#!/bin/bash
# Автоматический запуск всех тестов проекта ARA API.
#
# Этот скрипт выполняет комплексное тестирование проекта:
# - Проверка импортов всех модулей
# - Запуск unit тестов
# - Проверка типизации с mypy
# - Форматирование кода с black
# - Линтинг с flake8/ruff
#
# Использование:
#     ./test.sh              # Запустить все тесты
#     ./test.sh --imports    # Только тесты импортов
#     ./test.sh --help       # Показать справку

# Убираем set -e чтобы скрипт не прерывался при первой ошибке
# так как многие инструменты возвращают ненулевой код при обнаружении проблем

# Цвета для вывода
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Переменные
# Автоматическое определение корневой директории проекта
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
TESTS_DIR="$PROJECT_ROOT/test"
REPORTS_DIR="$TESTS_DIR/reports"
POETRY_CMD="poetry run"
EXIT_CODE=0

# Проверка что скрипт запускается из правильной директории
if [ ! -f "$PROJECT_ROOT/pyproject.toml" ]; then
    echo -e "${RED}❌ Ошибка: Проект не найден по пути $PROJECT_ROOT${NC}"
    echo -e "${RED}   Убедитесь что pyproject.toml существует в корне проекта${NC}"
    exit 1
fi

# Переход в корневую директорию проекта
cd "$PROJECT_ROOT"

# Создание папки для отчетов
mkdir -p "$REPORTS_DIR"

# Функции для вывода
print_header() {
    echo -e "\n${CYAN}=======================================${NC}"
    echo -e "${CYAN}$1${NC}"
    echo -e "${CYAN}=======================================${NC}\n"
}

print_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_error() {
    echo -e "${RED}❌ $1${NC}"
    EXIT_CODE=1
}

print_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

print_step() {
    echo -e "${PURPLE}🔄 $1${NC}"
}

# Проверка зависимостей
check_dependencies() {
    print_header "ПРОВЕРКА ЗАВИСИМОСТЕЙ"
    
    # Проверка Poetry
    if ! command -v poetry &> /dev/null; then
        print_error "Poetry не установлен. Установите Poetry для продолжения."
        exit 1
    fi
    print_success "Poetry найден"
    
    # Проверка виртуального окружения
    if [ ! -d "$PROJECT_ROOT/.venv" ]; then
        print_step "Создание виртуального окружения..."
        poetry install
    fi
    print_success "Виртуальное окружение готово"
    
    # Установка зависимостей
    print_step "Установка зависимостей..."
    poetry install --with dev 2>/dev/null || poetry install
    print_success "Зависимости установлены"
}

# Проверка импортов
run_import_tests() {
    print_header "ТЕСТИРОВАНИЕ ИМПОРТОВ"
    
    local import_report="$REPORTS_DIR/imports_report.txt"
    local import_unittest_report="$REPORTS_DIR/imports_unittest_report.txt"
    
    # Инициализация отчета
    echo "================================================" > "$import_report"
    echo "ОТЧЕТ О ТЕСТИРОВАНИИ ИМПОРТОВ" >> "$import_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$import_report"
    echo "================================================" >> "$import_report"
    echo "" >> "$import_report"
    
    print_step "Запуск unit тестов импортов..."
    echo "2. UNIT ТЕСТЫ ИМПОРТОВ:" >> "$import_report"
    echo "----------------------" >> "$import_report"
    
    if $POETRY_CMD python -m unittest "$TESTS_DIR/test_import.py" -v > "$import_unittest_report" 2>&1; then
        print_success "Unit тесты импортов прошли успешно"
        echo "✅ РЕЗУЛЬТАТ: УСПЕШНО" >> "$import_report"
    else
        print_warning "Unit тесты импортов завершились с предупреждениями"
        echo "⚠️ РЕЗУЛЬТАТ: ПРЕДУПРЕЖДЕНИЯ" >> "$import_report"
    fi
    
    # Добавляем результаты unit тестов в основной отчет
    cat "$import_unittest_report" >> "$import_report"
    echo "" >> "$import_report"
    echo "================================================" >> "$import_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$import_report"
    echo "================================================" >> "$import_report"
    
    print_info "Отчет сохранен: $import_report"
}

# Unit тесты
run_unit_tests() {
    print_header "UNIT ТЕСТИРОВАНИЕ"
    
    local unit_report="$REPORTS_DIR/unit_tests_report.txt"
    local unittest_output="$REPORTS_DIR/unittest_output.txt"
    local pytest_output="$REPORTS_DIR/pytest_output.txt"
    
    # Инициализация отчета
    echo "================================================" > "$unit_report"
    echo "ОТЧЕТ О UNIT ТЕСТИРОВАНИИ" >> "$unit_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$unit_report"
    echo "================================================" >> "$unit_report"
    echo "" >> "$unit_report"
    
    # Поиск всех тестовых файлов
    test_files=($(find "$TESTS_DIR" -name "test_*.py" -type f))
    
    if [ ${#test_files[@]} -eq 0 ]; then
        print_warning "Тестовые файлы не найдены"
        echo "⚠️ РЕЗУЛЬТАТ: Тестовые файлы не найдены" >> "$unit_report"
        return
    fi
    
    print_info "Найдено ${#test_files[@]} тестовых файлов"
    echo "Найдено тестовых файлов: ${#test_files[@]}" >> "$unit_report"
    echo "" >> "$unit_report"
    
    # Запуск с unittest
    print_step "Запуск тестов через unittest..."
    echo "1. ТЕСТИРОВАНИЕ ЧЕРЕЗ UNITTEST:" >> "$unit_report"
    echo "------------------------------" >> "$unit_report"
    
    local unittest_success=true
    for test_file in "${test_files[@]}"; do
        test_name=$(basename "$test_file" .py)
        print_step "Тестирование: $test_name"
        
        local test_output="$REPORTS_DIR/unittest_${test_name}_output.txt"
        
        if $POETRY_CMD python -m unittest "test.$test_name" -v > "$test_output" 2>&1; then
            print_success "$test_name: прошел"
            echo "✅ $test_name: УСПЕШНО" >> "$unit_report"
        else
            print_warning "$test_name: завершился с предупреждениями"
            echo "⚠️ $test_name: ПРЕДУПРЕЖДЕНИЯ" >> "$unit_report"
            unittest_success=false
        fi
        
        # Добавляем результаты теста в отчет
        echo "   Детали $test_name:" >> "$unit_report"
        cat "$test_output" | sed 's/^/   /' >> "$unit_report"
        echo "" >> "$unit_report"
    done
    
    # Общий результат unittest
    if [ "$unittest_success" = true ]; then
        echo "✅ ОБЩИЙ РЕЗУЛЬТАТ UNITTEST: УСПЕШНО" >> "$unit_report"
    else
        echo "⚠️ ОБЩИЙ РЕЗУЛЬТАТ UNITTEST: ЕСТЬ ПРЕДУПРЕЖДЕНИЯ" >> "$unit_report"
    fi
    echo "" >> "$unit_report"
    
    # Попытка запуска с pytest (если доступен)
    if $POETRY_CMD python -c "import pytest" 2>/dev/null; then
        print_step "Запуск тестов через pytest..."
        echo "2. ТЕСТИРОВАНИЕ ЧЕРЕЗ PYTEST:" >> "$unit_report"
        echo "----------------------------" >> "$unit_report"
        
        if $POETRY_CMD pytest "$TESTS_DIR" -v --tb=short > "$pytest_output" 2>&1; then
            print_success "Pytest тесты прошли успешно"
            echo "✅ РЕЗУЛЬТАТ PYTEST: УСПЕШНО" >> "$unit_report"
        else
            print_warning "Pytest тесты завершились с предупреждениями"
            echo "⚠️ РЕЗУЛЬТАТ PYTEST: ПРЕДУПРЕЖДЕНИЯ" >> "$unit_report"
        fi
        
        # Добавляем результаты pytest в отчет
        cat "$pytest_output" | sed 's/^/   /' >> "$unit_report"
    else
        print_info "Pytest не установлен, пропускаем"
        echo "ℹ️ Pytest не установлен, пропускаем" >> "$unit_report"
    fi
    
    echo "" >> "$unit_report"
    echo "================================================" >> "$unit_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$unit_report"
    echo "================================================" >> "$unit_report"
    
    print_info "Отчет сохранен: $unit_report"
}

# Проверка типизации
run_type_check() {
    print_header "ПРОВЕРКА ТИПИЗАЦИИ"
    
    local type_report="$REPORTS_DIR/type_check_report.txt"
    local mypy_output="$REPORTS_DIR/mypy_output.txt"
    
    # Инициализация отчета
    echo "================================================" > "$type_report"
    echo "ОТЧЕТ О ПРОВЕРКЕ ТИПИЗАЦИИ" >> "$type_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$type_report"
    echo "================================================" >> "$type_report"
    echo "" >> "$type_report"
    
    if $POETRY_CMD python -c "import mypy" 2>/dev/null; then
        print_step "Запуск проверки типов с mypy..."
        echo "ПРОВЕРКА ТИПОВ С MYPY:" >> "$type_report"
        echo "---------------------" >> "$type_report"
        
        if $POETRY_CMD mypy ara_api --ignore-missing-imports --no-strict-optional > "$mypy_output" 2>&1; then
            print_success "Проверка типов прошла успешно"
            echo "✅ РЕЗУЛЬТАТ: УСПЕШНО" >> "$type_report"
        else
            print_warning "Обнаружены проблемы с типизацией"
            echo "⚠️ РЕЗУЛЬТАТ: ПРОБЛЕМЫ ОБНАРУЖЕНЫ" >> "$type_report"
        fi
        
        # Добавляем результаты mypy в отчет
        echo "" >> "$type_report"
        echo "Детали проверки mypy:" >> "$type_report"
        cat "$mypy_output" | sed 's/^/   /' >> "$type_report"
    else
        print_info "MyPy не установлен, пропускаем проверку типов"
        echo "ℹ️ MyPy не установлен, проверка пропущена" >> "$type_report"
    fi
    
    echo "" >> "$type_report"
    echo "================================================" >> "$type_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$type_report"
    echo "================================================" >> "$type_report"
    
    print_info "Отчет сохранен: $type_report"
}

# Проверка форматирования
run_format_check() {
    print_header "ПРОВЕРКА ФОРМАТИРОВАНИЯ"
    
    local format_report="$REPORTS_DIR/format_check_report.txt"
    local black_output="$REPORTS_DIR/black_output.txt"
    
    # Инициализация отчета
    echo "================================================" > "$format_report"
    echo "ОТЧЕТ О ПРОВЕРКЕ ФОРМАТИРОВАНИЯ" >> "$format_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$format_report"
    echo "================================================" >> "$format_report"
    echo "" >> "$format_report"
    
    if $POETRY_CMD python -c "import black" 2>/dev/null; then
        print_step "Проверка форматирования с black..."
        echo "ПРОВЕРКА ФОРМАТИРОВАНИЯ С BLACK:" >> "$format_report"
        echo "-------------------------------" >> "$format_report"
        
        if $POETRY_CMD black --check ara_api test --diff > "$black_output" 2>&1; then
            print_success "Код отформатирован корректно"
            echo "✅ РЕЗУЛЬТАТ: КОД ОТФОРМАТИРОВАН КОРРЕКТНО" >> "$format_report"
        else
            print_warning "Код требует переформатирования"
            print_info "Запустите: poetry run black ara_api test"
            echo "⚠️ РЕЗУЛЬТАТ: КОД ТРЕБУЕТ ПЕРЕФОРМАТИРОВАНИЯ" >> "$format_report"
            echo "   Для исправления запустите: poetry run black ara_api test" >> "$format_report"
        fi
        
        # Добавляем результаты black в отчет
        echo "" >> "$format_report"
        echo "Детали проверки black:" >> "$format_report"
        if [ -s "$black_output" ]; then
            cat "$black_output" | sed 's/^/   /' >> "$format_report"
        else
            echo "   Нет изменений для форматирования" >> "$format_report"
        fi
    else
        print_info "Black не установлен, пропускаем проверку форматирования"
        echo "ℹ️ Black не установлен, проверка пропущена" >> "$format_report"
    fi
    
    echo "" >> "$format_report"
    echo "================================================" >> "$format_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$format_report"
    echo "================================================" >> "$format_report"
    
    print_info "Отчет сохранен: $format_report"
}

# Линтинг
run_linting() {
    print_header "ЛИНТИНГ КОДА"
    
    local lint_report="$REPORTS_DIR/linting_report.txt"
    local ruff_output="$REPORTS_DIR/ruff_output.txt"
    local flake8_output="$REPORTS_DIR/flake8_output.txt"
    
    # Инициализация отчета
    echo "================================================" > "$lint_report"
    echo "ОТЧЕТ О ЛИНТИНГЕ КОДА" >> "$lint_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$lint_report"
    echo "================================================" >> "$lint_report"
    echo "" >> "$lint_report"
    
    local lint_success=true
    
    # Проверка с ruff (если доступен)
    if $POETRY_CMD python -c "import ruff" 2>/dev/null; then
        print_step "Запуск линтинга с ruff..."
        echo "ЛИНТИНГ С RUFF:" >> "$lint_report"
        echo "--------------" >> "$lint_report"
        
        if $POETRY_CMD ruff check ara_api test > "$ruff_output" 2>&1; then
            print_success "Ruff проверка прошла успешно"
            echo "✅ РЕЗУЛЬТАТ RUFF: УСПЕШНО" >> "$lint_report"
        else
            print_warning "Ruff обнаружил проблемы в коде"
            echo "⚠️ РЕЗУЛЬТАТ RUFF: ПРОБЛЕМЫ ОБНАРУЖЕНЫ" >> "$lint_report"
            lint_success=false
        fi
        
        # Добавляем результаты ruff в отчет
        echo "" >> "$lint_report"
        echo "Детали проверки ruff:" >> "$lint_report"
        if [ -s "$ruff_output" ]; then
            cat "$ruff_output" | sed 's/^/   /' >> "$lint_report"
        else
            echo "   Проблемы не обнаружены" >> "$lint_report"
        fi
        echo "" >> "$lint_report"
    fi
    
    # Проверка с flake8 (если доступен)
    if $POETRY_CMD python -c "import flake8" 2>/dev/null; then
        print_step "Запуск линтинга с flake8..."
        echo "ЛИНТИНГ С FLAKE8:" >> "$lint_report"
        echo "----------------" >> "$lint_report"
        
        if $POETRY_CMD flake8 ara_api test --max-line-length=88 --ignore=E203,W503 > "$flake8_output" 2>&1; then
            print_success "Flake8 проверка прошла успешно"
            echo "✅ РЕЗУЛЬТАТ FLAKE8: УСПЕШНО" >> "$lint_report"
        else
            print_warning "Flake8 обнаружил проблемы в коде"
            echo "⚠️ РЕЗУЛЬТАТ FLAKE8: ПРОБЛЕМЫ ОБНАРУЖЕНЫ" >> "$lint_report"
            lint_success=false
        fi
        
        # Добавляем результаты flake8 в отчет
        echo "" >> "$lint_report"
        echo "Детали проверки flake8:" >> "$lint_report"
        if [ -s "$flake8_output" ]; then
            cat "$flake8_output" | sed 's/^/   /' >> "$lint_report"
        else
            echo "   Проблемы не обнаружены" >> "$lint_report"
        fi
        echo "" >> "$lint_report"
    fi
    
    # Если нет доступных линтеров
    if ! $POETRY_CMD python -c "import ruff" 2>/dev/null && ! $POETRY_CMD python -c "import flake8" 2>/dev/null; then
        print_info "Линтеры не установлены, пропускаем проверку"
        echo "ℹ️ Линтеры не установлены, проверка пропущена" >> "$lint_report"
    fi
    
    # Общий результат
    if [ "$lint_success" = true ]; then
        echo "✅ ОБЩИЙ РЕЗУЛЬТАТ ЛИНТИНГА: УСПЕШНО" >> "$lint_report"
    else
        echo "⚠️ ОБЩИЙ РЕЗУЛЬТАТ ЛИНТИНГА: ПРОБЛЕМЫ ОБНАРУЖЕНЫ" >> "$lint_report"
    fi
    
    echo "" >> "$lint_report"
    echo "================================================" >> "$lint_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$lint_report"
    echo "================================================" >> "$lint_report"
    
    print_info "Отчет сохранен: $lint_report"
}

# Проверка покрытия
run_coverage() {
    print_header "АНАЛИЗ ПОКРЫТИЯ КОДА"
    
    local coverage_report="$REPORTS_DIR/coverage_report.txt"
    local coverage_html_dir="$REPORTS_DIR/coverage_html"
    local coverage_output="$REPORTS_DIR/coverage_output.txt"
    
    # Инициализация отчета
    echo "================================================" > "$coverage_report"
    echo "ОТЧЕТ ОБ АНАЛИЗЕ ПОКРЫТИЯ КОДА" >> "$coverage_report"
    echo "Дата: $(date '+%Y-%m-%d %H:%M:%S')" >> "$coverage_report"
    echo "================================================" >> "$coverage_report"
    echo "" >> "$coverage_report"
    
    if $POETRY_CMD python -c "import coverage" 2>/dev/null; then
        print_step "Запуск анализа покрытия..."
        echo "АНАЛИЗ ПОКРЫТИЯ КОДА:" >> "$coverage_report"
        echo "--------------------" >> "$coverage_report"
        
        # Запуск анализа покрытия
        if $POETRY_CMD coverage run -m unittest discover "$TESTS_DIR" > "$coverage_output" 2>&1; then
            echo "✅ Тесты выполнены для анализа покрытия" >> "$coverage_report"
        else
            echo "⚠️ Тесты выполнены с предупреждениями" >> "$coverage_report"
        fi
        
        print_step "Генерация отчета покрытия..."
        echo "" >> "$coverage_report"
        echo "ОТЧЕТ О ПОКРЫТИИ:" >> "$coverage_report"
        echo "----------------" >> "$coverage_report"
        
        # Генерация текстового отчета
        $POETRY_CMD coverage report -m >> "$coverage_report" 2>&1
        
        print_step "Генерация HTML отчета..."
        # Создаем папку для HTML отчета
        mkdir -p "$coverage_html_dir"
        
        # Генерация HTML отчета
        if $POETRY_CMD coverage html -d "$coverage_html_dir" >> "$coverage_output" 2>&1; then
            print_info "HTML отчет создан в $coverage_html_dir/index.html"
            echo "" >> "$coverage_report"
            echo "📊 HTML отчет создан: $coverage_html_dir/index.html" >> "$coverage_report"
        else
            echo "⚠️ Ошибка при создании HTML отчета" >> "$coverage_report"
        fi
        
        # Генерация XML отчета для CI/CD
        $POETRY_CMD coverage xml -o "$REPORTS_DIR/coverage.xml" 2>/dev/null || true
        
        print_success "Анализ покрытия завершен"
        echo "" >> "$coverage_report"
        echo "✅ РЕЗУЛЬТАТ: АНАЛИЗ ЗАВЕРШЕН УСПЕШНО" >> "$coverage_report"
        
        # Добавляем детали выполнения тестов
        echo "" >> "$coverage_report"
        echo "ДЕТАЛИ ВЫПОЛНЕНИЯ ТЕСТОВ:" >> "$coverage_report"
        echo "------------------------" >> "$coverage_report"
        cat "$coverage_output" | sed 's/^/   /' >> "$coverage_report"
    else
        print_info "Coverage не установлен, пропускаем анализ покрытия"
        echo "ℹ️ Coverage не установлен, анализ пропущен" >> "$coverage_report"
    fi
    
    echo "" >> "$coverage_report"
    echo "================================================" >> "$coverage_report"
    echo "ОТЧЕТ ЗАВЕРШЕН: $(date '+%Y-%m-%d %H:%M:%S')" >> "$coverage_report"
    echo "================================================" >> "$coverage_report"
    
    print_info "Отчет сохранен: $coverage_report"
}

# Создание сводного отчета
create_summary_report() {
    print_header "СОЗДАНИЕ СВОДНОГО ОТЧЕТА"
    
    local summary_report="$REPORTS_DIR/summary_report.txt"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    
    # Инициализация сводного отчета
    echo "================================================" > "$summary_report"
    echo "СВОДНЫЙ ОТЧЕТ ТЕСТИРОВАНИЯ ARA API" >> "$summary_report"
    echo "Дата: $timestamp" >> "$summary_report"
    echo "================================================" >> "$summary_report"
    echo "" >> "$summary_report"
    
    # Информация о проекте
    echo "ИНФОРМАЦИЯ О ПРОЕКТЕ:" >> "$summary_report"
    echo "--------------------" >> "$summary_report"
    echo "Корневая папка: $PROJECT_ROOT" >> "$summary_report"
    echo "Папка тестов: $TESTS_DIR" >> "$summary_report"
    echo "Папка отчетов: $REPORTS_DIR" >> "$summary_report"
    echo "" >> "$summary_report"
    
    # Статус каждого типа тестов
    echo "СТАТУС ТЕСТИРОВАНИЯ:" >> "$summary_report"
    echo "-------------------" >> "$summary_report"
    
    # Проверяем какие отчеты существуют и их статус
    if [ -f "$REPORTS_DIR/imports_report.txt" ]; then
        echo "📋 Тесты импортов: ВЫПОЛНЕНЫ" >> "$summary_report"
        if grep -q "✅.*УСПЕШНО" "$REPORTS_DIR/imports_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ЕСТЬ ПРОБЛЕМЫ" >> "$summary_report"
        fi
    else
        echo "📋 Тесты импортов: НЕ ВЫПОЛНЯЛИСЬ" >> "$summary_report"
    fi
    
    if [ -f "$REPORTS_DIR/unit_tests_report.txt" ]; then
        echo "🧪 Unit тесты: ВЫПОЛНЕНЫ" >> "$summary_report"
        if grep -q "✅.*УСПЕШНО" "$REPORTS_DIR/unit_tests_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ЕСТЬ ПРОБЛЕМЫ" >> "$summary_report"
        fi
    else
        echo "🧪 Unit тесты: НЕ ВЫПОЛНЯЛИСЬ" >> "$summary_report"
    fi
    
    if [ -f "$REPORTS_DIR/type_check_report.txt" ]; then
        echo "🔍 Проверка типов: ВЫПОЛНЕНА" >> "$summary_report"
        if grep -q "✅.*УСПЕШНО" "$REPORTS_DIR/type_check_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ЕСТЬ ПРОБЛЕМЫ" >> "$summary_report"
        fi
    else
        echo "🔍 Проверка типов: НЕ ВЫПОЛНЯЛАСЬ" >> "$summary_report"
    fi
    
    if [ -f "$REPORTS_DIR/format_check_report.txt" ]; then
        echo "🎨 Проверка форматирования: ВЫПОЛНЕНА" >> "$summary_report"
        if grep -q "✅.*КОРРЕКТНО" "$REPORTS_DIR/format_check_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ТРЕБУЕТСЯ ИСПРАВЛЕНИЕ" >> "$summary_report"
        fi
    else
        echo "🎨 Проверка форматирования: НЕ ВЫПОЛНЯЛАСЬ" >> "$summary_report"
    fi
    
    if [ -f "$REPORTS_DIR/linting_report.txt" ]; then
        echo "🔧 Линтинг: ВЫПОЛНЕН" >> "$summary_report"
        if grep -q "✅.*УСПЕШНО" "$REPORTS_DIR/linting_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ЕСТЬ ПРОБЛЕМЫ" >> "$summary_report"
        fi
    else
        echo "🔧 Линтинг: НЕ ВЫПОЛНЯЛСЯ" >> "$summary_report"
    fi
    
    if [ -f "$REPORTS_DIR/coverage_report.txt" ]; then
        echo "📊 Анализ покрытия: ВЫПОЛНЕН" >> "$summary_report"
        if grep -q "✅.*УСПЕШНО" "$REPORTS_DIR/coverage_report.txt"; then
            echo "   Результат: ✅ УСПЕШНО" >> "$summary_report"
        else
            echo "   Результат: ⚠️ ЕСТЬ ПРОБЛЕМЫ" >> "$summary_report"
        fi
    else
        echo "📊 Анализ покрытия: НЕ ВЫПОЛНЯЛСЯ" >> "$summary_report"
    fi
    
    echo "" >> "$summary_report"
    
    # Список всех созданных файлов отчетов
    echo "СОЗДАННЫЕ ОТЧЕТЫ:" >> "$summary_report"
    echo "----------------" >> "$summary_report"
    find "$REPORTS_DIR" -name "*.txt" -o -name "*.xml" | sort | while read report; do
        filename=$(basename "$report")
        filesize=$(ls -lh "$report" | awk '{print $5}')
        echo "📄 $filename ($filesize)" >> "$summary_report"
    done
    
    # Проверяем HTML отчеты
    if [ -d "$REPORTS_DIR/coverage_html" ]; then
        echo "📊 coverage_html/ (HTML отчет покрытия)" >> "$summary_report"
    fi
    
    echo "" >> "$summary_report"
    
    # Общий статус тестирования
    echo "ОБЩИЙ СТАТУС:" >> "$summary_report"
    echo "------------" >> "$summary_report"
    if [ $EXIT_CODE -eq 0 ]; then
        echo "🎉 ТЕСТИРОВАНИЕ ЗАВЕРШЕНО УСПЕШНО" >> "$summary_report"
    else
        echo "⚠️ ТЕСТИРОВАНИЕ ЗАВЕРШЕНО С ПРОБЛЕМАМИ" >> "$summary_report"
        echo "   Код завершения: $EXIT_CODE" >> "$summary_report"
    fi
    
    echo "" >> "$summary_report"
    echo "================================================" >> "$summary_report"
    echo "ОТЧЕТ СОЗДАН: $timestamp" >> "$summary_report"
    echo "================================================" >> "$summary_report"
    
    print_success "Сводный отчет создан: $summary_report"
}

# Справка
show_help() {
    echo -e "${CYAN}Использование: $0 [ОПЦИИ]${NC}\n"
    echo "Опции:"
    echo "  --imports       Только тесты импортов"
    echo "  --unit          Только unit тесты"
    echo "  --types         Только проверка типизации"
    echo "  --format        Только проверка форматирования"
    echo "  --lint          Только линтинг"
    echo "  --coverage      Только анализ покрытия"
    echo "  --no-deps       Пропустить проверку зависимостей"
    echo "  --help          Показать эту справку"
    echo
    echo "Примеры:"
    echo "  $0                    # Запустить все тесты"
    echo "  $0 --imports --unit  # Только импорты и unit тесты"
    echo
    echo "Отчеты:"
    echo "  Все отчеты сохраняются в папке: test/reports/"
    echo "  - imports_report.txt         # Отчет о тестах импортов"
    echo "  - unit_tests_report.txt      # Отчет о unit тестах"
    echo "  - type_check_report.txt      # Отчет о проверке типизации"
    echo "  - format_check_report.txt    # Отчет о проверке форматирования"
    echo "  - linting_report.txt         # Отчет о линтинге"
    echo "  - coverage_report.txt        # Отчет об анализе покрытия"
    echo "  - summary_report.txt         # Сводный отчет всех тестов"
    echo "  - coverage_html/             # HTML отчет покрытия"
}

# Основная функция
main() {
    local run_imports=false
    local run_unit=false
    local run_types=false
    local run_format=false
    local run_lint=false
    local run_coverage_check=false
    local check_deps=true
    local run_all=true
    
    # Парсинг аргументов
    while [[ $# -gt 0 ]]; do
        case $1 in
            --imports)
                run_imports=true
                run_all=false
                shift
                ;;
            --unit)
                run_unit=true
                run_all=false
                shift
                ;;
            --types)
                run_types=true
                run_all=false
                shift
                ;;
            --format)
                run_format=true
                run_all=false
                shift
                ;;
            --lint)
                run_lint=true
                run_all=false
                shift
                ;;
            --coverage)
                run_coverage_check=true
                run_all=false
                shift
                ;;
            --no-deps)
                check_deps=false
                shift
                ;;
            --help)
                show_help
                exit 0
                ;;
            *)
                print_error "Неизвестная опция: $1"
                show_help
                exit 1
                ;;
        esac
    done
    
    # Начало тестирования
    print_header "ЗАПУСК ТЕСТИРОВАНИЯ ARA API"
    print_info "Корневая папка проекта: $PROJECT_ROOT"
    
    # Проверка зависимостей
    if [ "$check_deps" = true ]; then
        check_dependencies
    fi
    
    # Выполнение тестов
    if [ "$run_all" = true ]; then
        run_import_tests
        run_unit_tests
        run_type_check
        run_format_check
        run_linting
        run_coverage
    else
        [ "$run_imports" = true ] && run_import_tests
        [ "$run_unit" = true ] && run_unit_tests
        [ "$run_types" = true ] && run_type_check
        [ "$run_format" = true ] && run_format_check
        [ "$run_lint" = true ] && run_linting
        [ "$run_coverage_check" = true ] && run_coverage
    fi
    
    # Итоговый результат
    print_header "РЕЗУЛЬТАТЫ ТЕСТИРОВАНИЯ"
    
    # Создание сводного отчета
    create_summary_report
    
    if [ $EXIT_CODE -eq 0 ]; then
        print_success "Все проверки завершены успешно! 🎉"
    else
        print_error "Обнаружены проблемы в коде. Проверьте вывод выше."
    fi
    
    print_info "Все отчеты сохранены в папке: $REPORTS_DIR"
    
    exit $EXIT_CODE
}

# Запуск
main "$@"