#!/usr/bin/env python3
"""
Тесты для проверки корректности импортов всех модулей проекта ARA API.

Этот модуль автоматически обнаруживает все Python модули в проекте
(папки с файлами __init__.py) и проверяет возможность их импорта,
а также валидирует экспортируемые символы.
"""

import sys
import unittest
from pathlib import Path
from typing import Any, Dict

from test.subutils.import_checker import ImportChecker

# Добавляем корневую папку проекта в sys.path
PROJECT_ROOT = Path(__file__).parent.parent
sys.path.insert(0, str(PROJECT_ROOT))


class TestImports(unittest.TestCase):
    """Тестовый класс для проверки импортов модулей."""

    @classmethod
    def setUpClass(cls) -> None:
        """Инициализация тестового класса."""
        cls.checker = ImportChecker(PROJECT_ROOT)
        cls.results = cls.checker.run_all_tests()

    def test_all_modules_importable(self) -> None:
        """Проверяет, что все модули можно импортировать."""
        failed_imports = []

        for detail in self.results["details"]:
            if not detail["import_success"]:
                failed_imports.append(
                    (detail["module_name"], detail["error_message"])
                )

        if failed_imports:
            error_msg = "Не удалось импортировать следующие модули:\n"
            for module, error in failed_imports:
                error_msg += f"  - {module}: {error}\n"
            self.fail(error_msg)

    def test_module_exports_available(self) -> None:
        """Проверяет, что все ожидаемые экспорты доступны в модулях."""
        modules_with_missing = []

        for detail in self.results["details"]:
            if detail["import_success"] and detail["missing_attributes"]:
                modules_with_missing.append(
                    (detail["module_name"], detail["missing_attributes"])
                )

        if modules_with_missing:
            error_msg = "В следующих модулях отсутствуют ожидаемые атрибуты:\n"
            for module, missing_attrs in modules_with_missing:
                error_msg += f"  - {module}: {', '.join(missing_attrs)}\n"
            self.fail(error_msg)

    def test_no_circular_imports(self) -> None:
        """Проверяет отсутствие циклических импортов."""
        # Эта проверка косвенная - если все модули импортируются успешно,
        # значит нет критических циклических зависимостей
        self.assertGreater(
            self.results["successful_imports"],
            0,
            "Ни один модуль не удалось импортировать - возможны циклические зависимости",
        )


def print_detailed_report(results: Dict[str, Any]) -> None:
    """
    Выводит детальный отчет о результатах тестирования.

    Args:
        results: Результаты тестирования.
    """
    print("\n" + "=" * 80)
    print("ОТЧЕТ О ТЕСТИРОВАНИИ ИМПОРТОВ")
    print("=" * 80)

    print("\nОбщая статистика:")
    print(f"  Всего модулей: {results['total_modules']}")
    print(f"  Успешных импортов: {results['successful_imports']}")
    print(f"  Неудачных импортов: {results['failed_imports']}")
    print(
        f"  Модулей с отсутствующими атрибутами: {results['modules_with_missing_attrs']}"
    )

    # Успешные импорты
    if results["successful_imports"] > 0:
        print(f"\n✅ УСПЕШНЫЕ ИМПОРТЫ ({results['successful_imports']}):")
        for detail in results["details"]:
            if detail["import_success"]:
                print(f"  ✓ {detail['module_name']}")
                if detail["expected_exports"]:
                    print(
                        f"    Экспорты: {', '.join(detail['expected_exports'])}"
                    )
                if detail["missing_attributes"]:
                    print(
                        f"    ⚠️  Отсутствуют: {', '.join(detail['missing_attributes'])}"
                    )

    # Неудачные импорты
    if results["failed_imports"] > 0:
        print(f"\n❌ НЕУДАЧНЫЕ ИМПОРТЫ ({results['failed_imports']}):")
        for detail in results["details"]:
            if not detail["import_success"]:
                print(f"  ✗ {detail['module_name']}")
                print(f"    Ошибка: {detail['error_message']}")
                print(f"    Путь: {detail['module_path']}")

    # Итоговая оценка
    success_rate = (
        results["successful_imports"] / results["total_modules"]
    ) * 100
    print(f"\n{'=' * 80}")
    print(
        f"ИТОГОВАЯ ОЦЕНКА: {success_rate:.1f}% модулей импортируется успешно"
    )

    if success_rate == 100:
        print("🎉 Отлично! Все модули импортируются без ошибок!")
    elif success_rate >= 80:
        print("👍 Хорошо! Большинство модулей работают корректно.")
    else:
        print("⚠️  Требуется внимание! Много модулей с ошибками импорта.")


if __name__ == "__main__":
    # Запуск в режиме отчета
    if len(sys.argv) > 1:
        checker = ImportChecker(PROJECT_ROOT)
        if sys.argv[1] == "--report":
            results = checker.run_all_tests()
            print_detailed_report(results)
    else:
        # Запуск unit тестов
        unittest.main(verbosity=2)
