#!/usr/bin/env python3
"""
Тестовый скрипт для проверки подключения к MSP сервису.

Использует gRPCSync для получения данных телеметрии с дрона и логирует их.
"""

import time

import grpc

from ara_api._utils import Logger, gRPCSync
from ara_api._utils.data.msp import attitude, velocity


def test_msp_connection():
    """Тестирует подключение к MSP сервису и получение данных."""

    # Создаем логгер
    logger = Logger(
        log_level="DEBUG",
        log_to_file=True,
        log_to_terminal=True,
        log_dir="logs/test",
    )

    logger.info("=" * 80)
    logger.info("Запуск теста подключения к MSP сервису")
    logger.info("=" * 80)

    try:
        # Создаем gRPC клиент
        logger.info("Инициализация gRPCSync клиента...")
        grpc_client = gRPCSync.get_instance()
        logger.info("gRPCSync клиент успешно создан")

        # Тестируем получение различных данных
        test_iterations = 10
        logger.info(
            f"Запуск цикла тестирования ({test_iterations} итераций)..."
        )

        for i in range(test_iterations):
            logger.info(f"\n--- Итерация {i + 1}/{test_iterations} ---")

            # 1. Получаем данные IMU
            logger.debug("Запрос данных IMU...")
            imu_data = grpc_client.msp_get_raw_imu()
            if imu_data:
                logger.info(
                    f"IMU данные получены: "
                    f"acc=[{imu_data.grpc.acc.x:.2f}, {imu_data.grpc.acc.y:.2f}, {imu_data.grpc.acc.z:.2f}], "
                    f"gyro=[{imu_data.grpc.gyro.x:.2f}, {imu_data.grpc.gyro.y:.2f}, {imu_data.grpc.gyro.z:.2f}]"
                )
            else:
                logger.warning("Не удалось получить IMU данные")

            # 2. Получаем данные ориентации
            logger.debug("Запрос данных Attitude...")
            attitude_data = grpc_client.msp_get_attitude()
            if attitude_data:
                logger.info(
                    f"Attitude данные получены: "
                    f"roll={attitude_data.json['roll']:.2f}°, "
                    f"pitch={attitude_data.json['pitch']:.2f}°, "
                    f"yaw={attitude_data.json['yaw']:.2f}°"
                )
            else:
                logger.warning("Не удалось получить Attitude данные")

            # 3. Получаем данные высоты
            logger.debug("Запрос данных Altitude...")
            altitude_data = grpc_client.msp_get_altitude()
            if altitude_data:
                logger.info(
                    f"Altitude данные получены: alt={altitude_data.alt:.2f}м"
                )
            else:
                logger.warning("Не удалось получить Altitude данные")

            # 4. Получаем данные позиции
            logger.debug("Запрос данных Position...")
            position_data = grpc_client.msp_get_position()
            if position_data:
                logger.info(
                    f"Position данные получены: "
                    f"x={position_data.json['x']:.2f}м, "
                    f"y={position_data.json['y']:.2f}м, "
                    f"z={position_data.json['z']:.2f}м"
                )
            else:
                logger.warning("Не удалось получить Position данные")

            # 5. Получаем данные моторов
            logger.debug("Запрос данных Motor...")
            motor_data = grpc_client.msp_get_motor()
            if motor_data:
                logger.info(
                    f"Motor данные получены: "
                    f"M1={motor_data.grpc.data[0]}, M2={motor_data.grpc.data[1]}, "
                    f"M3={motor_data.grpc.data[2]}, M4={motor_data.grpc.data[3]}"
                )
            else:
                logger.warning("Не удалось получить Motor данные")

            # 6. Получаем данные аналоговых входов
            logger.debug("Запрос данных Analog...")
            analog_data = grpc_client.msp_get_analog()
            if analog_data:
                logger.info(
                    f"Analog данные получены: "
                    f"vbat={analog_data.json.get('vbat', 0):.2f}В, "
                    f"current={analog_data.json.get('current', 0):.0f}mA"
                )
            else:
                logger.warning("Не удалось получить Analog данные")

            # Ждем перед следующей итерацией
            if i < test_iterations - 1:
                time.sleep(0.5)

        logger.info("\n" + "=" * 80)
        logger.info("Тест завершен успешно!")
        logger.info("=" * 80)

    except KeyboardInterrupt:
        logger.warning("\nТест прерван пользователем (Ctrl+C)")

    except Exception as e:
        logger.error(f"Ошибка при выполнении теста: {e}")
        import traceback

        logger.error(f"Traceback:\n{traceback.format_exc()}")

    finally:
        logger.info("Завершение работы тестового скрипта")


def test_msp_connection_continuous():
    """Непрерывный мониторинг телеметрии MSP (для длительного тестирования)."""

    logger = Logger(
        log_level="INFO",
        log_to_file=True,
        log_to_terminal=True,
        log_dir="logs/test",
    )

    logger.info("=" * 80)
    logger.info("Запуск непрерывного мониторинга MSP телеметрии")
    logger.info("Нажмите Ctrl+C для остановки")
    logger.info("=" * 80)

    try:
        grpc_client = gRPCSync.get_instance()
        iteration = 0

        while True:
            iteration += 1

            # Получаем только основные данные
            position_data = grpc_client.msp_get_position()
            if position_data:
                logger.info(
                    f"Position данные получены: "
                    f"x={position_data.json['x']:.2f}м, "
                    f"y={position_data.json['y']:.2f}м, "
                    f"z={position_data.json['z']:.2f}м"
                )
            else:
                logger.warning("Не удалось получить Position данные")

            velocity_data = grpc_client.msp_get_velocity()
            print(f"DEBUG: velocity_data = {velocity_data}")
            print(
                f"DEBUG: velocity_data.grpc.data = {velocity_data.grpc.data if velocity_data else 'None'}"
            )
            if velocity_data:
                logger.info(
                    f"Velocity данные получены: "
                    f"x={velocity_data.json['x']:.2f}м/c, "
                    f"y={velocity_data.json['y']:.2f}м/c, "
                    f"z={velocity_data.json['z']:.2f}м/c"
                )
            else:
                logger.warning("Не удалось получить Position данные")

            logger.info("")

            time.sleep(0.1)  # 10 Hz

    except KeyboardInterrupt:
        logger.info("\nМониторинг остановлен пользователем")

    except Exception as e:
        logger.error(f"Ошибка при мониторинге: {e}")
        import traceback

        logger.error(f"Traceback:\n{traceback.format_exc()}")


if __name__ == "__main__":
    import sys

    print("\n=== Тестовый скрипт MSP сервиса ===")
    print("1. Разовый тест (10 итераций)")
    print("2. Непрерывный мониторинг (до Ctrl+C)")
    print("===================================\n")

    choice = input("Выберите режим (1 или 2): ").strip()

    if choice == "1":
        test_msp_connection()
    elif choice == "2":
        test_msp_connection_continuous()
    else:
        print("Неверный выбор. Запуск разового теста по умолчанию...")
        test_msp_connection()
