#!/usr/bin/env python3
"""
Test script to send a sample daily digest to Slack
"""
import json
from datetime import date
import requests

# Sample daily digest data
sample_digest = {
    "date": date.today(),
    "total_runs": 3,
    "total_checks": 12,
    "checks_passed": 11,
    "total_changes": 45,
    "commits": 2,
    "success_rate": 91.7
}

# Create markdown message
message = f"""# Daily Paper Maintenance Digest
**Date:** {sample_digest['date'].strftime('%A, %B %d, %Y')}

---

## Summary
- **Runs today:** {sample_digest['total_runs']}
- **Total checks:** {sample_digest['total_checks']}
- **Checks passed:** {sample_digest['checks_passed']} ({sample_digest['success_rate']:.1f}%)
- **Lines changed:** {sample_digest['total_changes']}
- **Commits:** {sample_digest['commits']}

## Status
✅ All systems operational - {sample_digest['success_rate']:.1f}% success rate

## Run Details

### Run 1 - 09:00 AM
- ✅ BibTeX validation: PASSED
- ✅ Reference formatting: PASSED
- ✅ LaTeX compilation: PASSED
- ✅ PDF generation: PASSED
- **Changes:** 15 lines updated in references.bib

### Run 2 - 12:00 PM
- ✅ BibTeX validation: PASSED
- ✅ Reference formatting: PASSED
- ⚠️ LaTeX compilation: WARNING (minor style issue)
- ✅ PDF generation: PASSED
- **Changes:** 10 lines formatted

### Run 3 - 18:00 PM
- ✅ BibTeX validation: PASSED
- ✅ Reference formatting: PASSED
- ✅ LaTeX compilation: PASSED
- ✅ PDF generation: PASSED
- **Changes:** 20 lines in main.tex

---

_Generated by LaTeX Paper Automation System_
"""

def send_to_slack(webhook_url: str):
    """Send sample digest to Slack"""
    payload = {
        "username": "Paper Automation Bot",
        "icon_emoji": ":memo:",
        "text": f"*📊 Daily Paper Digest - {sample_digest['date'].strftime('%b %d')}*\n{message}"
    }

    try:
        response = requests.post(webhook_url, json=payload, timeout=10)
        if response.status_code == 200:
            print("✓ Sample daily digest sent to Slack successfully!")
            print(f"\nPreview of sent message:\n{'-'*60}")
            print(message)
            print('-'*60)
            return True
        else:
            print(f"✗ Failed to send: HTTP {response.status_code}")
            print(f"Response: {response.text}")
            return False
    except Exception as e:
        print(f"✗ Error sending to Slack: {e}")
        return False


if __name__ == "__main__":
    import sys

    if len(sys.argv) < 2:
        print("Usage: python3 test_slack_digest.py <SLACK_WEBHOOK_URL>")
        print("\nExample:")
        print('  python3 test_slack_digest.py "https://hooks.slack.com/services/YOUR/WEBHOOK/URL"')
        sys.exit(1)

    webhook_url = sys.argv[1]
    success = send_to_slack(webhook_url)
    sys.exit(0 if success else 1)
