# tests/test_policies.py

import unittest

from anypoint_sdk._http import HttpClient
from anypoint_sdk.resources.policies import Policies
from tests.fakes import FakeSession
from tests.helpers import make_response

BASE = "https://api.test.local"


class PoliciesTests(unittest.TestCase):
    def test_list_api_policies_normalises_and_dedupes(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 20473240
        payload = {
            "policies": [
                {
                    "policyTemplateId": "348742",
                    "order": 1,
                    "type": "system",
                    "policyId": 7038442,
                    "implementationAsset": {
                        "assetId": "rate-limiting-sla-based-mule",
                        "groupId": "68ef9520-24e9-4cf2-b2f5-620025690913",
                        "version": "1.3.0",
                        "technology": "mule4",
                    },
                    "configuration": {"exposeHeaders": True},
                },
                # duplicate policyId to prove dedupe
                {
                    "policyTemplateId": "348742",
                    "order": 1,
                    "type": "system",
                    "policyId": 7038442,
                },
            ]
        }
        resp = make_response(
            status=200,
            json_body=payload,
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/policies",
        )
        http = HttpClient(base_url=BASE, session=FakeSession([resp]), retries=0)

        out = Policies(http).list_api_policies(org_id, env_id, api_id)

        self.assertEqual(len(out), 1)
        p = out[0]
        self.assertEqual(p["policyId"], 7038442)
        self.assertEqual(p["type"], "system")
        self.assertEqual(p["order"], 1)
        self.assertIn("implementationAsset", p)

    def test_list_api_policies_handles_missing_or_non_list(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 1
        # No 'policies' key
        resp = make_response(
            status=200,
            json_body={"tiers": {"values": []}},
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/policies",
        )
        http = HttpClient(base_url=BASE, session=FakeSession([resp]), retries=0)

        out = Policies(http).list_api_policies(org_id, env_id, api_id)
        self.assertEqual(out, [])

    def test_list_api_policies_ignores_non_dict_entries(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 1
        resp = make_response(
            status=200,
            json_body={"policies": ["bad", 123, None]},
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/policies",
        )
        http = HttpClient(base_url=BASE, session=FakeSession([resp]), retries=0)

        out = Policies(http).list_api_policies(org_id, env_id, api_id)
        self.assertEqual(out, [])


if __name__ == "__main__":
    unittest.main()
