# tests/test_tiers.py
import unittest

from anypoint_sdk._http import HttpClient
from anypoint_sdk.resources.tiers import Tiers
from tests.fakes import FakeSession
from tests.helpers import make_response

BASE = "https://api.test.local"


class TiersTests(unittest.TestCase):
    def test_list_api_tiers_normalises_and_dedupes(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 20473240
        payload = {
            "total": 2,
            "tiers": [
                {
                    "id": 2247207,
                    "name": "gold",
                    "description": "gold ties",
                    "limits": [
                        {
                            "maximumRequests": 100,
                            "timePeriodInMilliseconds": 1000,
                            "visible": True,
                        }
                    ],
                    "status": "ACTIVE",
                    "autoApprove": False,
                    "applicationCount": 2,
                },
                # duplicate to prove dedupe
                {"id": 2247207, "name": "gold"},
            ],
        }
        resp = make_response(
            status=200,
            json_body=payload,
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/tiers",
        )
        http = HttpClient(base_url=BASE, session=FakeSession([resp]), retries=0)

        out = Tiers(http).list_api_tiers(org_id, env_id, api_id)

        self.assertEqual(len(out), 1)
        t = out[0]
        self.assertEqual(t["id"], 2247207)
        self.assertEqual(t["name"], "gold")
        self.assertEqual(t["status"], "ACTIVE")
        self.assertIsInstance(t["limits"], list)
        self.assertEqual(t["applicationCount"], 2)

    def test_list_api_tiers_handles_missing_or_non_list(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 1
        # No 'tiers' key
        resp1 = make_response(
            status=200,
            json_body={"total": 0},
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/tiers",
        )
        http1 = HttpClient(base_url=BASE, session=FakeSession([resp1]), retries=0)
        out1 = Tiers(http1).list_api_tiers(org_id, env_id, api_id)
        self.assertEqual(out1, [])

        # 'tiers' not a list, but 'values' is present
        resp2 = make_response(
            status=200,
            json_body={"values": [{"id": 1, "name": "bronze"}]},
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/tiers",
        )
        http2 = HttpClient(base_url=BASE, session=FakeSession([resp2]), retries=0)
        out2 = Tiers(http2).list_api_tiers(org_id, env_id, api_id)
        self.assertEqual(out2[0]["name"], "bronze")

        # 'tiers' and 'values' both wrong types
        resp3 = make_response(
            status=200,
            json_body={"tiers": {"unexpected": True}, "values": "oops"},
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/tiers",
        )
        http3 = HttpClient(base_url=BASE, session=FakeSession([resp3]), retries=0)
        out3 = Tiers(http3).list_api_tiers(org_id, env_id, api_id)
        self.assertEqual(out3, [])

    def test_list_api_tiers_ignores_non_dict_entries(self):
        org_id = "o1"
        env_id = "e1"
        api_id = 1
        payload = {"tiers": ["bad", 123, None]}
        resp = make_response(
            status=200,
            json_body=payload,
            url=f"{BASE}/apimanager/api/v1/organizations/{org_id}/environments/{env_id}/apis/{api_id}/tiers",
        )
        http = HttpClient(base_url=BASE, session=FakeSession([resp]), retries=0)

        out = Tiers(http).list_api_tiers(org_id, env_id, api_id)
        self.assertEqual(out, [])


if __name__ == "__main__":
    unittest.main()
