"""
BeastX CLI Tool
===============

Command-line interface for BeastX library.
"""

import click
from rich.console import Console
from rich.panel import Panel
from rich.text import Text
from rich.table import Table
from rich import print as rprint
from . import __version__, __author__, __credits__

console = Console()


@click.group()
@click.version_option(version=__version__, prog_name='BeastX')
def main():
    """BeastX - Python Telegram Client Library"""
    pass


@main.command()
def info():
    """Display BeastX information"""
    
    banner = Text()
    banner.append("BeastX\n", style="bold cyan")
    banner.append(f"Version: {__version__}\n\n", style="green")
    banner.append("A powerful Python library for Telegram's MTProto API\n\n", style="dim")
    banner.append("Developer: ", style="dim")
    banner.append("t.me/GODMRUNAL\n", style="bold yellow")
    banner.append("Channel: ", style="dim")
    banner.append("@BEASTX_BOTS\n", style="bold yellow")
    banner.append("GitHub: ", style="dim")
    banner.append("github.com/beastx-python\n\n", style="bold blue")
    banner.append("Credits: ", style="dim")
    banner.append("Based on Telethon by Lonami", style="italic cyan")
    
    console.print(Panel(banner, border_style="cyan", padding=(1, 2)))


@main.command()
def quickstart():
    """Show a quick start guide"""
    
    console.print("\n[bold cyan]BeastX Quick Start Guide[/bold cyan]\n", style="bold")
    
    console.print("[yellow]1. Get API Credentials:[/yellow]")
    console.print("   • Visit https://my.telegram.org")
    console.print("   • Log in and go to 'API Development Tools'")
    console.print("   • Create an application and get your api_id and api_hash\n")
    
    console.print("[yellow]2. Install BeastX:[/yellow]")
    console.print("   pip install beastx-python\n")
    
    console.print("[yellow]3. Basic Usage:[/yellow]")
    
    code = """
from beastx import TelegramClient

api_id = 12345
api_hash = 'your_api_hash'

client = TelegramClient('session_name', api_id, api_hash)

async def main():
    await client.start()
    await client.send_message('me', 'Hello from BeastX!')

with client:
    client.loop.run_until_completed(main())
    """
    
    console.print(Panel(code, title="[green]Example Code[/green]", border_style="green"))
    
    console.print("\n[yellow]4. Documentation:[/yellow]")
    console.print("   • Full Docs: https://beastx-python.github.io")
    console.print("   • Telegram: https://t.me/BEASTX_BOTS")
    console.print("   • Original: https://docs.telethon.dev\n")


@main.command()
def examples():
    """List available examples"""
    
    table = Table(title="[bold cyan]BeastX Examples[/bold cyan]")
    table.add_column("Example", style="cyan", no_wrap=True)
    table.add_column("Description", style="white")
    
    examples_list = [
        ("simple_bot.py", "Basic bot with event handlers"),
        ("send_message.py", "Send messages to users/groups"),
        ("download_media.py", "Download media from channels"),
        ("auto_reply.py", "Auto-reply bot with patterns"),
        ("file_sender.py", "Send files and documents"),
        ("user_scraper.py", "Extract user information"),
        ("conversation.py", "Interactive conversation flows"),
    ]
    
    for name, desc in examples_list:
        table.add_row(name, desc)
    
    console.print(table)
    console.print("\n[dim]See the 'examples/' folder for full code[/dim]\n")


@main.command()
def features():
    """Display key features"""
    
    console.print("\n[bold cyan]BeastX Key Features[/bold cyan]\n")
    
    features_list = [
        ("📨 Messaging", "Send, edit, delete messages with rich formatting"),
        ("📁 File Operations", "Upload and download files with progress tracking"),
        ("🎯 Events", "Real-time event handlers (NewMessage, CallbackQuery, etc.)"),
        ("👥 User Management", "Get users, participants, and manage permissions"),
        ("🤖 Bot Support", "Full bot API with inline buttons and keyboards"),
        ("🔐 Sessions", "Persistent authentication with session files"),
        ("⚡ Performance", "Async/await support for high performance"),
        ("🛡️ Error Handling", "Comprehensive error classes for all scenarios"),
    ]
    
    for icon_title, desc in features_list:
        console.print(f"[yellow]{icon_title}[/yellow]: {desc}")
    
    console.print()


@main.command()
@click.option('--output', '-o', default='my_session', help='Session file name')
def create_session(output):
    """Create a new session file interactively"""
    
    console.print("\n[bold cyan]BeastX Session Creator[/bold cyan]\n")
    
    api_id = click.prompt("Enter your API ID", type=int)
    api_hash = click.prompt("Enter your API Hash")
    phone = click.prompt("Enter your phone number (with country code)")
    
    console.print(f"\n[green]Creating session '{output}'...[/green]")
    console.print("[yellow]Note: You'll receive a code via Telegram[/yellow]\n")
    
    try:
        from beastx import TelegramClient
        
        client = TelegramClient(output, api_id, api_hash)
        
        with client:
            client.start(phone=phone)
            console.print(f"\n[green]✓ Session created successfully: {output}.session[/green]\n")
            
    except ImportError:
        console.print("[red]Error: BeastX library not properly installed[/red]\n")
    except Exception as e:
        console.print(f"[red]Error: {e}[/red]\n")


if __name__ == '__main__':
    main()
