![queueio](logo.svg)

Python background queues with an async twist
============================================

Use async functions to manage complex background task workflows,
and keep using synchronous functions for everything else.

Getting Started
---------------

Install `queueio`:

```sh
pip install queueio
```

Create your routines:

```python
# basic.py
from time import sleep

from queueio import gather
from queueio import pause
from queueio import routine


@routine(name="blocking", queue="queueio")
def blocking():
    sleep(0.1)  # Regular blocking call


@routine(name="yielding", queue="queueio")
async def yielding(iterations: int):
    # Do them two at a time
    for _ in range(iterations // 2):
        await gather(blocking(), blocking())
        await pause(0.2)  # Release processing capacity
    if iterations % 2 == 1:
        await blocking()
```

Add the configuration to your `pyproject.toml`:

```toml
[tool.queueio]
# Configure RabbitMQ
pika = "amqp://guest:guest@localhost:5672/"
# Register the modules that the worker should load to find your routines
register = ["basic"]
```

The pika configuration can be overridden with an environment variable
to allow a project to be deployed in multiple environments.

```sh
QUEUEIO_PIKA='amqp://guest:guest@localhost:5672/'
```

Sync the queues to the broker:

```python
queueio sync
```

Submit the routine to run on a worker:

```python
from queueio import activate
from basic import yielding

with activate():
    yielding(7).submit()
```

Then run the worker to process submitted routines:

```sh
queueio run queueio=4
```

Monitor the status of active routine invocations:

```sh
queueio monitor
```

Stability
---------

The design of the public API is under active development
and is likely to change with any release.
Release notes will provide upgrade instructions,
but backward compatibility and deprecation warnings
will not generally be implemented.
