"""Hello, Dear ImGui: cross-platform Gui apps for Windows / Mac / Linux / iOS / Android / Emscripten with the simplicity of a "Hello World" app
https://github.com/pthom/hello_imgui
"""

# ruff: noqa: F811, B008, F821
from typing import List, Any, Callable, Tuple, Optional, overload, Dict
import numpy as np
import enum

from imgui_bundle.imgui.internal import DockNodeFlags_, DockNodeFlags
from imgui_bundle.imgui import (
    ImVec2,
    ImVec4,
    ImVec2Like,
    ImVec4Like,
    ImFontConfig,
    ImFont,
    ImTextureID,
    Dir,
    Cond_,
)

from imgui_bundle.imgui import test_engine

# Manual code
import imgui_bundle.imgui

ImGuiID = int
ImGuiDefaultSettings = imgui_default_settings  # noqa: F821

ImGuiDockNodeFlags_ = DockNodeFlags_
ImGuiDockNodeFlags = DockNodeFlags

DockSpaceName = str
ImGuiTestEngine = test_engine.TestEngine

VoidFunction = Callable[[], None]
AnyEventCallback = Callable[[Any], None]
ScreenSize = Tuple[int, int]
ScreenPosition = Tuple[int, int]
ImGuiCond_FirstUseEver = Cond_.first_use_ever
ImGuiDockNodeFlags_None = DockNodeFlags_.none
ImGuiDockNodeFlags_PassthruCentralNode = DockNodeFlags_.passthru_central_node

DefaultScreenSize = (800, 600)
DefaultWindowSize = (800, 600)
DefaultScreenPosition = (40, 40)

ImGuiCond = imgui_bundle.imgui.Cond
ImGuiDir_Down = Dir.down
ImGuiTheme = imgui_bundle.hello_imgui
ImGuiStyle = imgui_bundle.imgui.Style
ImGuiDir = imgui_bundle.imgui.Dir
ImGuiWindowFlags = imgui_bundle.imgui.WindowFlags

# ImWcharPair is used to defined unicode glyph ranges
ImWchar = int
ImWcharPair = Tuple[int, int]

DictTypeInputTextData = dict[str, Any]

def EmptyVoidFunction() -> VoidFunction:
    pass

def EmptyEventCallback() -> AnyEventCallback:
    pass

def set_load_asset_file_data_function(fn: Callable[[str], bytes]) -> None:
    """Register a Python callback that receives an asset and returns the file contents as a **bytes** object.

    Example
    -------
    ```python
    def my_loader(filename: str) -> bytes:
        from pathlib import Path
        return Path(filename).read_bytes()

    hello_imgui.set_load_file_bytes_func(my_loader)
    """
    pass

# !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
# <litgen_stub> // Autogenerated code below! Do not edit!
####################    <generated_from:hello_imgui_amalgamation.h>    ####################
# THIS FILE WAS GENERATED AUTOMATICALLY. DO NOT EDIT.

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui.h                                                                          //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# @@md#DpiAwareParams

class DpiAwareParams:
    """
    Hello ImGui will try its best to automatically handle DPI scaling for you.

    Parameter to change the scaling behavior:
    ------------------------------------------
    - `dpiWindowSizeFactor`:
           factor by which window size should be multiplied
       By default, Hello ImGui will compute it automatically, when it is set to 0.


    How to set manually:
    ---------------------------------
    If it fails (i.e. your window and/or fonts are too big or too small),
    you may set them manually:
       (1) Either by setting them programmatically in your application
           (set their values in `runnerParams.dpiAwareParams`)
       (2) Either by setting them in a `hello_imgui.ini` file. See hello_imgui/hello_imgui_example.ini for more info
    Note: if several methods are used, the order of priority is (1) > (2)

    For more information, see the documentation on DPI handling, here: https://pthom.github.io/hello_imgui/book/doc_api.html#handling-screens-with-high-dpi

    """

    # float dpiWindowSizeFactor = 0.0f;    /* original C++ signature */
    # `dpiWindowSizeFactor`
    #     factor by which window size should be multiplied to get a similar
    #     physical size on different OSes (as if they were all displayed on a 96 PPI screen).
    #     This affects the size of native app windows,
    #     but *not* imgui Windows, and *not* the size of widgets and text.
    #  In a standard environment (i.e. outside of Hello ImGui), an application with a size of 960x480 pixels,
    #  may have a physical size (in mm or inches) that varies depending on the screen DPI, and the OS.
    #
    #  Inside Hello ImGui, the window size is always treated as targeting a 96 PPI screen, so that its size will
    #  look similar whatever the OS and the screen DPI.
    #  In our example, our 960x480 pixels window will try to correspond to a 10x5 inches window
    #
    #  Hello ImGui does its best to compute it on all OSes.
    #  However, if it fails you may set its value manually.
    #  If it is set to 0, Hello ImGui will compute it automatically,
    #  and the resulting value will be stored in `dpiWindowSizeFactor`.
    dpi_window_size_factor: float = 0.0

    # float DpiFontLoadingFactor() const {    /* original C++ signature */
    #         return dpiWindowSizeFactor;
    #     }
    def dpi_font_loading_factor(self) -> float:
        """`DpiFontLoadingFactor`
           factor by which font size should be multiplied at loading time to get a similar visible size on different OSes.
           This is equal to dpiWindowSizeFactor
        The size will be equivalent to a size given for a 96 PPI screen
        """
        pass
    # DpiAwareParams(float dpiWindowSizeFactor = 0.0f);    /* original C++ signature */
    def __init__(self, dpi_window_size_factor: float = 0.0) -> None:
        """Auto-generated default constructor with named params"""
        pass

# ----------------------------------------------------------------------------

# @@md

# *
# @@md#DocEmToVec2
#
# Special care must be taken in order to correctly handle screen with high DPI
# (for example, almost all recent laptops screens).
#
# Using ImVec2 with fixed values is *almost always a bad idea* if you intend your
# application to be used on high DPI screens!
# Otherwise, widgets might be misplaced or too small on different screens and/or OS.
#
# Instead, you should use scale your widgets and windows relatively to the font size,
# as is done with the [em CSS Unit](https://lyty.dev/css/css-unit.html).
#
# @@md
# *

# @@md#EmToVec2
#  __HelloImGui::EmToVec2()__ returns an ImVec2 that you can use to size
#  or place your widgets in a DPI independent way.
#  Values are in multiples of the font size (i.e. as in the em CSS unit).
# ImVec2 EmToVec2(float x, float y);    /* original C++ signature */
@overload
def em_to_vec2(x: float, y: float) -> ImVec2:
    pass

# ImVec2 EmToVec2(ImVec2 v);    /* original C++ signature */
@overload
def em_to_vec2(v: ImVec2Like) -> ImVec2:
    pass

# float EmSize();    /* original C++ signature */
@overload
def em_size() -> float:
    """__HelloImGui::EmSize()__ returns the visible font size on the screen."""
    pass

# float EmSize(float nbLines);    /* original C++ signature */
@overload
def em_size(nb_lines: float) -> float:
    """__HelloImGui::EmSize(nbLines)__ returns a size corresponding to nbLines text lines"""
    pass

# ImVec2 PixelsToEm(ImVec2 pixels);    /* original C++ signature */
def pixels_to_em(pixels: ImVec2Like) -> ImVec2:
    """__HelloImGui::PixelToEm()__ converts a Vec2 in pixels coord to a Vec2 in em units"""
    pass

# float  PixelSizeToEm(float pixelSize);    /* original C++ signature */
def pixel_size_to_em(pixel_size: float) -> float:
    """__HelloImGui::PixelSizeToEm()__ converts a size in pixels coord to a size in em units"""
    pass

# @@md

# DpiAwareParams* GetDpiAwareParams();    /* original C++ signature */
def get_dpi_aware_params() -> DpiAwareParams:
    """Returns the current DpiAwareParams, which are used
    for font loading and window size scaling
    """
    pass

# ----------------------------------------------------------------------------

#
# Legacy API, you should use RunnerParams.dpiAwareParams instead
#
# float DpiFontLoadingFactor();    /* original C++ signature */
def dpi_font_loading_factor() -> float:
    """Multiply font sizes by this factor when loading fonts manually with ImGui::GetIO().Fonts->AddFont...
    (HelloImGui::LoadFontTTF does this by default)
    """
    pass

# float DpiWindowSizeFactor();    /* original C++ signature */
def dpi_window_size_factor() -> float:
    """DpiWindowSizeFactor() is the factor by which window size should be multiplied to get a similar visible size on different OSes.
    It returns ApplicationScreenPixelPerInch / 96 under windows and linux. Under macOS, it will return 1.
    """
    pass

# ----------------------------------------------------------------------------
#           Handling screens with high DPI
# ----------------------------------------------------------------------------
#
# @@md#HandlingScreenHighDPI
#
# _Note: This part is relevant only for more advanced usages. If you use `HelloImGui::LoadFont()`,
# and always use `HelloImGui::EmToVec2()` to place widgets, you do not need to worry about DPI handling_
#
### OS specificities
#
# There are several important things to know about high-DPI handling within Hello ImGui and Dear ImGui:
#
# 1. (virtual) screen coordinates vs (physical) pixels
# 2. DisplayFramebufferScale: Frame buffer size vs window size
# 3. FontGlobalScale: display-time font scaling factor
# 4. How to load fonts with the correct size
# 5. How to get similar window sizes on different OSes/DPI
#
#
### Screen coordinates
#
# Screen coordinates are the coordinates you use to place and size windows on the screen.
#
# **Screen coordinates do not always correspond to physical pixels**
#
# - On macOS/iOS retina screens, a screen coordinate corresponds typically
#  to 2x2 physical pixels (but this may vary if you change the display scaling)
# - On most Linux distributions, whenever there is a high DPI screen
#  you can set the display scale. For example if you set the scale to 300%,
#  then a screen coordinate will correspond to 3x3 physical pixels
# - On Windows, there are two possible situations:
#    - If the application is DPI aware, a screen coordinate corresponds to 1x1 physical pixel,
#      and you can use the full extent of your screen resolution.
#    - If the application is not DPI aware, a screen coordinate may correspond to 2x2 physical pixels
#      (if the display scaling is set to 200% for example). However, the rendering of your application
#      will be blurry and will not use the full extent of your screen resolution.
#    - Notes:
#        - Applications created with HelloImGui are DPI aware by default (when using glfw and sdl backends).
#        - SDL applications are normally not DPI aware. However, HelloImGui makes them DPI aware.
#
#
### DisplayFramebufferScale
# `DisplayFramebufferScale` is the ratio between the frame buffer size and the window size.
#
# The frame buffer size is the size of the internal buffer used by the rendering backend.
# It might be bigger than the actual window size.
# `ImVec2 ImGui::GetIO().DisplayFramebufferScale` is a factor by which the frame buffer size is bigger than the window size.
# It is set by the platform backend after it was initialized, and typically reflects the scaling ratio between
# physical pixels and screen coordinates.
#
# Under windows, it will always be (1,1). Under macOS / linux, it will reflect the current display scaling.
# It will typically be (2,2) on a macOS retina screen.
#
# Notes:
# - You cannot change DisplayFramebufferScale manually, it will be reset at each new frame, by asking the platform backend.
#
#
### How to load fonts with the correct size
#
#### Using HelloImGui (recommended)
#
# [`HelloImGui::LoadFont()`](https://pthom.github.io/hello_imgui/book/doc_api.html#load-fonts) will load fonts
# with the correct size, taking into account the DPI scaling.
#
#### Using Dear ImGui
# `ImGui::GetIO().Fonts->AddFontFromFileTTF()` loads a font with a given size, in *physical pixels*.
# KKDYNFONT: TBC...
#
### Reproducible physical window sizes (in mm or inches)
#
#### Using HelloImGui
# Simply specify a window size that corresponds to theoretical 96 PPI screen (inside `RunnerParams.appWindowParams.windowGeometry.size`)
#
#### Using your own code to create the backend window
# If you prefer to create the window by yourself, its physical size in millimeters may vary widely,
# depending on the OS and the current screen DPI setting.
# Typically under Windows, your window may appear to be very small if your screen is high DPI.
#
# To get a similar window size on different OSes/DPI, you should multiply the window size by `HelloImGui::DpiWindowSizeFactor()`.
#
# Note: DpiWindowSizeFactor() is equal to `CurrentScreenPixelPerInch / 96` under windows and linux, and always 1 under macOS.
#
### Fine tune DPI Handling
#
# See [`HelloImGui::DpiAwareParams`](https://pthom.github.io/hello_imgui/book/doc_params.html#dpi-aware-params)
# for more information on how to fine tune DPI handling when using Hello ImGui.
# @@md
#

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/hello_imgui_assets.h included by hello_imgui.h                             //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# *
# @@md#AssetsStructure
#
# Assets located beside the application CMakeLists are embedded automatically.
#
# For example, you can have the following project structure:
# ```
# my_app/
# ├── CMakeLists.txt        # Your app's CMakeLists
# ├── assets/               # Its assets: for mobile devices and emscripten
# │         └── fonts/            # they are embedded automatically by hello_imgui_add_app.cmake
# │             └── my_font.ttf
# ├── my_app.main.cpp       # Its source code
# ```
#
# Then you can load the asset "fonts/my_font.ttf", on all platforms.
#
# @@md
#

# @@md#LoadAssetFileData

class AssetFileData:
    # void * data = nullptr;    /* original C++ signature */
    data: Any = None
    # size_t dataSize = 0;    /* original C++ signature */
    data_size: int = 0
    # AssetFileData(size_t dataSize = 0);    /* original C++ signature */
    def __init__(self, data_size: int = 0) -> None:
        """Auto-generated default constructor with named params"""
        pass

# AssetFileData LoadAssetFileData(const char *assetPath);    /* original C++ signature */
def load_asset_file_data(asset_path: str) -> AssetFileData:
    """LoadAssetFileData(const char *assetPath)`
    Will load an entire asset file into memory. This works on all platforms,
    including android.
    You *have* to call FreeAssetFileData to free the memory, except if you use
    ImGui::GetIO().Fonts->AddFontFromMemoryTTF, which will take ownership of the
    data and free it for you.
    This function can be redirected with setLoadAssetFileDataFunction. If not redirected,
    it calls DefaultLoadAssetFileData.
    """
    pass

# void FreeAssetFileData(AssetFileData * assetFileData);    /* original C++ signature */
def free_asset_file_data(asset_file_data: AssetFileData) -> None:
    """FreeAssetFileData(AssetFileData *)
    Will free the memory.
    Note: "ImGui::GetIO().Fonts->AddFontFromMemoryTTF" takes ownership of the data
    and will free the memory for you.
    """
    pass

# @@md

# AssetFileData DefaultLoadAssetFileData(const char *assetPath);    /* original C++ signature */
def default_load_asset_file_data(asset_path: str) -> AssetFileData:
    """This function actually performs the asset load, as described in
    LoadAssetFileData
    """
    pass

# @@md#assetFileFullPath

# std::string AssetFileFullPath(const std::string& assetRelativeFilename,    /* original C++ signature */
#                               bool assertIfNotFound = true);
def asset_file_full_path(
    asset_relative_filename: str, assert_if_not_found: bool = True
) -> str:
    """`std::string AssetFileFullPath(const std::string& assetRelativeFilename)`
    will return the path to assets.

    This works under all platforms *except Android*
    For compatibility with Android and other platforms, prefer to use `LoadAssetFileData`
    whenever possible.
       * Under iOS it will give a path in the app bundle (/private/XXX/....)
       * Under emscripten, it will be stored in the virtual filesystem at "/"
       * Under Android, assetFileFullPath is *not* implemented, and will throw an error:
         assets can be compressed under android, and you can't use standard file operations!
         Use LoadAssetFileData instead
    """
    pass

# bool AssetExists(const std::string& assetRelativeFilename);    /* original C++ signature */
def asset_exists(asset_relative_filename: str) -> bool:
    """Returns True if this asset file exists"""
    pass

# void SetAssetsFolder(const std::string& folder);    /* original C++ signature */
@overload
def set_assets_folder(folder: str) -> None:
    """Sets the assets folder location
    (when using this, automatic assets installation on mobile platforms may not work)
    """
    pass

# @@md

# void SetAssetsFolder(const char* folder);    /* original C++ signature */
@overload
def set_assets_folder(folder: str) -> None:
    """Legacy API, kept for compatibility"""
    pass

# inline std::string assetFileFullPath(const std::string& assetRelativeFilename, bool assertIfNotFound = true)    /* original C++ signature */
#     { return AssetFileFullPath(assetRelativeFilename, assertIfNotFound); }
def asset_file_full_path(
    asset_relative_filename: str, assert_if_not_found: bool = True
) -> str:
    pass

# void overrideAssetsFolder(const char* folder);     /* original C++ signature */
def override_assets_folder(folder: str) -> None:
    """synonym of SetAssetsFolder"""
    pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/hello_imgui_error.h included by hello_imgui.h                              //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/hello_imgui_logger.h included by hello_imgui.h                             //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////
class LogLevel(enum.IntEnum):
    # Debug,    /* original C++ signature */
    debug = enum.auto()  # (= 0)
    # Info,    /* original C++ signature */
    info = enum.auto()  # (= 1)
    # Warning,    /* original C++ signature */
    warning = enum.auto()  # (= 2)
    # Error    /* original C++ signature */
    #     }
    error = enum.auto()  # (= 3)

# void Log(LogLevel level, char const* const format, ...);    /* original C++ signature */
def log(level: LogLevel, format: str) -> None:
    pass

# void LogClear();    /* original C++ signature */
def log_clear() -> None:
    pass

# void LogGui(ImVec2 size=ImVec2(0.f, 0.f));    /* original C++ signature */
# }
def log_gui(size: Optional[ImVec2Like] = None) -> None:
    """Python bindings defaults:
    If size is None, then its default value will be: ImVec2(0., 0.)
    """
    pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/image_from_asset.h included by hello_imgui.h                               //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# @@md#HelloImGui::ImageFromAsset

#
# Images are loaded when first displayed, and then cached
# (they will be freed just before the application exits).
#
# For example, given this files structure:
# ```
# ├── CMakeLists.txt
# ├── assets/
# │         └── my_image.jpg
# └── my_app.main.cpp
# ```
#
# then, you can display "my_image.jpg", using:
#
#    ```cpp
#    HelloImGui::ImageFromAsset("my_image.jpg");
#    ```

# void ImageFromAsset(const char *assetPath, const ImVec2& size = ImVec2(0, 0),    /* original C++ signature */
#                     const ImVec2& uv0 = ImVec2(0, 0), const ImVec2& uv1 = ImVec2(1,1));
def image_from_asset(
    asset_path: str,
    size: Optional[ImVec2Like] = None,
    uv0: Optional[ImVec2Like] = None,
    uv1: Optional[ImVec2Like] = None,
) -> None:
    """`HelloImGui::ImageFromAsset(const char *assetPath, size, ...)`:
     will display a static image from the assets.


    Python bindings defaults:
        If any of the params below is None, then its default value below will be used:
            * size: ImVec2(0, 0)
            * uv0: ImVec2(0, 0)
            * uv1: ImVec2(1,1)
    """
    pass

# void ImageFromAssetWithBg(const char *assetPath, const ImVec2& size = ImVec2(0, 0),    /* original C++ signature */
#             const ImVec2& uv0 = ImVec2(0, 0), const ImVec2& uv1 = ImVec2(1,1),
#             const ImVec4& tint_col = ImVec4(1,1,1,1),
#             const ImVec4& border_col = ImVec4(0,0,0,0));
def image_from_asset_with_bg(
    asset_path: str,
    size: Optional[ImVec2Like] = None,
    uv0: Optional[ImVec2Like] = None,
    uv1: Optional[ImVec2Like] = None,
    tint_col: Optional[ImVec4Like] = None,
    border_col: Optional[ImVec4Like] = None,
) -> None:
    """`HelloImGui::ImageFromAsset(const char *assetPath, size, ...)`:
     will display a static image from the assets, with a colored background and a border.


    Python bindings defaults:
        If any of the params below is None, then its default value below will be used:
            * size: ImVec2(0, 0)
            * uv0: ImVec2(0, 0)
            * uv1: ImVec2(1,1)
            * tint_col: ImVec4(1,1,1,1)
            * border_col: ImVec4(0,0,0,0)
    """
    pass

# bool ImageButtonFromAsset(const char *assetPath, const ImVec2& size = ImVec2(0, 0),    /* original C++ signature */
#                           const ImVec2& uv0 = ImVec2(0, 0),  const ImVec2& uv1 = ImVec2(1,1),
#                           int frame_padding = -1,
#                           const ImVec4& bg_col = ImVec4(0,0,0,0),
#                           const ImVec4& tint_col = ImVec4(1,1,1,1));
def image_button_from_asset(
    asset_path: str,
    size: Optional[ImVec2Like] = None,
    uv0: Optional[ImVec2Like] = None,
    uv1: Optional[ImVec2Like] = None,
    frame_padding: int = -1,
    bg_col: Optional[ImVec4Like] = None,
    tint_col: Optional[ImVec4Like] = None,
) -> bool:
    """`bool HelloImGui::ImageButtonFromAsset(const char *assetPath, size, ...)`:
     will display a button using an image from the assets.


    Python bindings defaults:
        If any of the params below is None, then its default value below will be used:
            * size: ImVec2(0, 0)
            * uv0: ImVec2(0, 0)
            * uv1: ImVec2(1,1)
            * bg_col: ImVec4(0,0,0,0)
            * tint_col: ImVec4(1,1,1,1)
    """
    pass

# ImTextureID ImTextureIdFromAsset(const char *assetPath);    /* original C++ signature */
def im_texture_id_from_asset(asset_path: str) -> ImTextureID:
    """`ImTextureID HelloImGui::ImTextureIdFromAsset(assetPath)`:
    will return a texture ID for an image loaded from the assets.
    """
    pass

# ImVec2 ImageSizeFromAsset(const char *assetPath);    /* original C++ signature */
def image_size_from_asset(asset_path: str) -> ImVec2:
    """`ImVec2 HelloImGui::ImageSizeFromAsset(assetPath)`:
    will return the size of an image loaded from the assets.
    """
    pass

class ImageAndSize:
    """`HelloImGui::ImageAndSize HelloImGui::ImageAndSizeFromAsset(assetPath)`:
    will return the texture ID and the size of an image loaded from the assets.
    """

    # ImTextureID textureId = ImTextureID(0);    /* original C++ signature */
    texture_id: ImTextureID = ImTextureID(0)
    # ImVec2 size = ImVec2(0.f, 0.f);    /* original C++ signature */
    size: ImVec2 = ImVec2(0.0, 0.0)
    # ImageAndSize(ImTextureID textureId = ImTextureID(0), ImVec2 size = ImVec2(0.f, 0.f));    /* original C++ signature */
    def __init__(
        self,
        texture_id: Optional[ImTextureID] = None,
        size: Optional[ImVec2Like] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * textureId: ImTextureID(0)
                * size: ImVec2(0., 0.)
        """
        pass

# ImageAndSize ImageAndSizeFromAsset(const char *assetPath);    /* original C++ signature */
def image_and_size_from_asset(asset_path: str) -> ImageAndSize:
    pass

# ImVec2 ImageProportionalSize(const ImVec2& askedSize, const ImVec2& imageSize);    /* original C++ signature */
def image_proportional_size(asked_size: ImVec2Like, image_size: ImVec2Like) -> ImVec2:
    """`ImVec2 HelloImGui::ImageProportionalSize(askedSize, imageSize)`:
    will return the displayed size of an image.
       - if askedSize.x or askedSize.y is 0, then the corresponding dimension
         will be computed from the image size, keeping the aspect ratio.
       - if askedSize.x>0 and askedSize.y> 0, then the image will be scaled to fit
         exactly the askedSize, thus potentially changing the aspect ratio.
    Note: this function is used internally by ImageFromAsset and ImageButtonFromAsset,
          so you don't need to call it directly.
    """
    pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/imgui_theme.h included by hello_imgui.h                                    //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

#
# Theme tweak utilities for ImGui.
# Reuse and adaptation of imgui_theme.h and imgui_theme.cpp file is granted for other projects,
# provided the origin of those files is stated in the copied version
# Some themes were adapted by themes posted by ImGui users at https://github.com/ocornut/imgui/issues/707
#

class ImGuiTheme_(enum.IntEnum):
    # ImGuiTheme_ImGuiColorsClassic = 0,    /* original C++ signature */
    imgui_colors_classic = enum.auto()  # (= 0)
    # ImGuiTheme_ImGuiColorsDark,    /* original C++ signature */
    imgui_colors_dark = enum.auto()  # (= 1)
    # ImGuiTheme_ImGuiColorsLight,    /* original C++ signature */
    imgui_colors_light = enum.auto()  # (= 2)
    # ImGuiTheme_MaterialFlat,    /* original C++ signature */
    material_flat = enum.auto()  # (= 3)
    # ImGuiTheme_PhotoshopStyle,    /* original C++ signature */
    photoshop_style = enum.auto()  # (= 4)
    # ImGuiTheme_GrayVariations,    /* original C++ signature */
    gray_variations = enum.auto()  # (= 5)
    # ImGuiTheme_GrayVariations_Darker,    /* original C++ signature */
    gray_variations_darker = enum.auto()  # (= 6)
    # ImGuiTheme_MicrosoftStyle,    /* original C++ signature */
    microsoft_style = enum.auto()  # (= 7)
    # ImGuiTheme_Cherry,    /* original C++ signature */
    cherry = enum.auto()  # (= 8)
    # ImGuiTheme_Darcula,    /* original C++ signature */
    darcula = enum.auto()  # (= 9)
    # ImGuiTheme_DarculaDarker,    /* original C++ signature */
    darcula_darker = enum.auto()  # (= 10)
    # ImGuiTheme_LightRounded,    /* original C++ signature */
    light_rounded = enum.auto()  # (= 11)
    # ImGuiTheme_SoDark_AccentBlue,    /* original C++ signature */
    so_dark_accent_blue = enum.auto()  # (= 12)
    # ImGuiTheme_SoDark_AccentYellow,    /* original C++ signature */
    so_dark_accent_yellow = enum.auto()  # (= 13)
    # ImGuiTheme_SoDark_AccentRed,    /* original C++ signature */
    so_dark_accent_red = enum.auto()  # (= 14)
    # ImGuiTheme_BlackIsBlack,    /* original C++ signature */
    black_is_black = enum.auto()  # (= 15)
    # ImGuiTheme_WhiteIsWhite,    /* original C++ signature */
    white_is_white = enum.auto()  # (= 16)
    # ImGuiTheme_Count    /* original C++ signature */
    #     }
    count = enum.auto()  # (= 17)

# const char* ImGuiTheme_Name(ImGuiTheme_ theme);    /* original C++ signature */
def imgui_theme_name(theme: ImGuiTheme_) -> str:
    pass

# ImGuiTheme_ ImGuiTheme_FromName(const char* themeName);    /* original C++ signature */
def imgui_theme_from_name(theme_name: str) -> ImGuiTheme_:
    pass

# ImGuiStyle ThemeToStyle(ImGuiTheme_ theme);    /* original C++ signature */
def theme_to_style(theme: ImGuiTheme_) -> ImGuiStyle:
    pass

# void ApplyTheme(ImGuiTheme_ theme);    /* original C++ signature */
def apply_theme(theme: ImGuiTheme_) -> None:
    pass

class ImGuiThemeTweaks:
    # float Rounding = -1.f;    /* original C++ signature */
    # Common rounding for widgets. If < 0, this is ignored.
    rounding: float = -1.0
    # float RoundingScrollbarRatio = 4.f;    /* original C++ signature */
    # If rounding is applied, scrollbar rounding needs to be adjusted to be visually pleasing in conjunction with other widgets roundings. Only applied if Rounding > 0.)
    rounding_scrollbar_ratio: float = 4.0
    # float AlphaMultiplier = -1.f;    /* original C++ signature */
    # Change the alpha that will be applied to windows, popups, etc. If < 0, this is ignored.
    alpha_multiplier: float = -1.0

    # float Hue = -1.f;    /* original C++ signature */
    #
    # HSV Color tweaks
    #
    # Change the hue of all widgets (gray widgets will remain gray, since their saturation is zero). If < 0, this is ignored.
    hue: float = -1.0
    # float SaturationMultiplier = -1.f;    /* original C++ signature */
    # Multiply the saturation of all widgets (gray widgets will remain gray, since their saturation is zero). If < 0, this is ignored.
    saturation_multiplier: float = -1.0
    # float ValueMultiplierFront = -1.f;    /* original C++ signature */
    # Multiply the value (luminance) of all front widgets. If < 0, this is ignored.
    value_multiplier_front: float = -1.0
    # float ValueMultiplierBg = -1.f;    /* original C++ signature */
    # Multiply the value (luminance) of all backgrounds. If < 0, this is ignored.
    value_multiplier_bg: float = -1.0
    # float ValueMultiplierText = -1.f;    /* original C++ signature */
    # Multiply the value (luminance) of text. If < 0, this is ignored.
    value_multiplier_text: float = -1.0
    # float ValueMultiplierFrameBg = -1.f;    /* original C++ signature */
    # Multiply the value (luminance) of FrameBg. If < 0, this is ignored.
    # (Background of checkbox, radio button, plot, slider, text input)
    value_multiplier_frame_bg: float = -1.0

    # ImGuiThemeTweaks() {}    /* original C++ signature */
    def __init__(self) -> None:
        pass

class ImGuiTweakedTheme:
    # ImGuiTheme_ Theme = ImGuiTheme_DarculaDarker;    /* original C++ signature */
    theme: ImGuiTheme_ = ImGuiTheme_.darcula_darker
    # ImGuiThemeTweaks Tweaks = ImGuiThemeTweaks();    /* original C++ signature */
    tweaks: ImGuiThemeTweaks = ImGuiThemeTweaks()

    # ImGuiTweakedTheme(ImGuiTheme_ theme = ImGuiTheme_DarculaDarker, const ImGuiThemeTweaks& tweaks = ImGuiThemeTweaks())    /* original C++ signature */
    #             : Theme(theme), Tweaks(tweaks) {}
    def __init__(
        self,
        theme: ImGuiTheme_ = ImGuiTheme_.darcula_darker,
        tweaks: Optional[ImGuiThemeTweaks] = None,
    ) -> None:
        """Python bindings defaults:
        If tweaks is None, then its default value will be: ImGuiThemeTweaks()
        """
        pass

# ImGuiStyle TweakedThemeThemeToStyle(const ImGuiTweakedTheme& tweaked_theme);    /* original C++ signature */
def tweaked_theme_theme_to_style(tweaked_theme: ImGuiTweakedTheme) -> ImGuiStyle:
    pass

# void ApplyTweakedTheme(const ImGuiTweakedTheme& tweaked_theme);    /* original C++ signature */
def apply_tweaked_theme(tweaked_theme: ImGuiTweakedTheme) -> None:
    pass

# PushTweakedTheme() / PopTweakedTheme()
# Push and pop a tweaked theme
#
# Note: If you want the theme to apply globally to a window, you need to apply it
#       *before* calling ImGui::Begin
#
#     For example, within Hello ImGui, given a dockable window, you should set this option:
#        myDockableWindow.callBeginEnd = False;
#     And then:
#     - call ImGuiTheme::PushTweakedTheme
#     - call ImGui::Begin
#     - display your content
#     - call ImGui::End
#     - call ImGuiTheme::PopTweakedTheme
#
# See demo inside src/hello_imgui_demos/hello_imgui_demodocking/hello_imgui_demodocking.main.cpp:
#     look at `GuiWindowAlternativeTheme()`
# void PushTweakedTheme(const ImGuiTweakedTheme& tweaked_theme);    /* original C++ signature */
def push_tweaked_theme(tweaked_theme: ImGuiTweakedTheme) -> None:
    pass

# void PopTweakedTheme();    /* original C++ signature */
def pop_tweaked_theme() -> None:
    pass

# bool ShowThemeTweakGui(ImGuiTweakedTheme *tweaked_theme);    /* original C++ signature */
def show_theme_tweak_gui(tweaked_theme: ImGuiTweakedTheme) -> bool:
    """Show the theme selection listbox, the theme tweak widgets, as well as ImGui::ShowStyleEditor. Returns True if modified (Warning, when using ShowStyleEditor, no info about modification is transmitted)"""
    pass

# Some tweakable themes
# ImGuiStyle SoDark(float hue);    /* original C++ signature */
def so_dark(hue: float) -> ImGuiStyle:
    pass

# ImGuiStyle ShadesOfGray(float rounding=0.f, float value_multiplier_front=1.f, float value_multiplier_bg=1.f);    /* original C++ signature */
def shades_of_gray(
    rounding: float = 0.0,
    value_multiplier_front: float = 1.0,
    value_multiplier_bg: float = 1.0,
) -> ImGuiStyle:
    pass

# ImGuiStyle Darcula(    /* original C++ signature */
#         float rounding=1.f,
#         float hue=-1.f,
#         float saturation_multiplier=1.f,
#         float value_multiplier_front=1.f,
#         float value_multiplier_bg=1.f,
#         float alpha_bg_transparency=1.f
#     );
def darcula(
    rounding: float = 1.0,
    hue: float = -1.0,
    saturation_multiplier: float = 1.0,
    value_multiplier_front: float = 1.0,
    value_multiplier_bg: float = 1.0,
    alpha_bg_transparency: float = 1.0,
) -> ImGuiStyle:
    pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/hello_imgui_theme.h included by hello_imgui.h                              //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# void ShowThemeTweakGuiWindow(bool* p_open = nullptr);    /* original C++ signature */
# }
def show_theme_tweak_gui_window(p_open: Optional[bool] = None) -> Optional[bool]:
    pass

# @@md#Fonts

# When loading fonts, use
#          HelloImGui::LoadFont(..)
#      or
#      	HelloImGui::LoadDpiResponsiveFont()
#
# Use these functions instead of ImGui::GetIO().Fonts->AddFontFromFileTTF(),
# because they will automatically adjust the font size to account for HighDPI,
# and will help you to get consistent font size across different OSes.

class FontLoadingParams:
    """
    Font loading parameters: several options are available (color, merging, range, ...)
    """

    # bool adjustSizeToDpi = true;    /* original C++ signature */
    # if True, the font size will be adjusted automatically to account for HighDPI
    #
    adjust_size_to_dpi: bool = True

    # bool mergeToLastFont = false;    /* original C++ signature */
    # if True, the font will be merged to the last font
    merge_to_last_font: bool = False

    # bool loadColor = false;    /* original C++ signature */
    # if True, the font will be loaded using colors
    # (requires freetype, enabled by IMGUI_ENABLE_FREETYPE)
    load_color: bool = False

    # bool insideAssets = true;    /* original C++ signature */
    # if True, the font will be loaded using HelloImGui asset system.
    # Otherwise, it will be loaded from the filesystem
    inside_assets: bool = True

    # ImFontConfig fontConfig = ImFontConfig();    /* original C++ signature */
    # ImGui native font config to use
    font_config: ImFontConfig = ImFontConfig()
    # FontLoadingParams(bool adjustSizeToDpi = true, bool mergeToLastFont = false, bool loadColor = false, bool insideAssets = true, ImFontConfig fontConfig = ImFontConfig());    /* original C++ signature */
    def __init__(
        self,
        adjust_size_to_dpi: bool = True,
        merge_to_last_font: bool = False,
        load_color: bool = False,
        inside_assets: bool = True,
        font_config: Optional[ImFontConfig] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If fontConfig is None, then its default value will be: ImFontConfig()
        """
        pass

# ImFont* LoadFont(    /* original C++ signature */
#         const std::string & fontFilename, float fontSize,
#         const FontLoadingParams & params = __srcmlcpp_brace_init__());
def load_font(
    font_filename: str, font_size: float, params: Optional[FontLoadingParams] = None
) -> ImFont:
    """Python bindings defaults:
    If params is None, then its default value will be: initialized with default value
    """
    pass

# ImFont* LoadFontTTF(    /* original C++ signature */
#         const std::string & fontFilename,
#         float fontSize,
#         ImFontConfig config = ImFontConfig()
#     );
def load_font_ttf(
    font_filename: str, font_size: float, config: Optional[ImFontConfig] = None
) -> ImFont:
    """Python bindings defaults:
    If config is None, then its default value will be: ImFontConfig()
    """
    pass

# ImFont* LoadFontTTF_WithFontAwesomeIcons(    /* original C++ signature */
#         const std::string & fontFilename,
#         float fontSize,
#         ImFontConfig configFont = ImFontConfig()
#     );
def load_font_ttf_with_font_awesome_icons(
    font_filename: str, font_size: float, config_font: Optional[ImFontConfig] = None
) -> ImFont:
    """Python bindings defaults:
    If configFont is None, then its default value will be: ImFontConfig()
    """
    pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/runner_params.h included by hello_imgui.h                                  //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/app_window_params.h included by hello_imgui/runner_params.h                //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/screen_bounds.h included by hello_imgui/app_window_params.h                //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# Note: note related to DPI and high resolution screens:
# ScreenPosition and ScreenSize are in "Screen Coordinates":
# Screen coordinates *might* differ from real pixel on high dpi screens; but this depends on the OS.
# - For example, on apple a retina screenpixel size 3456x2052 might be seen as 1728x1026 in screen coordinates
# - Under windows, ScreenCoordinates correspond to pixels, even on high density screens

class ScreenBounds:
    # ScreenPosition position = DefaultScreenPosition;    /* original C++ signature */
    position: ScreenPosition = DefaultScreenPosition
    # ScreenSize size = DefaultWindowSize;    /* original C++ signature */
    size: ScreenSize = DefaultWindowSize

    # ScreenPosition TopLeftCorner() const{ return position; }    /* original C++ signature */
    def top_left_corner(self) -> ScreenPosition:
        pass
    # ScreenPosition BottomRightCorner() const{ return { position[0] + size[0], position[1] + size[1] }; }    /* original C++ signature */
    def bottom_right_corner(self) -> ScreenPosition:
        pass
    # ScreenPosition Center() const{ return { position[0] + size[0] / 2, position[1] + size[1] / 2 }; }    /* original C++ signature */
    def center(self) -> ScreenPosition:
        pass
    # bool Contains(ScreenPosition pixel) const;    /* original C++ signature */
    def contains(self, pixel: ScreenPosition) -> bool:
        pass
    # ScreenPosition WinPositionCentered(ScreenSize windowSize) const;    /* original C++ signature */
    def win_position_centered(self, window_size: ScreenSize) -> ScreenPosition:
        pass
    # int DistanceFromPixel(ScreenPosition point) const;    /* original C++ signature */
    def distance_from_pixel(self, point: ScreenPosition) -> int:
        pass
    # ScreenBounds EnsureWindowFitsThisMonitor(ScreenBounds windowBoundsOriginal) const;    /* original C++ signature */
    def ensure_window_fits_this_monitor(
        self, window_bounds_original: ScreenBounds
    ) -> ScreenBounds:
        pass
    # bool operator==(const ScreenBounds& other) const;    /* original C++ signature */
    def __eq__(self, other: ScreenBounds) -> bool:
        pass
    # ScreenBounds(ScreenPosition position = DefaultScreenPosition, ScreenSize size = DefaultWindowSize);    /* original C++ signature */
    def __init__(
        self,
        position: Optional[ScreenPosition] = None,
        size: Optional[ScreenSize] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * position: DefaultScreenPosition
                * size: DefaultWindowSize
        """
        pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/app_window_params.h continued                                              //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

class FullScreenMode(enum.IntEnum):
    # NoFullScreen,    /* original C++ signature */
    no_full_screen = enum.auto()  # (= 0)
    # FullScreen,                        /* original C++ signature */
    full_screen = enum.auto()  # (= 1)  # Full screen with specified resolution
    # FullScreenDesktopResolution,       /* original C++ signature */
    full_screen_desktop_resolution = (
        enum.auto()
    )  # (= 2)  # Full screen with current desktop mode & resolution
    # FullMonitorWorkArea                /* original C++ signature */
    full_monitor_work_area = (
        enum.auto()
    )  # (= 3)  # Fake full screen, maximized window on the selected monitor

class WindowSizeState(enum.IntEnum):
    # Standard,    /* original C++ signature */
    standard = enum.auto()  # (= 0)
    # Minimized,    /* original C++ signature */
    minimized = enum.auto()  # (= 1)
    # Maximized    /* original C++ signature */
    # }
    maximized = enum.auto()  # (= 2)

class WindowPositionMode(enum.IntEnum):
    # OsDefault,    /* original C++ signature */
    os_default = enum.auto()  # (= 0)
    # MonitorCenter,    /* original C++ signature */
    monitor_center = enum.auto()  # (= 1)
    # FromCoords,    /* original C++ signature */
    # }
    from_coords = enum.auto()  # (= 2)

class EmscriptenKeyboardElement(enum.IntEnum):
    # Window,    /* original C++ signature */
    window = enum.auto()  # (= 0)
    # Document,    /* original C++ signature */
    document = enum.auto()  # (= 1)
    # Screen,    /* original C++ signature */
    screen = enum.auto()  # (= 2)
    # Canvas,    /* original C++ signature */
    canvas = enum.auto()  # (= 3)
    # Default    /* original C++ signature */
    # }
    default = enum.auto()  # (= 4)

class WindowSizeMeasureMode(enum.IntEnum):
    # ScreenCoords,    /* original C++ signature */
    # ScreenCoords: measure window size in screen coords.
    #     Note: screen coordinates *might* differ from real pixel on high dpi screens; but this depends on the OS.
    #         - For example, on apple a retina screenpixel size 3456x2052 might be seen as 1728x1026 in screen
    #           coordinates
    #         - Under windows, and if the application is DPI aware, ScreenCoordinates correspond to real pixels,
    #           even on high density screens
    screen_coords = enum.auto()  # (= 0)

    # RelativeTo96Ppi    /* original C++ signature */
    # }
    # RelativeTo96Ppi enables to give screen size that are independant from the screen density.
    # For example, a window size expressed as 800x600 will correspond to a size
    #    800x600 (in screen coords) if the monitor dpi is 96
    #    1600x120 (in screen coords) if the monitor dpi is 192
    relative_to96_ppi = enum.auto()  # (= 1)

class WindowGeometry:
    """@@md#WindowGeometry

    WindowGeometry is a struct that defines the window geometry.
    """

    # --------------- Window Size ------------------

    # ScreenSize size = DefaultWindowSize;    /* original C++ signature */
    # Size of the application window
    # used if fullScreenMode==NoFullScreen and sizeAuto==False. Default=(800, 600)
    # The size will be handled as if it was specified for a 96PPI screen
    # (i.e. a given size will correspond to the same physical size on different screens, whatever their DPI)
    size: ScreenSize = DefaultWindowSize

    # bool sizeAuto = false;    /* original C++ signature */
    # If sizeAuto=True, adapt the app window size to the presented widgets.
    # After the first frame was displayed, HelloImGui will measure its size, and the
    # application window will be resized.
    # As a consequence, the application window may change between the 1st and 2nd frame.
    # If True, adapt the app window size to the presented widgets. This is done at startup
    size_auto: bool = False

    # WindowSizeState windowSizeState = WindowSizeState::Standard;    /* original C++ signature */
    # `windowSizeState`: _WindowSizeState, default=Standard_
    #  You can choose between several window size states:
    #      Standard,
    #      Minimized,
    #      Maximized
    window_size_state: WindowSizeState = WindowSizeState.standard

    # WindowSizeMeasureMode windowSizeMeasureMode = WindowSizeMeasureMode::RelativeTo96Ppi;    /* original C++ signature */
    # `windowSizeMeasureMode`: _WindowSizeMeasureMode_, default=RelativeTo96Ppi
    # Define how the window size is specified:
    #      * RelativeTo96Ppi enables to give a screen size whose physical result
    #      (in millimeters) is independent of the screen density.
    #         For example, a window size expressed as 800x600 will correspond to a size
    #            - 800x600 (in screen coords) if the monitor dpi is 96
    #            - 1600x120 (in screen coords) if the monitor dpi is 192
    #          (this works with Glfw. With SDL, it only works under windows)
    #      * ScreenCoords: measure window size in screen coords
    #        (Note: screen coordinates might differ from real pixels on high dpi screen)
    window_size_measure_mode: WindowSizeMeasureMode = (
        WindowSizeMeasureMode.relative_to96_ppi
    )

    # --------------- Position ------------------

    # WindowPositionMode positionMode = WindowPositionMode::OsDefault;    /* original C++ signature */
    # `positionMode`: you can choose between several window position modes:
    #      OsDefault,
    #      MonitorCenter,
    #      FromCoords,
    position_mode: WindowPositionMode = WindowPositionMode.os_default

    # ScreenPosition position = DefaultScreenPosition;    /* original C++ signature */
    # `position`: used if windowPositionMode==FromCoords, default=(40, 40)
    position: ScreenPosition = DefaultScreenPosition

    # int monitorIdx = 0;    /* original C++ signature */
    # `monitorIdx`: index of the monitor to use, default=0
    #  used if positionMode==MonitorCenter or if fullScreenMode!=NoFullScreen
    monitor_idx: int = 0

    # --------------- Full screen ------------------

    # FullScreenMode fullScreenMode = FullScreenMode::NoFullScreen;    /* original C++ signature */
    # `fullScreenMode`: you can choose between several full screen modes:
    #      NoFullScreen,
    #      FullScreen,                  // Full screen with specified resolution
    #      FullScreenDesktopResolution, // Full screen with current desktop mode & resolution
    #      FullMonitorWorkArea          // Fake full screen (maximized window) on the selected monitor
    full_screen_mode: FullScreenMode = FullScreenMode.no_full_screen

    # --------------- Auto Resize ------------------

    # bool resizeAppWindowAtNextFrame = false;    /* original C++ signature */
    # `resizeAppWindowAtNextFrame`: _bool_, default=False;
    #  If you set this to flag to True at any point during the execution, the application
    #  window will then try to resize based on its content on the next displayed frame,
    #  and this flag will subsequently be set to False.
    #  Example:
    #   ```cpp
    # // Will resize the app window at next displayed frame
    #   HelloImGui::GetRunnerParams()->appWindowParams.windowGeometry.resizeAppWindowAtNextFrame = True;
    #   ```
    #  Note: this flag is intended to be used during execution, not at startup
    #  (use sizeAuto at startup).
    resize_app_window_at_next_frame: bool = False
    # WindowGeometry(ScreenSize size = DefaultWindowSize, bool sizeAuto = false, WindowSizeState windowSizeState = WindowSizeState::Standard, WindowSizeMeasureMode windowSizeMeasureMode = WindowSizeMeasureMode::RelativeTo96Ppi, WindowPositionMode positionMode = WindowPositionMode::OsDefault, ScreenPosition position = DefaultScreenPosition, int monitorIdx = 0, FullScreenMode fullScreenMode = FullScreenMode::NoFullScreen, bool resizeAppWindowAtNextFrame = false);    /* original C++ signature */
    def __init__(
        self,
        size: Optional[ScreenSize] = None,
        size_auto: bool = False,
        window_size_state: WindowSizeState = WindowSizeState.standard,
        window_size_measure_mode: WindowSizeMeasureMode = WindowSizeMeasureMode.relative_to96_ppi,
        position_mode: WindowPositionMode = WindowPositionMode.os_default,
        position: Optional[ScreenPosition] = None,
        monitor_idx: int = 0,
        full_screen_mode: FullScreenMode = FullScreenMode.no_full_screen,
        resize_app_window_at_next_frame: bool = False,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * size: DefaultWindowSize
                * position: DefaultScreenPosition
        """
        pass

# @@md

class EdgeInsets:
    """If there is a notch on the iPhone, you should not display inside these insets"""

    # double top = 0.;    /* original C++ signature */
    top: float = 0.0  # Typically around 47
    # double left = 0.;    /* original C++ signature */
    left: float = 0.0  # Typically 0
    # double bottom = 0.;    /* original C++ signature */
    bottom: float = 0.0  # Typically around 34
    # double right = 0.;    /* original C++ signature */
    right: float = 0.0  # Typically 0
    # EdgeInsets(double top = 0., double left = 0., double bottom = 0., double right = 0.);    /* original C++ signature */
    def __init__(
        self,
        top: float = 0.0,
        left: float = 0.0,
        bottom: float = 0.0,
        right: float = 0.0,
    ) -> None:
        """Auto-generated default constructor with named params"""
        pass

class AppWindowParams:
    """@@md#AppWindowParams

     AppWindowParams is a struct that defines the application window display params.
    See https://raw.githubusercontent.com/pthom/hello_imgui/master/src/hello_imgui/doc_src/hello_imgui_diagram.jpg
     for details.
    """

    # --------------- Standard params ------------------

    # std::string windowTitle;    /* original C++ signature */
    # `windowTitle`: _string, default=""_. Title of the application window
    window_title: str

    # WindowGeometry windowGeometry;    /* original C++ signature */
    # `windowGeometry`: _WindowGeometry_
    #  Enables to precisely set the window geometry (position, monitor, size,
    #  full screen, fake full screen, etc.)
    #   _Note: on a mobile device, the application will always be full screen._
    window_geometry: WindowGeometry

    # bool restorePreviousGeometry = false;    /* original C++ signature */
    # `restorePreviousGeometry`: _bool, default=false_.
    # If True, then save & restore windowGeometry from last run (the geometry
    # will be written in imgui_app_window.ini)
    restore_previous_geometry: bool = False

    # bool resizable = true;    /* original C++ signature */
    # `resizable`: _bool, default = false_. Should the window be resizable.
    # This is taken into account at creation.
    resizable: bool = True
    # bool hidden = false;    /* original C++ signature */
    # `hidden`: _bool, default = false_. Should the window be hidden.
    # This is taken into account dynamically (you can show/hide the window with this).
    # Full screen windows cannot be hidden.
    hidden: bool = False

    # --------------- Borderless window params ------------------

    # bool   borderless = false;    /* original C++ signature */
    # `borderless`: _bool, default = false_. Should the window have borders.
    # This is taken into account at creation.
    borderless: bool = False
    # bool   borderlessMovable = true;    /* original C++ signature */
    # `borderlessMovable`: if the window is borderless, should it be movable.
    #   If so, a drag zone is displayed at the top of the window when the mouse is over it.
    borderless_movable: bool = True
    # bool   borderlessResizable = true;    /* original C++ signature */
    # `borderlessResizable`: if the window is borderless, should it be resizable.
    #  If so, a drag zone is displayed at the bottom-right of the window
    #  when the mouse is over it.
    borderless_resizable: bool = True
    # bool   borderlessClosable = true;    /* original C++ signature */
    # `borderlessClosable`: if the window is borderless, should it have a close button.
    #  If so, a close button is displayed at the top-right of the window
    #  when the mouse is over it.
    borderless_closable: bool = True
    # ImVec4 borderlessHighlightColor = ImVec4(0.2f, 0.4f, 1.f, 0.3f);    /* original C++ signature */
    # `borderlessHighlightColor`:
    #   Color of the highlight displayed on resize/move zones.
    #   If borderlessHighlightColor.w==0, then the highlightColor will be automatically
    #   set to ImGui::GetColorU32(ImGuiCol_TitleBgActive, 0.6)
    borderless_highlight_color: ImVec4 = ImVec4(0.2, 0.4, 1.0, 0.3)

    # --------------- iOS Notch ------------------

    # EdgeInsets edgeInsets;    /* original C++ signature */
    # `edgeInsets`: _EdgeInsets_. iOS only, out values filled by HelloImGui.
    # If there is a notch on the iPhone, you should not display inside these insets.
    # HelloImGui handles this automatically, if handleEdgeInsets is True and
    # if runnerParams.imGuiWindowParams.defaultImGuiWindowType is not NoDefaultWindow.
    # (warning, these values are updated only after a few frames,
    #  they are typically 0 for the first 4 frames)
    edge_insets: EdgeInsets
    # bool       handleEdgeInsets = true;    /* original C++ signature */
    # `handleEdgeInsets`: _bool, default = true_. iOS only.
    # If True, HelloImGui will handle the edgeInsets on iOS.
    handle_edge_insets: bool = True

    # EmscriptenKeyboardElement emscriptenKeyboardElement = EmscriptenKeyboardElement::Default;    /* original C++ signature */
    # --------------- Emscripten ------------------
    # `emscriptenKeyboardElement`: _EmscriptenKeyboardElement, default=Default_. HTML element in which SDL will capture the keyboard events.
    # (For Emscripten only)
    # Choose between: Window, Document, Screen, Canvas, Default.
    # If Default:
    # - the default SDL behavior is used, which is to capture the keyboard events for the window,
    #   if no previous hint was set in the javascript code.
    # - under Pyodide, the default behavior is to capture the keyboard events for the canvas.
    emscripten_keyboard_element: EmscriptenKeyboardElement = (
        EmscriptenKeyboardElement.default
    )

    # bool repaintDuringResize_GotchaReentrantRepaint = false;    /* original C++ signature */
    # ----------------- repaint the window during resize -----------------
    # Very advanced and reserved for advanced C++ users.
    # If you set this to True, the window will be repainted during resize.
    # Do read https://github.com/pthom/hello_imgui/issues/112 for info about the possible gotchas
    # (This API is not stable, as the name suggests, and this is not supported)
    repaint_during_resize_gotcha_reentrant_repaint: bool = False
    # AppWindowParams(std::string windowTitle = std::string(), WindowGeometry windowGeometry = WindowGeometry(), bool restorePreviousGeometry = false, bool resizable = true, bool hidden = false, bool borderless = false, bool borderlessMovable = true, bool borderlessResizable = true, bool borderlessClosable = true, ImVec4 borderlessHighlightColor = ImVec4(0.2f, 0.4f, 1.f, 0.3f), EdgeInsets edgeInsets = EdgeInsets(), bool handleEdgeInsets = true, EmscriptenKeyboardElement emscriptenKeyboardElement = EmscriptenKeyboardElement::Default, bool repaintDuringResize_GotchaReentrantRepaint = false);    /* original C++ signature */
    def __init__(
        self,
        window_title: str = "",
        window_geometry: Optional[WindowGeometry] = None,
        restore_previous_geometry: bool = False,
        resizable: bool = True,
        hidden: bool = False,
        borderless: bool = False,
        borderless_movable: bool = True,
        borderless_resizable: bool = True,
        borderless_closable: bool = True,
        borderless_highlight_color: Optional[ImVec4Like] = None,
        edge_insets: Optional[EdgeInsets] = None,
        handle_edge_insets: bool = True,
        emscripten_keyboard_element: EmscriptenKeyboardElement = EmscriptenKeyboardElement.default,
        repaint_during_resize_gotcha_reentrant_repaint: bool = False,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * windowGeometry: WindowGeometry()
                * borderlessHighlightColor: ImVec4(0.2, 0.4, 1., 0.3)
                * edgeInsets: EdgeInsets()
        """
        pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/imgui_window_params.h included by hello_imgui/runner_params.h              //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# @@md#DefaultImGuiWindowType

class DefaultImGuiWindowType(enum.IntEnum):
    """`DefaultImGuiWindowType` is an enum class that defines whether a full screen background
    window is provided or not
    """

    # ProvideFullScreenWindow,    /* original C++ signature */
    # `ProvideFullScreenWindow`: a full window is provided in the background
    provide_full_screen_window = enum.auto()  # (= 0)
    # ProvideFullScreenDockSpace,    /* original C++ signature */
    # `ProvideFullScreenDockSpace`: a full screen dockspace is provided in the background
    provide_full_screen_dock_space = enum.auto()  # (= 1)
    # NoDefaultWindow    /* original C++ signature */
    # }
    # `NoDefaultWindow`: No default window is provided
    # (except for ImGui's default "debug" window)
    no_default_window = enum.auto()  # (= 2)

# @@md

# @@md#ImGuiWindowParams

class ImGuiWindowParams:
    """`ImGuiWindowParams` is a struct that defines the ImGui inner windows params
    These settings affect the imgui inner windows inside the application window.
    In order to change the application window settings, change the `AppWindowsParams`
    """

    # ------------ Main Options  -------------------------------------------------------

    # DefaultImGuiWindowType defaultImGuiWindowType =    /* original C++ signature */
    #         DefaultImGuiWindowType::ProvideFullScreenWindow;
    # defaultImGuiWindowType: (enum DefaultImGuiWindowType)
    # Choose between:
    #    - ProvideFullScreenWindow (default)
    #      a full window is provided in the background
    #      You can still add windows on top of it, since the Z-order
    #      of this background window is always behind
    #    - ProvideFullScreenDockSpace:
    #      a full screen dockspace is provided in the background
    #      (use this if you intend to use docking)
    #    - NoDefaultWindow:
    #      no default window is provided
    default_imgui_window_type: DefaultImGuiWindowType = (
        DefaultImGuiWindowType.provide_full_screen_window
    )

    # bool enableViewports = false;    /* original C++ signature */
    # enableViewports: Enable multiple viewports (i.e. multiple native windows)
    # If True, you can drag windows outside the main window,
    # in order to put their content into new native windows.
    enable_viewports: bool = False

    # bool configWindowsMoveFromTitleBarOnly = true;    /* original C++ signature */
    # Make windows only movable from the title bar
    config_windows_move_from_title_bar_only: bool = True

    # ------------ Menus & Status bar --------------------------------------------------

    # std::string menuAppTitle = "";    /* original C++ signature */
    # Set the title of the App menu. If empty, the menu name will use
    # the "windowTitle" from AppWindowParams//
    menu_app_title: str = ""

    # bool showMenuBar = false;    /* original C++ signature */
    # Show Menu bar on top of imgui main window.
    # In order to fully customize the menu, set showMenuBar to True, and set showMenu_App
    # and showMenu_View params to False. Then, implement the callback
    # `RunnerParams.callbacks.ShowMenus`
    # which can optionally call `HelloImGui::ShowViewMenu` and `HelloImGui::ShowAppMenu`.
    show_menu_bar: bool = False

    # bool showMenu_App = true;    /* original C++ signature */
    #  If menu bar is shown, include or not the default app menu
    show_menu_app: bool = True

    # bool showMenu_App_Quit = true;    /* original C++ signature */
    # Include or not a "Quit" item in the default app menu.
    # Set this to False if you intend to provide your own quit callback
    # with possible user confirmation
    # (and implement it inside RunnerCallbacks.ShowAppMenuItems)
    show_menu_app_quit: bool = True

    # bool showMenu_View = true;    /* original C++ signature */
    # If menu bar is shown, include or not the default _View_ menu, that enables
    # to change the layout and set the docked windows and status bar visibility)
    show_menu_view: bool = True

    # bool showMenu_View_Themes = true;    /* original C++ signature */
    # Show theme selection in view menu
    show_menu_view_themes: bool = True
    # bool rememberTheme = true;    /* original C++ signature */
    # `rememberTheme`: _bool, default=true_. Remember selected theme
    remember_theme: bool = True

    # bool showStatusBar = false;    /* original C++ signature */
    # Flag that enable to show a Status bar at the bottom. You can customize
    # the status bar via RunnerCallbacks.ShowStatus()
    show_status_bar: bool = False

    # bool showStatus_Fps = true;    /* original C++ signature */
    # If set, display the FPS in the status bar.
    show_status_fps: bool = True
    # bool rememberStatusBarSettings = true;    /* original C++ signature */
    # If set, showStatusBar and showStatus_Fps are stored in the application settings.
    remember_status_bar_settings: bool = True

    # ------------ Change the dockspace or background window size -----------------------

    # If defaultImGuiWindowType = ProvideFullScreenWindow or ProvideFullScreenDockSpace,
    # you can set the position and size of the background window:
    #    - fullScreenWindow_MarginTopLeft is the window position
    #    - fullScreenWindow_MarginBottomRight is the margin between
    #      the "application window" bottom right corner
    #      and the "imgui background window" bottom right corner
    # Important note:
    #     In order to be Dpi aware, those sizes are in *em units*, not in pixels,
    #     i.e. in multiples of the font size! (See HelloImGui::EmToVec2)
    # ImVec2 fullScreenWindow_MarginTopLeft     = ImVec2(0.f, 0.f);    /* original C++ signature */
    full_screen_window_margin_top_left: ImVec2 = ImVec2(0.0, 0.0)
    # ImVec2 fullScreenWindow_MarginBottomRight = ImVec2(0.f, 0.f);    /* original C++ signature */
    full_screen_window_margin_bottom_right: ImVec2 = ImVec2(0.0, 0.0)

    # ------------ Theme ---------------------------------------------------------------

    # ImGuiTheme::ImGuiTweakedTheme tweakedTheme;    /* original C++ signature */
    # tweakedTheme: (enum ImGuiTheme::ImGuiTweakedTheme)
    # Changes the ImGui theme. Several themes are available, you can query the list
    # by calling HelloImGui::AvailableThemes()
    tweaked_theme: ImGuiTweakedTheme

    # ImVec4 backgroundColor = ImVec4(0.f, 0.f, 0.f, 0.f);    /* original C++ signature */
    # backgroundColor:
    # This is the "clearColor", i.e. the app window background color, is visible *only if*
    #    runnerParams.imGuiWindowParams.defaultImGuiWindowType = NoDefaultWindow
    # Alternatively, you can set your own RunnerCallbacks.CustomBackground to have full
    # control over what is drawn behind the Gui.
    background_color: ImVec4 = ImVec4(0.0, 0.0, 0.0, 0.0)

    # ImGuiWindowParams(DefaultImGuiWindowType defaultImGuiWindowType = DefaultImGuiWindowType::ProvideFullScreenWindow, bool enableViewports = false, bool configWindowsMoveFromTitleBarOnly = true, std::string menuAppTitle = "", bool showMenuBar = false, bool showMenu_App = true, bool showMenu_App_Quit = true, bool showMenu_View = true, bool showMenu_View_Themes = true, bool rememberTheme = true, bool showStatusBar = false, bool showStatus_Fps = true, bool rememberStatusBarSettings = true, ImVec2 fullScreenWindow_MarginTopLeft = ImVec2(0.f, 0.f), ImVec2 fullScreenWindow_MarginBottomRight = ImVec2(0.f, 0.f), ImGuiTheme::ImGuiTweakedTheme tweakedTheme = ImGuiTheme::ImGuiTweakedTheme(), ImVec4 backgroundColor = ImVec4(0.f, 0.f, 0.f, 0.f));    /* original C++ signature */
    def __init__(
        self,
        default_imgui_window_type: DefaultImGuiWindowType = DefaultImGuiWindowType.provide_full_screen_window,
        enable_viewports: bool = False,
        config_windows_move_from_title_bar_only: bool = True,
        menu_app_title: str = "",
        show_menu_bar: bool = False,
        show_menu_app: bool = True,
        show_menu_app_quit: bool = True,
        show_menu_view: bool = True,
        show_menu_view_themes: bool = True,
        remember_theme: bool = True,
        show_status_bar: bool = False,
        show_status_fps: bool = True,
        remember_status_bar_settings: bool = True,
        full_screen_window_margin_top_left: Optional[ImVec2Like] = None,
        full_screen_window_margin_bottom_right: Optional[ImVec2Like] = None,
        tweaked_theme: Optional[ImGuiTweakedTheme] = None,
        background_color: Optional[ImVec4Like] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * fullScreenWindow_MarginTopLeft: ImVec2(0., 0.)
                * fullScreenWindow_MarginBottomRight: ImVec2(0., 0.)
                * tweakedTheme: ImGuiTweakedTheme()
                * backgroundColor: ImVec4(0., 0., 0., 0.)
        """
        pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/runner_callbacks.h included by hello_imgui/runner_params.h                 //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/imgui_default_settings.h included by hello_imgui/runner_callbacks.h        //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/runner_callbacks.h continued                                               //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# --------------------------------------------------------------------------------------------------------------------

# @@md#VoidFunction_AnyEventCallback

# inline VoidFunction EmptyVoidFunction() { return {}; }    /* original C++ signature */
def empty_void_function() -> VoidFunction:
    pass

# VoidFunction SequenceFunctions(const VoidFunction& f1, const VoidFunction& f2);    /* original C++ signature */
def sequence_functions(f1: VoidFunction, f2: VoidFunction) -> VoidFunction:
    """SequenceFunctions: returns a function that will call f1 and f2 in sequence"""
    pass

# inline AnyEventCallback EmptyEventCallback() {return {}; }    /* original C++ signature */
def empty_event_callback() -> AnyEventCallback:
    pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#MobileCallbacks

class MobileCallbacks:
    """MobileCallbacks is a struct that contains callbacks that are called by the application
    when running under "Android, iOS and WinRT".
    These events are specific to mobile and embedded devices that have different
    requirements from your usual desktop application.
    These events must be handled quickly, since often the OS needs an immediate response
    and will terminate your process shortly after sending the event
    if you do not handle them appropriately.
    On mobile devices, it is not possible to "Quit" an application,
    it can only be put on Pause.
    """

    # VoidFunction OnDestroy = EmptyVoidFunction();    /* original C++ signature */
    # `OnDestroy`: The application is being terminated by the OS.
    on_destroy: VoidFunction = EmptyVoidFunction()

    # VoidFunction OnLowMemory = EmptyVoidFunction();    /* original C++ signature */
    # `OnLowMemory`: _VoidFunction, default=empty_.
    # When the application is low on memory, free memory if possible.
    on_low_memory: VoidFunction = EmptyVoidFunction()

    # VoidFunction OnPause = EmptyVoidFunction();    /* original C++ signature */
    # `OnPause`: The application is about to enter the background.
    on_pause: VoidFunction = EmptyVoidFunction()

    # VoidFunction OnResume = EmptyVoidFunction();    /* original C++ signature */
    # `OnResume`: The application came to foreground and is now interactive.
    # Note: 'OnPause' and 'OnResume' are called twice consecutively under iOS
    # (before and after entering background or foreground).
    on_resume: VoidFunction = EmptyVoidFunction()
    # MobileCallbacks(VoidFunction OnDestroy = EmptyVoidFunction(), VoidFunction OnLowMemory = EmptyVoidFunction(), VoidFunction OnPause = EmptyVoidFunction(), VoidFunction OnResume = EmptyVoidFunction());    /* original C++ signature */
    def __init__(
        self,
        on_destroy: Optional[VoidFunction] = None,
        on_low_memory: Optional[VoidFunction] = None,
        on_pause: Optional[VoidFunction] = None,
        on_resume: Optional[VoidFunction] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * OnDestroy: EmptyVoidFunction()
                * OnLowMemory: EmptyVoidFunction()
                * OnPause: EmptyVoidFunction()
                * OnResume: EmptyVoidFunction()
        """
        pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#EdgeToolbar

class EdgeToolbarType(enum.IntEnum):
    """EdgeToolbarType: location of an Edge Toolbar"""

    # Top,    /* original C++ signature */
    top = enum.auto()  # (= 0)
    # Bottom,    /* original C++ signature */
    bottom = enum.auto()  # (= 1)
    # Left,    /* original C++ signature */
    left = enum.auto()  # (= 2)
    # Right    /* original C++ signature */
    # }
    right = enum.auto()  # (= 3)

class EdgeToolbarOptions:
    # float sizeEm = 2.5f;    /* original C++ signature */
    # height or width the top toolbar, in em units
    # (i.e. multiples of the default font size, to be Dpi aware)
    size_em: float = 2.5

    # ImVec2 WindowPaddingEm = ImVec2(0.3f, 0.3f);    /* original C++ signature */
    # Padding inside the window, in em units
    window_padding_em: ImVec2 = ImVec2(0.3, 0.3)

    # ImVec4 WindowBg = ImVec4(0.f, 0.f, 0.f, 0.f);    /* original C++ signature */
    # Window background color, only used if WindowBg.w > 0
    window_bg: ImVec4 = ImVec4(0.0, 0.0, 0.0, 0.0)
    # EdgeToolbarOptions(float sizeEm = 2.5f, ImVec2 WindowPaddingEm = ImVec2(0.3f, 0.3f), ImVec4 WindowBg = ImVec4(0.f, 0.f, 0.f, 0.f));    /* original C++ signature */
    def __init__(
        self,
        size_em: float = 2.5,
        window_padding_em: Optional[ImVec2Like] = None,
        window_bg: Optional[ImVec4Like] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * WindowPaddingEm: ImVec2(0.3, 0.3)
                * WindowBg: ImVec4(0., 0., 0., 0.)
        """
        pass

class EdgeToolbar:
    """EdgeToolbar :a toolbar that can be placed on the edges of the App window
    It will be placed in a non-dockable window
    """

    # VoidFunction ShowToolbar = EmptyVoidFunction();    /* original C++ signature */
    show_toolbar: VoidFunction = EmptyVoidFunction()
    # EdgeToolbarOptions options;    /* original C++ signature */
    options: EdgeToolbarOptions
    # EdgeToolbar(VoidFunction ShowToolbar = EmptyVoidFunction(), EdgeToolbarOptions options = EdgeToolbarOptions());    /* original C++ signature */
    def __init__(
        self,
        show_toolbar: Optional[VoidFunction] = None,
        options: Optional[EdgeToolbarOptions] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * ShowToolbar: EmptyVoidFunction()
                * options: EdgeToolbarOptions()
        """
        pass

# std::vector<EdgeToolbarType> AllEdgeToolbarTypes();    /* original C++ signature */
def all_edge_toolbar_types() -> List[EdgeToolbarType]:
    pass

# std::string EdgeToolbarTypeName(EdgeToolbarType e);    /* original C++ signature */
def edge_toolbar_type_name(e: EdgeToolbarType) -> str:
    pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#DefaultIconFont

class DefaultIconFont(enum.IntEnum):
    """HelloImGui can optionally merge an icon font (FontAwesome 4 or 6) to the default font
    Breaking change in v1.5.0:
    - the default icon font is now FontAwesome 6, which includes many more icons.
    - you need to include manually icons_font_awesome_4.h or icons_font_awesome_6.h:
        #include "hello_imgui/icons_font_awesome_6.h" or #include "hello_imgui/icons_font_awesome_4.h"
    """

    # NoIcons,    /* original C++ signature */
    no_icons = enum.auto()  # (= 0)
    # FontAwesome4,    /* original C++ signature */
    font_awesome4 = enum.auto()  # (= 1)
    # FontAwesome6    /* original C++ signature */
    # }
    font_awesome6 = enum.auto()  # (= 2)

# @@md

# --------------------------------------------------------------------------------------------------------------------

class RunnerCallbacks:
    """@@md#RunnerCallbacks
    RunnerCallbacks is a struct that contains the callbacks
    that are called by the application

    """

    # --------------- GUI Callbacks -------------------

    # VoidFunction ShowGui = EmptyVoidFunction();    /* original C++ signature */
    # `ShowGui`: Fill it with a function that will add your widgets.
    # (ShowGui will be called at each frame, before rendering the Dockable windows, if any)
    show_gui: VoidFunction = EmptyVoidFunction()

    # VoidFunction ShowMenus = EmptyVoidFunction();    /* original C++ signature */
    # `ShowMenus`: Fill it with a function that will add ImGui menus by calling:
    #       ImGui::BeginMenu(...) / ImGui::MenuItem(...) / ImGui::EndMenu()
    #   Notes:
    #   * you do not need to call ImGui::BeginMenuBar and ImGui::EndMenuBar
    #   * Some default menus can be provided:
    #     see ImGuiWindowParams options:
    #         _showMenuBar, showMenu_App_QuitAbout, showMenu_View_
    show_menus: VoidFunction = EmptyVoidFunction()

    # VoidFunction ShowAppMenuItems = EmptyVoidFunction();    /* original C++ signature */
    # `ShowAppMenuItems`: A function that will render items that will be placed
    # in the App menu. They will be placed before the "Quit" MenuItem,
    # which is added automatically by HelloImGui.
    #  This will be displayed only if ImGuiWindowParams.showMenu_App is True
    show_app_menu_items: VoidFunction = EmptyVoidFunction()

    # VoidFunction ShowStatus = EmptyVoidFunction();    /* original C++ signature */
    # `ShowStatus`: A function that will add items to the status bar.
    #  Use small items (ImGui::Text for example), since the height of the status is 30.
    #  Also, remember to call ImGui::SameLine() between items.
    show_status: VoidFunction = EmptyVoidFunction()

    # std::map<EdgeToolbarType, EdgeToolbar> edgesToolbars;    /* original C++ signature */
    # `EdgesToolbars`:
    # A dict that contains toolbars that can be placed on the edges of the App window
    edges_toolbars: Dict[EdgeToolbarType, EdgeToolbar]

    # void AddEdgeToolbar(EdgeToolbarType edgeToolbarType,    /* original C++ signature */
    #                         VoidFunction guiFunction,
    #                         const EdgeToolbarOptions& options = EdgeToolbarOptions());
    def add_edge_toolbar(
        self,
        edge_toolbar_type: EdgeToolbarType,
        gui_function: VoidFunction,
        options: Optional[EdgeToolbarOptions] = None,
    ) -> None:
        """`AddEdgeToolbar`: Add a toolbar that can be placed on the edges of the App window


        Python bindings defaults:
            If options is None, then its default value will be: EdgeToolbarOptions()
        """
        pass
    # --------------- Startup sequence callbacks -------------------

    # VoidFunction PostInit_AddPlatformBackendCallbacks = EmptyVoidFunction();    /* original C++ signature */
    # `PostInit_AddPlatformBackendCallbacks`:
    #  You can here add a function that will be called once after OpenGL and ImGui are inited,
    #  but before the platform backend callbacks are initialized.
    #  If you, want to add your own glfw callbacks, you should use this function to do so
    #  (and then ImGui will call your callbacks followed by its own callbacks)
    post_init_add_platform_backend_callbacks: VoidFunction = EmptyVoidFunction()

    # VoidFunction PostInit = EmptyVoidFunction();    /* original C++ signature */
    # `PostInit`: You can here add a function that will be called once after everything
    #  is inited (ImGui, Platform and Renderer Backend)
    post_init: VoidFunction = EmptyVoidFunction()

    # void EnqueuePostInit(const VoidFunction& callback);    /* original C++ signature */
    def enqueue_post_init(self, callback: VoidFunction) -> None:
        """`EnqueuePostInit`: Add a function that will be called once after OpenGL
        and ImGui are inited, but before the backend callback are initialized.
        (this will modify the `PostInit` callback by appending the new callback (using `SequenceFunctions`)
        """
        pass
    # VoidFunction LoadAdditionalFonts = ImGuiDefaultSettings::LoadDefaultFont_WithFontAwesomeIcons;    /* original C++ signature */
    # `LoadAdditionalFonts`: default=_LoadDefaultFont_WithFontAwesome*.
    #  A function that is called in order to load fonts.
    # `LoadAdditionalFonts` will be called once, then *set to None*.
    # If you want to load additional fonts, during the app execution, you can
    # set LoadAdditionalFonts to a function that will load the additional fonts.
    load_additional_fonts: VoidFunction = (
        imgui_default_settings.LoadDefaultFont_WithFontAwesomeIcons
    )
    # DefaultIconFont defaultIconFont = DefaultIconFont::FontAwesome4;    /* original C++ signature */
    # If LoadAdditionalFonts==LoadDefaultFont_WithFontAwesomeIcons, this parameter control
    # which icon font will be loaded by default.
    default_icon_font: DefaultIconFont = DefaultIconFont.font_awesome4

    # VoidFunction SetupImGuiConfig = ImGuiDefaultSettings::SetupDefaultImGuiConfig;    /* original C++ signature */
    # `SetupImGuiConfig`: default=_ImGuiDefaultSettings::SetupDefaultImGuiConfig*.
    #  If needed, change ImGui config via SetupImGuiConfig
    #  (enable docking, gamepad, etc)
    setup_imgui_config: VoidFunction = imgui_default_settings.SetupDefaultImGuiConfig

    # VoidFunction SetupImGuiStyle = ImGuiDefaultSettings::SetupDefaultImGuiStyle;    /* original C++ signature */
    # `SetupImGuiStyle`: default=_ImGuiDefaultSettings::SetupDefaultImGuiConfig*.
    #  If needed, set your own style by providing your own SetupImGuiStyle callback
    setup_imgui_style: VoidFunction = imgui_default_settings.SetupDefaultImGuiStyle

    # VoidFunction RegisterTests = EmptyVoidFunction();    /* original C++ signature */
    # `RegisterTests`: A function that is called once ImGuiTestEngine is ready
    # to be filled with tests and automations definitions.
    register_tests: VoidFunction = EmptyVoidFunction()
    # bool         registerTestsCalled = false;    /* original C++ signature */
    # `registerTestsCalled`: will be set to True when RegisterTests was called
    # (you can set this to False if you want to RegisterTests to be called again
    #  during the app execution)
    register_tests_called: bool = False

    # --------------- Exit sequence callbacks -------------------

    # VoidFunction BeforeExit = EmptyVoidFunction();    /* original C++ signature */
    # `BeforeExit`: You can here add a function that will be called once before exiting
    #  (when OpenGL and ImGui are still inited)
    before_exit: VoidFunction = EmptyVoidFunction()

    # void EnqueueBeforeExit(const VoidFunction& callback);    /* original C++ signature */
    def enqueue_before_exit(self, callback: VoidFunction) -> None:
        """`EnqueueBeforeExit`: Add a function that will be called once before exiting
         (when OpenGL and ImGui are still inited)
        (this will modify the `BeforeExit` callback by appending the new callback (using `SequenceFunctions`)
        """
        pass
    # VoidFunction BeforeExit_PostCleanup = EmptyVoidFunction();    /* original C++ signature */
    # `BeforeExit_PostCleanup`: You can here add a function that will be called once
    # before exiting (after OpenGL and ImGui have been stopped)
    before_exit_post_cleanup: VoidFunction = EmptyVoidFunction()

    # --------------- Callbacks in the render loop -------------------

    # VoidFunction PreNewFrame = EmptyVoidFunction();    /* original C++ signature */
    # `PreNewFrame`: You can here add a function that will be called at each frame,
    #  and before the call to ImGui::NewFrame().
    #  It is a good place to add new dockable windows.
    pre_new_frame: VoidFunction = EmptyVoidFunction()

    # VoidFunction BeforeImGuiRender = EmptyVoidFunction();    /* original C++ signature */
    # `BeforeImGuiRender`: You can here add a function that will be called at each frame,
    #  after the user Gui code, and just before the call to
    #  ImGui::Render() (which will also call ImGui::EndFrame()).
    before_imgui_render: VoidFunction = EmptyVoidFunction()

    # VoidFunction AfterSwap = EmptyVoidFunction();    /* original C++ signature */
    # `AfterSwap`: You can here add a function that will be called at each frame,
    #  after the Gui was rendered and swapped to the screen.
    after_swap: VoidFunction = EmptyVoidFunction()

    # VoidFunction CustomBackground = EmptyVoidFunction();    /* original C++ signature */
    # `CustomBackground`:
    #  By default, the background is cleared using ImGuiWindowParams.backgroundColor.
    #  If set, this function gives you full control over the background that is drawn
    #  behind the Gui. An example use case is if you have a 3D application
    #  like a mesh editor, or game, and just want the Gui to be drawn
    #  on top of that content.
    custom_background: VoidFunction = EmptyVoidFunction()

    # VoidFunction PostRenderDockableWindows = EmptyVoidFunction();    /* original C++ signature */
    # `PostRenderDockableWindows`: Fill it with a function that will be called
    # after the dockable windows are rendered.
    post_render_dockable_windows: VoidFunction = EmptyVoidFunction()

    # AnyEventCallback AnyBackendEventCallback = EmptyEventCallback();    /* original C++ signature */
    # `AnyBackendEventCallback`:
    #  Callbacks for events from a specific backend. _Only implemented for SDL.
    #  where the event will be of type 'SDL_Event *'_
    #  This callback should return True if the event was handled
    #  and shall not be processed further.
    #  Note: in the case of GLFW, you should use register them in `PostInit`
    any_backend_event_callback: AnyEventCallback = EmptyEventCallback()

    # --------------- Mobile callbacks -------------------
    # RunnerCallbacks(VoidFunction ShowGui = EmptyVoidFunction(), VoidFunction ShowMenus = EmptyVoidFunction(), VoidFunction ShowAppMenuItems = EmptyVoidFunction(), VoidFunction ShowStatus = EmptyVoidFunction(), VoidFunction PostInit_AddPlatformBackendCallbacks = EmptyVoidFunction(), VoidFunction PostInit = EmptyVoidFunction(), VoidFunction LoadAdditionalFonts = ImGuiDefaultSettings::LoadDefaultFont_WithFontAwesomeIcons, DefaultIconFont defaultIconFont = DefaultIconFont::FontAwesome4, VoidFunction SetupImGuiConfig = ImGuiDefaultSettings::SetupDefaultImGuiConfig, VoidFunction SetupImGuiStyle = ImGuiDefaultSettings::SetupDefaultImGuiStyle, VoidFunction RegisterTests = EmptyVoidFunction(), bool registerTestsCalled = false, VoidFunction BeforeExit = EmptyVoidFunction(), VoidFunction BeforeExit_PostCleanup = EmptyVoidFunction(), VoidFunction PreNewFrame = EmptyVoidFunction(), VoidFunction BeforeImGuiRender = EmptyVoidFunction(), VoidFunction AfterSwap = EmptyVoidFunction(), VoidFunction CustomBackground = EmptyVoidFunction(), VoidFunction PostRenderDockableWindows = EmptyVoidFunction(), AnyEventCallback AnyBackendEventCallback = EmptyEventCallback());    /* original C++ signature */
    def __init__(
        self,
        show_gui: Optional[VoidFunction] = None,
        show_menus: Optional[VoidFunction] = None,
        show_app_menu_items: Optional[VoidFunction] = None,
        show_status: Optional[VoidFunction] = None,
        post_init_add_platform_backend_callbacks: Optional[VoidFunction] = None,
        post_init: Optional[VoidFunction] = None,
        load_additional_fonts: Optional[VoidFunction] = None,
        default_icon_font: DefaultIconFont = DefaultIconFont.font_awesome4,
        setup_imgui_config: Optional[VoidFunction] = None,
        setup_imgui_style: Optional[VoidFunction] = None,
        register_tests: Optional[VoidFunction] = None,
        register_tests_called: bool = False,
        before_exit: Optional[VoidFunction] = None,
        before_exit_post_cleanup: Optional[VoidFunction] = None,
        pre_new_frame: Optional[VoidFunction] = None,
        before_imgui_render: Optional[VoidFunction] = None,
        after_swap: Optional[VoidFunction] = None,
        custom_background: Optional[VoidFunction] = None,
        post_render_dockable_windows: Optional[VoidFunction] = None,
        any_backend_event_callback: Optional[AnyEventCallback] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * ShowGui: EmptyVoidFunction()
                * ShowMenus: EmptyVoidFunction()
                * ShowAppMenuItems: EmptyVoidFunction()
                * ShowStatus: EmptyVoidFunction()
                * PostInit_AddPlatformBackendCallbacks: EmptyVoidFunction()
                * PostInit: EmptyVoidFunction()
                * LoadAdditionalFonts: imgui_default_settings.LoadDefaultFont_WithFontAwesomeIcons
                * SetupImGuiConfig: imgui_default_settings.SetupDefaultImGuiConfig
                * SetupImGuiStyle: imgui_default_settings.SetupDefaultImGuiStyle
                * RegisterTests: EmptyVoidFunction()
                * BeforeExit: EmptyVoidFunction()
                * BeforeExit_PostCleanup: EmptyVoidFunction()
                * PreNewFrame: EmptyVoidFunction()
                * BeforeImGuiRender: EmptyVoidFunction()
                * AfterSwap: EmptyVoidFunction()
                * CustomBackground: EmptyVoidFunction()
                * PostRenderDockableWindows: EmptyVoidFunction()
                * AnyBackendEventCallback: EmptyEventCallback()
        """
        pass

# @@md

# VoidFunction AppendCallback(const VoidFunction& previousCallback, const VoidFunction& newCallback);    /* original C++ signature */
def append_callback(
    previous_callback: VoidFunction, new_callback: VoidFunction
) -> VoidFunction:
    """AppendCallback: legacy synonym for SequenceFunctions"""
    pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/docking_params.h included by hello_imgui/runner_params.h                   //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# *
# @@md#DockingIntro
#
# HelloImGui makes it easy to use dockable windows
# (based on ImGui [docking branch](https://github.com/ocornut/imgui/tree/docking)).
#
# You can define several layouts and switch between them:  each layout which will remember
# the user modifications and the list of opened windows
#
# HelloImGui will then provide a "View" menu with options to show/hide the dockable windows,
# restore the default layout, switch between layouts, etc.
#
#![demo docking](https://traineq.org/ImGuiBundle/HelloImGuiLayout.gif)
#
# * Source for this example: https://github.com/pthom/hello_imgui/tree/master/src/hello_imgui_demos/hello_imgui_demodocking
# * [Video explanation on YouTube](https://www.youtube.com/watch?v=XKxmz__F4ow) (5 minutes)
#
#
# The different available layouts are provided inside RunnerParams via the two members below:
# ```cpp
# struct RunnerParams
# {
#    ...
#    // default layout of the application
#    DockingParams dockingParams;
#
#    // optional alternative layouts
#    std::vector<DockingParams> alternativeDockingLayouts;
#
#    ...
# };
# ```
#
# And `DockingParams` contains members that define a layout:
#
# ```cpp
# struct DockingParams
# {
#    // displayed name of the layout
#    std::string layoutName = "Default";
#
#    // list of splits
#    // (which define spaces where the windows will be placed)
#    std::vector<DockingSplit> dockingSplits;
#
#    // list of windows
#    // (with their gui code, and specifying in which space they will be placed)
#    std::vector<DockableWindow> dockableWindows;
#
#    ...
# };
# ```
#
# Inside DockingParams, the member `dockingSplits` specifies the layout, and the member `dockableWindows`
# specifies the list of dockable windows, along with their default location, and their code (given by lambdas).
#
# @@md
#
#
# @@md#DockingExample
#
# Below is an example that shows how to instantiate a layout:
#
# 1. First, define the docking splits:
#
# ```cpp
# std::vector<HelloImGui::DockingSplit> CreateDefaultDockingSplits()
# {
#    //   Here, we want to split "MainDockSpace" (which is provided automatically)
#    //   into three zones, like this:
#    //    ___________________________________________
#    //    |        |                                |
#    //    | Command|                                |
#    //    | Space  |    MainDockSpace               |
#    //    |        |                                |
#    //    |        |                                |
#    //    |        |                                |
#    //    -------------------------------------------
#    //    |     MiscSpace                           |
#    //    -------------------------------------------
#    //
#
#    // add a space named "MiscSpace" whose height is 25% of the app height.
#    // This will split the preexisting default dockspace "MainDockSpace" in two parts.
#    HelloImGui::DockingSplit splitMainMisc;
#    splitMainMisc.initialDock = "MainDockSpace";
#    splitMainMisc.newDock = "MiscSpace";
#    splitMainMisc.direction = ImGuiDir_Down;
#    splitMainMisc.ratio = 0.25;
#
#    // Then, add a space to the left which occupies a column
#    // whose width is 25% of the app width
#    HelloImGui::DockingSplit splitMainCommand;
#    splitMainCommand.initialDock = "MainDockSpace";
#    splitMainCommand.newDock = "CommandSpace";
#    splitMainCommand.direction = ImGuiDir_Left;
#    splitMainCommand.ratio = 0.25;
#
#    std::vector<HelloImGui::DockingSplit> splits {splitMainMisc, splitMainCommand};
#    return splits;
# }
# ```
#
# 2. Then, define the dockable windows:
#
# ```cpp
# std::vector<HelloImGui::DockableWindow> CreateDockableWindows(AppState& appState)
# {
#    // A Command panel named "Commands" will be placed in "CommandSpace".
#    // Its Gui is provided calls "CommandGui"
#    HelloImGui::DockableWindow commandsWindow;
#    commandsWindow.label = "Commands";
#    commandsWindow.dockSpaceName = "CommandSpace";
#    commandsWindow.GuiFunction = [&] { CommandGui(appState); };
#
#    // A Log window named "Logs" will be placed in "MiscSpace".
#    // It uses the HelloImGui logger gui
#    HelloImGui::DockableWindow logsWindow;
#    logsWindow.label = "Logs";
#    logsWindow.dockSpaceName = "MiscSpace";
#    logsWindow.GuiFunction = [] { HelloImGui::LogGui(); };
#
#    ...
# }
# ```
#
# 3. Finally, fill the RunnerParams
#
# ```cpp
# HelloImGui::RunnerParams runnerParams;
# runnerParams.imGuiWindowParams.defaultImGuiWindowType =
#    HelloImGui::DefaultImGuiWindowType::ProvideFullScreenDockSpace;
#
# runnerParams.dockingParams.dockingSplits = CreateDefaultDockingSplits();
# runnerParams.dockingParams.dockableWindows = CreateDockableWindows();
#
#
# HelloImGui::Run(runnerParams);
# ```
#
# @@md
#

# ***************************************************************************

# @@md#DockingSplit

class DockingSplit:
    """DockingSplit is a struct that defines the way the docking splits should
    be applied on the screen in order to create new Dock Spaces.
    DockingParams contains a
        vector<DockingSplit>
    in order to partition the screen at your will.
    """

    # DockSpaceName initialDock;    /* original C++ signature */
    # `initialDock`: _DockSpaceName (aka string)_
    #  id of the space that should be split.
    #  At the start, there is only one Dock Space named "MainDockSpace".
    #  You should start by partitioning this space, in order to create a new dock space.
    initial_dock: DockSpaceName

    # DockSpaceName newDock;    /* original C++ signature */
    # `newDock`: _DockSpaceName (aka string)_.
    #  id of the new dock space that will be created.
    new_dock: DockSpaceName

    # ImGuiDir direction;    /* original C++ signature */
    # `direction`: *ImGuiDir_*
    #  (enum with ImGuiDir_Down, ImGuiDir_Down, ImGuiDir_Left, ImGuiDir_Right)*
    #  Direction where this dock space should be created.
    direction: ImGuiDir

    # float ratio = 0.25f;    /* original C++ signature */
    # `ratio`: _float, default=0.25_.
    #  Ratio of the initialDock size that should be used by the new dock space.
    ratio: float = 0.25

    # ImGuiDockNodeFlags nodeFlags = ImGuiDockNodeFlags_None;    /* original C++ signature */
    # `nodeFlags`: *ImGuiDockNodeFlags_ (enum)*.
    #  Flags to apply to the new dock space
    #  (enable/disable resizing, splitting, tab bar, etc.)
    node_flags: ImGuiDockNodeFlags = DockNodeFlags_.none

    # DockingSplit(const DockSpaceName& initialDock_ = "", const DockSpaceName& newDock_ = "",    /* original C++ signature */
    #                  ImGuiDir direction_ = ImGuiDir_Down, float ratio_ = 0.25f,
    #                  ImGuiDockNodeFlags nodeFlags_ = ImGuiDockNodeFlags_None)
    #         : initialDock(initialDock_), newDock(newDock_), direction(direction_), ratio(ratio_), nodeFlags(nodeFlags_) {}
    def __init__(
        self,
        initial_dock_: DockSpaceName = "",
        new_dock_: DockSpaceName = "",
        direction_: Optional[ImGuiDir] = None,
        ratio_: float = 0.25,
        node_flags_: Optional[ImGuiDockNodeFlags] = None,
    ) -> None:
        """Constructor


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * direction_: Dir.down
                * nodeFlags_: DockNodeFlags_.none
        """
        pass

# @@md

# @@md#DockableWindow

class DockableWindow:
    """DockableWindow is a struct that represents a window that can be docked."""

    # --------------- Main params -------------------

    # std::string label;    /* original C++ signature */
    # `label`: _string_. Title of the window. It should be unique! Use "##" to add a unique suffix if needed.
    label: str

    # DockSpaceName dockSpaceName;    /* original C++ signature */
    # `dockSpaceName`: _DockSpaceName (aka string)_.
    #  Id of the dock space where this window should initially be placed
    dock_space_name: DockSpaceName

    # VoidFunction GuiFunction = EmptyVoidFunction();    /* original C++ signature */
    # `GuiFunction`: _VoidFunction_.
    # Any function that will render this window's Gui
    gui_function: VoidFunction = EmptyVoidFunction()

    # --------------- Options --------------------------

    # bool isVisible = true;    /* original C++ signature */
    # `isVisible`: _bool, default=true_.
    #  Flag that indicates whether this window is visible or not.
    is_visible: bool = True

    # bool rememberIsVisible = true;    /* original C++ signature */
    # `rememberIsVisible`: _bool, default=true_.
    #  Flag that indicates whether the window visibility should be saved in settings.
    remember_is_visible: bool = True

    # bool canBeClosed = true;    /* original C++ signature */
    # `canBeClosed`: _bool, default=true_.
    #  Flag that indicates whether the user can close this window.
    can_be_closed: bool = True

    # bool callBeginEnd = true;    /* original C++ signature */
    # `callBeginEnd`: _bool, default=true_.
    #  Flag that indicates whether ImGui::Begin and ImGui::End
    #  calls should be added automatically (with the given "label").
    #  Set to False if you want to call ImGui::Begin/End yourself
    call_begin_end: bool = True

    # bool includeInViewMenu = true;    /* original C++ signature */
    # `includeInViewMenu`: _bool, default=true_.
    #  Flag that indicates whether this window should be mentioned in the view menu.
    include_in_view_menu: bool = True

    # ImGuiWindowFlags imGuiWindowFlags = 0;    /* original C++ signature */
    # `imGuiWindowFlags`: _ImGuiWindowFlags, default=0_.
    #  Window flags, see enum ImGuiWindowFlags_
    imgui_window_flags: ImGuiWindowFlags = 0

    # --------------- Focus window -----------------------------

    # bool focusWindowAtNextFrame = false;    /* original C++ signature */
    # `focusWindowAtNextFrame`: _bool, default = false_.
    #  If set to True this window will be focused at the next frame.
    focus_window_at_next_frame: bool = False

    # --------------- Size & Position --------------------------
    #              (only if not docked)

    # ImVec2 windowSize = ImVec2(0.f, 0.f);    /* original C++ signature */
    # `windowSize`: _ImVec2, default=(0., 0.) (i.e let the app decide)_.
    #  Window size (unused if docked)
    window_size: ImVec2 = ImVec2(0.0, 0.0)

    # ImGuiCond  windowSizeCondition = ImGuiCond_FirstUseEver;    /* original C++ signature */
    # `windowSizeCondition`: _ImGuiCond, default=ImGuiCond_FirstUseEver_.
    #  When to apply the window size.
    window_size_condition: ImGuiCond = Cond_.first_use_ever

    # ImVec2 windowPosition = ImVec2(0.f, 0.f);    /* original C++ signature */
    # `windowPos`: _ImVec2, default=(0., 0.) (i.e let the app decide)_.
    #  Window position (unused if docked)
    window_position: ImVec2 = ImVec2(0.0, 0.0)

    # ImGuiCond  windowPositionCondition = ImGuiCond_FirstUseEver;    /* original C++ signature */
    # `windowPosCondition`: _ImGuiCond, default=ImGuiCond_FirstUseEver_.
    #  When to apply the window position.
    window_position_condition: ImGuiCond = Cond_.first_use_ever

    # DockableWindow(    /* original C++ signature */
    #         const std::string & label_ = "",
    #         const DockSpaceName & dockSpaceName_ = "",
    #         const VoidFunction guiFunction_ = EmptyVoidFunction(),
    #         bool isVisible_ = true,
    #         bool canBeClosed_ = true)
    #         : label(label_), dockSpaceName(dockSpaceName_),
    #           GuiFunction(guiFunction_),
    #           isVisible(isVisible_),
    #           canBeClosed(canBeClosed_) {}
    def __init__(
        self,
        label_: str = "",
        dock_space_name_: DockSpaceName = "",
        gui_function_: Optional[VoidFunction] = None,
        is_visible_: bool = True,
        can_be_closed_: bool = True,
    ) -> None:
        """--------------- Constructor ------------------------------
         Constructor


        Python bindings defaults:
            If guiFunction_ is None, then its default value will be: EmptyVoidFunction()
        """
        pass

# @@md

class DockingLayoutCondition(enum.IntEnum):
    # FirstUseEver,    /* original C++ signature */
    first_use_ever = enum.auto()  # (= 0)
    # ApplicationStart,    /* original C++ signature */
    application_start = enum.auto()  # (= 1)
    # Never    /* original C++ signature */
    # }
    never = enum.auto()  # (= 2)

# @@md#DockingParams

class DockingParams:
    """DockingParams contains all the settings concerning the docking:
    - list of splits
    - list of dockable windows
    """

    # --------------- Main params -----------------------------

    # std::vector<DockingSplit>   dockingSplits;    /* original C++ signature */
    # `dockingSplits`: _vector[DockingSplit]_.
    #  Defines the way docking splits should be applied on the screen
    #  in order to create new Dock Spaces
    docking_splits: List[DockingSplit]

    # std::vector<DockableWindow> dockableWindows;    /* original C++ signature */
    # `dockableWindows`: _vector[DockableWindow]_.
    #  List of the dockable windows, together with their Gui code
    dockable_windows: List[DockableWindow]

    # std::string layoutName = "Default";    /* original C++ signature */
    # `layoutName`: _string, default="default"_.
    #  Displayed name of the layout.
    #  Only used in advanced cases, when several layouts are available.
    layout_name: str = "Default"

    # --------------- Options -----------------------------

    # ImGuiDockNodeFlags mainDockSpaceNodeFlags = ImGuiDockNodeFlags_PassthruCentralNode;    /* original C++ signature */
    # `mainDockSpaceNodeFlags`: _ImGuiDockNodeFlags (enum),
    #      default=ImGuiDockNodeFlags_PassthruCentralNode_
    #  Flags to apply to the main dock space
    #  (enable/disable resizing, splitting, tab bar, etc.).
    #  Most flags are inherited by children dock spaces.
    #  You can also set flags for specific dock spaces via `DockingSplit.nodeFlags`
    main_dock_space_node_flags: ImGuiDockNodeFlags = (
        DockNodeFlags_.passthru_central_node
    )

    # --------------- Layout handling -----------------------------

    # DockingLayoutCondition layoutCondition = DockingLayoutCondition::FirstUseEver;    /* original C++ signature */
    # `layoutCondition`: _enum DockingLayoutCondition, default=FirstUseEver_.
    #  When to apply the docking layout. Choose between
    #      FirstUseEver (apply once, then keep user preference),
    #      ApplicationStart (always reapply at application start)
    #      Never
    layout_condition: DockingLayoutCondition = DockingLayoutCondition.first_use_ever

    # bool layoutReset = false;    /* original C++ signature */
    # `layoutReset`: _bool, default=false_.
    #  Reset layout on next frame, i.e. drop the layout customizations which were
    #  applied manually by the user. layoutReset will be reset to False after this.
    layout_reset: bool = False

    # --------------- Helper Methods -----------------------------

    # DockableWindow * dockableWindowOfName(const std::string& name);    /* original C++ signature */
    def dockable_window_of_name(self, name: str) -> DockableWindow:
        """`DockableWindow * dockableWindowOfName(const std::string & name)`:
        returns a pointer to a dockable window
        """
        pass
    # bool focusDockableWindow(const std::string& windowName);    /* original C++ signature */
    def focus_dockable_window(self, window_name: str) -> bool:
        """`bool focusDockableWindow(const std::string& name)`:
        will focus a dockable window (and make its tab visible if needed)
        """
        pass
    # std::optional<ImGuiID> dockSpaceIdFromName(const std::string& dockSpaceName);    /* original C++ signature */
    def dock_space_id_from_name(self, dock_space_name: str) -> Optional[ImGuiID]:
        """`optional<ImGuiID> dockSpaceIdFromName(const std::string& dockSpaceName)`:
        returns the ImGuiID corresponding to the dockspace with this name
        """
        pass
    # DockingParams(std::vector<DockingSplit> dockingSplits = std::vector<DockingSplit>(), std::vector<DockableWindow> dockableWindows = std::vector<DockableWindow>(), std::string layoutName = "Default", ImGuiDockNodeFlags mainDockSpaceNodeFlags = ImGuiDockNodeFlags_PassthruCentralNode, DockingLayoutCondition layoutCondition = DockingLayoutCondition::FirstUseEver, bool layoutReset = false);    /* original C++ signature */
    def __init__(
        self,
        docking_splits: Optional[List[DockingSplit]] = None,
        dockable_windows: Optional[List[DockableWindow]] = None,
        layout_name: str = "Default",
        main_dock_space_node_flags: Optional[ImGuiDockNodeFlags] = None,
        layout_condition: DockingLayoutCondition = DockingLayoutCondition.first_use_ever,
        layout_reset: bool = False,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * dockingSplits: List[DockingSplit]()
                * dockableWindows: List[DockableWindow]()
                * mainDockSpaceNodeFlags: DockNodeFlags_.passthru_central_node
        """
        pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/backend_pointers.h included by hello_imgui/runner_params.h                 //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

class BackendPointers:
    """@@md#BackendPointers

    BackendPointers is a struct that contains optional pointers to the
    backend implementations (for SDL and GLFW).

    These pointers will be filled when the application starts, and you can use them
    to customize your application behavior using the selected backend.

    Note: If using the Metal, Vulkan or DirectX rendering backend, you can find
    some interesting pointers inside
        `src/hello_imgui/internal/backend_impls/rendering_metal.h`
        `src/hello_imgui/internal/backend_impls/rendering_vulkan.h`
        `src/hello_imgui/internal/backend_impls/rendering_dx11.h`
        `src/hello_imgui/internal/backend_impls/rendering_dx12.h`
    """

    # * `glfwWindow`: Pointer to the main GLFW window (of type `GLFWwindow*`).
    #  Only filled if the backend is GLFW.
    # void* glfwWindow     = nullptr;    /* original C++ signature */
    glfw_window: Any = None  # GLFWwindow*

    # * `sdlWindow`: Pointer to the main SDL window (of type `SDL_Window*`).
    #  Only filled if the backend is SDL (or emscripten + sdl)
    # void* sdlWindow      = nullptr;    /* original C++ signature */
    sdl_window: Any = None  # SDL_Window*

    # * `sdlGlContext`: Pointer to SDL's GlContext (of type `SDL_GLContext`).
    #  Only filled if the backend is SDL (or emscripten + sdl)
    # void* sdlGlContext   = nullptr;    /* original C++ signature */
    sdl_gl_context: Any = None  # SDL_GLContext
    # BackendPointers();    /* original C++ signature */
    def __init__(self) -> None:
        """Auto-generated default constructor"""
        pass

# @@md

# @@md#RemoteParams

class RemoteParams:
    """RemoteParams is a struct that contains the settings for displaying the application on a remote device.
    using https://github.com/sammyfreg/netImgui
    or using https://github.com/ggerganov/imgui-ws
    Those features are experimental and not supported with the standard version of HelloImGui,
    """

    # bool enableRemoting = false;    /* original C++ signature */
    enable_remoting: bool = False

    #
    # Params used only by imgui-ws
    #
    # int wsPort = 5003;    /* original C++ signature */
    ws_port: int = 5003
    # std::string wsHttpRootFolder = "";    /* original C++ signature */
    ws_http_root_folder: str = (
        ""  # Optional folder were some additional files can be served
    )
    # bool wsProvideIndexHtml = true;    /* original C++ signature */
    ws_provide_index_html: bool = (
        True  # If True, will automatically serve a simple index.html file that contains the canvas and the imgui-ws client code
    )

    #
    # Params used only by netImgui
    #
    # bool exitWhenServerDisconnected = false;    /* original C++ signature */
    exit_when_server_disconnected: bool = False
    # double durationMaxDisconnected = 30.0;    /* original C++ signature */
    duration_max_disconnected: float = 30.0
    # std::string serverHost = "localhost";    /* original C++ signature */
    # The server host (if empty, will use "localhost")
    # The server is the app that simply displays the application on a remote device
    server_host: str = "localhost"
    # uint32_t serverPort = 8888;    /* original C++ signature */
    # The server port (default is 8888)
    server_port: int = 8888
    # bool transmitWindowSize = false;    /* original C++ signature */
    # If True, transmit the window size to the server
    transmit_window_size: bool = False
    # RemoteParams(bool enableRemoting = false, int wsPort = 5003, std::string wsHttpRootFolder = "", bool wsProvideIndexHtml = true, bool exitWhenServerDisconnected = false, double durationMaxDisconnected = 30.0, std::string serverHost = "localhost", uint32_t serverPort = 8888, bool transmitWindowSize = false);    /* original C++ signature */
    def __init__(
        self,
        enable_remoting: bool = False,
        ws_port: int = 5003,
        ws_http_root_folder: str = "",
        ws_provide_index_html: bool = True,
        exit_when_server_disconnected: bool = False,
        duration_max_disconnected: float = 30.0,
        server_host: str = "localhost",
        server_port: int = 8888,
        transmit_window_size: bool = False,
    ) -> None:
        """Auto-generated default constructor with named params"""
        pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/renderer_backend_options.h included by hello_imgui/runner_params.h         //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# --------------------------------------------------------------------------------------------------------------------

# @@md#OpenGlOptions

class OpenGlOptions:
    """OpenGlOptions contains advanced options used at the startup of OpenGL.
    These parameters are reserved for advanced users.
    By default, Hello ImGui will select reasonable default values, and these parameters are not used.
    Use at your own risk, as they make break the multi-platform compatibility of your application!
    All these parameters are platform dependent.
    For real multiplatform examples, see
        hello_imgui/src/hello_imgui/internal/backend_impls/opengl_setup_helper/opengl_setup_glfw.cpp
    and
        hello_imgui/src/hello_imgui/internal/backend_impls/opengl_setup_helper/opengl_setup_sdl.cpp

    How to set those values manually:
    ---------------------------------
    you may set them manually:
       (1) Either by setting them programmatically in your application
           (set their values in `runnerParams.rendererBackendOptions.openGlOptions`)
       (2) Either by setting them in a `hello_imgui.ini` file in the current folder, or any of its parent folders.
          (this is useful when you want to set them for a specific app or set of apps, without modifying the app code)
          See hello_imgui/hello_imgui_example.ini for an example of such a file.
    Note: if several methods are used, the order of priority is (1) > (2)

    """

    # std::optional<std::string>  GlslVersion =  std::nullopt;    /* original C++ signature */
    # Could be for example:
    #    "150" on macOS
    #    "130" on Windows
    #    "300es" on GLES
    glsl_version: Optional[str] = None

    # OpenGL 3.3 (these options won't work for GlEs)
    # std::optional<int>          MajorVersion = std::nullopt;    /* original C++ signature */
    major_version: Optional[int] = None
    # std::optional<int>          MinorVersion = std::nullopt;    /* original C++ signature */
    minor_version: Optional[int] = None

    # std::optional<bool>         UseCoreProfile = std::nullopt;    /* original C++ signature */
    # OpenGL Core Profile (i.e. only includes the newer, maintained features of OpenGL)
    use_core_profile: Optional[bool] = None
    # std::optional<bool>         UseForwardCompat = std::nullopt;    /* original C++ signature */
    # OpenGL Forward Compatibility (required on macOS)
    use_forward_compat: Optional[bool] = None

    # std::optional<int> AntiAliasingSamples =  std::nullopt;    /* original C++ signature */
    # `AntiAliasingSamples`
    # If > 0, this value will be used to set the number of samples used for anti-aliasing.
    # This is used only when running with Glfw  + OpenGL (which is the default)
    # Notes:
    # - we query the maximum number of samples supported by the hardware, via glGetIntegerv(GL_MAX_SAMPLES)
    # - if you set this value to a non-zero value, it will be used instead of the default value of 8
    #   (except if it is greater than the maximum supported value, in which case a warning will be issued)
    # - if you set this value to 0, antialiasing will be disabled
    #
    # AntiAliasingSamples has a strong impact on the quality of the text rendering
    #     - 0: no antialiasing
    #     - 8: optimal
    #     - 16: optimal if using imgui-node-editor and you want to render very small text when unzooming
    anti_aliasing_samples: Optional[int] = None
    # OpenGlOptions(std::optional<std::string> GlslVersion = std::nullopt, std::optional<int> MajorVersion = std::nullopt, std::optional<int> MinorVersion = std::nullopt, std::optional<bool> UseCoreProfile = std::nullopt, std::optional<bool> UseForwardCompat = std::nullopt, std::optional<int> AntiAliasingSamples = std::nullopt);    /* original C++ signature */
    def __init__(
        self,
        glsl_version: Optional[str] = None,
        major_version: Optional[int] = None,
        minor_version: Optional[int] = None,
        use_core_profile: Optional[bool] = None,
        use_forward_compat: Optional[bool] = None,
        anti_aliasing_samples: Optional[int] = None,
    ) -> None:
        """Auto-generated default constructor with named params"""
        pass

# @@md

# @@md#RendererBackendOptions

# bool hasEdrSupport();    /* original C++ signature */
def has_edr_support() -> bool:
    """`bool hasEdrSupport()`:
    Check whether extended dynamic range (EDR), i.e. the ability to reproduce
    intensities exceeding the standard dynamic range from 0.0-1.0, is supported.

    To leverage EDR support, you need to set `floatBuffer=True` in `RendererBackendOptions`.
    Only the macOS Metal backend currently supports this.

    This currently returns False on all backends except Metal, where it checks whether
    this is supported on the current displays.
    """
    pass

class RendererBackendOptions:
    """RendererBackendOptions is a struct that contains options for the renderer backend
    (Metal, Vulkan, DirectX, OpenGL)
    """

    # bool requestFloatBuffer = false;    /* original C++ signature */
    # `requestFloatBuffer`:
    # Set to True to request a floating-point framebuffer.
    # Only available on Metal, if your display supports it.
    # Before setting this to True, first check `hasEdrSupport()`
    request_float_buffer: bool = False

    # OpenGlOptions openGlOptions;    /* original C++ signature */
    # `openGlOptions`:
    # Advanced options for OpenGL. Use at your own risk.
    open_gl_options: OpenGlOptions
    # RendererBackendOptions(bool requestFloatBuffer = false, OpenGlOptions openGlOptions = OpenGlOptions());    /* original C++ signature */
    def __init__(
        self,
        request_float_buffer: bool = False,
        open_gl_options: Optional[OpenGlOptions] = None,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If openGlOptions is None, then its default value will be: OpenGlOptions()
        """
        pass

# Note:
# If using Metal, Vulkan or DirectX, you can find interesting pointers inside:
#     src/hello_imgui/internal/backend_impls/rendering_metal.h
#     src/hello_imgui/internal/backend_impls/rendering_vulkan.h
#     src/hello_imgui/internal/backend_impls/rendering_dx11.h
#     src/hello_imgui/internal/backend_impls/rendering_dx12.h

# @@md

class OpenGlOptionsFilled_:
    """(Private structure, not part of the public API)
    OpenGlOptions after selecting the default platform-dependent values + after applying the user settings
    """

    # std::string  GlslVersion =  "150";    /* original C++ signature */
    glsl_version: str = "150"
    # int          MajorVersion = 3;    /* original C++ signature */
    major_version: int = 3
    # int          MinorVersion = 3;    /* original C++ signature */
    minor_version: int = 3
    # bool         UseCoreProfile = true;    /* original C++ signature */
    use_core_profile: bool = True
    # bool         UseForwardCompat = true;    /* original C++ signature */
    use_forward_compat: bool = True
    # int          AntiAliasingSamples = 8;    /* original C++ signature */
    anti_aliasing_samples: int = 8
    # OpenGlOptionsFilled_(std::string GlslVersion = "150", int MajorVersion = 3, int MinorVersion = 3, bool UseCoreProfile = true, bool UseForwardCompat = true, int AntiAliasingSamples = 8);    /* original C++ signature */
    def __init__(
        self,
        glsl_version: str = "150",
        major_version: int = 3,
        minor_version: int = 3,
        use_core_profile: bool = True,
        use_forward_compat: bool = True,
        anti_aliasing_samples: int = 8,
    ) -> None:
        """Auto-generated default constructor with named params"""
        pass

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui/runner_params.h continued                                                  //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# --------------------------------------------------------------------------------------------------------------------

# @@md#PlatformBackendType

# You can select the platform backend type (SDL, GLFW) and the rendering backend type
# via RunnerParams.platformBackendType and RunnerParams.renderingBackendType.

class PlatformBackendType(enum.IntEnum):
    """Platform backend type (SDL, GLFW)
    They are listed in the order of preference when FirstAvailable is selected.
    """

    # FirstAvailable,    /* original C++ signature */
    first_available = enum.auto()  # (= 0)
    # Glfw,    /* original C++ signature */
    glfw = enum.auto()  # (= 1)
    # Sdl,    /* original C++ signature */
    sdl = enum.auto()  # (= 2)
    # Null    /* original C++ signature */
    # }
    null = enum.auto()  # (= 3)

class RendererBackendType(enum.IntEnum):
    """Rendering backend type (OpenGL3, Metal, Vulkan, DirectX11, DirectX12)
    They are listed in the order of preference when FirstAvailable is selected.
    """

    # FirstAvailable,    /* original C++ signature */
    first_available = enum.auto()  # (= 0)
    # OpenGL3,    /* original C++ signature */
    open_gl3 = enum.auto()  # (= 1)
    # Metal,    /* original C++ signature */
    metal = enum.auto()  # (= 2)
    # Vulkan,    /* original C++ signature */
    vulkan = enum.auto()  # (= 3)
    # DirectX11,    /* original C++ signature */
    direct_x11 = enum.auto()  # (= 4)
    # DirectX12,    /* original C++ signature */
    direct_x12 = enum.auto()  # (= 5)
    # Null    /* original C++ signature */
    # }
    null = enum.auto()  # (= 6)

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#IniFolderType

class IniFolderType(enum.IntEnum):
    """IniFolderType is an enum which describes where is the base path to store
    the ini file for the application settings.

    You can use IniFolderLocation(iniFolderType) to get the corresponding path.

    RunnerParams contains the following members, which are used to compute
    the ini file location:
        iniFolderType           (IniFolderType::CurrentFolder by default)
        iniFilename             (empty string by default)
        iniFilename_useAppWindowTitle
            (True by default: iniFilename is derived from
             appWindowParams.windowTitle)

    iniFilename may contain a subfolder
    (which will be created inside the iniFolderType folder if needed)

    """

    # CurrentFolder,    /* original C++ signature */
    # CurrentFolder: the folder where the application is executed
    # (convenient for development, but not recommended for production)
    current_folder = enum.auto()  # (= 0)

    # AbsolutePath,    /* original C++ signature */
    # AbsolutePath: an absolute path
    # (convenient, but not recommended if targeting multiple platforms)
    absolute_path = enum.auto()  # (= 1)

    # AppUserConfigFolder,    /* original C++ signature */
    # AppUserConfigFolder:
    #      AppData under Windows (Example: C:\Users\[Username]\AppData\Roaming under windows)
    #      ~/.config under Linux
    #      "~/Library/Application Support" under macOS
    # (recommended for production, if settings do not need to be easily accessible by the user)
    app_user_config_folder = enum.auto()  # (= 2)

    # AppExecutableFolder,    /* original C++ signature */
    # AppExecutableFolder: the folder where the application executable is located
    # (this may be different from CurrentFolder if the application is launched from a shortcut)
    # (convenient for development, but not recommended for production)
    app_executable_folder = enum.auto()  # (= 3)

    # HomeFolder,    /* original C++ signature */
    # HomeFolder: the user home folder
    # (recommended for production, if settings need to be easily accessible by the user)
    home_folder = enum.auto()  # (= 4)

    # DocumentsFolder,    /* original C++ signature */
    # DocumentsFolder: the user documents folder
    documents_folder = enum.auto()  # (= 5)

    # TempFolder    /* original C++ signature */
    # }
    # TempFolder: the system temp folder
    temp_folder = enum.auto()  # (= 6)

# std::string IniFolderLocation(IniFolderType iniFolderType);    /* original C++ signature */
def ini_folder_location(ini_folder_type: IniFolderType) -> str:
    """Returns the path corresponding to the given IniFolderType"""
    pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#FpsIdling

class FpsIdlingMode(enum.IntEnum):
    """FpsIdlingMode is an enum that describes the different modes of idling when rendering the GUI.
    - Sleep: the application will sleep when idling to reduce CPU usage.
    - EarlyReturn: rendering will return immediately when idling.
      This is specifically designed for event-driven, and real-time applications.
      Avoid using it in a tight loop without pauses, as it may cause excessive CPU consumption.
    - Auto: use platform-specific default behavior.
       On most platforms, it will sleep. On Emscripten, `Render()` will return immediately
       to avoid blocking the main thread.
    Note: you can override the default behavior by explicitly setting Sleep or EarlyReturn.
    """

    # Sleep,    /* original C++ signature */
    sleep = enum.auto()  # (= 0)
    # EarlyReturn,    /* original C++ signature */
    early_return = enum.auto()  # (= 1)
    # Auto,    /* original C++ signature */
    # }
    auto = enum.auto()  # (= 2)

class FpsIdling:
    """FpsIdling is a struct that contains Fps Idling parameters"""

    # float fpsIdle = 9.f;    /* original C++ signature */
    # `fpsIdle`: _float, default=9_.
    #  ImGui applications can consume a lot of CPU, since they update the screen
    #  very frequently. In order to reduce the CPU usage, the FPS is reduced when
    #  no user interaction is detected.
    #  This is ok most of the time but if you are displaying animated widgets
    #  (for example a live video), you may want to ask for a faster refresh:
    #  either increase fpsIdle, or set it to 0 for maximum refresh speed
    #  (you can change this value during the execution depending on your application
    #  refresh needs)
    fps_idle: float = 9.0

    # float timeActiveAfterLastEvent = 3.f;    /* original C++ signature */
    # `timeActiveAfterLastEvent`: _float, default=3._.
    #  Time in seconds after the last event before the application is considered idling.
    time_active_after_last_event: float = 3.0

    # bool  enableIdling = true;    /* original C++ signature */
    # `enableIdling`: _bool, default=true_.
    #  Disable idling by setting this to False.
    #  (this can be changed dynamically during execution)
    enable_idling: bool = True

    # bool  isIdling = false;    /* original C++ signature */
    # `isIdling`: bool (dynamically updated during execution)
    #  This bool will be updated during the application execution,
    #  and will be set to True when it is idling.
    is_idling: bool = False

    # bool  rememberEnableIdling = false;    /* original C++ signature */
    # `rememberEnableIdling`: _bool, default=true_.
    #  If True, the last value of enableIdling is restored from the settings at startup.
    remember_enable_idling: bool = False

    # FpsIdlingMode fpsIdlingMode = FpsIdlingMode::Auto;    /* original C++ signature */
    # `fpsIdlingMode`: _FpsIdlingMode, default=FpsIdlingMode::Automatic_.
    # Sets the mode of idling when rendering the GUI (Sleep, EarlyReturn, Automatic)
    fps_idling_mode: FpsIdlingMode = FpsIdlingMode.auto
    # FpsIdling(float fpsIdle = 9.f, float timeActiveAfterLastEvent = 3.f, bool enableIdling = true, bool isIdling = false, bool rememberEnableIdling = false, FpsIdlingMode fpsIdlingMode = FpsIdlingMode::Auto);    /* original C++ signature */
    def __init__(
        self,
        fps_idle: float = 9.0,
        time_active_after_last_event: float = 3.0,
        enable_idling: bool = True,
        is_idling: bool = False,
        remember_enable_idling: bool = False,
        fps_idling_mode: FpsIdlingMode = FpsIdlingMode.auto,
    ) -> None:
        """Auto-generated default constructor with named params"""
        pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#RunnerParams

class RunnerParams:
    """RunnerParams contains the settings and callbacks needed to run an application."""

    # --------------- Callbacks and Window params -------------------

    # RunnerCallbacks callbacks;    /* original C++ signature */
    # `callbacks`: _see runner_callbacks.h_
    # callbacks.ShowGui() will render the gui, ShowMenus() will show the menus, etc.
    callbacks: RunnerCallbacks

    # AppWindowParams appWindowParams;    /* original C++ signature */
    # `appWindowParams`: _see app_window_params.h_
    # application Window Params (position, size, title)
    app_window_params: AppWindowParams

    # ImGuiWindowParams imGuiWindowParams;    /* original C++ signature */
    # `imGuiWindowParams`: _see imgui_window_params.h_
    # imgui window params (use docking, showMenuBar, ProvideFullScreenWindow, etc.)
    imgui_window_params: ImGuiWindowParams

    # --------------- Docking -------------------

    # DockingParams dockingParams;    /* original C++ signature */
    # `dockingParams`: _see docking_params.h_
    # dockable windows content and layout
    docking_params: DockingParams

    # std::vector<DockingParams> alternativeDockingLayouts;    /* original C++ signature */
    # `alternativeDockingLayouts`: _vector<DockingParams>, default=empty_
    # List of possible additional layout for the applications. Only used in advanced
    # cases when several layouts are available.
    alternative_docking_layouts: List[DockingParams]

    # bool rememberSelectedAlternativeLayout = true;    /* original C++ signature */
    # `rememberSelectedAlternativeLayout`: _bool, default=true_
    # Shall the application remember the last selected layout. Only used in advanced
    # cases when several layouts are available.
    remember_selected_alternative_layout: bool = True

    # --------------- Backends -------------------

    # BackendPointers backendPointers;    /* original C++ signature */
    # `backendPointers`: _see backend_pointers.h_
    # A struct that contains optional pointers to the backend implementations.
    # These pointers will be filled when the application starts
    backend_pointers: BackendPointers

    # RendererBackendOptions rendererBackendOptions;    /* original C++ signature */
    # `rendererBackendOptions`: _see renderer_backend_options.h_
    # Options for the renderer backend
    renderer_backend_options: RendererBackendOptions

    # PlatformBackendType platformBackendType = PlatformBackendType::FirstAvailable;    /* original C++ signature */
    # `platformBackendType`: _enum PlatformBackendType, default=PlatformBackendType::FirstAvailable_
    # Select the wanted platform backend type between `Sdl`, `Glfw`.
    # if `FirstAvailable`, Glfw will be preferred over Sdl when both are available.
    # Only useful when multiple backend are compiled and available.
    platform_backend_type: PlatformBackendType = PlatformBackendType.first_available

    # RendererBackendType rendererBackendType = RendererBackendType::FirstAvailable;    /* original C++ signature */
    # `renderingBackendType`: _enum RenderingBackendType, default=RenderingBackendType::FirstAvailable_
    # Select the wanted rendering backend type between `OpenGL3`, `Metal`, `Vulkan`, `DirectX11`, `DirectX12`.
    # if `FirstAvailable`, it will be selected in the order of preference mentioned above.
    # Only useful when multiple rendering backend are compiled and available.
    renderer_backend_type: RendererBackendType = RendererBackendType.first_available

    # --------------- Settings -------------------

    # IniFolderType iniFolderType = IniFolderType::CurrentFolder;    /* original C++ signature */
    # `iniFolderType`: _IniFolderType, default = IniFolderType::CurrentFolder_
    # Sets the folder where imgui will save its params.
    # (possible values are:
    #     CurrentFolder, AppUserConfigFolder, DocumentsFolder,
    #     HomeFolder, TempFolder, AppExecutableFolder)
    # AppUserConfigFolder is
    #     [Home]/AppData/Roaming under Windows,
    #     ~/.config under Linux,
    #     ~/Library/Application Support under macOS
    ini_folder_type: IniFolderType = IniFolderType.current_folder
    # `iniFilename`: _string, default = ""_
    # Sets the ini filename under which imgui will save its params.
    # Its path is relative to the path given by iniFolderType, and can include
    # a subfolder (which will be created if needed).
    # If iniFilename empty, then it will be derived from
    # appWindowParams.windowTitle
    # (if both are empty, the ini filename will be imgui.ini).
    # std::string iniFilename = "";    /* original C++ signature */
    ini_filename: str = ""  # relative to iniFolderType
    # bool iniFilename_useAppWindowTitle = true;    /* original C++ signature */
    # `iniFilename_useAppWindowTitle`: _bool, default = true_.
    # Shall the iniFilename be derived from appWindowParams.windowTitle (if not empty)
    ini_filename_use_app_window_title: bool = True

    # --------------- Exit -------------------

    # bool appShallExit = false;    /* original C++ signature */
    # * `appShallExit`: _bool, default=false_.
    # During execution, set this to True to exit the app.
    # _Note: 'appShallExit' has no effect on Mobile Devices (iOS, Android)
    # and under emscripten, since these apps shall not exit._
    app_shall_exit: bool = False

    # --------------- Idling -------------------

    # FpsIdling fpsIdling;    /* original C++ signature */
    # `fpsIdling`: _FpsIdling_. Idling parameters
    # (set fpsIdling.enableIdling to False to disable Idling)
    fps_idling: FpsIdling

    # DpiAwareParams dpiAwareParams;    /* original C++ signature */
    # --------------- DPI Handling -----------
    # Hello ImGui will try its best to automatically handle DPI scaling for you.
    # If it fails, look at DpiAwareParams (and the corresponding Ini file settings)
    dpi_aware_params: DpiAwareParams

    # --------------- Misc -------------------

    # bool useImGuiTestEngine = false;    /* original C++ signature */
    # `useImGuiTestEngine`: _bool, default=false_.
    # Set this to True if you intend to use Dear ImGui Test & Automation Engine
    #     ( https://github.com/ocornut/imgui_test_engine )
    # HelloImGui must be compiled with the option -DHELLOIMGUI_WITH_TEST_ENGINE=ON
    # See demo in src/hello_imgui_demos/hello_imgui_demo_test_engine.
    # License:
    # imgui_test_engine is subject to a specific license:
    #     https://github.com/ocornut/imgui_test_engine/blob/main/imgui_test_engine/LICENSE.txt)
    # (TL;DR: free for individuals, educational, open-source and small businesses uses.
    #  Paid for larger businesses.)
    use_imgui_test_engine: bool = False

    # int emscripten_fps = 0;    /* original C++ signature */
    # `emscripten_fps`: _int, default = 0_.
    # Set the application refresh rate
    # (only used on emscripten: 0 stands for "let the app or the browser decide")
    emscripten_fps: int = 0

    # RunnerParams(RunnerCallbacks callbacks = RunnerCallbacks(), AppWindowParams appWindowParams = AppWindowParams(), ImGuiWindowParams imGuiWindowParams = ImGuiWindowParams(), DockingParams dockingParams = DockingParams(), std::vector<DockingParams> alternativeDockingLayouts = std::vector<DockingParams>(), bool rememberSelectedAlternativeLayout = true, BackendPointers backendPointers = BackendPointers(), RendererBackendOptions rendererBackendOptions = RendererBackendOptions(), PlatformBackendType platformBackendType = PlatformBackendType::FirstAvailable, RendererBackendType rendererBackendType = RendererBackendType::FirstAvailable, IniFolderType iniFolderType = IniFolderType::CurrentFolder, std::string iniFilename = "", bool iniFilename_useAppWindowTitle = true, bool appShallExit = false, FpsIdling fpsIdling = FpsIdling(), DpiAwareParams dpiAwareParams = DpiAwareParams(), bool useImGuiTestEngine = false, int emscripten_fps = 0);    /* original C++ signature */
    def __init__(
        self,
        callbacks: Optional[RunnerCallbacks] = None,
        app_window_params: Optional[AppWindowParams] = None,
        imgui_window_params: Optional[ImGuiWindowParams] = None,
        docking_params: Optional[DockingParams] = None,
        alternative_docking_layouts: Optional[List[DockingParams]] = None,
        remember_selected_alternative_layout: bool = True,
        backend_pointers: Optional[BackendPointers] = None,
        renderer_backend_options: Optional[RendererBackendOptions] = None,
        platform_backend_type: PlatformBackendType = PlatformBackendType.first_available,
        renderer_backend_type: RendererBackendType = RendererBackendType.first_available,
        ini_folder_type: IniFolderType = IniFolderType.current_folder,
        ini_filename: str = "",
        ini_filename_use_app_window_title: bool = True,
        app_shall_exit: bool = False,
        fps_idling: Optional[FpsIdling] = None,
        dpi_aware_params: Optional[DpiAwareParams] = None,
        use_imgui_test_engine: bool = False,
        emscripten_fps: int = 0,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * callbacks: RunnerCallbacks()
                * appWindowParams: AppWindowParams()
                * imGuiWindowParams: ImGuiWindowParams()
                * dockingParams: DockingParams()
                * alternativeDockingLayouts: List[DockingParams]()
                * backendPointers: BackendPointers()
                * rendererBackendOptions: RendererBackendOptions()
                * fpsIdling: FpsIdling()
                * dpiAwareParams: DpiAwareParams()
        """
        pass

# @@md

# @@md#IniIniSettingsLocation

# std::string IniSettingsLocation(const RunnerParams& runnerParams);    /* original C++ signature */
def ini_settings_location(runner_params: RunnerParams) -> str:
    """IniSettingsLocation returns the path to the ini file for the application settings."""
    pass

# bool HasIniSettings(const RunnerParams& runnerParams);    /* original C++ signature */
def has_ini_settings(runner_params: RunnerParams) -> bool:
    """HasIniSettings returns True if the ini file for the application settings exists."""
    pass

# void DeleteIniSettings(const RunnerParams& runnerParams);    /* original C++ signature */
def delete_ini_settings(runner_params: RunnerParams) -> None:
    """DeleteIniSettings deletes the ini file for the application settings."""
    pass

# @@md

# --------------------------------------------------------------------------------------------------------------------

# @@md#SimpleRunnerParams

class SimpleRunnerParams:
    """SimpleRunnerParams is a struct that contains simpler params adapted for simple use cases.
    For example, this is sufficient to run an application:
        ```cpp
        None MyGui() {
            ImGui::Text("Hello, world");
            if (ImGui::Button("Exit"))
                HelloImGui::GetRunnerParams()->appShallExit = True;
        }

        int main(){
            auto params = HelloImGui::SimpleRunnerParams {
                .guiFunction = MyGui, .windowSizeAuto = True, .windowTitle = "Example"
            };
            HelloImGui::Run(params);
        }
        ```
    """

    # VoidFunction guiFunction = EmptyVoidFunction();    /* original C++ signature */
    # `guiFunction`: _VoidFunction_.
    #  Function that renders the Gui.
    gui_function: VoidFunction = EmptyVoidFunction()
    # std::string windowTitle = "";    /* original C++ signature */
    # `windowTitle`: _string, default=""_.
    #  Title of the application window
    window_title: str = ""

    # bool windowSizeAuto = false;    /* original C++ signature */
    # `windowSizeAuto`: _bool, default=false_.
    #  If True, the size of the window will be computed from its widgets.
    window_size_auto: bool = False

    # bool windowRestorePreviousGeometry = false;    /* original C++ signature */
    # `windowRestorePreviousGeometry`: _bool, default=true_.
    #  If True, restore the size and position of the window between runs.
    window_restore_previous_geometry: bool = False

    # ScreenSize windowSize = DefaultWindowSize;    /* original C++ signature */
    # `windowSize`: _ScreenSize, default={800, 600}_.
    #  Size of the window
    # The size will be handled as if it was specified for a 96PPI screen
    # (i.e. a given size will correspond to the same physical size on different screens, whatever their DPI)
    window_size: ScreenSize = DefaultWindowSize

    # float fpsIdle = 9.f;    /* original C++ signature */
    # `fpsIdle`: _float, default=9_.
    #  FPS of the application when idle (set to 0 for full speed).
    fps_idle: float = 9.0

    # bool  enableIdling = true;    /* original C++ signature */
    # `enableIdling`: _bool, default=true_.
    #  Disable idling at startup by setting this to False
    #  When running, use:
    #      HelloImGui::GetRunnerParams()->fpsIdling.enableIdling = False;
    enable_idling: bool = True

    # RunnerParams ToRunnerParams() const;    /* original C++ signature */
    def to_runner_params(self) -> RunnerParams:
        pass
    # SimpleRunnerParams(VoidFunction guiFunction = EmptyVoidFunction(), std::string windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, ScreenSize windowSize = DefaultWindowSize, float fpsIdle = 9.f, bool enableIdling = true);    /* original C++ signature */
    def __init__(
        self,
        gui_function: Optional[VoidFunction] = None,
        window_title: str = "",
        window_size_auto: bool = False,
        window_restore_previous_geometry: bool = False,
        window_size: Optional[ScreenSize] = None,
        fps_idle: float = 9.0,
        enable_idling: bool = True,
    ) -> None:
        """Auto-generated default constructor with named params


        Python bindings defaults:
            If any of the params below is None, then its default value below will be used:
                * guiFunction: EmptyVoidFunction()
                * windowSize: DefaultWindowSize
        """
        pass

# @@md

# void BeginGroupColumn();     /* original C++ signature */
def begin_group_column() -> None:
    """calls ImGui::BeginGroup()"""
    pass

# void EndGroupColumn();       /* original C++ signature */
def end_group_column() -> None:
    """calls ImGui::EndGroup() + ImGui::SameLine()"""
    pass

# @@md#WidgetWithResizeHandle

# ImVec2 WidgetWithResizeHandle(    /* original C++ signature */
#         const char* id,
#         VoidFunction widgetGuiFunction,
#         float handleSizeEm = 1.0f,
#         std::optional<VoidFunction> onItemResized = std::nullopt,
#         std::optional<VoidFunction> onItemHovered = std::nullopt
#         );
def widget_with_resize_handle(
    id: str,
    widget_gui_function: VoidFunction,
    handle_size_em: float = 1.0,
    on_item_resized: Optional[VoidFunction] = None,
    on_item_hovered: Optional[VoidFunction] = None,
) -> ImVec2:
    """WidgetWithResizeHandle: adds a resize handle to a widget
    Example usage with ImPlot:
           None gui()
           {
               static ImVec2 widget_size(200, 200);
               auto myWidgetFunction = []()
               {
                   if (ImPlot::BeginPlot("My Plot", widget_size)) {
                       ImPlot::PlotLine("My Line", x.data(), y.data(), 1000);
                       ImPlot::EndPlot();
                   }
               };
               widget_size = widget_with_resize_handle("plot", myWidgetFunction);
           }
    """
    pass

# @@md

# --------------------------------------------------------------------------------------------

# @@md#InputTextResizable

class InputTextData:
    """`InputTextResizable`: displays a resizable text input widget

    The `InputTextResizable` widget allows you to create a text input field that can be resized by the user.
    It supports both single-line and multi-line text input.
    Note: the size of the widget is expressed in em units.
    **Usage example:**
       C++:
          ```cpp
        // Somewhere in the application state
          (static) InputTextData textInput("My text", True, ImVec2(10, 3));
        // In the GUI function
          bool changed = InputTextResizable("Label", &textInput);
          ```
       Python:
          ```python
          # Somewhere in the application state
          text_input = hello_imgui.InputTextData("My text", multiline=True, size_em=ImVec2(10, 3))
          # In the GUI function
          changed = hello_imgui.input_text_resizable("Label", text_input)
          ```
    """

    # std::string Text;    /* original C++ signature */
    # The text edited in the input field
    text: str

    # std::string Hint;    /* original C++ signature */
    # An optional hint displayed when the input field is empty
    # (only works for single-line text input)
    hint: str

    # bool Multiline = false;    /* original C++ signature */
    # If True, the input field is multi-line
    multiline: bool = False

    # bool Resizable = true;    /* original C++ signature */
    # If True, the input field is resizable
    resizable: bool = True

    # ImVec2 SizeEm = ImVec2(0, 0);    /* original C++ signature */
    # The size of the input field in em units
    size_em: ImVec2 = ImVec2(0, 0)

    # InputTextData(const std::string& text = "", bool multiline = false, ImVec2 size_em = ImVec2(0, 0)) : Text(text), Multiline(multiline), SizeEm(size_em) {}    /* original C++ signature */
    def __init__(
        self,
        text: str = "",
        multiline: bool = False,
        size_em: Optional[ImVec2Like] = None,
    ) -> None:
        """Python bindings defaults:
        If size_em is None, then its default value will be: ImVec2(0, 0)
        """
        pass

# bool InputTextResizable(const char* label, InputTextData* textInput);    /* original C++ signature */
def input_text_resizable(label: str, text_input: InputTextData) -> bool:
    pass

# DictTypeInputTextData InputTextDataToDict(const InputTextData& data);    /* original C++ signature */
def input_text_data_to_dict(data: InputTextData) -> DictTypeInputTextData:
    pass

# InputTextData InputTextDataFromDict(const DictTypeInputTextData& dict);    /* original C++ signature */
def input_text_data_from_dict(dict: DictTypeInputTextData) -> InputTextData:
    pass

# to/from string
# std::string InputTextDataToString(const InputTextData& data);    /* original C++ signature */
def input_text_data_to_string(data: InputTextData) -> str:
    pass

# InputTextData InputTextDataFromString(const std::string& str);    /* original C++ signature */
def input_text_data_from_string(str: str) -> InputTextData:
    pass

# @@md

# ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#                       hello_imgui.h continued                                                                //
# //////////////////////////////////////////////////////////////////////////////////////////////////////////////

# =========================== HelloImGui::Run ==================================
# *
# @@md#HelloImGui::Run
#
# __HelloImGui::Run()__ will run an application with a single call.
#
# Three signatures are provided:
#
# * `HelloImGui::Run(RunnerParams &)`: full signature, the most customizable version.
#  Runs an application whose params and Gui are provided by runnerParams.
#
# * `HelloImGui::Run(const SimpleRunnerParams&)`:
#  Runs an application, using simpler params.
#
# * `HelloImGui::Run(guiFunction, windowTitle, windowSize, windowSizeAuto=False, restoreLastWindowGeometry=False, fpsIdle=10)`
#  Runs an application, by providing the Gui function, the window title, etc.
#
# Although the API is extremely simple, it is highly customizable, and you can set many options by filling
# the elements in the `RunnerParams` struct, or in the simpler  `SimpleRunnerParams`.
#
# __HelloImGui::GetRunnerParams()__  will return the runnerParams of the current application.
#
# @@md
#

# void Run(RunnerParams &runnerParams);    /* original C++ signature */
@overload
def run(runner_params: RunnerParams) -> None:
    """`HelloImGui::Run(RunnerParams &)`: full signature, the most customizable version.
    Runs an application whose params and Gui are provided by runnerParams.
    """
    pass

# void Run(const SimpleRunnerParams &simpleParams);    /* original C++ signature */
@overload
def run(simple_params: SimpleRunnerParams) -> None:
    """`HelloImGui::Run(const SimpleRunnerParams&)`:
    Runs an application, using simpler params.
    """
    pass

# void Run(    /* original C++ signature */
#     const VoidFunction &guiFunction,
#     const std::string &windowTitle = "",
#     bool windowSizeAuto = false,
#     bool windowRestorePreviousGeometry = false,
#     const ScreenSize &windowSize = DefaultWindowSize,
#     float fpsIdle = 10.f
# );
@overload
def run(
    gui_function: VoidFunction,
    window_title: str = "",
    window_size_auto: bool = False,
    window_restore_previous_geometry: bool = False,
    window_size: Optional[ScreenSize] = None,
    fps_idle: float = 10.0,
) -> None:
    """Runs an application, by providing the Gui function, the window title, etc.


    Python bindings defaults:
        If windowSize is None, then its default value will be: DefaultWindowSize
    """
    pass

# =========================== HelloImGui::ManualRender ==================================
# @@md#HelloImGui::ManualRender

# @@md

# ============================== Utility functions ===============================

# @@md#UtilityFunctions

# RunnerParams* GetRunnerParams();    /* original C++ signature */
def get_runner_params() -> RunnerParams:
    """`GetRunnerParams()`:  a convenience function that will return the runnerParams
    of the current application
    """
    pass

# bool IsUsingHelloImGui();    /* original C++ signature */
def is_using_hello_imgui() -> bool:
    """`IsUsingHelloImGui()`: returns True if the application is using HelloImGui"""
    pass

# float FrameRate(float durationForMean = 0.5f);    /* original C++ signature */
def frame_rate(duration_for_mean: float = 0.5) -> float:
    """`FrameRate(durationForMean = 0.5)`: Returns the current FrameRate.
    May differ from ImGui::GetIO().FrameRate, since one can choose the duration
    for the calculation of the mean value of the fps
    Returns the current FrameRate. May differ from ImGui::GetIO().FrameRate,
    since one can choose the duration for the calculation of the mean value of the fps
    (Will only lead to accurate values if you call it at each frame)
    """
    pass

# ImGuiTestEngine* GetImGuiTestEngine();    /* original C++ signature */
def get_imgui_test_engine() -> ImGuiTestEngine:
    """`ImGuiTestEngine* GetImGuiTestEngine()`: returns a pointer to the global instance
    of ImGuiTestEngine that was initialized by HelloImGui
    (iif ImGui Test Engine is active).
    """
    pass

# std::string GetBackendDescription();    /* original C++ signature */
def get_backend_description() -> str:
    """`GetBackendDescription()`: returns a string with the backend info
    Could be for example:
        "Glfw - OpenGL3"
        "Glfw - Metal"
        "Sdl - Vulkan"
    """
    pass

# void ChangeWindowSize(const ScreenSize &windowSize);    /* original C++ signature */
def change_window_size(window_size: ScreenSize) -> None:
    """`ChangeWindowSize(const ScreenSize &windowSize)`: sets the window size
    (useful if you want to change the window size during execution)
    """
    pass

# void UseWindowFullMonitorWorkArea();    /* original C++ signature */
def use_window_full_monitor_work_area() -> None:
    """`UseWindowFullMonitorWorkArea()`: sets the window size to the monitor work area
    (useful if you want to change the window size during execution)
    """
    pass

# @@md

# ============================== Layout Utils =============================

# @@md#HelloImGui::Layouts

# In advanced cases when several layouts are available, you can switch between layouts.
# See demo inside
#     https://github.com/pthom/hello_imgui/tree/master/src/hello_imgui_demos/hello_imgui_demodocking/hello_imgui_demodocking.main.cpp

# void           SwitchLayout(const std::string& layoutName);    /* original C++ signature */
def switch_layout(layout_name: str) -> None:
    """`SwitchLayout(layoutName)`
    Changes the application current layout. Only used in advanced cases
    when several layouts are available, i.e. if you filled
        runnerParams.alternativeDockingLayouts.
    """
    pass

# std::string    CurrentLayoutName();    /* original C++ signature */
def current_layout_name() -> str:
    """`CurrentLayoutName()`: returns the name of the current layout"""
    pass

# void AddDockableWindow(const DockableWindow& dockableWindow, bool forceDockspace = false);    /* original C++ signature */
def add_dockable_window(
    dockable_window: DockableWindow, force_dockspace: bool = False
) -> None:
    """`AddDockableWindow()`: will add a dockable window to the current layout.
    Will dock the window to the dockspace it belongs to if forceDockspace is True,
    otherwise will dock it to the last space it was docked to (using saved settings)
    """
    pass

# void RemoveDockableWindow(const std::string& dockableWindowName);    /* original C++ signature */
def remove_dockable_window(dockable_window_name: str) -> None:
    """`RemoveDockableWindow()`: will remove a dockable window from the current layout.
    (dockableWindowName is the label of the window, as provided in the DockableWindow struct)
    """
    pass

# @@md

# ============================== User prefs Utils =============================

# @@md#HelloImGui::UserPref

# You may store additional user settings in the application settings.
# This is provided as a convenience only, and it is not intended to store large
# quantities of text data. Use sparingly.

# void        SaveUserPref(const std::string& userPrefName, const std::string& userPrefContent);    /* original C++ signature */
def save_user_pref(user_pref_name: str, user_pref_content: str) -> None:
    """`SaveUserPref(string userPrefName, string userPrefContent)`:
    Shall be called in the callback runnerParams.callbacks.BeforeExit
    """
    pass

# std::string LoadUserPref(const std::string& userPrefName);    /* original C++ signature */
def load_user_pref(user_pref_name: str) -> str:
    """`string LoadUserPref(string& userPrefName)`
    Shall be called in the callback runnerParams.callbacks.PostInit
    """
    pass

# @@md

# ============================== Menus defaults =============================

# *
# @@md#MenuIntro
#
# Hello ImGui provides a default menu and status bar, which you can customize by using the params:
#        `RunnerParams.imGuiWindowParams.` `showMenuBar` / `showMenu_App` / `showMenu_View`
#
# If you want to fully customize the menu:
# * set `showMenuBar` to True, then set `showMenu_App` and `showMenu_View` params to False
# * implement the callback `RunnerParams.callbacks.ShowMenus`:
#  it can optionally call `ShowViewMenu` and `ShowAppMenu` (see below).
#
# @@md
#
# @@md#MenuFunctions

# void ShowViewMenu(RunnerParams & runnerParams);    /* original C++ signature */
def show_view_menu(runner_params: RunnerParams) -> None:
    """`ShowViewMenu(RunnerParams & runnerParams)`:
    shows the View menu (where you can select the layout and docked windows visibility
    """
    pass

# void ShowAppMenu(RunnerParams & runnerParams);    /* original C++ signature */
def show_app_menu(runner_params: RunnerParams) -> None:
    """`ShowAppMenu(RunnerParams & runnerParams)`:
    shows the default App menu (including the Quit item)
    """
    pass

# @@md

# <submodule imgui_default_settings>
class imgui_default_settings:  # Proxy class that introduces typings for the *submodule* imgui_default_settings
    pass  # (This corresponds to a C++ namespace. All method are static!)
    """ namespace ImGuiDefaultSettings"""
    # void LoadDefaultFont_WithFontAwesomeIcons();    /* original C++ signature */
    @staticmethod
    def load_default_font_with_font_awesome_icons() -> None:
        """LoadDefaultFont_WithFontAwesome will load from assets/fonts and reverts to the imgui embedded font if not found."""
        pass
    # void SetupDefaultImGuiConfig();    /* original C++ signature */
    @staticmethod
    def setup_default_imgui_config() -> None:
        pass
    # void SetupDefaultImGuiStyle();    /* original C++ signature */
    #     }
    @staticmethod
    def setup_default_imgui_style() -> None:
        pass

# </submodule imgui_default_settings>

# <submodule manual_render>
class manual_render:  # Proxy class that introduces typings for the *submodule* manual_render
    pass  # (This corresponds to a C++ namespace. All method are static!)
    """ namespace ManualRender"""
    # HelloImGui::ManualRender is a namespace that groups functions, allowing fine-grained control over the rendering process:
    # - It is customizable like HelloImGui::Run: initialize it with `RunnerParams` or `SimpleRunnerParams`
    # - `ManualRender::Render()` will render the application for one frame:
    # - Ensure that `ManualRender::Render()` is triggered regularly (e.g., through a loop or other mechanism)
    #   to maintain responsiveness. This method must be called on the main thread.
    #
    # A typical use case is:
    # C++
    #        ```cpp
    #        HelloImGui::RunnerParams runnerParams;
    #        runnerParams.callbacks.ShowGui = ...; // your GUI function
    #      // Optionally, choose between Sleep, EarlyReturn, or Auto for fps idling mode:
    #      // runnerParams.fpsIdling.fpsIdlingMode = HelloImGui::FpsIdlingMode::Sleep; // or EarlyReturn, Auto
    #        HelloImGui::ManualRender::SetupFromRunnerParams(runnerParams);
    #        while (!HelloImGui::GetRunnerParams()->appShallExit)
    #        {
    #            HelloImGui::ManualRender::Render();
    #        }
    #        HelloImGui::ManualRender::TearDown();
    #        ```
    # Python:
    #        ```python
    #        runnerParams = HelloImGui.RunnerParams()
    #        runnerParams.callbacks.show_gui = ... # your GUI function
    #        while not hello_imgui.get_runner_params().app_shall_exit:
    #            hello_imgui.manual_render.render()
    #        hello_imgui.manual_render.tear_down()
    #        ```
    #
    # **Notes:**
    #  1. Depending on the configuration (`runnerParams.fpsIdling.fpsIdlingMode`), `HelloImGui` may enter
    #     an idle state to reduce CPU usage, if no events are received (e.g., no input or interaction).
    #     In this case, `Render()` will either sleep or return immediately.
    #     By default,
    #       - On Emscripten, `ManualRender::Render()` will return immediately to avoid blocking the main thread.
    #       - On other platforms, it will sleep
    #  2. If initialized with `RunnerParams`, a copy of the `RunnerParams` will be made
    #     (which can be accessed with `HelloImGui::GetRunnerParams()`).

    # void SetupFromRunnerParams(const RunnerParams& runnerParams);    /* original C++ signature */
    @staticmethod
    def setup_from_runner_params(runner_params: RunnerParams) -> None:
        """Initializes the rendering with the full customizable `RunnerParams`.
        This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).
        A distinct copy of `RunnerParams` is stored internally.
        """
        pass
    # void SetupFromSimpleRunnerParams(const SimpleRunnerParams& simpleParams);    /* original C++ signature */
    @staticmethod
    def setup_from_simple_runner_params(simple_params: SimpleRunnerParams) -> None:
        """Initializes the rendering with `SimpleRunnerParams`.
        This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).
        """
        pass
    # void SetupFromGuiFunction(    /* original C++ signature */
    #         const VoidFunction& guiFunction,
    #         const std::string& windowTitle = "",
    #         bool windowSizeAuto = false,
    #         bool windowRestorePreviousGeometry = false,
    #         const ScreenSize& windowSize = DefaultWindowSize,
    #         float fpsIdle = 10.f
    #     );
    @staticmethod
    def setup_from_gui_function(
        gui_function: VoidFunction,
        window_title: str = "",
        window_size_auto: bool = False,
        window_restore_previous_geometry: bool = False,
        window_size: Optional[ScreenSize] = None,
        fps_idle: float = 10.0,
    ) -> None:
        """Initializes the renderer with a simple GUI function and additional parameters.
         This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).


        Python bindings defaults:
            If windowSize is None, then its default value will be: DefaultWindowSize
        """
        pass
    # void Render();    /* original C++ signature */
    @staticmethod
    def render() -> None:
        """Renders the current frame. Should be called regularly to maintain the application's responsiveness."""
        pass
    # void TearDown();    /* original C++ signature */
    # }
    @staticmethod
    def tear_down() -> None:
        """Tears down the renderer and releases all associated resources.
        This will release the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).
        After calling `TearDown()`, the InitFromXXX can be called with new parameters.
        """
        pass

# </submodule manual_render>
####################    </generated_from:hello_imgui_amalgamation.h>    ####################

# </litgen_stub>
# !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

def final_app_window_screenshot() -> np.ndarray:
    """Return a screenshot of the final screen of the last (exited) app"""
    pass

def final_app_window_screenshot_framebuffer_scale() -> float:
    """Returns the scale of the framebuffer used to take the screenshot"""
    pass
