// Part of ImGui Bundle - MIT License - Copyright (c) 2022-2024 Pascal Thomet - https://github.com/pthom/imgui_bundle
#include <nanobind/nanobind.h>
#include <nanobind/stl/string.h>
#include <nanobind/stl/array.h>
#include <nanobind/stl/optional.h>
#include <nanobind/stl/vector.h>
#include <nanobind/stl/function.h>
#include <nanobind/stl/map.h>
#include <nanobind/stl/variant.h>
#include <nanobind/trampoline.h>
#include <nanobind/stl/shared_ptr.h>
#include <nanobind/stl/unique_ptr.h>
#include <nanobind/stl/tuple.h>
#include <nanobind/make_iterator.h>
#include <nanobind/ndarray.h>

#include <cstring> // memcpy

#include "hello_imgui/hello_imgui.h"
#include "hello_imgui/hello_imgui_screenshot.h"
#include "imgui_internal.h"
#ifdef HELLOIMGUI_WITH_TEST_ENGINE
#include "imgui_test_engine/imgui_te_engine.h"
#include "imgui_test_engine/imgui_te_internal.h"
#else
struct ImGuiTestEngine {};
#endif

namespace nb = nanobind;


// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// <litgen_glue_code>  // Autogenerated code below! Do not edit!

// </litgen_glue_code> // Autogenerated code end
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

// Returns the final screenshot of the app window as a numpy array
nb::handle FinalAppWindowScreenshot()
{
    // Get the image buffer from HelloImGui
    HelloImGui::ImageBuffer imageBuffer = HelloImGui::FinalAppWindowScreenshotRgbBuffer();

    // Calculate total size
    size_t total_size = imageBuffer.bufferRgb.size();

    // Allocate new buffer
    uint8_t* ndarray_buffer = new uint8_t[total_size];

    // Copy data
    std::memcpy(ndarray_buffer, imageBuffer.bufferRgb.data(), total_size);

    // Create a capsule to manage buffer lifetime
    nb::capsule owner(ndarray_buffer, [](void* p) noexcept {
        delete[] static_cast<uint8_t*>(p);
    });

    auto array = nb::ndarray<uint8_t>(
        ndarray_buffer,                                      // Data pointer
        {imageBuffer.height, imageBuffer.width, (size_t)3},  // Shape
        owner,                                               // Owner capsule
        {(int64_t)(3 * imageBuffer.width), (int64_t)3, 1},   // Strides in elements
        nb::dtype<uint8_t>(),                              // Data type
        0,                                                 // Device type (CPU)
        0,                                                 // Device ID
        'C'                                                // Order ('C' for C-contiguous)
    );

    // Properly export the ndarray
    return nb::detail::ndarray_export(
        array.handle(),
        nb::numpy::value,
        nb::rv_policy::move,  // Transfer ownership to Python
        nullptr               // No cleanup list
    );
}

namespace HelloImGui { float FinalAppWindowScreenshotFramebufferScale(); }


//
// Tooling to expose set_load_asset_file_data_function
//
static nb::callable g_load_asset_file_data_function; // Keep a strong ref to the Python callback so it isn’t GC‑ed.
// Helper: copy a Python `bytes` object into a freshly‑malloc’ed buffer so that `FreeAssetFileData()` can later `free()` it.
static HelloImGui::AssetFileData to_asset_file_data(const nb::bytes &py_bytes)
{
    const size_t sz   = nb::len(py_bytes);
    void *buf         = ::malloc(sz);
    if (!buf)
        throw std::bad_alloc{};
    std::memcpy(buf, py_bytes.c_str(), sz);
    return {buf, sz};
}
// Exposed function to set the Python function that will be called to load asset files.
void set_load_asset_file_data_function(const nb::callable &py_func)
{
    g_load_asset_file_data_function = py_func;  // keep alive

    HelloImGui::SetLoadAssetFileDataFunction(
        [](const char *c_path) -> HelloImGui::AssetFileData
        {
            nb::gil_scoped_acquire gil;               // call back into Python
            nb::object ret = g_load_asset_file_data_function(nb::str(c_path));
            if (!nb::isinstance<nb::bytes>(ret))
                throw nb::type_error("loader must return a 'bytes' object");
            nb::bytes py_bytes = nb::cast<nb::bytes>(ret);
            return to_asset_file_data(py_bytes);
        });
}


void py_init_module_hello_imgui(nb::module_& m)
{
    using ImWcharPair = std::array<ImWchar, 2>;
    using ScreenPosition = std::array<int, 2>;
    using ScreenSize = std::array<int, 2>;
    using VoidFunction = std::function<void(void)>;
    using AnyEventCallback = std::function<bool(void * backendEvent)>;
    using DockSpaceName = std::string;

    m.def("final_app_window_screenshot", FinalAppWindowScreenshot);
    m.def("final_app_window_screenshot_framebuffer_scale", HelloImGui::FinalAppWindowScreenshotFramebufferScale);

    m.def("get_glfw_window_address", []() {
        return (size_t) HelloImGui::GetRunnerParams()->backendPointers.glfwWindow;
    });
    m.def("get_sdl_window_address", []() {
        return (size_t) HelloImGui::GetRunnerParams()->backendPointers.sdlWindow;
    });
    m.def("get_sdl_gl_context", []() {
        return (size_t) HelloImGui::GetRunnerParams()->backendPointers.sdlGlContext;
    });

    m.def("set_load_asset_file_data_function", &set_load_asset_file_data_function);


    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // <litgen_pydef> // Autogenerated code below! Do not edit!
    ////////////////////    <generated_from:hello_imgui_amalgamation.h>    ////////////////////
    auto pyClassDpiAwareParams =
        nb::class_<HelloImGui::DpiAwareParams>
            (m, "DpiAwareParams", "\n Hello ImGui will try its best to automatically handle DPI scaling for you.\n\n Parameter to change the scaling behavior:\n ------------------------------------------\n - `dpiWindowSizeFactor`:\n        factor by which window size should be multiplied\n    By default, Hello ImGui will compute it automatically, when it is set to 0.\n\n\n How to set manually:\n ---------------------------------\n If it fails (i.e. your window and/or fonts are too big or too small),\n you may set them manually:\n    (1) Either by setting them programmatically in your application\n        (set their values in `runnerParams.dpiAwareParams`)\n    (2) Either by setting them in a `hello_imgui.ini` file. See hello_imgui/hello_imgui_example.ini for more info\n Note: if several methods are used, the order of priority is (1) > (2)\n\n For more information, see the documentation on DPI handling, here: https://pthom.github.io/hello_imgui/book/doc_api.html#handling-screens-with-high-dpi\n")
        .def("__init__", [](HelloImGui::DpiAwareParams * self, float dpiWindowSizeFactor = 0.0f)
        {
            new (self) HelloImGui::DpiAwareParams();  // placement new
            auto r_ctor_ = self;
            r_ctor_->dpiWindowSizeFactor = dpiWindowSizeFactor;
        },
        nb::arg("dpi_window_size_factor") = 0.0f
        )
        .def_rw("dpi_window_size_factor", &HelloImGui::DpiAwareParams::dpiWindowSizeFactor, " `dpiWindowSizeFactor`\n     factor by which window size should be multiplied to get a similar\n     physical size on different OSes (as if they were all displayed on a 96 PPI screen).\n     This affects the size of native app windows,\n     but *not* imgui Windows, and *not* the size of widgets and text.\n  In a standard environment (i.e. outside of Hello ImGui), an application with a size of 960x480 pixels,\n  may have a physical size (in mm or inches) that varies depending on the screen DPI, and the OS.\n\n  Inside Hello ImGui, the window size is always treated as targeting a 96 PPI screen, so that its size will\n  look similar whatever the OS and the screen DPI.\n  In our example, our 960x480 pixels window will try to correspond to a 10x5 inches window\n\n  Hello ImGui does its best to compute it on all OSes.\n  However, if it fails you may set its value manually.\n  If it is set to 0, Hello ImGui will compute it automatically,\n  and the resulting value will be stored in `dpiWindowSizeFactor`.")
        .def("dpi_font_loading_factor",
            &HelloImGui::DpiAwareParams::DpiFontLoadingFactor, " `DpiFontLoadingFactor`\n     factor by which font size should be multiplied at loading time to get a similar visible size on different OSes.\n     This is equal to dpiWindowSizeFactor\n  The size will be equivalent to a size given for a 96 PPI screen")
        ;


    m.def("em_to_vec2",
        nb::overload_cast<float, float>(HelloImGui::EmToVec2), nb::arg("x"), nb::arg("y"));

    m.def("em_to_vec2",
        nb::overload_cast<ImVec2>(HelloImGui::EmToVec2), nb::arg("v"));

    m.def("em_size",
        nb::overload_cast<>(HelloImGui::EmSize), "__HelloImGui::EmSize()__ returns the visible font size on the screen.");

    m.def("em_size",
        nb::overload_cast<float>(HelloImGui::EmSize),
        nb::arg("nb_lines"),
        "__HelloImGui::EmSize(nbLines)__ returns a size corresponding to nbLines text lines");

    m.def("pixels_to_em",
        HelloImGui::PixelsToEm,
        nb::arg("pixels"),
        "__HelloImGui::PixelToEm()__ converts a Vec2 in pixels coord to a Vec2 in em units");

    m.def("pixel_size_to_em",
        HelloImGui::PixelSizeToEm,
        nb::arg("pixel_size"),
        "__HelloImGui::PixelSizeToEm()__ converts a size in pixels coord to a size in em units");

    m.def("get_dpi_aware_params",
        HelloImGui::GetDpiAwareParams,
        " Returns the current DpiAwareParams, which are used\n for font loading and window size scaling",
        nb::rv_policy::reference);


    m.def("dpi_font_loading_factor",
        HelloImGui::DpiFontLoadingFactor, " Multiply font sizes by this factor when loading fonts manually with ImGui::GetIO().Fonts->AddFont...\n (HelloImGui::LoadFontTTF does this by default)");

    m.def("dpi_window_size_factor",
        HelloImGui::DpiWindowSizeFactor, " DpiWindowSizeFactor() is the factor by which window size should be multiplied to get a similar visible size on different OSes.\n It returns ApplicationScreenPixelPerInch / 96 under windows and linux. Under macOS, it will return 1.");


    auto pyClassAssetFileData =
        nb::class_<HelloImGui::AssetFileData>
            (m, "AssetFileData", "")
        .def("__init__", [](HelloImGui::AssetFileData * self, size_t dataSize = 0)
        {
            new (self) HelloImGui::AssetFileData();  // placement new
            auto r_ctor_ = self;
            r_ctor_->dataSize = dataSize;
        },
        nb::arg("data_size") = 0
        )
        .def_rw("data", &HelloImGui::AssetFileData::data, "")
        .def_rw("data_size", &HelloImGui::AssetFileData::dataSize, "")
        ;


    m.def("load_asset_file_data",
        HelloImGui::LoadAssetFileData,
        nb::arg("asset_path"),
        " LoadAssetFileData(const char *assetPath)`\n Will load an entire asset file into memory. This works on all platforms,\n including android.\n You *have* to call FreeAssetFileData to free the memory, except if you use\n ImGui::GetIO().Fonts->AddFontFromMemoryTTF, which will take ownership of the\n data and free it for you.\n This function can be redirected with setLoadAssetFileDataFunction. If not redirected,\n it calls DefaultLoadAssetFileData.");

    m.def("free_asset_file_data",
        HelloImGui::FreeAssetFileData,
        nb::arg("asset_file_data"),
        " FreeAssetFileData(AssetFileData *)\n Will free the memory.\n Note: \"ImGui::GetIO().Fonts->AddFontFromMemoryTTF\" takes ownership of the data\n and will free the memory for you.");

    m.def("default_load_asset_file_data",
        HelloImGui::DefaultLoadAssetFileData,
        nb::arg("asset_path"),
        " This function actually performs the asset load, as described in\n LoadAssetFileData");

    m.def("asset_file_full_path",
        HelloImGui::AssetFileFullPath,
        nb::arg("asset_relative_filename"), nb::arg("assert_if_not_found") = true,
        "`std::string AssetFileFullPath(const std::string& assetRelativeFilename)`\n will return the path to assets.\n\n This works under all platforms *except Android*\n For compatibility with Android and other platforms, prefer to use `LoadAssetFileData`\n whenever possible.\n    * Under iOS it will give a path in the app bundle (/private/XXX/....)\n    * Under emscripten, it will be stored in the virtual filesystem at \"/\"\n    * Under Android, assetFileFullPath is *not* implemented, and will throw an error:\n      assets can be compressed under android, and you can't use standard file operations!\n      Use LoadAssetFileData instead");

    m.def("asset_exists",
        HelloImGui::AssetExists,
        nb::arg("asset_relative_filename"),
        "Returns True if this asset file exists");

    m.def("set_assets_folder",
        nb::overload_cast<const std::string &>(HelloImGui::SetAssetsFolder),
        nb::arg("folder"),
        " Sets the assets folder location\n (when using this, automatic assets installation on mobile platforms may not work)");

    m.def("set_assets_folder",
        nb::overload_cast<const char *>(HelloImGui::SetAssetsFolder),
        nb::arg("folder"),
        "Legacy API, kept for compatibility");

    m.def("asset_file_full_path",
        HelloImGui::assetFileFullPath, nb::arg("asset_relative_filename"), nb::arg("assert_if_not_found") = true);

    m.def("override_assets_folder",
        HelloImGui::overrideAssetsFolder,
        nb::arg("folder"),
        "synonym of SetAssetsFolder");


    auto pyEnumLogLevel =
        nb::enum_<HelloImGui::LogLevel>(m, "LogLevel", nb::is_arithmetic(), "")
            .value("debug", HelloImGui::LogLevel::Debug, "")
            .value("info", HelloImGui::LogLevel::Info, "")
            .value("warning", HelloImGui::LogLevel::Warning, "")
            .value("error", HelloImGui::LogLevel::Error, "");


    m.def("log",
        [](HelloImGui::LogLevel level, const char * const format)
        {
            auto Log_adapt_variadic_format = [](HelloImGui::LogLevel level, const char * const format)
            {
                HelloImGui::Log(level, "%s", format);
            };

            Log_adapt_variadic_format(level, format);
        },     nb::arg("level"), nb::arg("format"));

    m.def("log_clear",
        HelloImGui::LogClear);

    m.def("log_gui",
        [](const std::optional<const ImVec2> & size = std::nullopt)
        {
            auto LogGui_adapt_mutable_param_with_default_value = [](const std::optional<const ImVec2> & size = std::nullopt)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0.f, 0.f);
                }();

                HelloImGui::LogGui(size_or_default);
            };

            LogGui_adapt_mutable_param_with_default_value(size);
        },
        nb::arg("size").none() = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0., 0.)");


    m.def("image_from_asset",
        [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt)
        {
            auto ImageFromAsset_adapt_mutable_param_with_default_value = [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1,1);
                }();

                HelloImGui::ImageFromAsset(assetPath, size_or_default, uv0_or_default, uv1_or_default);
            };

            ImageFromAsset_adapt_mutable_param_with_default_value(assetPath, size, uv0, uv1);
        },
        nb::arg("asset_path"), nb::arg("size").none() = nb::none(), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(),
        " `HelloImGui::ImageFromAsset(const char *assetPath, size, ...)`:\n will display a static image from the assets.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * size: ImVec2(0, 0)\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1,1)");

    m.def("image_from_asset_with_bg",
        [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt, const std::optional<const ImVec4> & border_col = std::nullopt)
        {
            auto ImageFromAssetWithBg_adapt_mutable_param_with_default_value = [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt, const std::optional<const ImVec4> & border_col = std::nullopt)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1,1);
                }();

                const ImVec4& tint_col_or_default = [&]() -> const ImVec4 {
                    if (tint_col.has_value())
                        return tint_col.value();
                    else
                        return ImVec4(1,1,1,1);
                }();

                const ImVec4& border_col_or_default = [&]() -> const ImVec4 {
                    if (border_col.has_value())
                        return border_col.value();
                    else
                        return ImVec4(0,0,0,0);
                }();

                HelloImGui::ImageFromAssetWithBg(assetPath, size_or_default, uv0_or_default, uv1_or_default, tint_col_or_default, border_col_or_default);
            };

            ImageFromAssetWithBg_adapt_mutable_param_with_default_value(assetPath, size, uv0, uv1, tint_col, border_col);
        },
        nb::arg("asset_path"), nb::arg("size").none() = nb::none(), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(), nb::arg("tint_col").none() = nb::none(), nb::arg("border_col").none() = nb::none(),
        " `HelloImGui::ImageFromAsset(const char *assetPath, size, ...)`:\n will display a static image from the assets, with a colored background and a border.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * size: ImVec2(0, 0)\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1,1)\n        * tint_col: ImVec4(1,1,1,1)\n        * border_col: ImVec4(0,0,0,0)");

    m.def("image_button_from_asset",
        [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, int frame_padding = -1, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt) -> bool
        {
            auto ImageButtonFromAsset_adapt_mutable_param_with_default_value = [](const char * assetPath, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, int frame_padding = -1, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1,1);
                }();

                const ImVec4& bg_col_or_default = [&]() -> const ImVec4 {
                    if (bg_col.has_value())
                        return bg_col.value();
                    else
                        return ImVec4(0,0,0,0);
                }();

                const ImVec4& tint_col_or_default = [&]() -> const ImVec4 {
                    if (tint_col.has_value())
                        return tint_col.value();
                    else
                        return ImVec4(1,1,1,1);
                }();

                auto lambda_result = HelloImGui::ImageButtonFromAsset(assetPath, size_or_default, uv0_or_default, uv1_or_default, frame_padding, bg_col_or_default, tint_col_or_default);
                return lambda_result;
            };

            return ImageButtonFromAsset_adapt_mutable_param_with_default_value(assetPath, size, uv0, uv1, frame_padding, bg_col, tint_col);
        },
        nb::arg("asset_path"), nb::arg("size").none() = nb::none(), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(), nb::arg("frame_padding") = -1, nb::arg("bg_col").none() = nb::none(), nb::arg("tint_col").none() = nb::none(),
        " `bool HelloImGui::ImageButtonFromAsset(const char *assetPath, size, ...)`:\n will display a button using an image from the assets.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * size: ImVec2(0, 0)\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1,1)\n        * bg_col: ImVec4(0,0,0,0)\n        * tint_col: ImVec4(1,1,1,1)");

    m.def("im_texture_id_from_asset",
        HelloImGui::ImTextureIdFromAsset,
        nb::arg("asset_path"),
        " `ImTextureID HelloImGui::ImTextureIdFromAsset(assetPath)`:\n will return a texture ID for an image loaded from the assets.");

    m.def("image_size_from_asset",
        HelloImGui::ImageSizeFromAsset,
        nb::arg("asset_path"),
        " `ImVec2 HelloImGui::ImageSizeFromAsset(assetPath)`:\n will return the size of an image loaded from the assets.");


    auto pyClassImageAndSize =
        nb::class_<HelloImGui::ImageAndSize>
            (m, "ImageAndSize", " `HelloImGui::ImageAndSize HelloImGui::ImageAndSizeFromAsset(assetPath)`:\n will return the texture ID and the size of an image loaded from the assets.")
        .def("__init__", [](HelloImGui::ImageAndSize * self, const std::optional<const ImTextureID> & textureId = std::nullopt, const std::optional<const ImVec2> & size = std::nullopt)
        {
            new (self) HelloImGui::ImageAndSize();  // placement new
            auto r_ctor_ = self;
            if (textureId.has_value())
                r_ctor_->textureId = textureId.value();
            else
                r_ctor_->textureId = ImTextureID(0);
            if (size.has_value())
                r_ctor_->size = size.value();
            else
                r_ctor_->size = ImVec2(0.f, 0.f);
        },
        nb::arg("texture_id").none() = nb::none(), nb::arg("size").none() = nb::none()
        )
        .def_rw("texture_id", &HelloImGui::ImageAndSize::textureId, "")
        .def_rw("size", &HelloImGui::ImageAndSize::size, "")
        ;


    m.def("image_and_size_from_asset",
        HelloImGui::ImageAndSizeFromAsset, nb::arg("asset_path"));

    m.def("image_proportional_size",
        HelloImGui::ImageProportionalSize,
        nb::arg("asked_size"), nb::arg("image_size"),
        " `ImVec2 HelloImGui::ImageProportionalSize(askedSize, imageSize)`:\n  will return the displayed size of an image.\n     - if askedSize.x or askedSize.y is 0, then the corresponding dimension\n       will be computed from the image size, keeping the aspect ratio.\n     - if askedSize.x>0 and askedSize.y> 0, then the image will be scaled to fit\n       exactly the askedSize, thus potentially changing the aspect ratio.\n  Note: this function is used internally by ImageFromAsset and ImageButtonFromAsset,\n        so you don't need to call it directly.");


    auto pyEnumImGuiTheme_ =
        nb::enum_<ImGuiTheme::ImGuiTheme_>(m, "ImGuiTheme_", nb::is_arithmetic(), "")
            .value("imgui_colors_classic", ImGuiTheme::ImGuiTheme_ImGuiColorsClassic, "")
            .value("imgui_colors_dark", ImGuiTheme::ImGuiTheme_ImGuiColorsDark, "")
            .value("imgui_colors_light", ImGuiTheme::ImGuiTheme_ImGuiColorsLight, "")
            .value("material_flat", ImGuiTheme::ImGuiTheme_MaterialFlat, "")
            .value("photoshop_style", ImGuiTheme::ImGuiTheme_PhotoshopStyle, "")
            .value("gray_variations", ImGuiTheme::ImGuiTheme_GrayVariations, "")
            .value("gray_variations_darker", ImGuiTheme::ImGuiTheme_GrayVariations_Darker, "")
            .value("microsoft_style", ImGuiTheme::ImGuiTheme_MicrosoftStyle, "")
            .value("cherry", ImGuiTheme::ImGuiTheme_Cherry, "")
            .value("darcula", ImGuiTheme::ImGuiTheme_Darcula, "")
            .value("darcula_darker", ImGuiTheme::ImGuiTheme_DarculaDarker, "")
            .value("light_rounded", ImGuiTheme::ImGuiTheme_LightRounded, "")
            .value("so_dark_accent_blue", ImGuiTheme::ImGuiTheme_SoDark_AccentBlue, "")
            .value("so_dark_accent_yellow", ImGuiTheme::ImGuiTheme_SoDark_AccentYellow, "")
            .value("so_dark_accent_red", ImGuiTheme::ImGuiTheme_SoDark_AccentRed, "")
            .value("black_is_black", ImGuiTheme::ImGuiTheme_BlackIsBlack, "")
            .value("white_is_white", ImGuiTheme::ImGuiTheme_WhiteIsWhite, "")
            .value("count", ImGuiTheme::ImGuiTheme_Count, "");


    m.def("imgui_theme_name",
        ImGuiTheme::ImGuiTheme_Name,
        nb::arg("theme"),
        nb::rv_policy::reference);

    m.def("imgui_theme_from_name",
        ImGuiTheme::ImGuiTheme_FromName, nb::arg("theme_name"));

    m.def("theme_to_style",
        ImGuiTheme::ThemeToStyle, nb::arg("theme"));

    m.def("apply_theme",
        ImGuiTheme::ApplyTheme, nb::arg("theme"));


    auto pyClassImGuiThemeTweaks =
        nb::class_<ImGuiTheme::ImGuiThemeTweaks>
            (m, "ImGuiThemeTweaks", "")
        .def_rw("rounding", &ImGuiTheme::ImGuiThemeTweaks::Rounding, "Common rounding for widgets. If < 0, this is ignored.")
        .def_rw("rounding_scrollbar_ratio", &ImGuiTheme::ImGuiThemeTweaks::RoundingScrollbarRatio, "If rounding is applied, scrollbar rounding needs to be adjusted to be visually pleasing in conjunction with other widgets roundings. Only applied if Rounding > 0.)")
        .def_rw("alpha_multiplier", &ImGuiTheme::ImGuiThemeTweaks::AlphaMultiplier, "Change the alpha that will be applied to windows, popups, etc. If < 0, this is ignored.")
        .def_rw("hue", &ImGuiTheme::ImGuiThemeTweaks::Hue, "\n HSV Color tweaks\n\n Change the hue of all widgets (gray widgets will remain gray, since their saturation is zero). If < 0, this is ignored.")
        .def_rw("saturation_multiplier", &ImGuiTheme::ImGuiThemeTweaks::SaturationMultiplier, "Multiply the saturation of all widgets (gray widgets will remain gray, since their saturation is zero). If < 0, this is ignored.")
        .def_rw("value_multiplier_front", &ImGuiTheme::ImGuiThemeTweaks::ValueMultiplierFront, "Multiply the value (luminance) of all front widgets. If < 0, this is ignored.")
        .def_rw("value_multiplier_bg", &ImGuiTheme::ImGuiThemeTweaks::ValueMultiplierBg, "Multiply the value (luminance) of all backgrounds. If < 0, this is ignored.")
        .def_rw("value_multiplier_text", &ImGuiTheme::ImGuiThemeTweaks::ValueMultiplierText, "Multiply the value (luminance) of text. If < 0, this is ignored.")
        .def_rw("value_multiplier_frame_bg", &ImGuiTheme::ImGuiThemeTweaks::ValueMultiplierFrameBg, " Multiply the value (luminance) of FrameBg. If < 0, this is ignored.\n (Background of checkbox, radio button, plot, slider, text input)")
        .def(nb::init<>())
        ;


    auto pyClassImGuiTweakedTheme =
        nb::class_<ImGuiTheme::ImGuiTweakedTheme>
            (m, "ImGuiTweakedTheme", "")
        .def_rw("theme", &ImGuiTheme::ImGuiTweakedTheme::Theme, "")
        .def_rw("tweaks", &ImGuiTheme::ImGuiTweakedTheme::Tweaks, "")
        .def("__init__",
            [](ImGuiTheme::ImGuiTweakedTheme * self, ImGuiTheme::ImGuiTheme_ theme = ImGuiTheme::ImGuiTheme_DarculaDarker, const std::optional<const ImGuiTheme::ImGuiThemeTweaks> & tweaks = std::nullopt)
            {
                auto ctor_wrapper = [](ImGuiTheme::ImGuiTweakedTheme* self, ImGuiTheme::ImGuiTheme_ theme = ImGuiTheme::ImGuiTheme_DarculaDarker, const ImGuiTheme::ImGuiThemeTweaks & tweaks = ImGuiTheme::ImGuiThemeTweaks()) ->  void
                {
                    new(self) ImGuiTheme::ImGuiTweakedTheme(theme, tweaks); // placement new
                };
                auto ctor_wrapper_adapt_mutable_param_with_default_value = [&ctor_wrapper](ImGuiTheme::ImGuiTweakedTheme * self, ImGuiTheme::ImGuiTheme_ theme = ImGuiTheme::ImGuiTheme_DarculaDarker, const std::optional<const ImGuiTheme::ImGuiThemeTweaks> & tweaks = std::nullopt)
                {

                    const ImGuiTheme::ImGuiThemeTweaks& tweaks_or_default = [&]() -> const ImGuiTheme::ImGuiThemeTweaks {
                        if (tweaks.has_value())
                            return tweaks.value();
                        else
                            return ImGuiTheme::ImGuiThemeTweaks();
                    }();

                    ctor_wrapper(self, theme, tweaks_or_default);
                };

                ctor_wrapper_adapt_mutable_param_with_default_value(self, theme, tweaks);
            },
            nb::arg("theme") = ImGuiTheme::ImGuiTheme_DarculaDarker, nb::arg("tweaks").none() = nb::none(),
            "Python bindings defaults:\n    If tweaks is None, then its default value will be: ImGuiThemeTweaks()")
        ;


    m.def("tweaked_theme_theme_to_style",
        ImGuiTheme::TweakedThemeThemeToStyle, nb::arg("tweaked_theme"));

    m.def("apply_tweaked_theme",
        ImGuiTheme::ApplyTweakedTheme, nb::arg("tweaked_theme"));

    m.def("push_tweaked_theme",
        ImGuiTheme::PushTweakedTheme, nb::arg("tweaked_theme"));

    m.def("pop_tweaked_theme",
        ImGuiTheme::PopTweakedTheme);

    m.def("show_theme_tweak_gui",
        ImGuiTheme::ShowThemeTweakGui,
        nb::arg("tweaked_theme"),
        "Show the theme selection listbox, the theme tweak widgets, as well as ImGui::ShowStyleEditor. Returns True if modified (Warning, when using ShowStyleEditor, no info about modification is transmitted)");

    m.def("so_dark",
        ImGuiTheme::SoDark, nb::arg("hue"));

    m.def("shades_of_gray",
        ImGuiTheme::ShadesOfGray, nb::arg("rounding") = 0.f, nb::arg("value_multiplier_front") = 1.f, nb::arg("value_multiplier_bg") = 1.f);

    m.def("darcula",
        ImGuiTheme::Darcula, nb::arg("rounding") = 1.f, nb::arg("hue") = -1.f, nb::arg("saturation_multiplier") = 1.f, nb::arg("value_multiplier_front") = 1.f, nb::arg("value_multiplier_bg") = 1.f, nb::arg("alpha_bg_transparency") = 1.f);


    m.def("show_theme_tweak_gui_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowThemeTweakGuiWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                HelloImGui::ShowThemeTweakGuiWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowThemeTweakGuiWindow_adapt_modifiable_immutable_to_return(p_open);
        },     nb::arg("p_open").none() = nb::none());


    auto pyClassFontLoadingParams =
        nb::class_<HelloImGui::FontLoadingParams>
            (m, "FontLoadingParams", "\n Font loading parameters: several options are available (color, merging, range, ...)")
        .def("__init__", [](HelloImGui::FontLoadingParams * self, bool adjustSizeToDpi = true, bool mergeToLastFont = false, bool loadColor = false, bool insideAssets = true, const std::optional<const ImFontConfig> & fontConfig = std::nullopt)
        {
            new (self) HelloImGui::FontLoadingParams();  // placement new
            auto r_ctor_ = self;
            r_ctor_->adjustSizeToDpi = adjustSizeToDpi;
            r_ctor_->mergeToLastFont = mergeToLastFont;
            r_ctor_->loadColor = loadColor;
            r_ctor_->insideAssets = insideAssets;
            if (fontConfig.has_value())
                r_ctor_->fontConfig = fontConfig.value();
            else
                r_ctor_->fontConfig = ImFontConfig();
        },
        nb::arg("adjust_size_to_dpi") = true, nb::arg("merge_to_last_font") = false, nb::arg("load_color") = false, nb::arg("inside_assets") = true, nb::arg("font_config").none() = nb::none()
        )
        .def_rw("adjust_size_to_dpi", &HelloImGui::FontLoadingParams::adjustSizeToDpi, " if True, the font size will be adjusted automatically to account for HighDPI\n")
        .def_rw("merge_to_last_font", &HelloImGui::FontLoadingParams::mergeToLastFont, "if True, the font will be merged to the last font")
        .def_rw("load_color", &HelloImGui::FontLoadingParams::loadColor, " if True, the font will be loaded using colors\n (requires freetype, enabled by IMGUI_ENABLE_FREETYPE)")
        .def_rw("inside_assets", &HelloImGui::FontLoadingParams::insideAssets, " if True, the font will be loaded using HelloImGui asset system.\n Otherwise, it will be loaded from the filesystem")
        .def_rw("font_config", &HelloImGui::FontLoadingParams::fontConfig, "ImGui native font config to use")
        ;


    m.def("load_font",
        [](const std::string & fontFilename, float fontSize, const std::optional<const HelloImGui::FontLoadingParams> & params = std::nullopt) -> ImFont *
        {
            auto LoadFont_adapt_mutable_param_with_default_value = [](const std::string & fontFilename, float fontSize, const std::optional<const HelloImGui::FontLoadingParams> & params = std::nullopt) -> ImFont *
            {

                const HelloImGui::FontLoadingParams& params_or_default = [&]() -> const HelloImGui::FontLoadingParams {
                    if (params.has_value())
                        return params.value();
                    else
                        return {};
                }();

                auto lambda_result = HelloImGui::LoadFont(fontFilename, fontSize, params_or_default);
                return lambda_result;
            };

            return LoadFont_adapt_mutable_param_with_default_value(fontFilename, fontSize, params);
        },
        nb::arg("font_filename"), nb::arg("font_size"), nb::arg("params").none() = nb::none(),
        "Python bindings defaults:\n    If params is None, then its default value will be: initialized with default value",
        nb::rv_policy::reference);

    m.def("load_font_ttf",
        [](const std::string & fontFilename, float fontSize, const std::optional<const ImFontConfig> & config = std::nullopt) -> ImFont *
        {
            auto LoadFontTTF_adapt_mutable_param_with_default_value = [](const std::string & fontFilename, float fontSize, const std::optional<const ImFontConfig> & config = std::nullopt) -> ImFont *
            {

                const ImFontConfig& config_or_default = [&]() -> const ImFontConfig {
                    if (config.has_value())
                        return config.value();
                    else
                        return ImFontConfig();
                }();

                auto lambda_result = HelloImGui::LoadFontTTF(fontFilename, fontSize, config_or_default);
                return lambda_result;
            };

            return LoadFontTTF_adapt_mutable_param_with_default_value(fontFilename, fontSize, config);
        },
        nb::arg("font_filename"), nb::arg("font_size"), nb::arg("config").none() = nb::none(),
        "Python bindings defaults:\n    If config is None, then its default value will be: ImFontConfig()",
        nb::rv_policy::reference);

    m.def("load_font_ttf_with_font_awesome_icons",
        [](const std::string & fontFilename, float fontSize, const std::optional<const ImFontConfig> & configFont = std::nullopt) -> ImFont *
        {
            auto LoadFontTTF_WithFontAwesomeIcons_adapt_mutable_param_with_default_value = [](const std::string & fontFilename, float fontSize, const std::optional<const ImFontConfig> & configFont = std::nullopt) -> ImFont *
            {

                const ImFontConfig& configFont_or_default = [&]() -> const ImFontConfig {
                    if (configFont.has_value())
                        return configFont.value();
                    else
                        return ImFontConfig();
                }();

                auto lambda_result = HelloImGui::LoadFontTTF_WithFontAwesomeIcons(fontFilename, fontSize, configFont_or_default);
                return lambda_result;
            };

            return LoadFontTTF_WithFontAwesomeIcons_adapt_mutable_param_with_default_value(fontFilename, fontSize, configFont);
        },
        nb::arg("font_filename"), nb::arg("font_size"), nb::arg("config_font").none() = nb::none(),
        "Python bindings defaults:\n    If configFont is None, then its default value will be: ImFontConfig()",
        nb::rv_policy::reference);


    auto pyClassScreenBounds =
        nb::class_<HelloImGui::ScreenBounds>
            (m, "ScreenBounds", "")
        .def("__init__", [](HelloImGui::ScreenBounds * self, const std::optional<const ScreenPosition> & position = std::nullopt, const std::optional<const ScreenSize> & size = std::nullopt)
        {
            new (self) HelloImGui::ScreenBounds();  // placement new
            auto r_ctor_ = self;
            if (position.has_value())
                r_ctor_->position = position.value();
            else
                r_ctor_->position = HelloImGui::DefaultScreenPosition;
            if (size.has_value())
                r_ctor_->size = size.value();
            else
                r_ctor_->size = HelloImGui::DefaultWindowSize;
        },
        nb::arg("position").none() = nb::none(), nb::arg("size").none() = nb::none()
        )
        .def_rw("position", &HelloImGui::ScreenBounds::position, "")
        .def_rw("size", &HelloImGui::ScreenBounds::size, "")
        .def("top_left_corner",
            &HelloImGui::ScreenBounds::TopLeftCorner)
        .def("bottom_right_corner",
            &HelloImGui::ScreenBounds::BottomRightCorner)
        .def("center",
            &HelloImGui::ScreenBounds::Center)
        .def("contains",
            &HelloImGui::ScreenBounds::Contains, nb::arg("pixel"))
        .def("win_position_centered",
            &HelloImGui::ScreenBounds::WinPositionCentered, nb::arg("window_size"))
        .def("distance_from_pixel",
            &HelloImGui::ScreenBounds::DistanceFromPixel, nb::arg("point"))
        .def("ensure_window_fits_this_monitor",
            &HelloImGui::ScreenBounds::EnsureWindowFitsThisMonitor, nb::arg("window_bounds_original"))
        .def("__eq__",
            &HelloImGui::ScreenBounds::operator==, nb::arg("other"))
        ;


    auto pyEnumFullScreenMode =
        nb::enum_<HelloImGui::FullScreenMode>(m, "FullScreenMode", nb::is_arithmetic(), "")
            .value("no_full_screen", HelloImGui::FullScreenMode::NoFullScreen, "")
            .value("full_screen", HelloImGui::FullScreenMode::FullScreen, "Full screen with specified resolution")
            .value("full_screen_desktop_resolution", HelloImGui::FullScreenMode::FullScreenDesktopResolution, "Full screen with current desktop mode & resolution")
            .value("full_monitor_work_area", HelloImGui::FullScreenMode::FullMonitorWorkArea, "Fake full screen, maximized window on the selected monitor");


    auto pyEnumWindowSizeState =
        nb::enum_<HelloImGui::WindowSizeState>(m, "WindowSizeState", nb::is_arithmetic(), "")
            .value("standard", HelloImGui::WindowSizeState::Standard, "")
            .value("minimized", HelloImGui::WindowSizeState::Minimized, "")
            .value("maximized", HelloImGui::WindowSizeState::Maximized, "");


    auto pyEnumWindowPositionMode =
        nb::enum_<HelloImGui::WindowPositionMode>(m, "WindowPositionMode", nb::is_arithmetic(), "")
            .value("os_default", HelloImGui::WindowPositionMode::OsDefault, "")
            .value("monitor_center", HelloImGui::WindowPositionMode::MonitorCenter, "")
            .value("from_coords", HelloImGui::WindowPositionMode::FromCoords, "");


    auto pyEnumEmscriptenKeyboardElement =
        nb::enum_<HelloImGui::EmscriptenKeyboardElement>(m, "EmscriptenKeyboardElement", nb::is_arithmetic(), "")
            .value("window", HelloImGui::EmscriptenKeyboardElement::Window, "")
            .value("document", HelloImGui::EmscriptenKeyboardElement::Document, "")
            .value("screen", HelloImGui::EmscriptenKeyboardElement::Screen, "")
            .value("canvas", HelloImGui::EmscriptenKeyboardElement::Canvas, "")
            .value("default", HelloImGui::EmscriptenKeyboardElement::Default, "");


    auto pyEnumWindowSizeMeasureMode =
        nb::enum_<HelloImGui::WindowSizeMeasureMode>(m, "WindowSizeMeasureMode", nb::is_arithmetic(), "")
            .value("screen_coords", HelloImGui::WindowSizeMeasureMode::ScreenCoords, " ScreenCoords: measure window size in screen coords.\n     Note: screen coordinates *might* differ from real pixel on high dpi screens; but this depends on the OS.\n         - For example, on apple a retina screenpixel size 3456x2052 might be seen as 1728x1026 in screen\n           coordinates\n         - Under windows, and if the application is DPI aware, ScreenCoordinates correspond to real pixels,\n           even on high density screens")
            .value("relative_to96_ppi", HelloImGui::WindowSizeMeasureMode::RelativeTo96Ppi, " RelativeTo96Ppi enables to give screen size that are independant from the screen density.\n For example, a window size expressed as 800x600 will correspond to a size\n    800x600 (in screen coords) if the monitor dpi is 96\n    1600x120 (in screen coords) if the monitor dpi is 192");


    auto pyClassWindowGeometry =
        nb::class_<HelloImGui::WindowGeometry>
            (m, "WindowGeometry", " @@md#WindowGeometry\n\n WindowGeometry is a struct that defines the window geometry.")
        .def("__init__", [](HelloImGui::WindowGeometry * self, const std::optional<const ScreenSize> & size = std::nullopt, bool sizeAuto = false, HelloImGui::WindowSizeState windowSizeState = HelloImGui::WindowSizeState::Standard, HelloImGui::WindowSizeMeasureMode windowSizeMeasureMode = HelloImGui::WindowSizeMeasureMode::RelativeTo96Ppi, HelloImGui::WindowPositionMode positionMode = HelloImGui::WindowPositionMode::OsDefault, const std::optional<const ScreenPosition> & position = std::nullopt, int monitorIdx = 0, HelloImGui::FullScreenMode fullScreenMode = HelloImGui::FullScreenMode::NoFullScreen, bool resizeAppWindowAtNextFrame = false)
        {
            new (self) HelloImGui::WindowGeometry();  // placement new
            auto r_ctor_ = self;
            if (size.has_value())
                r_ctor_->size = size.value();
            else
                r_ctor_->size = HelloImGui::DefaultWindowSize;
            r_ctor_->sizeAuto = sizeAuto;
            r_ctor_->windowSizeState = windowSizeState;
            r_ctor_->windowSizeMeasureMode = windowSizeMeasureMode;
            r_ctor_->positionMode = positionMode;
            if (position.has_value())
                r_ctor_->position = position.value();
            else
                r_ctor_->position = HelloImGui::DefaultScreenPosition;
            r_ctor_->monitorIdx = monitorIdx;
            r_ctor_->fullScreenMode = fullScreenMode;
            r_ctor_->resizeAppWindowAtNextFrame = resizeAppWindowAtNextFrame;
        },
        nb::arg("size").none() = nb::none(), nb::arg("size_auto") = false, nb::arg("window_size_state") = HelloImGui::WindowSizeState::Standard, nb::arg("window_size_measure_mode") = HelloImGui::WindowSizeMeasureMode::RelativeTo96Ppi, nb::arg("position_mode") = HelloImGui::WindowPositionMode::OsDefault, nb::arg("position").none() = nb::none(), nb::arg("monitor_idx") = 0, nb::arg("full_screen_mode") = HelloImGui::FullScreenMode::NoFullScreen, nb::arg("resize_app_window_at_next_frame") = false
        )
        .def_rw("size", &HelloImGui::WindowGeometry::size, " Size of the application window\n used if fullScreenMode==NoFullScreen and sizeAuto==False. Default=(800, 600)\n The size will be handled as if it was specified for a 96PPI screen\n (i.e. a given size will correspond to the same physical size on different screens, whatever their DPI)")
        .def_rw("size_auto", &HelloImGui::WindowGeometry::sizeAuto, " If sizeAuto=True, adapt the app window size to the presented widgets.\n After the first frame was displayed, HelloImGui will measure its size, and the\n application window will be resized.\n As a consequence, the application window may change between the 1st and 2nd frame.\n If True, adapt the app window size to the presented widgets. This is done at startup")
        .def_rw("window_size_state", &HelloImGui::WindowGeometry::windowSizeState, " `windowSizeState`: _WindowSizeState, default=Standard_\n  You can choose between several window size states:\n      Standard,\n      Minimized,\n      Maximized")
        .def_rw("window_size_measure_mode", &HelloImGui::WindowGeometry::windowSizeMeasureMode, " `windowSizeMeasureMode`: _WindowSizeMeasureMode_, default=RelativeTo96Ppi\n Define how the window size is specified:\n      * RelativeTo96Ppi enables to give a screen size whose physical result\n      (in millimeters) is independent of the screen density.\n         For example, a window size expressed as 800x600 will correspond to a size\n            - 800x600 (in screen coords) if the monitor dpi is 96\n            - 1600x120 (in screen coords) if the monitor dpi is 192\n          (this works with Glfw. With SDL, it only works under windows)\n      * ScreenCoords: measure window size in screen coords\n        (Note: screen coordinates might differ from real pixels on high dpi screen)")
        .def_rw("position_mode", &HelloImGui::WindowGeometry::positionMode, " `positionMode`: you can choose between several window position modes:\n      OsDefault,\n      MonitorCenter,\n      FromCoords,")
        .def_rw("position", &HelloImGui::WindowGeometry::position, "`position`: used if windowPositionMode==FromCoords, default=(40, 40)")
        .def_rw("monitor_idx", &HelloImGui::WindowGeometry::monitorIdx, " `monitorIdx`: index of the monitor to use, default=0\n  used if positionMode==MonitorCenter or if fullScreenMode!=NoFullScreen")
        .def_rw("full_screen_mode", &HelloImGui::WindowGeometry::fullScreenMode, " `fullScreenMode`: you can choose between several full screen modes:\n      NoFullScreen,\n      FullScreen,                  // Full screen with specified resolution\n      FullScreenDesktopResolution, // Full screen with current desktop mode & resolution\n      FullMonitorWorkArea          // Fake full screen (maximized window) on the selected monitor")
        .def_rw("resize_app_window_at_next_frame", &HelloImGui::WindowGeometry::resizeAppWindowAtNextFrame, " `resizeAppWindowAtNextFrame`: _bool_, default=False;\n  If you set this to flag to True at any point during the execution, the application\n  window will then try to resize based on its content on the next displayed frame,\n  and this flag will subsequently be set to False.\n  Example:\n   ```cpp\n // Will resize the app window at next displayed frame\n   HelloImGui::GetRunnerParams()->appWindowParams.windowGeometry.resizeAppWindowAtNextFrame = True;\n   ```\n  Note: this flag is intended to be used during execution, not at startup\n  (use sizeAuto at startup).")
        ;


    auto pyClassEdgeInsets =
        nb::class_<HelloImGui::EdgeInsets>
            (m, "EdgeInsets", "If there is a notch on the iPhone, you should not display inside these insets")
        .def("__init__", [](HelloImGui::EdgeInsets * self, double top = 0., double left = 0., double bottom = 0., double right = 0.)
        {
            new (self) HelloImGui::EdgeInsets();  // placement new
            auto r_ctor_ = self;
            r_ctor_->top = top;
            r_ctor_->left = left;
            r_ctor_->bottom = bottom;
            r_ctor_->right = right;
        },
        nb::arg("top") = 0., nb::arg("left") = 0., nb::arg("bottom") = 0., nb::arg("right") = 0.
        )
        .def_rw("top", &HelloImGui::EdgeInsets::top, "Typically around 47")
        .def_rw("left", &HelloImGui::EdgeInsets::left, "Typically 0")
        .def_rw("bottom", &HelloImGui::EdgeInsets::bottom, "Typically around 34")
        .def_rw("right", &HelloImGui::EdgeInsets::right, "Typically 0")
        ;


    auto pyClassAppWindowParams =
        nb::class_<HelloImGui::AppWindowParams>
            (m, "AppWindowParams", " @@md#AppWindowParams\n\n AppWindowParams is a struct that defines the application window display params.\nSee https://raw.githubusercontent.com/pthom/hello_imgui/master/src/hello_imgui/doc_src/hello_imgui_diagram.jpg\n for details.")
        .def("__init__", [](HelloImGui::AppWindowParams * self, std::string windowTitle = std::string(), const std::optional<const HelloImGui::WindowGeometry> & windowGeometry = std::nullopt, bool restorePreviousGeometry = false, bool resizable = true, bool hidden = false, bool borderless = false, bool borderlessMovable = true, bool borderlessResizable = true, bool borderlessClosable = true, const std::optional<const ImVec4> & borderlessHighlightColor = std::nullopt, const std::optional<const HelloImGui::EdgeInsets> & edgeInsets = std::nullopt, bool handleEdgeInsets = true, HelloImGui::EmscriptenKeyboardElement emscriptenKeyboardElement = HelloImGui::EmscriptenKeyboardElement::Default, bool repaintDuringResize_GotchaReentrantRepaint = false)
        {
            new (self) HelloImGui::AppWindowParams();  // placement new
            auto r_ctor_ = self;
            r_ctor_->windowTitle = windowTitle;
            if (windowGeometry.has_value())
                r_ctor_->windowGeometry = windowGeometry.value();
            else
                r_ctor_->windowGeometry = HelloImGui::WindowGeometry();
            r_ctor_->restorePreviousGeometry = restorePreviousGeometry;
            r_ctor_->resizable = resizable;
            r_ctor_->hidden = hidden;
            r_ctor_->borderless = borderless;
            r_ctor_->borderlessMovable = borderlessMovable;
            r_ctor_->borderlessResizable = borderlessResizable;
            r_ctor_->borderlessClosable = borderlessClosable;
            if (borderlessHighlightColor.has_value())
                r_ctor_->borderlessHighlightColor = borderlessHighlightColor.value();
            else
                r_ctor_->borderlessHighlightColor = ImVec4(0.2f, 0.4f, 1.f, 0.3f);
            if (edgeInsets.has_value())
                r_ctor_->edgeInsets = edgeInsets.value();
            else
                r_ctor_->edgeInsets = HelloImGui::EdgeInsets();
            r_ctor_->handleEdgeInsets = handleEdgeInsets;
            r_ctor_->emscriptenKeyboardElement = emscriptenKeyboardElement;
            r_ctor_->repaintDuringResize_GotchaReentrantRepaint = repaintDuringResize_GotchaReentrantRepaint;
        },
        nb::arg("window_title") = std::string(), nb::arg("window_geometry").none() = nb::none(), nb::arg("restore_previous_geometry") = false, nb::arg("resizable") = true, nb::arg("hidden") = false, nb::arg("borderless") = false, nb::arg("borderless_movable") = true, nb::arg("borderless_resizable") = true, nb::arg("borderless_closable") = true, nb::arg("borderless_highlight_color").none() = nb::none(), nb::arg("edge_insets").none() = nb::none(), nb::arg("handle_edge_insets") = true, nb::arg("emscripten_keyboard_element") = HelloImGui::EmscriptenKeyboardElement::Default, nb::arg("repaint_during_resize_gotcha_reentrant_repaint") = false
        )
        .def_rw("window_title", &HelloImGui::AppWindowParams::windowTitle, "`windowTitle`: _string, default=\"\"_. Title of the application window")
        .def_rw("window_geometry", &HelloImGui::AppWindowParams::windowGeometry, " `windowGeometry`: _WindowGeometry_\n  Enables to precisely set the window geometry (position, monitor, size,\n  full screen, fake full screen, etc.)\n   _Note: on a mobile device, the application will always be full screen._")
        .def_rw("restore_previous_geometry", &HelloImGui::AppWindowParams::restorePreviousGeometry, " `restorePreviousGeometry`: _bool, default=false_.\n If True, then save & restore windowGeometry from last run (the geometry\n will be written in imgui_app_window.ini)")
        .def_rw("resizable", &HelloImGui::AppWindowParams::resizable, " `resizable`: _bool, default = false_. Should the window be resizable.\n This is taken into account at creation.")
        .def_rw("hidden", &HelloImGui::AppWindowParams::hidden, " `hidden`: _bool, default = false_. Should the window be hidden.\n This is taken into account dynamically (you can show/hide the window with this).\n Full screen windows cannot be hidden.")
        .def_rw("borderless", &HelloImGui::AppWindowParams::borderless, " `borderless`: _bool, default = false_. Should the window have borders.\n This is taken into account at creation.")
        .def_rw("borderless_movable", &HelloImGui::AppWindowParams::borderlessMovable, " `borderlessMovable`: if the window is borderless, should it be movable.\n   If so, a drag zone is displayed at the top of the window when the mouse is over it.")
        .def_rw("borderless_resizable", &HelloImGui::AppWindowParams::borderlessResizable, " `borderlessResizable`: if the window is borderless, should it be resizable.\n  If so, a drag zone is displayed at the bottom-right of the window\n  when the mouse is over it.")
        .def_rw("borderless_closable", &HelloImGui::AppWindowParams::borderlessClosable, " `borderlessClosable`: if the window is borderless, should it have a close button.\n  If so, a close button is displayed at the top-right of the window\n  when the mouse is over it.")
        .def_rw("borderless_highlight_color", &HelloImGui::AppWindowParams::borderlessHighlightColor, " `borderlessHighlightColor`:\n   Color of the highlight displayed on resize/move zones.\n   If borderlessHighlightColor.w==0, then the highlightColor will be automatically\n   set to ImGui::GetColorU32(ImGuiCol_TitleBgActive, 0.6)")
        .def_rw("edge_insets", &HelloImGui::AppWindowParams::edgeInsets, " `edgeInsets`: _EdgeInsets_. iOS only, out values filled by HelloImGui.\n If there is a notch on the iPhone, you should not display inside these insets.\n HelloImGui handles this automatically, if handleEdgeInsets is True and\n if runnerParams.imGuiWindowParams.defaultImGuiWindowType is not NoDefaultWindow.\n (warning, these values are updated only after a few frames,\n  they are typically 0 for the first 4 frames)")
        .def_rw("handle_edge_insets", &HelloImGui::AppWindowParams::handleEdgeInsets, " `handleEdgeInsets`: _bool, default = true_. iOS only.\n If True, HelloImGui will handle the edgeInsets on iOS.")
        .def_rw("emscripten_keyboard_element", &HelloImGui::AppWindowParams::emscriptenKeyboardElement, " --------------- Emscripten ------------------\n `emscriptenKeyboardElement`: _EmscriptenKeyboardElement, default=Default_. HTML element in which SDL will capture the keyboard events.\n (For Emscripten only)\n Choose between: Window, Document, Screen, Canvas, Default.\n If Default:\n - the default SDL behavior is used, which is to capture the keyboard events for the window,\n   if no previous hint was set in the javascript code.\n - under Pyodide, the default behavior is to capture the keyboard events for the canvas.")
        .def_rw("repaint_during_resize_gotcha_reentrant_repaint", &HelloImGui::AppWindowParams::repaintDuringResize_GotchaReentrantRepaint, " ----------------- repaint the window during resize -----------------\n Very advanced and reserved for advanced C++ users.\n If you set this to True, the window will be repainted during resize.\n Do read https://github.com/pthom/hello_imgui/issues/112 for info about the possible gotchas\n (This API is not stable, as the name suggests, and this is not supported)")
        ;


    auto pyEnumDefaultImGuiWindowType =
        nb::enum_<HelloImGui::DefaultImGuiWindowType>(m, "DefaultImGuiWindowType", nb::is_arithmetic(), " `DefaultImGuiWindowType` is an enum class that defines whether a full screen background\n window is provided or not")
            .value("provide_full_screen_window", HelloImGui::DefaultImGuiWindowType::ProvideFullScreenWindow, "`ProvideFullScreenWindow`: a full window is provided in the background")
            .value("provide_full_screen_dock_space", HelloImGui::DefaultImGuiWindowType::ProvideFullScreenDockSpace, "`ProvideFullScreenDockSpace`: a full screen dockspace is provided in the background")
            .value("no_default_window", HelloImGui::DefaultImGuiWindowType::NoDefaultWindow, " `NoDefaultWindow`: No default window is provided\n (except for ImGui's default \"debug\" window)");


    auto pyClassImGuiWindowParams =
        nb::class_<HelloImGui::ImGuiWindowParams>
            (m, "ImGuiWindowParams", " `ImGuiWindowParams` is a struct that defines the ImGui inner windows params\n These settings affect the imgui inner windows inside the application window.\n In order to change the application window settings, change the `AppWindowsParams`")
        .def("__init__", [](HelloImGui::ImGuiWindowParams * self, HelloImGui::DefaultImGuiWindowType defaultImGuiWindowType = HelloImGui::DefaultImGuiWindowType::ProvideFullScreenWindow, bool enableViewports = false, bool configWindowsMoveFromTitleBarOnly = true, std::string menuAppTitle = "", bool showMenuBar = false, bool showMenu_App = true, bool showMenu_App_Quit = true, bool showMenu_View = true, bool showMenu_View_Themes = true, bool rememberTheme = true, bool showStatusBar = false, bool showStatus_Fps = true, bool rememberStatusBarSettings = true, const std::optional<const ImVec2> & fullScreenWindow_MarginTopLeft = std::nullopt, const std::optional<const ImVec2> & fullScreenWindow_MarginBottomRight = std::nullopt, const std::optional<const ImGuiTheme::ImGuiTweakedTheme> & tweakedTheme = std::nullopt, const std::optional<const ImVec4> & backgroundColor = std::nullopt)
        {
            new (self) HelloImGui::ImGuiWindowParams();  // placement new
            auto r_ctor_ = self;
            r_ctor_->defaultImGuiWindowType = defaultImGuiWindowType;
            r_ctor_->enableViewports = enableViewports;
            r_ctor_->configWindowsMoveFromTitleBarOnly = configWindowsMoveFromTitleBarOnly;
            r_ctor_->menuAppTitle = menuAppTitle;
            r_ctor_->showMenuBar = showMenuBar;
            r_ctor_->showMenu_App = showMenu_App;
            r_ctor_->showMenu_App_Quit = showMenu_App_Quit;
            r_ctor_->showMenu_View = showMenu_View;
            r_ctor_->showMenu_View_Themes = showMenu_View_Themes;
            r_ctor_->rememberTheme = rememberTheme;
            r_ctor_->showStatusBar = showStatusBar;
            r_ctor_->showStatus_Fps = showStatus_Fps;
            r_ctor_->rememberStatusBarSettings = rememberStatusBarSettings;
            if (fullScreenWindow_MarginTopLeft.has_value())
                r_ctor_->fullScreenWindow_MarginTopLeft = fullScreenWindow_MarginTopLeft.value();
            else
                r_ctor_->fullScreenWindow_MarginTopLeft = ImVec2(0.f, 0.f);
            if (fullScreenWindow_MarginBottomRight.has_value())
                r_ctor_->fullScreenWindow_MarginBottomRight = fullScreenWindow_MarginBottomRight.value();
            else
                r_ctor_->fullScreenWindow_MarginBottomRight = ImVec2(0.f, 0.f);
            if (tweakedTheme.has_value())
                r_ctor_->tweakedTheme = tweakedTheme.value();
            else
                r_ctor_->tweakedTheme = ImGuiTheme::ImGuiTweakedTheme();
            if (backgroundColor.has_value())
                r_ctor_->backgroundColor = backgroundColor.value();
            else
                r_ctor_->backgroundColor = ImVec4(0.f, 0.f, 0.f, 0.f);
        },
        nb::arg("default_imgui_window_type") = HelloImGui::DefaultImGuiWindowType::ProvideFullScreenWindow, nb::arg("enable_viewports") = false, nb::arg("config_windows_move_from_title_bar_only") = true, nb::arg("menu_app_title") = "", nb::arg("show_menu_bar") = false, nb::arg("show_menu_app") = true, nb::arg("show_menu_app_quit") = true, nb::arg("show_menu_view") = true, nb::arg("show_menu_view_themes") = true, nb::arg("remember_theme") = true, nb::arg("show_status_bar") = false, nb::arg("show_status_fps") = true, nb::arg("remember_status_bar_settings") = true, nb::arg("full_screen_window_margin_top_left").none() = nb::none(), nb::arg("full_screen_window_margin_bottom_right").none() = nb::none(), nb::arg("tweaked_theme").none() = nb::none(), nb::arg("background_color").none() = nb::none()
        )
        .def_rw("default_imgui_window_type", &HelloImGui::ImGuiWindowParams::defaultImGuiWindowType, " defaultImGuiWindowType: (enum DefaultImGuiWindowType)\n Choose between:\n    - ProvideFullScreenWindow (default)\n      a full window is provided in the background\n      You can still add windows on top of it, since the Z-order\n      of this background window is always behind\n    - ProvideFullScreenDockSpace:\n      a full screen dockspace is provided in the background\n      (use this if you intend to use docking)\n    - NoDefaultWindow:\n      no default window is provided")
        .def_rw("enable_viewports", &HelloImGui::ImGuiWindowParams::enableViewports, " enableViewports: Enable multiple viewports (i.e. multiple native windows)\n If True, you can drag windows outside the main window,\n in order to put their content into new native windows.")
        .def_rw("config_windows_move_from_title_bar_only", &HelloImGui::ImGuiWindowParams::configWindowsMoveFromTitleBarOnly, "Make windows only movable from the title bar")
        .def_rw("menu_app_title", &HelloImGui::ImGuiWindowParams::menuAppTitle, " Set the title of the App menu. If empty, the menu name will use\n the \"windowTitle\" from AppWindowParams//")
        .def_rw("show_menu_bar", &HelloImGui::ImGuiWindowParams::showMenuBar, " Show Menu bar on top of imgui main window.\n In order to fully customize the menu, set showMenuBar to True, and set showMenu_App\n and showMenu_View params to False. Then, implement the callback\n `RunnerParams.callbacks.ShowMenus`\n which can optionally call `HelloImGui::ShowViewMenu` and `HelloImGui::ShowAppMenu`.")
        .def_rw("show_menu_app", &HelloImGui::ImGuiWindowParams::showMenu_App, "If menu bar is shown, include or not the default app menu")
        .def_rw("show_menu_app_quit", &HelloImGui::ImGuiWindowParams::showMenu_App_Quit, " Include or not a \"Quit\" item in the default app menu.\n Set this to False if you intend to provide your own quit callback\n with possible user confirmation\n (and implement it inside RunnerCallbacks.ShowAppMenuItems)")
        .def_rw("show_menu_view", &HelloImGui::ImGuiWindowParams::showMenu_View, " If menu bar is shown, include or not the default _View_ menu, that enables\n to change the layout and set the docked windows and status bar visibility)")
        .def_rw("show_menu_view_themes", &HelloImGui::ImGuiWindowParams::showMenu_View_Themes, "Show theme selection in view menu")
        .def_rw("remember_theme", &HelloImGui::ImGuiWindowParams::rememberTheme, "`rememberTheme`: _bool, default=true_. Remember selected theme")
        .def_rw("show_status_bar", &HelloImGui::ImGuiWindowParams::showStatusBar, " Flag that enable to show a Status bar at the bottom. You can customize\n the status bar via RunnerCallbacks.ShowStatus()")
        .def_rw("show_status_fps", &HelloImGui::ImGuiWindowParams::showStatus_Fps, "If set, display the FPS in the status bar.")
        .def_rw("remember_status_bar_settings", &HelloImGui::ImGuiWindowParams::rememberStatusBarSettings, "If set, showStatusBar and showStatus_Fps are stored in the application settings.")
        .def_rw("full_screen_window_margin_top_left", &HelloImGui::ImGuiWindowParams::fullScreenWindow_MarginTopLeft, "")
        .def_rw("full_screen_window_margin_bottom_right", &HelloImGui::ImGuiWindowParams::fullScreenWindow_MarginBottomRight, "")
        .def_rw("tweaked_theme", &HelloImGui::ImGuiWindowParams::tweakedTheme, " tweakedTheme: (enum ImGuiTheme::ImGuiTweakedTheme)\n Changes the ImGui theme. Several themes are available, you can query the list\n by calling HelloImGui::AvailableThemes()")
        .def_rw("background_color", &HelloImGui::ImGuiWindowParams::backgroundColor, " backgroundColor:\n This is the \"clearColor\", i.e. the app window background color, is visible *only if*\n    runnerParams.imGuiWindowParams.defaultImGuiWindowType = NoDefaultWindow\n Alternatively, you can set your own RunnerCallbacks.CustomBackground to have full\n control over what is drawn behind the Gui.")
        ;


    m.def("empty_void_function",
        HelloImGui::EmptyVoidFunction);

    m.def("sequence_functions",
        HelloImGui::SequenceFunctions,
        nb::arg("f1"), nb::arg("f2"),
        "SequenceFunctions: returns a function that will call f1 and f2 in sequence");

    m.def("empty_event_callback",
        HelloImGui::EmptyEventCallback);


    auto pyClassMobileCallbacks =
        nb::class_<HelloImGui::MobileCallbacks>
            (m, "MobileCallbacks", " MobileCallbacks is a struct that contains callbacks that are called by the application\n when running under \"Android, iOS and WinRT\".\n These events are specific to mobile and embedded devices that have different\n requirements from your usual desktop application.\n These events must be handled quickly, since often the OS needs an immediate response\n and will terminate your process shortly after sending the event\n if you do not handle them appropriately.\n On mobile devices, it is not possible to \"Quit\" an application,\n it can only be put on Pause.")
        .def("__init__", [](HelloImGui::MobileCallbacks * self, const std::optional<const VoidFunction> & OnDestroy = std::nullopt, const std::optional<const VoidFunction> & OnLowMemory = std::nullopt, const std::optional<const VoidFunction> & OnPause = std::nullopt, const std::optional<const VoidFunction> & OnResume = std::nullopt)
        {
            new (self) HelloImGui::MobileCallbacks();  // placement new
            auto r_ctor_ = self;
            if (OnDestroy.has_value())
                r_ctor_->OnDestroy = OnDestroy.value();
            else
                r_ctor_->OnDestroy = HelloImGui::EmptyVoidFunction();
            if (OnLowMemory.has_value())
                r_ctor_->OnLowMemory = OnLowMemory.value();
            else
                r_ctor_->OnLowMemory = HelloImGui::EmptyVoidFunction();
            if (OnPause.has_value())
                r_ctor_->OnPause = OnPause.value();
            else
                r_ctor_->OnPause = HelloImGui::EmptyVoidFunction();
            if (OnResume.has_value())
                r_ctor_->OnResume = OnResume.value();
            else
                r_ctor_->OnResume = HelloImGui::EmptyVoidFunction();
        },
        nb::arg("on_destroy").none() = nb::none(), nb::arg("on_low_memory").none() = nb::none(), nb::arg("on_pause").none() = nb::none(), nb::arg("on_resume").none() = nb::none()
        )
        .def_rw("on_destroy", &HelloImGui::MobileCallbacks::OnDestroy, "`OnDestroy`: The application is being terminated by the OS.")
        .def_rw("on_low_memory", &HelloImGui::MobileCallbacks::OnLowMemory, "`OnLowMemory`: _VoidFunction, default=empty_.\n When the application is low on memory, free memory if possible.")
        .def_rw("on_pause", &HelloImGui::MobileCallbacks::OnPause, "`OnPause`: The application is about to enter the background.")
        .def_rw("on_resume", &HelloImGui::MobileCallbacks::OnResume, "`OnResume`: The application came to foreground and is now interactive.\n Note: 'OnPause' and 'OnResume' are called twice consecutively under iOS\n (before and after entering background or foreground).")
        ;


    auto pyEnumEdgeToolbarType =
        nb::enum_<HelloImGui::EdgeToolbarType>(m, "EdgeToolbarType", nb::is_arithmetic(), "EdgeToolbarType: location of an Edge Toolbar")
            .value("top", HelloImGui::EdgeToolbarType::Top, "")
            .value("bottom", HelloImGui::EdgeToolbarType::Bottom, "")
            .value("left", HelloImGui::EdgeToolbarType::Left, "")
            .value("right", HelloImGui::EdgeToolbarType::Right, "");


    auto pyClassEdgeToolbarOptions =
        nb::class_<HelloImGui::EdgeToolbarOptions>
            (m, "EdgeToolbarOptions", "")
        .def("__init__", [](HelloImGui::EdgeToolbarOptions * self, float sizeEm = 2.5f, const std::optional<const ImVec2> & WindowPaddingEm = std::nullopt, const std::optional<const ImVec4> & WindowBg = std::nullopt)
        {
            new (self) HelloImGui::EdgeToolbarOptions();  // placement new
            auto r_ctor_ = self;
            r_ctor_->sizeEm = sizeEm;
            if (WindowPaddingEm.has_value())
                r_ctor_->WindowPaddingEm = WindowPaddingEm.value();
            else
                r_ctor_->WindowPaddingEm = ImVec2(0.3f, 0.3f);
            if (WindowBg.has_value())
                r_ctor_->WindowBg = WindowBg.value();
            else
                r_ctor_->WindowBg = ImVec4(0.f, 0.f, 0.f, 0.f);
        },
        nb::arg("size_em") = 2.5f, nb::arg("window_padding_em").none() = nb::none(), nb::arg("window_bg").none() = nb::none()
        )
        .def_rw("size_em", &HelloImGui::EdgeToolbarOptions::sizeEm, " height or width the top toolbar, in em units\n (i.e. multiples of the default font size, to be Dpi aware)")
        .def_rw("window_padding_em", &HelloImGui::EdgeToolbarOptions::WindowPaddingEm, "Padding inside the window, in em units")
        .def_rw("window_bg", &HelloImGui::EdgeToolbarOptions::WindowBg, "Window background color, only used if WindowBg.w > 0")
        ;


    auto pyClassEdgeToolbar =
        nb::class_<HelloImGui::EdgeToolbar>
            (m, "EdgeToolbar", " EdgeToolbar :a toolbar that can be placed on the edges of the App window\n It will be placed in a non-dockable window")
        .def("__init__", [](HelloImGui::EdgeToolbar * self, const std::optional<const VoidFunction> & ShowToolbar = std::nullopt, const std::optional<const HelloImGui::EdgeToolbarOptions> & options = std::nullopt)
        {
            new (self) HelloImGui::EdgeToolbar();  // placement new
            auto r_ctor_ = self;
            if (ShowToolbar.has_value())
                r_ctor_->ShowToolbar = ShowToolbar.value();
            else
                r_ctor_->ShowToolbar = HelloImGui::EmptyVoidFunction();
            if (options.has_value())
                r_ctor_->options = options.value();
            else
                r_ctor_->options = HelloImGui::EdgeToolbarOptions();
        },
        nb::arg("show_toolbar").none() = nb::none(), nb::arg("options").none() = nb::none()
        )
        .def_rw("show_toolbar", &HelloImGui::EdgeToolbar::ShowToolbar, "")
        .def_rw("options", &HelloImGui::EdgeToolbar::options, "")
        ;


    m.def("all_edge_toolbar_types",
        HelloImGui::AllEdgeToolbarTypes);

    m.def("edge_toolbar_type_name",
        HelloImGui::EdgeToolbarTypeName, nb::arg("e"));


    auto pyEnumDefaultIconFont =
        nb::enum_<HelloImGui::DefaultIconFont>(m, "DefaultIconFont", nb::is_arithmetic(), " HelloImGui can optionally merge an icon font (FontAwesome 4 or 6) to the default font\n Breaking change in v1.5.0:\n - the default icon font is now FontAwesome 6, which includes many more icons.\n - you need to include manually icons_font_awesome_4.h or icons_font_awesome_6.h:\n     #include \"hello_imgui/icons_font_awesome_6.h\" or #include \"hello_imgui/icons_font_awesome_4.h\"")
            .value("no_icons", HelloImGui::DefaultIconFont::NoIcons, "")
            .value("font_awesome4", HelloImGui::DefaultIconFont::FontAwesome4, "")
            .value("font_awesome6", HelloImGui::DefaultIconFont::FontAwesome6, "");


    auto pyClassRunnerCallbacks =
        nb::class_<HelloImGui::RunnerCallbacks>
            (m, "RunnerCallbacks", " @@md#RunnerCallbacks\n RunnerCallbacks is a struct that contains the callbacks\n that are called by the application\n")
        .def("__init__", [](HelloImGui::RunnerCallbacks * self, const std::optional<const VoidFunction> & ShowGui = std::nullopt, const std::optional<const VoidFunction> & ShowMenus = std::nullopt, const std::optional<const VoidFunction> & ShowAppMenuItems = std::nullopt, const std::optional<const VoidFunction> & ShowStatus = std::nullopt, const std::optional<const VoidFunction> & PostInit_AddPlatformBackendCallbacks = std::nullopt, const std::optional<const VoidFunction> & PostInit = std::nullopt, const std::optional<const VoidFunction> & LoadAdditionalFonts = std::nullopt, HelloImGui::DefaultIconFont defaultIconFont = HelloImGui::DefaultIconFont::FontAwesome4, const std::optional<const VoidFunction> & SetupImGuiConfig = std::nullopt, const std::optional<const VoidFunction> & SetupImGuiStyle = std::nullopt, const std::optional<const VoidFunction> & RegisterTests = std::nullopt, bool registerTestsCalled = false, const std::optional<const VoidFunction> & BeforeExit = std::nullopt, const std::optional<const VoidFunction> & BeforeExit_PostCleanup = std::nullopt, const std::optional<const VoidFunction> & PreNewFrame = std::nullopt, const std::optional<const VoidFunction> & BeforeImGuiRender = std::nullopt, const std::optional<const VoidFunction> & AfterSwap = std::nullopt, const std::optional<const VoidFunction> & CustomBackground = std::nullopt, const std::optional<const VoidFunction> & PostRenderDockableWindows = std::nullopt, const std::optional<const AnyEventCallback> & AnyBackendEventCallback = std::nullopt)
        {
            new (self) HelloImGui::RunnerCallbacks();  // placement new
            auto r_ctor_ = self;
            if (ShowGui.has_value())
                r_ctor_->ShowGui = ShowGui.value();
            else
                r_ctor_->ShowGui = HelloImGui::EmptyVoidFunction();
            if (ShowMenus.has_value())
                r_ctor_->ShowMenus = ShowMenus.value();
            else
                r_ctor_->ShowMenus = HelloImGui::EmptyVoidFunction();
            if (ShowAppMenuItems.has_value())
                r_ctor_->ShowAppMenuItems = ShowAppMenuItems.value();
            else
                r_ctor_->ShowAppMenuItems = HelloImGui::EmptyVoidFunction();
            if (ShowStatus.has_value())
                r_ctor_->ShowStatus = ShowStatus.value();
            else
                r_ctor_->ShowStatus = HelloImGui::EmptyVoidFunction();
            if (PostInit_AddPlatformBackendCallbacks.has_value())
                r_ctor_->PostInit_AddPlatformBackendCallbacks = PostInit_AddPlatformBackendCallbacks.value();
            else
                r_ctor_->PostInit_AddPlatformBackendCallbacks = HelloImGui::EmptyVoidFunction();
            if (PostInit.has_value())
                r_ctor_->PostInit = PostInit.value();
            else
                r_ctor_->PostInit = HelloImGui::EmptyVoidFunction();
            if (LoadAdditionalFonts.has_value())
                r_ctor_->LoadAdditionalFonts = LoadAdditionalFonts.value();
            else
                r_ctor_->LoadAdditionalFonts = (VoidFunction)HelloImGui::ImGuiDefaultSettings::LoadDefaultFont_WithFontAwesomeIcons;
            r_ctor_->defaultIconFont = defaultIconFont;
            if (SetupImGuiConfig.has_value())
                r_ctor_->SetupImGuiConfig = SetupImGuiConfig.value();
            else
                r_ctor_->SetupImGuiConfig = (VoidFunction)HelloImGui::ImGuiDefaultSettings::SetupDefaultImGuiConfig;
            if (SetupImGuiStyle.has_value())
                r_ctor_->SetupImGuiStyle = SetupImGuiStyle.value();
            else
                r_ctor_->SetupImGuiStyle = (VoidFunction)HelloImGui::ImGuiDefaultSettings::SetupDefaultImGuiStyle;
            if (RegisterTests.has_value())
                r_ctor_->RegisterTests = RegisterTests.value();
            else
                r_ctor_->RegisterTests = HelloImGui::EmptyVoidFunction();
            r_ctor_->registerTestsCalled = registerTestsCalled;
            if (BeforeExit.has_value())
                r_ctor_->BeforeExit = BeforeExit.value();
            else
                r_ctor_->BeforeExit = HelloImGui::EmptyVoidFunction();
            if (BeforeExit_PostCleanup.has_value())
                r_ctor_->BeforeExit_PostCleanup = BeforeExit_PostCleanup.value();
            else
                r_ctor_->BeforeExit_PostCleanup = HelloImGui::EmptyVoidFunction();
            if (PreNewFrame.has_value())
                r_ctor_->PreNewFrame = PreNewFrame.value();
            else
                r_ctor_->PreNewFrame = HelloImGui::EmptyVoidFunction();
            if (BeforeImGuiRender.has_value())
                r_ctor_->BeforeImGuiRender = BeforeImGuiRender.value();
            else
                r_ctor_->BeforeImGuiRender = HelloImGui::EmptyVoidFunction();
            if (AfterSwap.has_value())
                r_ctor_->AfterSwap = AfterSwap.value();
            else
                r_ctor_->AfterSwap = HelloImGui::EmptyVoidFunction();
            if (CustomBackground.has_value())
                r_ctor_->CustomBackground = CustomBackground.value();
            else
                r_ctor_->CustomBackground = HelloImGui::EmptyVoidFunction();
            if (PostRenderDockableWindows.has_value())
                r_ctor_->PostRenderDockableWindows = PostRenderDockableWindows.value();
            else
                r_ctor_->PostRenderDockableWindows = HelloImGui::EmptyVoidFunction();
            if (AnyBackendEventCallback.has_value())
                r_ctor_->AnyBackendEventCallback = AnyBackendEventCallback.value();
            else
                r_ctor_->AnyBackendEventCallback = HelloImGui::EmptyEventCallback();
        },
        nb::arg("show_gui").none() = nb::none(), nb::arg("show_menus").none() = nb::none(), nb::arg("show_app_menu_items").none() = nb::none(), nb::arg("show_status").none() = nb::none(), nb::arg("post_init_add_platform_backend_callbacks").none() = nb::none(), nb::arg("post_init").none() = nb::none(), nb::arg("load_additional_fonts").none() = nb::none(), nb::arg("default_icon_font") = HelloImGui::DefaultIconFont::FontAwesome4, nb::arg("setup_imgui_config").none() = nb::none(), nb::arg("setup_imgui_style").none() = nb::none(), nb::arg("register_tests").none() = nb::none(), nb::arg("register_tests_called") = false, nb::arg("before_exit").none() = nb::none(), nb::arg("before_exit_post_cleanup").none() = nb::none(), nb::arg("pre_new_frame").none() = nb::none(), nb::arg("before_imgui_render").none() = nb::none(), nb::arg("after_swap").none() = nb::none(), nb::arg("custom_background").none() = nb::none(), nb::arg("post_render_dockable_windows").none() = nb::none(), nb::arg("any_backend_event_callback").none() = nb::none()
        )
        .def_rw("show_gui", &HelloImGui::RunnerCallbacks::ShowGui, " `ShowGui`: Fill it with a function that will add your widgets.\n (ShowGui will be called at each frame, before rendering the Dockable windows, if any)")
        .def_rw("show_menus", &HelloImGui::RunnerCallbacks::ShowMenus, " `ShowMenus`: Fill it with a function that will add ImGui menus by calling:\n       ImGui::BeginMenu(...) / ImGui::MenuItem(...) / ImGui::EndMenu()\n   Notes:\n   * you do not need to call ImGui::BeginMenuBar and ImGui::EndMenuBar\n   * Some default menus can be provided:\n     see ImGuiWindowParams options:\n         _showMenuBar, showMenu_App_QuitAbout, showMenu_View_")
        .def_rw("show_app_menu_items", &HelloImGui::RunnerCallbacks::ShowAppMenuItems, " `ShowAppMenuItems`: A function that will render items that will be placed\n in the App menu. They will be placed before the \"Quit\" MenuItem,\n which is added automatically by HelloImGui.\n  This will be displayed only if ImGuiWindowParams.showMenu_App is True")
        .def_rw("show_status", &HelloImGui::RunnerCallbacks::ShowStatus, " `ShowStatus`: A function that will add items to the status bar.\n  Use small items (ImGui::Text for example), since the height of the status is 30.\n  Also, remember to call ImGui::SameLine() between items.")
        .def_rw("edges_toolbars", &HelloImGui::RunnerCallbacks::edgesToolbars, " `EdgesToolbars`:\n A dict that contains toolbars that can be placed on the edges of the App window")
        .def("add_edge_toolbar",
            [](HelloImGui::RunnerCallbacks & self, HelloImGui::EdgeToolbarType edgeToolbarType, VoidFunction guiFunction, const std::optional<const HelloImGui::EdgeToolbarOptions> & options = std::nullopt)
            {
                auto AddEdgeToolbar_adapt_mutable_param_with_default_value = [&self](HelloImGui::EdgeToolbarType edgeToolbarType, VoidFunction guiFunction, const std::optional<const HelloImGui::EdgeToolbarOptions> & options = std::nullopt)
                {

                    const HelloImGui::EdgeToolbarOptions& options_or_default = [&]() -> const HelloImGui::EdgeToolbarOptions {
                        if (options.has_value())
                            return options.value();
                        else
                            return HelloImGui::EdgeToolbarOptions();
                    }();

                    self.AddEdgeToolbar(edgeToolbarType, guiFunction, options_or_default);
                };

                AddEdgeToolbar_adapt_mutable_param_with_default_value(edgeToolbarType, guiFunction, options);
            },
            nb::arg("edge_toolbar_type"), nb::arg("gui_function"), nb::arg("options").none() = nb::none(),
            " `AddEdgeToolbar`: Add a toolbar that can be placed on the edges of the App window\n\n\nPython bindings defaults:\n    If options is None, then its default value will be: EdgeToolbarOptions()")
        .def_rw("post_init_add_platform_backend_callbacks", &HelloImGui::RunnerCallbacks::PostInit_AddPlatformBackendCallbacks, " `PostInit_AddPlatformBackendCallbacks`:\n  You can here add a function that will be called once after OpenGL and ImGui are inited,\n  but before the platform backend callbacks are initialized.\n  If you, want to add your own glfw callbacks, you should use this function to do so\n  (and then ImGui will call your callbacks followed by its own callbacks)")
        .def_rw("post_init", &HelloImGui::RunnerCallbacks::PostInit, " `PostInit`: You can here add a function that will be called once after everything\n  is inited (ImGui, Platform and Renderer Backend)")
        .def("enqueue_post_init",
            &HelloImGui::RunnerCallbacks::EnqueuePostInit,
            nb::arg("callback"),
            " `EnqueuePostInit`: Add a function that will be called once after OpenGL\n  and ImGui are inited, but before the backend callback are initialized.\n  (this will modify the `PostInit` callback by appending the new callback (using `SequenceFunctions`)")
        .def_rw("load_additional_fonts", &HelloImGui::RunnerCallbacks::LoadAdditionalFonts, " `LoadAdditionalFonts`: default=_LoadDefaultFont_WithFontAwesome*.\n  A function that is called in order to load fonts.\n `LoadAdditionalFonts` will be called once, then *set to None*.\n If you want to load additional fonts, during the app execution, you can\n set LoadAdditionalFonts to a function that will load the additional fonts.")
        .def_rw("default_icon_font", &HelloImGui::RunnerCallbacks::defaultIconFont, " If LoadAdditionalFonts==LoadDefaultFont_WithFontAwesomeIcons, this parameter control\n which icon font will be loaded by default.")
        .def_rw("setup_imgui_config", &HelloImGui::RunnerCallbacks::SetupImGuiConfig, " `SetupImGuiConfig`: default=_ImGuiDefaultSettings::SetupDefaultImGuiConfig*.\n  If needed, change ImGui config via SetupImGuiConfig\n  (enable docking, gamepad, etc)")
        .def_rw("setup_imgui_style", &HelloImGui::RunnerCallbacks::SetupImGuiStyle, " `SetupImGuiStyle`: default=_ImGuiDefaultSettings::SetupDefaultImGuiConfig*.\n  If needed, set your own style by providing your own SetupImGuiStyle callback")
        .def_rw("register_tests", &HelloImGui::RunnerCallbacks::RegisterTests, " `RegisterTests`: A function that is called once ImGuiTestEngine is ready\n to be filled with tests and automations definitions.")
        .def_rw("register_tests_called", &HelloImGui::RunnerCallbacks::registerTestsCalled, " `registerTestsCalled`: will be set to True when RegisterTests was called\n (you can set this to False if you want to RegisterTests to be called again\n  during the app execution)")
        .def_rw("before_exit", &HelloImGui::RunnerCallbacks::BeforeExit, " `BeforeExit`: You can here add a function that will be called once before exiting\n  (when OpenGL and ImGui are still inited)")
        .def("enqueue_before_exit",
            &HelloImGui::RunnerCallbacks::EnqueueBeforeExit,
            nb::arg("callback"),
            " `EnqueueBeforeExit`: Add a function that will be called once before exiting\n  (when OpenGL and ImGui are still inited)\n (this will modify the `BeforeExit` callback by appending the new callback (using `SequenceFunctions`)")
        .def_rw("before_exit_post_cleanup", &HelloImGui::RunnerCallbacks::BeforeExit_PostCleanup, " `BeforeExit_PostCleanup`: You can here add a function that will be called once\n before exiting (after OpenGL and ImGui have been stopped)")
        .def_rw("pre_new_frame", &HelloImGui::RunnerCallbacks::PreNewFrame, " `PreNewFrame`: You can here add a function that will be called at each frame,\n  and before the call to ImGui::NewFrame().\n  It is a good place to add new dockable windows.")
        .def_rw("before_imgui_render", &HelloImGui::RunnerCallbacks::BeforeImGuiRender, " `BeforeImGuiRender`: You can here add a function that will be called at each frame,\n  after the user Gui code, and just before the call to\n  ImGui::Render() (which will also call ImGui::EndFrame()).")
        .def_rw("after_swap", &HelloImGui::RunnerCallbacks::AfterSwap, " `AfterSwap`: You can here add a function that will be called at each frame,\n  after the Gui was rendered and swapped to the screen.")
        .def_rw("custom_background", &HelloImGui::RunnerCallbacks::CustomBackground, " `CustomBackground`:\n  By default, the background is cleared using ImGuiWindowParams.backgroundColor.\n  If set, this function gives you full control over the background that is drawn\n  behind the Gui. An example use case is if you have a 3D application\n  like a mesh editor, or game, and just want the Gui to be drawn\n  on top of that content.")
        .def_rw("post_render_dockable_windows", &HelloImGui::RunnerCallbacks::PostRenderDockableWindows, " `PostRenderDockableWindows`: Fill it with a function that will be called\n after the dockable windows are rendered.")
        .def_rw("any_backend_event_callback", &HelloImGui::RunnerCallbacks::AnyBackendEventCallback, " `AnyBackendEventCallback`:\n  Callbacks for events from a specific backend. _Only implemented for SDL.\n  where the event will be of type 'SDL_Event *'_\n  This callback should return True if the event was handled\n  and shall not be processed further.\n  Note: in the case of GLFW, you should use register them in `PostInit`")
        ;


    m.def("append_callback",
        HelloImGui::AppendCallback,
        nb::arg("previous_callback"), nb::arg("new_callback"),
        "AppendCallback: legacy synonym for SequenceFunctions");


    auto pyClassDockingSplit =
        nb::class_<HelloImGui::DockingSplit>
            (m, "DockingSplit", " DockingSplit is a struct that defines the way the docking splits should\n be applied on the screen in order to create new Dock Spaces.\n DockingParams contains a\n     vector<DockingSplit>\n in order to partition the screen at your will.")
        .def_rw("initial_dock", &HelloImGui::DockingSplit::initialDock, " `initialDock`: _DockSpaceName (aka string)_\n  id of the space that should be split.\n  At the start, there is only one Dock Space named \"MainDockSpace\".\n  You should start by partitioning this space, in order to create a new dock space.")
        .def_rw("new_dock", &HelloImGui::DockingSplit::newDock, " `newDock`: _DockSpaceName (aka string)_.\n  id of the new dock space that will be created.")
        .def_rw("direction", &HelloImGui::DockingSplit::direction, " `direction`: *ImGuiDir_*\n  (enum with ImGuiDir_Down, ImGuiDir_Down, ImGuiDir_Left, ImGuiDir_Right)*\n  Direction where this dock space should be created.")
        .def_rw("ratio", &HelloImGui::DockingSplit::ratio, " `ratio`: _float, default=0.25_.\n  Ratio of the initialDock size that should be used by the new dock space.")
        .def_rw("node_flags", &HelloImGui::DockingSplit::nodeFlags, " `nodeFlags`: *ImGuiDockNodeFlags_ (enum)*.\n  Flags to apply to the new dock space\n  (enable/disable resizing, splitting, tab bar, etc.)")
        .def("__init__",
            [](HelloImGui::DockingSplit * self, const DockSpaceName & initialDock_ = "", const DockSpaceName & newDock_ = "", const std::optional<const ImGuiDir> & direction_ = std::nullopt, float ratio_ = 0.25f, const std::optional<const ImGuiDockNodeFlags> & nodeFlags_ = std::nullopt)
            {
                auto ctor_wrapper = [](HelloImGui::DockingSplit* self, const DockSpaceName & initialDock_ = "", const DockSpaceName & newDock_ = "", ImGuiDir direction_ = ImGuiDir_Down, float ratio_ = 0.25f, ImGuiDockNodeFlags nodeFlags_ = ImGuiDockNodeFlags_None) ->  void
                {
                    new(self) HelloImGui::DockingSplit(initialDock_, newDock_, direction_, ratio_, nodeFlags_); // placement new
                };
                auto ctor_wrapper_adapt_mutable_param_with_default_value = [&ctor_wrapper](HelloImGui::DockingSplit * self, const DockSpaceName & initialDock_ = "", const DockSpaceName & newDock_ = "", const std::optional<const ImGuiDir> & direction_ = std::nullopt, float ratio_ = 0.25f, const std::optional<const ImGuiDockNodeFlags> & nodeFlags_ = std::nullopt)
                {

                    const ImGuiDir& direction__or_default = [&]() -> const ImGuiDir {
                        if (direction_.has_value())
                            return direction_.value();
                        else
                            return ImGuiDir_Down;
                    }();

                    const ImGuiDockNodeFlags& nodeFlags__or_default = [&]() -> const ImGuiDockNodeFlags {
                        if (nodeFlags_.has_value())
                            return nodeFlags_.value();
                        else
                            return ImGuiDockNodeFlags_None;
                    }();

                    ctor_wrapper(self, initialDock_, newDock_, direction__or_default, ratio_, nodeFlags__or_default);
                };

                ctor_wrapper_adapt_mutable_param_with_default_value(self, initialDock_, newDock_, direction_, ratio_, nodeFlags_);
            },
            nb::arg("initial_dock_") = "", nb::arg("new_dock_") = "", nb::arg("direction_").none() = nb::none(), nb::arg("ratio_") = 0.25f, nb::arg("node_flags_").none() = nb::none(),
            " Constructor\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * direction_: Dir.down\n        * nodeFlags_: DockNodeFlags_.none")
        ;


    auto pyClassDockableWindow =
        nb::class_<HelloImGui::DockableWindow>
            (m, "DockableWindow", "DockableWindow is a struct that represents a window that can be docked.")
        .def_rw("label", &HelloImGui::DockableWindow::label, "`label`: _string_. Title of the window. It should be unique! Use \"##\" to add a unique suffix if needed.")
        .def_rw("dock_space_name", &HelloImGui::DockableWindow::dockSpaceName, " `dockSpaceName`: _DockSpaceName (aka string)_.\n  Id of the dock space where this window should initially be placed")
        .def_rw("gui_function", &HelloImGui::DockableWindow::GuiFunction, " `GuiFunction`: _VoidFunction_.\n Any function that will render this window's Gui")
        .def_rw("is_visible", &HelloImGui::DockableWindow::isVisible, " `isVisible`: _bool, default=true_.\n  Flag that indicates whether this window is visible or not.")
        .def_rw("remember_is_visible", &HelloImGui::DockableWindow::rememberIsVisible, " `rememberIsVisible`: _bool, default=true_.\n  Flag that indicates whether the window visibility should be saved in settings.")
        .def_rw("can_be_closed", &HelloImGui::DockableWindow::canBeClosed, " `canBeClosed`: _bool, default=true_.\n  Flag that indicates whether the user can close this window.")
        .def_rw("call_begin_end", &HelloImGui::DockableWindow::callBeginEnd, " `callBeginEnd`: _bool, default=true_.\n  Flag that indicates whether ImGui::Begin and ImGui::End\n  calls should be added automatically (with the given \"label\").\n  Set to False if you want to call ImGui::Begin/End yourself")
        .def_rw("include_in_view_menu", &HelloImGui::DockableWindow::includeInViewMenu, " `includeInViewMenu`: _bool, default=true_.\n  Flag that indicates whether this window should be mentioned in the view menu.")
        .def_rw("imgui_window_flags", &HelloImGui::DockableWindow::imGuiWindowFlags, " `imGuiWindowFlags`: _ImGuiWindowFlags, default=0_.\n  Window flags, see enum ImGuiWindowFlags_")
        .def_rw("focus_window_at_next_frame", &HelloImGui::DockableWindow::focusWindowAtNextFrame, " `focusWindowAtNextFrame`: _bool, default = false_.\n  If set to True this window will be focused at the next frame.")
        .def_rw("window_size", &HelloImGui::DockableWindow::windowSize, " `windowSize`: _ImVec2, default=(0., 0.) (i.e let the app decide)_.\n  Window size (unused if docked)")
        .def_rw("window_size_condition", &HelloImGui::DockableWindow::windowSizeCondition, " `windowSizeCondition`: _ImGuiCond, default=ImGuiCond_FirstUseEver_.\n  When to apply the window size.")
        .def_rw("window_position", &HelloImGui::DockableWindow::windowPosition, " `windowPos`: _ImVec2, default=(0., 0.) (i.e let the app decide)_.\n  Window position (unused if docked)")
        .def_rw("window_position_condition", &HelloImGui::DockableWindow::windowPositionCondition, " `windowPosCondition`: _ImGuiCond, default=ImGuiCond_FirstUseEver_.\n  When to apply the window position.")
        .def("__init__",
            [](HelloImGui::DockableWindow * self, const std::string & label_ = "", const DockSpaceName & dockSpaceName_ = "", const std::optional<const VoidFunction> & guiFunction_ = std::nullopt, bool isVisible_ = true, bool canBeClosed_ = true)
            {
                auto ctor_wrapper = [](HelloImGui::DockableWindow* self, const std::string & label_ = "", const DockSpaceName & dockSpaceName_ = "", const VoidFunction guiFunction_ = HelloImGui::EmptyVoidFunction(), bool isVisible_ = true, bool canBeClosed_ = true) ->  void
                {
                    new(self) HelloImGui::DockableWindow(label_, dockSpaceName_, guiFunction_, isVisible_, canBeClosed_); // placement new
                };
                auto ctor_wrapper_adapt_mutable_param_with_default_value = [&ctor_wrapper](HelloImGui::DockableWindow * self, const std::string & label_ = "", const DockSpaceName & dockSpaceName_ = "", const std::optional<const VoidFunction> & guiFunction_ = std::nullopt, bool isVisible_ = true, bool canBeClosed_ = true)
                {

                    const VoidFunction& guiFunction__or_default = [&]() -> const VoidFunction {
                        if (guiFunction_.has_value())
                            return guiFunction_.value();
                        else
                            return HelloImGui::EmptyVoidFunction();
                    }();

                    ctor_wrapper(self, label_, dockSpaceName_, guiFunction__or_default, isVisible_, canBeClosed_);
                };

                ctor_wrapper_adapt_mutable_param_with_default_value(self, label_, dockSpaceName_, guiFunction_, isVisible_, canBeClosed_);
            },
            nb::arg("label_") = "", nb::arg("dock_space_name_") = "", nb::arg("gui_function_").none() = nb::none(), nb::arg("is_visible_") = true, nb::arg("can_be_closed_") = true,
            " --------------- Constructor ------------------------------\n Constructor\n\n\nPython bindings defaults:\n    If guiFunction_ is None, then its default value will be: EmptyVoidFunction()")
        ;


    auto pyEnumDockingLayoutCondition =
        nb::enum_<HelloImGui::DockingLayoutCondition>(m, "DockingLayoutCondition", nb::is_arithmetic(), "")
            .value("first_use_ever", HelloImGui::DockingLayoutCondition::FirstUseEver, "")
            .value("application_start", HelloImGui::DockingLayoutCondition::ApplicationStart, "")
            .value("never", HelloImGui::DockingLayoutCondition::Never, "");


    auto pyClassDockingParams =
        nb::class_<HelloImGui::DockingParams>
            (m, "DockingParams", " DockingParams contains all the settings concerning the docking:\n     - list of splits\n     - list of dockable windows")
        .def("__init__", [](HelloImGui::DockingParams * self, const std::optional<const std::vector<HelloImGui::DockingSplit>> & dockingSplits = std::nullopt, const std::optional<const std::vector<HelloImGui::DockableWindow>> & dockableWindows = std::nullopt, std::string layoutName = "Default", const std::optional<const ImGuiDockNodeFlags> & mainDockSpaceNodeFlags = std::nullopt, HelloImGui::DockingLayoutCondition layoutCondition = HelloImGui::DockingLayoutCondition::FirstUseEver, bool layoutReset = false)
        {
            new (self) HelloImGui::DockingParams();  // placement new
            auto r_ctor_ = self;
            if (dockingSplits.has_value())
                r_ctor_->dockingSplits = dockingSplits.value();
            else
                r_ctor_->dockingSplits = std::vector<HelloImGui::DockingSplit>();
            if (dockableWindows.has_value())
                r_ctor_->dockableWindows = dockableWindows.value();
            else
                r_ctor_->dockableWindows = std::vector<HelloImGui::DockableWindow>();
            r_ctor_->layoutName = layoutName;
            if (mainDockSpaceNodeFlags.has_value())
                r_ctor_->mainDockSpaceNodeFlags = mainDockSpaceNodeFlags.value();
            else
                r_ctor_->mainDockSpaceNodeFlags = ImGuiDockNodeFlags_PassthruCentralNode;
            r_ctor_->layoutCondition = layoutCondition;
            r_ctor_->layoutReset = layoutReset;
        },
        nb::arg("docking_splits").none() = nb::none(), nb::arg("dockable_windows").none() = nb::none(), nb::arg("layout_name") = "Default", nb::arg("main_dock_space_node_flags").none() = nb::none(), nb::arg("layout_condition") = HelloImGui::DockingLayoutCondition::FirstUseEver, nb::arg("layout_reset") = false
        )
        .def_rw("docking_splits", &HelloImGui::DockingParams::dockingSplits, " `dockingSplits`: _vector[DockingSplit]_.\n  Defines the way docking splits should be applied on the screen\n  in order to create new Dock Spaces")
        .def_rw("dockable_windows", &HelloImGui::DockingParams::dockableWindows, " `dockableWindows`: _vector[DockableWindow]_.\n  List of the dockable windows, together with their Gui code")
        .def_rw("layout_name", &HelloImGui::DockingParams::layoutName, " `layoutName`: _string, default=\"default\"_.\n  Displayed name of the layout.\n  Only used in advanced cases, when several layouts are available.")
        .def_rw("main_dock_space_node_flags", &HelloImGui::DockingParams::mainDockSpaceNodeFlags, " `mainDockSpaceNodeFlags`: _ImGuiDockNodeFlags (enum),\n      default=ImGuiDockNodeFlags_PassthruCentralNode_\n  Flags to apply to the main dock space\n  (enable/disable resizing, splitting, tab bar, etc.).\n  Most flags are inherited by children dock spaces.\n  You can also set flags for specific dock spaces via `DockingSplit.nodeFlags`")
        .def_rw("layout_condition", &HelloImGui::DockingParams::layoutCondition, " `layoutCondition`: _enum DockingLayoutCondition, default=FirstUseEver_.\n  When to apply the docking layout. Choose between\n      FirstUseEver (apply once, then keep user preference),\n      ApplicationStart (always reapply at application start)\n      Never")
        .def_rw("layout_reset", &HelloImGui::DockingParams::layoutReset, " `layoutReset`: _bool, default=false_.\n  Reset layout on next frame, i.e. drop the layout customizations which were\n  applied manually by the user. layoutReset will be reset to False after this.")
        .def("dockable_window_of_name",
            &HelloImGui::DockingParams::dockableWindowOfName,
            nb::arg("name"),
            " `DockableWindow * dockableWindowOfName(const std::string & name)`:\n returns a pointer to a dockable window",
            nb::rv_policy::reference)
        .def("focus_dockable_window",
            &HelloImGui::DockingParams::focusDockableWindow,
            nb::arg("window_name"),
            " `bool focusDockableWindow(const std::string& name)`:\n will focus a dockable window (and make its tab visible if needed)")
        .def("dock_space_id_from_name",
            &HelloImGui::DockingParams::dockSpaceIdFromName,
            nb::arg("dock_space_name"),
            " `optional<ImGuiID> dockSpaceIdFromName(const std::string& dockSpaceName)`:\n returns the ImGuiID corresponding to the dockspace with this name")
        ;


    auto pyClassBackendPointers =
        nb::class_<HelloImGui::BackendPointers>
            (m, "BackendPointers", " @@md#BackendPointers\n\n BackendPointers is a struct that contains optional pointers to the\n backend implementations (for SDL and GLFW).\n\n These pointers will be filled when the application starts, and you can use them\n to customize your application behavior using the selected backend.\n\n Note: If using the Metal, Vulkan or DirectX rendering backend, you can find\n some interesting pointers inside\n     `src/hello_imgui/internal/backend_impls/rendering_metal.h`\n     `src/hello_imgui/internal/backend_impls/rendering_vulkan.h`\n     `src/hello_imgui/internal/backend_impls/rendering_dx11.h`\n     `src/hello_imgui/internal/backend_impls/rendering_dx12.h`")
        .def(nb::init<>()) // implicit default constructor
        .def_rw("glfw_window", &HelloImGui::BackendPointers::glfwWindow, "GLFWwindow*")
        .def_rw("sdl_window", &HelloImGui::BackendPointers::sdlWindow, "SDL_Window*")
        .def_rw("sdl_gl_context", &HelloImGui::BackendPointers::sdlGlContext, "SDL_GLContext")
        ;


    auto pyClassRemoteParams =
        nb::class_<HelloImGui::RemoteParams>
            (m, "RemoteParams", " RemoteParams is a struct that contains the settings for displaying the application on a remote device.\n using https://github.com/sammyfreg/netImgui\n or using https://github.com/ggerganov/imgui-ws\n Those features are experimental and not supported with the standard version of HelloImGui,")
        .def("__init__", [](HelloImGui::RemoteParams * self, bool enableRemoting = false, int wsPort = 5003, std::string wsHttpRootFolder = "", bool wsProvideIndexHtml = true, bool exitWhenServerDisconnected = false, double durationMaxDisconnected = 30.0, std::string serverHost = "localhost", uint32_t serverPort = 8888, bool transmitWindowSize = false)
        {
            new (self) HelloImGui::RemoteParams();  // placement new
            auto r_ctor_ = self;
            r_ctor_->enableRemoting = enableRemoting;
            r_ctor_->wsPort = wsPort;
            r_ctor_->wsHttpRootFolder = wsHttpRootFolder;
            r_ctor_->wsProvideIndexHtml = wsProvideIndexHtml;
            r_ctor_->exitWhenServerDisconnected = exitWhenServerDisconnected;
            r_ctor_->durationMaxDisconnected = durationMaxDisconnected;
            r_ctor_->serverHost = serverHost;
            r_ctor_->serverPort = serverPort;
            r_ctor_->transmitWindowSize = transmitWindowSize;
        },
        nb::arg("enable_remoting") = false, nb::arg("ws_port") = 5003, nb::arg("ws_http_root_folder") = "", nb::arg("ws_provide_index_html") = true, nb::arg("exit_when_server_disconnected") = false, nb::arg("duration_max_disconnected") = 30.0, nb::arg("server_host") = "localhost", nb::arg("server_port") = 8888, nb::arg("transmit_window_size") = false
        )
        .def_rw("enable_remoting", &HelloImGui::RemoteParams::enableRemoting, "")
        .def_rw("ws_port", &HelloImGui::RemoteParams::wsPort, "")
        .def_rw("ws_http_root_folder", &HelloImGui::RemoteParams::wsHttpRootFolder, "Optional folder were some additional files can be served")
        .def_rw("ws_provide_index_html", &HelloImGui::RemoteParams::wsProvideIndexHtml, "If True, will automatically serve a simple index.html file that contains the canvas and the imgui-ws client code")
        .def_rw("exit_when_server_disconnected", &HelloImGui::RemoteParams::exitWhenServerDisconnected, "")
        .def_rw("duration_max_disconnected", &HelloImGui::RemoteParams::durationMaxDisconnected, "")
        .def_rw("server_host", &HelloImGui::RemoteParams::serverHost, " The server host (if empty, will use \"localhost\")\n The server is the app that simply displays the application on a remote device")
        .def_rw("server_port", &HelloImGui::RemoteParams::serverPort, "The server port (default is 8888)")
        .def_rw("transmit_window_size", &HelloImGui::RemoteParams::transmitWindowSize, "If True, transmit the window size to the server")
        ;


    auto pyClassOpenGlOptions =
        nb::class_<HelloImGui::OpenGlOptions>
            (m, "OpenGlOptions", " OpenGlOptions contains advanced options used at the startup of OpenGL.\n These parameters are reserved for advanced users.\n By default, Hello ImGui will select reasonable default values, and these parameters are not used.\n Use at your own risk, as they make break the multi-platform compatibility of your application!\n All these parameters are platform dependent.\n For real multiplatform examples, see\n     hello_imgui/src/hello_imgui/internal/backend_impls/opengl_setup_helper/opengl_setup_glfw.cpp\n and\n     hello_imgui/src/hello_imgui/internal/backend_impls/opengl_setup_helper/opengl_setup_sdl.cpp\n\n How to set those values manually:\n ---------------------------------\n you may set them manually:\n    (1) Either by setting them programmatically in your application\n        (set their values in `runnerParams.rendererBackendOptions.openGlOptions`)\n    (2) Either by setting them in a `hello_imgui.ini` file in the current folder, or any of its parent folders.\n       (this is useful when you want to set them for a specific app or set of apps, without modifying the app code)\n       See hello_imgui/hello_imgui_example.ini for an example of such a file.\n Note: if several methods are used, the order of priority is (1) > (2)\n")
        .def("__init__", [](HelloImGui::OpenGlOptions * self, std::optional<std::string> GlslVersion = std::nullopt, std::optional<int> MajorVersion = std::nullopt, std::optional<int> MinorVersion = std::nullopt, std::optional<bool> UseCoreProfile = std::nullopt, std::optional<bool> UseForwardCompat = std::nullopt, std::optional<int> AntiAliasingSamples = std::nullopt)
        {
            new (self) HelloImGui::OpenGlOptions();  // placement new
            auto r_ctor_ = self;
            r_ctor_->GlslVersion = GlslVersion;
            r_ctor_->MajorVersion = MajorVersion;
            r_ctor_->MinorVersion = MinorVersion;
            r_ctor_->UseCoreProfile = UseCoreProfile;
            r_ctor_->UseForwardCompat = UseForwardCompat;
            r_ctor_->AntiAliasingSamples = AntiAliasingSamples;
        },
        nb::arg("glsl_version").none() = nb::none(), nb::arg("major_version").none() = nb::none(), nb::arg("minor_version").none() = nb::none(), nb::arg("use_core_profile").none() = nb::none(), nb::arg("use_forward_compat").none() = nb::none(), nb::arg("anti_aliasing_samples").none() = nb::none()
        )
        .def_rw("glsl_version", &HelloImGui::OpenGlOptions::GlslVersion, " Could be for example:\n    \"150\" on macOS\n    \"130\" on Windows\n    \"300es\" on GLES")
        .def_rw("major_version", &HelloImGui::OpenGlOptions::MajorVersion, "")
        .def_rw("minor_version", &HelloImGui::OpenGlOptions::MinorVersion, "")
        .def_rw("use_core_profile", &HelloImGui::OpenGlOptions::UseCoreProfile, "OpenGL Core Profile (i.e. only includes the newer, maintained features of OpenGL)")
        .def_rw("use_forward_compat", &HelloImGui::OpenGlOptions::UseForwardCompat, "OpenGL Forward Compatibility (required on macOS)")
        .def_rw("anti_aliasing_samples", &HelloImGui::OpenGlOptions::AntiAliasingSamples, " `AntiAliasingSamples`\n If > 0, this value will be used to set the number of samples used for anti-aliasing.\n This is used only when running with Glfw  + OpenGL (which is the default)\n Notes:\n - we query the maximum number of samples supported by the hardware, via glGetIntegerv(GL_MAX_SAMPLES)\n - if you set this value to a non-zero value, it will be used instead of the default value of 8\n   (except if it is greater than the maximum supported value, in which case a warning will be issued)\n - if you set this value to 0, antialiasing will be disabled\n\n AntiAliasingSamples has a strong impact on the quality of the text rendering\n     - 0: no antialiasing\n     - 8: optimal\n     - 16: optimal if using imgui-node-editor and you want to render very small text when unzooming")
        ;


    m.def("has_edr_support",
        HelloImGui::hasEdrSupport, " `bool hasEdrSupport()`:\n Check whether extended dynamic range (EDR), i.e. the ability to reproduce\n intensities exceeding the standard dynamic range from 0.0-1.0, is supported.\n\n To leverage EDR support, you need to set `floatBuffer=True` in `RendererBackendOptions`.\n Only the macOS Metal backend currently supports this.\n\n This currently returns False on all backends except Metal, where it checks whether\n this is supported on the current displays.");


    auto pyClassRendererBackendOptions =
        nb::class_<HelloImGui::RendererBackendOptions>
            (m, "RendererBackendOptions", " RendererBackendOptions is a struct that contains options for the renderer backend\n (Metal, Vulkan, DirectX, OpenGL)")
        .def("__init__", [](HelloImGui::RendererBackendOptions * self, bool requestFloatBuffer = false, const std::optional<const HelloImGui::OpenGlOptions> & openGlOptions = std::nullopt)
        {
            new (self) HelloImGui::RendererBackendOptions();  // placement new
            auto r_ctor_ = self;
            r_ctor_->requestFloatBuffer = requestFloatBuffer;
            if (openGlOptions.has_value())
                r_ctor_->openGlOptions = openGlOptions.value();
            else
                r_ctor_->openGlOptions = HelloImGui::OpenGlOptions();
        },
        nb::arg("request_float_buffer") = false, nb::arg("open_gl_options").none() = nb::none()
        )
        .def_rw("request_float_buffer", &HelloImGui::RendererBackendOptions::requestFloatBuffer, " `requestFloatBuffer`:\n Set to True to request a floating-point framebuffer.\n Only available on Metal, if your display supports it.\n Before setting this to True, first check `hasEdrSupport()`")
        .def_rw("open_gl_options", &HelloImGui::RendererBackendOptions::openGlOptions, " `openGlOptions`:\n Advanced options for OpenGL. Use at your own risk.")
        ;


    auto pyClassOpenGlOptionsFilled_ =
        nb::class_<HelloImGui::OpenGlOptionsFilled_>
            (m, "OpenGlOptionsFilled_", " (Private structure, not part of the public API)\n OpenGlOptions after selecting the default platform-dependent values + after applying the user settings")
        .def("__init__", [](HelloImGui::OpenGlOptionsFilled_ * self, std::string GlslVersion = "150", int MajorVersion = 3, int MinorVersion = 3, bool UseCoreProfile = true, bool UseForwardCompat = true, int AntiAliasingSamples = 8)
        {
            new (self) HelloImGui::OpenGlOptionsFilled_();  // placement new
            auto r_ctor_ = self;
            r_ctor_->GlslVersion = GlslVersion;
            r_ctor_->MajorVersion = MajorVersion;
            r_ctor_->MinorVersion = MinorVersion;
            r_ctor_->UseCoreProfile = UseCoreProfile;
            r_ctor_->UseForwardCompat = UseForwardCompat;
            r_ctor_->AntiAliasingSamples = AntiAliasingSamples;
        },
        nb::arg("glsl_version") = "150", nb::arg("major_version") = 3, nb::arg("minor_version") = 3, nb::arg("use_core_profile") = true, nb::arg("use_forward_compat") = true, nb::arg("anti_aliasing_samples") = 8
        )
        .def_rw("glsl_version", &HelloImGui::OpenGlOptionsFilled_::GlslVersion, "")
        .def_rw("major_version", &HelloImGui::OpenGlOptionsFilled_::MajorVersion, "")
        .def_rw("minor_version", &HelloImGui::OpenGlOptionsFilled_::MinorVersion, "")
        .def_rw("use_core_profile", &HelloImGui::OpenGlOptionsFilled_::UseCoreProfile, "")
        .def_rw("use_forward_compat", &HelloImGui::OpenGlOptionsFilled_::UseForwardCompat, "")
        .def_rw("anti_aliasing_samples", &HelloImGui::OpenGlOptionsFilled_::AntiAliasingSamples, "")
        ;


    auto pyEnumPlatformBackendType =
        nb::enum_<HelloImGui::PlatformBackendType>(m, "PlatformBackendType", nb::is_arithmetic(), " Platform backend type (SDL, GLFW)\n They are listed in the order of preference when FirstAvailable is selected.")
            .value("first_available", HelloImGui::PlatformBackendType::FirstAvailable, "")
            .value("glfw", HelloImGui::PlatformBackendType::Glfw, "")
            .value("sdl", HelloImGui::PlatformBackendType::Sdl, "")
            .value("null", HelloImGui::PlatformBackendType::Null, "");


    auto pyEnumRendererBackendType =
        nb::enum_<HelloImGui::RendererBackendType>(m, "RendererBackendType", nb::is_arithmetic(), " Rendering backend type (OpenGL3, Metal, Vulkan, DirectX11, DirectX12)\n They are listed in the order of preference when FirstAvailable is selected.")
            .value("first_available", HelloImGui::RendererBackendType::FirstAvailable, "")
            .value("open_gl3", HelloImGui::RendererBackendType::OpenGL3, "")
            .value("metal", HelloImGui::RendererBackendType::Metal, "")
            .value("vulkan", HelloImGui::RendererBackendType::Vulkan, "")
            .value("direct_x11", HelloImGui::RendererBackendType::DirectX11, "")
            .value("direct_x12", HelloImGui::RendererBackendType::DirectX12, "")
            .value("null", HelloImGui::RendererBackendType::Null, "");


    auto pyEnumIniFolderType =
        nb::enum_<HelloImGui::IniFolderType>(m, "IniFolderType", nb::is_arithmetic(), " IniFolderType is an enum which describes where is the base path to store\n the ini file for the application settings.\n\n You can use IniFolderLocation(iniFolderType) to get the corresponding path.\n\n RunnerParams contains the following members, which are used to compute\n the ini file location:\n     iniFolderType           (IniFolderType::CurrentFolder by default)\n     iniFilename             (empty string by default)\n     iniFilename_useAppWindowTitle\n         (True by default: iniFilename is derived from\n          appWindowParams.windowTitle)\n\n iniFilename may contain a subfolder\n (which will be created inside the iniFolderType folder if needed)\n")
            .value("current_folder", HelloImGui::IniFolderType::CurrentFolder, " CurrentFolder: the folder where the application is executed\n (convenient for development, but not recommended for production)")
            .value("absolute_path", HelloImGui::IniFolderType::AbsolutePath, " AbsolutePath: an absolute path\n (convenient, but not recommended if targeting multiple platforms)")
            .value("app_user_config_folder", HelloImGui::IniFolderType::AppUserConfigFolder, " AppUserConfigFolder:\n      AppData under Windows (Example: C:\\Users\\[Username]\\AppData\\Roaming under windows)\n      ~/.config under Linux\n      \"~/Library/Application Support\" under macOS\n (recommended for production, if settings do not need to be easily accessible by the user)")
            .value("app_executable_folder", HelloImGui::IniFolderType::AppExecutableFolder, " AppExecutableFolder: the folder where the application executable is located\n (this may be different from CurrentFolder if the application is launched from a shortcut)\n (convenient for development, but not recommended for production)")
            .value("home_folder", HelloImGui::IniFolderType::HomeFolder, " HomeFolder: the user home folder\n (recommended for production, if settings need to be easily accessible by the user)")
            .value("documents_folder", HelloImGui::IniFolderType::DocumentsFolder, "DocumentsFolder: the user documents folder")
            .value("temp_folder", HelloImGui::IniFolderType::TempFolder, "TempFolder: the system temp folder");


    m.def("ini_folder_location",
        HelloImGui::IniFolderLocation,
        nb::arg("ini_folder_type"),
        "Returns the path corresponding to the given IniFolderType");


    auto pyEnumFpsIdlingMode =
        nb::enum_<HelloImGui::FpsIdlingMode>(m, "FpsIdlingMode", nb::is_arithmetic(), " FpsIdlingMode is an enum that describes the different modes of idling when rendering the GUI.\n - Sleep: the application will sleep when idling to reduce CPU usage.\n - EarlyReturn: rendering will return immediately when idling.\n   This is specifically designed for event-driven, and real-time applications.\n   Avoid using it in a tight loop without pauses, as it may cause excessive CPU consumption.\n - Auto: use platform-specific default behavior.\n    On most platforms, it will sleep. On Emscripten, `Render()` will return immediately\n    to avoid blocking the main thread.\n Note: you can override the default behavior by explicitly setting Sleep or EarlyReturn.")
            .value("sleep", HelloImGui::FpsIdlingMode::Sleep, "")
            .value("early_return", HelloImGui::FpsIdlingMode::EarlyReturn, "")
            .value("auto", HelloImGui::FpsIdlingMode::Auto, "");


    auto pyClassFpsIdling =
        nb::class_<HelloImGui::FpsIdling>
            (m, "FpsIdling", "FpsIdling is a struct that contains Fps Idling parameters")
        .def("__init__", [](HelloImGui::FpsIdling * self, float fpsIdle = 9.f, float timeActiveAfterLastEvent = 3.f, bool enableIdling = true, bool isIdling = false, bool rememberEnableIdling = false, HelloImGui::FpsIdlingMode fpsIdlingMode = HelloImGui::FpsIdlingMode::Auto)
        {
            new (self) HelloImGui::FpsIdling();  // placement new
            auto r_ctor_ = self;
            r_ctor_->fpsIdle = fpsIdle;
            r_ctor_->timeActiveAfterLastEvent = timeActiveAfterLastEvent;
            r_ctor_->enableIdling = enableIdling;
            r_ctor_->isIdling = isIdling;
            r_ctor_->rememberEnableIdling = rememberEnableIdling;
            r_ctor_->fpsIdlingMode = fpsIdlingMode;
        },
        nb::arg("fps_idle") = 9.f, nb::arg("time_active_after_last_event") = 3.f, nb::arg("enable_idling") = true, nb::arg("is_idling") = false, nb::arg("remember_enable_idling") = false, nb::arg("fps_idling_mode") = HelloImGui::FpsIdlingMode::Auto
        )
        .def_rw("fps_idle", &HelloImGui::FpsIdling::fpsIdle, " `fpsIdle`: _float, default=9_.\n  ImGui applications can consume a lot of CPU, since they update the screen\n  very frequently. In order to reduce the CPU usage, the FPS is reduced when\n  no user interaction is detected.\n  This is ok most of the time but if you are displaying animated widgets\n  (for example a live video), you may want to ask for a faster refresh:\n  either increase fpsIdle, or set it to 0 for maximum refresh speed\n  (you can change this value during the execution depending on your application\n  refresh needs)")
        .def_rw("time_active_after_last_event", &HelloImGui::FpsIdling::timeActiveAfterLastEvent, " `timeActiveAfterLastEvent`: _float, default=3._.\n  Time in seconds after the last event before the application is considered idling.")
        .def_rw("enable_idling", &HelloImGui::FpsIdling::enableIdling, " `enableIdling`: _bool, default=true_.\n  Disable idling by setting this to False.\n  (this can be changed dynamically during execution)")
        .def_rw("is_idling", &HelloImGui::FpsIdling::isIdling, " `isIdling`: bool (dynamically updated during execution)\n  This bool will be updated during the application execution,\n  and will be set to True when it is idling.")
        .def_rw("remember_enable_idling", &HelloImGui::FpsIdling::rememberEnableIdling, " `rememberEnableIdling`: _bool, default=true_.\n  If True, the last value of enableIdling is restored from the settings at startup.")
        .def_rw("fps_idling_mode", &HelloImGui::FpsIdling::fpsIdlingMode, " `fpsIdlingMode`: _FpsIdlingMode, default=FpsIdlingMode::Automatic_.\n Sets the mode of idling when rendering the GUI (Sleep, EarlyReturn, Automatic)")
        ;


    auto pyClassRunnerParams =
        nb::class_<HelloImGui::RunnerParams>
            (m, "RunnerParams", " RunnerParams contains the settings and callbacks needed to run an application.\n")
        .def("__init__", [](HelloImGui::RunnerParams * self, const std::optional<const HelloImGui::RunnerCallbacks> & callbacks = std::nullopt, const std::optional<const HelloImGui::AppWindowParams> & appWindowParams = std::nullopt, const std::optional<const HelloImGui::ImGuiWindowParams> & imGuiWindowParams = std::nullopt, const std::optional<const HelloImGui::DockingParams> & dockingParams = std::nullopt, const std::optional<const std::vector<HelloImGui::DockingParams>> & alternativeDockingLayouts = std::nullopt, bool rememberSelectedAlternativeLayout = true, const std::optional<const HelloImGui::BackendPointers> & backendPointers = std::nullopt, const std::optional<const HelloImGui::RendererBackendOptions> & rendererBackendOptions = std::nullopt, HelloImGui::PlatformBackendType platformBackendType = HelloImGui::PlatformBackendType::FirstAvailable, HelloImGui::RendererBackendType rendererBackendType = HelloImGui::RendererBackendType::FirstAvailable, HelloImGui::IniFolderType iniFolderType = HelloImGui::IniFolderType::CurrentFolder, std::string iniFilename = "", bool iniFilename_useAppWindowTitle = true, bool appShallExit = false, const std::optional<const HelloImGui::FpsIdling> & fpsIdling = std::nullopt, const std::optional<const HelloImGui::DpiAwareParams> & dpiAwareParams = std::nullopt, bool useImGuiTestEngine = false, int emscripten_fps = 0)
        {
            new (self) HelloImGui::RunnerParams();  // placement new
            auto r_ctor_ = self;
            if (callbacks.has_value())
                r_ctor_->callbacks = callbacks.value();
            else
                r_ctor_->callbacks = HelloImGui::RunnerCallbacks();
            if (appWindowParams.has_value())
                r_ctor_->appWindowParams = appWindowParams.value();
            else
                r_ctor_->appWindowParams = HelloImGui::AppWindowParams();
            if (imGuiWindowParams.has_value())
                r_ctor_->imGuiWindowParams = imGuiWindowParams.value();
            else
                r_ctor_->imGuiWindowParams = HelloImGui::ImGuiWindowParams();
            if (dockingParams.has_value())
                r_ctor_->dockingParams = dockingParams.value();
            else
                r_ctor_->dockingParams = HelloImGui::DockingParams();
            if (alternativeDockingLayouts.has_value())
                r_ctor_->alternativeDockingLayouts = alternativeDockingLayouts.value();
            else
                r_ctor_->alternativeDockingLayouts = std::vector<HelloImGui::DockingParams>();
            r_ctor_->rememberSelectedAlternativeLayout = rememberSelectedAlternativeLayout;
            if (backendPointers.has_value())
                r_ctor_->backendPointers = backendPointers.value();
            else
                r_ctor_->backendPointers = HelloImGui::BackendPointers();
            if (rendererBackendOptions.has_value())
                r_ctor_->rendererBackendOptions = rendererBackendOptions.value();
            else
                r_ctor_->rendererBackendOptions = HelloImGui::RendererBackendOptions();
            r_ctor_->platformBackendType = platformBackendType;
            r_ctor_->rendererBackendType = rendererBackendType;
            r_ctor_->iniFolderType = iniFolderType;
            r_ctor_->iniFilename = iniFilename;
            r_ctor_->iniFilename_useAppWindowTitle = iniFilename_useAppWindowTitle;
            r_ctor_->appShallExit = appShallExit;
            if (fpsIdling.has_value())
                r_ctor_->fpsIdling = fpsIdling.value();
            else
                r_ctor_->fpsIdling = HelloImGui::FpsIdling();
            if (dpiAwareParams.has_value())
                r_ctor_->dpiAwareParams = dpiAwareParams.value();
            else
                r_ctor_->dpiAwareParams = HelloImGui::DpiAwareParams();
            r_ctor_->useImGuiTestEngine = useImGuiTestEngine;
            r_ctor_->emscripten_fps = emscripten_fps;
        },
        nb::arg("callbacks").none() = nb::none(), nb::arg("app_window_params").none() = nb::none(), nb::arg("imgui_window_params").none() = nb::none(), nb::arg("docking_params").none() = nb::none(), nb::arg("alternative_docking_layouts").none() = nb::none(), nb::arg("remember_selected_alternative_layout") = true, nb::arg("backend_pointers").none() = nb::none(), nb::arg("renderer_backend_options").none() = nb::none(), nb::arg("platform_backend_type") = HelloImGui::PlatformBackendType::FirstAvailable, nb::arg("renderer_backend_type") = HelloImGui::RendererBackendType::FirstAvailable, nb::arg("ini_folder_type") = HelloImGui::IniFolderType::CurrentFolder, nb::arg("ini_filename") = "", nb::arg("ini_filename_use_app_window_title") = true, nb::arg("app_shall_exit") = false, nb::arg("fps_idling").none() = nb::none(), nb::arg("dpi_aware_params").none() = nb::none(), nb::arg("use_imgui_test_engine") = false, nb::arg("emscripten_fps") = 0
        )
        .def_rw("callbacks", &HelloImGui::RunnerParams::callbacks, " `callbacks`: _see runner_callbacks.h_\n callbacks.ShowGui() will render the gui, ShowMenus() will show the menus, etc.")
        .def_rw("app_window_params", &HelloImGui::RunnerParams::appWindowParams, " `appWindowParams`: _see app_window_params.h_\n application Window Params (position, size, title)")
        .def_rw("imgui_window_params", &HelloImGui::RunnerParams::imGuiWindowParams, " `imGuiWindowParams`: _see imgui_window_params.h_\n imgui window params (use docking, showMenuBar, ProvideFullScreenWindow, etc.)")
        .def_rw("docking_params", &HelloImGui::RunnerParams::dockingParams, " `dockingParams`: _see docking_params.h_\n dockable windows content and layout")
        .def_rw("alternative_docking_layouts", &HelloImGui::RunnerParams::alternativeDockingLayouts, " `alternativeDockingLayouts`: _vector<DockingParams>, default=empty_\n List of possible additional layout for the applications. Only used in advanced\n cases when several layouts are available.")
        .def_rw("remember_selected_alternative_layout", &HelloImGui::RunnerParams::rememberSelectedAlternativeLayout, " `rememberSelectedAlternativeLayout`: _bool, default=true_\n Shall the application remember the last selected layout. Only used in advanced\n cases when several layouts are available.")
        .def_rw("backend_pointers", &HelloImGui::RunnerParams::backendPointers, " `backendPointers`: _see backend_pointers.h_\n A struct that contains optional pointers to the backend implementations.\n These pointers will be filled when the application starts")
        .def_rw("renderer_backend_options", &HelloImGui::RunnerParams::rendererBackendOptions, " `rendererBackendOptions`: _see renderer_backend_options.h_\n Options for the renderer backend")
        .def_rw("platform_backend_type", &HelloImGui::RunnerParams::platformBackendType, " `platformBackendType`: _enum PlatformBackendType, default=PlatformBackendType::FirstAvailable_\n Select the wanted platform backend type between `Sdl`, `Glfw`.\n if `FirstAvailable`, Glfw will be preferred over Sdl when both are available.\n Only useful when multiple backend are compiled and available.")
        .def_rw("renderer_backend_type", &HelloImGui::RunnerParams::rendererBackendType, " `renderingBackendType`: _enum RenderingBackendType, default=RenderingBackendType::FirstAvailable_\n Select the wanted rendering backend type between `OpenGL3`, `Metal`, `Vulkan`, `DirectX11`, `DirectX12`.\n if `FirstAvailable`, it will be selected in the order of preference mentioned above.\n Only useful when multiple rendering backend are compiled and available.")
        .def_rw("ini_folder_type", &HelloImGui::RunnerParams::iniFolderType, " `iniFolderType`: _IniFolderType, default = IniFolderType::CurrentFolder_\n Sets the folder where imgui will save its params.\n (possible values are:\n     CurrentFolder, AppUserConfigFolder, DocumentsFolder,\n     HomeFolder, TempFolder, AppExecutableFolder)\n AppUserConfigFolder is\n     [Home]/AppData/Roaming under Windows,\n     ~/.config under Linux,\n     ~/Library/Application Support under macOS")
        .def_rw("ini_filename", &HelloImGui::RunnerParams::iniFilename, "relative to iniFolderType")
        .def_rw("ini_filename_use_app_window_title", &HelloImGui::RunnerParams::iniFilename_useAppWindowTitle, " `iniFilename_useAppWindowTitle`: _bool, default = true_.\n Shall the iniFilename be derived from appWindowParams.windowTitle (if not empty)")
        .def_rw("app_shall_exit", &HelloImGui::RunnerParams::appShallExit, " * `appShallExit`: _bool, default=false_.\n During execution, set this to True to exit the app.\n _Note: 'appShallExit' has no effect on Mobile Devices (iOS, Android)\n and under emscripten, since these apps shall not exit._")
        .def_rw("fps_idling", &HelloImGui::RunnerParams::fpsIdling, " `fpsIdling`: _FpsIdling_. Idling parameters\n (set fpsIdling.enableIdling to False to disable Idling)")
        .def_rw("dpi_aware_params", &HelloImGui::RunnerParams::dpiAwareParams, " --------------- DPI Handling -----------\n Hello ImGui will try its best to automatically handle DPI scaling for you.\n If it fails, look at DpiAwareParams (and the corresponding Ini file settings)")
        .def_rw("use_imgui_test_engine", &HelloImGui::RunnerParams::useImGuiTestEngine, " `useImGuiTestEngine`: _bool, default=false_.\n Set this to True if you intend to use Dear ImGui Test & Automation Engine\n     ( https://github.com/ocornut/imgui_test_engine )\n HelloImGui must be compiled with the option -DHELLOIMGUI_WITH_TEST_ENGINE=ON\n See demo in src/hello_imgui_demos/hello_imgui_demo_test_engine.\n License:\n imgui_test_engine is subject to a specific license:\n     https://github.com/ocornut/imgui_test_engine/blob/main/imgui_test_engine/LICENSE.txt)\n (TL;DR: free for individuals, educational, open-source and small businesses uses.\n  Paid for larger businesses.)")
        .def_rw("emscripten_fps", &HelloImGui::RunnerParams::emscripten_fps, " `emscripten_fps`: _int, default = 0_.\n Set the application refresh rate\n (only used on emscripten: 0 stands for \"let the app or the browser decide\")")
        ;


    m.def("ini_settings_location",
        HelloImGui::IniSettingsLocation,
        nb::arg("runner_params"),
        "IniSettingsLocation returns the path to the ini file for the application settings.");

    m.def("has_ini_settings",
        HelloImGui::HasIniSettings,
        nb::arg("runner_params"),
        "HasIniSettings returns True if the ini file for the application settings exists.");

    m.def("delete_ini_settings",
        HelloImGui::DeleteIniSettings,
        nb::arg("runner_params"),
        "DeleteIniSettings deletes the ini file for the application settings.");


    auto pyClassSimpleRunnerParams =
        nb::class_<HelloImGui::SimpleRunnerParams>
            (m, "SimpleRunnerParams", " SimpleRunnerParams is a struct that contains simpler params adapted for simple use cases.\nFor example, this is sufficient to run an application:\n    ```cpp\n    None MyGui() {\n        ImGui::Text(\"Hello, world\");\n        if (ImGui::Button(\"Exit\"))\n            HelloImGui::GetRunnerParams()->appShallExit = True;\n    }\n\n    int main(){\n        auto params = HelloImGui::SimpleRunnerParams {\n            .guiFunction = MyGui, .windowSizeAuto = True, .windowTitle = \"Example\"\n        };\n        HelloImGui::Run(params);\n    }\n    ```")
        .def("__init__", [](HelloImGui::SimpleRunnerParams * self, const std::optional<const VoidFunction> & guiFunction = std::nullopt, std::string windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, const std::optional<const ScreenSize> & windowSize = std::nullopt, float fpsIdle = 9.f, bool enableIdling = true)
        {
            new (self) HelloImGui::SimpleRunnerParams();  // placement new
            auto r_ctor_ = self;
            if (guiFunction.has_value())
                r_ctor_->guiFunction = guiFunction.value();
            else
                r_ctor_->guiFunction = HelloImGui::EmptyVoidFunction();
            r_ctor_->windowTitle = windowTitle;
            r_ctor_->windowSizeAuto = windowSizeAuto;
            r_ctor_->windowRestorePreviousGeometry = windowRestorePreviousGeometry;
            if (windowSize.has_value())
                r_ctor_->windowSize = windowSize.value();
            else
                r_ctor_->windowSize = HelloImGui::DefaultWindowSize;
            r_ctor_->fpsIdle = fpsIdle;
            r_ctor_->enableIdling = enableIdling;
        },
        nb::arg("gui_function").none() = nb::none(), nb::arg("window_title") = "", nb::arg("window_size_auto") = false, nb::arg("window_restore_previous_geometry") = false, nb::arg("window_size").none() = nb::none(), nb::arg("fps_idle") = 9.f, nb::arg("enable_idling") = true
        )
        .def_rw("gui_function", &HelloImGui::SimpleRunnerParams::guiFunction, " `guiFunction`: _VoidFunction_.\n  Function that renders the Gui.")
        .def_rw("window_title", &HelloImGui::SimpleRunnerParams::windowTitle, " `windowTitle`: _string, default=\"\"_.\n  Title of the application window")
        .def_rw("window_size_auto", &HelloImGui::SimpleRunnerParams::windowSizeAuto, " `windowSizeAuto`: _bool, default=false_.\n  If True, the size of the window will be computed from its widgets.")
        .def_rw("window_restore_previous_geometry", &HelloImGui::SimpleRunnerParams::windowRestorePreviousGeometry, " `windowRestorePreviousGeometry`: _bool, default=true_.\n  If True, restore the size and position of the window between runs.")
        .def_rw("window_size", &HelloImGui::SimpleRunnerParams::windowSize, " `windowSize`: _ScreenSize, default={800, 600}_.\n  Size of the window\n The size will be handled as if it was specified for a 96PPI screen\n (i.e. a given size will correspond to the same physical size on different screens, whatever their DPI)")
        .def_rw("fps_idle", &HelloImGui::SimpleRunnerParams::fpsIdle, " `fpsIdle`: _float, default=9_.\n  FPS of the application when idle (set to 0 for full speed).")
        .def_rw("enable_idling", &HelloImGui::SimpleRunnerParams::enableIdling, " `enableIdling`: _bool, default=true_.\n  Disable idling at startup by setting this to False\n  When running, use:\n      HelloImGui::GetRunnerParams()->fpsIdling.enableIdling = False;")
        .def("to_runner_params",
            &HelloImGui::SimpleRunnerParams::ToRunnerParams)
        ;


    m.def("begin_group_column",
        HelloImGui::BeginGroupColumn, "calls ImGui::BeginGroup()");

    m.def("end_group_column",
        HelloImGui::EndGroupColumn, "calls ImGui::EndGroup() + ImGui::SameLine()");

    m.def("widget_with_resize_handle",
        HelloImGui::WidgetWithResizeHandle,
        nb::arg("id"), nb::arg("widget_gui_function"), nb::arg("handle_size_em") = 1.0f, nb::arg("on_item_resized").none() = nb::none(), nb::arg("on_item_hovered").none() = nb::none(),
        " WidgetWithResizeHandle: adds a resize handle to a widget\n Example usage with ImPlot:\n        None gui()\n        {\n            static ImVec2 widget_size(200, 200);\n            auto myWidgetFunction = []()\n            {\n                if (ImPlot::BeginPlot(\"My Plot\", widget_size)) {\n                    ImPlot::PlotLine(\"My Line\", x.data(), y.data(), 1000);\n                    ImPlot::EndPlot();\n                }\n            };\n            widget_size = widget_with_resize_handle(\"plot\", myWidgetFunction);\n        }");


    auto pyClassInputTextData =
        nb::class_<HelloImGui::InputTextData>
            (m, "InputTextData", " `InputTextResizable`: displays a resizable text input widget\n\n The `InputTextResizable` widget allows you to create a text input field that can be resized by the user.\n It supports both single-line and multi-line text input.\n Note: the size of the widget is expressed in em units.\n **Usage example:**\n    C++:\n       ```cpp\n     // Somewhere in the application state\n       (static) InputTextData textInput(\"My text\", True, ImVec2(10, 3));\n     // In the GUI function\n       bool changed = InputTextResizable(\"Label\", &textInput);\n       ```\n    Python:\n       ```python\n       # Somewhere in the application state\n       text_input = hello_imgui.InputTextData(\"My text\", multiline=True, size_em=ImVec2(10, 3))\n       # In the GUI function\n       changed = hello_imgui.input_text_resizable(\"Label\", text_input)\n       ```")
        .def_rw("text", &HelloImGui::InputTextData::Text, "The text edited in the input field")
        .def_rw("hint", &HelloImGui::InputTextData::Hint, " An optional hint displayed when the input field is empty\n (only works for single-line text input)")
        .def_rw("multiline", &HelloImGui::InputTextData::Multiline, "If True, the input field is multi-line")
        .def_rw("resizable", &HelloImGui::InputTextData::Resizable, "If True, the input field is resizable")
        .def_rw("size_em", &HelloImGui::InputTextData::SizeEm, "The size of the input field in em units")
        .def("__init__",
            [](HelloImGui::InputTextData * self, const std::string & text = "", bool multiline = false, const std::optional<const ImVec2> & size_em = std::nullopt)
            {
                auto ctor_wrapper = [](HelloImGui::InputTextData* self, const std::string & text = "", bool multiline = false, ImVec2 size_em = ImVec2(0, 0)) ->  void
                {
                    new(self) HelloImGui::InputTextData(text, multiline, size_em); // placement new
                };
                auto ctor_wrapper_adapt_mutable_param_with_default_value = [&ctor_wrapper](HelloImGui::InputTextData * self, const std::string & text = "", bool multiline = false, const std::optional<const ImVec2> & size_em = std::nullopt)
                {

                    const ImVec2& size_em_or_default = [&]() -> const ImVec2 {
                        if (size_em.has_value())
                            return size_em.value();
                        else
                            return ImVec2(0, 0);
                    }();

                    ctor_wrapper(self, text, multiline, size_em_or_default);
                };

                ctor_wrapper_adapt_mutable_param_with_default_value(self, text, multiline, size_em);
            },
            nb::arg("text") = "", nb::arg("multiline") = false, nb::arg("size_em").none() = nb::none(),
            "Python bindings defaults:\n    If size_em is None, then its default value will be: ImVec2(0, 0)")
        ;


    m.def("input_text_resizable",
        HelloImGui::InputTextResizable, nb::arg("label"), nb::arg("text_input"));

    m.def("input_text_data_to_dict",
        HelloImGui::InputTextDataToDict, nb::arg("data"));

    m.def("input_text_data_from_dict",
        HelloImGui::InputTextDataFromDict, nb::arg("dict"));

    m.def("input_text_data_to_string",
        HelloImGui::InputTextDataToString, nb::arg("data"));

    m.def("input_text_data_from_string",
        HelloImGui::InputTextDataFromString, nb::arg("str"));


    m.def("run",
        nb::overload_cast<HelloImGui::RunnerParams &>(HelloImGui::Run),
        nb::arg("runner_params"),
        " `HelloImGui::Run(RunnerParams &)`: full signature, the most customizable version.\n Runs an application whose params and Gui are provided by runnerParams.");

    m.def("run",
        nb::overload_cast<const HelloImGui::SimpleRunnerParams &>(HelloImGui::Run),
        nb::arg("simple_params"),
        " `HelloImGui::Run(const SimpleRunnerParams&)`:\n Runs an application, using simpler params.");

    m.def("run",
        [](const VoidFunction & guiFunction, const std::string & windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, const std::optional<const ScreenSize> & windowSize = std::nullopt, float fpsIdle = 10.f)
        {
            auto Run_adapt_mutable_param_with_default_value = [](const VoidFunction & guiFunction, const std::string & windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, const std::optional<const ScreenSize> & windowSize = std::nullopt, float fpsIdle = 10.f)
            {

                const ScreenSize& windowSize_or_default = [&]() -> const ScreenSize {
                    if (windowSize.has_value())
                        return windowSize.value();
                    else
                        return HelloImGui::DefaultWindowSize;
                }();

                HelloImGui::Run(guiFunction, windowTitle, windowSizeAuto, windowRestorePreviousGeometry, windowSize_or_default, fpsIdle);
            };

            Run_adapt_mutable_param_with_default_value(guiFunction, windowTitle, windowSizeAuto, windowRestorePreviousGeometry, windowSize, fpsIdle);
        },
        nb::arg("gui_function"), nb::arg("window_title") = "", nb::arg("window_size_auto") = false, nb::arg("window_restore_previous_geometry") = false, nb::arg("window_size").none() = nb::none(), nb::arg("fps_idle") = 10.f,
        " Runs an application, by providing the Gui function, the window title, etc.\n\n\nPython bindings defaults:\n    If windowSize is None, then its default value will be: DefaultWindowSize");

    m.def("get_runner_params",
        HelloImGui::GetRunnerParams,
        " `GetRunnerParams()`:  a convenience function that will return the runnerParams\n of the current application",
        nb::rv_policy::reference);

    m.def("is_using_hello_imgui",
        HelloImGui::IsUsingHelloImGui, "`IsUsingHelloImGui()`: returns True if the application is using HelloImGui");

    m.def("frame_rate",
        HelloImGui::FrameRate,
        nb::arg("duration_for_mean") = 0.5f,
        " `FrameRate(durationForMean = 0.5)`: Returns the current FrameRate.\n  May differ from ImGui::GetIO().FrameRate, since one can choose the duration\n  for the calculation of the mean value of the fps\n  Returns the current FrameRate. May differ from ImGui::GetIO().FrameRate,\n  since one can choose the duration for the calculation of the mean value of the fps\n  (Will only lead to accurate values if you call it at each frame)");

    m.def("get_imgui_test_engine",
        HelloImGui::GetImGuiTestEngine,
        " `ImGuiTestEngine* GetImGuiTestEngine()`: returns a pointer to the global instance\n  of ImGuiTestEngine that was initialized by HelloImGui\n  (iif ImGui Test Engine is active).",
        nb::rv_policy::reference);

    m.def("get_backend_description",
        HelloImGui::GetBackendDescription, " `GetBackendDescription()`: returns a string with the backend info\n Could be for example:\n     \"Glfw - OpenGL3\"\n     \"Glfw - Metal\"\n     \"Sdl - Vulkan\"");

    m.def("change_window_size",
        HelloImGui::ChangeWindowSize,
        nb::arg("window_size"),
        " `ChangeWindowSize(const ScreenSize &windowSize)`: sets the window size\n (useful if you want to change the window size during execution)");

    m.def("use_window_full_monitor_work_area",
        HelloImGui::UseWindowFullMonitorWorkArea, " `UseWindowFullMonitorWorkArea()`: sets the window size to the monitor work area\n (useful if you want to change the window size during execution)");

    m.def("switch_layout",
        HelloImGui::SwitchLayout,
        nb::arg("layout_name"),
        " `SwitchLayout(layoutName)`\n  Changes the application current layout. Only used in advanced cases\n  when several layouts are available, i.e. if you filled\n      runnerParams.alternativeDockingLayouts.");

    m.def("current_layout_name",
        HelloImGui::CurrentLayoutName, "`CurrentLayoutName()`: returns the name of the current layout");

    m.def("add_dockable_window",
        HelloImGui::AddDockableWindow,
        nb::arg("dockable_window"), nb::arg("force_dockspace") = false,
        " `AddDockableWindow()`: will add a dockable window to the current layout.\n Will dock the window to the dockspace it belongs to if forceDockspace is True,\n otherwise will dock it to the last space it was docked to (using saved settings)");

    m.def("remove_dockable_window",
        HelloImGui::RemoveDockableWindow,
        nb::arg("dockable_window_name"),
        " `RemoveDockableWindow()`: will remove a dockable window from the current layout.\n (dockableWindowName is the label of the window, as provided in the DockableWindow struct)");

    m.def("save_user_pref",
        HelloImGui::SaveUserPref,
        nb::arg("user_pref_name"), nb::arg("user_pref_content"),
        " `SaveUserPref(string userPrefName, string userPrefContent)`:\n  Shall be called in the callback runnerParams.callbacks.BeforeExit");

    m.def("load_user_pref",
        HelloImGui::LoadUserPref,
        nb::arg("user_pref_name"),
        " `string LoadUserPref(string& userPrefName)`\n  Shall be called in the callback runnerParams.callbacks.PostInit");

    m.def("show_view_menu",
        HelloImGui::ShowViewMenu,
        nb::arg("runner_params"),
        " `ShowViewMenu(RunnerParams & runnerParams)`:\n shows the View menu (where you can select the layout and docked windows visibility");

    m.def("show_app_menu",
        HelloImGui::ShowAppMenu,
        nb::arg("runner_params"),
        " `ShowAppMenu(RunnerParams & runnerParams)`:\n shows the default App menu (including the Quit item)");

    { // <namespace ImGuiDefaultSettings>
        nb::module_ pyNsImGuiDefaultSettings = m.def_submodule("imgui_default_settings", "namespace ImGuiDefaultSettings");
        pyNsImGuiDefaultSettings.def("load_default_font_with_font_awesome_icons",
            HelloImGui::ImGuiDefaultSettings::LoadDefaultFont_WithFontAwesomeIcons, "LoadDefaultFont_WithFontAwesome will load from assets/fonts and reverts to the imgui embedded font if not found.");

        pyNsImGuiDefaultSettings.def("setup_default_imgui_config",
            HelloImGui::ImGuiDefaultSettings::SetupDefaultImGuiConfig);

        pyNsImGuiDefaultSettings.def("setup_default_imgui_style",
            HelloImGui::ImGuiDefaultSettings::SetupDefaultImGuiStyle);
    } // </namespace ImGuiDefaultSettings>

    { // <namespace ManualRender>
        nb::module_ pyNsManualRender = m.def_submodule("manual_render", "namespace ManualRender");
        pyNsManualRender.def("setup_from_runner_params",
            HelloImGui::ManualRender::SetupFromRunnerParams,
            nb::arg("runner_params"),
            " Initializes the rendering with the full customizable `RunnerParams`.\n This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).\n A distinct copy of `RunnerParams` is stored internally.");

        pyNsManualRender.def("setup_from_simple_runner_params",
            HelloImGui::ManualRender::SetupFromSimpleRunnerParams,
            nb::arg("simple_params"),
            " Initializes the rendering with `SimpleRunnerParams`.\n This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).");

        pyNsManualRender.def("setup_from_gui_function",
            [](const VoidFunction & guiFunction, const std::string & windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, const std::optional<const ScreenSize> & windowSize = std::nullopt, float fpsIdle = 10.f)
            {
                auto SetupFromGuiFunction_adapt_mutable_param_with_default_value = [](const VoidFunction & guiFunction, const std::string & windowTitle = "", bool windowSizeAuto = false, bool windowRestorePreviousGeometry = false, const std::optional<const ScreenSize> & windowSize = std::nullopt, float fpsIdle = 10.f)
                {

                    const ScreenSize& windowSize_or_default = [&]() -> const ScreenSize {
                        if (windowSize.has_value())
                            return windowSize.value();
                        else
                            return HelloImGui::DefaultWindowSize;
                    }();

                    HelloImGui::ManualRender::SetupFromGuiFunction(guiFunction, windowTitle, windowSizeAuto, windowRestorePreviousGeometry, windowSize_or_default, fpsIdle);
                };

                SetupFromGuiFunction_adapt_mutable_param_with_default_value(guiFunction, windowTitle, windowSizeAuto, windowRestorePreviousGeometry, windowSize, fpsIdle);
            },
            nb::arg("gui_function"), nb::arg("window_title") = "", nb::arg("window_size_auto") = false, nb::arg("window_restore_previous_geometry") = false, nb::arg("window_size").none() = nb::none(), nb::arg("fps_idle") = 10.f,
            " Initializes the renderer with a simple GUI function and additional parameters.\n This will initialize the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).\n\n\nPython bindings defaults:\n    If windowSize is None, then its default value will be: DefaultWindowSize");

        pyNsManualRender.def("render",
            HelloImGui::ManualRender::Render, "Renders the current frame. Should be called regularly to maintain the application's responsiveness.");

        pyNsManualRender.def("tear_down",
            HelloImGui::ManualRender::TearDown, " Tears down the renderer and releases all associated resources.\n This will release the platform backend (SDL, Glfw, etc.) and the rendering backend (OpenGL, Vulkan, etc.).\n After calling `TearDown()`, the InitFromXXX can be called with new parameters.");
    } // </namespace ManualRender>
    ////////////////////    </generated_from:hello_imgui_amalgamation.h>    ////////////////////

    // </litgen_pydef> // Autogenerated code end
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
}
