// Part of ImGui Bundle - MIT License - Copyright (c) 2022-2024 Pascal Thomet - https://github.com/pthom/imgui_bundle
#include <nanobind/nanobind.h>
#include <nanobind/trampoline.h>
#include <nanobind/stl/array.h>
#include <nanobind/stl/string.h>
#include <nanobind/stl/vector.h>
#include <nanobind/stl/optional.h>
#include <nanobind/stl/function.h>
#include <nanobind/stl/shared_ptr.h>
#include <nanobind/stl/unique_ptr.h>
#include <nanobind/stl/map.h>
#include <nanobind/stl/tuple.h>
#include <nanobind/make_iterator.h>
#include <nanobind/ndarray.h>


namespace nb = nanobind;


#include "imgui.h"
#include "imgui_internal.h"
#include "imgui_pywrappers.h"
#include "misc/cpp/imgui_stdlib.h"

using uint = unsigned int;
using uchar = unsigned char;


namespace nanobind { namespace detail {

// Publish ImGuiNpBuffer as a 1D numpy array of bytes
// (only valid from C++ to Python, used to transfer buffers from C++ to Python)
template <>
struct type_caster<ImGuiNpBuffer> {
    NB_TYPE_CASTER(ImGuiNpBuffer, const_name("numpy.array"))

    bool from_python(handle /*src*/, uint8_t /*flags*/, cleanup_list * /*cleanup*/) noexcept {
        // Not supporting Python->C++ conversion for now.
        return false;
    }

    static handle from_cpp(const ImGuiNpBuffer &buf, rv_policy policy, cleanup_list *cleanup) noexcept {
        try {
            // Only allow reference policies since memory is owned by C++
            if (policy != rv_policy::reference
                && policy != rv_policy::reference_internal
                && policy != rv_policy::automatic
                && policy != rv_policy::automatic_reference
                )
            {
                throw std::runtime_error("ImGuiNpBuffer only supports reference policies since memory is owned by C++");
            }

            nb::handle no_owner = {};

            nb::ndarray<uint8_t> array = nb::ndarray<uint8_t>(
                buf.Data,
                {(size_t)buf.Size},
                no_owner,  // no owner, as lifetime is managed on C++
                {(int64_t)1},
                nb::dtype<uint8_t>(),
                0, 0, 'C'
            );

            return nb::detail::ndarray_export(array.handle(), nb::numpy::value, policy, cleanup);
        }
        catch (const std::exception& e) {
            PyErr_WarnFormat(PyExc_Warning, 1, "ImGuiNpBuffer caster from_cpp exception: %s", e.what());
            return {};
        }
    }
};

}} // namespace nanobind::detail

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// <litgen_glue_code>  // Autogenerated code below! Do not edit!

// </litgen_glue_code> // Autogenerated code end
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

nb::class_<ImVec2>* pyClassImVec2Ptr  = nullptr;
nb::class_<ImVec4>* pyClassImVec4Ptr = nullptr;
nb::class_<ImColor>* pyClassImColorPtr = nullptr;
nb::class_<ImFontAtlas>* pyClassImFontAtlasPtr = nullptr;

void imgui_manual_binding(nb::module_& m);



void py_init_module_imgui_main(nb::module_& m)
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // <litgen_pydef> // Autogenerated code below! Do not edit!
    ////////////////////    <generated_from:imgui.h>    ////////////////////
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //
    // #endif
    //
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //
    // #endif
    //
    // #ifndef IMGUI_DISABLE
    //
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //
    // #else
    //
    // #endif
    //


    auto pyClassImVec2 =
        nb::class_<ImVec2>
            (m, "ImVec2", "")
        .def_rw("x", &ImVec2::x, "")
        .def_rw("y", &ImVec2::y, "")
        .def(nb::init<>())
        .def(nb::init<float, float>(),
            nb::arg("_x"), nb::arg("_y"))
        .def("__getitem__",
            nb::overload_cast<size_t>(&ImVec2::operator[]),
            nb::arg("idx"),
            "(private API)\n\n We very rarely use this [] operator, so the assert overhead is fine.",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<size_t>(&ImVec2::operator[], nb::const_),
            nb::arg("idx"),
            "(private API)")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("to_dict",
            &ImVec2::to_dict, "(private API)")
        .def_static("from_dict",
            &ImVec2::from_dict,
            nb::arg("d"),
            "(private API)")
        // #endif
        //
        .def("__copy__",  [](const ImVec2 &self) {
            return ImVec2(self);
        })    ;


    auto pyClassImVec4 =
        nb::class_<ImVec4>
            (m, "ImVec4", "ImVec4: 4D vector used to store clipping rectangles, colors etc. [Compile-time configurable type]")
        .def_rw("x", &ImVec4::x, "")
        .def_rw("y", &ImVec4::y, "")
        .def_rw("z", &ImVec4::z, "")
        .def_rw("w", &ImVec4::w, "")
        .def(nb::init<>())
        .def(nb::init<float, float, float, float>(),
            nb::arg("_x"), nb::arg("_y"), nb::arg("_z"), nb::arg("_w"))
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("to_dict",
            &ImVec4::to_dict, "(private API)")
        .def_static("from_dict",
            &ImVec4::from_dict,
            nb::arg("d"),
            "(private API)")
        // #endif
        //
        .def("__copy__",  [](const ImVec4 &self) {
            return ImVec4(self);
        })    ;


    auto pyClassImTextureRef =
        nb::class_<ImTextureRef>
            (m, "ImTextureRef", "")
        .def(nb::init<>())
        .def(nb::init<ImTextureID>(),
            nb::arg("tex_id"))
        .def("get_tex_id",
            &ImTextureRef::GetTexID, "(private API)\n\n == (_TexData ? _TexData->TexID : _TexID) // Implemented below in the file.")
        .def_rw("_tex_data", &ImTextureRef::_TexData, "A texture, generally owned by a ImFontAtlas. Will convert to ImTextureID during render loop, after texture has been uploaded.")
        .def_rw("_tex_id", &ImTextureRef::_TexID, "_OR_ Low-level backend texture identifier, if already uploaded or created by user/app. Generally provided to e.g. ImGui::Image() calls.")
        ;


    m.def("create_context",
        ImGui::CreateContext,
        nb::arg("shared_font_atlas") = nb::none(),
        nb::rv_policy::reference);

    m.def("destroy_context",
        ImGui::DestroyContext,
        nb::arg("ctx") = nb::none(),
        "None = destroy current context");

    m.def("get_current_context",
        ImGui::GetCurrentContext, nb::rv_policy::reference);

    m.def("set_current_context",
        ImGui::SetCurrentContext, nb::arg("ctx"));

    m.def("get_io",
        nb::overload_cast<>(ImGui::GetIO),
        "access the ImGuiIO structure (mouse/keyboard/gamepad inputs, time, various configuration options/flags)",
        nb::rv_policy::reference);

    m.def("get_platform_io",
        nb::overload_cast<>(ImGui::GetPlatformIO),
        "access the ImGuiPlatformIO structure (mostly hooks/functions to connect to platform/renderer and OS Clipboard, IME etc.)",
        nb::rv_policy::reference);

    m.def("get_style",
        ImGui::GetStyle,
        "access the Style structure (colors, sizes). Always use PushStyleColor(), PushStyleVar() to modify style mid-frame!",
        nb::rv_policy::reference);

    m.def("new_frame",
        ImGui::NewFrame, "start a new Dear ImGui frame, you can submit any command from this point until Render()/EndFrame().");

    m.def("end_frame",
        ImGui::EndFrame, "ends the Dear ImGui frame. automatically called by Render(). If you don't need to render data (skipping rendering) you may call EndFrame() without Render()... but you'll have wasted CPU already! If you don't need to render, better to not create any windows and not call NewFrame() at all!");

    m.def("render",
        ImGui::Render, "ends the Dear ImGui frame, finalize the draw data. You can then get call GetDrawData().");

    m.def("get_draw_data",
        ImGui::GetDrawData,
        "valid after Render() and until the next call to NewFrame(). Call ImGui_ImplXXXX_RenderDrawData() function in your Renderer Backend to render.",
        nb::rv_policy::reference);

    m.def("show_demo_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowDemoWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImGui::ShowDemoWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowDemoWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "create Demo window. demonstrate most ImGui features. call this to learn about the library! try to make it always available in your application!");

    m.def("show_metrics_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowMetricsWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImGui::ShowMetricsWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowMetricsWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "create Metrics/Debugger window. display Dear ImGui internals: windows, draw commands, various internal state, etc.");

    m.def("show_debug_log_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowDebugLogWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImGui::ShowDebugLogWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowDebugLogWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "create Debug Log window. display a simplified log of important dear imgui events.");

    m.def("show_id_stack_tool_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowIDStackToolWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImGui::ShowIDStackToolWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowIDStackToolWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "create Stack Tool window. hover items with mouse to query information about the source of their unique ID.");

    m.def("show_about_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowAboutWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImGui::ShowAboutWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowAboutWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "create About window. display Dear ImGui version, credits and build/system information.");

    m.def("show_style_editor",
        ImGui::ShowStyleEditor,
        nb::arg("ref") = nb::none(),
        "add style editor block (not a window). you can pass in a reference ImGuiStyle structure to compare to, revert to and save to (else it uses the default style)");

    m.def("show_style_selector",
        ImGui::ShowStyleSelector,
        nb::arg("label"),
        "add style selector block (not a window), essentially a combo listing the default styles.");

    m.def("show_font_selector",
        ImGui::ShowFontSelector,
        nb::arg("label"),
        "add font selector block (not a window), essentially a combo listing the loaded fonts.");

    m.def("show_user_guide",
        ImGui::ShowUserGuide, "add basic help/info block (not a window): how to manipulate ImGui as an end-user (mouse/keyboard controls).");

    m.def("get_version",
        ImGui::GetVersion,
        "get the compiled version string e.g. \"1.80 WIP\" (essentially the value for IMGUI_VERSION from the compiled version of imgui.cpp)",
        nb::rv_policy::reference);

    m.def("style_colors_dark",
        ImGui::StyleColorsDark,
        nb::arg("dst") = nb::none(),
        "new, recommended style (default)");

    m.def("style_colors_light",
        ImGui::StyleColorsLight,
        nb::arg("dst") = nb::none(),
        "best used with borders and a custom, thicker font");

    m.def("style_colors_classic",
        ImGui::StyleColorsClassic,
        nb::arg("dst") = nb::none(),
        "classic imgui style");

    m.def("begin",
        [](const char * name, std::optional<bool> p_open = std::nullopt, ImGuiWindowFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
        {
            auto Begin_adapt_modifiable_immutable_to_return = [](const char * name, std::optional<bool> p_open = std::nullopt, ImGuiWindowFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                bool r = ImGui::Begin(name, p_open_adapt_modifiable, flags);
                return std::make_tuple(r, p_open);
            };

            return Begin_adapt_modifiable_immutable_to_return(name, p_open, flags);
        },     nb::arg("name"), nb::arg("p_open").none() = nb::none(), nb::arg("flags") = 0);

    m.def("end",
        ImGui::End);

    m.def("begin_child",
        [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiChildFlags child_flags = 0, ImGuiWindowFlags window_flags = 0) -> bool
        {
            auto BeginChild_adapt_mutable_param_with_default_value = [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiChildFlags child_flags = 0, ImGuiWindowFlags window_flags = 0) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::BeginChild(str_id, size_or_default, child_flags, window_flags);
                return lambda_result;
            };

            return BeginChild_adapt_mutable_param_with_default_value(str_id, size, child_flags, window_flags);
        },
        nb::arg("str_id"), nb::arg("size").none() = nb::none(), nb::arg("child_flags") = 0, nb::arg("window_flags") = 0,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("begin_child",
        [](ImGuiID id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiChildFlags child_flags = 0, ImGuiWindowFlags window_flags = 0) -> bool
        {
            auto BeginChild_adapt_mutable_param_with_default_value = [](ImGuiID id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiChildFlags child_flags = 0, ImGuiWindowFlags window_flags = 0) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::BeginChild(id, size_or_default, child_flags, window_flags);
                return lambda_result;
            };

            return BeginChild_adapt_mutable_param_with_default_value(id, size, child_flags, window_flags);
        },
        nb::arg("id_"), nb::arg("size").none() = nb::none(), nb::arg("child_flags") = 0, nb::arg("window_flags") = 0,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("end_child",
        ImGui::EndChild);

    m.def("is_window_appearing",
        ImGui::IsWindowAppearing);

    m.def("is_window_collapsed",
        ImGui::IsWindowCollapsed);

    m.def("is_window_focused",
        ImGui::IsWindowFocused,
        nb::arg("flags") = 0,
        "is current window focused? or its root/child, depending on flags. see flags for options.");

    m.def("is_window_hovered",
        ImGui::IsWindowHovered,
        nb::arg("flags") = 0,
        "is current window hovered and hoverable (e.g. not blocked by a popup/modal)? See ImGuiHoveredFlags_ for options. IMPORTANT: If you are trying to check whether your mouse should be dispatched to Dear ImGui or to your underlying app, you should not use this function! Use the 'io.WantCaptureMouse' boolean for that! Refer to FAQ entry \"How can I tell whether to dispatch mouse/keyboard to Dear ImGui or my application?\" for details.");

    m.def("get_window_draw_list",
        nb::overload_cast<>(ImGui::GetWindowDrawList),
        "get draw list associated to the current window, to append your own drawing primitives",
        nb::rv_policy::reference);

    m.def("get_window_dpi_scale",
        ImGui::GetWindowDpiScale, "get DPI scale currently associated to the current window's viewport.");

    m.def("get_window_pos",
        ImGui::GetWindowPos, "get current window position in screen space (IT IS UNLIKELY YOU EVER NEED TO USE THIS. Consider always using GetCursorScreenPos() and GetContentRegionAvail() instead)");

    m.def("get_window_size",
        ImGui::GetWindowSize, "get current window size (IT IS UNLIKELY YOU EVER NEED TO USE THIS. Consider always using GetCursorScreenPos() and GetContentRegionAvail() instead)");

    m.def("get_window_width",
        ImGui::GetWindowWidth, "get current window width (IT IS UNLIKELY YOU EVER NEED TO USE THIS). Shortcut for GetWindowSize().x.");

    m.def("get_window_height",
        ImGui::GetWindowHeight, "get current window height (IT IS UNLIKELY YOU EVER NEED TO USE THIS). Shortcut for GetWindowSize().y.");

    m.def("get_window_viewport",
        ImGui::GetWindowViewport,
        "get viewport currently associated to the current window.",
        nb::rv_policy::reference);

    m.def("set_next_window_pos",
        [](const ImVec2 & pos, ImGuiCond cond = 0, const std::optional<const ImVec2> & pivot = std::nullopt)
        {
            auto SetNextWindowPos_adapt_mutable_param_with_default_value = [](const ImVec2 & pos, ImGuiCond cond = 0, const std::optional<const ImVec2> & pivot = std::nullopt)
            {

                const ImVec2& pivot_or_default = [&]() -> const ImVec2 {
                    if (pivot.has_value())
                        return pivot.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::SetNextWindowPos(pos, cond, pivot_or_default);
            };

            SetNextWindowPos_adapt_mutable_param_with_default_value(pos, cond, pivot);
        },
        nb::arg("pos"), nb::arg("cond") = 0, nb::arg("pivot").none() = nb::none(),
        "Python bindings defaults:\n    If pivot is None, then its default value will be: ImVec2(0, 0)\n\n set next window position. call before Begin(). use pivot=(0.5,0.5) to center on given point, etc.");

    m.def("set_next_window_size",
        ImGui::SetNextWindowSize,
        nb::arg("size"), nb::arg("cond") = 0,
        "set next window size. set axis to 0.0 to force an auto-fit on this axis. call before Begin()");

    m.def("set_next_window_size_constraints",
        ImGui::SetNextWindowSizeConstraints,
        nb::arg("size_min"), nb::arg("size_max"), nb::arg("custom_callback") = nb::none(), nb::arg("custom_callback_data") = nb::none(),
        "set next window size limits. use 0.0 or FLT_MAX if you don't want limits. Use -1 for both min and max of same axis to preserve current size (which itself is a constraint). Use callback to apply non-trivial programmatic constraints.");

    m.def("set_next_window_content_size",
        ImGui::SetNextWindowContentSize,
        nb::arg("size"),
        "set next window content size (~ scrollable client area, which enforce the range of scrollbars). Not including window decorations (title bar, menu bar, etc.) nor WindowPadding. set an axis to 0.0 to leave it automatic. call before Begin()");

    m.def("set_next_window_collapsed",
        ImGui::SetNextWindowCollapsed,
        nb::arg("collapsed"), nb::arg("cond") = 0,
        "set next window collapsed state. call before Begin()");

    m.def("set_next_window_focus",
        ImGui::SetNextWindowFocus, "set next window to be focused / top-most. call before Begin()");

    m.def("set_next_window_scroll",
        ImGui::SetNextWindowScroll,
        nb::arg("scroll"),
        "set next window scrolling value (use < 0.0 to not affect a given axis).");

    m.def("set_next_window_bg_alpha",
        ImGui::SetNextWindowBgAlpha,
        nb::arg("alpha"),
        "set next window background color alpha. helper to easily override the Alpha component of ImGuiCol_WindowBg/ChildBg/PopupBg. you may also use ImGuiWindowFlags_NoBackground.");

    m.def("set_next_window_viewport",
        ImGui::SetNextWindowViewport,
        nb::arg("viewport_id"),
        "set next window viewport");

    m.def("set_window_pos",
        nb::overload_cast<const ImVec2 &, ImGuiCond>(ImGui::SetWindowPos),
        nb::arg("pos"), nb::arg("cond") = 0,
        "(not recommended) set current window position - call within Begin()/End(). prefer using SetNextWindowPos(), as this may incur tearing and side-effects.");

    m.def("set_window_size",
        nb::overload_cast<const ImVec2 &, ImGuiCond>(ImGui::SetWindowSize),
        nb::arg("size"), nb::arg("cond") = 0,
        "(not recommended) set current window size - call within Begin()/End(). set to ImVec2(0, 0) to force an auto-fit. prefer using SetNextWindowSize(), as this may incur tearing and minor side-effects.");

    m.def("set_window_collapsed",
        nb::overload_cast<bool, ImGuiCond>(ImGui::SetWindowCollapsed),
        nb::arg("collapsed"), nb::arg("cond") = 0,
        "(not recommended) set current window collapsed state. prefer using SetNextWindowCollapsed().");

    m.def("set_window_focus",
        nb::overload_cast<>(ImGui::SetWindowFocus), "(not recommended) set current window to be focused / top-most. prefer using SetNextWindowFocus().");

    m.def("set_window_pos",
        nb::overload_cast<const char *, const ImVec2 &, ImGuiCond>(ImGui::SetWindowPos),
        nb::arg("name"), nb::arg("pos"), nb::arg("cond") = 0,
        "set named window position.");

    m.def("set_window_size",
        nb::overload_cast<const char *, const ImVec2 &, ImGuiCond>(ImGui::SetWindowSize),
        nb::arg("name"), nb::arg("size"), nb::arg("cond") = 0,
        "set named window size. set axis to 0.0 to force an auto-fit on this axis.");

    m.def("set_window_collapsed",
        nb::overload_cast<const char *, bool, ImGuiCond>(ImGui::SetWindowCollapsed),
        nb::arg("name"), nb::arg("collapsed"), nb::arg("cond") = 0,
        "set named window collapsed state");

    m.def("set_window_focus",
        nb::overload_cast<const char *>(ImGui::SetWindowFocus),
        nb::arg("name"),
        "set named window to be focused / top-most. use None to remove focus.");

    m.def("get_scroll_x",
        ImGui::GetScrollX, "get scrolling amount [0 .. GetScrollMaxX()]");

    m.def("get_scroll_y",
        ImGui::GetScrollY, "get scrolling amount [0 .. GetScrollMaxY()]");

    m.def("set_scroll_x",
        nb::overload_cast<float>(ImGui::SetScrollX),
        nb::arg("scroll_x"),
        "set scrolling amount [0 .. GetScrollMaxX()]");

    m.def("set_scroll_y",
        nb::overload_cast<float>(ImGui::SetScrollY),
        nb::arg("scroll_y"),
        "set scrolling amount [0 .. GetScrollMaxY()]");

    m.def("get_scroll_max_x",
        ImGui::GetScrollMaxX, "get maximum scrolling amount ~~ ContentSize.x - WindowSize.x - DecorationsSize.x");

    m.def("get_scroll_max_y",
        ImGui::GetScrollMaxY, "get maximum scrolling amount ~~ ContentSize.y - WindowSize.y - DecorationsSize.y");

    m.def("set_scroll_here_x",
        nb::overload_cast<float>(ImGui::SetScrollHereX),
        nb::arg("center_x_ratio") = 0.5f,
        "adjust scrolling amount to make current cursor position visible. center_x_ratio=0.0: left, 0.5: center, 1.0: right. When using to make a \"default/current item\" visible, consider using SetItemDefaultFocus() instead.");

    m.def("set_scroll_here_y",
        nb::overload_cast<float>(ImGui::SetScrollHereY),
        nb::arg("center_y_ratio") = 0.5f,
        "adjust scrolling amount to make current cursor position visible. center_y_ratio=0.0: top, 0.5: center, 1.0: bottom. When using to make a \"default/current item\" visible, consider using SetItemDefaultFocus() instead.");

    m.def("set_scroll_from_pos_x",
        nb::overload_cast<float, float>(ImGui::SetScrollFromPosX),
        nb::arg("local_x"), nb::arg("center_x_ratio") = 0.5f,
        "adjust scrolling amount to make given position visible. Generally GetCursorStartPos() + offset to compute a valid position.");

    m.def("set_scroll_from_pos_y",
        nb::overload_cast<float, float>(ImGui::SetScrollFromPosY),
        nb::arg("local_y"), nb::arg("center_y_ratio") = 0.5f,
        "adjust scrolling amount to make given position visible. Generally GetCursorStartPos() + offset to compute a valid position.");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("push_font",
        nb::overload_cast<std::optional<ImFont*>, float>(ImGui::PushFont),
        nb::arg("font").none(), nb::arg("font_size_base_unscaled"),
        "Use None as a shortcut to keep current font. Use 0.0 to keep current size.");
    // #endif
    //

    m.def("pop_font",
        ImGui::PopFont);

    m.def("get_font",
        ImGui::GetFont,
        "get current font",
        nb::rv_policy::reference);

    m.def("get_font_size",
        ImGui::GetFontSize, "get current scaled font size (= height in pixels). AFTER global scale factors applied. *IMPORTANT* DO NOT PASS THIS VALUE TO PushFont()! Use ImGui::GetStyle().FontSizeBase to get value before global scale factors.");

    m.def("get_font_baked",
        ImGui::GetFontBaked,
        "get current font bound at current size // == GetFont()->GetFontBaked(GetFontSize())",
        nb::rv_policy::reference);

    m.def("push_style_color",
        nb::overload_cast<ImGuiCol, ImU32>(ImGui::PushStyleColor),
        nb::arg("idx"), nb::arg("col"),
        "modify a style color. always use this if you modify the style after NewFrame().");

    m.def("push_style_color",
        nb::overload_cast<ImGuiCol, const ImVec4 &>(ImGui::PushStyleColor), nb::arg("idx"), nb::arg("col"));

    m.def("pop_style_color",
        ImGui::PopStyleColor, nb::arg("count") = 1);

    m.def("push_style_var",
        nb::overload_cast<ImGuiStyleVar, float>(ImGui::PushStyleVar),
        nb::arg("idx"), nb::arg("val"),
        "modify a style float variable. always use this if you modify the style after NewFrame()!");

    m.def("push_style_var",
        nb::overload_cast<ImGuiStyleVar, const ImVec2 &>(ImGui::PushStyleVar),
        nb::arg("idx"), nb::arg("val"),
        "modify a style ImVec2 variable. \"");

    m.def("push_style_var_x",
        ImGui::PushStyleVarX,
        nb::arg("idx"), nb::arg("val_x"),
        "modify X component of a style ImVec2 variable. \"");

    m.def("push_style_var_y",
        ImGui::PushStyleVarY,
        nb::arg("idx"), nb::arg("val_y"),
        "modify Y component of a style ImVec2 variable. \"");

    m.def("pop_style_var",
        ImGui::PopStyleVar, nb::arg("count") = 1);

    m.def("push_item_flag",
        ImGui::PushItemFlag,
        nb::arg("option"), nb::arg("enabled"),
        "modify specified shared item flag, e.g. PushItemFlag(ImGuiItemFlags_NoTabStop, True)");

    m.def("pop_item_flag",
        ImGui::PopItemFlag);

    m.def("push_item_width",
        ImGui::PushItemWidth,
        nb::arg("item_width"),
        "push width of items for common large \"item+label\" widgets. >0.0: width in pixels, <0.0 align xx pixels to the right of window (so -FLT_MIN always align width to the right side).");

    m.def("pop_item_width",
        ImGui::PopItemWidth);

    m.def("set_next_item_width",
        ImGui::SetNextItemWidth,
        nb::arg("item_width"),
        "set width of the _next_ common large \"item+label\" widget. >0.0: width in pixels, <0.0 align xx pixels to the right of window (so -FLT_MIN always align width to the right side)");

    m.def("calc_item_width",
        ImGui::CalcItemWidth, "width of item given pushed settings and current cursor position. NOT necessarily the width of last item unlike most 'Item' functions.");

    m.def("push_text_wrap_pos",
        ImGui::PushTextWrapPos,
        nb::arg("wrap_local_pos_x") = 0.0f,
        "push word-wrapping position for Text*() commands. < 0.0: no wrapping; 0.0: wrap to end of window (or column); > 0.0: wrap at 'wrap_pos_x' position in window local space");

    m.def("pop_text_wrap_pos",
        ImGui::PopTextWrapPos);

    m.def("get_font_tex_uv_white_pixel",
        ImGui::GetFontTexUvWhitePixel, "get UV coordinate for a white pixel, useful to draw custom shapes via the ImDrawList API");

    m.def("get_color_u32",
        nb::overload_cast<ImGuiCol, float>(ImGui::GetColorU32),
        nb::arg("idx"), nb::arg("alpha_mul") = 1.0f,
        "retrieve given style color with style alpha applied and optional extra alpha multiplier, packed as a 32-bit value suitable for ImDrawList");

    m.def("get_color_u32",
        nb::overload_cast<const ImVec4 &>(ImGui::GetColorU32),
        nb::arg("col"),
        "retrieve given color with style alpha applied, packed as a 32-bit value suitable for ImDrawList");

    m.def("get_color_u32",
        nb::overload_cast<ImU32, float>(ImGui::GetColorU32),
        nb::arg("col"), nb::arg("alpha_mul") = 1.0f,
        "retrieve given color with style alpha applied, packed as a 32-bit value suitable for ImDrawList");

    m.def("get_style_color_vec4",
        ImGui::GetStyleColorVec4,
        nb::arg("idx"),
        "retrieve style color as stored in ImGuiStyle structure. use to feed back into PushStyleColor(), otherwise use GetColorU32() to get style color with style alpha baked in.",
        nb::rv_policy::reference);

    m.def("get_cursor_screen_pos",
        ImGui::GetCursorScreenPos, "cursor position, absolute coordinates. THIS IS YOUR BEST FRIEND (prefer using this rather than GetCursorPos(), also more useful to work with ImDrawList API).");

    m.def("set_cursor_screen_pos",
        ImGui::SetCursorScreenPos,
        nb::arg("pos"),
        "cursor position, absolute coordinates. THIS IS YOUR BEST FRIEND.");

    m.def("get_content_region_avail",
        ImGui::GetContentRegionAvail, "available space from current position. THIS IS YOUR BEST FRIEND.");

    m.def("get_cursor_pos",
        ImGui::GetCursorPos, "[window-local] cursor position in window-local coordinates. This is not your best friend.");

    m.def("get_cursor_pos_x",
        ImGui::GetCursorPosX, "[window-local] \"");

    m.def("get_cursor_pos_y",
        ImGui::GetCursorPosY, "[window-local] \"");

    m.def("set_cursor_pos",
        ImGui::SetCursorPos,
        nb::arg("local_pos"),
        "[window-local] \"");

    m.def("set_cursor_pos_x",
        ImGui::SetCursorPosX,
        nb::arg("local_x"),
        "[window-local] \"");

    m.def("set_cursor_pos_y",
        ImGui::SetCursorPosY,
        nb::arg("local_y"),
        "[window-local] \"");

    m.def("get_cursor_start_pos",
        ImGui::GetCursorStartPos, "[window-local] initial cursor position, in window-local coordinates. Call GetCursorScreenPos() after Begin() to get the absolute coordinates version.");

    m.def("separator",
        ImGui::Separator, "separator, generally horizontal. inside a menu bar or in horizontal layout mode, this becomes a vertical separator.");

    m.def("same_line",
        ImGui::SameLine,
        nb::arg("offset_from_start_x") = 0.0f, nb::arg("spacing") = -1.0f,
        "call between widgets or groups to layout them horizontally. X position given in window coordinates.");

    m.def("new_line",
        ImGui::NewLine, "undo a SameLine() or force a new line when in a horizontal-layout context.");

    m.def("spacing",
        ImGui::Spacing, "add vertical spacing.");

    m.def("dummy",
        ImGui::Dummy,
        nb::arg("size"),
        "add a dummy item of given size. unlike InvisibleButton(), Dummy() won't take the mouse click or be navigable into.");

    m.def("indent",
        ImGui::Indent,
        nb::arg("indent_w") = 0.0f,
        "move content position toward the right, by indent_w, or style.IndentSpacing if indent_w <= 0");

    m.def("unindent",
        ImGui::Unindent,
        nb::arg("indent_w") = 0.0f,
        "move content position back to the left, by indent_w, or style.IndentSpacing if indent_w <= 0");

    m.def("begin_group",
        ImGui::BeginGroup, "lock horizontal starting position");

    m.def("end_group",
        ImGui::EndGroup, "unlock horizontal starting position + capture the whole group bounding box into one \"item\" (so you can use IsItemHovered() or layout primitives such as SameLine() on whole group, etc.)");

    m.def("align_text_to_frame_padding",
        ImGui::AlignTextToFramePadding, "vertically align upcoming text baseline to FramePadding.y so that it will align properly to regularly framed items (call if you have text on a line before a framed item)");

    m.def("get_text_line_height",
        ImGui::GetTextLineHeight, "~ FontSize");

    m.def("get_text_line_height_with_spacing",
        ImGui::GetTextLineHeightWithSpacing, "~ FontSize + style.ItemSpacing.y (distance in pixels between 2 consecutive lines of text)");

    m.def("get_frame_height",
        ImGui::GetFrameHeight, "~ FontSize + style.FramePadding.y * 2");

    m.def("get_frame_height_with_spacing",
        ImGui::GetFrameHeightWithSpacing, "~ FontSize + style.FramePadding.y * 2 + style.ItemSpacing.y (distance in pixels between 2 consecutive lines of framed widgets)");

    m.def("push_id",
        nb::overload_cast<const char *>(ImGui::PushID),
        nb::arg("str_id"),
        "push string into the ID stack (will hash string).");

    m.def("push_id",
        nb::overload_cast<const char *, const char *>(ImGui::PushID),
        nb::arg("str_id_begin"), nb::arg("str_id_end"),
        "push string into the ID stack (will hash string).");

    m.def("push_id",
        nb::overload_cast<const void *>(ImGui::PushID),
        nb::arg("ptr_id"),
        "push pointer into the ID stack (will hash pointer).");

    m.def("push_id",
        nb::overload_cast<int>(ImGui::PushID),
        nb::arg("int_id"),
        "push integer into the ID stack (will hash integer).");

    m.def("pop_id",
        ImGui::PopID, "pop from the ID stack.");

    m.def("get_id",
        nb::overload_cast<const char *>(ImGui::GetID),
        nb::arg("str_id"),
        "calculate unique ID (hash of whole ID stack + given parameter). e.g. if you want to query into ImGuiStorage yourself");

    m.def("get_id",
        nb::overload_cast<const char *, const char *>(ImGui::GetID), nb::arg("str_id_begin"), nb::arg("str_id_end"));

    m.def("get_id",
        nb::overload_cast<const void *>(ImGui::GetID), nb::arg("ptr_id"));

    m.def("get_id",
        nb::overload_cast<int>(ImGui::GetID), nb::arg("int_id"));

    m.def("text_unformatted",
        [](const char * text, std::optional<std::string> text_end = std::nullopt)
        {
            auto TextUnformatted_adapt_const_char_pointer_with_default_null = [](const char * text, std::optional<std::string> text_end = std::nullopt)
            {
                const char * text_end_adapt_default_null = nullptr;
                if (text_end.has_value())
                    text_end_adapt_default_null = text_end.value().c_str();

                ImGui::TextUnformatted(text, text_end_adapt_default_null);
            };

            TextUnformatted_adapt_const_char_pointer_with_default_null(text, text_end);
        },
        nb::arg("text"), nb::arg("text_end").none() = nb::none(),
        "raw text without formatting. Roughly equivalent to Text(\"%s\", text) but: A) doesn't require null terminated string if 'text_end' is specified, B) it's faster, no memory copy is done, no buffer size limits, recommended for long chunks of text.");

    m.def("text",
        [](const char * fmt)
        {
            auto Text_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::Text("%s", fmt);
            };

            Text_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "formatted text");

    m.def("text_colored",
        [](const ImVec4 & col, const char * fmt)
        {
            auto TextColored_adapt_variadic_format = [](const ImVec4 & col, const char * fmt)
            {
                ImGui::TextColored(col, "%s", fmt);
            };

            TextColored_adapt_variadic_format(col, fmt);
        },
        nb::arg("col"), nb::arg("fmt"),
        "shortcut for PushStyleColor(ImGuiCol_Text, col); Text(fmt, ...); PopStyleColor();");

    m.def("text_disabled",
        [](const char * fmt)
        {
            auto TextDisabled_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::TextDisabled("%s", fmt);
            };

            TextDisabled_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "shortcut for PushStyleColor(ImGuiCol_Text, style.Colors[ImGuiCol_TextDisabled]); Text(fmt, ...); PopStyleColor();");

    m.def("text_wrapped",
        [](const char * fmt)
        {
            auto TextWrapped_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::TextWrapped("%s", fmt);
            };

            TextWrapped_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "shortcut for PushTextWrapPos(0.0); Text(fmt, ...); PopTextWrapPos();. Note that this won't work on an auto-resizing window if there's no other widgets to extend the window width, yoy may need to set a size using SetNextWindowSize().");

    m.def("label_text",
        [](const char * label, const char * fmt)
        {
            auto LabelText_adapt_variadic_format = [](const char * label, const char * fmt)
            {
                ImGui::LabelText(label, "%s", fmt);
            };

            LabelText_adapt_variadic_format(label, fmt);
        },
        nb::arg("label"), nb::arg("fmt"),
        "display text+label aligned the same way as value+label widgets");

    m.def("bullet_text",
        [](const char * fmt)
        {
            auto BulletText_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::BulletText("%s", fmt);
            };

            BulletText_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "shortcut for Bullet()+Text()");

    m.def("separator_text",
        ImGui::SeparatorText,
        nb::arg("label"),
        "currently: formatted text with a horizontal line");

    m.def("button",
        [](const char * label, const std::optional<const ImVec2> & size = std::nullopt) -> bool
        {
            auto Button_adapt_mutable_param_with_default_value = [](const char * label, const std::optional<const ImVec2> & size = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::Button(label, size_or_default);
                return lambda_result;
            };

            return Button_adapt_mutable_param_with_default_value(label, size);
        },
        nb::arg("label"), nb::arg("size").none() = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)\n\n button");

    m.def("small_button",
        ImGui::SmallButton,
        nb::arg("label"),
        "button with (FramePadding.y == 0) to easily embed within text");

    m.def("invisible_button",
        ImGui::InvisibleButton,
        nb::arg("str_id"), nb::arg("size"), nb::arg("flags") = 0,
        "flexible button behavior without the visuals, frequently useful to build custom behaviors using the public api (along with IsItemActive, IsItemHovered, etc.)");

    m.def("arrow_button",
        ImGui::ArrowButton,
        nb::arg("str_id"), nb::arg("dir"),
        "square button with an arrow shape");

    m.def("checkbox",
        [](const char * label, bool v) -> std::tuple<bool, bool>
        {
            auto Checkbox_adapt_modifiable_immutable_to_return = [](const char * label, bool v) -> std::tuple<bool, bool>
            {
                bool * v_adapt_modifiable = & v;

                bool r = ImGui::Checkbox(label, v_adapt_modifiable);
                return std::make_tuple(r, v);
            };

            return Checkbox_adapt_modifiable_immutable_to_return(label, v);
        },     nb::arg("label"), nb::arg("v"));

    m.def("checkbox_flags",
        [](const char * label, int flags, int flags_value) -> std::tuple<bool, int>
        {
            auto CheckboxFlags_adapt_modifiable_immutable_to_return = [](const char * label, int flags, int flags_value) -> std::tuple<bool, int>
            {
                int * flags_adapt_modifiable = & flags;

                bool r = ImGui::CheckboxFlags(label, flags_adapt_modifiable, flags_value);
                return std::make_tuple(r, flags);
            };

            return CheckboxFlags_adapt_modifiable_immutable_to_return(label, flags, flags_value);
        },     nb::arg("label"), nb::arg("flags"), nb::arg("flags_value"));

    m.def("checkbox_flags",
        [](const char * label, unsigned int flags, unsigned int flags_value) -> std::tuple<bool, unsigned int>
        {
            auto CheckboxFlags_adapt_modifiable_immutable_to_return = [](const char * label, unsigned int flags, unsigned int flags_value) -> std::tuple<bool, unsigned int>
            {
                unsigned int * flags_adapt_modifiable = & flags;

                bool r = ImGui::CheckboxFlags(label, flags_adapt_modifiable, flags_value);
                return std::make_tuple(r, flags);
            };

            return CheckboxFlags_adapt_modifiable_immutable_to_return(label, flags, flags_value);
        },     nb::arg("label"), nb::arg("flags"), nb::arg("flags_value"));

    m.def("radio_button",
        nb::overload_cast<const char *, bool>(ImGui::RadioButton),
        nb::arg("label"), nb::arg("active"),
        "use with e.g. if (RadioButton(\"one\", my_value==1)) { my_value = 1; }");

    m.def("radio_button",
        [](const char * label, int v, int v_button) -> std::tuple<bool, int>
        {
            auto RadioButton_adapt_modifiable_immutable_to_return = [](const char * label, int v, int v_button) -> std::tuple<bool, int>
            {
                int * v_adapt_modifiable = & v;

                bool r = ImGui::RadioButton(label, v_adapt_modifiable, v_button);
                return std::make_tuple(r, v);
            };

            return RadioButton_adapt_modifiable_immutable_to_return(label, v, v_button);
        },
        nb::arg("label"), nb::arg("v"), nb::arg("v_button"),
        "shortcut to handle the above pattern when value is an integer");

    m.def("progress_bar",
        [](float fraction, const std::optional<const ImVec2> & size_arg = std::nullopt, std::optional<std::string> overlay = std::nullopt)
        {
            auto ProgressBar_adapt_mutable_param_with_default_value = [](float fraction, const std::optional<const ImVec2> & size_arg = std::nullopt, const char * overlay = NULL)
            {

                const ImVec2& size_arg_or_default = [&]() -> const ImVec2 {
                    if (size_arg.has_value())
                        return size_arg.value();
                    else
                        return ImVec2(-FLT_MIN, 0);
                }();

                ImGui::ProgressBar(fraction, size_arg_or_default, overlay);
            };
            auto ProgressBar_adapt_const_char_pointer_with_default_null = [&ProgressBar_adapt_mutable_param_with_default_value](float fraction, const std::optional<const ImVec2> & size_arg = std::nullopt, std::optional<std::string> overlay = std::nullopt)
            {
                const char * overlay_adapt_default_null = nullptr;
                if (overlay.has_value())
                    overlay_adapt_default_null = overlay.value().c_str();

                ProgressBar_adapt_mutable_param_with_default_value(fraction, size_arg, overlay_adapt_default_null);
            };

            ProgressBar_adapt_const_char_pointer_with_default_null(fraction, size_arg, overlay);
        },
        nb::arg("fraction"), nb::arg("size_arg").none() = nb::none(), nb::arg("overlay").none() = nb::none(),
        "Python bindings defaults:\n    If size_arg is None, then its default value will be: ImVec2(-sys.float_info.min, 0)");

    m.def("bullet",
        ImGui::Bullet, "draw a small circle + keep the cursor on the same line. advance cursor x position by GetTreeNodeToLabelSpacing(), same distance that TreeNode() uses");

    m.def("text_link",
        ImGui::TextLink,
        nb::arg("label"),
        "hyperlink text button, return True when clicked");

    m.def("text_link_open_url",
        [](const char * label, std::optional<std::string> url = std::nullopt) -> bool
        {
            auto TextLinkOpenURL_adapt_const_char_pointer_with_default_null = [](const char * label, std::optional<std::string> url = std::nullopt) -> bool
            {
                const char * url_adapt_default_null = nullptr;
                if (url.has_value())
                    url_adapt_default_null = url.value().c_str();

                auto lambda_result = ImGui::TextLinkOpenURL(label, url_adapt_default_null);
                return lambda_result;
            };

            return TextLinkOpenURL_adapt_const_char_pointer_with_default_null(label, url);
        },
        nb::arg("label"), nb::arg("url").none() = nb::none(),
        "hyperlink text button, automatically open file/url when clicked");

    m.def("image",
        [](ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt)
        {
            auto Image_adapt_mutable_param_with_default_value = [](ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt)
            {

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1, 1);
                }();

                ImGui::Image(tex_ref, image_size, uv0_or_default, uv1_or_default);
            };

            Image_adapt_mutable_param_with_default_value(tex_ref, image_size, uv0, uv1);
        },
        nb::arg("tex_ref"), nb::arg("image_size"), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1, 1)");

    m.def("image_with_bg",
        [](ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt)
        {
            auto ImageWithBg_adapt_mutable_param_with_default_value = [](ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt)
            {

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1, 1);
                }();

                const ImVec4& bg_col_or_default = [&]() -> const ImVec4 {
                    if (bg_col.has_value())
                        return bg_col.value();
                    else
                        return ImVec4(0, 0, 0, 0);
                }();

                const ImVec4& tint_col_or_default = [&]() -> const ImVec4 {
                    if (tint_col.has_value())
                        return tint_col.value();
                    else
                        return ImVec4(1, 1, 1, 1);
                }();

                ImGui::ImageWithBg(tex_ref, image_size, uv0_or_default, uv1_or_default, bg_col_or_default, tint_col_or_default);
            };

            ImageWithBg_adapt_mutable_param_with_default_value(tex_ref, image_size, uv0, uv1, bg_col, tint_col);
        },
        nb::arg("tex_ref"), nb::arg("image_size"), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(), nb::arg("bg_col").none() = nb::none(), nb::arg("tint_col").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1, 1)\n        * bg_col: ImVec4(0, 0, 0, 0)\n        * tint_col: ImVec4(1, 1, 1, 1)");

    m.def("image_button",
        [](const char * str_id, ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt) -> bool
        {
            auto ImageButton_adapt_mutable_param_with_default_value = [](const char * str_id, ImTextureRef tex_ref, const ImVec2 & image_size, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & bg_col = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt) -> bool
            {

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1, 1);
                }();

                const ImVec4& bg_col_or_default = [&]() -> const ImVec4 {
                    if (bg_col.has_value())
                        return bg_col.value();
                    else
                        return ImVec4(0, 0, 0, 0);
                }();

                const ImVec4& tint_col_or_default = [&]() -> const ImVec4 {
                    if (tint_col.has_value())
                        return tint_col.value();
                    else
                        return ImVec4(1, 1, 1, 1);
                }();

                auto lambda_result = ImGui::ImageButton(str_id, tex_ref, image_size, uv0_or_default, uv1_or_default, bg_col_or_default, tint_col_or_default);
                return lambda_result;
            };

            return ImageButton_adapt_mutable_param_with_default_value(str_id, tex_ref, image_size, uv0, uv1, bg_col, tint_col);
        },
        nb::arg("str_id"), nb::arg("tex_ref"), nb::arg("image_size"), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(), nb::arg("bg_col").none() = nb::none(), nb::arg("tint_col").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1, 1)\n        * bg_col: ImVec4(0, 0, 0, 0)\n        * tint_col: ImVec4(1, 1, 1, 1)");

    m.def("begin_combo",
        ImGui::BeginCombo, nb::arg("label"), nb::arg("preview_value"), nb::arg("flags") = 0);

    m.def("end_combo",
        ImGui::EndCombo, "only call EndCombo() if BeginCombo() returns True!");

    m.def("combo",
        [](const char * label, int current_item, const std::vector<std::string> & items, int popup_max_height_in_items = -1) -> std::tuple<bool, int>
        {
            auto Combo_adapt_modifiable_immutable_to_return = [](const char * label, int current_item, const char * const items[], int items_count, int popup_max_height_in_items = -1) -> std::tuple<bool, int>
            {
                int * current_item_adapt_modifiable = & current_item;

                bool r = ImGui::Combo(label, current_item_adapt_modifiable, items, items_count, popup_max_height_in_items);
                return std::make_tuple(r, current_item);
            };
            auto Combo_adapt_c_string_list = [&Combo_adapt_modifiable_immutable_to_return](const char * label, int current_item, const std::vector<std::string> & items, int popup_max_height_in_items = -1) -> std::tuple<bool, int>
            {
                std::vector<const char *> items_ptrs;
                items_ptrs.reserve(items.size());
                for (const auto& v: items)
                    items_ptrs.push_back(v.c_str());
                int items_count = static_cast<int>(items.size());

                auto lambda_result = Combo_adapt_modifiable_immutable_to_return(label, current_item, items_ptrs.data(), items_count, popup_max_height_in_items);
                return lambda_result;
            };

            return Combo_adapt_c_string_list(label, current_item, items, popup_max_height_in_items);
        },     nb::arg("label"), nb::arg("current_item"), nb::arg("items"), nb::arg("popup_max_height_in_items") = -1);

    m.def("combo",
        [](const char * label, int current_item, const char * items_separated_by_zeros, int popup_max_height_in_items = -1) -> std::tuple<bool, int>
        {
            auto Combo_adapt_modifiable_immutable_to_return = [](const char * label, int current_item, const char * items_separated_by_zeros, int popup_max_height_in_items = -1) -> std::tuple<bool, int>
            {
                int * current_item_adapt_modifiable = & current_item;

                bool r = ImGui::Combo(label, current_item_adapt_modifiable, items_separated_by_zeros, popup_max_height_in_items);
                return std::make_tuple(r, current_item);
            };

            return Combo_adapt_modifiable_immutable_to_return(label, current_item, items_separated_by_zeros, popup_max_height_in_items);
        },
        nb::arg("label"), nb::arg("current_item"), nb::arg("items_separated_by_zeros"), nb::arg("popup_max_height_in_items") = -1,
        "Separate items with \\0 within a string, end item-list with \\0\\0. e.g. \"One\\0Two\\0Three\\0\"");

    m.def("drag_float",
        [](const char * label, float v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
        {
            auto DragFloat_adapt_modifiable_immutable_to_return = [](const char * label, float v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
            {
                float * v_adapt_modifiable = & v;

                bool r = ImGui::DragFloat(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragFloat_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },
        nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0.0f, nb::arg("v_max") = 0.0f, nb::arg("format") = "%.3f", nb::arg("flags") = 0,
        "If v_min >= v_max we have no bound");

    m.def("drag_float2",
        [](const char * label, std::array<float, 2> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
        {
            auto DragFloat2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 2> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::DragFloat2(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragFloat2_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0.0f, nb::arg("v_max") = 0.0f, nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("drag_float3",
        [](const char * label, std::array<float, 3> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
        {
            auto DragFloat3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 3> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::DragFloat3(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragFloat3_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0.0f, nb::arg("v_max") = 0.0f, nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("drag_float4",
        [](const char * label, std::array<float, 4> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
        {
            auto DragFloat4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 4> v, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::DragFloat4(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragFloat4_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0.0f, nb::arg("v_max") = 0.0f, nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("drag_float_range2",
        [](const char * label, float v_current_min, float v_current_max, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> std::tuple<bool, float, float>
        {
            auto DragFloatRange2_adapt_const_char_pointer_with_default_null = [](const char * label, float * v_current_min, float * v_current_max, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_max_adapt_default_null = nullptr;
                if (format_max.has_value())
                    format_max_adapt_default_null = format_max.value().c_str();

                auto lambda_result = ImGui::DragFloatRange2(label, v_current_min, v_current_max, v_speed, v_min, v_max, format, format_max_adapt_default_null, flags);
                return lambda_result;
            };
            auto DragFloatRange2_adapt_modifiable_immutable_to_return = [&DragFloatRange2_adapt_const_char_pointer_with_default_null](const char * label, float v_current_min, float v_current_max, float v_speed = 1.0f, float v_min = 0.0f, float v_max = 0.0f, const char * format = "%.3f", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> std::tuple<bool, float, float>
            {
                float * v_current_min_adapt_modifiable = & v_current_min;
                float * v_current_max_adapt_modifiable = & v_current_max;

                bool r = DragFloatRange2_adapt_const_char_pointer_with_default_null(label, v_current_min_adapt_modifiable, v_current_max_adapt_modifiable, v_speed, v_min, v_max, format, format_max, flags);
                return std::make_tuple(r, v_current_min, v_current_max);
            };

            return DragFloatRange2_adapt_modifiable_immutable_to_return(label, v_current_min, v_current_max, v_speed, v_min, v_max, format, format_max, flags);
        },     nb::arg("label"), nb::arg("v_current_min"), nb::arg("v_current_max"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0.0f, nb::arg("v_max") = 0.0f, nb::arg("format") = "%.3f", nb::arg("format_max").none() = nb::none(), nb::arg("flags") = 0);

    m.def("drag_int",
        [](const char * label, int v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
        {
            auto DragInt_adapt_modifiable_immutable_to_return = [](const char * label, int v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
            {
                int * v_adapt_modifiable = & v;

                bool r = ImGui::DragInt(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragInt_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },
        nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0, nb::arg("v_max") = 0, nb::arg("format") = "%d", nb::arg("flags") = 0,
        "If v_min >= v_max we have no bound");

    m.def("drag_int2",
        [](const char * label, std::array<int, 2> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
        {
            auto DragInt2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 2> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::DragInt2(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragInt2_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0, nb::arg("v_max") = 0, nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("drag_int3",
        [](const char * label, std::array<int, 3> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
        {
            auto DragInt3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 3> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::DragInt3(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragInt3_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0, nb::arg("v_max") = 0, nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("drag_int4",
        [](const char * label, std::array<int, 4> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
        {
            auto DragInt4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 4> v, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::DragInt4(label, v_adapt_modifiable, v_speed, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return DragInt4_adapt_modifiable_immutable_to_return(label, v, v_speed, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0, nb::arg("v_max") = 0, nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("drag_int_range2",
        [](const char * label, int v_current_min, int v_current_max, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> std::tuple<bool, int, int>
        {
            auto DragIntRange2_adapt_const_char_pointer_with_default_null = [](const char * label, int * v_current_min, int * v_current_max, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_max_adapt_default_null = nullptr;
                if (format_max.has_value())
                    format_max_adapt_default_null = format_max.value().c_str();

                auto lambda_result = ImGui::DragIntRange2(label, v_current_min, v_current_max, v_speed, v_min, v_max, format, format_max_adapt_default_null, flags);
                return lambda_result;
            };
            auto DragIntRange2_adapt_modifiable_immutable_to_return = [&DragIntRange2_adapt_const_char_pointer_with_default_null](const char * label, int v_current_min, int v_current_max, float v_speed = 1.0f, int v_min = 0, int v_max = 0, const char * format = "%d", std::optional<std::string> format_max = std::nullopt, ImGuiSliderFlags flags = 0) -> std::tuple<bool, int, int>
            {
                int * v_current_min_adapt_modifiable = & v_current_min;
                int * v_current_max_adapt_modifiable = & v_current_max;

                bool r = DragIntRange2_adapt_const_char_pointer_with_default_null(label, v_current_min_adapt_modifiable, v_current_max_adapt_modifiable, v_speed, v_min, v_max, format, format_max, flags);
                return std::make_tuple(r, v_current_min, v_current_max);
            };

            return DragIntRange2_adapt_modifiable_immutable_to_return(label, v_current_min, v_current_max, v_speed, v_min, v_max, format, format_max, flags);
        },     nb::arg("label"), nb::arg("v_current_min"), nb::arg("v_current_max"), nb::arg("v_speed") = 1.0f, nb::arg("v_min") = 0, nb::arg("v_max") = 0, nb::arg("format") = "%d", nb::arg("format_max").none() = nb::none(), nb::arg("flags") = 0);

    m.def("drag_scalar",
        [](const char * label, ImGuiDataType data_type, void * p_data, float v_speed = 1.0f, const void * p_min = NULL, const void * p_max = NULL, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
        {
            auto DragScalar_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, float v_speed = 1.0f, const void * p_min = NULL, const void * p_max = NULL, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::DragScalar(label, data_type, p_data, v_speed, p_min, p_max, format_adapt_default_null, flags);
                return lambda_result;
            };

            return DragScalar_adapt_const_char_pointer_with_default_null(label, data_type, p_data, v_speed, p_min, p_max, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("v_speed") = 1.0f, nb::arg("p_min") = nb::none(), nb::arg("p_max") = nb::none(), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("drag_scalar_n",
        [](const char * label, ImGuiDataType data_type, void * p_data, int components, float v_speed = 1.0f, const void * p_min = NULL, const void * p_max = NULL, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
        {
            auto DragScalarN_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, int components, float v_speed = 1.0f, const void * p_min = NULL, const void * p_max = NULL, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::DragScalarN(label, data_type, p_data, components, v_speed, p_min, p_max, format_adapt_default_null, flags);
                return lambda_result;
            };

            return DragScalarN_adapt_const_char_pointer_with_default_null(label, data_type, p_data, components, v_speed, p_min, p_max, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("components"), nb::arg("v_speed") = 1.0f, nb::arg("p_min") = nb::none(), nb::arg("p_max") = nb::none(), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("slider_float",
        [](const char * label, float v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
        {
            auto SliderFloat_adapt_modifiable_immutable_to_return = [](const char * label, float v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
            {
                float * v_adapt_modifiable = & v;

                bool r = ImGui::SliderFloat(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderFloat_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },
        nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%.3f", nb::arg("flags") = 0,
        "adjust format to decorate the value with a prefix or a suffix for in-slider labels or unit display.");

    m.def("slider_float2",
        [](const char * label, std::array<float, 2> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
        {
            auto SliderFloat2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 2> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderFloat2(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderFloat2_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("slider_float3",
        [](const char * label, std::array<float, 3> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
        {
            auto SliderFloat3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 3> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderFloat3(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderFloat3_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("slider_float4",
        [](const char * label, std::array<float, 4> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
        {
            auto SliderFloat4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 4> v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderFloat4(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderFloat4_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("slider_angle",
        [](const char * label, float v_rad, float v_degrees_min = -360.0f, float v_degrees_max = +360.0f, const char * format = "%.0f deg", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
        {
            auto SliderAngle_adapt_modifiable_immutable_to_return = [](const char * label, float v_rad, float v_degrees_min = -360.0f, float v_degrees_max = +360.0f, const char * format = "%.0f deg", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
            {
                float * v_rad_adapt_modifiable = & v_rad;

                bool r = ImGui::SliderAngle(label, v_rad_adapt_modifiable, v_degrees_min, v_degrees_max, format, flags);
                return std::make_tuple(r, v_rad);
            };

            return SliderAngle_adapt_modifiable_immutable_to_return(label, v_rad, v_degrees_min, v_degrees_max, format, flags);
        },     nb::arg("label"), nb::arg("v_rad"), nb::arg("v_degrees_min") = -360.0f, nb::arg("v_degrees_max") = +360.0f, nb::arg("format") = "%.0f deg", nb::arg("flags") = 0);

    m.def("slider_int",
        [](const char * label, int v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
        {
            auto SliderInt_adapt_modifiable_immutable_to_return = [](const char * label, int v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
            {
                int * v_adapt_modifiable = & v;

                bool r = ImGui::SliderInt(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderInt_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("slider_int2",
        [](const char * label, std::array<int, 2> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
        {
            auto SliderInt2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 2> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderInt2(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderInt2_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("slider_int3",
        [](const char * label, std::array<int, 3> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
        {
            auto SliderInt3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 3> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderInt3(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderInt3_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("slider_int4",
        [](const char * label, std::array<int, 4> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
        {
            auto SliderInt4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 4> v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::SliderInt4(label, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return SliderInt4_adapt_modifiable_immutable_to_return(label, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("slider_scalar",
        [](const char * label, ImGuiDataType data_type, void * p_data, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
        {
            auto SliderScalar_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::SliderScalar(label, data_type, p_data, p_min, p_max, format_adapt_default_null, flags);
                return lambda_result;
            };

            return SliderScalar_adapt_const_char_pointer_with_default_null(label, data_type, p_data, p_min, p_max, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("p_min"), nb::arg("p_max"), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("slider_scalar_n",
        [](const char * label, ImGuiDataType data_type, void * p_data, int components, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
        {
            auto SliderScalarN_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, int components, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::SliderScalarN(label, data_type, p_data, components, p_min, p_max, format_adapt_default_null, flags);
                return lambda_result;
            };

            return SliderScalarN_adapt_const_char_pointer_with_default_null(label, data_type, p_data, components, p_min, p_max, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("components"), nb::arg("p_min"), nb::arg("p_max"), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("v_slider_float",
        [](const char * label, const ImVec2 & size, float v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
        {
            auto VSliderFloat_adapt_modifiable_immutable_to_return = [](const char * label, const ImVec2 & size, float v, float v_min, float v_max, const char * format = "%.3f", ImGuiSliderFlags flags = 0) -> std::tuple<bool, float>
            {
                float * v_adapt_modifiable = & v;

                bool r = ImGui::VSliderFloat(label, size, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return VSliderFloat_adapt_modifiable_immutable_to_return(label, size, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("size"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("v_slider_int",
        [](const char * label, const ImVec2 & size, int v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
        {
            auto VSliderInt_adapt_modifiable_immutable_to_return = [](const char * label, const ImVec2 & size, int v, int v_min, int v_max, const char * format = "%d", ImGuiSliderFlags flags = 0) -> std::tuple<bool, int>
            {
                int * v_adapt_modifiable = & v;

                bool r = ImGui::VSliderInt(label, size, v_adapt_modifiable, v_min, v_max, format, flags);
                return std::make_tuple(r, v);
            };

            return VSliderInt_adapt_modifiable_immutable_to_return(label, size, v, v_min, v_max, format, flags);
        },     nb::arg("label"), nb::arg("size"), nb::arg("v"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("format") = "%d", nb::arg("flags") = 0);

    m.def("v_slider_scalar",
        [](const char * label, const ImVec2 & size, ImGuiDataType data_type, void * p_data, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
        {
            auto VSliderScalar_adapt_const_char_pointer_with_default_null = [](const char * label, const ImVec2 & size, ImGuiDataType data_type, void * p_data, const void * p_min, const void * p_max, std::optional<std::string> format = std::nullopt, ImGuiSliderFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::VSliderScalar(label, size, data_type, p_data, p_min, p_max, format_adapt_default_null, flags);
                return lambda_result;
            };

            return VSliderScalar_adapt_const_char_pointer_with_default_null(label, size, data_type, p_data, p_min, p_max, format, flags);
        },     nb::arg("label"), nb::arg("size"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("p_min"), nb::arg("p_max"), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("input_float",
        [](const char * label, float v, float step = 0.0f, float step_fast = 0.0f, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, float>
        {
            auto InputFloat_adapt_modifiable_immutable_to_return = [](const char * label, float v, float step = 0.0f, float step_fast = 0.0f, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, float>
            {
                float * v_adapt_modifiable = & v;

                bool r = ImGui::InputFloat(label, v_adapt_modifiable, step, step_fast, format, flags);
                return std::make_tuple(r, v);
            };

            return InputFloat_adapt_modifiable_immutable_to_return(label, v, step, step_fast, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("step") = 0.0f, nb::arg("step_fast") = 0.0f, nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("input_float2",
        [](const char * label, std::array<float, 2> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
        {
            auto InputFloat2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 2> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 2>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::InputFloat2(label, v_adapt_modifiable, format, flags);
                return std::make_tuple(r, v);
            };

            return InputFloat2_adapt_modifiable_immutable_to_return(label, v, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("input_float3",
        [](const char * label, std::array<float, 3> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
        {
            auto InputFloat3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 3> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::InputFloat3(label, v_adapt_modifiable, format, flags);
                return std::make_tuple(r, v);
            };

            return InputFloat3_adapt_modifiable_immutable_to_return(label, v, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("input_float4",
        [](const char * label, std::array<float, 4> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
        {
            auto InputFloat4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 4> v, const char * format = "%.3f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
            {
                float * v_adapt_modifiable = v.data();

                bool r = ImGui::InputFloat4(label, v_adapt_modifiable, format, flags);
                return std::make_tuple(r, v);
            };

            return InputFloat4_adapt_modifiable_immutable_to_return(label, v, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("format") = "%.3f", nb::arg("flags") = 0);

    m.def("input_int",
        [](const char * label, int v, int step = 1, int step_fast = 100, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, int>
        {
            auto InputInt_adapt_modifiable_immutable_to_return = [](const char * label, int v, int step = 1, int step_fast = 100, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, int>
            {
                int * v_adapt_modifiable = & v;

                bool r = ImGui::InputInt(label, v_adapt_modifiable, step, step_fast, flags);
                return std::make_tuple(r, v);
            };

            return InputInt_adapt_modifiable_immutable_to_return(label, v, step, step_fast, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("step") = 1, nb::arg("step_fast") = 100, nb::arg("flags") = 0);

    m.def("input_int2",
        [](const char * label, std::array<int, 2> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
        {
            auto InputInt2_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 2> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 2>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::InputInt2(label, v_adapt_modifiable, flags);
                return std::make_tuple(r, v);
            };

            return InputInt2_adapt_modifiable_immutable_to_return(label, v, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("flags") = 0);

    m.def("input_int3",
        [](const char * label, std::array<int, 3> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
        {
            auto InputInt3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 3> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 3>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::InputInt3(label, v_adapt_modifiable, flags);
                return std::make_tuple(r, v);
            };

            return InputInt3_adapt_modifiable_immutable_to_return(label, v, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("flags") = 0);

    m.def("input_int4",
        [](const char * label, std::array<int, 4> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
        {
            auto InputInt4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<int, 4> v, ImGuiInputTextFlags flags = 0) -> std::tuple<bool, std::array<int, 4>>
            {
                int * v_adapt_modifiable = v.data();

                bool r = ImGui::InputInt4(label, v_adapt_modifiable, flags);
                return std::make_tuple(r, v);
            };

            return InputInt4_adapt_modifiable_immutable_to_return(label, v, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("flags") = 0);

    m.def("input_double",
        [](const char * label, double v, double step = 0.0, double step_fast = 0.0, const char * format = "%.6f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, double>
        {
            auto InputDouble_adapt_modifiable_immutable_to_return = [](const char * label, double v, double step = 0.0, double step_fast = 0.0, const char * format = "%.6f", ImGuiInputTextFlags flags = 0) -> std::tuple<bool, double>
            {
                double * v_adapt_modifiable = & v;

                bool r = ImGui::InputDouble(label, v_adapt_modifiable, step, step_fast, format, flags);
                return std::make_tuple(r, v);
            };

            return InputDouble_adapt_modifiable_immutable_to_return(label, v, step, step_fast, format, flags);
        },     nb::arg("label"), nb::arg("v"), nb::arg("step") = 0.0, nb::arg("step_fast") = 0.0, nb::arg("format") = "%.6f", nb::arg("flags") = 0);

    m.def("input_scalar",
        [](const char * label, ImGuiDataType data_type, void * p_data, const void * p_step = NULL, const void * p_step_fast = NULL, std::optional<std::string> format = std::nullopt, ImGuiInputTextFlags flags = 0) -> bool
        {
            auto InputScalar_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, const void * p_step = NULL, const void * p_step_fast = NULL, std::optional<std::string> format = std::nullopt, ImGuiInputTextFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::InputScalar(label, data_type, p_data, p_step, p_step_fast, format_adapt_default_null, flags);
                return lambda_result;
            };

            return InputScalar_adapt_const_char_pointer_with_default_null(label, data_type, p_data, p_step, p_step_fast, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("p_step") = nb::none(), nb::arg("p_step_fast") = nb::none(), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("input_scalar_n",
        [](const char * label, ImGuiDataType data_type, void * p_data, int components, const void * p_step = NULL, const void * p_step_fast = NULL, std::optional<std::string> format = std::nullopt, ImGuiInputTextFlags flags = 0) -> bool
        {
            auto InputScalarN_adapt_const_char_pointer_with_default_null = [](const char * label, ImGuiDataType data_type, void * p_data, int components, const void * p_step = NULL, const void * p_step_fast = NULL, std::optional<std::string> format = std::nullopt, ImGuiInputTextFlags flags = 0) -> bool
            {
                const char * format_adapt_default_null = nullptr;
                if (format.has_value())
                    format_adapt_default_null = format.value().c_str();

                auto lambda_result = ImGui::InputScalarN(label, data_type, p_data, components, p_step, p_step_fast, format_adapt_default_null, flags);
                return lambda_result;
            };

            return InputScalarN_adapt_const_char_pointer_with_default_null(label, data_type, p_data, components, p_step, p_step_fast, format, flags);
        },     nb::arg("label"), nb::arg("data_type"), nb::arg("p_data"), nb::arg("components"), nb::arg("p_step") = nb::none(), nb::arg("p_step_fast") = nb::none(), nb::arg("format").none() = nb::none(), nb::arg("flags") = 0);

    m.def("color_edit3",
        [](const char * label, std::array<float, 3> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
        {
            auto ColorEdit3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 3> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
            {
                float * col_adapt_modifiable = col.data();

                bool r = ImGui::ColorEdit3(label, col_adapt_modifiable, flags);
                return std::make_tuple(r, col);
            };

            return ColorEdit3_adapt_modifiable_immutable_to_return(label, col, flags);
        },     nb::arg("label"), nb::arg("col"), nb::arg("flags") = 0);

    m.def("color_edit4",
        [](const char * label, std::array<float, 4> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
        {
            auto ColorEdit4_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 4> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 4>>
            {
                float * col_adapt_modifiable = col.data();

                bool r = ImGui::ColorEdit4(label, col_adapt_modifiable, flags);
                return std::make_tuple(r, col);
            };

            return ColorEdit4_adapt_modifiable_immutable_to_return(label, col, flags);
        },     nb::arg("label"), nb::arg("col"), nb::arg("flags") = 0);

    m.def("color_picker3",
        [](const char * label, std::array<float, 3> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
        {
            auto ColorPicker3_adapt_modifiable_immutable_to_return = [](const char * label, std::array<float, 3> col, ImGuiColorEditFlags flags = 0) -> std::tuple<bool, std::array<float, 3>>
            {
                float * col_adapt_modifiable = col.data();

                bool r = ImGui::ColorPicker3(label, col_adapt_modifiable, flags);
                return std::make_tuple(r, col);
            };

            return ColorPicker3_adapt_modifiable_immutable_to_return(label, col, flags);
        },     nb::arg("label"), nb::arg("col"), nb::arg("flags") = 0);
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("color_picker4",
        nb::overload_cast<const std::string &, ImVec4, ImGuiColorEditFlags, std::optional<ImVec4>>(ImGui::ColorPicker4), nb::arg("label"), nb::arg("col"), nb::arg("flags") = 0, nb::arg("ref_col").none() = nb::none());
    // #endif
    //

    m.def("color_button",
        [](const char * desc_id, const ImVec4 & col, ImGuiColorEditFlags flags = 0, const std::optional<const ImVec2> & size = std::nullopt) -> bool
        {
            auto ColorButton_adapt_mutable_param_with_default_value = [](const char * desc_id, const ImVec4 & col, ImGuiColorEditFlags flags = 0, const std::optional<const ImVec2> & size = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::ColorButton(desc_id, col, flags, size_or_default);
                return lambda_result;
            };

            return ColorButton_adapt_mutable_param_with_default_value(desc_id, col, flags, size);
        },
        nb::arg("desc_id"), nb::arg("col"), nb::arg("flags") = 0, nb::arg("size").none() = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)\n\n display a color square/button, hover for details, return True when pressed.");

    m.def("set_color_edit_options",
        ImGui::SetColorEditOptions,
        nb::arg("flags"),
        "initialize current options (generally on application startup) if you want to select a default format, picker type, etc. User will be able to change many settings, unless you pass the _NoOptions flag to your calls.");

    m.def("tree_node",
        nb::overload_cast<const char *>(ImGui::TreeNode), nb::arg("label"));

    m.def("tree_node",
        [](const char * str_id, const char * fmt) -> bool
        {
            auto TreeNode_adapt_variadic_format = [](const char * str_id, const char * fmt) -> bool
            {
                auto lambda_result = ImGui::TreeNode(str_id, "%s", fmt);
                return lambda_result;
            };

            return TreeNode_adapt_variadic_format(str_id, fmt);
        },
        nb::arg("str_id"), nb::arg("fmt"),
        "helper variation to easily decorelate the id from the displayed string. Read the FAQ about why and how to use ID. to align arbitrary text at the same level as a TreeNode() you can use Bullet().");

    m.def("tree_node",
        [](const void * ptr_id, const char * fmt) -> bool
        {
            auto TreeNode_adapt_variadic_format = [](const void * ptr_id, const char * fmt) -> bool
            {
                auto lambda_result = ImGui::TreeNode(ptr_id, "%s", fmt);
                return lambda_result;
            };

            return TreeNode_adapt_variadic_format(ptr_id, fmt);
        },
        nb::arg("ptr_id"), nb::arg("fmt"),
        "\"");

    m.def("tree_node_ex",
        nb::overload_cast<const char *, ImGuiTreeNodeFlags>(ImGui::TreeNodeEx), nb::arg("label"), nb::arg("flags") = 0);

    m.def("tree_node_ex",
        [](const char * str_id, ImGuiTreeNodeFlags flags, const char * fmt) -> bool
        {
            auto TreeNodeEx_adapt_variadic_format = [](const char * str_id, ImGuiTreeNodeFlags flags, const char * fmt) -> bool
            {
                auto lambda_result = ImGui::TreeNodeEx(str_id, flags, "%s", fmt);
                return lambda_result;
            };

            return TreeNodeEx_adapt_variadic_format(str_id, flags, fmt);
        },     nb::arg("str_id"), nb::arg("flags"), nb::arg("fmt"));

    m.def("tree_node_ex",
        [](const void * ptr_id, ImGuiTreeNodeFlags flags, const char * fmt) -> bool
        {
            auto TreeNodeEx_adapt_variadic_format = [](const void * ptr_id, ImGuiTreeNodeFlags flags, const char * fmt) -> bool
            {
                auto lambda_result = ImGui::TreeNodeEx(ptr_id, flags, "%s", fmt);
                return lambda_result;
            };

            return TreeNodeEx_adapt_variadic_format(ptr_id, flags, fmt);
        },     nb::arg("ptr_id"), nb::arg("flags"), nb::arg("fmt"));

    m.def("tree_push",
        nb::overload_cast<const char *>(ImGui::TreePush),
        nb::arg("str_id"),
        "~ Indent()+PushID(). Already called by TreeNode() when returning True, but you can call TreePush/TreePop yourself if desired.");

    m.def("tree_push",
        nb::overload_cast<const void *>(ImGui::TreePush),
        nb::arg("ptr_id"),
        "\"");

    m.def("tree_pop",
        ImGui::TreePop, "~ Unindent()+PopID()");

    m.def("get_tree_node_to_label_spacing",
        ImGui::GetTreeNodeToLabelSpacing, "horizontal distance preceding label when using TreeNode*() or Bullet() == (g.FontSize + style.FramePadding.x*2) for a regular unframed TreeNode");

    m.def("collapsing_header",
        nb::overload_cast<const char *, ImGuiTreeNodeFlags>(ImGui::CollapsingHeader),
        nb::arg("label"), nb::arg("flags") = 0,
        "if returning 'True' the header is open. doesn't indent nor push on ID stack. user doesn't have to call TreePop().");

    m.def("collapsing_header",
        [](const char * label, bool p_visible, ImGuiTreeNodeFlags flags = 0) -> std::tuple<bool, bool>
        {
            auto CollapsingHeader_adapt_modifiable_immutable_to_return = [](const char * label, bool p_visible, ImGuiTreeNodeFlags flags = 0) -> std::tuple<bool, bool>
            {
                bool * p_visible_adapt_modifiable = & p_visible;

                bool r = ImGui::CollapsingHeader(label, p_visible_adapt_modifiable, flags);
                return std::make_tuple(r, p_visible);
            };

            return CollapsingHeader_adapt_modifiable_immutable_to_return(label, p_visible, flags);
        },
        nb::arg("label"), nb::arg("p_visible"), nb::arg("flags") = 0,
        "when 'p_visible != None': if '*p_visible==True' display an additional small close button on upper right of the header which will set the bool to False when clicked, if '*p_visible==False' don't display the header.");

    m.def("set_next_item_open",
        ImGui::SetNextItemOpen,
        nb::arg("is_open"), nb::arg("cond") = 0,
        "set next TreeNode/CollapsingHeader open state.");

    m.def("set_next_item_storage_id",
        ImGui::SetNextItemStorageID,
        nb::arg("storage_id"),
        "set id to use for open/close storage (default to same as item id).");

    m.def("selectable",
        [](const char * label, bool p_selected, ImGuiSelectableFlags flags = 0, const std::optional<const ImVec2> & size = std::nullopt) -> std::tuple<bool, bool>
        {
            auto Selectable_adapt_mutable_param_with_default_value = [](const char * label, bool * p_selected, ImGuiSelectableFlags flags = 0, const std::optional<const ImVec2> & size = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::Selectable(label, p_selected, flags, size_or_default);
                return lambda_result;
            };
            auto Selectable_adapt_modifiable_immutable_to_return = [&Selectable_adapt_mutable_param_with_default_value](const char * label, bool p_selected, ImGuiSelectableFlags flags = 0, const std::optional<const ImVec2> & size = std::nullopt) -> std::tuple<bool, bool>
            {
                bool * p_selected_adapt_modifiable = & p_selected;

                bool r = Selectable_adapt_mutable_param_with_default_value(label, p_selected_adapt_modifiable, flags, size);
                return std::make_tuple(r, p_selected);
            };

            return Selectable_adapt_modifiable_immutable_to_return(label, p_selected, flags, size);
        },
        nb::arg("label"), nb::arg("p_selected"), nb::arg("flags") = 0, nb::arg("size").none() = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)\n\n \"bool* p_selected\" point to the selection state (read-write), as a convenient helper.");

    m.def("begin_multi_select",
        ImGui::BeginMultiSelect,
        nb::arg("flags"), nb::arg("selection_size") = -1, nb::arg("items_count") = -1,
        nb::rv_policy::reference);

    m.def("end_multi_select",
        ImGui::EndMultiSelect, nb::rv_policy::reference);

    m.def("set_next_item_selection_user_data",
        ImGui::SetNextItemSelectionUserData, nb::arg("selection_user_data"));

    m.def("is_item_toggled_selection",
        ImGui::IsItemToggledSelection, "Was the last item selection state toggled? Useful if you need the per-item information _before_ reaching EndMultiSelect(). We only returns toggle _event_ in order to handle clipping correctly.");

    m.def("begin_list_box",
        [](const char * label, const std::optional<const ImVec2> & size = std::nullopt) -> bool
        {
            auto BeginListBox_adapt_mutable_param_with_default_value = [](const char * label, const std::optional<const ImVec2> & size = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::BeginListBox(label, size_or_default);
                return lambda_result;
            };

            return BeginListBox_adapt_mutable_param_with_default_value(label, size);
        },
        nb::arg("label"), nb::arg("size").none() = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)\n\n open a framed scrolling region");

    m.def("end_list_box",
        ImGui::EndListBox, "only call EndListBox() if BeginListBox() returned True!");

    m.def("list_box",
        [](const char * label, int current_item, const std::vector<std::string> & items, int height_in_items = -1) -> std::tuple<bool, int>
        {
            auto ListBox_adapt_modifiable_immutable_to_return = [](const char * label, int current_item, const char * const items[], int items_count, int height_in_items = -1) -> std::tuple<bool, int>
            {
                int * current_item_adapt_modifiable = & current_item;

                bool r = ImGui::ListBox(label, current_item_adapt_modifiable, items, items_count, height_in_items);
                return std::make_tuple(r, current_item);
            };
            auto ListBox_adapt_c_string_list = [&ListBox_adapt_modifiable_immutable_to_return](const char * label, int current_item, const std::vector<std::string> & items, int height_in_items = -1) -> std::tuple<bool, int>
            {
                std::vector<const char *> items_ptrs;
                items_ptrs.reserve(items.size());
                for (const auto& v: items)
                    items_ptrs.push_back(v.c_str());
                int items_count = static_cast<int>(items.size());

                auto lambda_result = ListBox_adapt_modifiable_immutable_to_return(label, current_item, items_ptrs.data(), items_count, height_in_items);
                return lambda_result;
            };

            return ListBox_adapt_c_string_list(label, current_item, items, height_in_items);
        },     nb::arg("label"), nb::arg("current_item"), nb::arg("items"), nb::arg("height_in_items") = -1);

    m.def("plot_lines",
        [](const char * label, const nb::ndarray<> & values, int values_offset = 0, std::optional<std::string> overlay_text = std::nullopt, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
        {
            auto PlotLines_adapt_c_buffers = [](const char * label, const nb::ndarray<> & values, int values_offset = 0, const char * overlay_text = NULL, float scale_min = FLT_MAX, float scale_max = FLT_MAX, ImVec2 graph_size = ImVec2(0, 0), int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);
                // Check the type of the ndarray (generic type and size)
                //   - Step 1: check the generic type (one of dtype_code::Int, UInt, Float, Bfloat, Complex, Bool = 6);
                uint8_t dtype_code_python_1 = values.dtype().code;
                uint8_t dtype_code_cpp_1 = static_cast<uint8_t>(nb::dlpack::dtype_code::Float);
                if (dtype_code_python_1 != dtype_code_cpp_1)
                    throw std::runtime_error(std::string(R"msg(
                            Bad type! While checking the generic type (dtype_code=Float)!
                        )msg"));
                //   - Step 2: check the size of the type
                size_t size_python_1 = values.dtype().bits / 8;
                size_t size_cpp_1 = sizeof(float);
                if (size_python_1 != size_cpp_1)
                    throw std::runtime_error(std::string(R"msg(
                            Bad type! Size mismatch, while checking the size of the type (for param "values")!
                        )msg"));

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                ImGui::PlotLines(label, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), values_offset, overlay_text, scale_min, scale_max, graph_size, values_stride);
            };
            auto PlotLines_adapt_mutable_param_with_default_value = [&PlotLines_adapt_c_buffers](const char * label, const nb::ndarray<> & values, int values_offset = 0, const char * overlay_text = NULL, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
            {

                const ImVec2& graph_size_or_default = [&]() -> const ImVec2 {
                    if (graph_size.has_value())
                        return graph_size.value();
                    else
                        return ImVec2(0, 0);
                }();

                PlotLines_adapt_c_buffers(label, values, values_offset, overlay_text, scale_min, scale_max, graph_size_or_default, stride);
            };
            auto PlotLines_adapt_const_char_pointer_with_default_null = [&PlotLines_adapt_mutable_param_with_default_value](const char * label, const nb::ndarray<> & values, int values_offset = 0, std::optional<std::string> overlay_text = std::nullopt, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
            {
                const char * overlay_text_adapt_default_null = nullptr;
                if (overlay_text.has_value())
                    overlay_text_adapt_default_null = overlay_text.value().c_str();

                PlotLines_adapt_mutable_param_with_default_value(label, values, values_offset, overlay_text_adapt_default_null, scale_min, scale_max, graph_size, stride);
            };

            PlotLines_adapt_const_char_pointer_with_default_null(label, values, values_offset, overlay_text, scale_min, scale_max, graph_size, stride);
        },
        nb::arg("label"), nb::arg("values"), nb::arg("values_offset") = 0, nb::arg("overlay_text").none() = nb::none(), nb::arg("scale_min") = FLT_MAX, nb::arg("scale_max") = FLT_MAX, nb::arg("graph_size").none() = nb::none(), nb::arg("stride") = -1,
        "Python bindings defaults:\n    If graph_size is None, then its default value will be: ImVec2(0, 0)");

    m.def("plot_histogram",
        [](const char * label, const nb::ndarray<> & values, int values_offset = 0, std::optional<std::string> overlay_text = std::nullopt, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
        {
            auto PlotHistogram_adapt_c_buffers = [](const char * label, const nb::ndarray<> & values, int values_offset = 0, const char * overlay_text = NULL, float scale_min = FLT_MAX, float scale_max = FLT_MAX, ImVec2 graph_size = ImVec2(0, 0), int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);
                // Check the type of the ndarray (generic type and size)
                //   - Step 1: check the generic type (one of dtype_code::Int, UInt, Float, Bfloat, Complex, Bool = 6);
                uint8_t dtype_code_python_1 = values.dtype().code;
                uint8_t dtype_code_cpp_1 = static_cast<uint8_t>(nb::dlpack::dtype_code::Float);
                if (dtype_code_python_1 != dtype_code_cpp_1)
                    throw std::runtime_error(std::string(R"msg(
                            Bad type! While checking the generic type (dtype_code=Float)!
                        )msg"));
                //   - Step 2: check the size of the type
                size_t size_python_1 = values.dtype().bits / 8;
                size_t size_cpp_1 = sizeof(float);
                if (size_python_1 != size_cpp_1)
                    throw std::runtime_error(std::string(R"msg(
                            Bad type! Size mismatch, while checking the size of the type (for param "values")!
                        )msg"));

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                ImGui::PlotHistogram(label, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), values_offset, overlay_text, scale_min, scale_max, graph_size, values_stride);
            };
            auto PlotHistogram_adapt_mutable_param_with_default_value = [&PlotHistogram_adapt_c_buffers](const char * label, const nb::ndarray<> & values, int values_offset = 0, const char * overlay_text = NULL, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
            {

                const ImVec2& graph_size_or_default = [&]() -> const ImVec2 {
                    if (graph_size.has_value())
                        return graph_size.value();
                    else
                        return ImVec2(0, 0);
                }();

                PlotHistogram_adapt_c_buffers(label, values, values_offset, overlay_text, scale_min, scale_max, graph_size_or_default, stride);
            };
            auto PlotHistogram_adapt_const_char_pointer_with_default_null = [&PlotHistogram_adapt_mutable_param_with_default_value](const char * label, const nb::ndarray<> & values, int values_offset = 0, std::optional<std::string> overlay_text = std::nullopt, float scale_min = FLT_MAX, float scale_max = FLT_MAX, const std::optional<const ImVec2> & graph_size = std::nullopt, int stride = -1)
            {
                const char * overlay_text_adapt_default_null = nullptr;
                if (overlay_text.has_value())
                    overlay_text_adapt_default_null = overlay_text.value().c_str();

                PlotHistogram_adapt_mutable_param_with_default_value(label, values, values_offset, overlay_text_adapt_default_null, scale_min, scale_max, graph_size, stride);
            };

            PlotHistogram_adapt_const_char_pointer_with_default_null(label, values, values_offset, overlay_text, scale_min, scale_max, graph_size, stride);
        },
        nb::arg("label"), nb::arg("values"), nb::arg("values_offset") = 0, nb::arg("overlay_text").none() = nb::none(), nb::arg("scale_min") = FLT_MAX, nb::arg("scale_max") = FLT_MAX, nb::arg("graph_size").none() = nb::none(), nb::arg("stride") = -1,
        "Python bindings defaults:\n    If graph_size is None, then its default value will be: ImVec2(0, 0)");

    m.def("value",
        nb::overload_cast<const char *, bool>(ImGui::Value), nb::arg("prefix"), nb::arg("b"));

    m.def("value",
        nb::overload_cast<const char *, int>(ImGui::Value), nb::arg("prefix"), nb::arg("v"));

    m.def("value",
        nb::overload_cast<const char *, unsigned int>(ImGui::Value), nb::arg("prefix"), nb::arg("v"));

    m.def("value",
        [](const char * prefix, float v, std::optional<std::string> float_format = std::nullopt)
        {
            auto Value_adapt_const_char_pointer_with_default_null = [](const char * prefix, float v, std::optional<std::string> float_format = std::nullopt)
            {
                const char * float_format_adapt_default_null = nullptr;
                if (float_format.has_value())
                    float_format_adapt_default_null = float_format.value().c_str();

                ImGui::Value(prefix, v, float_format_adapt_default_null);
            };

            Value_adapt_const_char_pointer_with_default_null(prefix, v, float_format);
        },     nb::arg("prefix"), nb::arg("v"), nb::arg("float_format").none() = nb::none());

    m.def("begin_menu_bar",
        ImGui::BeginMenuBar, "append to menu-bar of current window (requires ImGuiWindowFlags_MenuBar flag set on parent window).");

    m.def("end_menu_bar",
        ImGui::EndMenuBar, "only call EndMenuBar() if BeginMenuBar() returns True!");

    m.def("begin_main_menu_bar",
        ImGui::BeginMainMenuBar, "create and append to a full screen menu-bar.");

    m.def("end_main_menu_bar",
        ImGui::EndMainMenuBar, "only call EndMainMenuBar() if BeginMainMenuBar() returns True!");

    m.def("begin_menu",
        ImGui::BeginMenu,
        nb::arg("label"), nb::arg("enabled") = true,
        "create a sub-menu entry. only call EndMenu() if this returns True!");

    m.def("end_menu",
        ImGui::EndMenu, "only call EndMenu() if BeginMenu() returns True!");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("menu_item_simple",
        [](const char * label, std::optional<std::string> shortcut = std::nullopt, bool selected = false, bool enabled = true) -> bool
        {
            auto MenuItemSimple_adapt_const_char_pointer_with_default_null = [](const char * label, std::optional<std::string> shortcut = std::nullopt, bool selected = false, bool enabled = true) -> bool
            {
                const char * shortcut_adapt_default_null = nullptr;
                if (shortcut.has_value())
                    shortcut_adapt_default_null = shortcut.value().c_str();

                auto lambda_result = ImGui::MenuItemSimple(label, shortcut_adapt_default_null, selected, enabled);
                return lambda_result;
            };

            return MenuItemSimple_adapt_const_char_pointer_with_default_null(label, shortcut, selected, enabled);
        },
        nb::arg("label"), nb::arg("shortcut").none() = nb::none(), nb::arg("selected") = false, nb::arg("enabled") = true,
        "(private API)");
    // #endif
    //

    m.def("menu_item",
        [](const char * label, const char * shortcut, bool p_selected, bool enabled = true) -> std::tuple<bool, bool>
        {
            auto MenuItem_adapt_modifiable_immutable_to_return = [](const char * label, const char * shortcut, bool p_selected, bool enabled = true) -> std::tuple<bool, bool>
            {
                bool * p_selected_adapt_modifiable = & p_selected;

                bool r = ImGui::MenuItem(label, shortcut, p_selected_adapt_modifiable, enabled);
                return std::make_tuple(r, p_selected);
            };

            return MenuItem_adapt_modifiable_immutable_to_return(label, shortcut, p_selected, enabled);
        },
        nb::arg("label"), nb::arg("shortcut"), nb::arg("p_selected"), nb::arg("enabled") = true,
        "return True when activated + toggle (*p_selected) if p_selected != None");

    m.def("begin_tooltip",
        ImGui::BeginTooltip, "begin/append a tooltip window.");

    m.def("end_tooltip",
        ImGui::EndTooltip, "only call EndTooltip() if BeginTooltip()/BeginItemTooltip() returns True!");

    m.def("set_tooltip",
        [](const char * fmt)
        {
            auto SetTooltip_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::SetTooltip("%s", fmt);
            };

            SetTooltip_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "set a text-only tooltip. Often used after a ImGui::IsItemHovered() check. Override any previous call to SetTooltip().");

    m.def("begin_item_tooltip",
        ImGui::BeginItemTooltip, "begin/append a tooltip window if preceding item was hovered.");

    m.def("set_item_tooltip",
        [](const char * fmt)
        {
            auto SetItemTooltip_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::SetItemTooltip("%s", fmt);
            };

            SetItemTooltip_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "set a text-only tooltip if preceding item was hovered. override any previous call to SetTooltip().");

    m.def("begin_popup",
        nb::overload_cast<const char *, ImGuiWindowFlags>(ImGui::BeginPopup),
        nb::arg("str_id"), nb::arg("flags") = 0,
        "return True if the popup is open, and you can start outputting to it.");

    m.def("begin_popup_modal",
        [](const char * name, std::optional<bool> p_open = std::nullopt, ImGuiWindowFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
        {
            auto BeginPopupModal_adapt_modifiable_immutable_to_return = [](const char * name, std::optional<bool> p_open = std::nullopt, ImGuiWindowFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                bool r = ImGui::BeginPopupModal(name, p_open_adapt_modifiable, flags);
                return std::make_tuple(r, p_open);
            };

            return BeginPopupModal_adapt_modifiable_immutable_to_return(name, p_open, flags);
        },
        nb::arg("name"), nb::arg("p_open").none() = nb::none(), nb::arg("flags") = 0,
        "return True if the modal is open, and you can start outputting to it.");

    m.def("end_popup",
        nb::overload_cast<>(ImGui::EndPopup), "only call EndPopup() if BeginPopupXXX() returns True!");

    m.def("open_popup",
        nb::overload_cast<const char *, ImGuiPopupFlags>(ImGui::OpenPopup),
        nb::arg("str_id"), nb::arg("popup_flags") = 0,
        "call to mark popup as open (don't call every frame!).");

    m.def("open_popup",
        nb::overload_cast<ImGuiID, ImGuiPopupFlags>(ImGui::OpenPopup),
        nb::arg("id_"), nb::arg("popup_flags") = 0,
        "id overload to facilitate calling from nested stacks");

    m.def("open_popup_on_item_click",
        [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1)
        {
            auto OpenPopupOnItemClick_adapt_const_char_pointer_with_default_null = [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1)
            {
                const char * str_id_adapt_default_null = nullptr;
                if (str_id.has_value())
                    str_id_adapt_default_null = str_id.value().c_str();

                ImGui::OpenPopupOnItemClick(str_id_adapt_default_null, popup_flags);
            };

            OpenPopupOnItemClick_adapt_const_char_pointer_with_default_null(str_id, popup_flags);
        },
        nb::arg("str_id").none() = nb::none(), nb::arg("popup_flags") = 1,
        "helper to open popup when clicked on last item. Default to ImGuiPopupFlags_MouseButtonRight == 1. (note: actually triggers on the mouse _released_ event to be consistent with popup behaviors)");

    m.def("close_current_popup",
        nb::overload_cast<>(ImGui::CloseCurrentPopup), "manually close the popup we have begin-ed into.");

    m.def("begin_popup_context_item",
        [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
        {
            auto BeginPopupContextItem_adapt_const_char_pointer_with_default_null = [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
            {
                const char * str_id_adapt_default_null = nullptr;
                if (str_id.has_value())
                    str_id_adapt_default_null = str_id.value().c_str();

                auto lambda_result = ImGui::BeginPopupContextItem(str_id_adapt_default_null, popup_flags);
                return lambda_result;
            };

            return BeginPopupContextItem_adapt_const_char_pointer_with_default_null(str_id, popup_flags);
        },
        nb::arg("str_id").none() = nb::none(), nb::arg("popup_flags") = 1,
        "open+begin popup when clicked on last item. Use str_id==None to associate the popup to previous item. If you want to use that on a non-interactive item such as Text() you need to pass in an explicit ID here. read comments in .cpp!");

    m.def("begin_popup_context_window",
        [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
        {
            auto BeginPopupContextWindow_adapt_const_char_pointer_with_default_null = [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
            {
                const char * str_id_adapt_default_null = nullptr;
                if (str_id.has_value())
                    str_id_adapt_default_null = str_id.value().c_str();

                auto lambda_result = ImGui::BeginPopupContextWindow(str_id_adapt_default_null, popup_flags);
                return lambda_result;
            };

            return BeginPopupContextWindow_adapt_const_char_pointer_with_default_null(str_id, popup_flags);
        },
        nb::arg("str_id").none() = nb::none(), nb::arg("popup_flags") = 1,
        "open+begin popup when clicked on current window.");

    m.def("begin_popup_context_void",
        [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
        {
            auto BeginPopupContextVoid_adapt_const_char_pointer_with_default_null = [](std::optional<std::string> str_id = std::nullopt, ImGuiPopupFlags popup_flags = 1) -> bool
            {
                const char * str_id_adapt_default_null = nullptr;
                if (str_id.has_value())
                    str_id_adapt_default_null = str_id.value().c_str();

                auto lambda_result = ImGui::BeginPopupContextVoid(str_id_adapt_default_null, popup_flags);
                return lambda_result;
            };

            return BeginPopupContextVoid_adapt_const_char_pointer_with_default_null(str_id, popup_flags);
        },
        nb::arg("str_id").none() = nb::none(), nb::arg("popup_flags") = 1,
        "open+begin popup when clicked in None (where there are no windows).");

    m.def("is_popup_open",
        nb::overload_cast<const char *, ImGuiPopupFlags>(ImGui::IsPopupOpen),
        nb::arg("str_id"), nb::arg("flags") = 0,
        "return True if the popup is open.");

    m.def("begin_table",
        [](const char * str_id, int columns, ImGuiTableFlags flags = 0, const std::optional<const ImVec2> & outer_size = std::nullopt, float inner_width = 0.0f) -> bool
        {
            auto BeginTable_adapt_mutable_param_with_default_value = [](const char * str_id, int columns, ImGuiTableFlags flags = 0, const std::optional<const ImVec2> & outer_size = std::nullopt, float inner_width = 0.0f) -> bool
            {

                const ImVec2& outer_size_or_default = [&]() -> const ImVec2 {
                    if (outer_size.has_value())
                        return outer_size.value();
                    else
                        return ImVec2(0.0f, 0.0f);
                }();

                auto lambda_result = ImGui::BeginTable(str_id, columns, flags, outer_size_or_default, inner_width);
                return lambda_result;
            };

            return BeginTable_adapt_mutable_param_with_default_value(str_id, columns, flags, outer_size, inner_width);
        },
        nb::arg("str_id"), nb::arg("columns"), nb::arg("flags") = 0, nb::arg("outer_size").none() = nb::none(), nb::arg("inner_width") = 0.0f,
        "Python bindings defaults:\n    If outer_size is None, then its default value will be: ImVec2(0.0, 0.0)");

    m.def("end_table",
        ImGui::EndTable, "only call EndTable() if BeginTable() returns True!");

    m.def("table_next_row",
        nb::overload_cast<ImGuiTableRowFlags, float>(ImGui::TableNextRow),
        nb::arg("row_flags") = 0, nb::arg("min_row_height") = 0.0f,
        "append into the first cell of a new row.");

    m.def("table_next_column",
        nb::overload_cast<>(ImGui::TableNextColumn), "append into the next column (or first column of next row if currently in last column). Return True when column is visible.");

    m.def("table_set_column_index",
        nb::overload_cast<int>(ImGui::TableSetColumnIndex),
        nb::arg("column_n"),
        "append into the specified column. Return True when column is visible.");

    m.def("table_setup_column",
        nb::overload_cast<const char *, ImGuiTableColumnFlags, float, ImGuiID>(ImGui::TableSetupColumn), nb::arg("label"), nb::arg("flags") = 0, nb::arg("init_width_or_weight") = 0.0f, nb::arg("user_id") = 0);

    m.def("table_setup_scroll_freeze",
        nb::overload_cast<int, int>(ImGui::TableSetupScrollFreeze),
        nb::arg("cols"), nb::arg("rows"),
        "lock columns/rows so they stay visible when scrolled.");

    m.def("table_header",
        nb::overload_cast<const char *>(ImGui::TableHeader),
        nb::arg("label"),
        "submit one header cell manually (rarely used)");

    m.def("table_headers_row",
        nb::overload_cast<>(ImGui::TableHeadersRow), "submit a row with headers cells based on data provided to TableSetupColumn() + submit context menu");

    m.def("table_angled_headers_row",
        nb::overload_cast<>(ImGui::TableAngledHeadersRow), "submit a row with angled headers for every column with the ImGuiTableColumnFlags_AngledHeader flag. MUST BE FIRST ROW.");

    m.def("table_get_sort_specs",
        nb::overload_cast<>(ImGui::TableGetSortSpecs),
        "get latest sort specs for the table (None if not sorting).  Lifetime: don't hold on this pointer over multiple frames or past any subsequent call to BeginTable().",
        nb::rv_policy::reference);

    m.def("table_get_column_count",
        nb::overload_cast<>(ImGui::TableGetColumnCount), "return number of columns (value passed to BeginTable)");

    m.def("table_get_column_index",
        nb::overload_cast<>(ImGui::TableGetColumnIndex), "return current column index.");

    m.def("table_get_row_index",
        nb::overload_cast<>(ImGui::TableGetRowIndex), "return current row index (header rows are accounted for)");

    m.def("table_get_column_name",
        nb::overload_cast<int>(ImGui::TableGetColumnName),
        nb::arg("column_n") = -1,
        "return \"\" if column didn't have a name declared by TableSetupColumn(). Pass -1 to use current column.",
        nb::rv_policy::reference);

    m.def("table_get_column_flags",
        nb::overload_cast<int>(ImGui::TableGetColumnFlags),
        nb::arg("column_n") = -1,
        "return column flags so you can query their Enabled/Visible/Sorted/Hovered status flags. Pass -1 to use current column.");

    m.def("table_set_column_enabled",
        nb::overload_cast<int, bool>(ImGui::TableSetColumnEnabled),
        nb::arg("column_n"), nb::arg("v"),
        "change user accessible enabled/disabled state of a column. Set to False to hide the column. User can use the context menu to change this themselves (right-click in headers, or right-click in columns body with ImGuiTableFlags_ContextMenuInBody)");

    m.def("table_get_hovered_column",
        nb::overload_cast<>(ImGui::TableGetHoveredColumn), "return hovered column. return -1 when table is not hovered. return columns_count if the unused space at the right of visible columns is hovered. Can also use (TableGetColumnFlags() & ImGuiTableColumnFlags_IsHovered) instead.");

    m.def("table_set_bg_color",
        nb::overload_cast<ImGuiTableBgTarget, ImU32, int>(ImGui::TableSetBgColor),
        nb::arg("target"), nb::arg("color"), nb::arg("column_n") = -1,
        "change the color of a cell, row, or column. See ImGuiTableBgTarget_ flags for details.");

    m.def("columns",
        [](int count = 1, std::optional<std::string> id = std::nullopt, bool borders = true)
        {
            auto Columns_adapt_const_char_pointer_with_default_null = [](int count = 1, std::optional<std::string> id = std::nullopt, bool borders = true)
            {
                const char * id_adapt_default_null = nullptr;
                if (id.has_value())
                    id_adapt_default_null = id.value().c_str();

                ImGui::Columns(count, id_adapt_default_null, borders);
            };

            Columns_adapt_const_char_pointer_with_default_null(count, id, borders);
        },     nb::arg("count") = 1, nb::arg("id_").none() = nb::none(), nb::arg("borders") = true);

    m.def("next_column",
        ImGui::NextColumn, "next column, defaults to current row or next row if the current row is finished");

    m.def("get_column_index",
        ImGui::GetColumnIndex, "get current column index");

    m.def("get_column_width",
        ImGui::GetColumnWidth,
        nb::arg("column_index") = -1,
        "get column width (in pixels). pass -1 to use current column");

    m.def("set_column_width",
        ImGui::SetColumnWidth,
        nb::arg("column_index"), nb::arg("width"),
        "set column width (in pixels). pass -1 to use current column");

    m.def("get_column_offset",
        ImGui::GetColumnOffset,
        nb::arg("column_index") = -1,
        "get position of column line (in pixels, from the left side of the contents region). pass -1 to use current column, otherwise 0..GetColumnsCount() inclusive. column 0 is typically 0.0");

    m.def("set_column_offset",
        ImGui::SetColumnOffset,
        nb::arg("column_index"), nb::arg("offset_x"),
        "set position of column line (in pixels, from the left side of the contents region). pass -1 to use current column");

    m.def("get_columns_count",
        ImGui::GetColumnsCount);

    m.def("begin_tab_bar",
        ImGui::BeginTabBar,
        nb::arg("str_id"), nb::arg("flags") = 0,
        "create and append into a TabBar");

    m.def("end_tab_bar",
        ImGui::EndTabBar, "only call EndTabBar() if BeginTabBar() returns True!");

    m.def("begin_tab_item",
        [](const char * label, std::optional<bool> p_open = std::nullopt, ImGuiTabItemFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
        {
            auto BeginTabItem_adapt_modifiable_immutable_to_return = [](const char * label, std::optional<bool> p_open = std::nullopt, ImGuiTabItemFlags flags = 0) -> std::tuple<bool, std::optional<bool>>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                bool r = ImGui::BeginTabItem(label, p_open_adapt_modifiable, flags);
                return std::make_tuple(r, p_open);
            };

            return BeginTabItem_adapt_modifiable_immutable_to_return(label, p_open, flags);
        },
        nb::arg("label"), nb::arg("p_open").none() = nb::none(), nb::arg("flags") = 0,
        "create a Tab. Returns True if the Tab is selected.");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("begin_tab_item_simple",
        ImGui::BeginTabItemSimple,
        nb::arg("label"), nb::arg("flags") = 0,
        "create a Tab (non-closable). Returns True if the Tab is selected.");
    // #endif
    //

    m.def("end_tab_item",
        ImGui::EndTabItem, "only call EndTabItem() if BeginTabItem() returns True!");

    m.def("tab_item_button",
        ImGui::TabItemButton,
        nb::arg("label"), nb::arg("flags") = 0,
        "create a Tab behaving like a button. return True when clicked. cannot be selected in the tab bar.");

    m.def("set_tab_item_closed",
        ImGui::SetTabItemClosed,
        nb::arg("tab_or_docked_window_label"),
        "notify TabBar or Docking system of a closed tab/window ahead (useful to reduce visual flicker on reorderable tab bars). For tab-bar: call after BeginTabBar() and before Tab submissions. Otherwise call with a window name.");

    m.def("dock_space",
        [](ImGuiID dockspace_id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiDockNodeFlags flags = 0, const ImGuiWindowClass * window_class = NULL) -> ImGuiID
        {
            auto DockSpace_adapt_mutable_param_with_default_value = [](ImGuiID dockspace_id, const std::optional<const ImVec2> & size = std::nullopt, ImGuiDockNodeFlags flags = 0, const ImGuiWindowClass * window_class = NULL) -> ImGuiID
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::DockSpace(dockspace_id, size_or_default, flags, window_class);
                return lambda_result;
            };

            return DockSpace_adapt_mutable_param_with_default_value(dockspace_id, size, flags, window_class);
        },
        nb::arg("dockspace_id"), nb::arg("size").none() = nb::none(), nb::arg("flags") = 0, nb::arg("window_class") = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("dock_space_over_viewport",
        ImGui::DockSpaceOverViewport, nb::arg("dockspace_id") = 0, nb::arg("viewport") = nb::none(), nb::arg("flags") = 0, nb::arg("window_class") = nb::none());

    m.def("set_next_window_dock_id",
        ImGui::SetNextWindowDockID,
        nb::arg("dock_id"), nb::arg("cond") = 0,
        "set next window dock id");

    m.def("set_next_window_class",
        ImGui::SetNextWindowClass,
        nb::arg("window_class"),
        "set next window class (control docking compatibility + provide hints to platform backend via custom viewport flags and platform parent/child relationship)");

    m.def("get_window_dock_id",
        ImGui::GetWindowDockID);

    m.def("is_window_docked",
        ImGui::IsWindowDocked, "is current window docked into another window?");

    m.def("log_to_tty",
        ImGui::LogToTTY,
        nb::arg("auto_open_depth") = -1,
        "start logging to tty (stdout)");

    m.def("log_to_file",
        [](int auto_open_depth = -1, std::optional<std::string> filename = std::nullopt)
        {
            auto LogToFile_adapt_const_char_pointer_with_default_null = [](int auto_open_depth = -1, std::optional<std::string> filename = std::nullopt)
            {
                const char * filename_adapt_default_null = nullptr;
                if (filename.has_value())
                    filename_adapt_default_null = filename.value().c_str();

                ImGui::LogToFile(auto_open_depth, filename_adapt_default_null);
            };

            LogToFile_adapt_const_char_pointer_with_default_null(auto_open_depth, filename);
        },
        nb::arg("auto_open_depth") = -1, nb::arg("filename").none() = nb::none(),
        "start logging to file");

    m.def("log_to_clipboard",
        ImGui::LogToClipboard,
        nb::arg("auto_open_depth") = -1,
        "start logging to OS clipboard");

    m.def("log_finish",
        ImGui::LogFinish, "stop logging (close file, etc.)");

    m.def("log_buttons",
        ImGui::LogButtons, "helper to display buttons for logging to tty/file/clipboard");

    m.def("log_text",
        [](const char * fmt)
        {
            auto LogText_adapt_variadic_format = [](const char * fmt)
            {
                ImGui::LogText("%s", fmt);
            };

            LogText_adapt_variadic_format(fmt);
        },
        nb::arg("fmt"),
        "pass text data straight to log (without being displayed)");

    m.def("begin_drag_drop_source",
        ImGui::BeginDragDropSource,
        nb::arg("flags") = 0,
        "call after submitting an item which may be dragged. when this return True, you can call SetDragDropPayload() + EndDragDropSource()");

    m.def("set_drag_drop_payload",
        ImGui::SetDragDropPayload,
        nb::arg("type"), nb::arg("data"), nb::arg("sz"), nb::arg("cond") = 0,
        "type is a user defined string of maximum 32 characters. Strings starting with '_' are reserved for dear imgui internal types. Data is copied and held by imgui. Return True when payload has been accepted.");

    m.def("end_drag_drop_source",
        ImGui::EndDragDropSource, "only call EndDragDropSource() if BeginDragDropSource() returns True!");

    m.def("begin_drag_drop_target",
        ImGui::BeginDragDropTarget, "call after submitting an item that may receive a payload. If this returns True, you can call AcceptDragDropPayload() + EndDragDropTarget()");

    m.def("end_drag_drop_target",
        ImGui::EndDragDropTarget, "only call EndDragDropTarget() if BeginDragDropTarget() returns True!");

    m.def("begin_disabled",
        ImGui::BeginDisabled, nb::arg("disabled") = true);

    m.def("end_disabled",
        ImGui::EndDisabled);

    m.def("push_clip_rect",
        ImGui::PushClipRect, nb::arg("clip_rect_min"), nb::arg("clip_rect_max"), nb::arg("intersect_with_current_clip_rect"));

    m.def("pop_clip_rect",
        ImGui::PopClipRect);

    m.def("set_item_default_focus",
        ImGui::SetItemDefaultFocus, "make last item the default focused item of a newly appearing window.");

    m.def("set_keyboard_focus_here",
        ImGui::SetKeyboardFocusHere,
        nb::arg("offset") = 0,
        "focus keyboard on the next widget. Use positive 'offset' to access sub components of a multiple component widget. Use -1 to access previous widget.");

    m.def("set_nav_cursor_visible",
        ImGui::SetNavCursorVisible,
        nb::arg("visible"),
        "alter visibility of keyboard/gamepad cursor. by default: show when using an arrow key, hide when clicking with mouse.");

    m.def("set_next_item_allow_overlap",
        ImGui::SetNextItemAllowOverlap, "allow next item to be overlapped by a subsequent item. Useful with invisible buttons, selectable, treenode covering an area where subsequent items may need to be added. Note that both Selectable() and TreeNode() have dedicated flags doing this.");

    m.def("is_item_hovered",
        ImGui::IsItemHovered,
        nb::arg("flags") = 0,
        "is the last item hovered? (and usable, aka not blocked by a popup, etc.). See ImGuiHoveredFlags for more options.");

    m.def("is_item_active",
        ImGui::IsItemActive, "is the last item active? (e.g. button being held, text field being edited. This will continuously return True while holding mouse button on an item. Items that don't interact will always return False)");

    m.def("is_item_focused",
        ImGui::IsItemFocused, "is the last item focused for keyboard/gamepad navigation?");

    m.def("is_item_clicked",
        ImGui::IsItemClicked,
        nb::arg("mouse_button") = 0,
        "is the last item hovered and mouse clicked on? (**)  == IsMouseClicked(mouse_button) && IsItemHovered()Important. (**) this is NOT equivalent to the behavior of e.g. Button(). Read comments in function definition.");

    m.def("is_item_visible",
        ImGui::IsItemVisible, "is the last item visible? (items may be out of sight because of clipping/scrolling)");

    m.def("is_item_edited",
        ImGui::IsItemEdited, "did the last item modify its underlying value this frame? or was pressed? This is generally the same as the \"bool\" return value of many widgets.");

    m.def("is_item_activated",
        ImGui::IsItemActivated, "was the last item just made active (item was previously inactive).");

    m.def("is_item_deactivated",
        ImGui::IsItemDeactivated, "was the last item just made inactive (item was previously active). Useful for Undo/Redo patterns with widgets that require continuous editing.");

    m.def("is_item_deactivated_after_edit",
        ImGui::IsItemDeactivatedAfterEdit, "was the last item just made inactive and made a value change when it was active? (e.g. Slider/Drag moved). Useful for Undo/Redo patterns with widgets that require continuous editing. Note that you may get False positives (some widgets such as Combo()/ListBox()/Selectable() will return True even when clicking an already selected item).");

    m.def("is_item_toggled_open",
        ImGui::IsItemToggledOpen, "was the last item open state toggled? set by TreeNode().");

    m.def("is_any_item_hovered",
        ImGui::IsAnyItemHovered, "is any item hovered?");

    m.def("is_any_item_active",
        ImGui::IsAnyItemActive, "is any item active?");

    m.def("is_any_item_focused",
        ImGui::IsAnyItemFocused, "is any item focused?");

    m.def("get_item_id",
        ImGui::GetItemID, "get ID of last item (~~ often same ImGui::GetID(label) beforehand)");

    m.def("get_item_rect_min",
        ImGui::GetItemRectMin, "get upper-left bounding rectangle of the last item (screen space)");

    m.def("get_item_rect_max",
        ImGui::GetItemRectMax, "get lower-right bounding rectangle of the last item (screen space)");

    m.def("get_item_rect_size",
        ImGui::GetItemRectSize, "get size of last item");

    m.def("get_main_viewport",
        ImGui::GetMainViewport,
        "return primary/default viewport. This can never be None.",
        nb::rv_policy::reference);

    m.def("get_background_draw_list",
        nb::overload_cast<ImGuiViewport *>(ImGui::GetBackgroundDrawList),
        nb::arg("viewport") = nb::none(),
        "get background draw list for the given viewport or viewport associated to the current window. this draw list will be the first rendering one. Useful to quickly draw shapes/text behind dear imgui contents.",
        nb::rv_policy::reference);

    m.def("get_foreground_draw_list",
        nb::overload_cast<ImGuiViewport *>(ImGui::GetForegroundDrawList),
        nb::arg("viewport") = nb::none(),
        "get foreground draw list for the given viewport or viewport associated to the current window. this draw list will be the top-most rendered one. Useful to quickly draw shapes/text over dear imgui contents.",
        nb::rv_policy::reference);

    m.def("is_rect_visible",
        nb::overload_cast<const ImVec2 &>(ImGui::IsRectVisible),
        nb::arg("size"),
        "test if rectangle (of given size, starting from cursor position) is visible / not clipped.");

    m.def("is_rect_visible",
        nb::overload_cast<const ImVec2 &, const ImVec2 &>(ImGui::IsRectVisible),
        nb::arg("rect_min"), nb::arg("rect_max"),
        "test if rectangle (in screen space) is visible / not clipped. to perform coarse clipping on user's side.");

    m.def("get_time",
        ImGui::GetTime, "get global imgui time. incremented by io.DeltaTime every frame.");

    m.def("get_frame_count",
        ImGui::GetFrameCount, "get global imgui frame count. incremented by 1 every frame.");

    m.def("get_draw_list_shared_data",
        nb::overload_cast<>(ImGui::GetDrawListSharedData),
        "you may use this when creating your own ImDrawList instances.",
        nb::rv_policy::reference);

    m.def("get_style_color_name",
        ImGui::GetStyleColorName,
        nb::arg("idx"),
        "get a string corresponding to the enum value (for display, saving, etc.).",
        nb::rv_policy::reference);

    m.def("set_state_storage",
        ImGui::SetStateStorage,
        nb::arg("storage"),
        "replace current window storage with our own (if you want to manipulate it yourself, typically clear subsection of it)");

    m.def("get_state_storage",
        ImGui::GetStateStorage, nb::rv_policy::reference);

    m.def("calc_text_size",
        [](const char * text, std::optional<std::string> text_end = std::nullopt, bool hide_text_after_double_hash = false, float wrap_width = -1.0f) -> ImVec2
        {
            auto CalcTextSize_adapt_const_char_pointer_with_default_null = [](const char * text, std::optional<std::string> text_end = std::nullopt, bool hide_text_after_double_hash = false, float wrap_width = -1.0f) -> ImVec2
            {
                const char * text_end_adapt_default_null = nullptr;
                if (text_end.has_value())
                    text_end_adapt_default_null = text_end.value().c_str();

                auto lambda_result = ImGui::CalcTextSize(text, text_end_adapt_default_null, hide_text_after_double_hash, wrap_width);
                return lambda_result;
            };

            return CalcTextSize_adapt_const_char_pointer_with_default_null(text, text_end, hide_text_after_double_hash, wrap_width);
        },
        nb::arg("text"), nb::arg("text_end").none() = nb::none(), nb::arg("hide_text_after_double_hash") = false, nb::arg("wrap_width") = -1.0f,
        "Text Utilities");

    m.def("color_convert_u32_to_float4",
        ImGui::ColorConvertU32ToFloat4, nb::arg("in_"));

    m.def("color_convert_float4_to_u32",
        ImGui::ColorConvertFloat4ToU32, nb::arg("in_"));

    m.def("color_convert_rgb_to_hsv",
        [](float r, float g, float b, float out_h, float out_s, float out_v) -> std::tuple<float, float, float>
        {
            auto ColorConvertRGBtoHSV_adapt_modifiable_immutable_to_return = [](float r, float g, float b, float out_h, float out_s, float out_v) -> std::tuple<float, float, float>
            {
                float & out_h_adapt_modifiable = out_h;
                float & out_s_adapt_modifiable = out_s;
                float & out_v_adapt_modifiable = out_v;

                ImGui::ColorConvertRGBtoHSV(r, g, b, out_h_adapt_modifiable, out_s_adapt_modifiable, out_v_adapt_modifiable);
                return std::make_tuple(out_h, out_s, out_v);
            };

            return ColorConvertRGBtoHSV_adapt_modifiable_immutable_to_return(r, g, b, out_h, out_s, out_v);
        },     nb::arg("r"), nb::arg("g"), nb::arg("b"), nb::arg("out_h"), nb::arg("out_s"), nb::arg("out_v"));

    m.def("color_convert_hsv_to_rgb",
        [](float h, float s, float v, float out_r, float out_g, float out_b) -> std::tuple<float, float, float>
        {
            auto ColorConvertHSVtoRGB_adapt_modifiable_immutable_to_return = [](float h, float s, float v, float out_r, float out_g, float out_b) -> std::tuple<float, float, float>
            {
                float & out_r_adapt_modifiable = out_r;
                float & out_g_adapt_modifiable = out_g;
                float & out_b_adapt_modifiable = out_b;

                ImGui::ColorConvertHSVtoRGB(h, s, v, out_r_adapt_modifiable, out_g_adapt_modifiable, out_b_adapt_modifiable);
                return std::make_tuple(out_r, out_g, out_b);
            };

            return ColorConvertHSVtoRGB_adapt_modifiable_immutable_to_return(h, s, v, out_r, out_g, out_b);
        },     nb::arg("h"), nb::arg("s"), nb::arg("v"), nb::arg("out_r"), nb::arg("out_g"), nb::arg("out_b"));

    m.def("is_key_down",
        nb::overload_cast<ImGuiKey>(ImGui::IsKeyDown),
        nb::arg("key"),
        "is key being held.");

    m.def("is_key_pressed",
        nb::overload_cast<ImGuiKey, bool>(ImGui::IsKeyPressed),
        nb::arg("key"), nb::arg("repeat") = true,
        "was key pressed (went from !Down to Down)? if repeat=True, uses io.KeyRepeatDelay / KeyRepeatRate");

    m.def("is_key_released",
        nb::overload_cast<ImGuiKey>(ImGui::IsKeyReleased),
        nb::arg("key"),
        "was key released (went from Down to !Down)?");

    m.def("is_key_chord_pressed",
        nb::overload_cast<ImGuiKeyChord>(ImGui::IsKeyChordPressed),
        nb::arg("key_chord"),
        "was key chord (mods + key) pressed, e.g. you can pass 'ImGuiMod_Ctrl | ImGuiKey_S' as a key-chord. This doesn't do any routing or focus check, please consider using Shortcut() function instead.");

    m.def("get_key_pressed_amount",
        ImGui::GetKeyPressedAmount,
        nb::arg("key"), nb::arg("repeat_delay"), nb::arg("rate"),
        "uses provided repeat rate/delay. return a count, most often 0 or 1 but might be >1 if RepeatRate is small enough that DeltaTime > RepeatRate");

    m.def("get_key_name",
        ImGui::GetKeyName,
        nb::arg("key"),
        "[DEBUG] returns English name of the key. Those names are provided for debugging purpose and are not meant to be saved persistently nor compared.",
        nb::rv_policy::reference);

    m.def("set_next_frame_want_capture_keyboard",
        ImGui::SetNextFrameWantCaptureKeyboard,
        nb::arg("want_capture_keyboard"),
        "Override io.WantCaptureKeyboard flag next frame (said flag is left for your application to handle, typically when True it instructs your app to ignore inputs). e.g. force capture keyboard when your widget is being hovered. This is equivalent to setting \"io.WantCaptureKeyboard = want_capture_keyboard\"; after the next NewFrame() call.");

    m.def("shortcut",
        nb::overload_cast<ImGuiKeyChord, ImGuiInputFlags>(ImGui::Shortcut), nb::arg("key_chord"), nb::arg("flags") = 0);

    m.def("set_next_item_shortcut",
        ImGui::SetNextItemShortcut, nb::arg("key_chord"), nb::arg("flags") = 0);

    m.def("set_item_key_owner",
        nb::overload_cast<ImGuiKey>(ImGui::SetItemKeyOwner),
        nb::arg("key"),
        "Set key owner to last item ID if it is hovered or active. Equivalent to 'if (IsItemHovered() || IsItemActive()) { SetKeyOwner(key, GetItemID());'.");

    m.def("is_mouse_down",
        nb::overload_cast<ImGuiMouseButton>(ImGui::IsMouseDown),
        nb::arg("button"),
        "is mouse button held?");

    m.def("is_mouse_clicked",
        nb::overload_cast<ImGuiMouseButton, bool>(ImGui::IsMouseClicked),
        nb::arg("button"), nb::arg("repeat") = false,
        "did mouse button clicked? (went from !Down to Down). Same as GetMouseClickedCount() == 1.");

    m.def("is_mouse_released",
        nb::overload_cast<ImGuiMouseButton>(ImGui::IsMouseReleased),
        nb::arg("button"),
        "did mouse button released? (went from Down to !Down)");

    m.def("is_mouse_double_clicked",
        nb::overload_cast<ImGuiMouseButton>(ImGui::IsMouseDoubleClicked),
        nb::arg("button"),
        "did mouse button double-clicked? Same as GetMouseClickedCount() == 2. (note that a double-click will also report IsMouseClicked() == True)");

    m.def("is_mouse_released_with_delay",
        nb::overload_cast<ImGuiMouseButton, float>(ImGui::IsMouseReleasedWithDelay),
        nb::arg("button"), nb::arg("delay"),
        "delayed mouse release (use very sparingly!). Generally used with 'delay >= io.MouseDoubleClickTime' + combined with a 'io.MouseClickedLastCount==1' test. This is a very rarely used UI idiom, but some apps use this: e.g. MS Explorer single click on an icon to rename.");

    m.def("get_mouse_clicked_count",
        ImGui::GetMouseClickedCount,
        nb::arg("button"),
        "return the number of successive mouse-clicks at the time where a click happen (otherwise 0).");

    m.def("is_mouse_hovering_rect",
        nb::overload_cast<const ImVec2 &, const ImVec2 &, bool>(ImGui::IsMouseHoveringRect),
        nb::arg("r_min"), nb::arg("r_max"), nb::arg("clip") = true,
        "is mouse hovering given bounding rect (in screen space). clipped by current clipping settings, but disregarding of other consideration of focus/window ordering/popup-block.");

    m.def("is_mouse_pos_valid",
        nb::overload_cast<const ImVec2 *>(ImGui::IsMousePosValid),
        nb::arg("mouse_pos") = nb::none(),
        "by convention we use (-FLT_MAX,-FLT_MAX) to denote that there is no mouse available");

    m.def("is_any_mouse_down",
        ImGui::IsAnyMouseDown, "[WILL OBSOLETE] is any mouse button held? This was designed for backends, but prefer having backend maintain a mask of held mouse buttons, because upcoming input queue system will make this invalid.");

    m.def("get_mouse_pos",
        ImGui::GetMousePos, "shortcut to ImGui::GetIO().MousePos provided by user, to be consistent with other calls");

    m.def("get_mouse_pos_on_opening_current_popup",
        nb::overload_cast<>(ImGui::GetMousePosOnOpeningCurrentPopup), "retrieve mouse position at the time of opening popup we have BeginPopup() into (helper to avoid user backing that value themselves)");

    m.def("is_mouse_dragging",
        nb::overload_cast<ImGuiMouseButton, float>(ImGui::IsMouseDragging),
        nb::arg("button"), nb::arg("lock_threshold") = -1.0f,
        "is mouse dragging? (uses io.MouseDraggingThreshold if lock_threshold < 0.0)");

    m.def("get_mouse_drag_delta",
        ImGui::GetMouseDragDelta,
        nb::arg("button") = 0, nb::arg("lock_threshold") = -1.0f,
        "return the delta from the initial clicking position while the mouse button is pressed or was just released. This is locked and return 0.0 until the mouse moves past a distance threshold at least once (uses io.MouseDraggingThreshold if lock_threshold < 0.0)");

    m.def("reset_mouse_drag_delta",
        ImGui::ResetMouseDragDelta, nb::arg("button") = 0);

    m.def("get_mouse_cursor",
        ImGui::GetMouseCursor, "get desired mouse cursor shape. Important: reset in ImGui::NewFrame(), this is updated during the frame. valid before Render(). If you use software rendering by setting io.MouseDrawCursor ImGui will render those for you");

    m.def("set_mouse_cursor",
        ImGui::SetMouseCursor,
        nb::arg("cursor_type"),
        "set desired mouse cursor shape");

    m.def("set_next_frame_want_capture_mouse",
        ImGui::SetNextFrameWantCaptureMouse,
        nb::arg("want_capture_mouse"),
        "Override io.WantCaptureMouse flag next frame (said flag is left for your application to handle, typical when True it instructs your app to ignore inputs). This is equivalent to setting \"io.WantCaptureMouse = want_capture_mouse;\" after the next NewFrame() call.");

    m.def("get_clipboard_text",
        ImGui::GetClipboardText, nb::rv_policy::reference);

    m.def("set_clipboard_text",
        ImGui::SetClipboardText, nb::arg("text"));

    m.def("load_ini_settings_from_disk",
        ImGui::LoadIniSettingsFromDisk,
        nb::arg("ini_filename"),
        "call after CreateContext() and before the first call to NewFrame(). NewFrame() automatically calls LoadIniSettingsFromDisk(io.IniFilename).");

    m.def("load_ini_settings_from_memory",
        ImGui::LoadIniSettingsFromMemory,
        nb::arg("ini_data"), nb::arg("ini_size") = 0,
        "call after CreateContext() and before the first call to NewFrame() to provide .ini data from your own data source.");

    m.def("save_ini_settings_to_disk",
        ImGui::SaveIniSettingsToDisk,
        nb::arg("ini_filename"),
        "this is automatically called (if io.IniFilename is not empty) a few seconds after any modification that should be reflected in the .ini file (and also by DestroyContext).");

    m.def("save_ini_settings_to_memory",
        []() -> const char *
        {
            auto SaveIniSettingsToMemory_adapt_exclude_params = []() -> const char *
            {
                auto lambda_result = ImGui::SaveIniSettingsToMemory(NULL);
                return lambda_result;
            };

            return SaveIniSettingsToMemory_adapt_exclude_params();
        },
        "return a zero-terminated string with the .ini data which you can save by your own mean. call when io.WantSaveIniSettings is set, then save data by your own mean and clear io.WantSaveIniSettings.",
        nb::rv_policy::reference);

    m.def("debug_text_encoding",
        ImGui::DebugTextEncoding, nb::arg("text"));

    m.def("debug_flash_style_color",
        ImGui::DebugFlashStyleColor, nb::arg("idx"));

    m.def("debug_start_item_picker",
        ImGui::DebugStartItemPicker);

    m.def("debug_check_version_and_data_layout",
        ImGui::DebugCheckVersionAndDataLayout,
        nb::arg("version_str"), nb::arg("sz_io"), nb::arg("sz_style"), nb::arg("sz_vec2"), nb::arg("sz_vec4"), nb::arg("sz_drawvert"), nb::arg("sz_drawidx"),
        "This is called by IMGUI_CHECKVERSION() macro.");

    m.def("update_platform_windows",
        ImGui::UpdatePlatformWindows, "call in main loop. will call CreateWindow/ResizeWindow/etc. platform functions for each secondary viewport, and DestroyWindow for each inactive viewport.");

    m.def("render_platform_windows_default",
        ImGui::RenderPlatformWindowsDefault,
        nb::arg("platform_render_arg") = nb::none(), nb::arg("renderer_render_arg") = nb::none(),
        "call in main loop. will call RenderWindow/SwapBuffers platform functions for each secondary viewport which doesn't have the ImGuiViewportFlags_Minimized flag set. May be reimplemented by user for custom rendering needs.");

    m.def("destroy_platform_windows",
        ImGui::DestroyPlatformWindows, "call DestroyWindow platform functions for all viewports. call from backend Shutdown() if you need to close platform windows before imgui shutdown. otherwise will be called by DestroyContext().");

    m.def("find_viewport_by_id",
        ImGui::FindViewportByID,
        nb::arg("id_"),
        "this is a helper for backends.",
        nb::rv_policy::reference);

    m.def("find_viewport_by_platform_handle",
        ImGui::FindViewportByPlatformHandle,
        nb::arg("platform_handle"),
        "this is a helper for backends. the type platform_handle is decided by the backend (e.g. HWND, MyWindow*, GLFWwindow* etc.)",
        nb::rv_policy::reference);


    auto pyEnumWindowFlags_ =
        nb::enum_<ImGuiWindowFlags_>(m, "WindowFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::Begin()\n (Those are per-window flags. There are shared flags in ImGuiIO: io.ConfigWindowsResizeFromEdges and io.ConfigWindowsMoveFromTitleBarOnly)")
            .value("none", ImGuiWindowFlags_None, "")
            .value("no_title_bar", ImGuiWindowFlags_NoTitleBar, "Disable title-bar")
            .value("no_resize", ImGuiWindowFlags_NoResize, "Disable user resizing with the lower-right grip")
            .value("no_move", ImGuiWindowFlags_NoMove, "Disable user moving the window")
            .value("no_scrollbar", ImGuiWindowFlags_NoScrollbar, "Disable scrollbars (window can still scroll with mouse or programmatically)")
            .value("no_scroll_with_mouse", ImGuiWindowFlags_NoScrollWithMouse, "Disable user vertically scrolling with mouse wheel. On child window, mouse wheel will be forwarded to the parent unless NoScrollbar is also set.")
            .value("no_collapse", ImGuiWindowFlags_NoCollapse, "Disable user collapsing window by double-clicking on it. Also referred to as Window Menu Button (e.g. within a docking node).")
            .value("always_auto_resize", ImGuiWindowFlags_AlwaysAutoResize, "Resize every window to its content every frame")
            .value("no_background", ImGuiWindowFlags_NoBackground, "Disable drawing background color (WindowBg, etc.) and outside border. Similar as using SetNextWindowBgAlpha(0.0).")
            .value("no_saved_settings", ImGuiWindowFlags_NoSavedSettings, "Never load/save settings in .ini file")
            .value("no_mouse_inputs", ImGuiWindowFlags_NoMouseInputs, "Disable catching mouse, hovering test with pass through.")
            .value("menu_bar", ImGuiWindowFlags_MenuBar, "Has a menu-bar")
            .value("horizontal_scrollbar", ImGuiWindowFlags_HorizontalScrollbar, "Allow horizontal scrollbar to appear (off by default). You may use SetNextWindowContentSize(ImVec2(width,0.0)); prior to calling Begin() to specify width. Read code in imgui_demo in the \"Horizontal Scrolling\" section.")
            .value("no_focus_on_appearing", ImGuiWindowFlags_NoFocusOnAppearing, "Disable taking focus when transitioning from hidden to visible state")
            .value("no_bring_to_front_on_focus", ImGuiWindowFlags_NoBringToFrontOnFocus, "Disable bringing window to front when taking focus (e.g. clicking on it or programmatically giving it focus)")
            .value("always_vertical_scrollbar", ImGuiWindowFlags_AlwaysVerticalScrollbar, "Always show vertical scrollbar (even if ContentSize.y < Size.y)")
            .value("always_horizontal_scrollbar", ImGuiWindowFlags_AlwaysHorizontalScrollbar, "Always show horizontal scrollbar (even if ContentSize.x < Size.x)")
            .value("no_nav_inputs", ImGuiWindowFlags_NoNavInputs, "No keyboard/gamepad navigation within the window")
            .value("no_nav_focus", ImGuiWindowFlags_NoNavFocus, "No focusing toward this window with keyboard/gamepad navigation (e.g. skipped by CTRL+TAB)")
            .value("unsaved_document", ImGuiWindowFlags_UnsavedDocument, "Display a dot next to the title. When used in a tab/docking context, tab is selected when clicking the X + closure is not assumed (will wait for user to stop submitting the tab). Otherwise closure is assumed when pressing the X, so if you keep submitting the tab may reappear at end of tab bar.")
            .value("no_docking", ImGuiWindowFlags_NoDocking, "Disable docking of this window")
            .value("no_nav", ImGuiWindowFlags_NoNav, "")
            .value("no_decoration", ImGuiWindowFlags_NoDecoration, "")
            .value("no_inputs", ImGuiWindowFlags_NoInputs, "")
            .value("dock_node_host", ImGuiWindowFlags_DockNodeHost, "Don't use! For internal use by Begin()/NewFrame()")
            .value("child_window", ImGuiWindowFlags_ChildWindow, "Don't use! For internal use by BeginChild()")
            .value("tooltip", ImGuiWindowFlags_Tooltip, "Don't use! For internal use by BeginTooltip()")
            .value("popup", ImGuiWindowFlags_Popup, "Don't use! For internal use by BeginPopup()")
            .value("modal", ImGuiWindowFlags_Modal, "Don't use! For internal use by BeginPopupModal()")
            .value("child_menu", ImGuiWindowFlags_ChildMenu, "Don't use! For internal use by BeginMenu()");


    auto pyEnumChildFlags_ =
        nb::enum_<ImGuiChildFlags_>(m, "ChildFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::BeginChild()\n (Legacy: bit 0 must always correspond to ImGuiChildFlags_Borders to be backward compatible with old API using 'bool border = False'.)\n About using AutoResizeX/AutoResizeY flags:\n - May be combined with SetNextWindowSizeConstraints() to set a min/max size for each axis (see \"Demo->Child->Auto-resize with Constraints\").\n - Size measurement for a given axis is only performed when the child window is within visible boundaries, or is just appearing.\n   - This allows BeginChild() to return False when not within boundaries (e.g. when scrolling), which is more optimal. BUT it won't update its auto-size while clipped.\n     While not perfect, it is a better default behavior as the always-on performance gain is more valuable than the occasional \"resizing after becoming visible again\" glitch.\n   - You may also use ImGuiChildFlags_AlwaysAutoResize to force an update even when child window is not in view.\n     HOWEVER PLEASE UNDERSTAND THAT DOING SO WILL PREVENT BeginChild() FROM EVER RETURNING FALSE, disabling benefits of coarse clipping.")
            .value("none", ImGuiChildFlags_None, "")
            .value("borders", ImGuiChildFlags_Borders, "Show an outer border and enable WindowPadding. (IMPORTANT: this is always == 1 == True for legacy reason)")
            .value("always_use_window_padding", ImGuiChildFlags_AlwaysUseWindowPadding, "Pad with style.WindowPadding even if no border are drawn (no padding by default for non-bordered child windows because it makes more sense)")
            .value("resize_x", ImGuiChildFlags_ResizeX, "Allow resize from right border (layout direction). Enable .ini saving (unless ImGuiWindowFlags_NoSavedSettings passed to window flags)")
            .value("resize_y", ImGuiChildFlags_ResizeY, "Allow resize from bottom border (layout direction). \"")
            .value("auto_resize_x", ImGuiChildFlags_AutoResizeX, "Enable auto-resizing width. Read \"IMPORTANT: Size measurement\" details above.")
            .value("auto_resize_y", ImGuiChildFlags_AutoResizeY, "Enable auto-resizing height. Read \"IMPORTANT: Size measurement\" details above.")
            .value("always_auto_resize", ImGuiChildFlags_AlwaysAutoResize, "Combined with AutoResizeX/AutoResizeY. Always measure size even when child is hidden, always return True, always disable clipping optimization! NOT RECOMMENDED.")
            .value("frame_style", ImGuiChildFlags_FrameStyle, "Style the child window like a framed item: use FrameBg, FrameRounding, FrameBorderSize, FramePadding instead of ChildBg, ChildRounding, ChildBorderSize, WindowPadding.")
            .value("nav_flattened", ImGuiChildFlags_NavFlattened, "[BETA] Share focus scope, allow keyboard/gamepad navigation to cross over parent border to this child or between sibling child windows.");


    auto pyEnumItemFlags_ =
        nb::enum_<ImGuiItemFlags_>(m, "ItemFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::PushItemFlag()\n (Those are shared by all items)")
            .value("none", ImGuiItemFlags_None, "(Default)")
            .value("no_tab_stop", ImGuiItemFlags_NoTabStop, "False    // Disable keyboard tabbing. This is a \"lighter\" version of ImGuiItemFlags_NoNav.")
            .value("no_nav", ImGuiItemFlags_NoNav, "False    // Disable any form of focusing (keyboard/gamepad directional navigation and SetKeyboardFocusHere() calls).")
            .value("no_nav_default_focus", ImGuiItemFlags_NoNavDefaultFocus, "False    // Disable item being a candidate for default focus (e.g. used by title bar items).")
            .value("button_repeat", ImGuiItemFlags_ButtonRepeat, "False    // Any button-like behavior will have repeat mode enabled (based on io.KeyRepeatDelay and io.KeyRepeatRate values). Note that you can also call IsItemActive() after any button to tell if it is being held.")
            .value("auto_close_popups", ImGuiItemFlags_AutoClosePopups, "True     // MenuItem()/Selectable() automatically close their parent popup window.")
            .value("allow_duplicate_id", ImGuiItemFlags_AllowDuplicateId, "False    // Allow submitting an item with the same identifier as an item already submitted this frame without triggering a warning tooltip if io.ConfigDebugHighlightIdConflicts is set.");


    auto pyEnumInputTextFlags_ =
        nb::enum_<ImGuiInputTextFlags_>(m, "InputTextFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::InputText()\n (Those are per-item flags. There are shared flags in ImGuiIO: io.ConfigInputTextCursorBlink and io.ConfigInputTextEnterKeepActive)")
            .value("none", ImGuiInputTextFlags_None, "")
            .value("chars_decimal", ImGuiInputTextFlags_CharsDecimal, "Allow 0123456789.+-*/")
            .value("chars_hexadecimal", ImGuiInputTextFlags_CharsHexadecimal, "Allow 0123456789ABCDEFabcdef")
            .value("chars_scientific", ImGuiInputTextFlags_CharsScientific, "Allow 0123456789.+-*/eE (Scientific notation input)")
            .value("chars_uppercase", ImGuiInputTextFlags_CharsUppercase, "Turn a..z into A..Z")
            .value("chars_no_blank", ImGuiInputTextFlags_CharsNoBlank, "Filter out spaces, tabs")
            .value("allow_tab_input", ImGuiInputTextFlags_AllowTabInput, "Pressing TAB input a '\t' character into the text field")
            .value("enter_returns_true", ImGuiInputTextFlags_EnterReturnsTrue, "Return 'True' when Enter is pressed (as opposed to every time the value was modified). Consider using IsItemDeactivatedAfterEdit() instead!")
            .value("escape_clears_all", ImGuiInputTextFlags_EscapeClearsAll, "Escape key clears content if not empty, and deactivate otherwise (contrast to default behavior of Escape to revert)")
            .value("ctrl_enter_for_new_line", ImGuiInputTextFlags_CtrlEnterForNewLine, "In multi-line mode, validate with Enter, add new line with Ctrl+Enter (default is opposite: validate with Ctrl+Enter, add line with Enter).")
            .value("read_only", ImGuiInputTextFlags_ReadOnly, "Read-only mode")
            .value("password", ImGuiInputTextFlags_Password, "Password mode, display all characters as '*', disable copy")
            .value("always_overwrite", ImGuiInputTextFlags_AlwaysOverwrite, "Overwrite mode")
            .value("auto_select_all", ImGuiInputTextFlags_AutoSelectAll, "Select entire text when first taking mouse focus")
            .value("parse_empty_ref_val", ImGuiInputTextFlags_ParseEmptyRefVal, "InputFloat(), InputInt(), InputScalar() etc. only: parse empty string as zero value.")
            .value("display_empty_ref_val", ImGuiInputTextFlags_DisplayEmptyRefVal, "InputFloat(), InputInt(), InputScalar() etc. only: when value is zero, do not display it. Generally used with ImGuiInputTextFlags_ParseEmptyRefVal.")
            .value("no_horizontal_scroll", ImGuiInputTextFlags_NoHorizontalScroll, "Disable following the cursor horizontally")
            .value("no_undo_redo", ImGuiInputTextFlags_NoUndoRedo, "Disable undo/redo. Note that input text owns the text data while active, if you want to provide your own undo/redo stack you need e.g. to call ClearActiveID().")
            .value("elide_left", ImGuiInputTextFlags_ElideLeft, "When text doesn't fit, elide left side to ensure right side stays visible. Useful for path/filenames. Single-line only!")
            .value("callback_completion", ImGuiInputTextFlags_CallbackCompletion, "Callback on pressing TAB (for completion handling)")
            .value("callback_history", ImGuiInputTextFlags_CallbackHistory, "Callback on pressing Up/Down arrows (for history handling)")
            .value("callback_always", ImGuiInputTextFlags_CallbackAlways, "Callback on each iteration. User code may query cursor position, modify text buffer.")
            .value("callback_char_filter", ImGuiInputTextFlags_CallbackCharFilter, "Callback on character inputs to replace or discard them. Modify 'EventChar' to replace or discard, or return 1 in callback to discard.")
            .value("callback_resize", ImGuiInputTextFlags_CallbackResize, "Callback on buffer capacity changes request (beyond 'buf_size' parameter value), allowing the string to grow. Notify when the string wants to be resized (for string types which hold a cache of their Size). You will be provided a new BufSize in the callback and NEED to honor it. (see misc/cpp/imgui_stdlib.h for an example of using this)")
            .value("callback_edit", ImGuiInputTextFlags_CallbackEdit, "Callback on any edit. Note that InputText() already returns True on edit + you can always use IsItemEdited(). The callback is useful to manipulate the underlying buffer while focus is active.")
            .value("word_wrap", ImGuiInputTextFlags_WordWrap, "InputTextMultine(): word-wrap lines that are too long.");


    auto pyEnumTreeNodeFlags_ =
        nb::enum_<ImGuiTreeNodeFlags_>(m, "TreeNodeFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::TreeNodeEx(), ImGui::CollapsingHeader*()")
            .value("none", ImGuiTreeNodeFlags_None, "")
            .value("selected", ImGuiTreeNodeFlags_Selected, "Draw as selected")
            .value("framed", ImGuiTreeNodeFlags_Framed, "Draw frame with background (e.g. for CollapsingHeader)")
            .value("allow_overlap", ImGuiTreeNodeFlags_AllowOverlap, "Hit testing to allow subsequent widgets to overlap this one")
            .value("no_tree_push_on_open", ImGuiTreeNodeFlags_NoTreePushOnOpen, "Don't do a TreePush() when open (e.g. for CollapsingHeader) = no extra indent nor pushing on ID stack")
            .value("no_auto_open_on_log", ImGuiTreeNodeFlags_NoAutoOpenOnLog, "Don't automatically and temporarily open node when Logging is active (by default logging will automatically open tree nodes)")
            .value("default_open", ImGuiTreeNodeFlags_DefaultOpen, "Default node to be open")
            .value("open_on_double_click", ImGuiTreeNodeFlags_OpenOnDoubleClick, "Open on double-click instead of simple click (default for multi-select unless any _OpenOnXXX behavior is set explicitly). Both behaviors may be combined.")
            .value("open_on_arrow", ImGuiTreeNodeFlags_OpenOnArrow, "Open when clicking on the arrow part (default for multi-select unless any _OpenOnXXX behavior is set explicitly). Both behaviors may be combined.")
            .value("leaf", ImGuiTreeNodeFlags_Leaf, "No collapsing, no arrow (use as a convenience for leaf nodes).")
            .value("bullet", ImGuiTreeNodeFlags_Bullet, "Display a bullet instead of arrow. IMPORTANT: node can still be marked open/close if you don't set the _Leaf flag!")
            .value("frame_padding", ImGuiTreeNodeFlags_FramePadding, "Use FramePadding (even for an unframed text node) to vertically align text baseline to regular widget height. Equivalent to calling AlignTextToFramePadding() before the node.")
            .value("span_avail_width", ImGuiTreeNodeFlags_SpanAvailWidth, "Extend hit box to the right-most edge, even if not framed. This is not the default in order to allow adding other items on the same line without using AllowOverlap mode.")
            .value("span_full_width", ImGuiTreeNodeFlags_SpanFullWidth, "Extend hit box to the left-most and right-most edges (cover the indent area).")
            .value("span_label_width", ImGuiTreeNodeFlags_SpanLabelWidth, "Narrow hit box + narrow hovering highlight, will only cover the label text.")
            .value("span_all_columns", ImGuiTreeNodeFlags_SpanAllColumns, "Frame will span all columns of its container table (label will still fit in current column)")
            .value("label_span_all_columns", ImGuiTreeNodeFlags_LabelSpanAllColumns, "Label will span all columns of its container table")
            .value("nav_left_jumps_to_parent", ImGuiTreeNodeFlags_NavLeftJumpsToParent, "Nav: left arrow moves back to parent. This is processed in TreePop() when there's an unfullfilled Left nav request remaining.")
            .value("collapsing_header", ImGuiTreeNodeFlags_CollapsingHeader, "")
            .value("draw_lines_none", ImGuiTreeNodeFlags_DrawLinesNone, "No lines drawn")
            .value("draw_lines_full", ImGuiTreeNodeFlags_DrawLinesFull, "Horizontal lines to child nodes. Vertical line drawn down to TreePop() position: cover full contents. Faster (for large trees).")
            .value("draw_lines_to_nodes", ImGuiTreeNodeFlags_DrawLinesToNodes, "Horizontal lines to child nodes. Vertical line drawn down to bottom-most child node. Slower (for large trees).");


    auto pyEnumPopupFlags_ =
        nb::enum_<ImGuiPopupFlags_>(m, "PopupFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for OpenPopup*(), BeginPopupContext*(), IsPopupOpen() functions.\n - To be backward compatible with older API which took an 'int mouse_button = 1' argument instead of 'ImGuiPopupFlags flags',\n   we need to treat small flags values as a mouse button index, so we encode the mouse button in the first few bits of the flags.\n   It is therefore guaranteed to be legal to pass a mouse button index in ImGuiPopupFlags.\n - For the same reason, we exceptionally default the ImGuiPopupFlags argument of BeginPopupContextXXX functions to 1 instead of 0.\n   IMPORTANT: because the default parameter is 1 (==ImGuiPopupFlags_MouseButtonRight), if you rely on the default parameter\n   and want to use another flag, you need to pass in the ImGuiPopupFlags_MouseButtonRight flag explicitly.\n - Multiple buttons currently cannot be combined/or-ed in those functions (we could allow it later).")
            .value("none", ImGuiPopupFlags_None, "")
            .value("mouse_button_left", ImGuiPopupFlags_MouseButtonLeft, "For BeginPopupContext*(): open on Left Mouse release. Guaranteed to always be == 0 (same as ImGuiMouseButton_Left)")
            .value("mouse_button_right", ImGuiPopupFlags_MouseButtonRight, "For BeginPopupContext*(): open on Right Mouse release. Guaranteed to always be == 1 (same as ImGuiMouseButton_Right)")
            .value("mouse_button_middle", ImGuiPopupFlags_MouseButtonMiddle, "For BeginPopupContext*(): open on Middle Mouse release. Guaranteed to always be == 2 (same as ImGuiMouseButton_Middle)")
            .value("mouse_button_mask_", ImGuiPopupFlags_MouseButtonMask_, "")
            .value("mouse_button_default_", ImGuiPopupFlags_MouseButtonDefault_, "")
            .value("no_reopen", ImGuiPopupFlags_NoReopen, "For OpenPopup*(), BeginPopupContext*(): don't reopen same popup if already open (won't reposition, won't reinitialize navigation)")
            .value("no_open_over_existing_popup", ImGuiPopupFlags_NoOpenOverExistingPopup, "For OpenPopup*(), BeginPopupContext*(): don't open if there's already a popup at the same level of the popup stack")
            .value("no_open_over_items", ImGuiPopupFlags_NoOpenOverItems, "For BeginPopupContextWindow(): don't return True when hovering items, only when hovering empty space")
            .value("any_popup_id", ImGuiPopupFlags_AnyPopupId, "For IsPopupOpen(): ignore the ImGuiID parameter and test for any popup.")
            .value("any_popup_level", ImGuiPopupFlags_AnyPopupLevel, "For IsPopupOpen(): search/test at any level of the popup stack (default test in the current level)")
            .value("any_popup", ImGuiPopupFlags_AnyPopup, "");


    auto pyEnumSelectableFlags_ =
        nb::enum_<ImGuiSelectableFlags_>(m, "SelectableFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::Selectable()")
            .value("none", ImGuiSelectableFlags_None, "")
            .value("no_auto_close_popups", ImGuiSelectableFlags_NoAutoClosePopups, "Clicking this doesn't close parent popup window (overrides ImGuiItemFlags_AutoClosePopups)")
            .value("span_all_columns", ImGuiSelectableFlags_SpanAllColumns, "Frame will span all columns of its container table (text will still fit in current column)")
            .value("allow_double_click", ImGuiSelectableFlags_AllowDoubleClick, "Generate press events on double clicks too")
            .value("disabled", ImGuiSelectableFlags_Disabled, "Cannot be selected, display grayed out text")
            .value("allow_overlap", ImGuiSelectableFlags_AllowOverlap, "(WIP) Hit testing to allow subsequent widgets to overlap this one")
            .value("highlight", ImGuiSelectableFlags_Highlight, "Make the item be displayed as if it is hovered")
            .value("select_on_nav", ImGuiSelectableFlags_SelectOnNav, "Auto-select when moved into, unless Ctrl is held. Automatic when in a BeginMultiSelect() block.");


    auto pyEnumComboFlags_ =
        nb::enum_<ImGuiComboFlags_>(m, "ComboFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::BeginCombo()")
            .value("none", ImGuiComboFlags_None, "")
            .value("popup_align_left", ImGuiComboFlags_PopupAlignLeft, "Align the popup toward the left by default")
            .value("height_small", ImGuiComboFlags_HeightSmall, "Max ~4 items visible. Tip: If you want your combo popup to be a specific size you can use SetNextWindowSizeConstraints() prior to calling BeginCombo()")
            .value("height_regular", ImGuiComboFlags_HeightRegular, "Max ~8 items visible (default)")
            .value("height_large", ImGuiComboFlags_HeightLarge, "Max ~20 items visible")
            .value("height_largest", ImGuiComboFlags_HeightLargest, "As many fitting items as possible")
            .value("no_arrow_button", ImGuiComboFlags_NoArrowButton, "Display on the preview box without the square arrow button")
            .value("no_preview", ImGuiComboFlags_NoPreview, "Display only a square arrow button")
            .value("width_fit_preview", ImGuiComboFlags_WidthFitPreview, "Width dynamically calculated from preview contents")
            .value("height_mask_", ImGuiComboFlags_HeightMask_, "");


    auto pyEnumTabBarFlags_ =
        nb::enum_<ImGuiTabBarFlags_>(m, "TabBarFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::BeginTabBar()")
            .value("none", ImGuiTabBarFlags_None, "")
            .value("reorderable", ImGuiTabBarFlags_Reorderable, "Allow manually dragging tabs to re-order them + New tabs are appended at the end of list")
            .value("auto_select_new_tabs", ImGuiTabBarFlags_AutoSelectNewTabs, "Automatically select new tabs when they appear")
            .value("tab_list_popup_button", ImGuiTabBarFlags_TabListPopupButton, "Disable buttons to open the tab list popup")
            .value("no_close_with_middle_mouse_button", ImGuiTabBarFlags_NoCloseWithMiddleMouseButton, "Disable behavior of closing tabs (that are submitted with p_open != None) with middle mouse button. You may handle this behavior manually on user's side with if (IsItemHovered() && IsMouseClicked(2)) *p_open = False.")
            .value("no_tab_list_scrolling_buttons", ImGuiTabBarFlags_NoTabListScrollingButtons, "Disable scrolling buttons (apply when fitting policy is ImGuiTabBarFlags_FittingPolicyScroll)")
            .value("no_tooltip", ImGuiTabBarFlags_NoTooltip, "Disable tooltips when hovering a tab")
            .value("draw_selected_overline", ImGuiTabBarFlags_DrawSelectedOverline, "Draw selected overline markers over selected tab")
            .value("fitting_policy_mixed", ImGuiTabBarFlags_FittingPolicyMixed, "Shrink down tabs when they don't fit, until width is style.TabMinWidthShrink, then enable scrolling buttons.")
            .value("fitting_policy_shrink", ImGuiTabBarFlags_FittingPolicyShrink, "Shrink down tabs when they don't fit")
            .value("fitting_policy_scroll", ImGuiTabBarFlags_FittingPolicyScroll, "Enable scrolling buttons when tabs don't fit")
            .value("fitting_policy_mask_", ImGuiTabBarFlags_FittingPolicyMask_, "")
            .value("fitting_policy_default_", ImGuiTabBarFlags_FittingPolicyDefault_, "");


    auto pyEnumTabItemFlags_ =
        nb::enum_<ImGuiTabItemFlags_>(m, "TabItemFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::BeginTabItem()")
            .value("none", ImGuiTabItemFlags_None, "")
            .value("unsaved_document", ImGuiTabItemFlags_UnsavedDocument, "Display a dot next to the title + set ImGuiTabItemFlags_NoAssumedClosure.")
            .value("set_selected", ImGuiTabItemFlags_SetSelected, "Trigger flag to programmatically make the tab selected when calling BeginTabItem()")
            .value("no_close_with_middle_mouse_button", ImGuiTabItemFlags_NoCloseWithMiddleMouseButton, "Disable behavior of closing tabs (that are submitted with p_open != None) with middle mouse button. You may handle this behavior manually on user's side with if (IsItemHovered() && IsMouseClicked(2)) *p_open = False.")
            .value("no_push_id", ImGuiTabItemFlags_NoPushId, "Don't call PushID()/PopID() on BeginTabItem()/EndTabItem()")
            .value("no_tooltip", ImGuiTabItemFlags_NoTooltip, "Disable tooltip for the given tab")
            .value("no_reorder", ImGuiTabItemFlags_NoReorder, "Disable reordering this tab or having another tab cross over this tab")
            .value("leading", ImGuiTabItemFlags_Leading, "Enforce the tab position to the left of the tab bar (after the tab list popup button)")
            .value("trailing", ImGuiTabItemFlags_Trailing, "Enforce the tab position to the right of the tab bar (before the scrolling buttons)")
            .value("no_assumed_closure", ImGuiTabItemFlags_NoAssumedClosure, "Tab is selected when trying to close + closure is not immediately assumed (will wait for user to stop submitting the tab). Otherwise closure is assumed when pressing the X, so if you keep submitting the tab may reappear at end of tab bar.");


    auto pyEnumFocusedFlags_ =
        nb::enum_<ImGuiFocusedFlags_>(m, "FocusedFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::IsWindowFocused()")
            .value("none", ImGuiFocusedFlags_None, "")
            .value("child_windows", ImGuiFocusedFlags_ChildWindows, "Return True if any children of the window is focused")
            .value("root_window", ImGuiFocusedFlags_RootWindow, "Test from root window (top most parent of the current hierarchy)")
            .value("any_window", ImGuiFocusedFlags_AnyWindow, "Return True if any window is focused. Important: If you are trying to tell how to dispatch your low-level inputs, do NOT use this. Use 'io.WantCaptureMouse' instead! Please read the FAQ!")
            .value("no_popup_hierarchy", ImGuiFocusedFlags_NoPopupHierarchy, "Do not consider popup hierarchy (do not treat popup emitter as parent of popup) (when used with _ChildWindows or _RootWindow)")
            .value("dock_hierarchy", ImGuiFocusedFlags_DockHierarchy, "Consider docking hierarchy (treat dockspace host as parent of docked window) (when used with _ChildWindows or _RootWindow)")
            .value("root_and_child_windows", ImGuiFocusedFlags_RootAndChildWindows, "");


    auto pyEnumHoveredFlags_ =
        nb::enum_<ImGuiHoveredFlags_>(m, "HoveredFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::IsItemHovered(), ImGui::IsWindowHovered()\n Note: if you are trying to check whether your mouse should be dispatched to Dear ImGui or to your app, you should use 'io.WantCaptureMouse' instead! Please read the FAQ!\n Note: windows with the ImGuiWindowFlags_NoInputs flag are ignored by IsWindowHovered() calls.")
            .value("none", ImGuiHoveredFlags_None, "Return True if directly over the item/window, not obstructed by another window, not obstructed by an active popup or modal blocking inputs under them.")
            .value("child_windows", ImGuiHoveredFlags_ChildWindows, "IsWindowHovered() only: Return True if any children of the window is hovered")
            .value("root_window", ImGuiHoveredFlags_RootWindow, "IsWindowHovered() only: Test from root window (top most parent of the current hierarchy)")
            .value("any_window", ImGuiHoveredFlags_AnyWindow, "IsWindowHovered() only: Return True if any window is hovered")
            .value("no_popup_hierarchy", ImGuiHoveredFlags_NoPopupHierarchy, "IsWindowHovered() only: Do not consider popup hierarchy (do not treat popup emitter as parent of popup) (when used with _ChildWindows or _RootWindow)")
            .value("dock_hierarchy", ImGuiHoveredFlags_DockHierarchy, "IsWindowHovered() only: Consider docking hierarchy (treat dockspace host as parent of docked window) (when used with _ChildWindows or _RootWindow)")
            .value("allow_when_blocked_by_popup", ImGuiHoveredFlags_AllowWhenBlockedByPopup, "Return True even if a popup window is normally blocking access to this item/window")
            .value("allow_when_blocked_by_active_item", ImGuiHoveredFlags_AllowWhenBlockedByActiveItem, "Return True even if an active item is blocking access to this item/window. Useful for Drag and Drop patterns.")
            .value("allow_when_overlapped_by_item", ImGuiHoveredFlags_AllowWhenOverlappedByItem, "IsItemHovered() only: Return True even if the item uses AllowOverlap mode and is overlapped by another hoverable item.")
            .value("allow_when_overlapped_by_window", ImGuiHoveredFlags_AllowWhenOverlappedByWindow, "IsItemHovered() only: Return True even if the position is obstructed or overlapped by another window.")
            .value("allow_when_disabled", ImGuiHoveredFlags_AllowWhenDisabled, "IsItemHovered() only: Return True even if the item is disabled")
            .value("no_nav_override", ImGuiHoveredFlags_NoNavOverride, "IsItemHovered() only: Disable using keyboard/gamepad navigation state when active, always query mouse")
            .value("allow_when_overlapped", ImGuiHoveredFlags_AllowWhenOverlapped, "")
            .value("rect_only", ImGuiHoveredFlags_RectOnly, "")
            .value("root_and_child_windows", ImGuiHoveredFlags_RootAndChildWindows, "")
            .value("for_tooltip", ImGuiHoveredFlags_ForTooltip, "Shortcut for standard flags when using IsItemHovered() + SetTooltip() sequence.")
            .value("stationary", ImGuiHoveredFlags_Stationary, "Require mouse to be stationary for style.HoverStationaryDelay (~0.15 sec) _at least one time_. After this, can move on same item/window. Using the stationary test tends to reduces the need for a long delay.")
            .value("delay_none", ImGuiHoveredFlags_DelayNone, "IsItemHovered() only: Return True immediately (default). As this is the default you generally ignore this.")
            .value("delay_short", ImGuiHoveredFlags_DelayShort, "IsItemHovered() only: Return True after style.HoverDelayShort elapsed (~0.15 sec) (shared between items) + requires mouse to be stationary for style.HoverStationaryDelay (once per item).")
            .value("delay_normal", ImGuiHoveredFlags_DelayNormal, "IsItemHovered() only: Return True after style.HoverDelayNormal elapsed (~0.40 sec) (shared between items) + requires mouse to be stationary for style.HoverStationaryDelay (once per item).")
            .value("no_shared_delay", ImGuiHoveredFlags_NoSharedDelay, "IsItemHovered() only: Disable shared delay system where moving from one item to the next keeps the previous timer for a short time (standard for tooltips with long delays)");


    auto pyEnumDockNodeFlags_ =
        nb::enum_<ImGuiDockNodeFlags_>(m, "DockNodeFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::DockSpace(), shared/inherited by child nodes.\n (Some flags can be applied to individual nodes directly)\n FIXME-DOCK: Also see ImGuiDockNodeFlagsPrivate_ which may involve using the WIP and internal DockBuilder api.")
            .value("none", ImGuiDockNodeFlags_None, "")
            .value("keep_alive_only", ImGuiDockNodeFlags_KeepAliveOnly, "// Don't display the dockspace node but keep it alive. Windows docked into this dockspace node won't be undocked.")
            .value("no_docking_over_central_node", ImGuiDockNodeFlags_NoDockingOverCentralNode, "// Disable docking over the Central Node, which will be always kept empty.")
            .value("passthru_central_node", ImGuiDockNodeFlags_PassthruCentralNode, "// Enable passthru dockspace: 1) DockSpace() will render a ImGuiCol_WindowBg background covering everything excepted the Central Node when empty. Meaning the host window should probably use SetNextWindowBgAlpha(0.0) prior to Begin() when using this. 2) When Central Node is empty: let inputs pass-through + won't display a DockingEmptyBg background. See demo for details.")
            .value("no_docking_split", ImGuiDockNodeFlags_NoDockingSplit, "// Disable other windows/nodes from splitting this node.")
            .value("no_resize", ImGuiDockNodeFlags_NoResize, "Saved // Disable resizing node using the splitter/separators. Useful with programmatically setup dockspaces.")
            .value("auto_hide_tab_bar", ImGuiDockNodeFlags_AutoHideTabBar, "// Tab bar will automatically hide when there is a single window in the dock node.")
            .value("no_undocking", ImGuiDockNodeFlags_NoUndocking, "// Disable undocking this node.");


    auto pyEnumDragDropFlags_ =
        nb::enum_<ImGuiDragDropFlags_>(m, "DragDropFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::BeginDragDropSource(), ImGui::AcceptDragDropPayload()")
            .value("none", ImGuiDragDropFlags_None, "")
            .value("source_no_preview_tooltip", ImGuiDragDropFlags_SourceNoPreviewTooltip, "Disable preview tooltip. By default, a successful call to BeginDragDropSource opens a tooltip so you can display a preview or description of the source contents. This flag disables this behavior.")
            .value("source_no_disable_hover", ImGuiDragDropFlags_SourceNoDisableHover, "By default, when dragging we clear data so that IsItemHovered() will return False, to avoid subsequent user code submitting tooltips. This flag disables this behavior so you can still call IsItemHovered() on the source item.")
            .value("source_no_hold_to_open_others", ImGuiDragDropFlags_SourceNoHoldToOpenOthers, "Disable the behavior that allows to open tree nodes and collapsing header by holding over them while dragging a source item.")
            .value("source_allow_null_id", ImGuiDragDropFlags_SourceAllowNullID, "Allow items such as Text(), Image() that have no unique identifier to be used as drag source, by manufacturing a temporary identifier based on their window-relative position. This is extremely unusual within the dear imgui ecosystem and so we made it explicit.")
            .value("source_extern", ImGuiDragDropFlags_SourceExtern, "External source (from outside of dear imgui), won't attempt to read current item/window info. Will always return True. Only one Extern source can be active simultaneously.")
            .value("payload_auto_expire", ImGuiDragDropFlags_PayloadAutoExpire, "Automatically expire the payload if the source cease to be submitted (otherwise payloads are persisting while being dragged)")
            .value("payload_no_cross_context", ImGuiDragDropFlags_PayloadNoCrossContext, "Hint to specify that the payload may not be copied outside current dear imgui context.")
            .value("payload_no_cross_process", ImGuiDragDropFlags_PayloadNoCrossProcess, "Hint to specify that the payload may not be copied outside current process.")
            .value("accept_before_delivery", ImGuiDragDropFlags_AcceptBeforeDelivery, "AcceptDragDropPayload() will returns True even before the mouse button is released. You can then call IsDelivery() to test if the payload needs to be delivered.")
            .value("accept_no_draw_default_rect", ImGuiDragDropFlags_AcceptNoDrawDefaultRect, "Do not draw the default highlight rectangle when hovering over target.")
            .value("accept_no_preview_tooltip", ImGuiDragDropFlags_AcceptNoPreviewTooltip, "Request hiding the BeginDragDropSource tooltip from the BeginDragDropTarget site.")
            .value("accept_peek_only", ImGuiDragDropFlags_AcceptPeekOnly, "For peeking ahead and inspecting the payload before delivery.");


    auto pyEnumDataType_ =
        nb::enum_<ImGuiDataType_>(m, "DataType_", nb::is_arithmetic(), nb::is_flag(), "A primary data type")
            .value("s8", ImGuiDataType_S8, "signed char / char (with sensible compilers)")
            .value("u8", ImGuiDataType_U8, "uchar")
            .value("s16", ImGuiDataType_S16, "short")
            .value("u16", ImGuiDataType_U16, "unsigned short")
            .value("s32", ImGuiDataType_S32, "int")
            .value("u32", ImGuiDataType_U32, "unsigned int")
            .value("s64", ImGuiDataType_S64, "long long / __int64")
            .value("u64", ImGuiDataType_U64, "unsigned long long / unsigned __int64")
            .value("float", ImGuiDataType_Float, "float")
            .value("double", ImGuiDataType_Double, "double")
            .value("bool", ImGuiDataType_Bool, "bool (provided for user convenience, not supported by scalar widgets)")
            .value("string", ImGuiDataType_String, "char* (provided for user convenience, not supported by scalar widgets)")
            .value("count", ImGuiDataType_COUNT, "");


    auto pyEnumDir =
        nb::enum_<ImGuiDir>(m, "Dir", nb::is_arithmetic(), nb::is_flag(), "A cardinal direction")
            .value("none", ImGuiDir_None, "")
            .value("left", ImGuiDir_Left, "")
            .value("right", ImGuiDir_Right, "")
            .value("up", ImGuiDir_Up, "")
            .value("down", ImGuiDir_Down, "")
            .value("count", ImGuiDir_COUNT, "");


    auto pyEnumSortDirection =
        nb::enum_<ImGuiSortDirection>(m, "SortDirection", nb::is_arithmetic(), nb::is_flag(), "A sorting direction")
            .value("none", ImGuiSortDirection_None, "")
            .value("ascending", ImGuiSortDirection_Ascending, "Ascending = 0->9, A->Z etc.")
            .value("descending", ImGuiSortDirection_Descending, "Descending = 9->0, Z->A etc.");


    auto pyEnumKey =
        nb::enum_<ImGuiKey>(m, "Key", nb::is_arithmetic(), nb::is_flag(), " A key identifier (ImGuiKey_XXX or ImGuiMod_XXX value): can represent Keyboard, Mouse and Gamepad values.\n All our named keys are >= 512. Keys value 0 to 511 are left unused and were legacy native/opaque key values (< 1.87).\n Support for legacy keys was completely removed in 1.91.5.\n Read details about the 1.87+ transition : https://github.com/ocornut/imgui/issues/4921\n Note that \"Keys\" related to physical keys and are not the same concept as input \"Characters\", the later are submitted via io.AddInputCharacter().\n The keyboard key enum values are named after the keys on a standard US keyboard, and on other keyboard types the keys reported may not match the keycaps.")
            .value("none", ImGuiKey_None, "")
            .value("named_key_begin", ImGuiKey_NamedKey_BEGIN, "First valid key value (other than 0)")
            .value("tab", ImGuiKey_Tab, "== ImGuiKey_NamedKey_BEGIN")
            .value("left_arrow", ImGuiKey_LeftArrow, "")
            .value("right_arrow", ImGuiKey_RightArrow, "")
            .value("up_arrow", ImGuiKey_UpArrow, "")
            .value("down_arrow", ImGuiKey_DownArrow, "")
            .value("page_up", ImGuiKey_PageUp, "")
            .value("page_down", ImGuiKey_PageDown, "")
            .value("home", ImGuiKey_Home, "")
            .value("end", ImGuiKey_End, "")
            .value("insert", ImGuiKey_Insert, "")
            .value("delete", ImGuiKey_Delete, "")
            .value("backspace", ImGuiKey_Backspace, "")
            .value("space", ImGuiKey_Space, "")
            .value("enter", ImGuiKey_Enter, "")
            .value("escape", ImGuiKey_Escape, "")
            .value("left_ctrl", ImGuiKey_LeftCtrl, "")
            .value("left_shift", ImGuiKey_LeftShift, "")
            .value("left_alt", ImGuiKey_LeftAlt, "")
            .value("left_super", ImGuiKey_LeftSuper, "Also see ImGuiMod_Ctrl, ImGuiMod_Shift, ImGuiMod_Alt, ImGuiMod_Super below!")
            .value("right_ctrl", ImGuiKey_RightCtrl, "")
            .value("right_shift", ImGuiKey_RightShift, "")
            .value("right_alt", ImGuiKey_RightAlt, "")
            .value("right_super", ImGuiKey_RightSuper, "")
            .value("menu", ImGuiKey_Menu, "")
            .value("_0", ImGuiKey_0, "")
            .value("_1", ImGuiKey_1, "")
            .value("_2", ImGuiKey_2, "")
            .value("_3", ImGuiKey_3, "")
            .value("_4", ImGuiKey_4, "")
            .value("_5", ImGuiKey_5, "")
            .value("_6", ImGuiKey_6, "")
            .value("_7", ImGuiKey_7, "")
            .value("_8", ImGuiKey_8, "")
            .value("_9", ImGuiKey_9, "")
            .value("a", ImGuiKey_A, "")
            .value("b", ImGuiKey_B, "")
            .value("c", ImGuiKey_C, "")
            .value("d", ImGuiKey_D, "")
            .value("e", ImGuiKey_E, "")
            .value("f", ImGuiKey_F, "")
            .value("g", ImGuiKey_G, "")
            .value("h", ImGuiKey_H, "")
            .value("i", ImGuiKey_I, "")
            .value("j", ImGuiKey_J, "")
            .value("k", ImGuiKey_K, "")
            .value("l", ImGuiKey_L, "")
            .value("m", ImGuiKey_M, "")
            .value("n", ImGuiKey_N, "")
            .value("o", ImGuiKey_O, "")
            .value("p", ImGuiKey_P, "")
            .value("q", ImGuiKey_Q, "")
            .value("r", ImGuiKey_R, "")
            .value("s", ImGuiKey_S, "")
            .value("t", ImGuiKey_T, "")
            .value("u", ImGuiKey_U, "")
            .value("v", ImGuiKey_V, "")
            .value("w", ImGuiKey_W, "")
            .value("x", ImGuiKey_X, "")
            .value("y", ImGuiKey_Y, "")
            .value("z", ImGuiKey_Z, "")
            .value("f1", ImGuiKey_F1, "")
            .value("f2", ImGuiKey_F2, "")
            .value("f3", ImGuiKey_F3, "")
            .value("f4", ImGuiKey_F4, "")
            .value("f5", ImGuiKey_F5, "")
            .value("f6", ImGuiKey_F6, "")
            .value("f7", ImGuiKey_F7, "")
            .value("f8", ImGuiKey_F8, "")
            .value("f9", ImGuiKey_F9, "")
            .value("f10", ImGuiKey_F10, "")
            .value("f11", ImGuiKey_F11, "")
            .value("f12", ImGuiKey_F12, "")
            .value("f13", ImGuiKey_F13, "")
            .value("f14", ImGuiKey_F14, "")
            .value("f15", ImGuiKey_F15, "")
            .value("f16", ImGuiKey_F16, "")
            .value("f17", ImGuiKey_F17, "")
            .value("f18", ImGuiKey_F18, "")
            .value("f19", ImGuiKey_F19, "")
            .value("f20", ImGuiKey_F20, "")
            .value("f21", ImGuiKey_F21, "")
            .value("f22", ImGuiKey_F22, "")
            .value("f23", ImGuiKey_F23, "")
            .value("f24", ImGuiKey_F24, "")
            .value("apostrophe", ImGuiKey_Apostrophe, "'")
            .value("comma", ImGuiKey_Comma, ",")
            .value("minus", ImGuiKey_Minus, "-")
            .value("period", ImGuiKey_Period, ".")
            .value("slash", ImGuiKey_Slash, "/")
            .value("semicolon", ImGuiKey_Semicolon, ";")
            .value("equal", ImGuiKey_Equal, "=")
            .value("left_bracket", ImGuiKey_LeftBracket, "[")
            .value("backslash", ImGuiKey_Backslash, "\\ (this text inhibit multiline comment caused by backslash)")
            .value("right_bracket", ImGuiKey_RightBracket, "]")
            .value("grave_accent", ImGuiKey_GraveAccent, "`")
            .value("caps_lock", ImGuiKey_CapsLock, "")
            .value("scroll_lock", ImGuiKey_ScrollLock, "")
            .value("num_lock", ImGuiKey_NumLock, "")
            .value("print_screen", ImGuiKey_PrintScreen, "")
            .value("pause", ImGuiKey_Pause, "")
            .value("keypad0", ImGuiKey_Keypad0, "")
            .value("keypad1", ImGuiKey_Keypad1, "")
            .value("keypad2", ImGuiKey_Keypad2, "")
            .value("keypad3", ImGuiKey_Keypad3, "")
            .value("keypad4", ImGuiKey_Keypad4, "")
            .value("keypad5", ImGuiKey_Keypad5, "")
            .value("keypad6", ImGuiKey_Keypad6, "")
            .value("keypad7", ImGuiKey_Keypad7, "")
            .value("keypad8", ImGuiKey_Keypad8, "")
            .value("keypad9", ImGuiKey_Keypad9, "")
            .value("keypad_decimal", ImGuiKey_KeypadDecimal, "")
            .value("keypad_divide", ImGuiKey_KeypadDivide, "")
            .value("keypad_multiply", ImGuiKey_KeypadMultiply, "")
            .value("keypad_subtract", ImGuiKey_KeypadSubtract, "")
            .value("keypad_add", ImGuiKey_KeypadAdd, "")
            .value("keypad_enter", ImGuiKey_KeypadEnter, "")
            .value("keypad_equal", ImGuiKey_KeypadEqual, "")
            .value("app_back", ImGuiKey_AppBack, "Available on some keyboard/mouses. Often referred as \"Browser Back\"")
            .value("app_forward", ImGuiKey_AppForward, "")
            .value("oem102", ImGuiKey_Oem102, "Non-US backslash.")
            .value("gamepad_start", ImGuiKey_GamepadStart, "Menu        | +       | Options  |")
            .value("gamepad_back", ImGuiKey_GamepadBack, "View        | -       | Share    |")
            .value("gamepad_face_left", ImGuiKey_GamepadFaceLeft, "X           | Y       | Square   | Tap: Toggle Menu. Hold: Windowing mode (Focus/Move/Resize windows)")
            .value("gamepad_face_right", ImGuiKey_GamepadFaceRight, "B           | A       | Circle   | Cancel / Close / Exit")
            .value("gamepad_face_up", ImGuiKey_GamepadFaceUp, "Y           | X       | Triangle | Text Input / On-screen Keyboard")
            .value("gamepad_face_down", ImGuiKey_GamepadFaceDown, "A           | B       | Cross    | Activate / Open / Toggle / Tweak")
            .value("gamepad_dpad_left", ImGuiKey_GamepadDpadLeft, "D-pad Left  | \"       | \"        | Move / Tweak / Resize Window (in Windowing mode)")
            .value("gamepad_dpad_right", ImGuiKey_GamepadDpadRight, "D-pad Right | \"       | \"        | Move / Tweak / Resize Window (in Windowing mode)")
            .value("gamepad_dpad_up", ImGuiKey_GamepadDpadUp, "D-pad Up    | \"       | \"        | Move / Tweak / Resize Window (in Windowing mode)")
            .value("gamepad_dpad_down", ImGuiKey_GamepadDpadDown, "D-pad Down  | \"       | \"        | Move / Tweak / Resize Window (in Windowing mode)")
            .value("gamepad_l1", ImGuiKey_GamepadL1, "L Bumper    | L       | L1       | Tweak Slower / Focus Previous (in Windowing mode)")
            .value("gamepad_r1", ImGuiKey_GamepadR1, "R Bumper    | R       | R1       | Tweak Faster / Focus Next (in Windowing mode)")
            .value("gamepad_l2", ImGuiKey_GamepadL2, "L Trigger   | ZL      | L2       | [Analog]")
            .value("gamepad_r2", ImGuiKey_GamepadR2, "R Trigger   | ZR      | R2       | [Analog]")
            .value("gamepad_l3", ImGuiKey_GamepadL3, "L Stick     | L3      | L3       |")
            .value("gamepad_r3", ImGuiKey_GamepadR3, "R Stick     | R3      | R3       |")
            .value("gamepad_l_stick_left", ImGuiKey_GamepadLStickLeft, "|         |          | [Analog] Move Window (in Windowing mode)")
            .value("gamepad_l_stick_right", ImGuiKey_GamepadLStickRight, "|         |          | [Analog] Move Window (in Windowing mode)")
            .value("gamepad_l_stick_up", ImGuiKey_GamepadLStickUp, "|         |          | [Analog] Move Window (in Windowing mode)")
            .value("gamepad_l_stick_down", ImGuiKey_GamepadLStickDown, "|         |          | [Analog] Move Window (in Windowing mode)")
            .value("gamepad_r_stick_left", ImGuiKey_GamepadRStickLeft, "|         |          | [Analog]")
            .value("gamepad_r_stick_right", ImGuiKey_GamepadRStickRight, "|         |          | [Analog]")
            .value("gamepad_r_stick_up", ImGuiKey_GamepadRStickUp, "|         |          | [Analog]")
            .value("gamepad_r_stick_down", ImGuiKey_GamepadRStickDown, "|         |          | [Analog]")
            .value("mouse_left", ImGuiKey_MouseLeft, " Aliases: Mouse Buttons (auto-submitted from AddMouseButtonEvent() calls)\n - This is mirroring the data also written to io.MouseDown[], io.MouseWheel, in a format allowing them to be accessed via standard key API.")
            .value("mouse_right", ImGuiKey_MouseRight, "")
            .value("mouse_middle", ImGuiKey_MouseMiddle, "")
            .value("mouse_x1", ImGuiKey_MouseX1, "")
            .value("mouse_x2", ImGuiKey_MouseX2, "")
            .value("mouse_wheel_x", ImGuiKey_MouseWheelX, "")
            .value("mouse_wheel_y", ImGuiKey_MouseWheelY, "")
            .value("reserved_for_mod_ctrl", ImGuiKey_ReservedForModCtrl, "[Internal] Reserved for mod storage")
            .value("reserved_for_mod_shift", ImGuiKey_ReservedForModShift, "")
            .value("reserved_for_mod_alt", ImGuiKey_ReservedForModAlt, "")
            .value("reserved_for_mod_super", ImGuiKey_ReservedForModSuper, "")
            .value("named_key_end", ImGuiKey_NamedKey_END, "")
            .value("named_key_count", ImGuiKey_NamedKey_COUNT, "")
            .value("mod_none", ImGuiMod_None, "")
            .value("mod_ctrl", ImGuiMod_Ctrl, "Ctrl (non-macOS), Cmd (macOS)")
            .value("mod_shift", ImGuiMod_Shift, "Shift")
            .value("mod_alt", ImGuiMod_Alt, "Option/Menu")
            .value("mod_super", ImGuiMod_Super, "Windows/Super (non-macOS), Ctrl (macOS)")
            .value("mod_mask_", ImGuiMod_Mask_, "4-bits");


    auto pyEnumInputFlags_ =
        nb::enum_<ImGuiInputFlags_>(m, "InputFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for Shortcut(), SetNextItemShortcut(),\n (and for upcoming extended versions of IsKeyPressed(), IsMouseClicked(), Shortcut(), SetKeyOwner(), SetItemKeyOwner() that are still in imgui_internal.h)\n Don't mistake with ImGuiInputTextFlags! (which is for ImGui::InputText() function)")
            .value("none", ImGuiInputFlags_None, "")
            .value("repeat", ImGuiInputFlags_Repeat, "Enable repeat. Return True on successive repeats. Default for legacy IsKeyPressed(). NOT Default for legacy IsMouseClicked(). MUST BE == 1.")
            .value("route_active", ImGuiInputFlags_RouteActive, "Route to active item only.")
            .value("route_focused", ImGuiInputFlags_RouteFocused, "Route to windows in the focus stack (DEFAULT). Deep-most focused window takes inputs. Active item takes inputs over deep-most focused window.")
            .value("route_global", ImGuiInputFlags_RouteGlobal, "Global route (unless a focused window or active item registered the route).")
            .value("route_always", ImGuiInputFlags_RouteAlways, "Do not register route, poll keys directly.")
            .value("route_over_focused", ImGuiInputFlags_RouteOverFocused, "Option: global route: higher priority than focused route (unless active item in focused route).")
            .value("route_over_active", ImGuiInputFlags_RouteOverActive, "Option: global route: higher priority than active item. Unlikely you need to use that: will interfere with every active items, e.g. CTRL+A registered by InputText will be overridden by this. May not be fully honored as user/internal code is likely to always assume they can access keys when active.")
            .value("route_unless_bg_focused", ImGuiInputFlags_RouteUnlessBgFocused, "Option: global route: will not be applied if underlying background/None is focused (== no Dear ImGui windows are focused). Useful for overlay applications.")
            .value("route_from_root_window", ImGuiInputFlags_RouteFromRootWindow, "Option: route evaluated from the point of view of root window rather than current window.")
            .value("tooltip", ImGuiInputFlags_Tooltip, "Automatically display a tooltip when hovering item [BETA] Unsure of right api (opt-in/opt-out)");


    auto pyEnumConfigFlags_ =
        nb::enum_<ImGuiConfigFlags_>(m, "ConfigFlags_", nb::is_arithmetic(), nb::is_flag(), "Configuration flags stored in io.ConfigFlags. Set by user/application.")
            .value("none", ImGuiConfigFlags_None, "")
            .value("nav_enable_keyboard", ImGuiConfigFlags_NavEnableKeyboard, "Master keyboard navigation enable flag. Enable full Tabbing + directional arrows + space/enter to activate.")
            .value("nav_enable_gamepad", ImGuiConfigFlags_NavEnableGamepad, "Master gamepad navigation enable flag. Backend also needs to set ImGuiBackendFlags_HasGamepad.")
            .value("no_mouse", ImGuiConfigFlags_NoMouse, "Instruct dear imgui to disable mouse inputs and interactions.")
            .value("no_mouse_cursor_change", ImGuiConfigFlags_NoMouseCursorChange, "Instruct backend to not alter mouse cursor shape and visibility. Use if the backend cursor changes are interfering with yours and you don't want to use SetMouseCursor() to change mouse cursor. You may want to honor requests from imgui by reading GetMouseCursor() yourself instead.")
            .value("no_keyboard", ImGuiConfigFlags_NoKeyboard, "Instruct dear imgui to disable keyboard inputs and interactions. This is done by ignoring keyboard events and clearing existing states.")
            .value("docking_enable", ImGuiConfigFlags_DockingEnable, "Docking enable flags.")
            .value("viewports_enable", ImGuiConfigFlags_ViewportsEnable, "Viewport enable flags (require both ImGuiBackendFlags_PlatformHasViewports + ImGuiBackendFlags_RendererHasViewports set by the respective backends)")
            .value("is_srgb", ImGuiConfigFlags_IsSRGB, "Application is SRGB-aware.")
            .value("is_touch_screen", ImGuiConfigFlags_IsTouchScreen, "Application is using a touch screen instead of a mouse.");


    auto pyEnumBackendFlags_ =
        nb::enum_<ImGuiBackendFlags_>(m, "BackendFlags_", nb::is_arithmetic(), nb::is_flag(), "Backend capabilities flags stored in io.BackendFlags. Set by imgui_impl_xxx or custom backend.")
            .value("none", ImGuiBackendFlags_None, "")
            .value("has_gamepad", ImGuiBackendFlags_HasGamepad, "Backend Platform supports gamepad and currently has one connected.")
            .value("has_mouse_cursors", ImGuiBackendFlags_HasMouseCursors, "Backend Platform supports honoring GetMouseCursor() value to change the OS cursor shape.")
            .value("has_set_mouse_pos", ImGuiBackendFlags_HasSetMousePos, "Backend Platform supports io.WantSetMousePos requests to reposition the OS mouse position (only used if io.ConfigNavMoveSetMousePos is set).")
            .value("renderer_has_vtx_offset", ImGuiBackendFlags_RendererHasVtxOffset, "Backend Renderer supports ImDrawCmd::VtxOffset. This enables output of large meshes (64K+ vertices) while still using 16-bit indices.")
            .value("renderer_has_textures", ImGuiBackendFlags_RendererHasTextures, "Backend Renderer supports ImTextureData requests to create/update/destroy textures. This enables incremental texture updates and texture reloads. See https://github.com/ocornut/imgui/blob/master/docs/BACKENDS.md for instructions on how to upgrade your custom backend.")
            .value("platform_has_viewports", ImGuiBackendFlags_PlatformHasViewports, "Backend Platform supports multiple viewports.")
            .value("has_mouse_hovered_viewport", ImGuiBackendFlags_HasMouseHoveredViewport, "Backend Platform supports calling io.AddMouseViewportEvent() with the viewport under the mouse. IF POSSIBLE, ignore viewports with the ImGuiViewportFlags_NoInputs flag (Win32 backend, GLFW 3.30+ backend can do this, SDL backend cannot). If this cannot be done, Dear ImGui needs to use a flawed heuristic to find the viewport under.")
            .value("renderer_has_viewports", ImGuiBackendFlags_RendererHasViewports, "Backend Renderer supports multiple viewports.");


    auto pyEnumCol_ =
        nb::enum_<ImGuiCol_>(m, "Col_", nb::is_arithmetic(), nb::is_flag(), "Enumeration for PushStyleColor() / PopStyleColor()")
            .value("text", ImGuiCol_Text, "")
            .value("text_disabled", ImGuiCol_TextDisabled, "")
            .value("window_bg", ImGuiCol_WindowBg, "Background of normal windows")
            .value("child_bg", ImGuiCol_ChildBg, "Background of child windows")
            .value("popup_bg", ImGuiCol_PopupBg, "Background of popups, menus, tooltips windows")
            .value("border", ImGuiCol_Border, "")
            .value("border_shadow", ImGuiCol_BorderShadow, "")
            .value("frame_bg", ImGuiCol_FrameBg, "Background of checkbox, radio button, plot, slider, text input")
            .value("frame_bg_hovered", ImGuiCol_FrameBgHovered, "")
            .value("frame_bg_active", ImGuiCol_FrameBgActive, "")
            .value("title_bg", ImGuiCol_TitleBg, "Title bar")
            .value("title_bg_active", ImGuiCol_TitleBgActive, "Title bar when focused")
            .value("title_bg_collapsed", ImGuiCol_TitleBgCollapsed, "Title bar when collapsed")
            .value("menu_bar_bg", ImGuiCol_MenuBarBg, "")
            .value("scrollbar_bg", ImGuiCol_ScrollbarBg, "")
            .value("scrollbar_grab", ImGuiCol_ScrollbarGrab, "")
            .value("scrollbar_grab_hovered", ImGuiCol_ScrollbarGrabHovered, "")
            .value("scrollbar_grab_active", ImGuiCol_ScrollbarGrabActive, "")
            .value("check_mark", ImGuiCol_CheckMark, "Checkbox tick and RadioButton circle")
            .value("slider_grab", ImGuiCol_SliderGrab, "")
            .value("slider_grab_active", ImGuiCol_SliderGrabActive, "")
            .value("button", ImGuiCol_Button, "")
            .value("button_hovered", ImGuiCol_ButtonHovered, "")
            .value("button_active", ImGuiCol_ButtonActive, "")
            .value("header", ImGuiCol_Header, "Header* colors are used for CollapsingHeader, TreeNode, Selectable, MenuItem")
            .value("header_hovered", ImGuiCol_HeaderHovered, "")
            .value("header_active", ImGuiCol_HeaderActive, "")
            .value("separator", ImGuiCol_Separator, "")
            .value("separator_hovered", ImGuiCol_SeparatorHovered, "")
            .value("separator_active", ImGuiCol_SeparatorActive, "")
            .value("resize_grip", ImGuiCol_ResizeGrip, "Resize grip in lower-right and lower-left corners of windows.")
            .value("resize_grip_hovered", ImGuiCol_ResizeGripHovered, "")
            .value("resize_grip_active", ImGuiCol_ResizeGripActive, "")
            .value("input_text_cursor", ImGuiCol_InputTextCursor, "InputText cursor/caret")
            .value("tab_hovered", ImGuiCol_TabHovered, "Tab background, when hovered")
            .value("tab", ImGuiCol_Tab, "Tab background, when tab-bar is focused & tab is unselected")
            .value("tab_selected", ImGuiCol_TabSelected, "Tab background, when tab-bar is focused & tab is selected")
            .value("tab_selected_overline", ImGuiCol_TabSelectedOverline, "Tab horizontal overline, when tab-bar is focused & tab is selected")
            .value("tab_dimmed", ImGuiCol_TabDimmed, "Tab background, when tab-bar is unfocused & tab is unselected")
            .value("tab_dimmed_selected", ImGuiCol_TabDimmedSelected, "Tab background, when tab-bar is unfocused & tab is selected")
            .value("tab_dimmed_selected_overline", ImGuiCol_TabDimmedSelectedOverline, "..horizontal overline, when tab-bar is unfocused & tab is selected")
            .value("docking_preview", ImGuiCol_DockingPreview, "Preview overlay color when about to docking something")
            .value("docking_empty_bg", ImGuiCol_DockingEmptyBg, "Background color for empty node (e.g. CentralNode with no window docked into it)")
            .value("plot_lines", ImGuiCol_PlotLines, "")
            .value("plot_lines_hovered", ImGuiCol_PlotLinesHovered, "")
            .value("plot_histogram", ImGuiCol_PlotHistogram, "")
            .value("plot_histogram_hovered", ImGuiCol_PlotHistogramHovered, "")
            .value("table_header_bg", ImGuiCol_TableHeaderBg, "Table header background")
            .value("table_border_strong", ImGuiCol_TableBorderStrong, "Table outer and header borders (prefer using Alpha=1.0 here)")
            .value("table_border_light", ImGuiCol_TableBorderLight, "Table inner borders (prefer using Alpha=1.0 here)")
            .value("table_row_bg", ImGuiCol_TableRowBg, "Table row background (even rows)")
            .value("table_row_bg_alt", ImGuiCol_TableRowBgAlt, "Table row background (odd rows)")
            .value("text_link", ImGuiCol_TextLink, "Hyperlink color")
            .value("text_selected_bg", ImGuiCol_TextSelectedBg, "Selected text inside an InputText")
            .value("tree_lines", ImGuiCol_TreeLines, "Tree node hierarchy outlines when using ImGuiTreeNodeFlags_DrawLines")
            .value("drag_drop_target", ImGuiCol_DragDropTarget, "Rectangle highlighting a drop target")
            .value("nav_cursor", ImGuiCol_NavCursor, "Color of keyboard/gamepad navigation cursor/rectangle, when visible")
            .value("nav_windowing_highlight", ImGuiCol_NavWindowingHighlight, "Highlight window when using CTRL+TAB")
            .value("nav_windowing_dim_bg", ImGuiCol_NavWindowingDimBg, "Darken/colorize entire screen behind the CTRL+TAB window list, when active")
            .value("modal_window_dim_bg", ImGuiCol_ModalWindowDimBg, "Darken/colorize entire screen behind a modal window, when one is active")
            .value("count", ImGuiCol_COUNT, "");


    auto pyEnumStyleVar_ =
        nb::enum_<ImGuiStyleVar_>(m, "StyleVar_", nb::is_arithmetic(), nb::is_flag(), " Enumeration for PushStyleVar() / PopStyleVar() to temporarily modify the ImGuiStyle structure.\n - The enum only refers to fields of ImGuiStyle which makes sense to be pushed/popped inside UI code.\n   During initialization or between frames, feel free to just poke into ImGuiStyle directly.\n - Tip: Use your programming IDE navigation facilities on the names in the _second column_ below to find the actual members and their description.\n   - In Visual Studio: CTRL+comma (\"Edit.GoToAll\") can follow symbols inside comments, whereas CTRL+F12 (\"Edit.GoToImplementation\") cannot.\n   - In Visual Studio w/ Visual Assist installed: ALT+G (\"VAssistX.GoToImplementation\") can also follow symbols inside comments.\n   - In VS Code, CLion, etc.: CTRL+click can follow symbols inside comments.\n - When changing this enum, you need to update the associated internal table GStyleVarInfo[] accordingly. This is where we link enum values to members offset/type.")
            .value("alpha", ImGuiStyleVar_Alpha, "float     Alpha")
            .value("disabled_alpha", ImGuiStyleVar_DisabledAlpha, "float     DisabledAlpha")
            .value("window_padding", ImGuiStyleVar_WindowPadding, "ImVec2    WindowPadding")
            .value("window_rounding", ImGuiStyleVar_WindowRounding, "float     WindowRounding")
            .value("window_border_size", ImGuiStyleVar_WindowBorderSize, "float     WindowBorderSize")
            .value("window_min_size", ImGuiStyleVar_WindowMinSize, "ImVec2    WindowMinSize")
            .value("window_title_align", ImGuiStyleVar_WindowTitleAlign, "ImVec2    WindowTitleAlign")
            .value("child_rounding", ImGuiStyleVar_ChildRounding, "float     ChildRounding")
            .value("child_border_size", ImGuiStyleVar_ChildBorderSize, "float     ChildBorderSize")
            .value("popup_rounding", ImGuiStyleVar_PopupRounding, "float     PopupRounding")
            .value("popup_border_size", ImGuiStyleVar_PopupBorderSize, "float     PopupBorderSize")
            .value("frame_padding", ImGuiStyleVar_FramePadding, "ImVec2    FramePadding")
            .value("frame_rounding", ImGuiStyleVar_FrameRounding, "float     FrameRounding")
            .value("frame_border_size", ImGuiStyleVar_FrameBorderSize, "float     FrameBorderSize")
            .value("item_spacing", ImGuiStyleVar_ItemSpacing, "ImVec2    ItemSpacing")
            .value("item_inner_spacing", ImGuiStyleVar_ItemInnerSpacing, "ImVec2    ItemInnerSpacing")
            .value("indent_spacing", ImGuiStyleVar_IndentSpacing, "float     IndentSpacing")
            .value("cell_padding", ImGuiStyleVar_CellPadding, "ImVec2    CellPadding")
            .value("scrollbar_size", ImGuiStyleVar_ScrollbarSize, "float     ScrollbarSize")
            .value("scrollbar_rounding", ImGuiStyleVar_ScrollbarRounding, "float     ScrollbarRounding")
            .value("scrollbar_padding", ImGuiStyleVar_ScrollbarPadding, "float     ScrollbarPadding")
            .value("grab_min_size", ImGuiStyleVar_GrabMinSize, "float     GrabMinSize")
            .value("grab_rounding", ImGuiStyleVar_GrabRounding, "float     GrabRounding")
            .value("image_border_size", ImGuiStyleVar_ImageBorderSize, "float     ImageBorderSize")
            .value("layout_align", ImGuiStyleVar_LayoutAlign, "float     LayoutAlign")
            .value("tab_rounding", ImGuiStyleVar_TabRounding, "float     TabRounding")
            .value("tab_border_size", ImGuiStyleVar_TabBorderSize, "float     TabBorderSize")
            .value("tab_min_width_base", ImGuiStyleVar_TabMinWidthBase, "float     TabMinWidthBase")
            .value("tab_min_width_shrink", ImGuiStyleVar_TabMinWidthShrink, "float     TabMinWidthShrink")
            .value("tab_bar_border_size", ImGuiStyleVar_TabBarBorderSize, "float     TabBarBorderSize")
            .value("tab_bar_overline_size", ImGuiStyleVar_TabBarOverlineSize, "float     TabBarOverlineSize")
            .value("table_angled_headers_angle", ImGuiStyleVar_TableAngledHeadersAngle, "float     TableAngledHeadersAngle")
            .value("table_angled_headers_text_align", ImGuiStyleVar_TableAngledHeadersTextAlign, "ImVec2  TableAngledHeadersTextAlign")
            .value("tree_lines_size", ImGuiStyleVar_TreeLinesSize, "float     TreeLinesSize")
            .value("tree_lines_rounding", ImGuiStyleVar_TreeLinesRounding, "float     TreeLinesRounding")
            .value("button_text_align", ImGuiStyleVar_ButtonTextAlign, "ImVec2    ButtonTextAlign")
            .value("selectable_text_align", ImGuiStyleVar_SelectableTextAlign, "ImVec2    SelectableTextAlign")
            .value("separator_text_border_size", ImGuiStyleVar_SeparatorTextBorderSize, "float     SeparatorTextBorderSize")
            .value("separator_text_align", ImGuiStyleVar_SeparatorTextAlign, "ImVec2    SeparatorTextAlign")
            .value("separator_text_padding", ImGuiStyleVar_SeparatorTextPadding, "ImVec2    SeparatorTextPadding")
            .value("docking_separator_size", ImGuiStyleVar_DockingSeparatorSize, "float     DockingSeparatorSize")
            .value("count", ImGuiStyleVar_COUNT, "");


    auto pyEnumButtonFlags_ =
        nb::enum_<ImGuiButtonFlags_>(m, "ButtonFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for InvisibleButton() [extended in imgui_internal.h]")
            .value("none", ImGuiButtonFlags_None, "")
            .value("mouse_button_left", ImGuiButtonFlags_MouseButtonLeft, "React on left mouse button (default)")
            .value("mouse_button_right", ImGuiButtonFlags_MouseButtonRight, "React on right mouse button")
            .value("mouse_button_middle", ImGuiButtonFlags_MouseButtonMiddle, "React on center mouse button")
            .value("mouse_button_mask_", ImGuiButtonFlags_MouseButtonMask_, "[Internal]")
            .value("enable_nav", ImGuiButtonFlags_EnableNav, "InvisibleButton(): do not disable navigation/tabbing. Otherwise disabled by default.");


    auto pyEnumColorEditFlags_ =
        nb::enum_<ImGuiColorEditFlags_>(m, "ColorEditFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ColorEdit3() / ColorEdit4() / ColorPicker3() / ColorPicker4() / ColorButton()")
            .value("none", ImGuiColorEditFlags_None, "")
            .value("no_alpha", ImGuiColorEditFlags_NoAlpha, "// ColorEdit, ColorPicker, ColorButton: ignore Alpha component (will only read 3 components from the input pointer).")
            .value("no_picker", ImGuiColorEditFlags_NoPicker, "// ColorEdit: disable picker when clicking on color square.")
            .value("no_options", ImGuiColorEditFlags_NoOptions, "// ColorEdit: disable toggling options menu when right-clicking on inputs/small preview.")
            .value("no_small_preview", ImGuiColorEditFlags_NoSmallPreview, "// ColorEdit, ColorPicker: disable color square preview next to the inputs. (e.g. to show only the inputs)")
            .value("no_inputs", ImGuiColorEditFlags_NoInputs, "// ColorEdit, ColorPicker: disable inputs sliders/text widgets (e.g. to show only the small preview color square).")
            .value("no_tooltip", ImGuiColorEditFlags_NoTooltip, "// ColorEdit, ColorPicker, ColorButton: disable tooltip when hovering the preview.")
            .value("no_label", ImGuiColorEditFlags_NoLabel, "// ColorEdit, ColorPicker: disable display of inline text label (the label is still forwarded to the tooltip and picker).")
            .value("no_side_preview", ImGuiColorEditFlags_NoSidePreview, "// ColorPicker: disable bigger color preview on right side of the picker, use small color square preview instead.")
            .value("no_drag_drop", ImGuiColorEditFlags_NoDragDrop, "// ColorEdit: disable drag and drop target. ColorButton: disable drag and drop source.")
            .value("no_border", ImGuiColorEditFlags_NoBorder, "// ColorButton: disable border (which is enforced by default)")
            .value("alpha_opaque", ImGuiColorEditFlags_AlphaOpaque, "// ColorEdit, ColorPicker, ColorButton: disable alpha in the preview,. Contrary to _NoAlpha it may still be edited when calling ColorEdit4()/ColorPicker4(). For ColorButton() this does the same as _NoAlpha.")
            .value("alpha_no_bg", ImGuiColorEditFlags_AlphaNoBg, "// ColorEdit, ColorPicker, ColorButton: disable rendering a checkerboard background behind transparent color.")
            .value("alpha_preview_half", ImGuiColorEditFlags_AlphaPreviewHalf, "// ColorEdit, ColorPicker, ColorButton: display half opaque / half transparent preview.")
            .value("alpha_bar", ImGuiColorEditFlags_AlphaBar, "// ColorEdit, ColorPicker: show vertical alpha bar/gradient in picker.")
            .value("hdr", ImGuiColorEditFlags_HDR, "// (WIP) ColorEdit: Currently only disable 0.0..1.0 limits in RGBA edition (note: you probably want to use ImGuiColorEditFlags_Float flag as well).")
            .value("display_rgb", ImGuiColorEditFlags_DisplayRGB, "[Display]    // ColorEdit: override _display_ type among RGB/HSV/Hex. ColorPicker: select any combination using one or more of RGB/HSV/Hex.")
            .value("display_hsv", ImGuiColorEditFlags_DisplayHSV, "[Display]    // \"")
            .value("display_hex", ImGuiColorEditFlags_DisplayHex, "[Display]    // \"")
            .value("uint8", ImGuiColorEditFlags_Uint8, "[DataType]   // ColorEdit, ColorPicker, ColorButton: _display_ values formatted as 0..255.")
            .value("float", ImGuiColorEditFlags_Float, "[DataType]   // ColorEdit, ColorPicker, ColorButton: _display_ values formatted as 0.0..1.0 floats instead of 0..255 integers. No round-trip of value via integers.")
            .value("picker_hue_bar", ImGuiColorEditFlags_PickerHueBar, "[Picker]     // ColorPicker: bar for Hue, rectangle for Sat/Value.")
            .value("picker_hue_wheel", ImGuiColorEditFlags_PickerHueWheel, "[Picker]     // ColorPicker: wheel for Hue, triangle for Sat/Value.")
            .value("input_rgb", ImGuiColorEditFlags_InputRGB, "[Input]      // ColorEdit, ColorPicker: input and output data in RGB format.")
            .value("input_hsv", ImGuiColorEditFlags_InputHSV, "[Input]      // ColorEdit, ColorPicker: input and output data in HSV format.")
            .value("default_options_", ImGuiColorEditFlags_DefaultOptions_, " Defaults Options. You can set application defaults using SetColorEditOptions(). The intent is that you probably don't want to\n override them in most of your calls. Let the user choose via the option menu and/or call SetColorEditOptions() once during startup.")
            .value("alpha_mask_", ImGuiColorEditFlags_AlphaMask_, "")
            .value("display_mask_", ImGuiColorEditFlags_DisplayMask_, "")
            .value("data_type_mask_", ImGuiColorEditFlags_DataTypeMask_, "")
            .value("picker_mask_", ImGuiColorEditFlags_PickerMask_, "")
            .value("input_mask_", ImGuiColorEditFlags_InputMask_, "");


    auto pyEnumSliderFlags_ =
        nb::enum_<ImGuiSliderFlags_>(m, "SliderFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for DragFloat(), DragInt(), SliderFloat(), SliderInt() etc.\n We use the same sets of flags for DragXXX() and SliderXXX() functions as the features are the same and it makes it easier to swap them.\n (Those are per-item flags. There is shared behavior flag too: ImGuiIO: io.ConfigDragClickToInputText)")
            .value("none", ImGuiSliderFlags_None, "")
            .value("logarithmic", ImGuiSliderFlags_Logarithmic, "Make the widget logarithmic (linear otherwise). Consider using ImGuiSliderFlags_NoRoundToFormat with this if using a format-string with small amount of digits.")
            .value("no_round_to_format", ImGuiSliderFlags_NoRoundToFormat, "Disable rounding underlying value to match precision of the display format string (e.g. %.3 values are rounded to those 3 digits).")
            .value("no_input", ImGuiSliderFlags_NoInput, "Disable CTRL+Click or Enter key allowing to input text directly into the widget.")
            .value("wrap_around", ImGuiSliderFlags_WrapAround, "Enable wrapping around from max to min and from min to max. Only supported by DragXXX() functions for now.")
            .value("clamp_on_input", ImGuiSliderFlags_ClampOnInput, "Clamp value to min/max bounds when input manually with CTRL+Click. By default CTRL+Click allows going out of bounds.")
            .value("clamp_zero_range", ImGuiSliderFlags_ClampZeroRange, "Clamp even if min==max==0.0. Otherwise due to legacy reason DragXXX functions don't clamp with those values. When your clamping limits are dynamic you almost always want to use it.")
            .value("no_speed_tweaks", ImGuiSliderFlags_NoSpeedTweaks, "Disable keyboard modifiers altering tweak speed. Useful if you want to alter tweak speed yourself based on your own logic.")
            .value("always_clamp", ImGuiSliderFlags_AlwaysClamp, "")
            .value("invalid_mask_", ImGuiSliderFlags_InvalidMask_, "[Internal] We treat using those bits as being potentially a 'float power' argument from the previous API that has got miscast to this enum, and will trigger an assert if needed.");


    auto pyEnumMouseButton_ =
        nb::enum_<ImGuiMouseButton_>(m, "MouseButton_", nb::is_arithmetic(), nb::is_flag(), " Identify a mouse button.\n Those values are guaranteed to be stable and we frequently use 0/1 directly. Named enums provided for convenience.")
            .value("left", ImGuiMouseButton_Left, "")
            .value("right", ImGuiMouseButton_Right, "")
            .value("middle", ImGuiMouseButton_Middle, "")
            .value("count", ImGuiMouseButton_COUNT, "");


    auto pyEnumMouseCursor_ =
        nb::enum_<ImGuiMouseCursor_>(m, "MouseCursor_", nb::is_arithmetic(), nb::is_flag(), " Enumeration for GetMouseCursor()\n User code may request backend to display given cursor by calling SetMouseCursor(), which is why we have some cursors that are marked unused here")
            .value("none", ImGuiMouseCursor_None, "")
            .value("arrow", ImGuiMouseCursor_Arrow, "")
            .value("text_input", ImGuiMouseCursor_TextInput, "When hovering over InputText, etc.")
            .value("resize_all", ImGuiMouseCursor_ResizeAll, "(Unused by Dear ImGui functions)")
            .value("resize_ns", ImGuiMouseCursor_ResizeNS, "When hovering over a horizontal border")
            .value("resize_ew", ImGuiMouseCursor_ResizeEW, "When hovering over a vertical border or a column")
            .value("resize_nesw", ImGuiMouseCursor_ResizeNESW, "When hovering over the bottom-left corner of a window")
            .value("resize_nwse", ImGuiMouseCursor_ResizeNWSE, "When hovering over the bottom-right corner of a window")
            .value("hand", ImGuiMouseCursor_Hand, "(Unused by Dear ImGui functions. Use for e.g. hyperlinks)")
            .value("wait", ImGuiMouseCursor_Wait, "When waiting for something to process/load.")
            .value("progress", ImGuiMouseCursor_Progress, "When waiting for something to process/load, but application is still interactive.")
            .value("not_allowed", ImGuiMouseCursor_NotAllowed, "When hovering something with disallowed interaction. Usually a crossed circle.")
            .value("count", ImGuiMouseCursor_COUNT, "");


    auto pyEnumMouseSource =
        nb::enum_<ImGuiMouseSource>(m, "MouseSource", nb::is_arithmetic(), nb::is_flag(), " Enumeration for AddMouseSourceEvent() actual source of Mouse Input data.\n Historically we use \"Mouse\" terminology everywhere to indicate pointer data, e.g. MousePos, IsMousePressed(), io.AddMousePosEvent()\n But that \"Mouse\" data can come from different source which occasionally may be useful for application to know about.\n You can submit a change of pointer type using io.AddMouseSourceEvent().")
            .value("mouse", ImGuiMouseSource_Mouse, "Input is coming from an actual mouse.")
            .value("touch_screen", ImGuiMouseSource_TouchScreen, "Input is coming from a touch screen (no hovering prior to initial press, less precise initial press aiming, dual-axis wheeling possible).")
            .value("pen", ImGuiMouseSource_Pen, "Input is coming from a pressure/magnetic pen (often used in conjunction with high-sampling rates).")
            .value("count", ImGuiMouseSource_COUNT, "");


    auto pyEnumCond_ =
        nb::enum_<ImGuiCond_>(m, "Cond_", nb::is_arithmetic(), nb::is_flag(), " Enumeration for ImGui::SetNextWindow***(), SetWindow***(), SetNextItem***() functions\n Represent a condition.\n Important: Treat as a regular enum! Do NOT combine multiple values using binary operators! All the functions above treat 0 as a shortcut to ImGuiCond_Always.")
            .value("none", ImGuiCond_None, "No condition (always set the variable), same as _Always")
            .value("always", ImGuiCond_Always, "No condition (always set the variable), same as _None")
            .value("once", ImGuiCond_Once, "Set the variable once per runtime session (only the first call will succeed)")
            .value("first_use_ever", ImGuiCond_FirstUseEver, "Set the variable if the object/window has no persistently saved data (no entry in .ini file)")
            .value("appearing", ImGuiCond_Appearing, "Set the variable if the object/window is appearing after being hidden/inactive (or the first time)");


    auto pyEnumTableFlags_ =
        nb::enum_<ImGuiTableFlags_>(m, "TableFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImGui::BeginTable()\n - Important! Sizing policies have complex and subtle side effects, much more so than you would expect.\n   Read comments/demos carefully + experiment with live demos to get acquainted with them.\n - The DEFAULT sizing policies are:\n    - Default to ImGuiTableFlags_SizingFixedFit    if ScrollX is on, or if host window has ImGuiWindowFlags_AlwaysAutoResize.\n    - Default to ImGuiTableFlags_SizingStretchSame if ScrollX is off.\n - When ScrollX is off:\n    - Table defaults to ImGuiTableFlags_SizingStretchSame -> all Columns defaults to ImGuiTableColumnFlags_WidthStretch with same weight.\n    - Columns sizing policy allowed: Stretch (default), Fixed/Auto.\n    - Fixed Columns (if any) will generally obtain their requested width (unless the table cannot fit them all).\n    - Stretch Columns will share the remaining width according to their respective weight.\n    - Mixed Fixed/Stretch columns is possible but has various side-effects on resizing behaviors.\n      The typical use of mixing sizing policies is: any number of LEADING Fixed columns, followed by one or two TRAILING Stretch columns.\n      (this is because the visible order of columns have subtle but necessary effects on how they react to manual resizing).\n - When ScrollX is on:\n    - Table defaults to ImGuiTableFlags_SizingFixedFit -> all Columns defaults to ImGuiTableColumnFlags_WidthFixed\n    - Columns sizing policy allowed: Fixed/Auto mostly.\n    - Fixed Columns can be enlarged as needed. Table will show a horizontal scrollbar if needed.\n    - When using auto-resizing (non-resizable) fixed columns, querying the content width to use item right-alignment e.g. SetNextItemWidth(-FLT_MIN) doesn't make sense, would create a feedback loop.\n    - Using Stretch columns OFTEN DOES NOT MAKE SENSE if ScrollX is on, UNLESS you have specified a value for 'inner_width' in BeginTable().\n      If you specify a value for 'inner_width' then effectively the scrolling space is known and Stretch or mixed Fixed/Stretch columns become meaningful again.\n - Read on documentation at the top of imgui_tables.cpp for details.")
            .value("none", ImGuiTableFlags_None, "")
            .value("resizable", ImGuiTableFlags_Resizable, "Enable resizing columns.")
            .value("reorderable", ImGuiTableFlags_Reorderable, "Enable reordering columns in header row (need calling TableSetupColumn() + TableHeadersRow() to display headers)")
            .value("hideable", ImGuiTableFlags_Hideable, "Enable hiding/disabling columns in context menu.")
            .value("sortable", ImGuiTableFlags_Sortable, "Enable sorting. Call TableGetSortSpecs() to obtain sort specs. Also see ImGuiTableFlags_SortMulti and ImGuiTableFlags_SortTristate.")
            .value("no_saved_settings", ImGuiTableFlags_NoSavedSettings, "Disable persisting columns order, width and sort settings in the .ini file.")
            .value("context_menu_in_body", ImGuiTableFlags_ContextMenuInBody, "Right-click on columns body/contents will display table context menu. By default it is available in TableHeadersRow().")
            .value("row_bg", ImGuiTableFlags_RowBg, "Set each RowBg color with ImGuiCol_TableRowBg or ImGuiCol_TableRowBgAlt (equivalent of calling TableSetBgColor with ImGuiTableBgFlags_RowBg0 on each row manually)")
            .value("borders_inner_h", ImGuiTableFlags_BordersInnerH, "Draw horizontal borders between rows.")
            .value("borders_outer_h", ImGuiTableFlags_BordersOuterH, "Draw horizontal borders at the top and bottom.")
            .value("borders_inner_v", ImGuiTableFlags_BordersInnerV, "Draw vertical borders between columns.")
            .value("borders_outer_v", ImGuiTableFlags_BordersOuterV, "Draw vertical borders on the left and right sides.")
            .value("borders_h", ImGuiTableFlags_BordersH, "Draw horizontal borders.")
            .value("borders_v", ImGuiTableFlags_BordersV, "Draw vertical borders.")
            .value("borders_inner", ImGuiTableFlags_BordersInner, "Draw inner borders.")
            .value("borders_outer", ImGuiTableFlags_BordersOuter, "Draw outer borders.")
            .value("borders", ImGuiTableFlags_Borders, "Draw all borders.")
            .value("no_borders_in_body", ImGuiTableFlags_NoBordersInBody, "[ALPHA] Disable vertical borders in columns Body (borders will always appear in Headers). -> May move to style")
            .value("no_borders_in_body_until_resize", ImGuiTableFlags_NoBordersInBodyUntilResize, "[ALPHA] Disable vertical borders in columns Body until hovered for resize (borders will always appear in Headers). -> May move to style")
            .value("sizing_fixed_fit", ImGuiTableFlags_SizingFixedFit, "Columns default to _WidthFixed or _WidthAuto (if resizable or not resizable), matching contents width.")
            .value("sizing_fixed_same", ImGuiTableFlags_SizingFixedSame, "Columns default to _WidthFixed or _WidthAuto (if resizable or not resizable), matching the maximum contents width of all columns. Implicitly enable ImGuiTableFlags_NoKeepColumnsVisible.")
            .value("sizing_stretch_prop", ImGuiTableFlags_SizingStretchProp, "Columns default to _WidthStretch with default weights proportional to each columns contents widths.")
            .value("sizing_stretch_same", ImGuiTableFlags_SizingStretchSame, "Columns default to _WidthStretch with default weights all equal, unless overridden by TableSetupColumn().")
            .value("no_host_extend_x", ImGuiTableFlags_NoHostExtendX, "Make outer width auto-fit to columns, overriding outer_size.x value. Only available when ScrollX/ScrollY are disabled and Stretch columns are not used.")
            .value("no_host_extend_y", ImGuiTableFlags_NoHostExtendY, "Make outer height stop exactly at outer_size.y (prevent auto-extending table past the limit). Only available when ScrollX/ScrollY are disabled. Data below the limit will be clipped and not visible.")
            .value("no_keep_columns_visible", ImGuiTableFlags_NoKeepColumnsVisible, "Disable keeping column always minimally visible when ScrollX is off and table gets too small. Not recommended if columns are resizable.")
            .value("precise_widths", ImGuiTableFlags_PreciseWidths, "Disable distributing remainder width to stretched columns (width allocation on a 100-wide table with 3 columns: Without this flag: 33,33,34. With this flag: 33,33,33). With larger number of columns, resizing will appear to be less smooth.")
            .value("no_clip", ImGuiTableFlags_NoClip, "Disable clipping rectangle for every individual columns (reduce draw command count, items will be able to overflow into other columns). Generally incompatible with TableSetupScrollFreeze().")
            .value("pad_outer_x", ImGuiTableFlags_PadOuterX, "Default if BordersOuterV is on. Enable outermost padding. Generally desirable if you have headers.")
            .value("no_pad_outer_x", ImGuiTableFlags_NoPadOuterX, "Default if BordersOuterV is off. Disable outermost padding.")
            .value("no_pad_inner_x", ImGuiTableFlags_NoPadInnerX, "Disable inner padding between columns (double inner padding if BordersOuterV is on, single inner padding if BordersOuterV is off).")
            .value("scroll_x", ImGuiTableFlags_ScrollX, "Enable horizontal scrolling. Require 'outer_size' parameter of BeginTable() to specify the container size. Changes default sizing policy. Because this creates a child window, ScrollY is currently generally recommended when using ScrollX.")
            .value("scroll_y", ImGuiTableFlags_ScrollY, "Enable vertical scrolling. Require 'outer_size' parameter of BeginTable() to specify the container size.")
            .value("sort_multi", ImGuiTableFlags_SortMulti, "Hold shift when clicking headers to sort on multiple column. TableGetSortSpecs() may return specs where (SpecsCount > 1).")
            .value("sort_tristate", ImGuiTableFlags_SortTristate, "Allow no sorting, disable default sorting. TableGetSortSpecs() may return specs where (SpecsCount == 0).")
            .value("highlight_hovered_column", ImGuiTableFlags_HighlightHoveredColumn, "Highlight column headers when hovered (may evolve into a fuller highlight)")
            .value("sizing_mask_", ImGuiTableFlags_SizingMask_, "[Internal] Combinations and masks");


    auto pyEnumTableColumnFlags_ =
        nb::enum_<ImGuiTableColumnFlags_>(m, "TableColumnFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::TableSetupColumn()")
            .value("none", ImGuiTableColumnFlags_None, "")
            .value("disabled", ImGuiTableColumnFlags_Disabled, "Overriding/master disable flag: hide column, won't show in context menu (unlike calling TableSetColumnEnabled() which manipulates the user accessible state)")
            .value("default_hide", ImGuiTableColumnFlags_DefaultHide, "Default as a hidden/disabled column.")
            .value("default_sort", ImGuiTableColumnFlags_DefaultSort, "Default as a sorting column.")
            .value("width_stretch", ImGuiTableColumnFlags_WidthStretch, "Column will stretch. Preferable with horizontal scrolling disabled (default if table sizing policy is _SizingStretchSame or _SizingStretchProp).")
            .value("width_fixed", ImGuiTableColumnFlags_WidthFixed, "Column will not stretch. Preferable with horizontal scrolling enabled (default if table sizing policy is _SizingFixedFit and table is resizable).")
            .value("no_resize", ImGuiTableColumnFlags_NoResize, "Disable manual resizing.")
            .value("no_reorder", ImGuiTableColumnFlags_NoReorder, "Disable manual reordering this column, this will also prevent other columns from crossing over this column.")
            .value("no_hide", ImGuiTableColumnFlags_NoHide, "Disable ability to hide/disable this column.")
            .value("no_clip", ImGuiTableColumnFlags_NoClip, "Disable clipping for this column (all NoClip columns will render in a same draw command).")
            .value("no_sort", ImGuiTableColumnFlags_NoSort, "Disable ability to sort on this field (even if ImGuiTableFlags_Sortable is set on the table).")
            .value("no_sort_ascending", ImGuiTableColumnFlags_NoSortAscending, "Disable ability to sort in the ascending direction.")
            .value("no_sort_descending", ImGuiTableColumnFlags_NoSortDescending, "Disable ability to sort in the descending direction.")
            .value("no_header_label", ImGuiTableColumnFlags_NoHeaderLabel, "TableHeadersRow() will submit an empty label for this column. Convenient for some small columns. Name will still appear in context menu or in angled headers. You may append into this cell by calling TableSetColumnIndex() right after the TableHeadersRow() call.")
            .value("no_header_width", ImGuiTableColumnFlags_NoHeaderWidth, "Disable header text width contribution to automatic column width.")
            .value("prefer_sort_ascending", ImGuiTableColumnFlags_PreferSortAscending, "Make the initial sort direction Ascending when first sorting on this column (default).")
            .value("prefer_sort_descending", ImGuiTableColumnFlags_PreferSortDescending, "Make the initial sort direction Descending when first sorting on this column.")
            .value("indent_enable", ImGuiTableColumnFlags_IndentEnable, "Use current Indent value when entering cell (default for column 0).")
            .value("indent_disable", ImGuiTableColumnFlags_IndentDisable, "Ignore current Indent value when entering cell (default for columns > 0). Indentation changes _within_ the cell will still be honored.")
            .value("angled_header", ImGuiTableColumnFlags_AngledHeader, "TableHeadersRow() will submit an angled header row for this column. Note this will add an extra row.")
            .value("is_enabled", ImGuiTableColumnFlags_IsEnabled, "Status: is enabled == not hidden by user/api (referred to as \"Hide\" in _DefaultHide and _NoHide) flags.")
            .value("is_visible", ImGuiTableColumnFlags_IsVisible, "Status: is visible == is enabled AND not clipped by scrolling.")
            .value("is_sorted", ImGuiTableColumnFlags_IsSorted, "Status: is currently part of the sort specs")
            .value("is_hovered", ImGuiTableColumnFlags_IsHovered, "Status: is hovered by mouse")
            .value("width_mask_", ImGuiTableColumnFlags_WidthMask_, "")
            .value("indent_mask_", ImGuiTableColumnFlags_IndentMask_, "")
            .value("status_mask_", ImGuiTableColumnFlags_StatusMask_, "")
            .value("no_direct_resize_", ImGuiTableColumnFlags_NoDirectResize_, "[Internal] Disable user resizing this column directly (it may however we resized indirectly from its left edge)");


    auto pyEnumTableRowFlags_ =
        nb::enum_<ImGuiTableRowFlags_>(m, "TableRowFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGui::TableNextRow()")
            .value("none", ImGuiTableRowFlags_None, "")
            .value("headers", ImGuiTableRowFlags_Headers, "Identify header row (set default background color + width of its contents accounted differently for auto column width)");


    auto pyEnumTableBgTarget_ =
        nb::enum_<ImGuiTableBgTarget_>(m, "TableBgTarget_", nb::is_arithmetic(), nb::is_flag(), " Enum for ImGui::TableSetBgColor()\n Background colors are rendering in 3 layers:\n  - Layer 0: draw with RowBg0 color if set, otherwise draw with ColumnBg0 if set.\n  - Layer 1: draw with RowBg1 color if set, otherwise draw with ColumnBg1 if set.\n  - Layer 2: draw with CellBg color if set.\n The purpose of the two row/columns layers is to let you decide if a background color change should override or blend with the existing color.\n When using ImGuiTableFlags_RowBg on the table, each row has the RowBg0 color automatically set for odd/even rows.\n If you set the color of RowBg0 target, your color will override the existing RowBg0 color.\n If you set the color of RowBg1 or ColumnBg1 target, your color will blend over the RowBg0 color.")
            .value("none", ImGuiTableBgTarget_None, "")
            .value("row_bg0", ImGuiTableBgTarget_RowBg0, "Set row background color 0 (generally used for background, automatically set when ImGuiTableFlags_RowBg is used)")
            .value("row_bg1", ImGuiTableBgTarget_RowBg1, "Set row background color 1 (generally used for selection marking)")
            .value("cell_bg", ImGuiTableBgTarget_CellBg, "Set cell background color (top-most color)");


    auto pyClassImGuiTableSortSpecs =
        nb::class_<ImGuiTableSortSpecs>
            (m, "TableSortSpecs", " Sorting specifications for a table (often handling sort specs for a single column, occasionally more)\n Obtained by calling TableGetSortSpecs().\n When 'SpecsDirty == True' you can sort your data. It will be True with sorting specs have changed since last call, or the first time.\n Make sure to set 'SpecsDirty = False' after sorting, else you may wastefully sort your data every frame!")
        .def_ro("specs", &ImGuiTableSortSpecs::Specs, "Pointer to sort spec array.")
        .def_rw("specs_count", &ImGuiTableSortSpecs::SpecsCount, "Sort spec count. Most often 1. May be > 1 when ImGuiTableFlags_SortMulti is enabled. May be == 0 when ImGuiTableFlags_SortTristate is enabled.")
        .def_rw("specs_dirty", &ImGuiTableSortSpecs::SpecsDirty, "Set to True when specs have changed since last time! Use this to sort again, then clear the flag.")
        .def(nb::init<>())
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("get_specs",
            &ImGuiTableSortSpecs::GetSpecs,
            nb::arg("idx"),
            nb::rv_policy::reference)
        // #endif
        //
        ;


    auto pyClassImGuiTableColumnSortSpecs =
        nb::class_<ImGuiTableColumnSortSpecs>
            (m, "TableColumnSortSpecs", "Sorting specification for one column of a table (sizeof == 12 bytes)")
        .def_rw("column_user_id", &ImGuiTableColumnSortSpecs::ColumnUserID, "User id of the column (if specified by a TableSetupColumn() call)")
        .def_rw("column_index", &ImGuiTableColumnSortSpecs::ColumnIndex, "Index of the column")
        .def_rw("sort_order", &ImGuiTableColumnSortSpecs::SortOrder, "Index within parent ImGuiTableSortSpecs (always stored in order starting from 0, tables sorted on a single criteria will always have a 0 here)")
        .def_rw("sort_direction", &ImGuiTableColumnSortSpecs::SortDirection, "ImGuiSortDirection_Ascending or ImGuiSortDirection_Descending")
        .def(nb::init<>())
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("get_sort_direction",
            &ImGuiTableColumnSortSpecs::GetSortDirection)
        .def("set_sort_direction",
            &ImGuiTableColumnSortSpecs::SetSortDirection, nb::arg("direction"))
        // #endif
        //
        ;


    auto pyClassImNewWrapper =
        nb::class_<ImNewWrapper>
            (m, "ImNewWrapper", "")
        .def(nb::init<>()) // implicit default constructor
        ;


    auto pyClassImVector_int =
        nb::class_<ImVector<int>>
            (m, "ImVector_int", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<int>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<int> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<int>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<int>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<int>::empty, "(private API)")
        .def("size",
            &ImVector<int>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<int>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<int>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<int>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<int>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<int>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<int> &v) {
                return nb::make_iterator(nb::type<ImVector<int>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<int> &v) { return v.size(); })
        ;
    auto pyClassImVector_uint =
        nb::class_<ImVector<uint>>
            (m, "ImVector_uint", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<uint>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<uint> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<uint>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<uint>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<uint>::empty, "(private API)")
        .def("size",
            &ImVector<uint>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<uint>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<uint>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<uint>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<uint>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<uint>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<uint> &v) {
                return nb::make_iterator(nb::type<ImVector<uint>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<uint> &v) { return v.size(); })
        ;
    auto pyClassImVector_float =
        nb::class_<ImVector<float>>
            (m, "ImVector_float", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<float>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<float> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<float>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<float>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<float>::empty, "(private API)")
        .def("size",
            &ImVector<float>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<float>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<float>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<float>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<float>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<float>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<float> &v) {
                return nb::make_iterator(nb::type<ImVector<float>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<float> &v) { return v.size(); })
        ;
    auto pyClassImVector_char =
        nb::class_<ImVector<char>>
            (m, "ImVector_char", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<char>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<char> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<char>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<char>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<char>::empty, "(private API)")
        .def("size",
            &ImVector<char>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<char>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<char>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<char>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<char>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<char>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<char> &v) {
                return nb::make_iterator(nb::type<ImVector<char>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<char> &v) { return v.size(); })
        ;
    auto pyClassImVector_uchar =
        nb::class_<ImVector<uchar>>
            (m, "ImVector_uchar", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<uchar>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<uchar> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<uchar>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<uchar>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<uchar>::empty, "(private API)")
        .def("size",
            &ImVector<uchar>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<uchar>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<uchar>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<uchar>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<uchar>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<uchar>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<uchar> &v) {
                return nb::make_iterator(nb::type<ImVector<uchar>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<uchar> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImDrawCmd =
        nb::class_<ImVector<ImDrawCmd>>
            (m, "ImVector_ImDrawCmd", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImDrawCmd>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImDrawCmd> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImDrawCmd>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImDrawCmd>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImDrawCmd>::empty, "(private API)")
        .def("size",
            &ImVector<ImDrawCmd>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawCmd>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawCmd>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImDrawCmd>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImDrawCmd>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImDrawCmd>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImDrawCmd> &v) {
                return nb::make_iterator(nb::type<ImVector<ImDrawCmd>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImDrawCmd> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImDrawChannel =
        nb::class_<ImVector<ImDrawChannel>>
            (m, "ImVector_ImDrawChannel", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImDrawChannel>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImDrawChannel> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImDrawChannel>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImDrawChannel>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImDrawChannel>::empty, "(private API)")
        .def("size",
            &ImVector<ImDrawChannel>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawChannel>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawChannel>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImDrawChannel>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImDrawChannel>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImDrawChannel>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImDrawChannel> &v) {
                return nb::make_iterator(nb::type<ImVector<ImDrawChannel>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImDrawChannel> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImDrawVert =
        nb::class_<ImVector<ImDrawVert>>
            (m, "ImVector_ImDrawVert", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImDrawVert>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImDrawVert> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImDrawVert>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImDrawVert>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImDrawVert>::empty, "(private API)")
        .def("size",
            &ImVector<ImDrawVert>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawVert>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawVert>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImDrawVert>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImDrawVert>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImDrawVert>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImDrawVert> &v) {
                return nb::make_iterator(nb::type<ImVector<ImDrawVert>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImDrawVert> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImVec4 =
        nb::class_<ImVector<ImVec4>>
            (m, "ImVector_ImVec4", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImVec4>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImVec4> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImVec4>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImVec4>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImVec4>::empty, "(private API)")
        .def("size",
            &ImVector<ImVec4>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImVec4>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImVec4>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImVec4>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImVec4>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImVec4>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImVec4> &v) {
                return nb::make_iterator(nb::type<ImVector<ImVec4>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImVec4> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImVec2 =
        nb::class_<ImVector<ImVec2>>
            (m, "ImVector_ImVec2", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImVec2>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImVec2> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImVec2>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImVec2>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImVec2>::empty, "(private API)")
        .def("size",
            &ImVector<ImVec2>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImVec2>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImVec2>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImVec2>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImVec2>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImVec2>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImVec2> &v) {
                return nb::make_iterator(nb::type<ImVector<ImVec2>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImVec2> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImDrawList_ptr =
        nb::class_<ImVector<ImDrawList *>>
            (m, "ImVector_ImDrawList_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImDrawList *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImDrawList *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImDrawList *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImDrawList *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImDrawList *>::empty, "(private API)")
        .def("size",
            &ImVector<ImDrawList *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawList *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImDrawList *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImDrawList *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImDrawList *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImDrawList *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImDrawList *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImDrawList *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImDrawList *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImFont_ptr =
        nb::class_<ImVector<ImFont *>>
            (m, "ImVector_ImFont_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImFont *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImFont *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImFont *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImFont *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImFont *>::empty, "(private API)")
        .def("size",
            &ImVector<ImFont *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFont *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFont *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImFont *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImFont *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImFont *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImFont *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImFont *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImFont *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImFontGlyph =
        nb::class_<ImVector<ImFontGlyph>>
            (m, "ImVector_ImFontGlyph", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImFontGlyph>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImFontGlyph> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImFontGlyph>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImFontGlyph>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImFontGlyph>::empty, "(private API)")
        .def("size",
            &ImVector<ImFontGlyph>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontGlyph>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontGlyph>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImFontGlyph>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImFontGlyph>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImFontGlyph>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImFontGlyph> &v) {
                return nb::make_iterator(nb::type<ImVector<ImFontGlyph>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImFontGlyph> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiPlatformMonitor =
        nb::class_<ImVector<ImGuiPlatformMonitor>>
            (m, "ImVector_PlatformMonitor", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiPlatformMonitor>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiPlatformMonitor> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiPlatformMonitor>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiPlatformMonitor>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiPlatformMonitor>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiPlatformMonitor>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPlatformMonitor>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPlatformMonitor>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiPlatformMonitor>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiPlatformMonitor>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiPlatformMonitor>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiPlatformMonitor> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiPlatformMonitor>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiPlatformMonitor> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiViewport_ptr =
        nb::class_<ImVector<ImGuiViewport *>>
            (m, "ImVector_Viewport_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiViewport *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiViewport *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiViewport *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiViewport *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiViewport *>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiViewport *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiViewport *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiViewport *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiViewport *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiViewport *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiViewport *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiViewport *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiViewport *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiViewport *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiWindow_ptr =
        nb::class_<ImVector<ImGuiWindow *>>
            (m, "ImVector_Window_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiWindow *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiWindow *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiWindow *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiWindow *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiWindow *>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiWindow *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiWindow *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiWindow *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiWindow *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiWindow *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiWindow *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiWindow *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiWindow *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiWindow *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImFontConfig =
        nb::class_<ImVector<ImFontConfig>>
            (m, "ImVector_ImFontConfig", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImFontConfig>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImFontConfig> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImFontConfig>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImFontConfig>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImFontConfig>::empty, "(private API)")
        .def("size",
            &ImVector<ImFontConfig>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontConfig>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontConfig>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImFontConfig>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImFontConfig>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImFontConfig>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImFontConfig> &v) {
                return nb::make_iterator(nb::type<ImVector<ImFontConfig>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImFontConfig> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImFontConfig_ptr =
        nb::class_<ImVector<ImFontConfig *>>
            (m, "ImVector_ImFontConfig_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImFontConfig *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImFontConfig *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImFontConfig *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImFontConfig *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImFontConfig *>::empty, "(private API)")
        .def("size",
            &ImVector<ImFontConfig *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontConfig *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImFontConfig *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImFontConfig *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImFontConfig *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImFontConfig *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImFontConfig *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImFontConfig *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImFontConfig *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiFocusScopeData =
        nb::class_<ImVector<ImGuiFocusScopeData>>
            (m, "ImVector_FocusScopeData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiFocusScopeData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiFocusScopeData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiFocusScopeData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiFocusScopeData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiFocusScopeData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiFocusScopeData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiFocusScopeData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiFocusScopeData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiFocusScopeData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiFocusScopeData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiFocusScopeData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiFocusScopeData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiFocusScopeData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiFocusScopeData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiSelectionRequest =
        nb::class_<ImVector<ImGuiSelectionRequest>>
            (m, "ImVector_SelectionRequest", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiSelectionRequest>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiSelectionRequest> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiSelectionRequest>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiSelectionRequest>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiSelectionRequest>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiSelectionRequest>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiSelectionRequest>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiSelectionRequest>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiSelectionRequest>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiSelectionRequest>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiSelectionRequest>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiSelectionRequest> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiSelectionRequest>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiSelectionRequest> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImRect =
        nb::class_<ImVector<ImRect>>
            (m, "ImVector_ImRect", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImRect>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImRect> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImRect>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImRect>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImRect>::empty, "(private API)")
        .def("size",
            &ImVector<ImRect>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImRect>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImRect>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImRect>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImRect>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImRect>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImRect> &v) {
                return nb::make_iterator(nb::type<ImVector<ImRect>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImRect> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiColorMod =
        nb::class_<ImVector<ImGuiColorMod>>
            (m, "ImVector_ColorMod", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiColorMod>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiColorMod> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiColorMod>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiColorMod>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiColorMod>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiColorMod>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiColorMod>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiColorMod>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiColorMod>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiColorMod>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiColorMod>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiColorMod> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiColorMod>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiColorMod> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiGroupData =
        nb::class_<ImVector<ImGuiGroupData>>
            (m, "ImVector_GroupData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiGroupData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiGroupData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiGroupData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiGroupData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiGroupData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiGroupData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiGroupData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiGroupData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiGroupData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiGroupData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiGroupData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiGroupData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiGroupData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiGroupData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiPopupData =
        nb::class_<ImVector<ImGuiPopupData>>
            (m, "ImVector_PopupData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiPopupData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiPopupData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiPopupData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiPopupData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiPopupData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiPopupData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPopupData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPopupData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiPopupData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiPopupData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiPopupData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiPopupData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiPopupData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiPopupData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiViewportP_ptr =
        nb::class_<ImVector<ImGuiViewportP *>>
            (m, "ImVector_ViewportP_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiViewportP *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiViewportP *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiViewportP *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiViewportP *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiViewportP *>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiViewportP *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiViewportP *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiViewportP *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiViewportP *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiViewportP *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiViewportP *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiViewportP *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiViewportP *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiViewportP *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiInputEvent =
        nb::class_<ImVector<ImGuiInputEvent>>
            (m, "ImVector_InputEvent", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiInputEvent>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiInputEvent> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiInputEvent>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiInputEvent>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiInputEvent>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiInputEvent>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiInputEvent>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiInputEvent>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiInputEvent>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiInputEvent>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiInputEvent>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiInputEvent> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiInputEvent>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiInputEvent> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiWindowStackData =
        nb::class_<ImVector<ImGuiWindowStackData>>
            (m, "ImVector_WindowStackData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiWindowStackData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiWindowStackData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiWindowStackData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiWindowStackData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiWindowStackData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiWindowStackData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiWindowStackData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiWindowStackData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiWindowStackData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiWindowStackData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiWindowStackData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiWindowStackData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiWindowStackData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiWindowStackData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTableColumnSortSpecs =
        nb::class_<ImVector<ImGuiTableColumnSortSpecs>>
            (m, "ImVector_TableColumnSortSpecs", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTableColumnSortSpecs>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTableColumnSortSpecs> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTableColumnSortSpecs>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTableColumnSortSpecs>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTableColumnSortSpecs>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTableColumnSortSpecs>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableColumnSortSpecs>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableColumnSortSpecs>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTableColumnSortSpecs>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTableColumnSortSpecs>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTableColumnSortSpecs>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTableColumnSortSpecs> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTableColumnSortSpecs>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTableColumnSortSpecs> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTableInstanceData =
        nb::class_<ImVector<ImGuiTableInstanceData>>
            (m, "ImVector_TableInstanceData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTableInstanceData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTableInstanceData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTableInstanceData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTableInstanceData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTableInstanceData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTableInstanceData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableInstanceData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableInstanceData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTableInstanceData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTableInstanceData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTableInstanceData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTableInstanceData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTableInstanceData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTableInstanceData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTableTempData =
        nb::class_<ImVector<ImGuiTableTempData>>
            (m, "ImVector_TableTempData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTableTempData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTableTempData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTableTempData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTableTempData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTableTempData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTableTempData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableTempData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableTempData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTableTempData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTableTempData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTableTempData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTableTempData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTableTempData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTableTempData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiPtrOrIndex =
        nb::class_<ImVector<ImGuiPtrOrIndex>>
            (m, "ImVector_PtrOrIndex", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiPtrOrIndex>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiPtrOrIndex> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiPtrOrIndex>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiPtrOrIndex>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiPtrOrIndex>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiPtrOrIndex>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPtrOrIndex>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiPtrOrIndex>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiPtrOrIndex>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiPtrOrIndex>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiPtrOrIndex>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiPtrOrIndex> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiPtrOrIndex>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiPtrOrIndex> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiSettingsHandler =
        nb::class_<ImVector<ImGuiSettingsHandler>>
            (m, "ImVector_SettingsHandler", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiSettingsHandler>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiSettingsHandler> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiSettingsHandler>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiSettingsHandler>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiSettingsHandler>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiSettingsHandler>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiSettingsHandler>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiSettingsHandler>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiSettingsHandler>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiSettingsHandler>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiSettingsHandler>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiSettingsHandler> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiSettingsHandler>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiSettingsHandler> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiShrinkWidthItem =
        nb::class_<ImVector<ImGuiShrinkWidthItem>>
            (m, "ImVector_ShrinkWidthItem", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiShrinkWidthItem>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiShrinkWidthItem> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiShrinkWidthItem>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiShrinkWidthItem>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiShrinkWidthItem>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiShrinkWidthItem>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiShrinkWidthItem>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiShrinkWidthItem>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiShrinkWidthItem>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiShrinkWidthItem>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiShrinkWidthItem>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiShrinkWidthItem> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiShrinkWidthItem>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiShrinkWidthItem> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiStackLevelInfo =
        nb::class_<ImVector<ImGuiStackLevelInfo>>
            (m, "ImVector_StackLevelInfo", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiStackLevelInfo>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiStackLevelInfo> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiStackLevelInfo>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiStackLevelInfo>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiStackLevelInfo>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiStackLevelInfo>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiStackLevelInfo>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiStackLevelInfo>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiStackLevelInfo>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiStackLevelInfo>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiStackLevelInfo>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiStackLevelInfo> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiStackLevelInfo>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiStackLevelInfo> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTabItem =
        nb::class_<ImVector<ImGuiTabItem>>
            (m, "ImVector_TabItem", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTabItem>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTabItem> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTabItem>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTabItem>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTabItem>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTabItem>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTabItem>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTabItem>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTabItem>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTabItem>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTabItem>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTabItem> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTabItem>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTabItem> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiKeyRoutingData =
        nb::class_<ImVector<ImGuiKeyRoutingData>>
            (m, "ImVector_KeyRoutingData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiKeyRoutingData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiKeyRoutingData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiKeyRoutingData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiKeyRoutingData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiKeyRoutingData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiKeyRoutingData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiKeyRoutingData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiKeyRoutingData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiKeyRoutingData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiKeyRoutingData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiKeyRoutingData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiKeyRoutingData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiKeyRoutingData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiKeyRoutingData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiListClipperData =
        nb::class_<ImVector<ImGuiListClipperData>>
            (m, "ImVector_ListClipperData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiListClipperData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiListClipperData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiListClipperData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiListClipperData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiListClipperData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiListClipperData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiListClipperData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiListClipperData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiListClipperData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiListClipperData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiListClipperData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiListClipperData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiListClipperData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiListClipperData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiListClipperRange =
        nb::class_<ImVector<ImGuiListClipperRange>>
            (m, "ImVector_ListClipperRange", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiListClipperRange>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiListClipperRange> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiListClipperRange>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiListClipperRange>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiListClipperRange>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiListClipperRange>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiListClipperRange>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiListClipperRange>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiListClipperRange>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiListClipperRange>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiListClipperRange>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiListClipperRange> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiListClipperRange>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiListClipperRange> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiOldColumnData =
        nb::class_<ImVector<ImGuiOldColumnData>>
            (m, "ImVector_OldColumnData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiOldColumnData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiOldColumnData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiOldColumnData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiOldColumnData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiOldColumnData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiOldColumnData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiOldColumnData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiOldColumnData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiOldColumnData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiOldColumnData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiOldColumnData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiOldColumnData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiOldColumnData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiOldColumnData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiOldColumns =
        nb::class_<ImVector<ImGuiOldColumns>>
            (m, "ImVector_OldColumns", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiOldColumns>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiOldColumns> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiOldColumns>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiOldColumns>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiOldColumns>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiOldColumns>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiOldColumns>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiOldColumns>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiOldColumns>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiOldColumns>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiOldColumns>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiOldColumns> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiOldColumns>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiOldColumns> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiStyleMod =
        nb::class_<ImVector<ImGuiStyleMod>>
            (m, "ImVector_StyleMod", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiStyleMod>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiStyleMod> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiStyleMod>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiStyleMod>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiStyleMod>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiStyleMod>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiStyleMod>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiStyleMod>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiStyleMod>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiStyleMod>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiStyleMod>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiStyleMod> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiStyleMod>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiStyleMod> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTableHeaderData =
        nb::class_<ImVector<ImGuiTableHeaderData>>
            (m, "ImVector_TableHeaderData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTableHeaderData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTableHeaderData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTableHeaderData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTableHeaderData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTableHeaderData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTableHeaderData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableHeaderData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTableHeaderData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTableHeaderData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTableHeaderData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTableHeaderData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTableHeaderData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTableHeaderData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTableHeaderData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiTreeNodeStackData =
        nb::class_<ImVector<ImGuiTreeNodeStackData>>
            (m, "ImVector_TreeNodeStackData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiTreeNodeStackData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiTreeNodeStackData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiTreeNodeStackData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiTreeNodeStackData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiTreeNodeStackData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiTreeNodeStackData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTreeNodeStackData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiTreeNodeStackData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiTreeNodeStackData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiTreeNodeStackData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiTreeNodeStackData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiTreeNodeStackData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiTreeNodeStackData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiTreeNodeStackData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImGuiMultiSelectTempData =
        nb::class_<ImVector<ImGuiMultiSelectTempData>>
            (m, "ImVector_MultiSelectTempData", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImGuiMultiSelectTempData>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImGuiMultiSelectTempData> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImGuiMultiSelectTempData>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImGuiMultiSelectTempData>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImGuiMultiSelectTempData>::empty, "(private API)")
        .def("size",
            &ImVector<ImGuiMultiSelectTempData>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiMultiSelectTempData>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImGuiMultiSelectTempData>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImGuiMultiSelectTempData>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImGuiMultiSelectTempData>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImGuiMultiSelectTempData>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImGuiMultiSelectTempData> &v) {
                return nb::make_iterator(nb::type<ImVector<ImGuiMultiSelectTempData>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImGuiMultiSelectTempData> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImTextureData_ptr =
        nb::class_<ImVector<ImTextureData *>>
            (m, "ImVector_ImTextureData_ptr", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImTextureData *>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImTextureData *> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImTextureData *>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImTextureData *>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImTextureData *>::empty, "(private API)")
        .def("size",
            &ImVector<ImTextureData *>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureData *>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureData *>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImTextureData *>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImTextureData *>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImTextureData *>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImTextureData *> &v) {
                return nb::make_iterator(nb::type<ImVector<ImTextureData *>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImTextureData *> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImTextureRef =
        nb::class_<ImVector<ImTextureRef>>
            (m, "ImVector_ImTextureRef", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImTextureRef>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImTextureRef> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImTextureRef>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImTextureRef>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImTextureRef>::empty, "(private API)")
        .def("size",
            &ImVector<ImTextureRef>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureRef>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureRef>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImTextureRef>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImTextureRef>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImTextureRef>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImTextureRef> &v) {
                return nb::make_iterator(nb::type<ImVector<ImTextureRef>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImTextureRef> &v) { return v.size(); })
        ;
    auto pyClassImVector_ImTextureRect =
        nb::class_<ImVector<ImTextureRect>>
            (m, "ImVector_ImTextureRect", "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("data_address",
            &ImVector<ImTextureRect>::DataAddress, "(private API)")
        // #endif
        //
        .def(nb::init<>())
        .def(nb::init<const ImVector<ImTextureRect> &>(),
            nb::arg("src"))
        .def("clear",
            &ImVector<ImTextureRect>::clear, " Important: does not destruct anything\n(private API)")
        .def("clear_destruct",
            &ImVector<ImTextureRect>::clear_destruct, " Important: never called automatically! always explicit.\n(private API)")
        .def("empty",
            &ImVector<ImTextureRect>::empty, "(private API)")
        .def("size",
            &ImVector<ImTextureRect>::size, "(private API)")
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureRect>::operator[], nb::const_),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<int>(&ImVector<ImTextureRect>::operator[]),
            nb::arg("i"),
            "(private API)",
            nb::rv_policy::reference)
        .def("push_back",
            &ImVector<ImTextureRect>::push_back,
            nb::arg("v"),
            "(private API)")
        .def("pop_back",
            &ImVector<ImTextureRect>::pop_back, "(private API)")
        .def("push_front",
            &ImVector<ImTextureRect>::push_front,
            nb::arg("v"),
            "(private API)")
        .def("__iter__", [](const ImVector<ImTextureRect> &v) {
                return nb::make_iterator(nb::type<ImVector<ImTextureRect>>(), "iterator", v.begin(), v.end());
            }, nb::keep_alive<0, 1>())
        .def("__len__", [](const ImVector<ImTextureRect> &v) { return v.size(); })
        ;


    auto pyClassImGuiStyle =
        nb::class_<ImGuiStyle>
            (m, "Style", "")
        .def_rw("font_size_base", &ImGuiStyle::FontSizeBase, "Current base font size before external global factors are applied. Use PushFont(None, size) to modify. Use ImGui::GetFontSize() to obtain scaled value.")
        .def_rw("font_scale_main", &ImGuiStyle::FontScaleMain, "Main global scale factor. May be set by application once, or exposed to end-user.")
        .def_rw("font_scale_dpi", &ImGuiStyle::FontScaleDpi, "Additional global scale factor from viewport/monitor contents scale. When io.ConfigDpiScaleFonts is enabled, this is automatically overwritten when changing monitor DPI.")
        .def_rw("alpha", &ImGuiStyle::Alpha, "Global alpha applies to everything in Dear ImGui.")
        .def_rw("disabled_alpha", &ImGuiStyle::DisabledAlpha, "Additional alpha multiplier applied by BeginDisabled(). Multiply over current value of Alpha.")
        .def_rw("window_padding", &ImGuiStyle::WindowPadding, "Padding within a window.")
        .def_rw("window_rounding", &ImGuiStyle::WindowRounding, "Radius of window corners rounding. Set to 0.0 to have rectangular windows. Large values tend to lead to variety of artifacts and are not recommended.")
        .def_rw("window_border_size", &ImGuiStyle::WindowBorderSize, "Thickness of border around windows. Generally set to 0.0 or 1.0. (Other values are not well tested and more CPU/GPU costly).")
        .def_rw("window_border_hover_padding", &ImGuiStyle::WindowBorderHoverPadding, "Hit-testing extent outside/inside resizing border. Also extend determination of hovered window. Generally meaningfully larger than WindowBorderSize to make it easy to reach borders.")
        .def_rw("window_min_size", &ImGuiStyle::WindowMinSize, "Minimum window size. This is a global setting. If you want to constrain individual windows, use SetNextWindowSizeConstraints().")
        .def_rw("window_title_align", &ImGuiStyle::WindowTitleAlign, "Alignment for title bar text. Defaults to (0.0,0.5) for left-aligned,vertically centered.")
        .def_rw("window_menu_button_position", &ImGuiStyle::WindowMenuButtonPosition, "Side of the collapsing/docking button in the title bar (None/Left/Right). Defaults to ImGuiDir_Left.")
        .def_rw("child_rounding", &ImGuiStyle::ChildRounding, "Radius of child window corners rounding. Set to 0.0 to have rectangular windows.")
        .def_rw("child_border_size", &ImGuiStyle::ChildBorderSize, "Thickness of border around child windows. Generally set to 0.0 or 1.0. (Other values are not well tested and more CPU/GPU costly).")
        .def_rw("popup_rounding", &ImGuiStyle::PopupRounding, "Radius of popup window corners rounding. (Note that tooltip windows use WindowRounding)")
        .def_rw("popup_border_size", &ImGuiStyle::PopupBorderSize, "Thickness of border around popup/tooltip windows. Generally set to 0.0 or 1.0. (Other values are not well tested and more CPU/GPU costly).")
        .def_rw("frame_padding", &ImGuiStyle::FramePadding, "Padding within a framed rectangle (used by most widgets).")
        .def_rw("frame_rounding", &ImGuiStyle::FrameRounding, "Radius of frame corners rounding. Set to 0.0 to have rectangular frame (used by most widgets).")
        .def_rw("frame_border_size", &ImGuiStyle::FrameBorderSize, "Thickness of border around frames. Generally set to 0.0 or 1.0. (Other values are not well tested and more CPU/GPU costly).")
        .def_rw("item_spacing", &ImGuiStyle::ItemSpacing, "Horizontal and vertical spacing between widgets/lines.")
        .def_rw("item_inner_spacing", &ImGuiStyle::ItemInnerSpacing, "Horizontal and vertical spacing between within elements of a composed widget (e.g. a slider and its label).")
        .def_rw("cell_padding", &ImGuiStyle::CellPadding, "Padding within a table cell. Cellpadding.x is locked for entire table. CellPadding.y may be altered between different rows.")
        .def_rw("touch_extra_padding", &ImGuiStyle::TouchExtraPadding, "Expand reactive bounding box for touch-based system where touch position is not accurate enough. Unfortunately we don't sort widgets so priority on overlap will always be given to the first widget. So don't grow this too much!")
        .def_rw("indent_spacing", &ImGuiStyle::IndentSpacing, "Horizontal indentation when e.g. entering a tree node. Generally == (FontSize + FramePadding.x*2).")
        .def_rw("columns_min_spacing", &ImGuiStyle::ColumnsMinSpacing, "Minimum horizontal spacing between two columns. Preferably > (FramePadding.x + 1).")
        .def_rw("scrollbar_size", &ImGuiStyle::ScrollbarSize, "Width of the vertical scrollbar, Height of the horizontal scrollbar.")
        .def_rw("scrollbar_rounding", &ImGuiStyle::ScrollbarRounding, "Radius of grab corners for scrollbar.")
        .def_rw("scrollbar_padding", &ImGuiStyle::ScrollbarPadding, "Padding of scrollbar grab within its frame (same for both axises).")
        .def_rw("grab_min_size", &ImGuiStyle::GrabMinSize, "Minimum width/height of a grab box for slider/scrollbar.")
        .def_rw("grab_rounding", &ImGuiStyle::GrabRounding, "Radius of grabs corners rounding. Set to 0.0 to have rectangular slider grabs.")
        .def_rw("layout_align", &ImGuiStyle::LayoutAlign, "Element alignment inside horizontal and vertical layouts (0.0 - left/top, 1.0 - right/bottom, 0.5 - center).")
        .def_rw("log_slider_deadzone", &ImGuiStyle::LogSliderDeadzone, "The size in pixels of the dead-zone around zero on logarithmic sliders that cross zero.")
        .def_rw("image_border_size", &ImGuiStyle::ImageBorderSize, "Thickness of border around Image() calls.")
        .def_rw("tab_rounding", &ImGuiStyle::TabRounding, "Radius of upper corners of a tab. Set to 0.0 to have rectangular tabs.")
        .def_rw("tab_border_size", &ImGuiStyle::TabBorderSize, "Thickness of border around tabs.")
        .def_rw("tab_min_width_base", &ImGuiStyle::TabMinWidthBase, "Minimum tab width, to make tabs larger than their contents. TabBar buttons are not affected.")
        .def_rw("tab_min_width_shrink", &ImGuiStyle::TabMinWidthShrink, "Minimum tab width after shrinking, when using ImGuiTabBarFlags_FittingPolicyMixed policy.")
        .def_rw("tab_close_button_min_width_selected", &ImGuiStyle::TabCloseButtonMinWidthSelected, "-1: always visible. 0.0: visible when hovered. >0.0: visible when hovered if minimum width.")
        .def_rw("tab_close_button_min_width_unselected", &ImGuiStyle::TabCloseButtonMinWidthUnselected, "-1: always visible. 0.0: visible when hovered. >0.0: visible when hovered if minimum width. FLT_MAX: never show close button when unselected.")
        .def_rw("tab_bar_border_size", &ImGuiStyle::TabBarBorderSize, "Thickness of tab-bar separator, which takes on the tab active color to denote focus.")
        .def_rw("tab_bar_overline_size", &ImGuiStyle::TabBarOverlineSize, "Thickness of tab-bar overline, which highlights the selected tab-bar.")
        .def_rw("table_angled_headers_angle", &ImGuiStyle::TableAngledHeadersAngle, "Angle of angled headers (supported values range from -50.0 degrees to +50.0 degrees).")
        .def_rw("table_angled_headers_text_align", &ImGuiStyle::TableAngledHeadersTextAlign, "Alignment of angled headers within the cell")
        .def_rw("tree_lines_flags", &ImGuiStyle::TreeLinesFlags, "Default way to draw lines connecting TreeNode hierarchy. ImGuiTreeNodeFlags_DrawLinesNone or ImGuiTreeNodeFlags_DrawLinesFull or ImGuiTreeNodeFlags_DrawLinesToNodes.")
        .def_rw("tree_lines_size", &ImGuiStyle::TreeLinesSize, "Thickness of outlines when using ImGuiTreeNodeFlags_DrawLines.")
        .def_rw("tree_lines_rounding", &ImGuiStyle::TreeLinesRounding, "Radius of lines connecting child nodes to the vertical line.")
        .def_rw("color_button_position", &ImGuiStyle::ColorButtonPosition, "Side of the color button in the ColorEdit4 widget (left/right). Defaults to ImGuiDir_Right.")
        .def_rw("button_text_align", &ImGuiStyle::ButtonTextAlign, "Alignment of button text when button is larger than text. Defaults to (0.5, 0.5) (centered).")
        .def_rw("selectable_text_align", &ImGuiStyle::SelectableTextAlign, "Alignment of selectable text. Defaults to (0.0, 0.0) (top-left aligned). It's generally important to keep this left-aligned if you want to lay multiple items on a same line.")
        .def_rw("separator_text_border_size", &ImGuiStyle::SeparatorTextBorderSize, "Thickness of border in SeparatorText()")
        .def_rw("separator_text_align", &ImGuiStyle::SeparatorTextAlign, "Alignment of text within the separator. Defaults to (0.0, 0.5) (left aligned, center).")
        .def_rw("separator_text_padding", &ImGuiStyle::SeparatorTextPadding, "Horizontal offset of text from each edge of the separator + spacing on other axis. Generally small values. .y is recommended to be == FramePadding.y.")
        .def_rw("display_window_padding", &ImGuiStyle::DisplayWindowPadding, "Apply to regular windows: amount which we enforce to keep visible when moving near edges of your screen.")
        .def_rw("display_safe_area_padding", &ImGuiStyle::DisplaySafeAreaPadding, "Apply to every windows, menus, popups, tooltips: amount where we avoid displaying contents. Adjust if you cannot see the edges of your screen (e.g. on a TV where scaling has not been configured).")
        .def_rw("docking_node_has_close_button", &ImGuiStyle::DockingNodeHasCloseButton, "Docking node has their own CloseButton() to close all docked windows.")
        .def_rw("docking_separator_size", &ImGuiStyle::DockingSeparatorSize, "Thickness of resizing border between docked windows")
        .def_rw("mouse_cursor_scale", &ImGuiStyle::MouseCursorScale, "Scale software rendered mouse cursor (when io.MouseDrawCursor is enabled). We apply per-monitor DPI scaling over this scale. May be removed later.")
        .def_rw("anti_aliased_lines", &ImGuiStyle::AntiAliasedLines, "Enable anti-aliased lines/borders. Disable if you are really tight on CPU/GPU. Latched at the beginning of the frame (copied to ImDrawList).")
        .def_rw("anti_aliased_lines_use_tex", &ImGuiStyle::AntiAliasedLinesUseTex, "Enable anti-aliased lines/borders using textures where possible. Require backend to render with bilinear filtering (NOT point/nearest filtering). Latched at the beginning of the frame (copied to ImDrawList).")
        .def_rw("anti_aliased_fill", &ImGuiStyle::AntiAliasedFill, "Enable anti-aliased edges around filled shapes (rounded rectangles, circles, etc.). Disable if you are really tight on CPU/GPU. Latched at the beginning of the frame (copied to ImDrawList).")
        .def_rw("curve_tessellation_tol", &ImGuiStyle::CurveTessellationTol, "Tessellation tolerance when using PathBezierCurveTo() without a specific number of segments. Decrease for highly tessellated curves (higher quality, more polygons), increase to reduce quality.")
        .def_rw("circle_tessellation_max_error", &ImGuiStyle::CircleTessellationMaxError, "Maximum error (in pixels) allowed when using AddCircle()/AddCircleFilled() or drawing rounded corner rectangles with no explicit segment count specified. Decrease for higher quality but more geometry.")
        .def_rw("hover_stationary_delay", &ImGuiStyle::HoverStationaryDelay, "Delay for IsItemHovered(ImGuiHoveredFlags_Stationary). Time required to consider mouse stationary.")
        .def_rw("hover_delay_short", &ImGuiStyle::HoverDelayShort, "Delay for IsItemHovered(ImGuiHoveredFlags_DelayShort). Usually used along with HoverStationaryDelay.")
        .def_rw("hover_delay_normal", &ImGuiStyle::HoverDelayNormal, "Delay for IsItemHovered(ImGuiHoveredFlags_DelayNormal). \"")
        .def_rw("hover_flags_for_tooltip_mouse", &ImGuiStyle::HoverFlagsForTooltipMouse, "Default flags when using IsItemHovered(ImGuiHoveredFlags_ForTooltip) or BeginItemTooltip()/SetItemTooltip() while using mouse.")
        .def_rw("hover_flags_for_tooltip_nav", &ImGuiStyle::HoverFlagsForTooltipNav, "Default flags when using IsItemHovered(ImGuiHoveredFlags_ForTooltip) or BeginItemTooltip()/SetItemTooltip() while using keyboard/gamepad.")
        .def_rw("_main_scale", &ImGuiStyle::_MainScale, "FIXME-WIP: Reference scale, as applied by ScaleAllSizes().")
        .def_rw("_next_frame_font_size_base", &ImGuiStyle::_NextFrameFontSizeBase, "FIXME: Temporary hack until we finish remaining work.")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("color_",
            &ImGuiStyle::Color_,
            nb::arg("idx_color"),
            nb::rv_policy::reference)
        .def("set_color_",
            &ImGuiStyle::SetColor_, nb::arg("idx_color"), nb::arg("color"))
        // #endif
        //
        .def(nb::init<>())
        .def("scale_all_sizes",
            &ImGuiStyle::ScaleAllSizes, nb::arg("scale_factor"))
        ;


    auto pyClassImGuiKeyData =
        nb::class_<ImGuiKeyData>
            (m, "KeyData", " [Internal] Storage used by IsKeyDown(), IsKeyPressed() etc functions.\n If prior to 1.87 you used io.KeysDownDuration[] (which was marked as internal), you should use GetKeyData(key)->DownDuration and *NOT* io.KeysData[key]->DownDuration.")
        .def("__init__", [](ImGuiKeyData * self, bool Down = bool(), float DownDuration = float(), float DownDurationPrev = float(), float AnalogValue = float())
        {
            new (self) ImGuiKeyData();  // placement new
            auto r_ctor_ = self;
            r_ctor_->Down = Down;
            r_ctor_->DownDuration = DownDuration;
            r_ctor_->DownDurationPrev = DownDurationPrev;
            r_ctor_->AnalogValue = AnalogValue;
        },
        nb::arg("down") = bool(), nb::arg("down_duration") = float(), nb::arg("down_duration_prev") = float(), nb::arg("analog_value") = float()
        )
        .def_rw("down", &ImGuiKeyData::Down, "True for if key is down")
        .def_rw("down_duration", &ImGuiKeyData::DownDuration, "Duration the key has been down (<0.0: not pressed, 0.0: just pressed, >0.0: time held)")
        .def_rw("down_duration_prev", &ImGuiKeyData::DownDurationPrev, "Last frame duration the key has been down")
        .def_rw("analog_value", &ImGuiKeyData::AnalogValue, "0.0..1.0 for gamepad values")
        ;


    auto pyClassImGuiIO =
        nb::class_<ImGuiIO>
            (m, "IO", "")
        .def_rw("config_flags", &ImGuiIO::ConfigFlags, "= 0              // See ImGuiConfigFlags_ enum. Set by user/application. Keyboard/Gamepad navigation options, etc.")
        .def_rw("backend_flags", &ImGuiIO::BackendFlags, "= 0              // See ImGuiBackendFlags_ enum. Set by backend (imgui_impl_xxx files or custom backend) to communicate features supported by the backend.")
        .def_rw("display_size", &ImGuiIO::DisplaySize, "<unset>          // Main display size, in pixels (== GetMainViewport()->Size). May change every frame.")
        .def_rw("display_framebuffer_scale", &ImGuiIO::DisplayFramebufferScale, "= (1, 1)         // Main display density. For retina display where window coordinates are different from framebuffer coordinates. This will affect font density + will end up in ImDrawData::FramebufferScale.")
        .def_rw("delta_time", &ImGuiIO::DeltaTime, "= 1.0/60.0     // Time elapsed since last frame, in seconds. May change every frame.")
        .def_rw("ini_saving_rate", &ImGuiIO::IniSavingRate, "= 5.0           // Minimum time between saving positions/sizes to .ini file, in seconds.")
        .def_rw("user_data", &ImGuiIO::UserData, "= None           // Store your own data.")
        .def_rw("fonts", &ImGuiIO::Fonts, "<auto>           // Font atlas: load, rasterize and pack one or more fonts into a single texture.")
        .def_rw("font_default", &ImGuiIO::FontDefault, "= None           // Font to use on NewFrame(). Use None to uses Fonts->Fonts[0].")
        .def_rw("font_allow_user_scaling", &ImGuiIO::FontAllowUserScaling, "= False          // [OBSOLETE] Allow user scaling text of individual window with CTRL+Wheel.")
        .def_rw("config_nav_swap_gamepad_buttons", &ImGuiIO::ConfigNavSwapGamepadButtons, "= False          // Swap Activate<>Cancel (A<>B) buttons, matching typical \"Nintendo/Japanese style\" gamepad layout.")
        .def_rw("config_nav_move_set_mouse_pos", &ImGuiIO::ConfigNavMoveSetMousePos, "= False          // Directional/tabbing navigation teleports the mouse cursor. May be useful on TV/console systems where moving a virtual mouse is difficult. Will update io.MousePos and set io.WantSetMousePos=True.")
        .def_rw("config_nav_capture_keyboard", &ImGuiIO::ConfigNavCaptureKeyboard, "= True           // Sets io.WantCaptureKeyboard when io.NavActive is set.")
        .def_rw("config_nav_escape_clear_focus_item", &ImGuiIO::ConfigNavEscapeClearFocusItem, "= True           // Pressing Escape can clear focused item + navigation id/highlight. Set to False if you want to always keep highlight on.")
        .def_rw("config_nav_escape_clear_focus_window", &ImGuiIO::ConfigNavEscapeClearFocusWindow, "= False          // Pressing Escape can clear focused window as well (super set of io.ConfigNavEscapeClearFocusItem).")
        .def_rw("config_nav_cursor_visible_auto", &ImGuiIO::ConfigNavCursorVisibleAuto, "= True           // Using directional navigation key makes the cursor visible. Mouse click hides the cursor.")
        .def_rw("config_nav_cursor_visible_always", &ImGuiIO::ConfigNavCursorVisibleAlways, "= False          // Navigation cursor is always visible.")
        .def_rw("config_docking_no_split", &ImGuiIO::ConfigDockingNoSplit, "= False          // Simplified docking mode: disable window splitting, so docking is limited to merging multiple windows together into tab-bars.")
        .def_rw("config_docking_with_shift", &ImGuiIO::ConfigDockingWithShift, "= False          // Enable docking with holding Shift key (reduce visual noise, allows dropping in wider space)")
        .def_rw("config_docking_always_tab_bar", &ImGuiIO::ConfigDockingAlwaysTabBar, "= False          // [BETA] [FIXME: This currently creates regression with auto-sizing and general overhead] Make every single floating window display within a docking node.")
        .def_rw("config_docking_transparent_payload", &ImGuiIO::ConfigDockingTransparentPayload, "= False          // [BETA] Make window or viewport transparent when docking and only display docking boxes on the target viewport. Useful if rendering of multiple viewport cannot be synced. Best used with ConfigViewportsNoAutoMerge.")
        .def_rw("config_viewports_no_auto_merge", &ImGuiIO::ConfigViewportsNoAutoMerge, "= False;         // Set to make all floating imgui windows always create their own viewport. Otherwise, they are merged into the main host viewports when overlapping it. May also set ImGuiViewportFlags_NoAutoMerge on individual viewport.")
        .def_rw("config_viewports_no_task_bar_icon", &ImGuiIO::ConfigViewportsNoTaskBarIcon, "= False          // Disable default OS task bar icon flag for secondary viewports. When a viewport doesn't want a task bar icon, ImGuiViewportFlags_NoTaskBarIcon will be set on it.")
        .def_rw("config_viewports_no_decoration", &ImGuiIO::ConfigViewportsNoDecoration, "= True           // Disable default OS window decoration flag for secondary viewports. When a viewport doesn't want window decorations, ImGuiViewportFlags_NoDecoration will be set on it. Enabling decoration can create subsequent issues at OS levels (e.g. minimum window size).")
        .def_rw("config_viewports_no_default_parent", &ImGuiIO::ConfigViewportsNoDefaultParent, "= False          // Disable default OS parenting to main viewport for secondary viewports. By default, viewports are marked with ParentViewportId = <main_viewport>, expecting the platform backend to setup a parent/child relationship between the OS windows (some backend may ignore this). Set to True if you want the default to be 0, then all viewports will be top-level OS windows.")
        .def_rw("config_viewport_platform_focus_sets_focus", &ImGuiIO::ConfigViewportPlatformFocusSetsImGuiFocus, "= True // When a platform window is focused (e.g. using Alt+Tab, clicking Platform Title Bar), apply corresponding focus on imgui windows (may clear focus/active id from imgui windows location in other platform windows). In principle this is better enabled but we provide an opt-out, because some Linux window managers tend to eagerly focus windows (e.g. on mouse hover, or even a simple window pos/size change).")
        .def_rw("config_dpi_scale_fonts", &ImGuiIO::ConfigDpiScaleFonts, "= False          // [EXPERIMENTAL] Automatically overwrite style.FontScaleDpi when Monitor DPI changes. This will scale fonts but _NOT_ scale sizes/padding for now.")
        .def_rw("config_dpi_scale_viewports", &ImGuiIO::ConfigDpiScaleViewports, "= False          // [EXPERIMENTAL] Scale Dear ImGui and Platform Windows when Monitor DPI changes.")
        .def_rw("mouse_draw_cursor", &ImGuiIO::MouseDrawCursor, "= False          // Request ImGui to draw a mouse cursor for you (if you are on a platform without a mouse cursor). Cannot be easily renamed to 'io.ConfigXXX' because this is frequently used by backend implementations.")
        .def_rw("config_mac_osx_behaviors", &ImGuiIO::ConfigMacOSXBehaviors, "= defined(__APPLE__) // Swap Cmd<>Ctrl keys + OS X style text editing cursor movement using Alt instead of Ctrl, Shortcuts using Cmd/Super instead of Ctrl, Line/Text Start and End using Cmd+Arrows instead of Home/End, Double click selects by word instead of selecting whole text, Multi-selection in lists uses Cmd/Super instead of Ctrl.")
        .def_rw("config_input_trickle_event_queue", &ImGuiIO::ConfigInputTrickleEventQueue, "= True           // Enable input queue trickling: some types of events submitted during the same frame (e.g. button down + up) will be spread over multiple frames, improving interactions with low framerates.")
        .def_rw("config_input_text_cursor_blink", &ImGuiIO::ConfigInputTextCursorBlink, "= True           // Enable blinking cursor (optional as some users consider it to be distracting).")
        .def_rw("config_input_text_enter_keep_active", &ImGuiIO::ConfigInputTextEnterKeepActive, "= False          // [BETA] Pressing Enter will keep item active and select contents (single-line only).")
        .def_rw("config_drag_click_to_input_text", &ImGuiIO::ConfigDragClickToInputText, "= False          // [BETA] Enable turning DragXXX widgets into text input with a simple mouse click-release (without moving). Not desirable on devices without a keyboard.")
        .def_rw("config_windows_resize_from_edges", &ImGuiIO::ConfigWindowsResizeFromEdges, "= True           // Enable resizing of windows from their edges and from the lower-left corner. This requires ImGuiBackendFlags_HasMouseCursors for better mouse cursor feedback. (This used to be a per-window ImGuiWindowFlags_ResizeFromAnySide flag)")
        .def_rw("config_windows_move_from_title_bar_only", &ImGuiIO::ConfigWindowsMoveFromTitleBarOnly, "= False      // Enable allowing to move windows only when clicking on their title bar. Does not apply to windows without a title bar.")
        .def_rw("config_windows_copy_contents_with_ctrl_c", &ImGuiIO::ConfigWindowsCopyContentsWithCtrlC, "= False      // [EXPERIMENTAL] CTRL+C copy the contents of focused window into the clipboard. Experimental because: (1) has known issues with nested Begin/End pairs (2) text output quality varies (3) text output is in submission order rather than spatial order.")
        .def_rw("config_scrollbar_scroll_by_page", &ImGuiIO::ConfigScrollbarScrollByPage, "= True           // Enable scrolling page by page when clicking outside the scrollbar grab. When disabled, always scroll to clicked location. When enabled, Shift+Click scrolls to clicked location.")
        .def_rw("config_memory_compact_timer", &ImGuiIO::ConfigMemoryCompactTimer, "= 60.0          // Timer (in seconds) to free transient windows/tables memory buffers when unused. Set to -1.0 to disable.")
        .def_rw("mouse_double_click_time", &ImGuiIO::MouseDoubleClickTime, "= 0.30          // Time for a double-click, in seconds.")
        .def_rw("mouse_double_click_max_dist", &ImGuiIO::MouseDoubleClickMaxDist, "= 6.0           // Distance threshold to stay in to validate a double-click, in pixels.")
        .def_rw("mouse_drag_threshold", &ImGuiIO::MouseDragThreshold, "= 6.0           // Distance threshold before considering we are dragging.")
        .def_rw("key_repeat_delay", &ImGuiIO::KeyRepeatDelay, "= 0.275         // When holding a key/button, time before it starts repeating, in seconds (for buttons in Repeat mode, etc.).")
        .def_rw("key_repeat_rate", &ImGuiIO::KeyRepeatRate, "= 0.050         // When holding a key/button, rate at which it repeats, in seconds.")
        .def_rw("config_error_recovery", &ImGuiIO::ConfigErrorRecovery, "= True       // Enable error recovery support. Some errors won't be detected and lead to direct crashes if recovery is disabled.")
        .def_rw("config_error_recovery_enable_assert", &ImGuiIO::ConfigErrorRecoveryEnableAssert, "= True       // Enable asserts on recoverable error. By default call IM_ASSERT() when returning from a failing IM_ASSERT_USER_ERROR()")
        .def_rw("config_error_recovery_enable_debug_log", &ImGuiIO::ConfigErrorRecoveryEnableDebugLog, "= True       // Enable debug log output on recoverable errors.")
        .def_rw("config_error_recovery_enable_tooltip", &ImGuiIO::ConfigErrorRecoveryEnableTooltip, "= True       // Enable tooltip on recoverable errors. The tooltip include a way to enable asserts if they were disabled.")
        .def_rw("config_debug_is_debugger_present", &ImGuiIO::ConfigDebugIsDebuggerPresent, "= False          // Enable various tools calling IM_DEBUG_BREAK().")
        .def_rw("config_debug_highlight_id_conflicts", &ImGuiIO::ConfigDebugHighlightIdConflicts, "= True           // Highlight and show an error message popup when multiple items have conflicting identifiers.")
        .def_rw("config_debug_highlight_id_conflicts_show_item_picker", &ImGuiIO::ConfigDebugHighlightIdConflictsShowItemPicker, "=True // Show \"Item Picker\" button in aforementioned popup.")
        .def_rw("config_debug_begin_return_value_once", &ImGuiIO::ConfigDebugBeginReturnValueOnce, "= False          // First-time calls to Begin()/BeginChild() will return False. NEEDS TO BE SET AT APPLICATION BOOT TIME if you don't want to miss windows.")
        .def_rw("config_debug_begin_return_value_loop", &ImGuiIO::ConfigDebugBeginReturnValueLoop, "= False          // Some calls to Begin()/BeginChild() will return False. Will cycle through window depths then repeat. Suggested use: add \"io.ConfigDebugBeginReturnValue = io.KeyShift\" in your main loop then occasionally press SHIFT. Windows should be flickering while running.")
        .def_rw("config_debug_ignore_focus_loss", &ImGuiIO::ConfigDebugIgnoreFocusLoss, "= False          // Ignore io.AddFocusEvent(False), consequently not calling io.ClearInputKeys()/io.ClearInputMouse() in input processing.")
        .def_rw("config_debug_ini_settings", &ImGuiIO::ConfigDebugIniSettings, "= False          // Save .ini data with extra comments (particularly helpful for Docking, but makes saving slower)")
        .def_ro("backend_platform_name", &ImGuiIO::BackendPlatformName, "= None")
        .def_ro("backend_renderer_name", &ImGuiIO::BackendRendererName, "= None")
        .def_rw("backend_platform_user_data", &ImGuiIO::BackendPlatformUserData, "= None           // User data for platform backend")
        .def_rw("backend_renderer_user_data", &ImGuiIO::BackendRendererUserData, "= None           // User data for renderer backend")
        .def_rw("backend_language_user_data", &ImGuiIO::BackendLanguageUserData, "= None           // User data for non C++ programming language backend")
        .def("add_key_event",
            &ImGuiIO::AddKeyEvent,
            nb::arg("key"), nb::arg("down"),
            "Queue a new key down/up event. Key should be \"translated\" (as in, generally ImGuiKey_A matches the key end-user would use to emit an 'A' character)")
        .def("add_key_analog_event",
            &ImGuiIO::AddKeyAnalogEvent,
            nb::arg("key"), nb::arg("down"), nb::arg("v"),
            "Queue a new key down/up event for analog values (e.g. ImGuiKey_Gamepad_ values). Dead-zones should be handled by the backend.")
        .def("add_mouse_pos_event",
            &ImGuiIO::AddMousePosEvent,
            nb::arg("x"), nb::arg("y"),
            "Queue a mouse position update. Use -FLT_MAX,-FLT_MAX to signify no mouse (e.g. app not focused and not hovered)")
        .def("add_mouse_button_event",
            &ImGuiIO::AddMouseButtonEvent,
            nb::arg("button"), nb::arg("down"),
            "Queue a mouse button change")
        .def("add_mouse_wheel_event",
            &ImGuiIO::AddMouseWheelEvent,
            nb::arg("wheel_x"), nb::arg("wheel_y"),
            "Queue a mouse wheel update. wheel_y<0: scroll down, wheel_y>0: scroll up, wheel_x<0: scroll right, wheel_x>0: scroll left.")
        .def("add_mouse_source_event",
            &ImGuiIO::AddMouseSourceEvent,
            nb::arg("source"),
            "Queue a mouse source change (Mouse/TouchScreen/Pen)")
        .def("add_mouse_viewport_event",
            &ImGuiIO::AddMouseViewportEvent,
            nb::arg("id_"),
            "Queue a mouse hovered viewport. Requires backend to set ImGuiBackendFlags_HasMouseHoveredViewport to call this (for multi-viewport support).")
        .def("add_focus_event",
            &ImGuiIO::AddFocusEvent,
            nb::arg("focused"),
            "Queue a gain/loss of focus for the application (generally based on OS/platform focus of your window)")
        .def("add_input_character",
            &ImGuiIO::AddInputCharacter,
            nb::arg("c"),
            "Queue a new character input")
        .def("add_input_character_utf16",
            &ImGuiIO::AddInputCharacterUTF16,
            nb::arg("c"),
            "Queue a new character input from a UTF-16 character, it can be a surrogate")
        .def("add_input_characters_utf8",
            &ImGuiIO::AddInputCharactersUTF8,
            nb::arg("str"),
            "Queue a new characters input from a UTF-8 string")
        .def("set_key_event_native_data",
            &ImGuiIO::SetKeyEventNativeData,
            nb::arg("key"), nb::arg("native_keycode"), nb::arg("native_scancode"), nb::arg("native_legacy_index") = -1,
            "[Optional] Specify index for legacy <1.87 IsKeyXXX() functions with native indices + specify native keycode, scancode.")
        .def("set_app_accepting_events",
            &ImGuiIO::SetAppAcceptingEvents,
            nb::arg("accepting_events"),
            "Set master flag for accepting key/mouse/text events (default to True). Useful if you have native dialog boxes that are interrupting your application loop/refresh, and you want to disable events being queued while your app is frozen.")
        .def("clear_events_queue",
            &ImGuiIO::ClearEventsQueue, "Clear all incoming events.")
        .def("clear_input_keys",
            &ImGuiIO::ClearInputKeys, "Clear current keyboard/gamepad state + current frame text input buffer. Equivalent to releasing all keys/buttons.")
        .def("clear_input_mouse",
            &ImGuiIO::ClearInputMouse, "Clear current mouse state.")
        .def_rw("want_capture_mouse", &ImGuiIO::WantCaptureMouse, "Set when Dear ImGui will use mouse inputs, in this case do not dispatch them to your main game/application (either way, always pass on mouse inputs to imgui). (e.g. unclicked mouse is hovering over an imgui window, widget is active, mouse was clicked over an imgui window, etc.).")
        .def_rw("want_capture_keyboard", &ImGuiIO::WantCaptureKeyboard, "Set when Dear ImGui will use keyboard inputs, in this case do not dispatch them to your main game/application (either way, always pass keyboard inputs to imgui). (e.g. InputText active, or an imgui window is focused and navigation is enabled, etc.).")
        .def_rw("want_text_input", &ImGuiIO::WantTextInput, "Mobile/console: when set, you may display an on-screen keyboard. This is set by Dear ImGui when it wants textual keyboard input to happen (e.g. when a InputText widget is active).")
        .def_rw("want_set_mouse_pos", &ImGuiIO::WantSetMousePos, "MousePos has been altered, backend should reposition mouse on next frame. Rarely used! Set only when io.ConfigNavMoveSetMousePos is enabled.")
        .def_rw("want_save_ini_settings", &ImGuiIO::WantSaveIniSettings, "When manual .ini load/save is active (io.IniFilename == None), this will be set to notify your application that you can call SaveIniSettingsToMemory() and save yourself. Important: clear io.WantSaveIniSettings yourself after saving!")
        .def_rw("nav_active", &ImGuiIO::NavActive, "Keyboard/Gamepad navigation is currently allowed (will handle ImGuiKey_NavXXX events) = a window is focused and it doesn't use the ImGuiWindowFlags_NoNavInputs flag.")
        .def_rw("nav_visible", &ImGuiIO::NavVisible, "Keyboard/Gamepad navigation highlight is visible and allowed (will handle ImGuiKey_NavXXX events).")
        .def_rw("framerate", &ImGuiIO::Framerate, "Estimate of application framerate (rolling average over 60 frames, based on io.DeltaTime), in frame per second. Solely for convenience. Slow applications may not want to use a moving average or may want to reset underlying buffers occasionally.")
        .def_rw("metrics_render_vertices", &ImGuiIO::MetricsRenderVertices, "Vertices output during last call to Render()")
        .def_rw("metrics_render_indices", &ImGuiIO::MetricsRenderIndices, "Indices output during last call to Render() = number of triangles * 3")
        .def_rw("metrics_render_windows", &ImGuiIO::MetricsRenderWindows, "Number of visible windows")
        .def_rw("metrics_active_windows", &ImGuiIO::MetricsActiveWindows, "Number of active windows")
        .def_rw("mouse_delta", &ImGuiIO::MouseDelta, "Mouse delta. Note that this is zero if either current or previous position are invalid (-FLT_MAX,-FLT_MAX), so a disappearing/reappearing mouse won't have a huge delta.")
        .def_rw("ctx", &ImGuiIO::Ctx, "Parent UI context (needs to be set explicitly by parent).")
        .def_rw("mouse_pos", &ImGuiIO::MousePos, "Mouse position, in pixels. Set to ImVec2(-FLT_MAX, -FLT_MAX) if mouse is unavailable (on another screen, etc.)")
        .def_prop_ro("mouse_down",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDown;
            },
            "Mouse buttons: 0=left, 1=right, 2=middle + extras (ImGuiMouseButton_COUNT == 5). Dear ImGui mostly uses left and right buttons. Other buttons allow us to track if the mouse is being used by your application + available to user as a convenience via IsMouse** API.")
        .def_rw("mouse_wheel", &ImGuiIO::MouseWheel, "Mouse wheel Vertical: 1 unit scrolls about 5 lines text. >0 scrolls Up, <0 scrolls Down. Hold SHIFT to turn vertical scroll into horizontal scroll.")
        .def_rw("mouse_wheel_h", &ImGuiIO::MouseWheelH, "Mouse wheel Horizontal. >0 scrolls Left, <0 scrolls Right. Most users don't have a mouse with a horizontal wheel, may not be filled by all backends.")
        .def_rw("mouse_source", &ImGuiIO::MouseSource, "Mouse actual input peripheral (Mouse/TouchScreen/Pen).")
        .def_rw("mouse_hovered_viewport", &ImGuiIO::MouseHoveredViewport, "(Optional) Modify using io.AddMouseViewportEvent(). With multi-viewports: viewport the OS mouse is hovering. If possible _IGNORING_ viewports with the ImGuiViewportFlags_NoInputs flag is much better (few backends can handle that). Set io.BackendFlags |= ImGuiBackendFlags_HasMouseHoveredViewport if you can provide this info. If you don't imgui will infer the value using the rectangles and last focused time of the viewports it knows about (ignoring other OS windows).")
        .def_rw("key_ctrl", &ImGuiIO::KeyCtrl, "Keyboard modifier down: Ctrl (non-macOS), Cmd (macOS)")
        .def_rw("key_shift", &ImGuiIO::KeyShift, "Keyboard modifier down: Shift")
        .def_rw("key_alt", &ImGuiIO::KeyAlt, "Keyboard modifier down: Alt")
        .def_rw("key_super", &ImGuiIO::KeySuper, "Keyboard modifier down: Windows/Super (non-macOS), Ctrl (macOS)")
        .def_rw("key_mods", &ImGuiIO::KeyMods, "Key mods flags (any of ImGuiMod_Ctrl/ImGuiMod_Shift/ImGuiMod_Alt/ImGuiMod_Super flags, same as io.KeyCtrl/KeyShift/KeyAlt/KeySuper but merged into flags). Read-only, updated by NewFrame()")
        .def_rw("want_capture_mouse_unless_popup_close", &ImGuiIO::WantCaptureMouseUnlessPopupClose, "Alternative to WantCaptureMouse: (WantCaptureMouse == True && WantCaptureMouseUnlessPopupClose == False) when a click over None is expected to close a popup.")
        .def_rw("mouse_pos_prev", &ImGuiIO::MousePosPrev, "Previous mouse position (note that MouseDelta is not necessary == MousePos-MousePosPrev, in case either position is invalid)")
        .def_prop_ro("mouse_clicked_time",
            [](ImGuiIO &self) -> nb::ndarray<double, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseClickedTime;
            },
            "Time of last click (used to figure out double-click)")
        .def_prop_ro("mouse_clicked",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseClicked;
            },
            "Mouse button went from !Down to Down (same as MouseClickedCount[x] != 0)")
        .def_prop_ro("mouse_double_clicked",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDoubleClicked;
            },
            "Has mouse button been double-clicked? (same as MouseClickedCount[x] == 2)")
        .def_prop_ro("mouse_clicked_count",
            [](ImGuiIO &self) -> nb::ndarray<ImU16, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseClickedCount;
            },
            "== 0 (not clicked), == 1 (same as MouseClicked[]), == 2 (double-clicked), == 3 (triple-clicked) etc. when going from !Down to Down")
        .def_prop_ro("mouse_clicked_last_count",
            [](ImGuiIO &self) -> nb::ndarray<ImU16, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseClickedLastCount;
            },
            "Count successive number of clicks. Stays valid after mouse release. Reset after another click is done.")
        .def_prop_ro("mouse_released",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseReleased;
            },
            "Mouse button went from Down to !Down")
        .def_prop_ro("mouse_released_time",
            [](ImGuiIO &self) -> nb::ndarray<double, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseReleasedTime;
            },
            "Time of last released (rarely used! but useful to handle delayed single-click when trying to disambiguate them from double-click).")
        .def_prop_ro("mouse_down_owned",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDownOwned;
            },
            "Track if button was clicked inside a dear imgui window or over None blocked by a popup. We don't request mouse capture from the application if click started outside ImGui bounds.")
        .def_prop_ro("mouse_down_owned_unless_popup_close",
            [](ImGuiIO &self) -> nb::ndarray<bool, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDownOwnedUnlessPopupClose;
            },
            "Track if button was clicked inside a dear imgui window.")
        .def_rw("mouse_wheel_request_axis_swap", &ImGuiIO::MouseWheelRequestAxisSwap, "On a non-Mac system, holding SHIFT requests WheelY to perform the equivalent of a WheelX event. On a Mac system this is already enforced by the system.")
        .def_rw("mouse_ctrl_left_as_right_click", &ImGuiIO::MouseCtrlLeftAsRightClick, "(OSX) Set to True when the current click was a Ctrl+click that spawned a simulated right click")
        .def_prop_ro("mouse_down_duration",
            [](ImGuiIO &self) -> nb::ndarray<float, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDownDuration;
            },
            "Duration the mouse button has been down (0.0 == just clicked)")
        .def_prop_ro("mouse_down_duration_prev",
            [](ImGuiIO &self) -> nb::ndarray<float, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDownDurationPrev;
            },
            "Previous time the mouse button has been down")
        .def_prop_ro("mouse_drag_max_distance_sqr",
            [](ImGuiIO &self) -> nb::ndarray<float, nb::numpy, nb::shape<5>, nb::c_contig>
            {
                return self.MouseDragMaxDistanceSqr;
            },
            "Squared maximum distance of how much mouse has traveled from the clicking point (used for moving thresholds)")
        .def_rw("pen_pressure", &ImGuiIO::PenPressure, "Touch/Pen pressure (0.0 to 1.0, should be >0.0 only when MouseDown[0] == True). Helper storage currently unused by Dear ImGui.")
        .def_rw("app_focus_lost", &ImGuiIO::AppFocusLost, "Only modify via AddFocusEvent()")
        .def_rw("app_accepting_events", &ImGuiIO::AppAcceptingEvents, "Only modify via SetAppAcceptingEvents()")
        .def_rw("input_queue_surrogate", &ImGuiIO::InputQueueSurrogate, "For AddInputCharacterUTF16()")
        .def_rw("input_queue_characters", &ImGuiIO::InputQueueCharacters, "Queue of _characters_ input (obtained by platform backend). Fill using AddInputCharacter() helper.")
        .def(nb::init<>())
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("set_ini_filename",
            &ImGuiIO::SetIniFilename,
            nb::arg("filename").none(),
            " - The disk functions are automatically called if IniFilename != None\n - Set IniFilename to None to load/save manually. Read io.WantSaveIniSettings description about handling .ini saving manually.\n - Important: default value \"imgui.ini\" is relative to current working dir! Most apps will want to lock this to an absolute path (e.g. same path as executables).")
        .def("get_ini_filename",
            &ImGuiIO::GetIniFilename)
        .def("set_log_filename",
            &ImGuiIO::SetLogFilename, nb::arg("filename"))
        .def("get_log_filename",
            &ImGuiIO::GetLogFilename)
        // #endif
        //
        ;


    auto pyClassImGuiInputTextCallbackData =
        nb::class_<ImGuiInputTextCallbackData>
            (m, "InputTextCallbackData", " Shared state of InputText(), passed as an argument to your callback when a ImGuiInputTextFlags_Callback* flag is used.\n The callback function should return 0 by default.\n Callbacks (follow a flag name and see comments in ImGuiInputTextFlags_ declarations for more details)\n - ImGuiInputTextFlags_CallbackEdit:        Callback on buffer edit. Note that InputText() already returns True on edit + you can always use IsItemEdited(). The callback is useful to manipulate the underlying buffer while focus is active.\n - ImGuiInputTextFlags_CallbackAlways:      Callback on each iteration\n - ImGuiInputTextFlags_CallbackCompletion:  Callback on pressing TAB\n - ImGuiInputTextFlags_CallbackHistory:     Callback on pressing Up/Down arrows\n - ImGuiInputTextFlags_CallbackCharFilter:  Callback on character inputs to replace or discard them. Modify 'EventChar' to replace or discard, or return 1 in callback to discard.\n - ImGuiInputTextFlags_CallbackResize:      Callback on buffer capacity changes request (beyond 'buf_size' parameter value), allowing the string to grow.")
        .def_rw("ctx", &ImGuiInputTextCallbackData::Ctx, "Parent UI context")
        .def_rw("event_flag", &ImGuiInputTextCallbackData::EventFlag, "One ImGuiInputTextFlags_Callback*    // Read-only")
        .def_rw("flags", &ImGuiInputTextCallbackData::Flags, "What user passed to InputText()      // Read-only")
        .def_rw("user_data", &ImGuiInputTextCallbackData::UserData, "What user passed to InputText()      // Read-only")
        .def_rw("event_char", &ImGuiInputTextCallbackData::EventChar, "Character input                      // Read-write   // [CharFilter] Replace character with another one, or set to zero to drop. return 1 is equivalent to setting EventChar=0;")
        .def_rw("event_key", &ImGuiInputTextCallbackData::EventKey, "Key pressed (Up/Down/TAB)            // Read-only    // [Completion,History]")
        .def_ro("buf", &ImGuiInputTextCallbackData::Buf, "Text buffer                          // Read-write   // [Resize] Can replace pointer / [Completion,History,Always] Only write to pointed data, don't replace the actual pointer!")
        .def_rw("buf_text_len", &ImGuiInputTextCallbackData::BufTextLen, "Text length (in bytes)               // Read-write   // [Resize,Completion,History,Always] Exclude zero-terminator storage. In C land: == strlen(some_text), in C++ land: string.length()")
        .def_rw("buf_size", &ImGuiInputTextCallbackData::BufSize, "Buffer size (in bytes) = capacity+1  // Read-only    // [Resize,Completion,History,Always] Include zero-terminator storage. In C land: == ARRAYSIZE(my_char_array), in C++ land: string.capacity()+1")
        .def_rw("buf_dirty", &ImGuiInputTextCallbackData::BufDirty, "Set if you modify Buf/BufTextLen!    // Write        // [Completion,History,Always]")
        .def_rw("cursor_pos", &ImGuiInputTextCallbackData::CursorPos, "// Read-write   // [Completion,History,Always]")
        .def_rw("selection_start", &ImGuiInputTextCallbackData::SelectionStart, "// Read-write   // [Completion,History,Always] == to SelectionEnd when no selection")
        .def_rw("selection_end", &ImGuiInputTextCallbackData::SelectionEnd, "// Read-write   // [Completion,History,Always]")
        .def(nb::init<>())
        .def("delete_chars",
            &ImGuiInputTextCallbackData::DeleteChars, nb::arg("pos"), nb::arg("bytes_count"))
        .def("insert_chars",
            [](ImGuiInputTextCallbackData & self, int pos, const char * text, std::optional<std::string> text_end = std::nullopt)
            {
                auto InsertChars_adapt_const_char_pointer_with_default_null = [&self](int pos, const char * text, std::optional<std::string> text_end = std::nullopt)
                {
                    const char * text_end_adapt_default_null = nullptr;
                    if (text_end.has_value())
                        text_end_adapt_default_null = text_end.value().c_str();

                    self.InsertChars(pos, text, text_end_adapt_default_null);
                };

                InsertChars_adapt_const_char_pointer_with_default_null(pos, text, text_end);
            },     nb::arg("pos"), nb::arg("text"), nb::arg("text_end").none() = nb::none())
        .def("select_all",
            &ImGuiInputTextCallbackData::SelectAll, "(private API)")
        .def("clear_selection",
            &ImGuiInputTextCallbackData::ClearSelection, "(private API)")
        .def("has_selection",
            &ImGuiInputTextCallbackData::HasSelection, "(private API)")
        ;


    auto pyClassImGuiSizeCallbackData =
        nb::class_<ImGuiSizeCallbackData>
            (m, "SizeCallbackData", " Resizing callback data to apply custom constraint. As enabled by SetNextWindowSizeConstraints(). Callback is called during the next Begin().\n NB: For basic min/max size constraint on each axis you don't need to use the callback! The SetNextWindowSizeConstraints() parameters are enough.")
        .def("__init__", [](ImGuiSizeCallbackData * self, const std::optional<const ImVec2> & Pos = std::nullopt, const std::optional<const ImVec2> & CurrentSize = std::nullopt, const std::optional<const ImVec2> & DesiredSize = std::nullopt)
        {
            new (self) ImGuiSizeCallbackData();  // placement new
            auto r_ctor_ = self;
            if (Pos.has_value())
                r_ctor_->Pos = Pos.value();
            else
                r_ctor_->Pos = ImVec2();
            if (CurrentSize.has_value())
                r_ctor_->CurrentSize = CurrentSize.value();
            else
                r_ctor_->CurrentSize = ImVec2();
            if (DesiredSize.has_value())
                r_ctor_->DesiredSize = DesiredSize.value();
            else
                r_ctor_->DesiredSize = ImVec2();
        },
        nb::arg("pos").none() = nb::none(), nb::arg("current_size").none() = nb::none(), nb::arg("desired_size").none() = nb::none()
        )
        .def_rw("user_data", &ImGuiSizeCallbackData::UserData, "Read-only.   What user passed to SetNextWindowSizeConstraints(). Generally store an integer or float in here (need reinterpret_cast<>).")
        .def_rw("pos", &ImGuiSizeCallbackData::Pos, "Read-only.   Window position, for reference.")
        .def_rw("current_size", &ImGuiSizeCallbackData::CurrentSize, "Read-only.   Current window size.")
        .def_rw("desired_size", &ImGuiSizeCallbackData::DesiredSize, "Read-write.  Desired size, based on user's mouse position. Write to this field to restrain resizing.")
        ;


    auto pyClassImGuiWindowClass =
        nb::class_<ImGuiWindowClass>
            (m, "WindowClass", " [ALPHA] Rarely used / very advanced uses only. Use with SetNextWindowClass() and DockSpace() functions.\n Important: the content of this class is still highly WIP and likely to change and be refactored\n before we stabilize Docking features. Please be mindful if using this.\n Provide hints:\n - To the platform backend via altered viewport flags (enable/disable OS decoration, OS task bar icons, etc.)\n - To the platform backend for OS level parent/child relationships of viewport.\n - To the docking system for various options and filtering.")
        .def_rw("class_id", &ImGuiWindowClass::ClassId, "User data. 0 = Default class (unclassed). Windows of different classes cannot be docked with each others.")
        .def_rw("parent_viewport_id", &ImGuiWindowClass::ParentViewportId, "Hint for the platform backend. -1: use default. 0: request platform backend to not parent the platform. != 0: request platform backend to create a parent<>child relationship between the platform windows. Not conforming backends are free to e.g. parent every viewport to the main viewport or not.")
        .def_rw("focus_route_parent_window_id", &ImGuiWindowClass::FocusRouteParentWindowId, "ID of parent window for shortcut focus route evaluation, e.g. Shortcut() call from Parent Window will succeed when this window is focused.")
        .def_rw("viewport_flags_override_set", &ImGuiWindowClass::ViewportFlagsOverrideSet, "Viewport flags to set when a window of this class owns a viewport. This allows you to enforce OS decoration or task bar icon, override the defaults on a per-window basis.")
        .def_rw("viewport_flags_override_clear", &ImGuiWindowClass::ViewportFlagsOverrideClear, "Viewport flags to clear when a window of this class owns a viewport. This allows you to enforce OS decoration or task bar icon, override the defaults on a per-window basis.")
        .def_rw("tab_item_flags_override_set", &ImGuiWindowClass::TabItemFlagsOverrideSet, "[EXPERIMENTAL] TabItem flags to set when a window of this class gets submitted into a dock node tab bar. May use with ImGuiTabItemFlags_Leading or ImGuiTabItemFlags_Trailing.")
        .def_rw("dock_node_flags_override_set", &ImGuiWindowClass::DockNodeFlagsOverrideSet, "[EXPERIMENTAL] Dock node flags to set when a window of this class is hosted by a dock node (it doesn't have to be selected!)")
        .def_rw("docking_always_tab_bar", &ImGuiWindowClass::DockingAlwaysTabBar, "Set to True to enforce single floating windows of this class always having their own docking node (equivalent of setting the global io.ConfigDockingAlwaysTabBar)")
        .def_rw("docking_allow_unclassed", &ImGuiWindowClass::DockingAllowUnclassed, "Set to True to allow windows of this class to be docked/merged with an unclassed window. // FIXME-DOCK: Move to DockNodeFlags override?")
        .def(nb::init<>())
        ;


    auto pyClassImGuiPayload =
        nb::class_<ImGuiPayload>
            (m, "Payload", "Data payload for Drag and Drop operations: AcceptDragDropPayload(), GetDragDropPayload()")
        .def_rw("data", &ImGuiPayload::Data, "Data (copied and owned by dear imgui)")
        .def_rw("data_size", &ImGuiPayload::DataSize, "Data size")
        .def_rw("source_id", &ImGuiPayload::SourceId, "Source item id")
        .def_rw("source_parent_id", &ImGuiPayload::SourceParentId, "Source parent id (if available)")
        .def_rw("data_frame_count", &ImGuiPayload::DataFrameCount, "Data timestamp")
        .def_rw("preview", &ImGuiPayload::Preview, "Set when AcceptDragDropPayload() was called and mouse has been hovering the target item (nb: handle overlapping drag targets)")
        .def_rw("delivery", &ImGuiPayload::Delivery, "Set when AcceptDragDropPayload() was called and mouse button is released over the target item.")
        .def(nb::init<>())
        .def("clear",
            &ImGuiPayload::Clear, "(private API)")
        .def("is_data_type",
            &ImGuiPayload::IsDataType,
            nb::arg("type"),
            "(private API)")
        .def("is_preview",
            &ImGuiPayload::IsPreview, "(private API)")
        .def("is_delivery",
            &ImGuiPayload::IsDelivery, "(private API)")
        ;


    auto pyClassImGuiOnceUponAFrame =
        nb::class_<ImGuiOnceUponAFrame>
            (m, "OnceUponAFrame", " Helper: Execute a block of code at maximum once a frame. Convenient if you want to quickly create a UI within deep-nested code that runs multiple times every frame.\n Usage: static ImGuiOnceUponAFrame oaf; if (oaf) ImGui::Text(\"This will be called only once per frame\");")
        .def(nb::init<>())
        .def_rw("ref_frame", &ImGuiOnceUponAFrame::RefFrame, "")
        .def("__bool__",
            &ImGuiOnceUponAFrame::operator bool)
        ;


    auto pyClassImGuiTextFilter =
        nb::class_<ImGuiTextFilter>
            (m, "TextFilter", "Helper: Parse and apply text filters. In format \"aaaaa[,bbbb][,ccccc]\"");

    { // inner classes & enums of TextFilter
        auto pyClassImGuiTextFilter_ClassImGuiTextRange =
            nb::class_<ImGuiTextFilter::ImGuiTextRange>
                (pyClassImGuiTextFilter, "TextRange", "[Internal]")
            .def_ro("b", &ImGuiTextFilter::ImGuiTextRange::b, "")
            .def_ro("e", &ImGuiTextFilter::ImGuiTextRange::e, "")
            .def(nb::init<>())
            .def(nb::init<const char *, const char *>(),
                nb::arg("_b"), nb::arg("_e"))
            .def("empty",
                &ImGuiTextFilter::ImGuiTextRange::empty, "(private API)")
            ;
    } // end of inner classes & enums of TextFilter

    pyClassImGuiTextFilter
        .def(nb::init<const char *>(),
            nb::arg("default_filter") = "")
        .def("draw",
            &ImGuiTextFilter::Draw,
            nb::arg("label") = "Filter (inc,-exc)", nb::arg("width") = 0.0f,
            "Helper calling InputText+Build")
        .def("pass_filter",
            [](const ImGuiTextFilter & self, const char * text, std::optional<std::string> text_end = std::nullopt) -> bool
            {
                auto PassFilter_adapt_const_char_pointer_with_default_null = [&self](const char * text, std::optional<std::string> text_end = std::nullopt) -> bool
                {
                    const char * text_end_adapt_default_null = nullptr;
                    if (text_end.has_value())
                        text_end_adapt_default_null = text_end.value().c_str();

                    auto lambda_result = self.PassFilter(text, text_end_adapt_default_null);
                    return lambda_result;
                };

                return PassFilter_adapt_const_char_pointer_with_default_null(text, text_end);
            },     nb::arg("text"), nb::arg("text_end").none() = nb::none())
        .def("build",
            &ImGuiTextFilter::Build)
        .def("clear",
            &ImGuiTextFilter::Clear, "(private API)")
        .def("is_active",
            &ImGuiTextFilter::IsActive, "(private API)")
        .def_rw("count_grep", &ImGuiTextFilter::CountGrep, "")
        ;


    auto pyClassImGuiTextBuffer =
        nb::class_<ImGuiTextBuffer>
            (m, "TextBuffer", " Helper: Growable text buffer for logging/accumulating text\n (this could be called 'ImGuiTextBuilder' / 'ImGuiStringBuilder')")
        .def_rw("buf", &ImGuiTextBuffer::Buf, "")
        .def(nb::init<>())
        .def("__getitem__",
            &ImGuiTextBuffer::operator[],
            nb::arg("i"),
            "(private API)")
        .def("begin",
            &ImGuiTextBuffer::begin,
            "(private API)",
            nb::rv_policy::reference)
        .def("end",
            &ImGuiTextBuffer::end,
            "(private API)\n\n Buf is zero-terminated, so end() will point on the zero-terminator",
            nb::rv_policy::reference)
        .def("size",
            &ImGuiTextBuffer::size, "(private API)")
        .def("empty",
            &ImGuiTextBuffer::empty, "(private API)")
        .def("clear",
            &ImGuiTextBuffer::clear, "(private API)")
        .def("resize",
            &ImGuiTextBuffer::resize,
            nb::arg("size"),
            "(private API)\n\n Similar to resize(0) on ImVector: empty string but don't free buffer.")
        .def("reserve",
            &ImGuiTextBuffer::reserve,
            nb::arg("capacity"),
            "(private API)")
        .def("c_str",
            &ImGuiTextBuffer::c_str,
            "(private API)",
            nb::rv_policy::reference)
        .def("append",
            [](ImGuiTextBuffer & self, const char * str, std::optional<std::string> str_end = std::nullopt)
            {
                auto append_adapt_const_char_pointer_with_default_null = [&self](const char * str, std::optional<std::string> str_end = std::nullopt)
                {
                    const char * str_end_adapt_default_null = nullptr;
                    if (str_end.has_value())
                        str_end_adapt_default_null = str_end.value().c_str();

                    self.append(str, str_end_adapt_default_null);
                };

                append_adapt_const_char_pointer_with_default_null(str, str_end);
            },     nb::arg("str"), nb::arg("str_end").none() = nb::none())
        .def("appendf",
            [](ImGuiTextBuffer & self, const char * fmt)
            {
                auto appendf_adapt_variadic_format = [&self](const char * fmt)
                {
                    self.appendf("%s", fmt);
                };

                appendf_adapt_variadic_format(fmt);
            },     nb::arg("fmt"))
        ;


    auto pyClassImGuiStoragePair =
        nb::class_<ImGuiStoragePair>
            (m, "StoragePair", "[Internal] Key+Value for ImGuiStorage")
        .def_rw("key", &ImGuiStoragePair::key, "")
        .def(nb::init<ImGuiID, int>(),
            nb::arg("_key"), nb::arg("_val"))
        .def(nb::init<ImGuiID, float>(),
            nb::arg("_key"), nb::arg("_val"))
        .def(nb::init<ImGuiID, void *>(),
            nb::arg("_key"), nb::arg("_val"))
        ;


    auto pyClassImGuiStorage =
        nb::class_<ImGuiStorage>
            (m, "Storage", " Helper: Key->Value storage\n Typically you don't have to worry about this since a storage is held within each Window.\n We use it to e.g. store collapse state for a tree (Int 0/1)\n This is optimized for efficient lookup (dichotomy into a contiguous buffer) and rare insertion (typically tied to user interactions aka max once a frame)\n You can use it as custom user storage for temporary values. Declare your own storage if, for example:\n - You want to manipulate the open/close state of a particular sub-tree in your interface (tree node uses Int 0/1 to store their state).\n - You want to store custom debug data easily without adding or editing structures in your code (probably not efficient, but convenient)\n Types are NOT stored, so it is up to you to make sure your Key don't collide with different types.")
        .def(nb::init<>()) // implicit default constructor
        .def("clear",
            &ImGuiStorage::Clear, " - Get***() functions find pair, never add/allocate. Pairs are sorted so a query is O(log N)\n - Set***() functions find pair, insertion on demand if missing.\n - Sorted insertion is costly, paid once. A typical frame shouldn't need to insert any new pair.\n(private API)")
        .def("get_int",
            &ImGuiStorage::GetInt, nb::arg("key"), nb::arg("default_val") = 0)
        .def("set_int",
            &ImGuiStorage::SetInt, nb::arg("key"), nb::arg("val"))
        .def("get_bool",
            &ImGuiStorage::GetBool, nb::arg("key"), nb::arg("default_val") = false)
        .def("set_bool",
            &ImGuiStorage::SetBool, nb::arg("key"), nb::arg("val"))
        .def("get_float",
            &ImGuiStorage::GetFloat, nb::arg("key"), nb::arg("default_val") = 0.0f)
        .def("set_float",
            &ImGuiStorage::SetFloat, nb::arg("key"), nb::arg("val"))
        .def("get_void_ptr",
            &ImGuiStorage::GetVoidPtr,
            nb::arg("key"),
            "default_val is None",
            nb::rv_policy::reference)
        .def("set_void_ptr",
            &ImGuiStorage::SetVoidPtr, nb::arg("key"), nb::arg("val"))
        .def("get_int_ref",
            &ImGuiStorage::GetIntRef,
            nb::arg("key"), nb::arg("default_val") = 0,
            nb::rv_policy::reference)
        .def("get_bool_ref",
            &ImGuiStorage::GetBoolRef,
            nb::arg("key"), nb::arg("default_val") = false,
            nb::rv_policy::reference)
        .def("get_float_ref",
            &ImGuiStorage::GetFloatRef,
            nb::arg("key"), nb::arg("default_val") = 0.0f,
            nb::rv_policy::reference)
        .def("build_sort_by_key",
            &ImGuiStorage::BuildSortByKey, "Advanced: for quicker full rebuild of a storage (instead of an incremental one), you may add all your contents and then sort once.")
        .def("set_all_int",
            &ImGuiStorage::SetAllInt,
            nb::arg("val"),
            "Obsolete: use on your own storage if you know only integer are being stored (open/close all tree nodes)")
        ;


    auto pyEnumListClipperFlags_ =
        nb::enum_<ImGuiListClipperFlags_>(m, "ListClipperFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGuiListClipper (currently not fully exposed in function calls: a future refactor will likely add this to ImGuiListClipper::Begin function equivalent)")
            .value("none", ImGuiListClipperFlags_None, "")
            .value("no_set_table_row_counters", ImGuiListClipperFlags_NoSetTableRowCounters, "[Internal] Disabled modifying table row counters. Avoid assumption that 1 clipper item == 1 table row.");


    auto pyClassImGuiListClipper =
        nb::class_<ImGuiListClipper>
            (m, "ListClipper", " Helper: Manually clip large list of items.\n If you have lots evenly spaced items and you have random access to the list, you can perform coarse\n clipping based on visibility to only submit items that are in view.\n The clipper calculates the range of visible items and advance the cursor to compensate for the non-visible items we have skipped.\n (Dear ImGui already clip items based on their bounds but: it needs to first layout the item to do so, and generally\n  fetching/submitting your own data incurs additional cost. Coarse clipping using ImGuiListClipper allows you to easily\n  scale using lists with tens of thousands of items without a problem)\n Usage:\n   ImGuiListClipper clipper;\n   clipper.Begin(1000);         // We have 1000 elements, evenly spaced.\n   while (clipper.Step())\n       for (int i = clipper.DisplayStart; i < clipper.DisplayEnd; i++)\n           ImGui::Text(\"line number %d\", i);\n Generally what happens is:\n - Clipper lets you process the first element (DisplayStart = 0, DisplayEnd = 1) regardless of it being visible or not.\n - User code submit that one element.\n - Clipper can measure the height of the first element\n - Clipper calculate the actual range of elements to display based on the current clipping rectangle, position the cursor before the first visible element.\n - User code submit visible elements.\n - The clipper also handles various subtleties related to keyboard/gamepad navigation, wrapping etc.")
        .def_rw("ctx", &ImGuiListClipper::Ctx, "Parent UI context")
        .def_rw("display_start", &ImGuiListClipper::DisplayStart, "First item to display, updated by each call to Step()")
        .def_rw("display_end", &ImGuiListClipper::DisplayEnd, "End of items to display (exclusive)")
        .def_rw("items_count", &ImGuiListClipper::ItemsCount, "[Internal] Number of items")
        .def_rw("items_height", &ImGuiListClipper::ItemsHeight, "[Internal] Height of item after a first step and item submission can calculate it")
        .def_rw("start_pos_y", &ImGuiListClipper::StartPosY, "[Internal] Cursor position at the time of Begin() or after table frozen rows are all processed")
        .def_rw("start_seek_offset_y", &ImGuiListClipper::StartSeekOffsetY, "[Internal] Account for frozen rows in a table and initial loss of precision in very large windows.")
        .def_rw("temp_data", &ImGuiListClipper::TempData, "[Internal] Internal data")
        .def_rw("flags", &ImGuiListClipper::Flags, "[Internal] Flags, currently not yet well exposed.")
        .def(nb::init<>(),
            " items_count: Use INT_MAX if you don't know how many items you have (in which case the cursor won't be advanced in the final step, and you can call SeekCursorForItem() manually if you need)\n items_height: Use -1.0 to be calculated automatically on first step. Otherwise pass in the distance between your items, typically GetTextLineHeightWithSpacing() or GetFrameHeightWithSpacing().")
        .def("begin",
            &ImGuiListClipper::Begin, nb::arg("items_count"), nb::arg("items_height") = -1.0f)
        .def("end",
            &ImGuiListClipper::End, "Automatically called on the last call of Step() that returns False.")
        .def("step",
            &ImGuiListClipper::Step, "Call until it returns False. The DisplayStart/DisplayEnd fields will be set and you can process/draw those items.")
        .def("include_item_by_index",
            &ImGuiListClipper::IncludeItemByIndex,
            nb::arg("item_index"),
            " Call IncludeItemByIndex() or IncludeItemsByIndex() *BEFORE* first call to Step() if you need a range of items to not be clipped, regardless of their visibility.\n (Due to alignment / padding of certain items it is possible that an extra item may be included on either end of the display range).\n(private API)")
        .def("include_items_by_index",
            &ImGuiListClipper::IncludeItemsByIndex,
            nb::arg("item_begin"), nb::arg("item_end"),
            "item_end is exclusive e.g. use (42, 42+1) to make item 42 never clipped.")
        .def("seek_cursor_for_item",
            &ImGuiListClipper::SeekCursorForItem,
            nb::arg("item_index"),
            " Seek cursor toward given item. This is automatically called while stepping.\n - The only reason to call this is: you can use ImGuiListClipper::Begin(INT_MAX) if you don't know item count ahead of time.\n - In this case, after all steps are done, you'll want to call SeekCursorForItem(item_count).")
        ;


    auto pyClassImColor =
        nb::class_<ImColor>
            (m, "ImColor", " Helper: ImColor() implicitly converts colors to either ImU32 (packed 4x1 byte) or ImVec4 (4x1 float)\n Prefer using IM_COL32() macros if you want a guaranteed compile-time ImU32 for usage with ImDrawList API.\n **Avoid storing ImColor! Store either u32 of ImVec4. This is not a full-featured color class. MAY OBSOLETE.\n **None of the ImGui API are using ImColor directly but you can use it as a convenience to pass colors in either ImU32 or ImVec4 formats. Explicitly cast to ImU32 or ImVec4 if needed.")
        .def_rw("value", &ImColor::Value, "")
        .def(nb::init<>())
        .def(nb::init<float, float, float, float>(),
            nb::arg("r"), nb::arg("g"), nb::arg("b"), nb::arg("a") = 1.0f)
        .def(nb::init<const ImVec4 &>(),
            nb::arg("col"))
        .def(nb::init<int, int, int, int>(),
            nb::arg("r"), nb::arg("g"), nb::arg("b"), nb::arg("a") = 255)
        .def(nb::init<ImU32>(),
            nb::arg("rgba"))
        .def("set_hsv",
            &ImColor::SetHSV,
            nb::arg("h"), nb::arg("s"), nb::arg("v"), nb::arg("a") = 1.0f,
            "(private API)")
        .def_static("hsv",
            &ImColor::HSV,
            nb::arg("h"), nb::arg("s"), nb::arg("v"), nb::arg("a") = 1.0f,
            "(private API)")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("to_dict",
            &ImColor::to_dict, "(private API)")
        .def_static("from_dict",
            &ImColor::from_dict,
            nb::arg("d"),
            "(private API)")
        // #endif
        //
        .def("__copy__",  [](const ImColor &self) {
            return ImColor(self);
        })    ;


    auto pyEnumMultiSelectFlags_ =
        nb::enum_<ImGuiMultiSelectFlags_>(m, "MultiSelectFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for BeginMultiSelect()")
            .value("none", ImGuiMultiSelectFlags_None, "")
            .value("single_select", ImGuiMultiSelectFlags_SingleSelect, "Disable selecting more than one item. This is available to allow single-selection code to share same code/logic if desired. It essentially disables the main purpose of BeginMultiSelect() tho!")
            .value("no_select_all", ImGuiMultiSelectFlags_NoSelectAll, "Disable CTRL+A shortcut to select all.")
            .value("no_range_select", ImGuiMultiSelectFlags_NoRangeSelect, "Disable Shift+selection mouse/keyboard support (useful for unordered 2D selection). With BoxSelect is also ensure contiguous SetRange requests are not combined into one. This allows not handling interpolation in SetRange requests.")
            .value("no_auto_select", ImGuiMultiSelectFlags_NoAutoSelect, "Disable selecting items when navigating (useful for e.g. supporting range-select in a list of checkboxes).")
            .value("no_auto_clear", ImGuiMultiSelectFlags_NoAutoClear, "Disable clearing selection when navigating or selecting another one (generally used with ImGuiMultiSelectFlags_NoAutoSelect. useful for e.g. supporting range-select in a list of checkboxes).")
            .value("no_auto_clear_on_reselect", ImGuiMultiSelectFlags_NoAutoClearOnReselect, "Disable clearing selection when clicking/selecting an already selected item.")
            .value("box_select1d", ImGuiMultiSelectFlags_BoxSelect1d, "Enable box-selection with same width and same x pos items (e.g. full row Selectable()). Box-selection works better with little bit of spacing between items hit-box in order to be able to aim at empty space.")
            .value("box_select2d", ImGuiMultiSelectFlags_BoxSelect2d, "Enable box-selection with varying width or varying x pos items support (e.g. different width labels, or 2D layout/grid). This is slower: alters clipping logic so that e.g. horizontal movements will update selection of normally clipped items.")
            .value("box_select_no_scroll", ImGuiMultiSelectFlags_BoxSelectNoScroll, "Disable scrolling when box-selecting near edges of scope.")
            .value("clear_on_escape", ImGuiMultiSelectFlags_ClearOnEscape, "Clear selection when pressing Escape while scope is focused.")
            .value("clear_on_click_void", ImGuiMultiSelectFlags_ClearOnClickVoid, "Clear selection when clicking on empty location within scope.")
            .value("scope_window", ImGuiMultiSelectFlags_ScopeWindow, "Scope for _BoxSelect and _ClearOnClickVoid is whole window (Default). Use if BeginMultiSelect() covers a whole window or used a single time in same window.")
            .value("scope_rect", ImGuiMultiSelectFlags_ScopeRect, "Scope for _BoxSelect and _ClearOnClickVoid is rectangle encompassing BeginMultiSelect()/EndMultiSelect(). Use if BeginMultiSelect() is called multiple times in same window.")
            .value("select_on_click", ImGuiMultiSelectFlags_SelectOnClick, "Apply selection on mouse down when clicking on unselected item. (Default)")
            .value("select_on_click_release", ImGuiMultiSelectFlags_SelectOnClickRelease, "Apply selection on mouse release when clicking an unselected item. Allow dragging an unselected item without altering selection.")
            .value("nav_wrap_x", ImGuiMultiSelectFlags_NavWrapX, "[Temporary] Enable navigation wrapping on X axis. Provided as a convenience because we don't have a design for the general Nav API for this yet. When the more general feature be public we may obsolete this flag in favor of new one.");


    auto pyClassImGuiMultiSelectIO =
        nb::class_<ImGuiMultiSelectIO>
            (m, "MultiSelectIO", " Main IO structure returned by BeginMultiSelect()/EndMultiSelect().\n This mainly contains a list of selection requests.\n - Use 'Demo->Tools->Debug Log->Selection' to see requests as they happen.\n - Some fields are only useful if your list is dynamic and allows deletion (getting post-deletion focus/state right is shown in the demo)\n - Below: who reads/writes each fields? 'r'=read, 'w'=write, 'ms'=multi-select code, 'app'=application/user code.")
        .def("__init__", [](ImGuiMultiSelectIO * self, const std::optional<const ImVector<ImGuiSelectionRequest>> & Requests = std::nullopt, const std::optional<const ImGuiSelectionUserData> & RangeSrcItem = std::nullopt, const std::optional<const ImGuiSelectionUserData> & NavIdItem = std::nullopt, bool NavIdSelected = bool(), bool RangeSrcReset = bool(), int ItemsCount = int())
        {
            new (self) ImGuiMultiSelectIO();  // placement new
            auto r_ctor_ = self;
            if (Requests.has_value())
                r_ctor_->Requests = Requests.value();
            else
                r_ctor_->Requests = ImVector<ImGuiSelectionRequest>();
            if (RangeSrcItem.has_value())
                r_ctor_->RangeSrcItem = RangeSrcItem.value();
            else
                r_ctor_->RangeSrcItem = ImGuiSelectionUserData();
            if (NavIdItem.has_value())
                r_ctor_->NavIdItem = NavIdItem.value();
            else
                r_ctor_->NavIdItem = ImGuiSelectionUserData();
            r_ctor_->NavIdSelected = NavIdSelected;
            r_ctor_->RangeSrcReset = RangeSrcReset;
            r_ctor_->ItemsCount = ItemsCount;
        },
        nb::arg("requests").none() = nb::none(), nb::arg("range_src_item").none() = nb::none(), nb::arg("nav_id_item").none() = nb::none(), nb::arg("nav_id_selected") = bool(), nb::arg("range_src_reset") = bool(), nb::arg("items_count") = int()
        )
        .def_rw("requests", &ImGuiMultiSelectIO::Requests, "ms:w, app:r     /  ms:w  app:r   // Requests to apply to your selection data.")
        .def_rw("range_src_item", &ImGuiMultiSelectIO::RangeSrcItem, "ms:w  app:r     /                // (If using clipper) Begin: Source item (often the first selected item) must never be clipped: use clipper.IncludeItemByIndex() to ensure it is submitted.")
        .def_rw("nav_id_item", &ImGuiMultiSelectIO::NavIdItem, "ms:w, app:r     /                // (If using deletion) Last known SetNextItemSelectionUserData() value for NavId (if part of submitted items).")
        .def_rw("nav_id_selected", &ImGuiMultiSelectIO::NavIdSelected, "ms:w, app:r     /        app:r   // (If using deletion) Last known selection state for NavId (if part of submitted items).")
        .def_rw("range_src_reset", &ImGuiMultiSelectIO::RangeSrcReset, "app:w     /  ms:r          // (If using deletion) Set before EndMultiSelect() to reset ResetSrcItem (e.g. if deleted selection).")
        .def_rw("items_count", &ImGuiMultiSelectIO::ItemsCount, "ms:w, app:r     /        app:r   // 'int items_count' parameter to BeginMultiSelect() is copied here for convenience, allowing simpler calls to your ApplyRequests handler. Not used internally.")
        ;


    auto pyEnumSelectionRequestType =
        nb::enum_<ImGuiSelectionRequestType>(m, "SelectionRequestType", nb::is_arithmetic(), nb::is_flag(), "Selection request type")
            .value("none", ImGuiSelectionRequestType_None, "")
            .value("set_all", ImGuiSelectionRequestType_SetAll, "Request app to clear selection (if Selected==False) or select all items (if Selected==True). We cannot set RangeFirstItem/RangeLastItem as its contents is entirely up to user (not necessarily an index)")
            .value("set_range", ImGuiSelectionRequestType_SetRange, "Request app to select/unselect [RangeFirstItem..RangeLastItem] items (inclusive) based on value of Selected. Only EndMultiSelect() request this, app code can read after BeginMultiSelect() and it will always be False.");


    auto pyClassImGuiSelectionRequest =
        nb::class_<ImGuiSelectionRequest>
            (m, "SelectionRequest", "Selection request item")
        .def("__init__", [](ImGuiSelectionRequest * self, ImGuiSelectionRequestType Type = ImGuiSelectionRequestType(), bool Selected = bool(), ImS8 RangeDirection = ImS8(), const std::optional<const ImGuiSelectionUserData> & RangeFirstItem = std::nullopt, const std::optional<const ImGuiSelectionUserData> & RangeLastItem = std::nullopt)
        {
            new (self) ImGuiSelectionRequest();  // placement new
            auto r_ctor_ = self;
            r_ctor_->Type = Type;
            r_ctor_->Selected = Selected;
            r_ctor_->RangeDirection = RangeDirection;
            if (RangeFirstItem.has_value())
                r_ctor_->RangeFirstItem = RangeFirstItem.value();
            else
                r_ctor_->RangeFirstItem = ImGuiSelectionUserData();
            if (RangeLastItem.has_value())
                r_ctor_->RangeLastItem = RangeLastItem.value();
            else
                r_ctor_->RangeLastItem = ImGuiSelectionUserData();
        },
        nb::arg("type") = ImGuiSelectionRequestType(), nb::arg("selected") = bool(), nb::arg("range_direction") = ImS8(), nb::arg("range_first_item").none() = nb::none(), nb::arg("range_last_item").none() = nb::none()
        )
        .def_rw("type", &ImGuiSelectionRequest::Type, "ms:w, app:r     /  ms:w, app:r   // Request type. You'll most often receive 1 Clear + 1 SetRange with a single-item range.")
        .def_rw("selected", &ImGuiSelectionRequest::Selected, "ms:w, app:r     /  ms:w, app:r   // Parameter for SetAll/SetRange requests (True = select, False = unselect)")
        .def_rw("range_direction", &ImGuiSelectionRequest::RangeDirection, "/  ms:w  app:r   // Parameter for SetRange request: +1 when RangeFirstItem comes before RangeLastItem, -1 otherwise. Useful if you want to preserve selection order on a backward Shift+Click.")
        .def_rw("range_first_item", &ImGuiSelectionRequest::RangeFirstItem, "/  ms:w, app:r   // Parameter for SetRange request (this is generally == RangeSrcItem when shift selecting from top to bottom).")
        .def_rw("range_last_item", &ImGuiSelectionRequest::RangeLastItem, "/  ms:w, app:r   // Parameter for SetRange request (this is generally == RangeSrcItem when shift selecting from bottom to top). Inclusive!")
        ;


    auto pyClassImGuiSelectionBasicStorage =
        nb::class_<ImGuiSelectionBasicStorage>
            (m, "SelectionBasicStorage", " Optional helper to store multi-selection state + apply multi-selection requests.\n - Used by our demos and provided as a convenience to easily implement basic multi-selection.\n - Iterate selection with 'None* it = None; ImGuiID id; while (selection.GetNextSelectedItem(&it, &id)) { ... }'\n   Or you can check 'if (Contains(id)) { ... }' for each possible object if their number is not too high to iterate.\n - USING THIS IS NOT MANDATORY. This is only a helper and not a required API.\n To store a multi-selection, in your application you could:\n - Use this helper as a convenience. We use our simple key->value ImGuiStorage as a std::set<ImGuiID> replacement.\n - Use your own external storage: e.g. std::set<MyObjectId>, std::vector<MyObjectId>, interval trees, intrusively stored selection etc.\n In ImGuiSelectionBasicStorage we:\n - always use indices in the multi-selection API (passed to SetNextItemSelectionUserData(), retrieved in ImGuiMultiSelectIO)\n - use the AdapterIndexToStorageId() indirection layer to abstract how persistent selection data is derived from an index.\n - use decently optimized logic to allow queries and insertion of very large selection sets.\n - do not preserve selection order.\n Many combinations are possible depending on how you prefer to store your items and how you prefer to store your selection.\n Large applications are likely to eventually want to get rid of this indirection layer and do their own thing.\n See https://github.com/ocornut/imgui/wiki/Multi-Select for details and pseudo-code using this helper.")
        .def_rw("size", &ImGuiSelectionBasicStorage::Size, "// Number of selected items, maintained by this helper.")
        .def_rw("preserve_order", &ImGuiSelectionBasicStorage::PreserveOrder, "= False  // GetNextSelectedItem() will return ordered selection (currently implemented by two additional sorts of selection. Could be improved)")
        .def_rw("user_data", &ImGuiSelectionBasicStorage::UserData, "= None   // User data for use by adapter function        // e.g. selection.UserData = (None*)my_items;")
        .def_rw("_selection_order", &ImGuiSelectionBasicStorage::_SelectionOrder, "[Internal] Increasing counter to store selection order")
        .def_rw("_storage", &ImGuiSelectionBasicStorage::_Storage, "[Internal] Selection set. Think of this as similar to e.g. std::set<ImGuiID>. Prefer not accessing directly: iterate with GetNextSelectedItem().")
        .def(nb::init<>())
        .def("apply_requests",
            &ImGuiSelectionBasicStorage::ApplyRequests,
            nb::arg("ms_io"),
            "Apply selection requests coming from BeginMultiSelect() and EndMultiSelect() functions. It uses 'items_count' passed to BeginMultiSelect()")
        .def("contains",
            [](const ImGuiSelectionBasicStorage & self, ImGuiID id) -> bool
            {
                auto Contains_adapt_force_lambda = [&self](ImGuiID id) -> bool
                {
                    auto lambda_result = self.Contains(id);
                    return lambda_result;
                };

                return Contains_adapt_force_lambda(id);
            },
            nb::arg("id_"),
            "Query if an item id is in selection.")
        .def("clear",
            &ImGuiSelectionBasicStorage::Clear, "Clear selection")
        .def("swap",
            &ImGuiSelectionBasicStorage::Swap,
            nb::arg("r"),
            "Swap two selections")
        .def("set_item_selected",
            &ImGuiSelectionBasicStorage::SetItemSelected,
            nb::arg("id_"), nb::arg("selected"),
            "Add/remove an item from selection (generally done by ApplyRequests() function)")
        .def("get_storage_id_from_index",
            &ImGuiSelectionBasicStorage::GetStorageIdFromIndex,
            nb::arg("idx"),
            "(private API)\n\n Convert index to item id based on provided adapter.")
        ;


    auto pyClassImGuiSelectionExternalStorage =
        nb::class_<ImGuiSelectionExternalStorage>
            (m, "SelectionExternalStorage", " Optional helper to apply multi-selection requests to existing randomly accessible storage.\n Convenient if you want to quickly wire multi-select API on e.g. an array of bool or items storing their own selection state.")
        .def_rw("user_data", &ImGuiSelectionExternalStorage::UserData, "User data for use by adapter function                                // e.g. selection.UserData = (None*)my_items;")
        .def(nb::init<>())
        .def("apply_requests",
            &ImGuiSelectionExternalStorage::ApplyRequests,
            nb::arg("ms_io"),
            "Apply selection requests by using AdapterSetItemSelected() calls")
        ;


    auto pyClassImDrawCmd =
        nb::class_<ImDrawCmd>
            (m, "ImDrawCmd", " Typically, 1 command = 1 GPU draw call (unless command is a callback)\n - VtxOffset: When 'io.BackendFlags & ImGuiBackendFlags_RendererHasVtxOffset' is enabled,\n   this fields allow us to render meshes larger than 64K vertices while keeping 16-bit indices.\n   Backends made for <1.71. will typically ignore the VtxOffset fields.\n - The ClipRect/TexRef/VtxOffset fields must be contiguous as we memcmp() them together (this is asserted for).")
        .def_rw("clip_rect", &ImDrawCmd::ClipRect, "4*4  // Clipping rectangle (x1, y1, x2, y2). Subtract ImDrawData->DisplayPos to get clipping rectangle in \"viewport\" coordinates")
        .def_rw("tex_ref", &ImDrawCmd::TexRef, "16   // Reference to a font/texture atlas (where backend called ImTextureData::SetTexID()) or to a user-provided texture ID (via e.g. ImGui::Image() calls). Both will lead to a ImTextureID value.")
        .def_rw("vtx_offset", &ImDrawCmd::VtxOffset, "4    // Start offset in vertex buffer. ImGuiBackendFlags_RendererHasVtxOffset: always 0, otherwise may be >0 to support meshes larger than 64K vertices with 16-bit indices.")
        .def_rw("idx_offset", &ImDrawCmd::IdxOffset, "4    // Start offset in index buffer.")
        .def_rw("elem_count", &ImDrawCmd::ElemCount, "4    // Number of indices (multiple of 3) to be rendered as triangles. Vertices are stored in the callee ImDrawList's vtx_buffer[] array, indices in idx_buffer[].")
        .def_rw("user_callback_data", &ImDrawCmd::UserCallbackData, "4-8  // Callback user data (when UserCallback != None). If called AddCallback() with size == 0, this is a copy of the AddCallback() argument. If called AddCallback() with size > 0, this is pointing to a buffer where data is stored.")
        .def_rw("user_callback_data_size", &ImDrawCmd::UserCallbackDataSize, "4 // Size of callback user data when using storage, otherwise 0.")
        .def_rw("user_callback_data_offset", &ImDrawCmd::UserCallbackDataOffset, "4 // [Internal] Offset of callback user data when using storage, otherwise -1.")
        .def(nb::init<>(),
            "Also ensure our padding fields are zeroed")
        .def("get_tex_id",
            &ImDrawCmd::GetTexID, "(private API)\n\n == (TexRef._TexData ? TexRef._TexData->TexID : TexRef._TexID)")
        ;
    // #ifndef IMGUI_OVERRIDE_DRAWVERT_STRUCT_LAYOUT
    //


    auto pyClassImDrawVert =
        nb::class_<ImDrawVert>
            (m, "ImDrawVert", "")
        .def("__init__", [](ImDrawVert * self, const std::optional<const ImVec2> & pos = std::nullopt, const std::optional<const ImVec2> & uv = std::nullopt, ImU32 col = ImU32())
        {
            new (self) ImDrawVert();  // placement new
            auto r_ctor_ = self;
            if (pos.has_value())
                r_ctor_->pos = pos.value();
            else
                r_ctor_->pos = ImVec2();
            if (uv.has_value())
                r_ctor_->uv = uv.value();
            else
                r_ctor_->uv = ImVec2();
            r_ctor_->col = col;
        },
        nb::arg("pos").none() = nb::none(), nb::arg("uv").none() = nb::none(), nb::arg("col") = ImU32()
        )
        .def_rw("pos", &ImDrawVert::pos, "")
        .def_rw("uv", &ImDrawVert::uv, "")
        .def_rw("col", &ImDrawVert::col, "")
        ;
    // #else
    //
    // #endif
    //


    auto pyClassImDrawCmdHeader =
        nb::class_<ImDrawCmdHeader>
            (m, "ImDrawCmdHeader", "[Internal] For use by ImDrawList")
        .def("__init__", [](ImDrawCmdHeader * self, const std::optional<const ImVec4> & ClipRect = std::nullopt, const std::optional<const ImTextureRef> & TexRef = std::nullopt)
        {
            new (self) ImDrawCmdHeader();  // placement new
            auto r_ctor_ = self;
            if (ClipRect.has_value())
                r_ctor_->ClipRect = ClipRect.value();
            else
                r_ctor_->ClipRect = ImVec4();
            if (TexRef.has_value())
                r_ctor_->TexRef = TexRef.value();
            else
                r_ctor_->TexRef = ImTextureRef();
        },
        nb::arg("clip_rect").none() = nb::none(), nb::arg("tex_ref").none() = nb::none()
        )
        .def_rw("clip_rect", &ImDrawCmdHeader::ClipRect, "")
        .def_rw("tex_ref", &ImDrawCmdHeader::TexRef, "")
        .def_rw("vtx_offset", &ImDrawCmdHeader::VtxOffset, "")
        ;


    auto pyClassImDrawChannel =
        nb::class_<ImDrawChannel>
            (m, "ImDrawChannel", "[Internal] For use by ImDrawListSplitter")
        .def("__init__", [](ImDrawChannel * self, const std::optional<const ImVector<ImDrawCmd>> & _CmdBuffer = std::nullopt, const std::optional<const ImVector<ImDrawIdx>> & _IdxBuffer = std::nullopt)
        {
            new (self) ImDrawChannel();  // placement new
            auto r_ctor_ = self;
            if (_CmdBuffer.has_value())
                r_ctor_->_CmdBuffer = _CmdBuffer.value();
            else
                r_ctor_->_CmdBuffer = ImVector<ImDrawCmd>();
            if (_IdxBuffer.has_value())
                r_ctor_->_IdxBuffer = _IdxBuffer.value();
            else
                r_ctor_->_IdxBuffer = ImVector<ImDrawIdx>();
        },
        nb::arg("_cmd_buffer").none() = nb::none(), nb::arg("_idx_buffer").none() = nb::none()
        )
        .def_rw("_cmd_buffer", &ImDrawChannel::_CmdBuffer, "")
        .def_rw("_idx_buffer", &ImDrawChannel::_IdxBuffer, "")
        ;


    auto pyClassImDrawListSplitter =
        nb::class_<ImDrawListSplitter>
            (m, "ImDrawListSplitter", " Split/Merge functions are used to split the draw list into different layers which can be drawn into out of order.\n This is used by the Columns/Tables API, so items of each column can be batched together in a same draw call.")
        .def_rw("_current", &ImDrawListSplitter::_Current, "Current channel number (0)")
        .def_rw("_count", &ImDrawListSplitter::_Count, "Number of active channels (1+)")
        .def_rw("_channels", &ImDrawListSplitter::_Channels, "Draw channels (not resized down so _Count might be < Channels.Size)")
        .def(nb::init<>())
        .def("clear",
            &ImDrawListSplitter::Clear, "(private API)\n\n Do not clear Channels[] so our allocations are reused next frame")
        .def("clear_free_memory",
            &ImDrawListSplitter::ClearFreeMemory)
        .def("split",
            &ImDrawListSplitter::Split, nb::arg("draw_list"), nb::arg("count"))
        .def("merge",
            &ImDrawListSplitter::Merge, nb::arg("draw_list"))
        .def("set_current_channel",
            &ImDrawListSplitter::SetCurrentChannel, nb::arg("draw_list"), nb::arg("channel_idx"))
        ;


    auto pyEnumImDrawFlags_ =
        nb::enum_<ImDrawFlags_>(m, "ImDrawFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImDrawList functions\n (Legacy: bit 0 must always correspond to ImDrawFlags_Closed to be backward compatible with old API using a bool. Bits 1..3 must be unused)")
            .value("none", ImDrawFlags_None, "")
            .value("closed", ImDrawFlags_Closed, "PathStroke(), AddPolyline(): specify that shape should be closed (Important: this is always == 1 for legacy reason)")
            .value("round_corners_top_left", ImDrawFlags_RoundCornersTopLeft, "AddRect(), AddRectFilled(), PathRect(): enable rounding top-left corner only (when rounding > 0.0, we default to all corners). Was 0x01.")
            .value("round_corners_top_right", ImDrawFlags_RoundCornersTopRight, "AddRect(), AddRectFilled(), PathRect(): enable rounding top-right corner only (when rounding > 0.0, we default to all corners). Was 0x02.")
            .value("round_corners_bottom_left", ImDrawFlags_RoundCornersBottomLeft, "AddRect(), AddRectFilled(), PathRect(): enable rounding bottom-left corner only (when rounding > 0.0, we default to all corners). Was 0x04.")
            .value("round_corners_bottom_right", ImDrawFlags_RoundCornersBottomRight, "AddRect(), AddRectFilled(), PathRect(): enable rounding bottom-right corner only (when rounding > 0.0, we default to all corners). Wax 0x08.")
            .value("round_corners_none", ImDrawFlags_RoundCornersNone, "AddRect(), AddRectFilled(), PathRect(): disable rounding on all corners (when rounding > 0.0). This is NOT zero, NOT an implicit flag!")
            .value("round_corners_top", ImDrawFlags_RoundCornersTop, "")
            .value("round_corners_bottom", ImDrawFlags_RoundCornersBottom, "")
            .value("round_corners_left", ImDrawFlags_RoundCornersLeft, "")
            .value("round_corners_right", ImDrawFlags_RoundCornersRight, "")
            .value("round_corners_all", ImDrawFlags_RoundCornersAll, "")
            .value("round_corners_default_", ImDrawFlags_RoundCornersDefault_, "Default to ALL corners if none of the _RoundCornersXX flags are specified.")
            .value("round_corners_mask_", ImDrawFlags_RoundCornersMask_, "");


    auto pyEnumImDrawListFlags_ =
        nb::enum_<ImDrawListFlags_>(m, "ImDrawListFlags_", nb::is_arithmetic(), nb::is_flag(), " Flags for ImDrawList instance. Those are set automatically by ImGui:: functions from ImGuiIO settings, and generally not manipulated directly.\n It is however possible to temporarily alter flags between calls to ImDrawList:: functions.")
            .value("none", ImDrawListFlags_None, "")
            .value("anti_aliased_lines", ImDrawListFlags_AntiAliasedLines, "Enable anti-aliased lines/borders (*2 the number of triangles for 1.0 wide line or lines thin enough to be drawn using textures, otherwise *3 the number of triangles)")
            .value("anti_aliased_lines_use_tex", ImDrawListFlags_AntiAliasedLinesUseTex, "Enable anti-aliased lines/borders using textures when possible. Require backend to render with bilinear filtering (NOT point/nearest filtering).")
            .value("anti_aliased_fill", ImDrawListFlags_AntiAliasedFill, "Enable anti-aliased edge around filled shapes (rounded rectangles, circles).")
            .value("allow_vtx_offset", ImDrawListFlags_AllowVtxOffset, "Can emit 'VtxOffset > 0' to allow large meshes. Set when 'ImGuiBackendFlags_RendererHasVtxOffset' is enabled.");


    auto pyClassImDrawList =
        nb::class_<ImDrawList>
            (m, "ImDrawList", " Draw command list\n This is the low-level list of polygons that ImGui:: functions are filling. At the end of the frame,\n all command lists are passed to your ImGuiIO::RenderDrawListFn function for rendering.\n Each dear imgui window contains its own ImDrawList. You can use ImGui::GetWindowDrawList() to\n access the current window draw list and draw custom primitives.\n You can interleave normal ImGui:: calls and adding primitives to the current draw list.\n In single viewport mode, top-left is == GetMainViewport()->Pos (generally 0,0), bottom-right is == GetMainViewport()->Pos+Size (generally io.DisplaySize).\n You are totally free to apply whatever transformation matrix you want to the data (depending on the use of the transformation you may want to apply it to ClipRect as well!)\n Important: Primitives are always added to the list and not culled (culling is done at higher-level by ImGui:: functions), if you use this API a lot consider coarse culling your drawn objects.")
        .def_rw("cmd_buffer", &ImDrawList::CmdBuffer, "Draw commands. Typically 1 command = 1 GPU draw call, unless the command is a callback.")
        .def_rw("idx_buffer", &ImDrawList::IdxBuffer, "Index buffer. Each command consume ImDrawCmd::ElemCount of those")
        .def_rw("vtx_buffer", &ImDrawList::VtxBuffer, "Vertex buffer.")
        .def_rw("flags", &ImDrawList::Flags, "Flags, you may poke into these to adjust anti-aliasing settings per-primitive.")
        .def_rw("_vtx_current_idx", &ImDrawList::_VtxCurrentIdx, "[Internal] generally == VtxBuffer.Size unless we are past 64K vertices, in which case this gets reset to 0.")
        .def_rw("_data", &ImDrawList::_Data, "Pointer to shared draw data (you can use ImGui::GetDrawListSharedData() to get the one from current ImGui context)")
        .def_rw("_vtx_write_ptr", &ImDrawList::_VtxWritePtr, "[Internal] point within VtxBuffer.Data after each add command (to avoid using the ImVector<> operators too much)")
        .def_rw("_idx_write_ptr", &ImDrawList::_IdxWritePtr, "[Internal] point within IdxBuffer.Data after each add command (to avoid using the ImVector<> operators too much)")
        .def_rw("_path", &ImDrawList::_Path, "[Internal] current path building")
        .def_rw("_cmd_header", &ImDrawList::_CmdHeader, "[Internal] template of active commands. Fields should match those of CmdBuffer.back().")
        .def_rw("_splitter", &ImDrawList::_Splitter, "[Internal] for channels api (note: prefer using your own persistent instance of ImDrawListSplitter!)")
        .def_rw("_clip_rect_stack", &ImDrawList::_ClipRectStack, "[Internal]")
        .def_rw("_texture_stack", &ImDrawList::_TextureStack, "[Internal]")
        .def_rw("_callbacks_data_buf", &ImDrawList::_CallbacksDataBuf, "[Internal]")
        .def_rw("_fringe_scale", &ImDrawList::_FringeScale, "[Internal] anti-alias fringe is scaled by this value, this helps to keep things sharp while zooming at vertex buffer content")
        .def_ro("_owner_name", &ImDrawList::_OwnerName, "Pointer to owner window's name for debugging")
        .def(nb::init<ImDrawListSharedData *>(),
            nb::arg("shared_data"),
            " If you want to create ImDrawList instances, pass them ImGui::GetDrawListSharedData().\n (advanced: you may create and use your own ImDrawListSharedData so you can use ImDrawList without ImGui, but that's more involved)")
        .def("push_clip_rect",
            &ImDrawList::PushClipRect,
            nb::arg("clip_rect_min"), nb::arg("clip_rect_max"), nb::arg("intersect_with_current_clip_rect") = false,
            "Render-level scissoring. This is passed down to your render function but not used for CPU-side coarse clipping. Prefer using higher-level ImGui::PushClipRect() to affect logic (hit-testing and widget culling)")
        .def("push_clip_rect_full_screen",
            &ImDrawList::PushClipRectFullScreen)
        .def("pop_clip_rect",
            &ImDrawList::PopClipRect)
        .def("push_texture",
            &ImDrawList::PushTexture, nb::arg("tex_ref"))
        .def("pop_texture",
            &ImDrawList::PopTexture)
        .def("get_clip_rect_min",
            &ImDrawList::GetClipRectMin, "(private API)")
        .def("get_clip_rect_max",
            &ImDrawList::GetClipRectMax, "(private API)")
        .def("add_line",
            &ImDrawList::AddLine, nb::arg("p1"), nb::arg("p2"), nb::arg("col"), nb::arg("thickness") = 1.0f)
        .def("add_rect",
            &ImDrawList::AddRect,
            nb::arg("p_min"), nb::arg("p_max"), nb::arg("col"), nb::arg("rounding") = 0.0f, nb::arg("flags") = 0, nb::arg("thickness") = 1.0f,
            "a: upper-left, b: lower-right (== upper-left + size)")
        .def("add_rect_filled",
            &ImDrawList::AddRectFilled,
            nb::arg("p_min"), nb::arg("p_max"), nb::arg("col"), nb::arg("rounding") = 0.0f, nb::arg("flags") = 0,
            "a: upper-left, b: lower-right (== upper-left + size)")
        .def("add_rect_filled_multi_color",
            &ImDrawList::AddRectFilledMultiColor, nb::arg("p_min"), nb::arg("p_max"), nb::arg("col_upr_left"), nb::arg("col_upr_right"), nb::arg("col_bot_right"), nb::arg("col_bot_left"))
        .def("add_quad",
            &ImDrawList::AddQuad, nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("p4"), nb::arg("col"), nb::arg("thickness") = 1.0f)
        .def("add_quad_filled",
            &ImDrawList::AddQuadFilled, nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("p4"), nb::arg("col"))
        .def("add_triangle",
            &ImDrawList::AddTriangle, nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("col"), nb::arg("thickness") = 1.0f)
        .def("add_triangle_filled",
            &ImDrawList::AddTriangleFilled, nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("col"))
        .def("add_circle",
            &ImDrawList::AddCircle, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("num_segments") = 0, nb::arg("thickness") = 1.0f)
        .def("add_circle_filled",
            &ImDrawList::AddCircleFilled, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("num_segments") = 0)
        .def("add_ngon",
            &ImDrawList::AddNgon, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("num_segments"), nb::arg("thickness") = 1.0f)
        .def("add_ngon_filled",
            &ImDrawList::AddNgonFilled, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("num_segments"))
        .def("add_ellipse",
            &ImDrawList::AddEllipse, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("rot") = 0.0f, nb::arg("num_segments") = 0, nb::arg("thickness") = 1.0f)
        .def("add_ellipse_filled",
            &ImDrawList::AddEllipseFilled, nb::arg("center"), nb::arg("radius"), nb::arg("col"), nb::arg("rot") = 0.0f, nb::arg("num_segments") = 0)
        .def("add_text",
            [](ImDrawList & self, const ImVec2 & pos, ImU32 col, const char * text_begin, std::optional<std::string> text_end = std::nullopt)
            {
                auto AddText_adapt_const_char_pointer_with_default_null = [&self](const ImVec2 & pos, ImU32 col, const char * text_begin, std::optional<std::string> text_end = std::nullopt)
                {
                    const char * text_end_adapt_default_null = nullptr;
                    if (text_end.has_value())
                        text_end_adapt_default_null = text_end.value().c_str();

                    self.AddText(pos, col, text_begin, text_end_adapt_default_null);
                };

                AddText_adapt_const_char_pointer_with_default_null(pos, col, text_begin, text_end);
            },     nb::arg("pos"), nb::arg("col"), nb::arg("text_begin"), nb::arg("text_end").none() = nb::none())
        .def("add_text",
            [](ImDrawList & self, ImFont * font, float font_size, const ImVec2 & pos, ImU32 col, const char * text_begin, std::optional<std::string> text_end = std::nullopt, float wrap_width = 0.0f, const ImVec4 * cpu_fine_clip_rect = NULL)
            {
                auto AddText_adapt_const_char_pointer_with_default_null = [&self](ImFont * font, float font_size, const ImVec2 & pos, ImU32 col, const char * text_begin, std::optional<std::string> text_end = std::nullopt, float wrap_width = 0.0f, const ImVec4 * cpu_fine_clip_rect = NULL)
                {
                    const char * text_end_adapt_default_null = nullptr;
                    if (text_end.has_value())
                        text_end_adapt_default_null = text_end.value().c_str();

                    self.AddText(font, font_size, pos, col, text_begin, text_end_adapt_default_null, wrap_width, cpu_fine_clip_rect);
                };

                AddText_adapt_const_char_pointer_with_default_null(font, font_size, pos, col, text_begin, text_end, wrap_width, cpu_fine_clip_rect);
            },     nb::arg("font"), nb::arg("font_size"), nb::arg("pos"), nb::arg("col"), nb::arg("text_begin"), nb::arg("text_end").none() = nb::none(), nb::arg("wrap_width") = 0.0f, nb::arg("cpu_fine_clip_rect") = nb::none())
        .def("add_bezier_cubic",
            &ImDrawList::AddBezierCubic,
            nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("p4"), nb::arg("col"), nb::arg("thickness"), nb::arg("num_segments") = 0,
            "Cubic Bezier (4 control points)")
        .def("add_bezier_quadratic",
            &ImDrawList::AddBezierQuadratic,
            nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("col"), nb::arg("thickness"), nb::arg("num_segments") = 0,
            "Quadratic Bezier (3 control points)")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("add_polyline",
            nb::overload_cast<const std::vector<ImVec2> &, ImU32, ImDrawFlags, float>(&ImDrawList::AddPolyline), nb::arg("points"), nb::arg("col"), nb::arg("flags"), nb::arg("thickness"))
        .def("add_convex_poly_filled",
            nb::overload_cast<const std::vector<ImVec2> &, ImU32>(&ImDrawList::AddConvexPolyFilled), nb::arg("points"), nb::arg("col"))
        .def("add_concave_poly_filled",
            nb::overload_cast<const std::vector<ImVec2> &, ImU32>(&ImDrawList::AddConcavePolyFilled), nb::arg("points"), nb::arg("col"))
        // #endif
        //
        .def("add_image",
            [](ImDrawList & self, ImTextureRef tex_ref, const ImVec2 & p_min, const ImVec2 & p_max, const std::optional<const ImVec2> & uv_min = std::nullopt, const std::optional<const ImVec2> & uv_max = std::nullopt, ImU32 col = IM_COL32_WHITE)
            {
                auto AddImage_adapt_mutable_param_with_default_value = [&self](ImTextureRef tex_ref, const ImVec2 & p_min, const ImVec2 & p_max, const std::optional<const ImVec2> & uv_min = std::nullopt, const std::optional<const ImVec2> & uv_max = std::nullopt, ImU32 col = IM_COL32_WHITE)
                {

                    const ImVec2& uv_min_or_default = [&]() -> const ImVec2 {
                        if (uv_min.has_value())
                            return uv_min.value();
                        else
                            return ImVec2(0, 0);
                    }();

                    const ImVec2& uv_max_or_default = [&]() -> const ImVec2 {
                        if (uv_max.has_value())
                            return uv_max.value();
                        else
                            return ImVec2(1, 1);
                    }();

                    self.AddImage(tex_ref, p_min, p_max, uv_min_or_default, uv_max_or_default, col);
                };

                AddImage_adapt_mutable_param_with_default_value(tex_ref, p_min, p_max, uv_min, uv_max, col);
            },
            nb::arg("tex_ref"), nb::arg("p_min"), nb::arg("p_max"), nb::arg("uv_min").none() = nb::none(), nb::arg("uv_max").none() = nb::none(), nb::arg("col") = IM_COL32_WHITE,
            "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv_min: ImVec2(0, 0)\n        * uv_max: ImVec2(1, 1)")
        .def("add_image_quad",
            [](ImDrawList & self, ImTextureRef tex_ref, const ImVec2 & p1, const ImVec2 & p2, const ImVec2 & p3, const ImVec2 & p4, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec2> & uv2 = std::nullopt, const std::optional<const ImVec2> & uv3 = std::nullopt, const std::optional<const ImVec2> & uv4 = std::nullopt, ImU32 col = IM_COL32_WHITE)
            {
                auto AddImageQuad_adapt_mutable_param_with_default_value = [&self](ImTextureRef tex_ref, const ImVec2 & p1, const ImVec2 & p2, const ImVec2 & p3, const ImVec2 & p4, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec2> & uv2 = std::nullopt, const std::optional<const ImVec2> & uv3 = std::nullopt, const std::optional<const ImVec2> & uv4 = std::nullopt, ImU32 col = IM_COL32_WHITE)
                {

                    const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                        if (uv1.has_value())
                            return uv1.value();
                        else
                            return ImVec2(0, 0);
                    }();

                    const ImVec2& uv2_or_default = [&]() -> const ImVec2 {
                        if (uv2.has_value())
                            return uv2.value();
                        else
                            return ImVec2(1, 0);
                    }();

                    const ImVec2& uv3_or_default = [&]() -> const ImVec2 {
                        if (uv3.has_value())
                            return uv3.value();
                        else
                            return ImVec2(1, 1);
                    }();

                    const ImVec2& uv4_or_default = [&]() -> const ImVec2 {
                        if (uv4.has_value())
                            return uv4.value();
                        else
                            return ImVec2(0, 1);
                    }();

                    self.AddImageQuad(tex_ref, p1, p2, p3, p4, uv1_or_default, uv2_or_default, uv3_or_default, uv4_or_default, col);
                };

                AddImageQuad_adapt_mutable_param_with_default_value(tex_ref, p1, p2, p3, p4, uv1, uv2, uv3, uv4, col);
            },
            nb::arg("tex_ref"), nb::arg("p1"), nb::arg("p2"), nb::arg("p3"), nb::arg("p4"), nb::arg("uv1").none() = nb::none(), nb::arg("uv2").none() = nb::none(), nb::arg("uv3").none() = nb::none(), nb::arg("uv4").none() = nb::none(), nb::arg("col") = IM_COL32_WHITE,
            "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv1: ImVec2(0, 0)\n        * uv2: ImVec2(1, 0)\n        * uv3: ImVec2(1, 1)\n        * uv4: ImVec2(0, 1)")
        .def("add_image_rounded",
            &ImDrawList::AddImageRounded, nb::arg("tex_ref"), nb::arg("p_min"), nb::arg("p_max"), nb::arg("uv_min"), nb::arg("uv_max"), nb::arg("col"), nb::arg("rounding"), nb::arg("flags") = 0)
        .def("path_clear",
            &ImDrawList::PathClear, "(private API)")
        .def("path_line_to",
            &ImDrawList::PathLineTo,
            nb::arg("pos"),
            "(private API)")
        .def("path_line_to_merge_duplicate",
            &ImDrawList::PathLineToMergeDuplicate,
            nb::arg("pos"),
            "(private API)")
        .def("path_fill_convex",
            &ImDrawList::PathFillConvex,
            nb::arg("col"),
            "(private API)")
        .def("path_fill_concave",
            &ImDrawList::PathFillConcave,
            nb::arg("col"),
            "(private API)")
        .def("path_stroke",
            &ImDrawList::PathStroke,
            nb::arg("col"), nb::arg("flags") = 0, nb::arg("thickness") = 1.0f,
            "(private API)")
        .def("path_arc_to",
            &ImDrawList::PathArcTo, nb::arg("center"), nb::arg("radius"), nb::arg("a_min"), nb::arg("a_max"), nb::arg("num_segments") = 0)
        .def("path_arc_to_fast",
            &ImDrawList::PathArcToFast,
            nb::arg("center"), nb::arg("radius"), nb::arg("a_min_of_12"), nb::arg("a_max_of_12"),
            "Use precomputed angles for a 12 steps circle")
        .def("path_elliptical_arc_to",
            &ImDrawList::PathEllipticalArcTo,
            nb::arg("center"), nb::arg("radius"), nb::arg("rot"), nb::arg("a_min"), nb::arg("a_max"), nb::arg("num_segments") = 0,
            "Ellipse")
        .def("path_bezier_cubic_curve_to",
            &ImDrawList::PathBezierCubicCurveTo,
            nb::arg("p2"), nb::arg("p3"), nb::arg("p4"), nb::arg("num_segments") = 0,
            "Cubic Bezier (4 control points)")
        .def("path_bezier_quadratic_curve_to",
            &ImDrawList::PathBezierQuadraticCurveTo,
            nb::arg("p2"), nb::arg("p3"), nb::arg("num_segments") = 0,
            "Quadratic Bezier (3 control points)")
        .def("path_rect",
            &ImDrawList::PathRect, nb::arg("rect_min"), nb::arg("rect_max"), nb::arg("rounding") = 0.0f, nb::arg("flags") = 0)
        .def("add_draw_cmd",
            &ImDrawList::AddDrawCmd, "This is useful if you need to forcefully create a new draw call (to allow for dependent rendering / blending). Otherwise primitives are merged into the same draw-call as much as possible")
        .def("clone_output",
            &ImDrawList::CloneOutput,
            "Create a clone of the CmdBuffer/IdxBuffer/VtxBuffer. For multi-threaded rendering, consider using `imgui_threaded_rendering` from https://github.com/ocornut/imgui_club instead.",
            nb::rv_policy::reference)
        .def("channels_split",
            &ImDrawList::ChannelsSplit,
            nb::arg("count"),
            "(private API)")
        .def("channels_merge",
            &ImDrawList::ChannelsMerge, "(private API)")
        .def("channels_set_current",
            &ImDrawList::ChannelsSetCurrent,
            nb::arg("n"),
            "(private API)")
        .def("prim_reserve",
            &ImDrawList::PrimReserve, nb::arg("idx_count"), nb::arg("vtx_count"))
        .def("prim_unreserve",
            &ImDrawList::PrimUnreserve, nb::arg("idx_count"), nb::arg("vtx_count"))
        .def("prim_rect",
            &ImDrawList::PrimRect,
            nb::arg("a"), nb::arg("b"), nb::arg("col"),
            "Axis aligned rectangle (composed of two triangles)")
        .def("prim_rect_uv",
            &ImDrawList::PrimRectUV, nb::arg("a"), nb::arg("b"), nb::arg("uv_a"), nb::arg("uv_b"), nb::arg("col"))
        .def("prim_quad_uv",
            &ImDrawList::PrimQuadUV, nb::arg("a"), nb::arg("b"), nb::arg("c"), nb::arg("d"), nb::arg("uv_a"), nb::arg("uv_b"), nb::arg("uv_c"), nb::arg("uv_d"), nb::arg("col"))
        .def("prim_write_vtx",
            &ImDrawList::PrimWriteVtx,
            nb::arg("pos"), nb::arg("uv"), nb::arg("col"),
            "(private API)")
        .def("prim_write_idx",
            &ImDrawList::PrimWriteIdx,
            nb::arg("idx"),
            "(private API)")
        .def("prim_vtx",
            &ImDrawList::PrimVtx,
            nb::arg("pos"), nb::arg("uv"), nb::arg("col"),
            "(private API)\n\n Write vertex with unique index")
        .def("_set_draw_list_shared_data",
            nb::overload_cast<ImDrawListSharedData *>(&ImDrawList::_SetDrawListSharedData), nb::arg("data"))
        .def("_reset_for_new_frame",
            &ImDrawList::_ResetForNewFrame)
        .def("_clear_free_memory",
            &ImDrawList::_ClearFreeMemory)
        .def("_pop_unused_draw_cmd",
            &ImDrawList::_PopUnusedDrawCmd)
        .def("_try_merge_draw_cmds",
            &ImDrawList::_TryMergeDrawCmds)
        .def("_on_changed_clip_rect",
            &ImDrawList::_OnChangedClipRect)
        .def("_on_changed_texture",
            &ImDrawList::_OnChangedTexture)
        .def("_on_changed_vtx_offset",
            &ImDrawList::_OnChangedVtxOffset)
        .def("_set_texture",
            &ImDrawList::_SetTexture, nb::arg("tex_ref"))
        .def("_calc_circle_auto_segment_count",
            &ImDrawList::_CalcCircleAutoSegmentCount, nb::arg("radius"))
        .def("_path_arc_to_fast_ex",
            &ImDrawList::_PathArcToFastEx, nb::arg("center"), nb::arg("radius"), nb::arg("a_min_sample"), nb::arg("a_max_sample"), nb::arg("a_step"))
        .def("_path_arc_to_n",
            &ImDrawList::_PathArcToN, nb::arg("center"), nb::arg("radius"), nb::arg("a_min"), nb::arg("a_max"), nb::arg("num_segments"))
        ;


    auto pyClassImDrawData =
        nb::class_<ImDrawData>
            (m, "ImDrawData", " All draw data to render a Dear ImGui frame\n (NB: the style and the naming convention here is a little inconsistent, we currently preserve them for backward compatibility purpose,\n as this is one of the oldest structure exposed by the library! Basically, ImDrawList == CmdList)")
        .def_rw("valid", &ImDrawData::Valid, "Only valid after Render() is called and before the next NewFrame() is called.")
        .def_rw("cmd_lists_count", &ImDrawData::CmdListsCount, "== CmdLists.Size. (OBSOLETE: exists for legacy reasons). Number of ImDrawList* to render.")
        .def_rw("total_idx_count", &ImDrawData::TotalIdxCount, "For convenience, sum of all ImDrawList's IdxBuffer.Size")
        .def_rw("total_vtx_count", &ImDrawData::TotalVtxCount, "For convenience, sum of all ImDrawList's VtxBuffer.Size")
        .def_rw("cmd_lists", &ImDrawData::CmdLists, "Array of ImDrawList* to render. The ImDrawLists are owned by ImGuiContext and only pointed to from here.")
        .def_rw("display_pos", &ImDrawData::DisplayPos, "Top-left position of the viewport to render (== top-left of the orthogonal projection matrix to use) (== GetMainViewport()->Pos for the main viewport, == (0.0) in most single-viewport applications)")
        .def_rw("display_size", &ImDrawData::DisplaySize, "Size of the viewport to render (== GetMainViewport()->Size for the main viewport, == io.DisplaySize in most single-viewport applications)")
        .def_rw("framebuffer_scale", &ImDrawData::FramebufferScale, "Amount of pixels for each unit of DisplaySize. Copied from viewport->FramebufferScale (== io.DisplayFramebufferScale for main viewport). Generally (1,1) on normal display, (2,2) on OSX with Retina display.")
        .def_rw("owner_viewport", &ImDrawData::OwnerViewport, "Viewport carrying the ImDrawData instance, might be of use to the renderer (generally not).")
        .def_rw("textures", &ImDrawData::Textures, "List of textures to update. Most of the times the list is shared by all ImDrawData, has only 1 texture and it doesn't need any update. This almost always points to ImGui::GetPlatformIO().Textures[]. May be overriden or set to None if you want to manually update textures.")
        .def(nb::init<>(),
            "Functions")
        .def("clear",
            &ImDrawData::Clear)
        .def("add_draw_list",
            nb::overload_cast<ImDrawList *>(&ImDrawData::AddDrawList),
            nb::arg("draw_list"),
            "Helper to add an external draw list into an existing ImDrawData.")
        .def("de_index_all_buffers",
            &ImDrawData::DeIndexAllBuffers, "Helper to convert all buffers from indexed to non-indexed, in case you cannot render indexed. Note: this is slow and most likely a waste of resources. Always prefer indexed rendering!")
        .def("scale_clip_rects",
            &ImDrawData::ScaleClipRects,
            nb::arg("fb_scale"),
            "Helper to scale the ClipRect field of each ImDrawCmd. Use if your final output buffer is at a different scale than Dear ImGui expects, or if there is a difference between your window resolution and framebuffer resolution.")
        ;


    auto pyEnumImTextureFormat =
        nb::enum_<ImTextureFormat>(m, "ImTextureFormat", nb::is_arithmetic(), nb::is_flag(), " We intentionally support a limited amount of texture formats to limit burden on CPU-side code and extension.\n Most standard backends only support RGBA32 but we provide a single channel option for low-resource/embedded systems.")
            .value("rgba32", ImTextureFormat_RGBA32, "4 components per pixel, each is unsigned 8-bit. Total size = TexWidth * TexHeight * 4")
            .value("alpha8", ImTextureFormat_Alpha8, "1 component per pixel, each is unsigned 8-bit. Total size = TexWidth * TexHeight");


    auto pyEnumImTextureStatus =
        nb::enum_<ImTextureStatus>(m, "ImTextureStatus", nb::is_arithmetic(), nb::is_flag(), "Status of a texture to communicate with Renderer Backend.")
            .value("ok", ImTextureStatus_OK, "")
            .value("destroyed", ImTextureStatus_Destroyed, "Backend destroyed the texture.")
            .value("want_create", ImTextureStatus_WantCreate, "Requesting backend to create the texture. Set status OK when done.")
            .value("want_updates", ImTextureStatus_WantUpdates, "Requesting backend to update specific blocks of pixels (write to texture portions which have never been used before). Set status OK when done.")
            .value("want_destroy", ImTextureStatus_WantDestroy, "Requesting backend to destroy the texture. Set status to Destroyed when done.");


    auto pyClassImTextureRect =
        nb::class_<ImTextureRect>
            (m, "ImTextureRect", " Coordinates of a rectangle within a texture.\n When a texture is in ImTextureStatus_WantUpdates state, we provide a list of individual rectangles to copy to the graphics system.\n You may use ImTextureData::Updates[] for the list, or ImTextureData::UpdateBox for a single bounding box.")
        .def(nb::init<>()) // implicit default constructor
        .def_rw("x", &ImTextureRect::x, "Upper-left coordinates of rectangle to update")
        .def_rw("y", &ImTextureRect::y, "Upper-left coordinates of rectangle to update")
        .def_rw("w", &ImTextureRect::w, "Size of rectangle to update (in pixels)")
        .def_rw("h", &ImTextureRect::h, "Size of rectangle to update (in pixels)")
        ;


    auto pyClassImTextureData =
        nb::class_<ImTextureData>
            (m, "ImTextureData", " Specs and pixel storage for a texture used by Dear ImGui.\n This is only useful for (1) core library and (2) backends. End-user/applications do not need to care about this.\n Renderer Backends will create a GPU-side version of this.\n Why does we store two identifiers: TexID and BackendUserData?\n - ImTextureID    TexID           = lower-level identifier stored in ImDrawCmd. ImDrawCmd can refer to textures not created by the backend, and for which there's no ImTextureData.\n - None*          BackendUserData = higher-level opaque storage for backend own book-keeping. Some backends may have enough with TexID and not need both.\n In columns below: who reads/writes each fields? 'r'=read, 'w'=write, 'core'=main library, 'backend'=renderer backend")
        .def_rw("unique_id", &ImTextureData::UniqueID, "w    -   // [DEBUG] Sequential index to facilitate identifying a texture when debugging/printing. Unique per atlas.")
        .def_rw("status", &ImTextureData::Status, "rw   rw  // ImTextureStatus_OK/_WantCreate/_WantUpdates/_WantDestroy. Always use SetStatus() to modify!")
        .def_rw("backend_user_data", &ImTextureData::BackendUserData, "-    rw  // Convenience storage for backend. Some backends may have enough with TexID.")
        .def_rw("tex_id", &ImTextureData::TexID, "r    w   // Backend-specific texture identifier. Always use SetTexID() to modify! The identifier will stored in ImDrawCmd::GetTexID() and passed to backend's RenderDrawData function.")
        .def_rw("format", &ImTextureData::Format, "w    r   // ImTextureFormat_RGBA32 (default) or ImTextureFormat_Alpha8")
        .def_rw("width", &ImTextureData::Width, "w    r   // Texture width")
        .def_rw("height", &ImTextureData::Height, "w    r   // Texture height")
        .def_rw("bytes_per_pixel", &ImTextureData::BytesPerPixel, "w    r   // 4 or 1")
        .def_rw("pixels", &ImTextureData::Pixels, "w    r   // Pointer to buffer holding 'Width*Height' pixels and 'Width*Height*BytesPerPixels' bytes.")
        .def_rw("used_rect", &ImTextureData::UsedRect, "w    r   // Bounding box encompassing all past and queued Updates[].")
        .def_rw("update_rect", &ImTextureData::UpdateRect, "w    r   // Bounding box encompassing all queued Updates[].")
        .def_rw("updates", &ImTextureData::Updates, "w    r   // Array of individual updates.")
        .def_rw("unused_frames", &ImTextureData::UnusedFrames, "w    r   // In order to facilitate handling Status==WantDestroy in some backend: this is a count successive frames where the texture was not used. Always >0 when Status==WantDestroy.")
        .def_rw("ref_count", &ImTextureData::RefCount, "w    r   // Number of contexts using this texture. Used during backend shutdown.")
        .def_rw("use_colors", &ImTextureData::UseColors, "w    r   // Tell whether our texture data is known to use colors (rather than just white + alpha).")
        .def_rw("want_destroy_next_frame", &ImTextureData::WantDestroyNextFrame, "rw   -   // [Internal] Queued to set ImTextureStatus_WantDestroy next frame. May still be used in the current frame.")
        .def(nb::init<>(),
            "Functions")
        .def("create",
            &ImTextureData::Create, nb::arg("format"), nb::arg("w"), nb::arg("h"))
        .def("destroy_pixels",
            &ImTextureData::DestroyPixels)
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("get_pixels_array",
            &ImTextureData::GetPixelsArray, " GetPixelsArray(): returns the pixel data as a NumPy array.\n\n Note: GetPixelsAt(x, y) is not implemented for Python, but you can use the offset below:\n    offset = (y * tex.width + x) * tex.bytes_per_pixel\n(private API)")
        // #endif
        //
        .def("get_size_in_bytes",
            &ImTextureData::GetSizeInBytes, "(private API)")
        .def("get_pitch",
            &ImTextureData::GetPitch, "(private API)")
        .def("get_tex_ref",
            &ImTextureData::GetTexRef, "(private API)")
        .def("get_tex_id",
            &ImTextureData::GetTexID, "(private API)")
        .def("set_tex_id",
            &ImTextureData::SetTexID,
            nb::arg("tex_id"),
            "(private API)\n\n Call after creating or destroying the texture. Never modify TexID directly!")
        .def("set_status",
            &ImTextureData::SetStatus,
            nb::arg("status"),
            "(private API)\n\n Call after honoring a request. Never modify Status directly!")
        ;


    auto pyClassImFontConfig =
        nb::class_<ImFontConfig>
            (m, "ImFontConfig", "A font input/source (we may rename this to ImFontSource in the future)")
        .def_rw("font_data", &ImFontConfig::FontData, "// TTF/OTF data")
        .def_rw("font_data_size", &ImFontConfig::FontDataSize, "// TTF/OTF data size")
        .def_rw("font_data_owned_by_atlas", &ImFontConfig::FontDataOwnedByAtlas, "True     // TTF/OTF data ownership taken by the container ImFontAtlas (will delete memory itself).")
        .def_rw("merge_mode", &ImFontConfig::MergeMode, "False    // Merge into previous ImFont, so you can combine multiple inputs font into one ImFont (e.g. ASCII font + icons + Japanese glyphs). You may want to use GlyphOffset.y when merge font of different heights.")
        .def_rw("pixel_snap_h", &ImFontConfig::PixelSnapH, "False    // Align every glyph AdvanceX to pixel boundaries. Useful e.g. if you are merging a non-pixel aligned font with the default font. If enabled, you can set OversampleH/V to 1.")
        .def_rw("pixel_snap_v", &ImFontConfig::PixelSnapV, "True     // Align Scaled GlyphOffset.y to pixel boundaries.")
        .def_rw("oversample_h", &ImFontConfig::OversampleH, "0 (2)    // Rasterize at higher quality for sub-pixel positioning. 0 == auto == 1 or 2 depending on size. Note the difference between 2 and 3 is minimal. You can reduce this to 1 for large glyphs save memory. Read https://github.com/nothings/stb/blob/master/tests/oversample/README.md for details.")
        .def_rw("oversample_v", &ImFontConfig::OversampleV, "0 (1)    // Rasterize at higher quality for sub-pixel positioning. 0 == auto == 1. This is not really useful as we don't use sub-pixel positions on the Y axis.")
        .def_rw("ellipsis_char", &ImFontConfig::EllipsisChar, "0        // Explicitly specify Unicode codepoint of ellipsis character. When fonts are being merged first specified ellipsis will be used.")
        .def_rw("size_pixels", &ImFontConfig::SizePixels, "// Size in pixels for rasterizer (more or less maps to the resulting font height).")
        .def_rw("glyph_offset", &ImFontConfig::GlyphOffset, "0, 0     // Offset (in pixels) all glyphs from this font input. Absolute value for default size, other sizes will scale this value.")
        .def_rw("glyph_min_advance_x", &ImFontConfig::GlyphMinAdvanceX, "0        // Minimum AdvanceX for glyphs, set Min to align font icons, set both Min/Max to enforce mono-space font. Absolute value for default size, other sizes will scale this value.")
        .def_rw("glyph_max_advance_x", &ImFontConfig::GlyphMaxAdvanceX, "FLT_MAX  // Maximum AdvanceX for glyphs")
        .def_rw("glyph_extra_advance_x", &ImFontConfig::GlyphExtraAdvanceX, "0        // Extra spacing (in pixels) between glyphs. Please contact us if you are using this. // FIXME-NEWATLAS: Intentionally unscaled")
        .def_rw("font_no", &ImFontConfig::FontNo, "0        // Index of font within TTF/OTF file")
        .def_rw("font_loader_flags", &ImFontConfig::FontLoaderFlags, "0        // Settings for custom font builder. THIS IS BUILDER IMPLEMENTATION DEPENDENT. Leave as zero if unsure.")
        .def_rw("rasterizer_multiply", &ImFontConfig::RasterizerMultiply, "1.0     // Linearly brighten (>1.0) or darken (<1.0) font output. Brightening small fonts may be a good workaround to make them more readable. This is a silly thing we may remove in the future.")
        .def_rw("rasterizer_density", &ImFontConfig::RasterizerDensity, "1.0     // [LEGACY: this only makes sense when ImGuiBackendFlags_RendererHasTextures is not supported] DPI scale multiplier for rasterization. Not altering other font metrics: makes it easy to swap between e.g. a 100% and a 400% fonts for a zooming display, or handle Retina screen. IMPORTANT: If you change this it is expected that you increase/decrease font scale roughly to the inverse of this, otherwise quality may look lowered.")
        .def_rw("flags", &ImFontConfig::Flags, "Font flags (don't use just yet, will be exposed in upcoming 1.92.X updates)")
        .def_rw("dst_font", &ImFontConfig::DstFont, "Target font (as we merging fonts, multiple ImFontConfig may target the same font)")
        .def_rw("font_loader_data", &ImFontConfig::FontLoaderData, "Font loader opaque storage (per font config)")
        .def(nb::init<>())
        ;


    auto pyClassImFontGlyph =
        nb::class_<ImFontGlyph>
            (m, "ImFontGlyph", " Hold rendering data for one glyph.\n (Note: some language parsers may fail to convert the bitfield members, in this case maybe drop store a single u32 or we can rework this)")
        .def_rw("advance_x", &ImFontGlyph::AdvanceX, "Horizontal distance to advance cursor/layout position.")
        .def_rw("x0", &ImFontGlyph::X0, "Glyph corners. Offsets from current cursor/layout position.")
        .def_rw("y0", &ImFontGlyph::Y0, "Glyph corners. Offsets from current cursor/layout position.")
        .def_rw("x1", &ImFontGlyph::X1, "Glyph corners. Offsets from current cursor/layout position.")
        .def_rw("y1", &ImFontGlyph::Y1, "Glyph corners. Offsets from current cursor/layout position.")
        .def_rw("u0", &ImFontGlyph::U0, "Texture coordinates for the current value of ImFontAtlas->TexRef. Cached equivalent of calling GetCustomRect() with PackId.")
        .def_rw("v0", &ImFontGlyph::V0, "Texture coordinates for the current value of ImFontAtlas->TexRef. Cached equivalent of calling GetCustomRect() with PackId.")
        .def_rw("u1", &ImFontGlyph::U1, "Texture coordinates for the current value of ImFontAtlas->TexRef. Cached equivalent of calling GetCustomRect() with PackId.")
        .def_rw("v1", &ImFontGlyph::V1, "Texture coordinates for the current value of ImFontAtlas->TexRef. Cached equivalent of calling GetCustomRect() with PackId.")
        .def_rw("pack_id", &ImFontGlyph::PackId, "[Internal] ImFontAtlasRectId value (FIXME: Cold data, could be moved elsewhere?)")
        .def(nb::init<>())
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("is_colored",
            &ImFontGlyph::isColored, "(private API)")
        .def("is_visible",
            &ImFontGlyph::isVisible, "(private API)")
        .def("get_codepoint",
            &ImFontGlyph::getCodepoint, "(private API)")
        // #endif
        ;


    auto pyClassImFontGlyphRangesBuilder =
        nb::class_<ImFontGlyphRangesBuilder>
            (m, "ImFontGlyphRangesBuilder", " Helper to build glyph ranges from text/string data. Feed your application strings/characters to it then call BuildRanges().\n This is essentially a tightly packed of vector of 64k booleans = 8KB storage.")
        .def_rw("used_chars", &ImFontGlyphRangesBuilder::UsedChars, "Store 1-bit per Unicode code point (0=unused, 1=used)")
        .def(nb::init<>())
        .def("clear",
            &ImFontGlyphRangesBuilder::Clear, "(private API)")
        .def("get_bit",
            &ImFontGlyphRangesBuilder::GetBit,
            nb::arg("n"),
            "(private API)\n\n Get bit n in the array")
        .def("set_bit",
            &ImFontGlyphRangesBuilder::SetBit,
            nb::arg("n"),
            "(private API)\n\n Set bit n in the array")
        .def("add_char",
            &ImFontGlyphRangesBuilder::AddChar,
            nb::arg("c"),
            "(private API)\n\n Add character")
        .def("add_text",
            [](ImFontGlyphRangesBuilder & self, const char * text, std::optional<std::string> text_end = std::nullopt)
            {
                auto AddText_adapt_const_char_pointer_with_default_null = [&self](const char * text, std::optional<std::string> text_end = std::nullopt)
                {
                    const char * text_end_adapt_default_null = nullptr;
                    if (text_end.has_value())
                        text_end_adapt_default_null = text_end.value().c_str();

                    self.AddText(text, text_end_adapt_default_null);
                };

                AddText_adapt_const_char_pointer_with_default_null(text, text_end);
            },
            nb::arg("text"), nb::arg("text_end").none() = nb::none(),
            "Add string (each character of the UTF-8 string are added)")
        .def("build_ranges",
            &ImFontGlyphRangesBuilder::BuildRanges,
            nb::arg("out_ranges"),
            "Output new ranges")
        ;


    auto pyClassImFontAtlasRect =
        nb::class_<ImFontAtlasRect>
            (m, "ImFontAtlasRect", " Output of ImFontAtlas::GetCustomRect() when using custom rectangles.\n Those values may not be cached/stored as they are only valid for the current value of atlas->TexRef\n (this is in theory derived from ImTextureRect but we use separate structures for reasons)")
        .def_rw("x", &ImFontAtlasRect::x, "Position (in current texture)")
        .def_rw("y", &ImFontAtlasRect::y, "Position (in current texture)")
        .def_rw("w", &ImFontAtlasRect::w, "Size")
        .def_rw("h", &ImFontAtlasRect::h, "Size")
        .def_rw("uv0", &ImFontAtlasRect::uv0, "UV coordinates (in current texture)")
        .def_rw("uv1", &ImFontAtlasRect::uv1, "UV coordinates (in current texture)")
        .def(nb::init<>())
        ;


    auto pyEnumImFontAtlasFlags_ =
        nb::enum_<ImFontAtlasFlags_>(m, "ImFontAtlasFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImFontAtlas build")
            .value("none", ImFontAtlasFlags_None, "")
            .value("no_power_of_two_height", ImFontAtlasFlags_NoPowerOfTwoHeight, "Don't round the height to next power of two")
            .value("no_mouse_cursors", ImFontAtlasFlags_NoMouseCursors, "Don't build software mouse cursors into the atlas (save a little texture memory)")
            .value("no_baked_lines", ImFontAtlasFlags_NoBakedLines, "Don't build thick line textures into the atlas (save a little texture memory, allow support for point/nearest filtering). The AntiAliasedLinesUseTex features uses them, otherwise they will be rendered using polygons (more expensive for CPU/GPU).");


    auto pyClassImFontAtlas =
        nb::class_<ImFontAtlas>
            (m, "ImFontAtlas", " Load and rasterize multiple TTF/OTF fonts into a same texture. The font atlas will build a single texture holding:\n  - One or more fonts.\n  - Custom graphics data needed to render the shapes needed by Dear ImGui.\n  - Mouse cursor shapes for software cursor rendering (unless setting 'Flags |= ImFontAtlasFlags_NoMouseCursors' in the font atlas).\n  - If you don't call any AddFont*** functions, the default font embedded in the code will be loaded for you.\n It is the rendering backend responsibility to upload texture into your graphics API:\n  - ImGui_ImplXXXX_RenderDrawData() functions generally iterate platform_io->Textures[] to create/update/destroy each ImTextureData instance.\n  - Backend then set ImTextureData's TexID and BackendUserData.\n  - Texture id are passed back to you during rendering to identify the texture. Read FAQ entry about ImTextureID/ImTextureRef for more details.\n Legacy path:\n  - Call Build() + GetTexDataAsAlpha8() or GetTexDataAsRGBA32() to build and retrieve pixels data.\n  - Call SetTexID(my_tex_id); and pass the pointer/identifier to your texture in a format natural to your graphics API.\n Common pitfalls:\n - If you pass a 'glyph_ranges' array to AddFont*** functions, you need to make sure that your array persist up until the\n   atlas is build (when calling GetTexData*** or Build()). We only copy the pointer, not the data.\n - Important: By default, AddFontFromMemoryTTF() takes ownership of the data. Even though we are not writing to it, we will free the pointer on destruction.\n   You can set font_cfg->FontDataOwnedByAtlas=False to keep ownership of your data and it won't be freed,\n - Even though many functions are suffixed with \"TTF\", OTF data is supported just as well.\n - This is an old API and it is currently awkward for those and various other reasons! We will address them in the future!")
        .def(nb::init<>())
        .def("add_font",
            &ImFontAtlas::AddFont,
            nb::arg("font_cfg"),
            nb::rv_policy::reference)
        .def("add_font_default",
            &ImFontAtlas::AddFontDefault,
            nb::arg("font_cfg") = nb::none(),
            nb::rv_policy::reference)
        .def("remove_font",
            &ImFontAtlas::RemoveFont, nb::arg("font"))
        .def("clear",
            &ImFontAtlas::Clear, "Clear everything (input fonts, output glyphs/textures)")
        .def("compact_cache",
            &ImFontAtlas::CompactCache, "Compact cached glyphs and texture.")
        .def("set_font_loader",
            &ImFontAtlas::SetFontLoader,
            nb::arg("font_loader"),
            "Change font loader at runtime.")
        .def("clear_input_data",
            &ImFontAtlas::ClearInputData, "[OBSOLETE] Clear input data (all ImFontConfig structures including sizes, TTF data, glyph ranges, etc.) = all the data used to build the texture and fonts.")
        .def("clear_fonts",
            &ImFontAtlas::ClearFonts, "[OBSOLETE] Clear input+output font data (same as ClearInputData() + glyphs storage, UV coordinates).")
        .def("clear_tex_data",
            &ImFontAtlas::ClearTexData, "[OBSOLETE] Clear CPU-side copy of the texture data. Saves RAM once the texture has been copied to graphics memory.")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("add_font_from_file_ttf",
            &ImFontAtlas::_AddFontFromFileTTF,
            nb::arg("filename"), nb::arg("size_pixels"), nb::arg("font_cfg") = nb::none(),
            nb::rv_policy::reference)
        // #endif
        //
        .def("add_custom_rect",
            &ImFontAtlas::AddCustomRect,
            nb::arg("width"), nb::arg("height"), nb::arg("out_r") = nb::none(),
            "Register a rectangle. Return -1 (ImFontAtlasRectId_Invalid) on error.")
        .def("remove_custom_rect",
            &ImFontAtlas::RemoveCustomRect,
            nb::arg("id_"),
            "Unregister a rectangle. Existing pixels will stay in texture until resized / garbage collected.")
        .def("get_custom_rect",
            &ImFontAtlas::GetCustomRect,
            nb::arg("id_"), nb::arg("out_r"),
            "Get rectangle coordinates for current texture. Valid immediately, never store this (read above)!")
        .def_rw("flags", &ImFontAtlas::Flags, "Build flags (see ImFontAtlasFlags_)")
        .def_rw("tex_desired_format", &ImFontAtlas::TexDesiredFormat, "Desired texture format (default to ImTextureFormat_RGBA32 but may be changed to ImTextureFormat_Alpha8).")
        .def_rw("tex_glyph_padding", &ImFontAtlas::TexGlyphPadding, "FIXME: Should be called \"TexPackPadding\". Padding between glyphs within texture in pixels. Defaults to 1. If your rendering method doesn't rely on bilinear filtering you may set this to 0 (will also need to set AntiAliasedLinesUseTex = False).")
        .def_rw("tex_min_width", &ImFontAtlas::TexMinWidth, "Minimum desired texture width. Must be a power of two. Default to 512.")
        .def_rw("tex_min_height", &ImFontAtlas::TexMinHeight, "Minimum desired texture height. Must be a power of two. Default to 128.")
        .def_rw("tex_max_width", &ImFontAtlas::TexMaxWidth, "Maximum desired texture width. Must be a power of two. Default to 8192.")
        .def_rw("tex_max_height", &ImFontAtlas::TexMaxHeight, "Maximum desired texture height. Must be a power of two. Default to 8192.")
        .def_rw("user_data", &ImFontAtlas::UserData, "Store your own atlas related user-data (if e.g. you have multiple font atlas).")
        .def_rw("tex_data", &ImFontAtlas::TexData, "Latest texture.")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("python_set_texture_id",
            &ImFontAtlas::Python_SetTextureID,
            nb::arg("id_"),
            "(private API)")
        .def("python_get_texture_id",
            &ImFontAtlas::Python_GetTextureID, "(private API)")
        // #endif
        //
        .def_rw("tex_list", &ImFontAtlas::TexList, "Texture list (most often TexList.Size == 1). TexData is always == TexList.back(). DO NOT USE DIRECTLY, USE GetDrawData().Textures[]/GetPlatformIO().Textures[] instead!")
        .def_rw("locked", &ImFontAtlas::Locked, "Marked as locked during ImGui::NewFrame()..EndFrame() scope if TexUpdates are not supported. Any attempt to modify the atlas will assert.")
        .def_rw("renderer_has_textures", &ImFontAtlas::RendererHasTextures, "Copy of (BackendFlags & ImGuiBackendFlags_RendererHasTextures) from supporting context.")
        .def_rw("tex_is_built", &ImFontAtlas::TexIsBuilt, "Set when texture was built matching current font input. Mostly useful for legacy IsBuilt() call.")
        .def_rw("tex_pixels_use_colors", &ImFontAtlas::TexPixelsUseColors, "Tell whether our texture data is known to use colors (rather than just alpha channel), in order to help backend select a format or conversion process.")
        .def_rw("tex_uv_scale", &ImFontAtlas::TexUvScale, "= (1.0/TexData->TexWidth, 1.0/TexData->TexHeight). May change as new texture gets created.")
        .def_rw("tex_uv_white_pixel", &ImFontAtlas::TexUvWhitePixel, "Texture coordinates to a white pixel. May change as new texture gets created.")
        .def_rw("fonts", &ImFontAtlas::Fonts, "Hold all the fonts returned by AddFont*. Fonts[0] is the default font upon calling ImGui::NewFrame(), use ImGui::PushFont()/PopFont() to change the current font.")
        .def_rw("sources", &ImFontAtlas::Sources, "Source/configuration data")
        .def_rw("tex_next_unique_id", &ImFontAtlas::TexNextUniqueID, "Next value to be stored in TexData->UniqueID")
        .def_rw("font_next_unique_id", &ImFontAtlas::FontNextUniqueID, "Next value to be stored in ImFont->FontID")
        .def_ro("font_loader_name", &ImFontAtlas::FontLoaderName, "Font loader name (for display e.g. in About box) == FontLoader->Name")
        .def_rw("font_loader_data", &ImFontAtlas::FontLoaderData, "Font backend opaque storage")
        .def_rw("font_loader_flags", &ImFontAtlas::FontLoaderFlags, "Shared flags (for all fonts) for font loader. THIS IS BUILD IMPLEMENTATION DEPENDENT (e.g. Per-font override is also available in ImFontConfig).")
        .def_rw("ref_count", &ImFontAtlas::RefCount, "Number of contexts using this atlas")
        .def_rw("owner_context", &ImFontAtlas::OwnerContext, "Context which own the atlas will be in charge of updating and destroying it.")
        ;


    auto pyClassImFontBaked =
        nb::class_<ImFontBaked>
            (m, "ImFontBaked", " Font runtime data for a given size\n Important: pointers to ImFontBaked are only valid for the current frame.")
        .def_rw("index_advance_x", &ImFontBaked::IndexAdvanceX, "12-16 // out // Sparse. Glyphs->AdvanceX in a directly indexable way (cache-friendly for CalcTextSize functions which only this info, and are often bottleneck in large UI).")
        .def_rw("fallback_advance_x", &ImFontBaked::FallbackAdvanceX, "4     // out // FindGlyph(FallbackChar)->AdvanceX")
        .def_rw("size", &ImFontBaked::Size, "4     // in  // Height of characters/line, set during loading (doesn't change after loading)")
        .def_rw("rasterizer_density", &ImFontBaked::RasterizerDensity, "4     // in  // Density this is baked at")
        .def_rw("index_lookup", &ImFontBaked::IndexLookup, "12-16 // out // Sparse. Index glyphs by Unicode code-point.")
        .def_rw("glyphs", &ImFontBaked::Glyphs, "12-16 // out // All glyphs.")
        .def_rw("fallback_glyph_index", &ImFontBaked::FallbackGlyphIndex, "4     // out // Index of FontFallbackChar")
        .def_rw("ascent", &ImFontBaked::Ascent, "4+4   // out // Ascent: distance from top to bottom of e.g. 'A' [0..FontSize] (unscaled)")
        .def_rw("descent", &ImFontBaked::Descent, "4+4   // out // Ascent: distance from top to bottom of e.g. 'A' [0..FontSize] (unscaled)")
        .def_rw("last_used_frame", &ImFontBaked::LastUsedFrame, "4  //     // Record of that time this was bounds")
        .def_rw("baked_id", &ImFontBaked::BakedId, "4     //     // Unique ID for this baked storage")
        .def_rw("container_font", &ImFontBaked::ContainerFont, "4-8   // in  // Parent font")
        .def_rw("font_loader_datas", &ImFontBaked::FontLoaderDatas, "4-8   //     // Font loader opaque storage (per baked font * sources): single contiguous buffer allocated by imgui, passed to loader.")
        .def(nb::init<>())
        .def("clear_output_data",
            &ImFontBaked::ClearOutputData)
        .def("find_glyph",
            &ImFontBaked::FindGlyph,
            nb::arg("c"),
            "Return U+FFFD glyph if requested glyph doesn't exists.",
            nb::rv_policy::reference)
        .def("find_glyph_no_fallback",
            &ImFontBaked::FindGlyphNoFallback,
            nb::arg("c"),
            "Return None if glyph doesn't exist",
            nb::rv_policy::reference)
        .def("get_char_advance",
            &ImFontBaked::GetCharAdvance, nb::arg("c"))
        .def("is_glyph_loaded",
            &ImFontBaked::IsGlyphLoaded, nb::arg("c"))
        ;


    auto pyEnumImFontFlags_ =
        nb::enum_<ImFontFlags_>(m, "ImFontFlags_", nb::is_arithmetic(), nb::is_flag(), " Font flags\n (in future versions as we redesign font loading API, this will become more important and better documented. for now please consider this as internal/advanced use)")
            .value("none", ImFontFlags_None, "")
            .value("no_load_error", ImFontFlags_NoLoadError, "Disable throwing an error/assert when calling AddFontXXX() with missing file/data. Calling code is expected to check AddFontXXX() return value.")
            .value("no_load_glyphs", ImFontFlags_NoLoadGlyphs, "[Internal] Disable loading new glyphs.")
            .value("lock_baked_sizes", ImFontFlags_LockBakedSizes, "[Internal] Disable loading new baked sizes, disable garbage collecting current ones. e.g. if you want to lock a font to a single size. Important: if you use this to preload given sizes, consider the possibility of multiple font density used on Retina display.");


    auto pyClassImFont =
        nb::class_<ImFont>
            (m, "ImFont", " Font runtime data and rendering\n - ImFontAtlas automatically loads a default embedded font for you if you didn't load one manually.\n - Since 1.92.X a font may be rendered as any size! Therefore a font doesn't have one specific size.\n - Use 'font->GetFontBaked(size)' to retrieve the ImFontBaked* corresponding to a given size.\n - If you used g.Font + g.FontSize (which is frequent from the ImGui layer), you can use g.FontBaked as a shortcut, as g.FontBaked == g.Font->GetFontBaked(g.FontSize).")
        .def_rw("last_baked", &ImFont::LastBaked, "4-8   // Cache last bound baked. NEVER USE DIRECTLY. Use GetFontBaked().")
        .def_rw("container_atlas", &ImFont::ContainerAtlas, "4-8   // What we have been loaded into.")
        .def_rw("flags", &ImFont::Flags, "4     // Font flags.")
        .def_rw("current_rasterizer_density", &ImFont::CurrentRasterizerDensity, "Current rasterizer density. This is a varying state of the font.")
        .def_rw("font_id", &ImFont::FontId, "Unique identifier for the font")
        .def_rw("legacy_size", &ImFont::LegacySize, "4     // in  // Font size passed to AddFont(). Use for old code calling PushFont() expecting to use that size. (use ImGui::GetFontBaked() to get font baked at current bound size).")
        .def_rw("sources", &ImFont::Sources, "16    // in  // List of sources. Pointers within ContainerAtlas->Sources[]")
        .def_rw("ellipsis_char", &ImFont::EllipsisChar, "2-4   // out // Character used for ellipsis rendering ('...').")
        .def_rw("fallback_char", &ImFont::FallbackChar, "2-4   // out // Character used if a glyph isn't found (U+FFFD, '?')")
        .def_rw("ellipsis_auto_bake", &ImFont::EllipsisAutoBake, "1     //     // Mark when the \"...\" glyph needs to be generated.")
        .def_rw("remap_pairs", &ImFont::RemapPairs, "16    //     // Remapping pairs when using AddRemapChar(), otherwise empty.")
        .def(nb::init<>(),
            "Methods")
        .def("is_glyph_in_font",
            &ImFont::IsGlyphInFont, nb::arg("c"))
        .def("is_loaded",
            &ImFont::IsLoaded, "(private API)")
        .def("get_debug_name",
            &ImFont::GetDebugName,
            "(private API)\n\n Fill ImFontConfig::Name.",
            nb::rv_policy::reference)
        .def("get_font_baked",
            &ImFont::GetFontBaked,
            nb::arg("font_size"), nb::arg("density") = -1.0f,
            "Get or create baked data for given size",
            nb::rv_policy::reference)
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("calc_word_wrap_position_python",
            &ImFont::CalcWordWrapPositionPython,
            nb::arg("size"), nb::arg("text"), nb::arg("wrap_width"),
            "Python API for CalcWordWrapPosition (will return an index in the text, not a pointer)")
        // #endif
        //
        .def("render_char",
            &ImFont::RenderChar, nb::arg("draw_list"), nb::arg("size"), nb::arg("pos"), nb::arg("col"), nb::arg("c"), nb::arg("cpu_fine_clip") = nb::none())
        .def("render_text",
            &ImFont::RenderText, nb::arg("draw_list"), nb::arg("size"), nb::arg("pos"), nb::arg("col"), nb::arg("clip_rect"), nb::arg("text_begin"), nb::arg("text_end"), nb::arg("wrap_width") = 0.0f, nb::arg("flags") = 0)
        .def("clear_output_data",
            &ImFont::ClearOutputData)
        .def("add_remap_char",
            &ImFont::AddRemapChar,
            nb::arg("from_codepoint"), nb::arg("to_codepoint"),
            "Makes 'from_codepoint' character points to 'to_codepoint' glyph.")
        .def("is_glyph_range_unused",
            &ImFont::IsGlyphRangeUnused, nb::arg("c_begin"), nb::arg("c_last"))
        ;


    auto pyEnumViewportFlags_ =
        nb::enum_<ImGuiViewportFlags_>(m, "ViewportFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags stored in ImGuiViewport::Flags, giving indications to the platform backends.")
            .value("none", ImGuiViewportFlags_None, "")
            .value("is_platform_window", ImGuiViewportFlags_IsPlatformWindow, "Represent a Platform Window")
            .value("is_platform_monitor", ImGuiViewportFlags_IsPlatformMonitor, "Represent a Platform Monitor (unused yet)")
            .value("owned_by_app", ImGuiViewportFlags_OwnedByApp, "Platform Window: Is created/managed by the user application? (rather than our backend)")
            .value("no_decoration", ImGuiViewportFlags_NoDecoration, "Platform Window: Disable platform decorations: title bar, borders, etc. (generally set all windows, but if ImGuiConfigFlags_ViewportsDecoration is set we only set this on popups/tooltips)")
            .value("no_task_bar_icon", ImGuiViewportFlags_NoTaskBarIcon, "Platform Window: Disable platform task bar icon (generally set on popups/tooltips, or all windows if ImGuiConfigFlags_ViewportsNoTaskBarIcon is set)")
            .value("no_focus_on_appearing", ImGuiViewportFlags_NoFocusOnAppearing, "Platform Window: Don't take focus when created.")
            .value("no_focus_on_click", ImGuiViewportFlags_NoFocusOnClick, "Platform Window: Don't take focus when clicked on.")
            .value("no_inputs", ImGuiViewportFlags_NoInputs, "Platform Window: Make mouse pass through so we can drag this window while peaking behind it.")
            .value("no_renderer_clear", ImGuiViewportFlags_NoRendererClear, "Platform Window: Renderer doesn't need to clear the framebuffer ahead (because we will fill it entirely).")
            .value("no_auto_merge", ImGuiViewportFlags_NoAutoMerge, "Platform Window: Avoid merging this window into another host window. This can only be set via ImGuiWindowClass viewport flags override (because we need to now ahead if we are going to create a viewport in the first place!).")
            .value("top_most", ImGuiViewportFlags_TopMost, "Platform Window: Display on top (for tooltips only).")
            .value("can_host_other_windows", ImGuiViewportFlags_CanHostOtherWindows, "Viewport can host multiple imgui windows (secondary viewports are associated to a single window). // FIXME: In practice there's still probably code making the assumption that this is always and only on the MainViewport. Will fix once we add support for \"no main viewport\".")
            .value("is_minimized", ImGuiViewportFlags_IsMinimized, "Platform Window: Window is minimized, can skip render. When minimized we tend to avoid using the viewport pos/size for clipping window or testing if they are contained in the viewport.")
            .value("is_focused", ImGuiViewportFlags_IsFocused, "Platform Window: Window is focused (last call to Platform_GetWindowFocus() returned True)");


    auto pyClassImGuiViewport =
        nb::class_<ImGuiViewport>
            (m, "Viewport", " - Currently represents the Platform Window created by the application which is hosting our Dear ImGui windows.\n - With multi-viewport enabled, we extend this concept to have multiple active viewports.\n - In the future we will extend this concept further to also represent Platform Monitor and support a \"no main platform window\" operation mode.\n - About Main Area vs Work Area:\n   - Main Area = entire viewport.\n   - Work Area = entire viewport minus sections used by main menu bars (for platform windows), or by task bar (for platform monitor).\n   - Windows are generally trying to stay within the Work Area of their host viewport.")
        .def_rw("id_", &ImGuiViewport::ID, "Unique identifier for the viewport")
        .def_rw("flags", &ImGuiViewport::Flags, "See ImGuiViewportFlags_")
        .def_rw("pos", &ImGuiViewport::Pos, "Main Area: Position of the viewport (Dear ImGui coordinates are the same as OS desktop/native coordinates)")
        .def_rw("size", &ImGuiViewport::Size, "Main Area: Size of the viewport.")
        .def_rw("framebuffer_scale", &ImGuiViewport::FramebufferScale, "Density of the viewport for Retina display (always 1,1 on Windows, may be 2,2 etc on macOS/iOS). This will affect font rasterizer density.")
        .def_rw("work_pos", &ImGuiViewport::WorkPos, "Work Area: Position of the viewport minus task bars, menus bars, status bars (>= Pos)")
        .def_rw("work_size", &ImGuiViewport::WorkSize, "Work Area: Size of the viewport minus task bars, menu bars, status bars (<= Size)")
        .def_rw("dpi_scale", &ImGuiViewport::DpiScale, "1.0 = 96 DPI = No extra scale.")
        .def_rw("parent_viewport_id", &ImGuiViewport::ParentViewportId, "(Advanced) 0: no parent. Instruct the platform backend to setup a parent/child relationship between platform windows.")
        .def_rw("draw_data", &ImGuiViewport::DrawData, "The ImDrawData corresponding to this viewport. Valid after Render() and until the next call to NewFrame().")
        .def_rw("renderer_user_data", &ImGuiViewport::RendererUserData, "None* to hold custom data structure for the renderer (e.g. swap chain, framebuffers etc.). generally set by your Renderer_CreateWindow function.")
        .def_rw("platform_user_data", &ImGuiViewport::PlatformUserData, "None* to hold custom data structure for the OS / platform (e.g. windowing info, render context). generally set by your Platform_CreateWindow function.")
        .def_rw("platform_handle", &ImGuiViewport::PlatformHandle, "None* to hold higher-level, platform window handle (e.g. HWND for Win32 backend, Uint32 WindowID for SDL, GLFWWindow* for GLFW), for FindViewportByPlatformHandle().")
        .def_rw("platform_handle_raw", &ImGuiViewport::PlatformHandleRaw, "None* to hold lower-level, platform-native window handle (always HWND on Win32 platform, unused for other platforms).")
        .def_rw("platform_window_created", &ImGuiViewport::PlatformWindowCreated, "Platform window has been created (Platform_CreateWindow() has been called). This is False during the first frame where a viewport is being created.")
        .def_rw("platform_request_move", &ImGuiViewport::PlatformRequestMove, "Platform window requested move (e.g. window was moved by the OS / host window manager, authoritative position will be OS window position)")
        .def_rw("platform_request_resize", &ImGuiViewport::PlatformRequestResize, "Platform window requested resize (e.g. window was resized by the OS / host window manager, authoritative size will be OS window size)")
        .def_rw("platform_request_close", &ImGuiViewport::PlatformRequestClose, "Platform window requested closure (e.g. window was moved by the OS / host window manager, e.g. pressing ALT-F4)")
        .def(nb::init<>())
        .def("get_center",
            &ImGuiViewport::GetCenter, "(private API)")
        .def("get_work_center",
            &ImGuiViewport::GetWorkCenter, "(private API)")
        ;


    auto pyClassImGuiPlatformIO =
        nb::class_<ImGuiPlatformIO>
            (m, "PlatformIO", "Access via ImGui::GetPlatformIO()")
        .def(nb::init<>())
        .def_rw("platform_get_clipboard_text_fn", &ImGuiPlatformIO::Platform_GetClipboardTextFn, "")
        .def_rw("platform_set_clipboard_text_fn", &ImGuiPlatformIO::Platform_SetClipboardTextFn, "")
        .def_rw("platform_clipboard_user_data", &ImGuiPlatformIO::Platform_ClipboardUserData, "[/ADAPT_IMGUI_BUNDLE]")
        .def_rw("platform_open_in_shell_fn", &ImGuiPlatformIO::Platform_OpenInShellFn, " Optional: Open link/folder/file in OS Shell\n (default to use ShellExecuteW() on Windows, system() on Linux/Mac)\n [ADAPT_IMGUI_BUNDLE]\nbool        (*Platform_OpenInShellFn)(ImGuiContext* ctx, const char* path);")
        .def_rw("platform_open_in_shell_user_data", &ImGuiPlatformIO::Platform_OpenInShellUserData, "[/ADAPT_IMGUI_BUNDLE]")
        .def_rw("platform_ime_user_data", &ImGuiPlatformIO::Platform_ImeUserData, "")
        .def_rw("platform_locale_decimal_point", &ImGuiPlatformIO::Platform_LocaleDecimalPoint, "'.'")
        .def_rw("renderer_texture_max_width", &ImGuiPlatformIO::Renderer_TextureMaxWidth, "")
        .def_rw("renderer_texture_max_height", &ImGuiPlatformIO::Renderer_TextureMaxHeight, "")
        .def_rw("renderer_render_state", &ImGuiPlatformIO::Renderer_RenderState, "Written by some backends during ImGui_ImplXXXX_RenderDrawData() call to point backend_specific ImGui_ImplXXXX_RenderState* structure.")
        .def_rw("monitors", &ImGuiPlatformIO::Monitors, " (Optional) Monitor list\n - Updated by: app/backend. Update every frame to dynamically support changing monitor or DPI configuration.\n - Used by: dear imgui to query DPI info, clamp popups/tooltips within same monitor and not have them straddle monitors.")
        .def_rw("textures", &ImGuiPlatformIO::Textures, "List of textures used by Dear ImGui (most often 1) + contents of external texture list is automatically appended into this.")
        .def_rw("viewports", &ImGuiPlatformIO::Viewports, "Main viewports, followed by all secondary viewports.")
        ;


    auto pyClassImGuiPlatformMonitor =
        nb::class_<ImGuiPlatformMonitor>
            (m, "PlatformMonitor", " (Optional) This is required when enabling multi-viewport. Represent the bounds of each connected monitor/display and their DPI.\n We use this information for multiple DPI support + clamping the position of popups and tooltips so they don't straddle multiple monitors.")
        .def_rw("main_pos", &ImGuiPlatformMonitor::MainPos, "Coordinates of the area displayed on this monitor (Min = upper left, Max = bottom right)")
        .def_rw("main_size", &ImGuiPlatformMonitor::MainSize, "Coordinates of the area displayed on this monitor (Min = upper left, Max = bottom right)")
        .def_rw("work_pos", &ImGuiPlatformMonitor::WorkPos, "Coordinates without task bars / side bars / menu bars. Used to avoid positioning popups/tooltips inside this region. If you don't have this info, please copy the value for MainPos/MainSize.")
        .def_rw("work_size", &ImGuiPlatformMonitor::WorkSize, "Coordinates without task bars / side bars / menu bars. Used to avoid positioning popups/tooltips inside this region. If you don't have this info, please copy the value for MainPos/MainSize.")
        .def_rw("dpi_scale", &ImGuiPlatformMonitor::DpiScale, "1.0 = 96 DPI")
        .def_rw("platform_handle", &ImGuiPlatformMonitor::PlatformHandle, "Backend dependant data (e.g. HMONITOR, GLFWmonitor*, SDL Display Index, NSScreen*)")
        .def(nb::init<>())
        ;


    auto pyClassImGuiPlatformImeData =
        nb::class_<ImGuiPlatformImeData>
            (m, "PlatformImeData", "(Optional) Support for IME (Input Method Editor) via the platform_io.Platform_SetImeDataFn() function. Handler is called during EndFrame().")
        .def_rw("want_visible", &ImGuiPlatformImeData::WantVisible, "A widget wants the IME to be visible.")
        .def_rw("want_text_input", &ImGuiPlatformImeData::WantTextInput, "A widget wants text input, not necessarily IME to be visible. This is automatically set to the upcoming value of io.WantTextInput.")
        .def_rw("input_pos", &ImGuiPlatformImeData::InputPos, "Position of input cursor (for IME).")
        .def_rw("input_line_height", &ImGuiPlatformImeData::InputLineHeight, "Line height (for IME).")
        .def_rw("viewport_id", &ImGuiPlatformImeData::ViewportId, "ID of platform window/viewport.")
        .def(nb::init<>())
        ;
    // #ifdef IMGUI_INCLUDE_IMGUI_USER_H
    //
    // #endif
    //
    // #endif
    ////////////////////    </generated_from:imgui.h>    ////////////////////


    ////////////////////    <generated_from:imgui_stacklayout.h>    ////////////////////
    // #ifndef IMGUI_DISABLE
    //


    m.def("begin_horizontal",
        [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
        {
            auto BeginHorizontal_adapt_mutable_param_with_default_value = [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginHorizontal(str_id, size_or_default, align);
            };

            BeginHorizontal_adapt_mutable_param_with_default_value(str_id, size, align);
        },
        nb::arg("str_id"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1.0f,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("begin_horizontal",
        [](const void * ptr_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
        {
            auto BeginHorizontal_adapt_mutable_param_with_default_value = [](const void * ptr_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginHorizontal(ptr_id, size_or_default, align);
            };

            BeginHorizontal_adapt_mutable_param_with_default_value(ptr_id, size, align);
        },
        nb::arg("ptr_id"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1.0f,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("begin_horizontal",
        [](int id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1)
        {
            auto BeginHorizontal_adapt_mutable_param_with_default_value = [](int id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginHorizontal(id, size_or_default, align);
            };

            BeginHorizontal_adapt_mutable_param_with_default_value(id, size, align);
        },
        nb::arg("id_"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("end_horizontal",
        ImGui::EndHorizontal);

    m.def("begin_vertical",
        [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
        {
            auto BeginVertical_adapt_mutable_param_with_default_value = [](const char * str_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginVertical(str_id, size_or_default, align);
            };

            BeginVertical_adapt_mutable_param_with_default_value(str_id, size, align);
        },
        nb::arg("str_id"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1.0f,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("begin_vertical",
        [](const void * ptr_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
        {
            auto BeginVertical_adapt_mutable_param_with_default_value = [](const void * ptr_id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1.0f)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginVertical(ptr_id, size_or_default, align);
            };

            BeginVertical_adapt_mutable_param_with_default_value(ptr_id, size, align);
        },
        nb::arg("ptr_id"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1.0f,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("begin_vertical",
        [](int id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1)
        {
            auto BeginVertical_adapt_mutable_param_with_default_value = [](int id, const std::optional<const ImVec2> & size = std::nullopt, float align = -1)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                ImGui::BeginVertical(id, size_or_default, align);
            };

            BeginVertical_adapt_mutable_param_with_default_value(id, size, align);
        },
        nb::arg("id_"), nb::arg("size").none() = nb::none(), nb::arg("align") = -1,
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("end_vertical",
        ImGui::EndVertical);

    m.def("spring",
        ImGui::Spring, nb::arg("weight") = 1.0f, nb::arg("spacing") = -1.0f);

    m.def("suspend_layout",
        ImGui::SuspendLayout);

    m.def("resume_layout",
        ImGui::ResumeLayout);
    // #endif
    ////////////////////    </generated_from:imgui_stacklayout.h>    ////////////////////


    ////////////////////    <generated_from:imgui_stacklayout_internal.h>    ////////////////////
    // #ifndef IMGUI_DISABLE
    //
    // #endif
    ////////////////////    </generated_from:imgui_stacklayout_internal.h>    ////////////////////


    ////////////////////    <generated_from:imgui_stdlib.h>    ////////////////////
    // #ifndef IMGUI_DISABLE
    //


    m.def("input_text",
        [](const char * label, std::string str, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
        {
            auto InputText_adapt_modifiable_immutable_to_return = [](const char * label, std::string str, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
            {
                std::string * str_adapt_modifiable = & str;

                bool r = ImGui::InputText(label, str_adapt_modifiable, flags, callback, user_data);
                return std::make_tuple(r, str);
            };

            return InputText_adapt_modifiable_immutable_to_return(label, str, flags, callback, user_data);
        },     nb::arg("label"), nb::arg("str"), nb::arg("flags") = 0, nb::arg("callback") = nb::none(), nb::arg("user_data") = nb::none());

    m.def("input_text_multiline",
        [](const char * label, std::string str, const std::optional<const ImVec2> & size = std::nullopt, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
        {
            auto InputTextMultiline_adapt_mutable_param_with_default_value = [](const char * label, std::string * str, const std::optional<const ImVec2> & size = std::nullopt, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0, 0);
                }();

                auto lambda_result = ImGui::InputTextMultiline(label, str, size_or_default, flags, callback, user_data);
                return lambda_result;
            };
            auto InputTextMultiline_adapt_modifiable_immutable_to_return = [&InputTextMultiline_adapt_mutable_param_with_default_value](const char * label, std::string str, const std::optional<const ImVec2> & size = std::nullopt, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
            {
                std::string * str_adapt_modifiable = & str;

                bool r = InputTextMultiline_adapt_mutable_param_with_default_value(label, str_adapt_modifiable, size, flags, callback, user_data);
                return std::make_tuple(r, str);
            };

            return InputTextMultiline_adapt_modifiable_immutable_to_return(label, str, size, flags, callback, user_data);
        },
        nb::arg("label"), nb::arg("str"), nb::arg("size").none() = nb::none(), nb::arg("flags") = 0, nb::arg("callback") = nb::none(), nb::arg("user_data") = nb::none(),
        "Python bindings defaults:\n    If size is None, then its default value will be: ImVec2(0, 0)");

    m.def("input_text_with_hint",
        [](const char * label, const char * hint, std::string str, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
        {
            auto InputTextWithHint_adapt_modifiable_immutable_to_return = [](const char * label, const char * hint, std::string str, ImGuiInputTextFlags flags = 0, ImGuiInputTextCallback callback = nullptr, void * user_data = nullptr) -> std::tuple<bool, std::string>
            {
                std::string * str_adapt_modifiable = & str;

                bool r = ImGui::InputTextWithHint(label, hint, str_adapt_modifiable, flags, callback, user_data);
                return std::make_tuple(r, str);
            };

            return InputTextWithHint_adapt_modifiable_immutable_to_return(label, hint, str, flags, callback, user_data);
        },     nb::arg("label"), nb::arg("hint"), nb::arg("str"), nb::arg("flags") = 0, nb::arg("callback") = nb::none(), nb::arg("user_data") = nb::none());
    // #endif
    ////////////////////    </generated_from:imgui_stdlib.h>    ////////////////////


    ////////////////////    <generated_from:imgui_pywrappers.h>    ////////////////////
    auto pyClassImGuiPayload_PyId =
        nb::class_<ImGuiPayload_PyId>
            (m, "Payload_PyId", "")
        .def("__init__", [](ImGuiPayload_PyId * self, const std::optional<const ImGuiPayloadId> & DataId = std::nullopt, std::string Type = std::string())
        {
            new (self) ImGuiPayload_PyId();  // placement new
            auto r_ctor_ = self;
            if (DataId.has_value())
                r_ctor_->DataId = DataId.value();
            else
                r_ctor_->DataId = ImGuiPayloadId();
            r_ctor_->Type = Type;
        },
        nb::arg("data_id").none() = nb::none(), nb::arg("type") = std::string()
        )
        .def_rw("data_id", &ImGuiPayload_PyId::DataId, "Stores an id that represents the payload. For example, this could be given by python `id(object)`")
        .def_rw("type", &ImGuiPayload_PyId::Type, "A string representing the type of payload. It cannot exceed 32 characters.")
        ;


    m.def("set_drag_drop_payload_py_id",
        ImGui::SetDragDropPayload_PyId, nb::arg("type"), nb::arg("data_id"), nb::arg("cond") = 0);

    m.def("accept_drag_drop_payload_py_id",
        ImGui::AcceptDragDropPayload_PyId, nb::arg("type"), nb::arg("flags") = 0);

    m.def("get_drag_drop_payload_py_id",
        ImGui::GetDragDropPayload_PyId);
    ////////////////////    </generated_from:imgui_pywrappers.h>    ////////////////////

    // </litgen_pydef> // Autogenerated code end
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    // ****************************************************************************************************************
    //
    //             MANUAL PATCHES BELOW
    //
    // ****************************************************************************************************************
    pyClassImVec2Ptr = &pyClassImVec2;
    pyClassImVec4Ptr = &pyClassImVec4;
    pyClassImColorPtr = &pyClassImColor;
    pyClassImFontAtlasPtr = &pyClassImFontAtlas;

    imgui_manual_binding(m);
}


void imgui_manual_binding(nb::module_& m)
{
    IM_ASSERT(pyClassImVec4Ptr != nullptr);
    IM_ASSERT(pyClassImVec2Ptr != nullptr);
    IM_ASSERT(pyClassImColorPtr != nullptr);
    IM_ASSERT(pyClassImFontAtlasPtr != nullptr);
    nb::class_<ImVec2>& pyClassImVec2  = *pyClassImVec2Ptr;
    nb::class_<ImVec4>& pyClassImVec4 = *pyClassImVec4Ptr;
    nb::class_<ImColor>& pyClassImColor = *pyClassImColorPtr;

    // these are internal functions that should only be called with a tuple, list or array argument
    auto cast_to_imvec2 = [](nb::handle obj) -> ImVec2
    {
        if (len(obj) != 2)
            throw std::invalid_argument("Python tuple/list/array to imgui.ImVec2: size should be 2!");
        auto floats = nb::cast<std::vector<float>>(obj);
        return ImVec2(floats[0], floats[1]);
    };
    auto cast_to_imvec4 = [](nb::handle obj) -> ImVec4
    {
        if (len(obj) != 4)
            throw std::invalid_argument("Python tuple/list/array to imgui.ImVec4: size should be 4!");
        auto floats = nb::cast<std::vector<float>>(obj);
        return ImVec4(floats[0], floats[1], floats[2], floats[3]);
    };

    auto cast_to_imcolor = [](nb::handle obj) -> ImColor
    {
        if (len(obj) != 3 && len(obj) != 4)
            throw std::invalid_argument("Python tuple/list/array to imgui.ImColor: size should be 3 or 4!");
        auto floats = nb::cast<std::vector<float>>(obj);
        if (floats.size() == 3)
            return ImColor(floats[0], floats[1], floats[2]);
        else
            return ImColor(floats[0], floats[1], floats[2], floats[3]);
    };

    // FLT_MIN & FLT_MAX
    m.attr("FLT_MIN") = (float)FLT_MIN;
    m.attr("FLT_MAX") = (float)FLT_MAX;


    //
    //  Patches to ImVec2
    //
    pyClassImVec2.def("__str__", [](const ImVec2& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImVec2(%f, %f)", self.x, self.y);
        return r;
    });
    pyClassImVec2.def("__repr__", [](const ImVec2& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImVec2(%f, %f)", self.x, self.y);
        return r;
    });
    pyClassImVec2.def("__len__", [](const ImVec2& self) -> size_t {
        return 2;
    });
    pyClassImVec2.def("__iter__", [](const ImVec2& self) {
                          return nb::make_iterator(nb::type<ImVec2>(), "iterator", &self.x, &self.x+2);
                      },
                      nb::keep_alive<0, 1>() /* Essential: keep object alive while iterator exists */
    );

    pyClassImVec2.def("__init__", [&cast_to_imvec2](ImVec2 *self, nb::tuple t) {
        new (self) ImVec2();
        *self = cast_to_imvec2(t);
    });
    pyClassImVec2.def("__init__", [&cast_to_imvec2](ImVec2 *self, nb::list l) {
        new (self) ImVec2();
        *self = cast_to_imvec2(l);
    });
//    pyClassImVec2.def("__init__", [](ImVec2 *self, nb::ndarray<> a) {
//        new (self) ImVec2();
//        *self = cast_to_imvec2(a);
//    });


    nb::implicitly_convertible<nb::tuple, ImVec2>();
    nb::implicitly_convertible<nb::list, ImVec2>();
    //nb::implicitly_convertible<nb::array, ImVec2>();

    pyClassImVec2.def("__init__", [](ImVec2 *self, ImVec2 imv) {
        new (self) ImVec2();
        *self = ImVec2(imv.x, imv.y);
    });


    //
    // Add math operators to ImVec2
    //
    pyClassImVec2.def("__eq__", [](const ImVec2& self, const ImVec2& other) -> bool {
        return self.x == other.x && self.y == other.y;
    });
    pyClassImVec2.def("__ne__", [](const ImVec2& self, const ImVec2& other) -> bool {
        return self.x != other.x || self.y != other.y;
    });
    pyClassImVec2.def("__add__", [](const ImVec2& self, const ImVec2& other) -> ImVec2 {
        return ImVec2(self.x + other.x, self.y + other.y);
    });
    pyClassImVec2.def("__sub__", [](const ImVec2& self, const ImVec2& other) -> ImVec2 {
        return ImVec2(self.x - other.x, self.y - other.y);
    });
    pyClassImVec2.def("__mul__", [](const ImVec2& self, float other) -> ImVec2 {
        return ImVec2(self.x * other, self.y * other);
    });
    pyClassImVec2.def("__truediv__", [](const ImVec2& self, float other) -> ImVec2 {
        return ImVec2(self.x / other, self.y / other);
    });
    pyClassImVec2.def("__mul__", [](const ImVec2& self, const ImVec2& other) -> ImVec2 {
        return ImVec2(self.x * other.x, self.y * other.y);
    });
    pyClassImVec2.def("__truediv__", [](const ImVec2& self, const ImVec2& other) -> ImVec2 {
        return ImVec2(self.x / other.x, self.y / other.y);
    });
    // operator *=, +=, -=, /=
    pyClassImVec2.def("__imul__", [](ImVec2& self, float other) -> ImVec2& {
        self.x *= other;
        self.y *= other;
        return self;
    });
    pyClassImVec2.def("__itruediv__", [](ImVec2& self, float other) -> ImVec2& {
        self.x /= other;
        self.y /= other;
        return self;
    });
    pyClassImVec2.def("__imul__", [](ImVec2& self, const ImVec2& other) -> ImVec2& {
        self.x *= other.x;
        self.y *= other.y;
        return self;
    });
    pyClassImVec2.def("__itruediv__", [](ImVec2& self, const ImVec2& other) -> ImVec2& {
        self.x /= other.x;
        self.y /= other.y;
        return self;
    });
    pyClassImVec2.def("__iadd__", [](ImVec2& self, const ImVec2& other) -> ImVec2& {
        self.x += other.x;
        self.y += other.y;
        return self;
    });
    pyClassImVec2.def("__isub__", [](ImVec2& self, const ImVec2& other) -> ImVec2& {
        self.x -= other.x;
        self.y -= other.y;
        return self;
    });
    // unary operators
    pyClassImVec2.def("__neg__", [](const ImVec2& self) -> ImVec2 {
        return ImVec2(-self.x, -self.y);
    });

    // Pickle support
    pyClassImVec2.def("__getstate__", [](const ImVec2 &v) {
        return nb::make_tuple(v.x, v.y);
    });
    pyClassImVec2.def("__setstate__", [](ImVec2 &self, nb::tuple t) {
        new(&self) ImVec2();
        if (t.size() != 2)
            throw std::runtime_error("ImVec2 unpickling failed");
        ImVec2 r(nb::cast<float>(t[0]), nb::cast<float>(t[1]));
        self = r;
    });

    //
    //  Patches to ImVec4
    //
    pyClassImVec4.def("__str__", [](const ImVec4& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImVec4(%f, %f, %f, %f)", self.x, self.y, self.z, self.w);
        return r;
    });
    pyClassImVec4.def("__repr__", [](const ImVec4& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImVec4(%f, %f, %f, %f)", self.x, self.y, self.z, self.w);
        return r;
    });
    pyClassImVec4.def("__getitem__", [](const ImVec4& self, size_t idx) -> float {
        if (idx >= 4)
            throw nb::index_error();
        switch (idx)
        {
            case 0:
                return self.x;
            case 1:
                return self.y;
            case 2:
                return self.z;
            case 3:
                return self.w;
            default:
                // unreachable, but makes clang happy
                // (warning: non-void lambda does not return a value in all control paths [-Wreturn-type])
                return 0.f;
        }
    });
    pyClassImVec4.def("__len__", [](const ImVec4& self) -> size_t {
        return 4;
    });
    pyClassImVec4.def("__iter__", [](const ImVec4& self) {
                          return nb::make_iterator(nb::type<ImVec4>(), "iterator", &self.x, &self.x+4);
                      },
                      nb::keep_alive<0, 1>() /* Essential: keep object alive while iterator exists */
    );
    pyClassImVec4.def("__init__", [&cast_to_imvec4](ImVec4 *self, nb::tuple t) {
        new (self) ImVec4();
        *self = cast_to_imvec4(t);
    });
    pyClassImVec4.def("__init__", [&cast_to_imvec4](ImVec4 *self, nb::list l) {
        new (self) ImVec4();
        *self = cast_to_imvec4(l);
    });
//    pyClassImVec4.def("__init__", [](ImVec4 *self, nb::ndarray<> a) {
//        new (self) ImVec4();
//        *self = cast_to_imvec4(a);
//    });

    nb::implicitly_convertible<nb::tuple, ImVec4>();
    nb::implicitly_convertible<nb::list, ImVec4>();
    //nb::implicitly_convertible<nb::array, ImVec4>();

    pyClassImVec4.def("__init__", [](ImVec4 *self, ImVec4 imv) {
        new (self) ImVec4();
        *self = ImVec4(imv.x, imv.y, imv.z, imv.w);
    }, nb::arg("xyzw"));

    //
    // Add math operators to ImVec4
    //
    pyClassImVec4.def("__eq__", [](const ImVec4& self, const ImVec4& other) -> bool {
        return self.x == other.x && self.y == other.y && self.z == other.z && self.w == other.w;
    });
    pyClassImVec4.def("__ne__", [](const ImVec4& self, const ImVec4& other) -> bool {
        return self.x != other.x || self.y != other.y || self.z != other.z || self.w != other.w;
    });
    pyClassImVec4.def("__add__", [](const ImVec4& self, const ImVec4& other) -> ImVec4 {
        return ImVec4(self.x + other.x, self.y + other.y, self.z + other.z, self.w + other.w);
    });
    pyClassImVec4.def("__sub__", [](const ImVec4& self, const ImVec4& other) -> ImVec4 {
        return ImVec4(self.x - other.x, self.y - other.y, self.z - other.z, self.w - other.w);
    });
    pyClassImVec4.def("__mul__", [](const ImVec4& self, float other) -> ImVec4 {
        return ImVec4(self.x * other, self.y * other, self.z * other, self.w * other);
    });
    pyClassImVec4.def("__truediv__", [](const ImVec4& self, float other) -> ImVec4 {
        return ImVec4(self.x / other, self.y / other, self.z / other, self.w / other);
    });
    pyClassImVec4.def("__mul__", [](const ImVec4& self, const ImVec4& other) -> ImVec4 {
        return ImVec4(self.x * other.x, self.y * other.y, self.z * other.z, self.w * other.w);
    });
    pyClassImVec4.def("__truediv__", [](const ImVec4& self, const ImVec4& other) -> ImVec4 {
        return ImVec4(self.x / other.x, self.y / other.y, self.z / other.z, self.w / other.w);
    });
    // operator *=, +=, -=, /=
    pyClassImVec4.def("__imul__", [](ImVec4& self, float other) -> ImVec4& {
        self.x *= other;
        self.y *= other;
        self.z *= other;
        self.w *= other;
        return self;
    });
    pyClassImVec4.def("__itruediv__", [](ImVec4& self, float other) -> ImVec4& {
        self.x /= other;
        self.y /= other;
        self.z /= other;
        self.w /= other;
        return self;
    });
    pyClassImVec4.def("__imul__", [](ImVec4& self, const ImVec4& other) -> ImVec4& {
        self.x *= other.x;
        self.y *= other.y;
        self.z *= other.z;
        self.w *= other.w;
        return self;
    });
    pyClassImVec4.def("__itruediv__", [](ImVec4& self, const ImVec4& other) -> ImVec4& {
        self.x /= other.x;
        self.y /= other.y;
        self.z /= other.z;
        self.w /= other.w;
        return self;
    });
    pyClassImVec4.def("__iadd__", [](ImVec4& self, const ImVec4& other) -> ImVec4& {
        self.x += other.x;
        self.y += other.y;
        self.z += other.z;
        self.w += other.w;
        return self;
    });
    pyClassImVec4.def("__isub__", [](ImVec4& self, const ImVec4& other) -> ImVec4& {
        self.x -= other.x;
        self.y -= other.y;
        self.z -= other.z;
        self.w -= other.w;
        return self;
    });
    // unary operators
    pyClassImVec4.def("__neg__", [](const ImVec4& self) -> ImVec4 {
        return ImVec4(-self.x, -self.y, -self.z, -self.w);
    });

    // Pickle support
    pyClassImVec4.def("__getstate__", [](const ImVec4 &v) {
        return nb::make_tuple(v.x, v.y, v.z, v.w);
    });
    pyClassImVec4.def("__setstate__", [](ImVec4 &self, nb::tuple t) {
        new(&self) ImVec2();
        if (t.size() != 4)
            throw std::runtime_error("ImVec4 unpickling failed");
        ImVec4 r(nb::cast<float>(t[0]), nb::cast<float>(t[1]), nb::cast<float>(t[2]), nb::cast<float>(t[3]));
        self = r;
    });


    //
    //  Patches to ImColor
    //
    pyClassImColor.def("__str__", [](const ImColor& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImColor(%f, %f, %f, %f)", self.Value.x, self.Value.y, self.Value.z, self.Value.w);
        return r;
    });
    pyClassImColor.def("__repr__", [](const ImColor& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImColor(%f, %f, %f, %f)", self.Value.x, self.Value.y, self.Value.z, self.Value.w);
        return r;
    });
    pyClassImColor.def("__getitem__", [](const ImColor& self, size_t idx) -> float {
        if (idx >= 4)
            throw nb::index_error();
        switch (idx)
        {
            case 0:
                return self.Value.x;
            case 1:
                return self.Value.y;
            case 2:
                return self.Value.z;
            case 3:
                return self.Value.w;
            default:
                // unreachable, but makes clang happy
                // (warning: non-void lambda does not return a value in all control paths [-Wreturn-type])
                return 0.f;
        }
    });
    pyClassImColor.def("__len__", [](const ImColor& self) -> size_t {
        return 4;
    });
    pyClassImColor.def("__iter__", [](const ImColor& self) {
            return nb::make_iterator(nb::type<ImColor>(), "iterator", &self.Value.x, &self.Value.x + 4);
        },
        nb::keep_alive<0, 1>() /* Essential: keep object alive while iterator exists */
    );

    pyClassImColor.def("__init__", [&cast_to_imcolor](ImColor *self, nb::tuple t) {
        new (self) ImColor();
        *self = cast_to_imcolor(t);
    });
    pyClassImColor.def("__init__", [&cast_to_imcolor](ImColor *self, nb::list l) {
        new (self) ImColor();
        *self = cast_to_imcolor(l);
    });
//    pyClassImColor.def("__init__", [](ImColor *self, nb::ndarray<> a) {
//        new (self) ImColor();
//        *self = cast_to_imcolor(a);
//    });


    nb::implicitly_convertible<nb::tuple, ImColor>();
    nb::implicitly_convertible<nb::list, ImColor>();
    //nb::implicitly_convertible<nb::array, ImColor>();
    nb::implicitly_convertible<ImVec4, ImColor>();
    nb::implicitly_convertible<ImColor, ImVec4>();
    // below two are not possible, even if sorely needed for our API, with current pybind11
    //nb::implicitly_convertible<ImU32, ImColor>();
    //nb::implicitly_convertible<ImColor, ImU32>();

    pyClassImColor.def("__init__", [](ImColor *self, ImColor imc) {
        new (self) ImColor();
        *self = ImColor(imc.Value.x, imc.Value.y, imc.Value.z, imc.Value.w);
    });


    // Pickle support
    pyClassImColor.def("__getstate__", [](const ImColor &p) {
        return nb::make_tuple(p.Value.x, p.Value.y, p.Value.z, p.Value.w);
    });
    pyClassImColor.def("__setstate__", [](ImColor &self, nb::tuple t) {
        new(&self) ImColor();
        if (t.size() != 4)
            throw std::runtime_error("ImVec4 unpickling failed");
        ImVec4 r(nb::cast<float>(t[0]), nb::cast<float>(t[1]), nb::cast<float>(t[2]), nb::cast<float>(t[3]));
        self = ImColor(r);
    });


    pyClassImColor.def("__eq__", [](const ImColor& self, const ImColor& other) -> bool {
        return self.Value.x == other.Value.x && self.Value.y == other.Value.y && self.Value.z == other.Value.z && self.Value.w == other.Value.w;
    });
    pyClassImColor.def("__ne__", [](const ImColor& self, const ImColor& other) -> bool {
        return self.Value.x != other.Value.x || self.Value.y != other.Value.y || self.Value.z != other.Value.z || self.Value.w != other.Value.w;
    });


    //
    //  Other patches
    //

    // VERTEX_SIZE, VERTEX_BUFFER_POS_OFFSET, VERTEX_BUFFER_UV_OFFSET, etc.
    // Utilities to facilitate rendering in python backends: they provide buffer offsets info
    //
    //    def _py_vertex_buffer_vertex_size():
    //    return sizeof(cimgui.ImDrawVert)
    m.attr("VERTEX_SIZE") = sizeof(ImDrawVert);
    //    def _py_vertex_buffer_vertex_pos_offset():
    //    return <uintptr_t><size_t>&(<cimgui.ImDrawVert*>NULL).pos
    ImDrawVert* nullDrawVert = NULL;
    m.attr("VERTEX_BUFFER_POS_OFFSET") = (uintptr_t)(size_t) (& (nullDrawVert->pos));
    //    def _py_vertex_buffer_vertex_uv_offset():
    //    return <uintptr_t><size_t>&(<cimgui.ImDrawVert*>NULL).uv
    m.attr("VERTEX_BUFFER_UV_OFFSET") = (uintptr_t)(size_t) (& (nullDrawVert->uv));
    //    def _py_vertex_buffer_vertex_col_offset():
    //    return <uintptr_t><size_t>&(<cimgui.ImDrawVert*>NULL).col
    m.attr("VERTEX_BUFFER_COL_OFFSET") = (uintptr_t)(size_t) (& (nullDrawVert->col));
    //    def _py_index_buffer_index_size():
    //    return sizeof(cimgui.ImDrawIdx)
    m.attr("INDEX_SIZE") = sizeof(ImDrawIdx);


    m.def("IM_COL32", [](int r, int g, int b, int a){
        return IM_COL32(r, g, b, a);
    });

    //
    // API for imgui_demo
    //
    // Forward declaration of imgui_demo.cpp API
    void SetImGuiDemoWindowPos(ImVec2 pos, ImVec2 size, ImGuiCond cond);
    void SetImGuiDemoCodeWindowPos(ImVec2 pos, ImVec2 size, ImGuiCond cond);
    void ImGuiDemoSetShowPythonCode(bool showPythonCode);
    extern bool GImGuiDemoMarker_IsActive;

    m.def("set_imgui_demo_window_pos", SetImGuiDemoWindowPos);
    m.def("set_imgui_demo_code_window_pos", SetImGuiDemoCodeWindowPos);
    m.def("set_imgui_demo_marker_is_active", [](bool b) {
        GImGuiDemoMarker_IsActive = b;
    });
    m.def("get_imgui_demo_marker_is_active", []() {
        return GImGuiDemoMarker_IsActive;
    });
    m.def("set_imgui_demo_show_python_code", ImGuiDemoSetShowPythonCode);
}
