// Part of ImGui Bundle - MIT License - Copyright (c) 2022-2024 Pascal Thomet - https://github.com/pthom/imgui_bundle
#ifdef HELLOIMGUI_WITH_TEST_ENGINE
#include <nanobind/nanobind.h>
#include <nanobind/trampoline.h>
#include <nanobind/stl/array.h>
#include <nanobind/stl/string.h>
#include <nanobind/stl/vector.h>
#include <nanobind/stl/optional.h>
#include <nanobind/stl/function.h>
#include <nanobind/stl/shared_ptr.h>
#include <nanobind/stl/unique_ptr.h>
#include <nanobind/stl/map.h>
#include <nanobind/stl/tuple.h>
#include <nanobind/make_iterator.h>
#include <nanobind/ndarray.h>

#include "imgui.h"
#include "imgui_internal.h"
#include "imgui_test_engine/imgui_te_context.h"
#include "imgui_test_engine/imgui_te_engine.h"
#include "imgui_test_engine/imgui_te_ui.h"
#include "imgui_test_engine/imgui_te_internal.h"
#include "imgui_test_engine/imgui_te_exporters.h"
#include "imgui_test_engine/imgui_te_perftool.h"


namespace nb = nanobind;



// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// <litgen_glue_code>  // Autogenerated code below! Do not edit!

// </litgen_glue_code> // Autogenerated code end
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


void py_init_module_imgui_test_engine(nb::module_& m)
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // <litgen_pydef> // Autogenerated code below! Do not edit!
    ////////////////////    <generated_from:imgui_te_exporters.h>    ////////////////////
    auto pyEnumTestEngineExportFormat =
        nb::enum_<ImGuiTestEngineExportFormat>(m, "TestEngineExportFormat", nb::is_arithmetic(), nb::is_flag(), "")
            .value("none", ImGuiTestEngineExportFormat_None, "")
            .value("j_unit_xml", ImGuiTestEngineExportFormat_JUnitXml, "");


    m.def("print_result_summary",
        ImGuiTestEngine_PrintResultSummary,
        nb::arg("engine"),
        "(private API)");

    m.def("export",
        ImGuiTestEngine_Export,
        nb::arg("engine"),
        "(private API)");

    m.def("export_ex",
        ImGuiTestEngine_ExportEx,
        nb::arg("engine"), nb::arg("format"), nb::arg("filename"),
        "(private API)");
    ////////////////////    </generated_from:imgui_te_exporters.h>    ////////////////////


    ////////////////////    <generated_from:imgui_te_engine.h>    ////////////////////
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //
    // #endif
    //


    auto pyEnumTestActiveFunc =
        nb::enum_<ImGuiTestActiveFunc>(m, "TestActiveFunc", nb::is_arithmetic(), nb::is_flag(), "Stored in ImGuiTestContext: where we are currently running GuiFunc or TestFunc")
            .value("none", ImGuiTestActiveFunc_None, "")
            .value("gui_func", ImGuiTestActiveFunc_GuiFunc, "")
            .value("test_func", ImGuiTestActiveFunc_TestFunc, "");


    auto pyEnumTestRunSpeed =
        nb::enum_<ImGuiTestRunSpeed>(m, "TestRunSpeed", nb::is_arithmetic(), nb::is_flag(), "")
            .value("fast", ImGuiTestRunSpeed_Fast, "Run tests as fast as possible (teleport mouse, skip delays, etc.)")
            .value("normal", ImGuiTestRunSpeed_Normal, "Run tests at human watchable speed (for debugging)")
            .value("cinematic", ImGuiTestRunSpeed_Cinematic, "Run tests with pauses between actions (for e.g. tutorials)")
            .value("count", ImGuiTestRunSpeed_COUNT, "");


    auto pyEnumTestVerboseLevel =
        nb::enum_<ImGuiTestVerboseLevel>(m, "TestVerboseLevel", nb::is_arithmetic(), nb::is_flag(), "")
            .value("silent", ImGuiTestVerboseLevel_Silent, "-v0")
            .value("error", ImGuiTestVerboseLevel_Error, "-v1")
            .value("warning", ImGuiTestVerboseLevel_Warning, "-v2")
            .value("info", ImGuiTestVerboseLevel_Info, "-v3")
            .value("debug", ImGuiTestVerboseLevel_Debug, "-v4")
            .value("trace", ImGuiTestVerboseLevel_Trace, "")
            .value("count", ImGuiTestVerboseLevel_COUNT, "");


    auto pyEnumTestStatus =
        nb::enum_<ImGuiTestStatus>(m, "TestStatus", nb::is_arithmetic(), nb::is_flag(), "Test status (stored in ImGuiTest)")
            .value("unknown", ImGuiTestStatus_Unknown, "")
            .value("success", ImGuiTestStatus_Success, "")
            .value("queued", ImGuiTestStatus_Queued, "")
            .value("running", ImGuiTestStatus_Running, "")
            .value("error", ImGuiTestStatus_Error, "")
            .value("suspended", ImGuiTestStatus_Suspended, "")
            .value("count", ImGuiTestStatus_COUNT, "");


    auto pyEnumTestGroup =
        nb::enum_<ImGuiTestGroup>(m, "TestGroup", nb::is_arithmetic(), nb::is_flag(), "Test group: this is mostly used to categorize tests in our testing UI. (Stored in ImGuiTest)")
            .value("unknown", ImGuiTestGroup_Unknown, "")
            .value("tests", ImGuiTestGroup_Tests, "")
            .value("perfs", ImGuiTestGroup_Perfs, "")
            .value("count", ImGuiTestGroup_COUNT, "");


    auto pyEnumTestFlags_ =
        nb::enum_<ImGuiTestFlags_>(m, "TestFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags (stored in ImGuiTest)")
            .value("none", ImGuiTestFlags_None, "")
            .value("no_gui_warm_up", ImGuiTestFlags_NoGuiWarmUp, "Disable running the GUI func for 2 frames before starting test code. For tests which absolutely need to start before GuiFunc.")
            .value("no_auto_finish", ImGuiTestFlags_NoAutoFinish, "By default, tests with no TestFunc (only a GuiFunc) will end after warmup. Setting this require test to call ctx->Finish().")
            .value("no_recovery_warnings", ImGuiTestFlags_NoRecoveryWarnings, "Error/recovery warnings (missing End/Pop calls etc.) will be displayed as normal debug entries, for tests which may rely on those.");


    auto pyEnumTestCheckFlags_ =
        nb::enum_<ImGuiTestCheckFlags_>(m, "TestCheckFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for IM_CHECK* macros.")
            .value("none", ImGuiTestCheckFlags_None, "")
            .value("silent_success", ImGuiTestCheckFlags_SilentSuccess, "");


    auto pyEnumTestLogFlags_ =
        nb::enum_<ImGuiTestLogFlags_>(m, "TestLogFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ImGuiTestContext::Log* functions.")
            .value("none", ImGuiTestLogFlags_None, "")
            .value("no_header", ImGuiTestLogFlags_NoHeader, "Do not display frame count and depth padding");


    auto pyEnumTestRunFlags_ =
        nb::enum_<ImGuiTestRunFlags_>(m, "TestRunFlags_", nb::is_arithmetic(), nb::is_flag(), "")
            .value("none", ImGuiTestRunFlags_None, "")
            .value("gui_func_disable", ImGuiTestRunFlags_GuiFuncDisable, "Used internally to temporarily disable the GUI func (at the end of a test, etc)")
            .value("gui_func_only", ImGuiTestRunFlags_GuiFuncOnly, "Set when user selects \"Run GUI func\"")
            .value("no_success_msg", ImGuiTestRunFlags_NoSuccessMsg, "")
            .value("enable_raw_inputs", ImGuiTestRunFlags_EnableRawInputs, "Disable input submission to let test submission raw input event (in order to test e.g. IO queue)")
            .value("run_from_gui", ImGuiTestRunFlags_RunFromGui, "Test ran manually from GUI, will disable watchdog.")
            .value("run_from_command_line", ImGuiTestRunFlags_RunFromCommandLine, "Test queued from command-line.")
            .value("no_error", ImGuiTestRunFlags_NoError, "")
            .value("share_vars", ImGuiTestRunFlags_ShareVars, "Share generic vars and custom vars between child and parent tests (custom vars need to be same type)")
            .value("share_test_context", ImGuiTestRunFlags_ShareTestContext, "Share ImGuiTestContext instead of creating a new one (unsure what purpose this may be useful for yet)");


    auto pyClassImGuiTestEngineResultSummary =
        nb::class_<ImGuiTestEngineResultSummary>
            (m, "TestEngineResultSummary", "")
        .def("__init__", [](ImGuiTestEngineResultSummary * self, int CountTested = 0, int CountSuccess = 0, int CountInQueue = 0)
        {
            new (self) ImGuiTestEngineResultSummary();  // placement new
            auto r_ctor_ = self;
            r_ctor_->CountTested = CountTested;
            r_ctor_->CountSuccess = CountSuccess;
            r_ctor_->CountInQueue = CountInQueue;
        },
        nb::arg("count_tested") = 0, nb::arg("count_success") = 0, nb::arg("count_in_queue") = 0
        )
        .def_rw("count_tested", &ImGuiTestEngineResultSummary::CountTested, "Number of tests executed")
        .def_rw("count_success", &ImGuiTestEngineResultSummary::CountSuccess, "Number of tests succeeded")
        .def_rw("count_in_queue", &ImGuiTestEngineResultSummary::CountInQueue, "Number of tests remaining in queue (e.g. aborted, crashed)")
        ;


    m.def("hook_item_add",
        ImGuiTestEngineHook_ItemAdd,
        nb::arg("ui_ctx"), nb::arg("id_"), nb::arg("bb"), nb::arg("item_data"),
        " Hooks for core imgui/ library (generally called via macros)\n(private API)");

    m.def("hook_log",
        [](ImGuiContext * ui_ctx, const char * fmt)
        {
            auto ImGuiTestEngineHook_Log_adapt_variadic_format = [](ImGuiContext * ui_ctx, const char * fmt)
            {
                ImGuiTestEngineHook_Log(ui_ctx, "%s", fmt);
            };

            ImGuiTestEngineHook_Log_adapt_variadic_format(ui_ctx, fmt);
        },
        nb::arg("ui_ctx"), nb::arg("fmt"),
        "(private API)");

    m.def("find_item_debug_label",
        ImGuiTestEngine_FindItemDebugLabel,
        nb::arg("ui_ctx"), nb::arg("id_"),
        "(private API)",
        nb::rv_policy::reference);

    m.def("check",
        ImGuiTestEngine_Check, nb::arg("file"), nb::arg("func"), nb::arg("line"), nb::arg("flags"), nb::arg("result"), nb::arg("expr"));

    m.def("check_str_op",
        [](const char * file, const char * func, int line, ImGuiTestCheckFlags flags, const char * op, const char * lhs_var, const char * lhs_value, const char * rhs_var, const char * rhs_value, bool out_result) -> std::tuple<bool, bool>
        {
            auto ImGuiTestEngine_CheckStrOp_adapt_modifiable_immutable_to_return = [](const char * file, const char * func, int line, ImGuiTestCheckFlags flags, const char * op, const char * lhs_var, const char * lhs_value, const char * rhs_var, const char * rhs_value, bool out_result) -> std::tuple<bool, bool>
            {
                bool * out_result_adapt_modifiable = & out_result;

                bool r = ImGuiTestEngine_CheckStrOp(file, func, line, flags, op, lhs_var, lhs_value, rhs_var, rhs_value, out_result_adapt_modifiable);
                return std::make_tuple(r, out_result);
            };

            return ImGuiTestEngine_CheckStrOp_adapt_modifiable_immutable_to_return(file, func, line, flags, op, lhs_var, lhs_value, rhs_var, rhs_value, out_result);
        },     nb::arg("file"), nb::arg("func"), nb::arg("line"), nb::arg("flags"), nb::arg("op"), nb::arg("lhs_var"), nb::arg("lhs_value"), nb::arg("rhs_var"), nb::arg("rhs_value"), nb::arg("out_result"));

    m.def("error",
        [](const char * file, const char * func, int line, ImGuiTestCheckFlags flags, const char * fmt) -> bool
        {
            auto ImGuiTestEngine_Error_adapt_variadic_format = [](const char * file, const char * func, int line, ImGuiTestCheckFlags flags, const char * fmt) -> bool
            {
                auto lambda_result = ImGuiTestEngine_Error(file, func, line, flags, "%s", fmt);
                return lambda_result;
            };

            return ImGuiTestEngine_Error_adapt_variadic_format(file, func, line, flags, fmt);
        },     nb::arg("file"), nb::arg("func"), nb::arg("line"), nb::arg("flags"), nb::arg("fmt"));

    m.def("assert_log",
        ImGuiTestEngine_AssertLog, nb::arg("expr"), nb::arg("file"), nb::arg("function"), nb::arg("line"));

    m.def("get_temp_string_builder",
        ImGuiTestEngine_GetTempStringBuilder, nb::rv_policy::reference);

    m.def("create_context",
        ImGuiTestEngine_CreateContext,
        " ImGuiTestEngine_CreateContext: Create test engine\n Note for python bindings users:\n     Integrating ImGui TestEngine directly from python, and without using HelloImGui and ImmApp is very difficult:\n         ImGui Test Engine uses two different threads (one for the main gui, and one for the scenario runner).\n         Your python code will be called from two separate threads, and this breaks the GIL!\n         HelloImGui and ImmApp handle this well by transferring the GIL between threads (from C++)\n     For gory details, see https://github.com/pthom/imgui_test_engine/blob/imgui_bundle/imgui_test_engine/imgui_te_python_gil.jpg",
        nb::rv_policy::reference);

    m.def("destroy_context",
        ImGuiTestEngine_DestroyContext,
        nb::arg("engine"),
        "Destroy test engine. Call after ImGui::DestroyContext() so test engine specific ini data gets saved.");

    m.def("start",
        ImGuiTestEngine_Start,
        nb::arg("engine"), nb::arg("ui_ctx"),
        "Bind to a dear imgui context. Start coroutine.");

    m.def("stop",
        ImGuiTestEngine_Stop,
        nb::arg("engine"),
        "Stop coroutine and export if any. (Unbind will lazily happen on context shutdown)");

    m.def("post_swap",
        ImGuiTestEngine_PostSwap,
        nb::arg("engine"),
        "Call every frame after framebuffer swap, will process screen capture and call test_io.ScreenCaptureFunc()");

    m.def("get_io",
        ImGuiTestEngine_GetIO,
        nb::arg("engine"),
        nb::rv_policy::reference);

    m.def("register_test",
        [](ImGuiTestEngine * engine, const char * category, const char * name, std::optional<std::string> src_file = std::nullopt, int src_line = 0) -> ImGuiTest *
        {
            auto ImGuiTestEngine_RegisterTest_adapt_const_char_pointer_with_default_null = [](ImGuiTestEngine * engine, const char * category, const char * name, std::optional<std::string> src_file = std::nullopt, int src_line = 0) -> ImGuiTest *
            {
                const char * src_file_adapt_default_null = nullptr;
                if (src_file.has_value())
                    src_file_adapt_default_null = src_file.value().c_str();

                auto lambda_result = ImGuiTestEngine_RegisterTest(engine, category, name, src_file_adapt_default_null, src_line);
                return lambda_result;
            };

            return ImGuiTestEngine_RegisterTest_adapt_const_char_pointer_with_default_null(engine, category, name, src_file, src_line);
        },
        nb::arg("engine"), nb::arg("category"), nb::arg("name"), nb::arg("src_file").none() = nb::none(), nb::arg("src_line") = 0,
        "Prefer calling IM_REGISTER_TEST()",
        nb::rv_policy::reference);

    m.def("unregister_test",
        ImGuiTestEngine_UnregisterTest, nb::arg("engine"), nb::arg("test"));

    m.def("unregister_all_tests",
        ImGuiTestEngine_UnregisterAllTests, nb::arg("engine"));

    m.def("queue_test",
        ImGuiTestEngine_QueueTest, nb::arg("engine"), nb::arg("test"), nb::arg("run_flags") = 0);

    m.def("queue_tests",
        [](ImGuiTestEngine * engine, ImGuiTestGroup group, std::optional<std::string> filter = std::nullopt, ImGuiTestRunFlags run_flags = 0)
        {
            auto ImGuiTestEngine_QueueTests_adapt_const_char_pointer_with_default_null = [](ImGuiTestEngine * engine, ImGuiTestGroup group, std::optional<std::string> filter = std::nullopt, ImGuiTestRunFlags run_flags = 0)
            {
                const char * filter_adapt_default_null = nullptr;
                if (filter.has_value())
                    filter_adapt_default_null = filter.value().c_str();

                ImGuiTestEngine_QueueTests(engine, group, filter_adapt_default_null, run_flags);
            };

            ImGuiTestEngine_QueueTests_adapt_const_char_pointer_with_default_null(engine, group, filter, run_flags);
        },     nb::arg("engine"), nb::arg("group"), nb::arg("filter").none() = nb::none(), nb::arg("run_flags") = 0);

    m.def("try_abort_engine",
        ImGuiTestEngine_TryAbortEngine, nb::arg("engine"));

    m.def("abort_current_test",
        ImGuiTestEngine_AbortCurrentTest, nb::arg("engine"));

    m.def("find_test_by_name",
        ImGuiTestEngine_FindTestByName,
        nb::arg("engine"), nb::arg("category"), nb::arg("name"),
        nb::rv_policy::reference);

    m.def("is_test_queue_empty",
        ImGuiTestEngine_IsTestQueueEmpty, nb::arg("engine"));

    m.def("is_using_simulated_inputs",
        ImGuiTestEngine_IsUsingSimulatedInputs, nb::arg("engine"));

    m.def("get_result_summary",
        ImGuiTestEngine_GetResultSummary, nb::arg("engine"), nb::arg("out_results"));

    m.def("install_default_crash_handler",
        ImGuiTestEngine_InstallDefaultCrashHandler, "Install default crash handler (if you don't have one)");

    m.def("crash_handler",
        ImGuiTestEngine_CrashHandler, "Default crash handler, should be called from a custom crash handler if such exists");


    auto pyClassImGuiTestEngineIO =
        nb::class_<ImGuiTestEngineIO>
            (m, "TestEngineIO", "")
        .def("__init__", [](ImGuiTestEngineIO * self, bool ConfigSavedSettings = true, ImGuiTestRunSpeed ConfigRunSpeed = ImGuiTestRunSpeed_Fast, bool ConfigStopOnError = false, bool ConfigBreakOnError = false, bool ConfigKeepGuiFunc = false, ImGuiTestVerboseLevel ConfigVerboseLevel = ImGuiTestVerboseLevel_Warning, ImGuiTestVerboseLevel ConfigVerboseLevelOnError = ImGuiTestVerboseLevel_Info, bool ConfigLogToTTY = false, bool ConfigLogToDebugger = false, bool ConfigRestoreFocusAfterTests = true, bool ConfigCaptureEnabled = true, bool ConfigCaptureOnError = false, bool ConfigNoThrottle = false, bool ConfigMouseDrawCursor = true, float ConfigFixedDeltaTime = 0.0f, int PerfStressAmount = 1, float MouseSpeed = 600.0f, float MouseWobble = 0.25f, float ScrollSpeed = 1400.0f, float TypingSpeed = 20.0f, float ActionDelayShort = 0.15f, float ActionDelayStandard = 0.40f, float ConfigWatchdogWarning = 30.0f, float ConfigWatchdogKillTest = 60.0f, float ConfigWatchdogKillApp = FLT_MAX, ImGuiTestEngineExportFormat ExportResultsFormat = (ImGuiTestEngineExportFormat)0, bool CheckDrawDataIntegrity = false, bool IsRunningTests = false, bool IsRequestingMaxAppSpeed = false, bool IsCapturing = false)
        {
            new (self) ImGuiTestEngineIO();  // placement new
            auto r_ctor_ = self;
            r_ctor_->ConfigSavedSettings = ConfigSavedSettings;
            r_ctor_->ConfigRunSpeed = ConfigRunSpeed;
            r_ctor_->ConfigStopOnError = ConfigStopOnError;
            r_ctor_->ConfigBreakOnError = ConfigBreakOnError;
            r_ctor_->ConfigKeepGuiFunc = ConfigKeepGuiFunc;
            r_ctor_->ConfigVerboseLevel = ConfigVerboseLevel;
            r_ctor_->ConfigVerboseLevelOnError = ConfigVerboseLevelOnError;
            r_ctor_->ConfigLogToTTY = ConfigLogToTTY;
            r_ctor_->ConfigLogToDebugger = ConfigLogToDebugger;
            r_ctor_->ConfigRestoreFocusAfterTests = ConfigRestoreFocusAfterTests;
            r_ctor_->ConfigCaptureEnabled = ConfigCaptureEnabled;
            r_ctor_->ConfigCaptureOnError = ConfigCaptureOnError;
            r_ctor_->ConfigNoThrottle = ConfigNoThrottle;
            r_ctor_->ConfigMouseDrawCursor = ConfigMouseDrawCursor;
            r_ctor_->ConfigFixedDeltaTime = ConfigFixedDeltaTime;
            r_ctor_->PerfStressAmount = PerfStressAmount;
            r_ctor_->MouseSpeed = MouseSpeed;
            r_ctor_->MouseWobble = MouseWobble;
            r_ctor_->ScrollSpeed = ScrollSpeed;
            r_ctor_->TypingSpeed = TypingSpeed;
            r_ctor_->ActionDelayShort = ActionDelayShort;
            r_ctor_->ActionDelayStandard = ActionDelayStandard;
            r_ctor_->ConfigWatchdogWarning = ConfigWatchdogWarning;
            r_ctor_->ConfigWatchdogKillTest = ConfigWatchdogKillTest;
            r_ctor_->ConfigWatchdogKillApp = ConfigWatchdogKillApp;
            r_ctor_->ExportResultsFormat = ExportResultsFormat;
            r_ctor_->CheckDrawDataIntegrity = CheckDrawDataIntegrity;
            r_ctor_->IsRunningTests = IsRunningTests;
            r_ctor_->IsRequestingMaxAppSpeed = IsRequestingMaxAppSpeed;
            r_ctor_->IsCapturing = IsCapturing;
        },
        nb::arg("config_saved_settings") = true, nb::arg("config_run_speed") = ImGuiTestRunSpeed_Fast, nb::arg("config_stop_on_error") = false, nb::arg("config_break_on_error") = false, nb::arg("config_keep_gui_func") = false, nb::arg("config_verbose_level") = ImGuiTestVerboseLevel_Warning, nb::arg("config_verbose_level_on_error") = ImGuiTestVerboseLevel_Info, nb::arg("config_log_to_tty") = false, nb::arg("config_log_to_debugger") = false, nb::arg("config_restore_focus_after_tests") = true, nb::arg("config_capture_enabled") = true, nb::arg("config_capture_on_error") = false, nb::arg("config_no_throttle") = false, nb::arg("config_mouse_draw_cursor") = true, nb::arg("config_fixed_delta_time") = 0.0f, nb::arg("perf_stress_amount") = 1, nb::arg("mouse_speed") = 600.0f, nb::arg("mouse_wobble") = 0.25f, nb::arg("scroll_speed") = 1400.0f, nb::arg("typing_speed") = 20.0f, nb::arg("action_delay_short") = 0.15f, nb::arg("action_delay_standard") = 0.40f, nb::arg("config_watchdog_warning") = 30.0f, nb::arg("config_watchdog_kill_test") = 60.0f, nb::arg("config_watchdog_kill_app") = FLT_MAX, nb::arg("export_results_format") = (ImGuiTestEngineExportFormat)0, nb::arg("check_draw_data_integrity") = false, nb::arg("is_running_tests") = false, nb::arg("is_requesting_max_app_speed") = false, nb::arg("is_capturing") = false
        )
        .def_rw("src_file_open_user_data", &ImGuiTestEngineIO::SrcFileOpenUserData, "(Optional) User data for SrcFileOpenFunc")
        .def_rw("screen_capture_user_data", &ImGuiTestEngineIO::ScreenCaptureUserData, "(Optional) User data for ScreenCaptureFunc")
        .def_rw("config_saved_settings", &ImGuiTestEngineIO::ConfigSavedSettings, "Load/Save settings in main context .ini file.")
        .def_rw("config_run_speed", &ImGuiTestEngineIO::ConfigRunSpeed, "Run tests in fast/normal/cinematic mode")
        .def_rw("config_stop_on_error", &ImGuiTestEngineIO::ConfigStopOnError, "Stop queued tests on test error")
        .def_rw("config_break_on_error", &ImGuiTestEngineIO::ConfigBreakOnError, "Break debugger on test error by calling IM_DEBUG_BREAK()")
        .def_rw("config_keep_gui_func", &ImGuiTestEngineIO::ConfigKeepGuiFunc, "Keep test GUI running at the end of the test")
        .def_rw("config_verbose_level", &ImGuiTestEngineIO::ConfigVerboseLevel, "")
        .def_rw("config_verbose_level_on_error", &ImGuiTestEngineIO::ConfigVerboseLevelOnError, "")
        .def_rw("config_log_to_tty", &ImGuiTestEngineIO::ConfigLogToTTY, "")
        .def_rw("config_log_to_debugger", &ImGuiTestEngineIO::ConfigLogToDebugger, "")
        .def_rw("config_restore_focus_after_tests", &ImGuiTestEngineIO::ConfigRestoreFocusAfterTests, "Restore focus back after running tests")
        .def_rw("config_capture_enabled", &ImGuiTestEngineIO::ConfigCaptureEnabled, "Master enable flags for capturing and saving captures. Disable to avoid e.g. lengthy saving of large PNG files.")
        .def_rw("config_capture_on_error", &ImGuiTestEngineIO::ConfigCaptureOnError, "")
        .def_rw("config_no_throttle", &ImGuiTestEngineIO::ConfigNoThrottle, "Disable vsync for performance measurement or fast test running")
        .def_rw("config_mouse_draw_cursor", &ImGuiTestEngineIO::ConfigMouseDrawCursor, "Enable drawing of Dear ImGui software mouse cursor when running tests")
        .def_rw("config_fixed_delta_time", &ImGuiTestEngineIO::ConfigFixedDeltaTime, "Use fixed delta time instead of calculating it from wall clock")
        .def_rw("perf_stress_amount", &ImGuiTestEngineIO::PerfStressAmount, "Integer to scale the amount of items submitted in test")
        .def_rw("mouse_speed", &ImGuiTestEngineIO::MouseSpeed, "Mouse speed (pixel/second) when not running in fast mode")
        .def_rw("mouse_wobble", &ImGuiTestEngineIO::MouseWobble, "(0.0..1.0) How much wobble to apply to the mouse (pixels per pixel of move distance) when not running in fast mode")
        .def_rw("scroll_speed", &ImGuiTestEngineIO::ScrollSpeed, "Scroll speed (pixel/second) when not running in fast mode")
        .def_rw("typing_speed", &ImGuiTestEngineIO::TypingSpeed, "Char input speed (characters/second) when not running in fast mode")
        .def_rw("action_delay_short", &ImGuiTestEngineIO::ActionDelayShort, "Time between short actions")
        .def_rw("action_delay_standard", &ImGuiTestEngineIO::ActionDelayStandard, "Time between most actions")
        .def_rw("config_watchdog_warning", &ImGuiTestEngineIO::ConfigWatchdogWarning, "Warn when a test exceed this time (in second)")
        .def_rw("config_watchdog_kill_test", &ImGuiTestEngineIO::ConfigWatchdogKillTest, "Attempt to stop running a test when exceeding this time (in second)")
        .def_rw("config_watchdog_kill_app", &ImGuiTestEngineIO::ConfigWatchdogKillApp, "Stop application when exceeding this time (in second)")
        .def_rw("export_results_format", &ImGuiTestEngineIO::ExportResultsFormat, "")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("export_results_filename_set",
            &ImGuiTestEngineIO::ExportResultsFilename_Set,
            nb::arg("filename"),
            " Set ExportResultsFilename from Python\n(private API)")
        // #endif
        //
        .def_rw("check_draw_data_integrity", &ImGuiTestEngineIO::CheckDrawDataIntegrity, "Check ImDrawData integrity (buffer count, etc.). Currently cheap but may become a slow operation.")
        .def_rw("is_running_tests", &ImGuiTestEngineIO::IsRunningTests, "")
        .def_rw("is_requesting_max_app_speed", &ImGuiTestEngineIO::IsRequestingMaxAppSpeed, "When running in fast mode: request app to skip vsync or even skip rendering if it wants")
        .def_rw("is_capturing", &ImGuiTestEngineIO::IsCapturing, "Capture is in progress")
        ;


    auto pyClassImGuiTestItemInfo =
        nb::class_<ImGuiTestItemInfo>
            (m, "TestItemInfo", "Information about a given item or window, result of an ItemInfo() or WindowInfo() query")
        .def_rw("id_", &ImGuiTestItemInfo::ID, "Item ID")
        .def_rw("window", &ImGuiTestItemInfo::Window, "Item Window")
        .def_rw("timestamp_main", &ImGuiTestItemInfo::TimestampMain, "Timestamp of main result (all fields)")
        .def_rw("timestamp_status", &ImGuiTestItemInfo::TimestampStatus, "Timestamp of StatusFlags")
        .def_rw("parent_id", &ImGuiTestItemInfo::ParentID, "Item Parent ID (value at top of the ID stack)")
        .def_rw("rect_full", &ImGuiTestItemInfo::RectFull, "Item Rectangle")
        .def_rw("rect_clipped", &ImGuiTestItemInfo::RectClipped, "Item Rectangle (clipped with window->ClipRect at time of item submission)")
        .def_rw("item_flags", &ImGuiTestItemInfo::ItemFlags, "Item flags")
        .def_rw("status_flags", &ImGuiTestItemInfo::StatusFlags, "Item Status flags (fully updated for some items only, compare TimestampStatus to FrameCount)")
        .def(nb::init<>())
        ;


    auto pyClassImGuiTestItemList =
        nb::class_<ImGuiTestItemList>
            (m, "TestItemList", "Result of an GatherItems() query")
        .def(nb::init<>()) // implicit default constructor
        .def("clear",
            &ImGuiTestItemList::Clear, "(private API)")
        .def("reserve",
            &ImGuiTestItemList::Reserve,
            nb::arg("capacity"),
            "(private API)")
        .def("get_size",
            &ImGuiTestItemList::GetSize, "(private API)")
        .def("get_by_index",
            &ImGuiTestItemList::GetByIndex,
            nb::arg("n"),
            "(private API)",
            nb::rv_policy::reference)
        .def("get_by_id",
            &ImGuiTestItemList::GetByID,
            nb::arg("id_"),
            "(private API)",
            nb::rv_policy::reference)
        .def("size",
            &ImGuiTestItemList::size, "(private API)")
        .def("begin",
            &ImGuiTestItemList::begin,
            "(private API)",
            nb::rv_policy::reference)
        .def("end",
            &ImGuiTestItemList::end,
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            &ImGuiTestItemList::operator[],
            nb::arg("n"),
            "(private API)",
            nb::rv_policy::reference)
        ;


    auto pyClassImGuiTestLogLineInfo =
        nb::class_<ImGuiTestLogLineInfo>
            (m, "TestLogLineInfo", "")
        .def("__init__", [](ImGuiTestLogLineInfo * self, ImGuiTestVerboseLevel Level = ImGuiTestVerboseLevel(), int LineOffset = int())
        {
            new (self) ImGuiTestLogLineInfo();  // placement new
            auto r_ctor_ = self;
            r_ctor_->Level = Level;
            r_ctor_->LineOffset = LineOffset;
        },
        nb::arg("level") = ImGuiTestVerboseLevel(), nb::arg("line_offset") = int()
        )
        .def_rw("level", &ImGuiTestLogLineInfo::Level, "")
        .def_rw("line_offset", &ImGuiTestLogLineInfo::LineOffset, "")
        ;


    auto pyClassImGuiTestLog =
        nb::class_<ImGuiTestLog>
            (m, "TestLog", "")
        .def_rw("buffer", &ImGuiTestLog::Buffer, "")
        .def(nb::init<>(),
            "Functions")
        .def("is_empty",
            &ImGuiTestLog::IsEmpty, "(private API)")
        .def("clear",
            &ImGuiTestLog::Clear, "(private API)")
        .def("extract_lines_for_verbose_levels",
            &ImGuiTestLog::ExtractLinesForVerboseLevels,
            nb::arg("level_min"), nb::arg("level_max"), nb::arg("out_buffer"),
            " Extract log contents filtered per log-level.\n Output:\n - If 'buffer != None': all extracted lines are appended to 'buffer'. Use 'buffer->c_str()' on your side to obtain the text.\n - Return value: number of lines extracted (should be equivalent to number of '\n' inside buffer->c_str()).\n - You may call the function with buffer == None to only obtain a count without getting the data.\n Verbose levels are inclusive:\n - To get ONLY Error:                     Use level_min == ImGuiTestVerboseLevel_Error, level_max = ImGuiTestVerboseLevel_Error\n - To get ONLY Error and Warnings:        Use level_min == ImGuiTestVerboseLevel_Error, level_max = ImGuiTestVerboseLevel_Warning\n - To get All Errors, Warnings, Debug...  Use level_min == ImGuiTestVerboseLevel_Error, level_max = ImGuiTestVerboseLevel_Trace\n(private API)")
        .def("update_line_offsets",
            &ImGuiTestLog::UpdateLineOffsets,
            nb::arg("engine_io"), nb::arg("level"), nb::arg("start"),
            " [Internal]\n(private API)")
        ;


    auto pyClassImGuiTestOutput =
        nb::class_<ImGuiTestOutput>
            (m, "TestOutput", "Storage for the output of a test run")
        .def("__init__", [](ImGuiTestOutput * self, ImGuiTestStatus Status = ImGuiTestStatus_Unknown, const std::optional<const ImGuiTestLog> & Log = std::nullopt, ImU64 StartTime = 0, ImU64 EndTime = 0)
        {
            new (self) ImGuiTestOutput();  // placement new
            auto r_ctor_ = self;
            r_ctor_->Status = Status;
            if (Log.has_value())
                r_ctor_->Log = Log.value();
            else
                r_ctor_->Log = ImGuiTestLog();
            r_ctor_->StartTime = StartTime;
            r_ctor_->EndTime = EndTime;
        },
        nb::arg("status") = ImGuiTestStatus_Unknown, nb::arg("log").none() = nb::none(), nb::arg("start_time") = 0, nb::arg("end_time") = 0
        )
        .def_rw("status", &ImGuiTestOutput::Status, "")
        .def_rw("log", &ImGuiTestOutput::Log, "")
        .def_rw("start_time", &ImGuiTestOutput::StartTime, "")
        .def_rw("end_time", &ImGuiTestOutput::EndTime, "")
        ;


    auto pyClassImGuiTest =
        nb::class_<ImGuiTest>
            (m, "Test", "Storage for one test")
        .def_rw("category", &ImGuiTest::Category, "Stored on the stack if len<30")
        .def_rw("name", &ImGuiTest::Name, "Stored on the stack if len<30")
        .def_rw("group", &ImGuiTest::Group, "Coarse groups: 'Tests' or 'Perf'")
        .def_rw("arg_variant", &ImGuiTest::ArgVariant, "User parameter. Generally we use it to run variations of a same test by sharing GuiFunc/TestFunc")
        .def_rw("flags", &ImGuiTest::Flags, "See ImGuiTestFlags_")
        .def_rw("gui_func", &ImGuiTest::GuiFunc, "GUI function (optional if your test are running over an existing GUI application)")
        .def_rw("test_func", &ImGuiTest::TestFunc, "Test function")
        .def_rw("user_data", &ImGuiTest::UserData, "General purpose user data (if assigning capturing lambdas on GuiFunc/TestFunc you may not need to use this)")
        .def_rw("source_file", &ImGuiTest::SourceFile, "__FILE__")
        .def_rw("source_line", &ImGuiTest::SourceLine, "__LINE__")
        .def_rw("source_line_end", &ImGuiTest::SourceLineEnd, "end of line (when calculated by ImGuiTestEngine_StartCalcSourceLineEnds())")
        .def_rw("output", &ImGuiTest::Output, " Last Test Output/Status\n (this is the only part that may change after registration)")
        .def_rw("vars_size", &ImGuiTest::VarsSize, "")
        .def_rw("vars_post_constructor_user_fn", &ImGuiTest::VarsPostConstructorUserFn, "")
        .def(nb::init<>(),
            "Functions")
        ;


    auto pyClassImGuiTestRunTask =
        nb::class_<ImGuiTestRunTask>
            (m, "TestRunTask", "Stored in test queue")
        .def("__init__", [](ImGuiTestRunTask * self, ImGuiTestRunFlags RunFlags = ImGuiTestRunFlags_None)
        {
            new (self) ImGuiTestRunTask();  // placement new
            auto r_ctor_ = self;
            r_ctor_->RunFlags = RunFlags;
        },
        nb::arg("run_flags") = ImGuiTestRunFlags_None
        )
        .def_rw("test", &ImGuiTestRunTask::Test, "")
        .def_rw("run_flags", &ImGuiTestRunTask::RunFlags, "")
        ;
    ////////////////////    </generated_from:imgui_te_engine.h>    ////////////////////


    ////////////////////    <generated_from:imgui_te_context.h>    ////////////////////
    auto pyClassImGuiTestRef =
        nb::class_<ImGuiTestRef>
            (m, "TestRef", " Weak reference to an Item/Window given an hashed ID _or_ a string path ID.\n This is most often passed as argument to function and generally has a very short lifetime.\n Documentation: https://github.com/ocornut/imgui_test_engine/wiki/Named-References\n (SUGGESTION: add those constructors to \"VA Step Filter\" (Visual Assist) or a .natstepfilter file (Visual Studio) so they are skipped by F11 (StepInto)")
        .def_rw("id_", &ImGuiTestRef::ID, "Pre-hashed ID")
        .def(nb::init<>())
        .def(nb::init<ImGuiID>(),
            nb::arg("id_"))
        .def(nb::init<const char *>(),
            nb::arg("path"))
        .def("is_empty",
            &ImGuiTestRef::IsEmpty, "(private API)")
        ;


    auto pyClassImGuiTestRefDesc =
        nb::class_<ImGuiTestRefDesc>
            (m, "TestRefDesc", " Debug helper to output a string showing the Path, ID or Debug Label based on what is available (some items only have ID as we couldn't find/store a Path)\n (The size is arbitrary, this is only used for logging info the user/debugger)")
        .def("c_str",
            &ImGuiTestRefDesc::c_str,
            "(private API)",
            nb::rv_policy::reference)
        .def(nb::init<const ImGuiTestRef &>(),
            nb::arg("ref"))
        .def(nb::init<const ImGuiTestRef &, const ImGuiTestItemInfo &>(),
            nb::arg("ref"), nb::arg("item"))
        ;


    auto pyEnumTestAction =
        nb::enum_<ImGuiTestAction>(m, "TestAction", nb::is_arithmetic(), nb::is_flag(), "Named actions. Generally you will call the named helpers e.g. ItemClick(). This is used by shared/low-level functions such as ItemAction().")
            .value("unknown", ImGuiTestAction_Unknown, "")
            .value("hover", ImGuiTestAction_Hover, "Move mouse")
            .value("click", ImGuiTestAction_Click, "Move mouse and click")
            .value("double_click", ImGuiTestAction_DoubleClick, "Move mouse and double-click")
            .value("check", ImGuiTestAction_Check, "Check item if unchecked (Checkbox, MenuItem or any widget reporting ImGuiItemStatusFlags_Checkable)")
            .value("uncheck", ImGuiTestAction_Uncheck, "Uncheck item if checked")
            .value("open", ImGuiTestAction_Open, "Open item if closed (TreeNode, BeginMenu or any widget reporting ImGuiItemStatusFlags_Openable)")
            .value("close", ImGuiTestAction_Close, "Close item if opened")
            .value("input", ImGuiTestAction_Input, "Start text inputing into a field (e.g. CTRL+Click on Drags/Slider, click on InputText etc.)")
            .value("nav_activate", ImGuiTestAction_NavActivate, "Activate item with navigation")
            .value("count", ImGuiTestAction_COUNT, "");


    auto pyEnumTestOpFlags_ =
        nb::enum_<ImGuiTestOpFlags_>(m, "TestOpFlags_", nb::is_arithmetic(), nb::is_flag(), " Generic flags for many ImGuiTestContext functions\n Some flags are only supported by a handful of functions. Check function headers for list of supported flags.")
            .value("none", ImGuiTestOpFlags_None, "")
            .value("no_check_hovered_id", ImGuiTestOpFlags_NoCheckHoveredId, "Don't check for HoveredId after aiming for a widget. A few situations may want this: while e.g. dragging or another items prevents hovering, or for items that don't use ItemHoverable()")
            .value("no_error", ImGuiTestOpFlags_NoError, "Don't abort/error e.g. if the item cannot be found or the operation doesn't succeed.")
            .value("no_focus_window", ImGuiTestOpFlags_NoFocusWindow, "Don't focus window when aiming at an item")
            .value("no_auto_uncollapse", ImGuiTestOpFlags_NoAutoUncollapse, "Disable automatically uncollapsing windows (useful when specifically testing Collapsing behaviors)")
            .value("no_auto_open_full_path", ImGuiTestOpFlags_NoAutoOpenFullPath, "Disable automatically opening intermediaries (e.g. ItemClick(\"Hello/OK\") will automatically first open \"Hello\" if \"OK\" isn't found. Only works if ref is a string path.")
            .value("no_yield", ImGuiTestOpFlags_NoYield, "Don't yield (only supported by a few functions), in case you need to manage rigorous per-frame timing.")
            .value("is_second_attempt", ImGuiTestOpFlags_IsSecondAttempt, "Used by recursing functions to indicate a second attempt")
            .value("move_to_edge_l", ImGuiTestOpFlags_MoveToEdgeL, "Simple Dumb aiming helpers to test widget that care about clicking position. May need to replace will better functionalities.")
            .value("move_to_edge_r", ImGuiTestOpFlags_MoveToEdgeR, "")
            .value("move_to_edge_u", ImGuiTestOpFlags_MoveToEdgeU, "")
            .value("move_to_edge_d", ImGuiTestOpFlags_MoveToEdgeD, "");


    auto pyClassImGuiTestActionFilter =
        nb::class_<ImGuiTestActionFilter>
            (m, "TestActionFilter", "Advanced filtering for ItemActionAll()")
        .def_rw("max_depth", &ImGuiTestActionFilter::MaxDepth, "")
        .def_rw("max_passes", &ImGuiTestActionFilter::MaxPasses, "")
        .def_ro("max_item_count_per_depth", &ImGuiTestActionFilter::MaxItemCountPerDepth, "")
        .def_rw("require_all_status_flags", &ImGuiTestActionFilter::RequireAllStatusFlags, "")
        .def_rw("require_any_status_flags", &ImGuiTestActionFilter::RequireAnyStatusFlags, "")
        .def(nb::init<>())
        ;


    auto pyClassImGuiTestGenericItemStatus =
        nb::class_<ImGuiTestGenericItemStatus>
            (m, "TestGenericItemStatus", " Helper struct to store various query-able state of an item.\n This facilitate interactions between GuiFunc and TestFunc, since those state are frequently used.")
        .def_rw("ret_value", &ImGuiTestGenericItemStatus::RetValue, "return value")
        .def_rw("hovered", &ImGuiTestGenericItemStatus::Hovered, "result of IsItemHovered()")
        .def_rw("hovered_allow_disabled", &ImGuiTestGenericItemStatus::HoveredAllowDisabled, "result of IsItemHovered(ImGuiHoveredFlags_AllowWhenDisabled)")
        .def_rw("active", &ImGuiTestGenericItemStatus::Active, "result of IsItemActive()")
        .def_rw("focused", &ImGuiTestGenericItemStatus::Focused, "result of IsItemFocused()")
        .def_rw("clicked", &ImGuiTestGenericItemStatus::Clicked, "result of IsItemClicked()")
        .def_rw("visible", &ImGuiTestGenericItemStatus::Visible, "result of IsItemVisible()")
        .def_rw("edited", &ImGuiTestGenericItemStatus::Edited, "result of IsItemEdited()")
        .def_rw("activated", &ImGuiTestGenericItemStatus::Activated, "result of IsItemActivated()")
        .def_rw("deactivated", &ImGuiTestGenericItemStatus::Deactivated, "result of IsItemDeactivated()")
        .def_rw("deactivated_after_edit", &ImGuiTestGenericItemStatus::DeactivatedAfterEdit, "result of IsItemDeactivatedAfterEdit()")
        .def(nb::init<>())
        .def("clear",
            &ImGuiTestGenericItemStatus::Clear, "(private API)")
        .def("query_set",
            &ImGuiTestGenericItemStatus::QuerySet,
            nb::arg("ret_val") = false,
            "(private API)")
        .def("query_inc",
            &ImGuiTestGenericItemStatus::QueryInc,
            nb::arg("ret_val") = false,
            "(private API)")
        .def("draw",
            &ImGuiTestGenericItemStatus::Draw, "(private API)")
        ;


    auto pyClassImGuiTestGenericVars =
        nb::class_<ImGuiTestGenericVars>
            (m, "TestGenericVars", " Generic structure with various storage fields.\n This is useful for tests to quickly share data between GuiFunc and TestFunc without creating custom data structure.\n If those fields are not enough: using test->SetVarsDataType<>() + ctx->GetVars<>() it is possible to store custom data.")
        .def_rw("step", &ImGuiTestGenericVars::Step, "")
        .def_rw("count", &ImGuiTestGenericVars::Count, "")
        .def_rw("dock_id", &ImGuiTestGenericVars::DockId, "")
        .def_rw("owner_id", &ImGuiTestGenericVars::OwnerId, "")
        .def_rw("window_size", &ImGuiTestGenericVars::WindowSize, "")
        .def_rw("window_flags", &ImGuiTestGenericVars::WindowFlags, "")
        .def_rw("table_flags", &ImGuiTestGenericVars::TableFlags, "")
        .def_rw("popup_flags", &ImGuiTestGenericVars::PopupFlags, "")
        .def_rw("input_text_flags", &ImGuiTestGenericVars::InputTextFlags, "")
        .def_rw("status", &ImGuiTestGenericVars::Status, "")
        .def_rw("show_window1", &ImGuiTestGenericVars::ShowWindow1, "")
        .def_rw("show_window2", &ImGuiTestGenericVars::ShowWindow2, "")
        .def_rw("use_clipper", &ImGuiTestGenericVars::UseClipper, "")
        .def_rw("use_viewports", &ImGuiTestGenericVars::UseViewports, "")
        .def_rw("width", &ImGuiTestGenericVars::Width, "")
        .def_rw("pos", &ImGuiTestGenericVars::Pos, "")
        .def_rw("pivot", &ImGuiTestGenericVars::Pivot, "")
        .def_rw("item_size", &ImGuiTestGenericVars::ItemSize, "")
        .def_rw("color1", &ImGuiTestGenericVars::Color1, "")
        .def_rw("color2", &ImGuiTestGenericVars::Color2, "")
        .def_rw("int1", &ImGuiTestGenericVars::Int1, "")
        .def_rw("int2", &ImGuiTestGenericVars::Int2, "")
        .def_prop_ro("int_array",
            [](ImGuiTestGenericVars &self) -> nb::ndarray<int, nb::numpy, nb::shape<10>, nb::c_contig>
            {
                return self.IntArray;
            },
            "")
        .def_rw("float1", &ImGuiTestGenericVars::Float1, "")
        .def_rw("float2", &ImGuiTestGenericVars::Float2, "")
        .def_prop_ro("float_array",
            [](ImGuiTestGenericVars &self) -> nb::ndarray<float, nb::numpy, nb::shape<10>, nb::c_contig>
            {
                return self.FloatArray;
            },
            "")
        .def_rw("bool1", &ImGuiTestGenericVars::Bool1, "")
        .def_rw("bool2", &ImGuiTestGenericVars::Bool2, "")
        .def_prop_ro("bool_array",
            [](ImGuiTestGenericVars &self) -> nb::ndarray<bool, nb::numpy, nb::shape<10>, nb::c_contig>
            {
                return self.BoolArray;
            },
            "")
        .def_rw("id_", &ImGuiTestGenericVars::Id, "")
        .def_prop_ro("id_array",
            [](ImGuiTestGenericVars &self) -> nb::ndarray<ImGuiID, nb::numpy, nb::shape<10>, nb::c_contig>
            {
                return self.IdArray;
            },
            "")
        .def(nb::init<>())
        .def("clear",
            &ImGuiTestGenericVars::Clear, "(private API)")
        ;


    auto pyClassImGuiTestContext =
        nb::class_<ImGuiTestContext>
            (m, "TestContext", "")
        .def("__init__", [](ImGuiTestContext * self, const std::optional<const ImGuiTestGenericVars> & GenericVars = std::nullopt, ImGuiTestOpFlags OpFlags = ImGuiTestOpFlags_None, int PerfStressAmount = 0, int FrameCount = 0, int FirstTestFrameCount = 0, bool FirstGuiFrame = false, bool HasDock = false, ImGuiTestRunFlags RunFlags = ImGuiTestRunFlags_None, ImGuiTestActiveFunc ActiveFunc = ImGuiTestActiveFunc_None, double RunningTime = 0.0, int ActionDepth = 0, int CaptureCounter = 0, int ErrorCounter = 0, bool Abort = false, double PerfRefDt = -1.0, int PerfIterations = 400, ImGuiID RefID = 0, ImGuiID RefWindowID = 0, const std::optional<const ImGuiInputSource> & InputMode = std::nullopt, const std::optional<const ImVector<char>> & TempString = std::nullopt, const std::optional<const ImVector<char>> & Clipboard = std::nullopt, const std::optional<const ImVector<ImGuiWindow*>> & ForeignWindowsToHide = std::nullopt, const std::optional<const ImGuiTestItemInfo> & DummyItemInfoNull = std::nullopt, bool CachedLinesPrintedToTTY = false)
        {
            new (self) ImGuiTestContext();  // placement new
            auto r_ctor_ = self;
            if (GenericVars.has_value())
                r_ctor_->GenericVars = GenericVars.value();
            else
                r_ctor_->GenericVars = ImGuiTestGenericVars();
            r_ctor_->OpFlags = OpFlags;
            r_ctor_->PerfStressAmount = PerfStressAmount;
            r_ctor_->FrameCount = FrameCount;
            r_ctor_->FirstTestFrameCount = FirstTestFrameCount;
            r_ctor_->FirstGuiFrame = FirstGuiFrame;
            r_ctor_->HasDock = HasDock;
            r_ctor_->RunFlags = RunFlags;
            r_ctor_->ActiveFunc = ActiveFunc;
            r_ctor_->RunningTime = RunningTime;
            r_ctor_->ActionDepth = ActionDepth;
            r_ctor_->CaptureCounter = CaptureCounter;
            r_ctor_->ErrorCounter = ErrorCounter;
            r_ctor_->Abort = Abort;
            r_ctor_->PerfRefDt = PerfRefDt;
            r_ctor_->PerfIterations = PerfIterations;
            r_ctor_->RefID = RefID;
            r_ctor_->RefWindowID = RefWindowID;
            if (InputMode.has_value())
                r_ctor_->InputMode = InputMode.value();
            else
                r_ctor_->InputMode = ImGuiInputSource_Mouse;
            if (TempString.has_value())
                r_ctor_->TempString = TempString.value();
            else
                r_ctor_->TempString = ImVector<char>();
            if (Clipboard.has_value())
                r_ctor_->Clipboard = Clipboard.value();
            else
                r_ctor_->Clipboard = ImVector<char>();
            if (ForeignWindowsToHide.has_value())
                r_ctor_->ForeignWindowsToHide = ForeignWindowsToHide.value();
            else
                r_ctor_->ForeignWindowsToHide = ImVector<ImGuiWindow*>();
            if (DummyItemInfoNull.has_value())
                r_ctor_->DummyItemInfoNull = DummyItemInfoNull.value();
            else
                r_ctor_->DummyItemInfoNull = ImGuiTestItemInfo();
            r_ctor_->CachedLinesPrintedToTTY = CachedLinesPrintedToTTY;
        },
        nb::arg("generic_vars").none() = nb::none(), nb::arg("op_flags") = ImGuiTestOpFlags_None, nb::arg("perf_stress_amount") = 0, nb::arg("frame_count") = 0, nb::arg("first_test_frame_count") = 0, nb::arg("first_gui_frame") = false, nb::arg("has_dock") = false, nb::arg("run_flags") = ImGuiTestRunFlags_None, nb::arg("active_func") = ImGuiTestActiveFunc_None, nb::arg("running_time") = 0.0, nb::arg("action_depth") = 0, nb::arg("capture_counter") = 0, nb::arg("error_counter") = 0, nb::arg("abort") = false, nb::arg("perf_ref_dt") = -1.0, nb::arg("perf_iterations") = 400, nb::arg("ref_id") = 0, nb::arg("ref_window_id") = 0, nb::arg("input_mode").none() = nb::none(), nb::arg("temp_string").none() = nb::none(), nb::arg("clipboard").none() = nb::none(), nb::arg("foreign_windows_to_hide").none() = nb::none(), nb::arg("dummy_item_info_null").none() = nb::none(), nb::arg("cached_lines_printed_to_tty") = false
        )
        .def_rw("generic_vars", &ImGuiTestContext::GenericVars, "Generic variables holder for convenience.")
        .def_rw("user_vars", &ImGuiTestContext::UserVars, "Access using ctx->GetVars<Type>(). Setup with test->SetVarsDataType<>().")
        .def_rw("ui_context", &ImGuiTestContext::UiContext, "UI context")
        .def_rw("engine_io", &ImGuiTestContext::EngineIO, "Test Engine IO/settings")
        .def_rw("test", &ImGuiTestContext::Test, "Test currently running")
        .def_rw("test_output", &ImGuiTestContext::TestOutput, "Test output (generally == &Test->Output)")
        .def_rw("op_flags", &ImGuiTestContext::OpFlags, "Flags affecting all operation (supported: ImGuiTestOpFlags_NoAutoUncollapse)")
        .def_rw("perf_stress_amount", &ImGuiTestContext::PerfStressAmount, "Convenience copy of engine->IO.PerfStressAmount")
        .def_rw("frame_count", &ImGuiTestContext::FrameCount, "Test frame count (restarts from zero every time)")
        .def_rw("first_test_frame_count", &ImGuiTestContext::FirstTestFrameCount, "First frame where TestFunc is running (after warm-up frame). This is generally -1 or 0 depending on whether we have warm up enabled")
        .def_rw("first_gui_frame", &ImGuiTestContext::FirstGuiFrame, "")
        .def_rw("has_dock", &ImGuiTestContext::HasDock, "#ifdef IMGUI_HAS_DOCK expressed in an easier to test value")
        .def_rw("engine", &ImGuiTestContext::Engine, "")
        .def_rw("inputs", &ImGuiTestContext::Inputs, "")
        .def_rw("run_flags", &ImGuiTestContext::RunFlags, "")
        .def_rw("active_func", &ImGuiTestContext::ActiveFunc, "None/GuiFunc/TestFunc")
        .def_rw("running_time", &ImGuiTestContext::RunningTime, "Amount of wall clock time the Test has been running. Used by safety watchdog.")
        .def_rw("action_depth", &ImGuiTestContext::ActionDepth, "Nested depth of ctx-> function calls (used to decorate log)")
        .def_rw("capture_counter", &ImGuiTestContext::CaptureCounter, "Number of captures")
        .def_rw("error_counter", &ImGuiTestContext::ErrorCounter, "Number of errors (generally this maxxes at 1 as most functions will early out)")
        .def_rw("abort", &ImGuiTestContext::Abort, "")
        .def_rw("perf_ref_dt", &ImGuiTestContext::PerfRefDt, "")
        .def_rw("perf_iterations", &ImGuiTestContext::PerfIterations, "Number of frames for PerfCapture() measurements")
        .def_rw("ref_id", &ImGuiTestContext::RefID, "Reference ID over which all named references are based")
        .def_rw("ref_window_id", &ImGuiTestContext::RefWindowID, "ID of a window that contains RefID item")
        .def_rw("input_mode", &ImGuiTestContext::InputMode, "Prefer interacting with mouse/keyboard/gamepad")
        .def_rw("temp_string", &ImGuiTestContext::TempString, "")
        .def_rw("clipboard", &ImGuiTestContext::Clipboard, "Private clipboard for the test instance")
        .def_rw("foreign_windows_to_hide", &ImGuiTestContext::ForeignWindowsToHide, "")
        .def_rw("dummy_item_info_null", &ImGuiTestContext::DummyItemInfoNull, "Storage for ItemInfoNull()")
        .def_rw("cached_lines_printed_to_tty", &ImGuiTestContext::CachedLinesPrintedToTTY, "")
        .def("finish",
            &ImGuiTestContext::Finish,
            nb::arg("status") = ImGuiTestStatus_Success,
            "(private API)\n\n Set test status and stop running. Usually called when running test logic from GuiFunc() only.")
        .def("run_child_test",
            &ImGuiTestContext::RunChildTest,
            nb::arg("test_name"), nb::arg("flags") = 0,
            "(private API)\n\n [Experimental] Run another test from the current test.")
        .def("is_error",
            &ImGuiTestContext::IsError, "(private API)")
        .def("is_warm_up_gui_frame",
            &ImGuiTestContext::IsWarmUpGuiFrame, "(private API)\n\n Unless test->Flags has ImGuiTestFlags_NoGuiWarmUp, we run GuiFunc() twice before running TestFunc(). Those frames are called \"WarmUp\" frames.")
        .def("is_first_gui_frame",
            &ImGuiTestContext::IsFirstGuiFrame, "(private API)")
        .def("is_first_test_frame",
            &ImGuiTestContext::IsFirstTestFrame, "(private API)\n\n First frame where TestFunc is running (after warm-up frame).")
        .def("is_gui_func_only",
            &ImGuiTestContext::IsGuiFuncOnly, "(private API)")
        .def("suspend_test_func",
            [](ImGuiTestContext & self, std::optional<std::string> file = std::nullopt, int line = 0) -> bool
            {
                auto SuspendTestFunc_adapt_const_char_pointer_with_default_null = [&self](std::optional<std::string> file = std::nullopt, int line = 0) -> bool
                {
                    const char * file_adapt_default_null = nullptr;
                    if (file.has_value())
                        file_adapt_default_null = file.value().c_str();

                    auto lambda_result = self.SuspendTestFunc(file_adapt_default_null, line);
                    return lambda_result;
                };

                return SuspendTestFunc_adapt_const_char_pointer_with_default_null(file, line);
            },
            nb::arg("file").none() = nb::none(), nb::arg("line") = 0,
            "(private API)\n\n [DEBUG] Generally called via IM_SUSPEND_TESTFUNC")
        .def("log_ex",
            [](ImGuiTestContext & self, ImGuiTestVerboseLevel level, ImGuiTestLogFlags flags, const char * fmt)
            {
                auto LogEx_adapt_variadic_format = [&self](ImGuiTestVerboseLevel level, ImGuiTestLogFlags flags, const char * fmt)
                {
                    self.LogEx(level, flags, "%s", fmt);
                };

                LogEx_adapt_variadic_format(level, flags, fmt);
            },
            nb::arg("level"), nb::arg("flags"), nb::arg("fmt"),
            "(private API)")
        .def("log_to_tty",
            [](ImGuiTestContext & self, ImGuiTestVerboseLevel level, const char * message, std::optional<std::string> message_end = std::nullopt)
            {
                auto LogToTTY_adapt_const_char_pointer_with_default_null = [&self](ImGuiTestVerboseLevel level, const char * message, std::optional<std::string> message_end = std::nullopt)
                {
                    const char * message_end_adapt_default_null = nullptr;
                    if (message_end.has_value())
                        message_end_adapt_default_null = message_end.value().c_str();

                    self.LogToTTY(level, message, message_end_adapt_default_null);
                };

                LogToTTY_adapt_const_char_pointer_with_default_null(level, message, message_end);
            },
            nb::arg("level"), nb::arg("message"), nb::arg("message_end").none() = nb::none(),
            "(private API)")
        .def("log_to_debugger",
            &ImGuiTestContext::LogToDebugger,
            nb::arg("level"), nb::arg("message"),
            "(private API)")
        .def("log_debug",
            [](ImGuiTestContext & self, const char * fmt)
            {
                auto LogDebug_adapt_variadic_format = [&self](const char * fmt)
                {
                    self.LogDebug("%s", fmt);
                };

                LogDebug_adapt_variadic_format(fmt);
            },
            nb::arg("fmt"),
            "(private API)\n\n ImGuiTestVerboseLevel_Debug or ImGuiTestVerboseLevel_Trace depending on context depth")
        .def("log_info",
            [](ImGuiTestContext & self, const char * fmt)
            {
                auto LogInfo_adapt_variadic_format = [&self](const char * fmt)
                {
                    self.LogInfo("%s", fmt);
                };

                LogInfo_adapt_variadic_format(fmt);
            },
            nb::arg("fmt"),
            "(private API)\n\n ImGuiTestVerboseLevel_Info")
        .def("log_warning",
            [](ImGuiTestContext & self, const char * fmt)
            {
                auto LogWarning_adapt_variadic_format = [&self](const char * fmt)
                {
                    self.LogWarning("%s", fmt);
                };

                LogWarning_adapt_variadic_format(fmt);
            },
            nb::arg("fmt"),
            "(private API)\n\n ImGuiTestVerboseLevel_Warning")
        .def("log_error",
            [](ImGuiTestContext & self, const char * fmt)
            {
                auto LogError_adapt_variadic_format = [&self](const char * fmt)
                {
                    self.LogError("%s", fmt);
                };

                LogError_adapt_variadic_format(fmt);
            },
            nb::arg("fmt"),
            "(private API)\n\n ImGuiTestVerboseLevel_Error")
        .def("log_basic_ui_state",
            &ImGuiTestContext::LogBasicUiState, "(private API)")
        .def("log_item_list",
            &ImGuiTestContext::LogItemList,
            nb::arg("list"),
            "(private API)")
        .def("yield_",
            &ImGuiTestContext::Yield,
            nb::arg("count") = 1,
            "(private API)")
        .def("sleep",
            &ImGuiTestContext::Sleep,
            nb::arg("time_in_second"),
            "(private API)\n\n Sleep for a given simulation time, unless in Fast mode")
        .def("sleep_short",
            &ImGuiTestContext::SleepShort, "(private API)\n\n Standard short delay of io.ActionDelayShort (~0.15), unless in Fast mode.")
        .def("sleep_standard",
            &ImGuiTestContext::SleepStandard, "(private API)\n\n Standard regular delay of io.ActionDelayStandard (~0.40), unless in Fast mode.")
        .def("sleep_no_skip",
            &ImGuiTestContext::SleepNoSkip,
            nb::arg("time_in_second"), nb::arg("framestep_in_second"),
            "(private API)")
        .def("set_ref",
            nb::overload_cast<ImGuiTestRef>(&ImGuiTestContext::SetRef),
            nb::arg("ref"),
            "(private API)")
        .def("set_ref",
            nb::overload_cast<ImGuiWindow *>(&ImGuiTestContext::SetRef),
            nb::arg("window"),
            "(private API)\n\n Shortcut to SetRef(window->Name) which works for ChildWindow (see code)")
        .def("get_ref",
            &ImGuiTestContext::GetRef, "(private API)")
        .def("window_info",
            &ImGuiTestContext::WindowInfo,
            nb::arg("window_ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("window_close",
            &ImGuiTestContext::WindowClose,
            nb::arg("window_ref"),
            "(private API)")
        .def("window_collapse",
            &ImGuiTestContext::WindowCollapse,
            nb::arg("window_ref"), nb::arg("collapsed"),
            "(private API)")
        .def("window_focus",
            &ImGuiTestContext::WindowFocus,
            nb::arg("window_ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("window_bring_to_front",
            &ImGuiTestContext::WindowBringToFront,
            nb::arg("window_ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("window_move",
            [](ImGuiTestContext & self, ImGuiTestRef window_ref, ImVec2 pos, const std::optional<const ImVec2> & pivot = std::nullopt, ImGuiTestOpFlags flags = ImGuiTestOpFlags_None)
            {
                auto WindowMove_adapt_mutable_param_with_default_value = [&self](ImGuiTestRef window_ref, ImVec2 pos, const std::optional<const ImVec2> & pivot = std::nullopt, ImGuiTestOpFlags flags = ImGuiTestOpFlags_None)
                {

                    const ImVec2& pivot_or_default = [&]() -> const ImVec2 {
                        if (pivot.has_value())
                            return pivot.value();
                        else
                            return ImVec2(0.0f, 0.0f);
                    }();

                    self.WindowMove(window_ref, pos, pivot_or_default, flags);
                };

                WindowMove_adapt_mutable_param_with_default_value(window_ref, pos, pivot, flags);
            },
            nb::arg("window_ref"), nb::arg("pos"), nb::arg("pivot").none() = nb::none(), nb::arg("flags") = ImGuiTestOpFlags_None,
            "Python bindings defaults:\n    If pivot is None, then its default value will be: ImVec2(0.0, 0.0)\n(private API)")
        .def("window_resize",
            &ImGuiTestContext::WindowResize,
            nb::arg("window_ref"), nb::arg("sz"),
            "(private API)")
        .def("window_teleport_to_make_pos_visible",
            &ImGuiTestContext::WindowTeleportToMakePosVisible,
            nb::arg("window_ref"), nb::arg("pos_in_window"),
            "(private API)")
        .def("get_window_by_ref",
            &ImGuiTestContext::GetWindowByRef,
            nb::arg("window_ref"),
            "(private API)",
            nb::rv_policy::reference)
        .def("popup_close_one",
            [](ImGuiTestContext & self) { return self.PopupCloseOne(); }, "(private API)")
        .def("popup_close_all",
            [](ImGuiTestContext & self) { return self.PopupCloseAll(); }, "(private API)")
        .def("popup_get_window_id",
            nb::overload_cast<ImGuiTestRef>(&ImGuiTestContext::PopupGetWindowID),
            nb::arg("ref"),
            "(private API)")
        .def("get_id",
            nb::overload_cast<ImGuiTestRef>(&ImGuiTestContext::GetID),
            nb::arg("ref"),
            "(private API)")
        .def("get_id",
            nb::overload_cast<ImGuiTestRef, ImGuiTestRef>(&ImGuiTestContext::GetID),
            nb::arg("ref"), nb::arg("seed_ref"),
            "(private API)")
        .def("get_pos_on_void",
            &ImGuiTestContext::GetPosOnVoid,
            nb::arg("viewport"),
            "(private API)\n\n Find a point that has no windows // FIXME: This needs error return and flag to enable/disable forcefully finding None.")
        .def("get_window_titlebar_point",
            &ImGuiTestContext::GetWindowTitlebarPoint,
            nb::arg("window_ref"),
            "(private API)\n\n Return a clickable point on window title-bar (window tab for docked windows).")
        .def("get_main_monitor_work_pos",
            &ImGuiTestContext::GetMainMonitorWorkPos, "(private API)\n\n Work pos and size of main viewport when viewports are disabled, or work pos and size of monitor containing main viewport when viewports are enabled.")
        .def("get_main_monitor_work_size",
            &ImGuiTestContext::GetMainMonitorWorkSize, "(private API)")
        .def("capture_reset",
            &ImGuiTestContext::CaptureReset, "(private API)\n\n Reset state (use when doing multiple captures)")
        .def("capture_set_extension",
            &ImGuiTestContext::CaptureSetExtension,
            nb::arg("ext"),
            "(private API)\n\n Set capture file format (otherwise for video this default to EngineIO->VideoCaptureExtension)")
        .def("capture_add_window",
            &ImGuiTestContext::CaptureAddWindow,
            nb::arg("ref"),
            "(private API)\n\n Add window to be captured (default to capture everything)")
        .def("capture_screenshot_window",
            &ImGuiTestContext::CaptureScreenshotWindow,
            nb::arg("ref"), nb::arg("capture_flags") = 0,
            "(private API)\n\n Trigger a screen capture of a single window (== CaptureAddWindow() + CaptureScreenshot())")
        .def("capture_screenshot",
            &ImGuiTestContext::CaptureScreenshot,
            nb::arg("capture_flags") = 0,
            "(private API)\n\n Trigger a screen capture")
        .def("capture_begin_video",
            &ImGuiTestContext::CaptureBeginVideo, "(private API)\n\n Start a video capture")
        .def("capture_end_video",
            &ImGuiTestContext::CaptureEndVideo, "(private API)")
        .def("mouse_move",
            &ImGuiTestContext::MouseMove,
            nb::arg("ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("mouse_move_to_pos",
            &ImGuiTestContext::MouseMoveToPos,
            nb::arg("pos"),
            "(private API)")
        .def("mouse_teleport_to_pos",
            &ImGuiTestContext::MouseTeleportToPos,
            nb::arg("pos"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("mouse_click",
            &ImGuiTestContext::MouseClick,
            nb::arg("button") = 0,
            "(private API)")
        .def("mouse_click_multi",
            &ImGuiTestContext::MouseClickMulti,
            nb::arg("button"), nb::arg("count"),
            "(private API)")
        .def("mouse_double_click",
            &ImGuiTestContext::MouseDoubleClick,
            nb::arg("button") = 0,
            "(private API)")
        .def("mouse_down",
            &ImGuiTestContext::MouseDown,
            nb::arg("button") = 0,
            "(private API)")
        .def("mouse_up",
            &ImGuiTestContext::MouseUp,
            nb::arg("button") = 0,
            "(private API)")
        .def("mouse_lift_drag_threshold",
            &ImGuiTestContext::MouseLiftDragThreshold,
            nb::arg("button") = 0,
            "(private API)")
        .def("mouse_drag_with_delta",
            &ImGuiTestContext::MouseDragWithDelta,
            nb::arg("delta"), nb::arg("button") = 0,
            "(private API)")
        .def("mouse_wheel",
            &ImGuiTestContext::MouseWheel,
            nb::arg("delta"),
            "(private API)")
        .def("mouse_wheel_x",
            &ImGuiTestContext::MouseWheelX,
            nb::arg("dx"),
            "(private API)")
        .def("mouse_wheel_y",
            &ImGuiTestContext::MouseWheelY,
            nb::arg("dy"),
            "(private API)")
        .def("mouse_move_to_void",
            &ImGuiTestContext::MouseMoveToVoid,
            nb::arg("viewport") = nb::none(),
            "(private API)")
        .def("mouse_click_on_void",
            &ImGuiTestContext::MouseClickOnVoid,
            nb::arg("button") = 0, nb::arg("viewport") = nb::none(),
            "(private API)")
        .def("find_hovered_window_at_pos",
            &ImGuiTestContext::FindHoveredWindowAtPos,
            nb::arg("pos"),
            "(private API)",
            nb::rv_policy::reference)
        .def("find_existing_void_pos_on_viewport",
            &ImGuiTestContext::FindExistingVoidPosOnViewport,
            nb::arg("viewport"), nb::arg("out"),
            "(private API)")
        .def("mouse_set_viewport",
            &ImGuiTestContext::MouseSetViewport,
            nb::arg("window"),
            "(private API)")
        .def("mouse_set_viewport_id",
            &ImGuiTestContext::MouseSetViewportID,
            nb::arg("viewport_id"),
            "(private API)")
        .def("key_down",
            &ImGuiTestContext::KeyDown,
            nb::arg("key_chord"),
            "(private API)")
        .def("key_up",
            &ImGuiTestContext::KeyUp,
            nb::arg("key_chord"),
            "(private API)")
        .def("key_press",
            &ImGuiTestContext::KeyPress,
            nb::arg("key_chord"), nb::arg("count") = 1,
            "(private API)")
        .def("key_hold",
            &ImGuiTestContext::KeyHold,
            nb::arg("key_chord"), nb::arg("time"),
            "(private API)")
        .def("key_set_ex",
            &ImGuiTestContext::KeySetEx,
            nb::arg("key_chord"), nb::arg("is_down"), nb::arg("time"),
            "(private API)")
        .def("key_chars",
            &ImGuiTestContext::KeyChars,
            nb::arg("chars"),
            "(private API)\n\n Input characters")
        .def("key_chars_append",
            &ImGuiTestContext::KeyCharsAppend,
            nb::arg("chars"),
            "(private API)\n\n Input characters at end of field")
        .def("key_chars_append_enter",
            &ImGuiTestContext::KeyCharsAppendEnter,
            nb::arg("chars"),
            "(private API)\n\n Input characters at end of field, press Enter")
        .def("key_chars_replace",
            &ImGuiTestContext::KeyCharsReplace,
            nb::arg("chars"),
            "(private API)\n\n Delete existing field then input characters")
        .def("key_chars_replace_enter",
            &ImGuiTestContext::KeyCharsReplaceEnter,
            nb::arg("chars"),
            "(private API)\n\n Delete existing field then input characters, press Enter")
        .def("set_input_mode",
            &ImGuiTestContext::SetInputMode,
            nb::arg("input_mode"),
            "(private API)\n\n Mouse or Keyboard or Gamepad. In Keyboard or Gamepad mode, actions such as ItemClick or ItemInput are using nav facilities instead of Mouse.")
        .def("nav_move_to",
            &ImGuiTestContext::NavMoveTo,
            nb::arg("ref"),
            "(private API)")
        .def("nav_activate",
            &ImGuiTestContext::NavActivate, "(private API)\n\n Activate current selected item: activate button, tweak sliders/drags. Equivalent of pressing Space on keyboard, ImGuiKey_GamepadFaceUp on a gamepad.")
        .def("nav_input",
            &ImGuiTestContext::NavInput, "(private API)\n\n Input into select item: input sliders/drags. Equivalent of pressing Enter on keyboard, ImGuiKey_GamepadFaceDown on a gamepad.")
        .def("scroll_to",
            &ImGuiTestContext::ScrollTo,
            nb::arg("ref"), nb::arg("axis"), nb::arg("scroll_v"), nb::arg("flags") = ImGuiTestOpFlags_None,
            " Scrolling\n(private API)")
        .def("scroll_to_x",
            &ImGuiTestContext::ScrollToX,
            nb::arg("ref"), nb::arg("scroll_x"),
            "(private API)")
        .def("scroll_to_y",
            &ImGuiTestContext::ScrollToY,
            nb::arg("ref"), nb::arg("scroll_y"),
            "(private API)")
        .def("scroll_to_top",
            &ImGuiTestContext::ScrollToTop,
            nb::arg("ref"),
            "(private API)")
        .def("scroll_to_bottom",
            &ImGuiTestContext::ScrollToBottom,
            nb::arg("ref"),
            "(private API)")
        .def("scroll_to_pos",
            &ImGuiTestContext::ScrollToPos,
            nb::arg("window_ref"), nb::arg("pos_v"), nb::arg("axis"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("scroll_to_pos_x",
            &ImGuiTestContext::ScrollToPosX,
            nb::arg("window_ref"), nb::arg("pos_x"),
            "(private API)")
        .def("scroll_to_pos_y",
            &ImGuiTestContext::ScrollToPosY,
            nb::arg("window_ref"), nb::arg("pos_y"),
            "(private API)")
        .def("scroll_to_item",
            &ImGuiTestContext::ScrollToItem,
            nb::arg("ref"), nb::arg("axis"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("scroll_to_item_x",
            &ImGuiTestContext::ScrollToItemX,
            nb::arg("ref"),
            "(private API)")
        .def("scroll_to_item_y",
            &ImGuiTestContext::ScrollToItemY,
            nb::arg("ref"),
            "(private API)")
        .def("scroll_to_tab_item",
            &ImGuiTestContext::ScrollToTabItem,
            nb::arg("tab_bar"), nb::arg("tab_id"),
            "(private API)")
        .def("scroll_error_check",
            [](ImGuiTestContext & self, ImGuiAxis axis, float expected, float actual, int remaining_attempts) -> std::tuple<bool, int>
            {
                auto ScrollErrorCheck_adapt_modifiable_immutable_to_return = [&self](ImGuiAxis axis, float expected, float actual, int remaining_attempts) -> std::tuple<bool, int>
                {
                    int * remaining_attempts_adapt_modifiable = & remaining_attempts;

                    bool r = self.ScrollErrorCheck(axis, expected, actual, remaining_attempts_adapt_modifiable);
                    return std::make_tuple(r, remaining_attempts);
                };

                return ScrollErrorCheck_adapt_modifiable_immutable_to_return(axis, expected, actual, remaining_attempts);
            },
            nb::arg("axis"), nb::arg("expected"), nb::arg("actual"), nb::arg("remaining_attempts"),
            "(private API)")
        .def("scroll_verify_scroll_max",
            &ImGuiTestContext::ScrollVerifyScrollMax,
            nb::arg("ref"),
            "(private API)")
        .def("item_info",
            &ImGuiTestContext::ItemInfo,
            nb::arg("ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("item_info_open_full_path",
            &ImGuiTestContext::ItemInfoOpenFullPath,
            nb::arg("ref"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("item_info_handle_wildcard_search",
            &ImGuiTestContext::ItemInfoHandleWildcardSearch,
            nb::arg("wildcard_prefix_start"), nb::arg("wildcard_prefix_end"), nb::arg("wildcard_suffix_start"),
            "(private API)")
        .def("item_info_null",
            &ImGuiTestContext::ItemInfoNull, "(private API)")
        .def("gather_items",
            &ImGuiTestContext::GatherItems,
            nb::arg("out_list"), nb::arg("parent"), nb::arg("depth") = -1,
            "(private API)")
        .def("item_action",
            &ImGuiTestContext::ItemAction,
            nb::arg("action"), nb::arg("ref"), nb::arg("flags") = 0, nb::arg("action_arg") = nb::none(),
            " Item/Widgets manipulation\n(private API)")
        .def("item_click",
            &ImGuiTestContext::ItemClick,
            nb::arg("ref"), nb::arg("button") = 0, nb::arg("flags") = 0,
            "(private API)")
        .def("item_double_click",
            &ImGuiTestContext::ItemDoubleClick,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_check",
            &ImGuiTestContext::ItemCheck,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_uncheck",
            &ImGuiTestContext::ItemUncheck,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_open",
            &ImGuiTestContext::ItemOpen,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_close",
            &ImGuiTestContext::ItemClose,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_input",
            &ImGuiTestContext::ItemInput,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_nav_activate",
            &ImGuiTestContext::ItemNavActivate,
            nb::arg("ref"), nb::arg("flags") = 0,
            "(private API)")
        .def("item_action_all",
            &ImGuiTestContext::ItemActionAll,
            nb::arg("action"), nb::arg("ref_parent"), nb::arg("filter") = nb::none(),
            "(private API)")
        .def("item_open_all",
            &ImGuiTestContext::ItemOpenAll,
            nb::arg("ref_parent"), nb::arg("depth") = -1, nb::arg("passes") = -1,
            "(private API)")
        .def("item_close_all",
            &ImGuiTestContext::ItemCloseAll,
            nb::arg("ref_parent"), nb::arg("depth") = -1, nb::arg("passes") = -1,
            "(private API)")
        .def("item_input_value",
            nb::overload_cast<ImGuiTestRef, int>(&ImGuiTestContext::ItemInputValue),
            nb::arg("ref"), nb::arg("v"),
            "(private API)")
        .def("item_input_value",
            nb::overload_cast<ImGuiTestRef, float>(&ImGuiTestContext::ItemInputValue),
            nb::arg("ref"), nb::arg("f"),
            "(private API)")
        .def("item_input_value",
            nb::overload_cast<ImGuiTestRef, const char *>(&ImGuiTestContext::ItemInputValue),
            nb::arg("ref"), nb::arg("str"),
            "(private API)")
        .def("item_read_as_int",
            &ImGuiTestContext::ItemReadAsInt,
            nb::arg("ref"),
            "(private API)")
        .def("item_read_as_float",
            &ImGuiTestContext::ItemReadAsFloat,
            nb::arg("ref"),
            "(private API)")
        .def("item_read_as_scalar",
            &ImGuiTestContext::ItemReadAsScalar,
            nb::arg("ref"), nb::arg("data_type"), nb::arg("out_data"), nb::arg("flags") = ImGuiTestOpFlags_None,
            "(private API)")
        .def("item_exists",
            &ImGuiTestContext::ItemExists,
            nb::arg("ref"),
            "(private API)")
        .def("item_is_checked",
            &ImGuiTestContext::ItemIsChecked,
            nb::arg("ref"),
            "(private API)")
        .def("item_is_opened",
            &ImGuiTestContext::ItemIsOpened,
            nb::arg("ref"),
            "(private API)")
        .def("item_verify_checked_if_alive",
            &ImGuiTestContext::ItemVerifyCheckedIfAlive,
            nb::arg("ref"), nb::arg("checked"),
            "(private API)")
        .def("item_hold",
            &ImGuiTestContext::ItemHold,
            nb::arg("ref"), nb::arg("time"),
            "(private API)")
        .def("item_hold_for_frames",
            &ImGuiTestContext::ItemHoldForFrames,
            nb::arg("ref"), nb::arg("frames"),
            "(private API)")
        .def("item_drag_over_and_hold",
            &ImGuiTestContext::ItemDragOverAndHold,
            nb::arg("ref_src"), nb::arg("ref_dst"),
            "(private API)")
        .def("item_drag_and_drop",
            &ImGuiTestContext::ItemDragAndDrop,
            nb::arg("ref_src"), nb::arg("ref_dst"), nb::arg("button") = 0,
            "(private API)")
        .def("item_drag_with_delta",
            &ImGuiTestContext::ItemDragWithDelta,
            nb::arg("ref_src"), nb::arg("pos_delta"),
            "(private API)")
        .def("tab_close",
            &ImGuiTestContext::TabClose,
            nb::arg("ref"),
            "(private API)")
        .def("menu_action",
            &ImGuiTestContext::MenuAction,
            nb::arg("action"), nb::arg("ref"),
            "(private API)")
        .def("menu_action_all",
            &ImGuiTestContext::MenuActionAll,
            nb::arg("action"), nb::arg("ref_parent"),
            "(private API)")
        .def("menu_click",
            &ImGuiTestContext::MenuClick,
            nb::arg("ref"),
            "(private API)")
        .def("menu_check",
            &ImGuiTestContext::MenuCheck,
            nb::arg("ref"),
            "(private API)")
        .def("menu_uncheck",
            &ImGuiTestContext::MenuUncheck,
            nb::arg("ref"),
            "(private API)")
        .def("menu_check_all",
            &ImGuiTestContext::MenuCheckAll,
            nb::arg("ref_parent"),
            "(private API)")
        .def("menu_uncheck_all",
            &ImGuiTestContext::MenuUncheckAll,
            nb::arg("ref_parent"),
            "(private API)")
        .def("combo_click",
            &ImGuiTestContext::ComboClick,
            nb::arg("ref"),
            "(private API)")
        .def("combo_click_all",
            &ImGuiTestContext::ComboClickAll,
            nb::arg("ref"),
            "(private API)")
        .def("table_open_context_menu",
            nb::overload_cast<ImGuiTestRef, int>(&ImGuiTestContext::TableOpenContextMenu),
            nb::arg("ref"), nb::arg("column_n") = -1,
            "(private API)")
        .def("table_click_header",
            nb::overload_cast<ImGuiTestRef, const char *, ImGuiKeyChord>(&ImGuiTestContext::TableClickHeader),
            nb::arg("ref"), nb::arg("label"), nb::arg("key_mods") = 0,
            "(private API)")
        .def("table_set_column_enabled",
            nb::overload_cast<ImGuiTestRef, const char *, bool>(&ImGuiTestContext::TableSetColumnEnabled),
            nb::arg("ref"), nb::arg("label"), nb::arg("enabled"),
            "(private API)")
        .def("table_resize_column",
            nb::overload_cast<ImGuiTestRef, int, float>(&ImGuiTestContext::TableResizeColumn),
            nb::arg("ref"), nb::arg("column_n"), nb::arg("width"),
            "(private API)")
        .def("table_get_sort_specs",
            nb::overload_cast<ImGuiTestRef>(&ImGuiTestContext::TableGetSortSpecs),
            nb::arg("ref"),
            "(private API)",
            nb::rv_policy::reference)
        // #ifdef IMGUI_HAS_DOCK
        //
        .def("dock_clear",
            [](ImGuiTestContext & self, const char * window_name)
            {
                auto DockClear_adapt_variadic_format = [&self](const char * window_name)
                {
                    self.DockClear("%s", window_name);
                };

                DockClear_adapt_variadic_format(window_name);
            },
            nb::arg("window_name"),
            "(private API)")
        .def("dock_into",
            [](ImGuiTestContext & self, ImGuiTestRef src_id, ImGuiTestRef dst_id, const std::optional<const ImGuiDir> & split_dir = std::nullopt, bool is_outer_docking = false, ImGuiTestOpFlags flags = 0)
            {
                auto DockInto_adapt_mutable_param_with_default_value = [&self](ImGuiTestRef src_id, ImGuiTestRef dst_id, const std::optional<const ImGuiDir> & split_dir = std::nullopt, bool is_outer_docking = false, ImGuiTestOpFlags flags = 0)
                {

                    const ImGuiDir& split_dir_or_default = [&]() -> const ImGuiDir {
                        if (split_dir.has_value())
                            return split_dir.value();
                        else
                            return ImGuiDir_None;
                    }();

                    self.DockInto(src_id, dst_id, split_dir_or_default, is_outer_docking, flags);
                };

                DockInto_adapt_mutable_param_with_default_value(src_id, dst_id, split_dir, is_outer_docking, flags);
            },
            nb::arg("src_id"), nb::arg("dst_id"), nb::arg("split_dir").none() = nb::none(), nb::arg("is_outer_docking") = false, nb::arg("flags") = 0,
            "Python bindings defaults:\n    If split_dir is None, then its default value will be: Dir_None\n(private API)")
        .def("undock_node",
            &ImGuiTestContext::UndockNode,
            nb::arg("dock_id"),
            "(private API)")
        .def("undock_window",
            &ImGuiTestContext::UndockWindow,
            nb::arg("window_name"),
            "(private API)")
        .def("window_is_undocked_or_standalone",
            &ImGuiTestContext::WindowIsUndockedOrStandalone,
            nb::arg("window"),
            "(private API)")
        .def("dock_id_is_undocked_or_standalone",
            &ImGuiTestContext::DockIdIsUndockedOrStandalone,
            nb::arg("dock_id"),
            "(private API)")
        .def("dock_node_hide_tab_bar",
            &ImGuiTestContext::DockNodeHideTabBar,
            nb::arg("node"), nb::arg("hidden"),
            "(private API)")
        // #endif
        //
        .def("perf_calc_ref",
            &ImGuiTestContext::PerfCalcRef, "(private API)")
        .def("perf_capture",
            [](ImGuiTestContext & self, std::optional<std::string> category = std::nullopt, std::optional<std::string> test_name = std::nullopt, std::optional<std::string> csv_file = std::nullopt)
            {
                auto PerfCapture_adapt_const_char_pointer_with_default_null = [&self](std::optional<std::string> category = std::nullopt, std::optional<std::string> test_name = std::nullopt, std::optional<std::string> csv_file = std::nullopt)
                {
                    const char * category_adapt_default_null = nullptr;
                    if (category.has_value())
                        category_adapt_default_null = category.value().c_str();
                    const char * test_name_adapt_default_null = nullptr;
                    if (test_name.has_value())
                        test_name_adapt_default_null = test_name.value().c_str();
                    const char * csv_file_adapt_default_null = nullptr;
                    if (csv_file.has_value())
                        csv_file_adapt_default_null = csv_file.value().c_str();

                    self.PerfCapture(category_adapt_default_null, test_name_adapt_default_null, csv_file_adapt_default_null);
                };

                PerfCapture_adapt_const_char_pointer_with_default_null(category, test_name, csv_file);
            },
            nb::arg("category").none() = nb::none(), nb::arg("test_name").none() = nb::none(), nb::arg("csv_file").none() = nb::none(),
            "(private API)")
        .def("_make_aiming_space_over_pos",
            &ImGuiTestContext::_MakeAimingSpaceOverPos,
            nb::arg("viewport"), nb::arg("over_window"), nb::arg("over_pos"),
            "(private API)\n\n Move windows covering 'window' at pos.")
        .def("_foreign_windows_unhide_all",
            &ImGuiTestContext::_ForeignWindowsUnhideAll, "(private API)\n\n FIXME: Aim to remove this system...")
        ;
    ////////////////////    </generated_from:imgui_te_context.h>    ////////////////////


    ////////////////////    <generated_from:imgui_te_internal.h>    ////////////////////
    auto pyClassImGuiTestInfoTask =
        nb::class_<ImGuiTestInfoTask>
            (m, "TestInfoTask", "Query item position/window/state given ID.")
        .def("__init__", [](ImGuiTestInfoTask * self, ImGuiID ID = 0, int FrameCount = -1, const std::optional<const ImGuiTestItemInfo> & Result = std::nullopt)
        {
            new (self) ImGuiTestInfoTask();  // placement new
            auto r_ctor_ = self;
            r_ctor_->ID = ID;
            r_ctor_->FrameCount = FrameCount;
            if (Result.has_value())
                r_ctor_->Result = Result.value();
            else
                r_ctor_->Result = ImGuiTestItemInfo();
        },
        nb::arg("id_") = 0, nb::arg("frame_count") = -1, nb::arg("result").none() = nb::none()
        )
        .def_rw("id_", &ImGuiTestInfoTask::ID, "")
        .def_rw("frame_count", &ImGuiTestInfoTask::FrameCount, "Timestamp of request")
        .def_rw("result", &ImGuiTestInfoTask::Result, "Output")
        ;


    auto pyClassImGuiTestGatherTask =
        nb::class_<ImGuiTestGatherTask>
            (m, "TestGatherTask", "Gather item list in given parent ID.")
        .def("__init__", [](ImGuiTestGatherTask * self, ImGuiID InParentID = 0, int InMaxDepth = 0, short InLayerMask = 0)
        {
            new (self) ImGuiTestGatherTask();  // placement new
            auto r_ctor_ = self;
            r_ctor_->InParentID = InParentID;
            r_ctor_->InMaxDepth = InMaxDepth;
            r_ctor_->InLayerMask = InLayerMask;
        },
        nb::arg("in_parent_id") = 0, nb::arg("in_max_depth") = 0, nb::arg("in_layer_mask") = 0
        )
        .def_rw("in_parent_id", &ImGuiTestGatherTask::InParentID, "")
        .def_rw("in_max_depth", &ImGuiTestGatherTask::InMaxDepth, "")
        .def_rw("in_layer_mask", &ImGuiTestGatherTask::InLayerMask, "")
        .def_rw("out_list", &ImGuiTestGatherTask::OutList, "")
        .def_rw("last_item_info", &ImGuiTestGatherTask::LastItemInfo, "")
        .def("clear",
            &ImGuiTestGatherTask::Clear, "(private API)")
        ;


    auto pyClassImGuiTestFindByLabelTask =
        nb::class_<ImGuiTestFindByLabelTask>
            (m, "TestFindByLabelTask", " Find item ID given a label and a parent id\n Usually used by queries with wildcards such as ItemInfo(\"hello/**/foo/bar\")")
        .def("__init__", [](ImGuiTestFindByLabelTask * self, ImGuiID InPrefixId = 0, int InSuffixDepth = 0, ImGuiID InSuffixLastItemHash = 0, ImGuiItemStatusFlags InFilterItemStatusFlags = 0, ImGuiID OutItemId = 0)
        {
            new (self) ImGuiTestFindByLabelTask();  // placement new
            auto r_ctor_ = self;
            r_ctor_->InPrefixId = InPrefixId;
            r_ctor_->InSuffixDepth = InSuffixDepth;
            r_ctor_->InSuffixLastItemHash = InSuffixLastItemHash;
            r_ctor_->InFilterItemStatusFlags = InFilterItemStatusFlags;
            r_ctor_->OutItemId = OutItemId;
        },
        nb::arg("in_prefix_id") = 0, nb::arg("in_suffix_depth") = 0, nb::arg("in_suffix_last_item_hash") = 0, nb::arg("in_filter_item_status_flags") = 0, nb::arg("out_item_id") = 0
        )
        .def_rw("in_prefix_id", &ImGuiTestFindByLabelTask::InPrefixId, "A known base ID which appears BEFORE the wildcard ID (for \"hello/**/foo/bar\" it would be hash of \"hello\")")
        .def_rw("in_suffix_depth", &ImGuiTestFindByLabelTask::InSuffixDepth, "Number of labels in a path, after unknown base ID (for \"hello/**/foo/bar\" it would be 2)")
        .def_ro("in_suffix", &ImGuiTestFindByLabelTask::InSuffix, "A label string which appears on ID stack after unknown base ID (for \"hello/**/foo/bar\" it would be \"foo/bar\")")
        .def_ro("in_suffix_last_item", &ImGuiTestFindByLabelTask::InSuffixLastItem, "A last label string (for \"hello/**/foo/bar\" it would be \"bar\")")
        .def_rw("in_suffix_last_item_hash", &ImGuiTestFindByLabelTask::InSuffixLastItemHash, "")
        .def_rw("in_filter_item_status_flags", &ImGuiTestFindByLabelTask::InFilterItemStatusFlags, "Flags required for item to be returned")
        .def_rw("out_item_id", &ImGuiTestFindByLabelTask::OutItemId, "Result item ID")
        ;


    auto pyEnumTestInputType =
        nb::enum_<ImGuiTestInputType>(m, "TestInputType", nb::is_arithmetic(), nb::is_flag(), "")
            .value("none", ImGuiTestInputType_None, "")
            .value("key", ImGuiTestInputType_Key, "")
            .value("char", ImGuiTestInputType_Char, "")
            .value("viewport_focus", ImGuiTestInputType_ViewportFocus, "")
            .value("viewport_set_pos", ImGuiTestInputType_ViewportSetPos, "")
            .value("viewport_set_size", ImGuiTestInputType_ViewportSetSize, "")
            .value("viewport_close", ImGuiTestInputType_ViewportClose, "");


    auto pyClassImGuiTestInput =
        nb::class_<ImGuiTestInput>
            (m, "TestInput", "FIXME: May want to strip further now that core imgui is using its own input queue")
        .def("__init__", [](ImGuiTestInput * self, ImGuiTestInputType Type = ImGuiTestInputType_None, const std::optional<const ImGuiKeyChord> & KeyChord = std::nullopt, ImWchar Char = 0, bool Down = false, ImGuiID ViewportId = 0, const std::optional<const ImVec2> & ViewportPosSize = std::nullopt)
        {
            new (self) ImGuiTestInput();  // placement new
            auto r_ctor_ = self;
            r_ctor_->Type = Type;
            if (KeyChord.has_value())
                r_ctor_->KeyChord = KeyChord.value();
            else
                r_ctor_->KeyChord = ImGuiKey_None;
            r_ctor_->Char = Char;
            r_ctor_->Down = Down;
            r_ctor_->ViewportId = ViewportId;
            if (ViewportPosSize.has_value())
                r_ctor_->ViewportPosSize = ViewportPosSize.value();
            else
                r_ctor_->ViewportPosSize = ImVec2();
        },
        nb::arg("type") = ImGuiTestInputType_None, nb::arg("key_chord").none() = nb::none(), nb::arg("char") = 0, nb::arg("down") = false, nb::arg("viewport_id") = 0, nb::arg("viewport_pos_size").none() = nb::none()
        )
        .def_rw("type", &ImGuiTestInput::Type, "")
        .def_rw("key_chord", &ImGuiTestInput::KeyChord, "")
        .def_rw("char", &ImGuiTestInput::Char, "")
        .def_rw("down", &ImGuiTestInput::Down, "")
        .def_rw("viewport_id", &ImGuiTestInput::ViewportId, "")
        .def_rw("viewport_pos_size", &ImGuiTestInput::ViewportPosSize, "")
        .def_static("for_key_chord",
            &ImGuiTestInput::ForKeyChord,
            nb::arg("key_chord"), nb::arg("down"),
            "(private API)")
        .def_static("for_char",
            &ImGuiTestInput::ForChar,
            nb::arg("v"),
            "(private API)")
        .def_static("for_viewport_focus",
            &ImGuiTestInput::ForViewportFocus,
            nb::arg("viewport_id"),
            "(private API)")
        .def_static("for_viewport_set_pos",
            &ImGuiTestInput::ForViewportSetPos,
            nb::arg("viewport_id"), nb::arg("pos"),
            "(private API)")
        .def_static("for_viewport_set_size",
            &ImGuiTestInput::ForViewportSetSize,
            nb::arg("viewport_id"), nb::arg("size"),
            "(private API)")
        .def_static("for_viewport_close",
            &ImGuiTestInput::ForViewportClose,
            nb::arg("viewport_id"),
            "(private API)")
        ;


    auto pyClassImGuiTestInputs =
        nb::class_<ImGuiTestInputs>
            (m, "TestInputs", "")
        .def("__init__", [](ImGuiTestInputs * self, const std::optional<const ImVec2> & MousePosValue = std::nullopt, const std::optional<const ImVec2> & MouseWheel = std::nullopt, ImGuiID MouseHoveredViewport = 0, int MouseButtonsValue = 0x00, bool HostEscDown = false, float HostEscDownDuration = -1.0f)
        {
            new (self) ImGuiTestInputs();  // placement new
            auto r_ctor_ = self;
            if (MousePosValue.has_value())
                r_ctor_->MousePosValue = MousePosValue.value();
            else
                r_ctor_->MousePosValue = ImVec2();
            if (MouseWheel.has_value())
                r_ctor_->MouseWheel = MouseWheel.value();
            else
                r_ctor_->MouseWheel = ImVec2();
            r_ctor_->MouseHoveredViewport = MouseHoveredViewport;
            r_ctor_->MouseButtonsValue = MouseButtonsValue;
            r_ctor_->HostEscDown = HostEscDown;
            r_ctor_->HostEscDownDuration = HostEscDownDuration;
        },
        nb::arg("mouse_pos_value").none() = nb::none(), nb::arg("mouse_wheel").none() = nb::none(), nb::arg("mouse_hovered_viewport") = 0, nb::arg("mouse_buttons_value") = 0x00, nb::arg("host_esc_down") = false, nb::arg("host_esc_down_duration") = -1.0f
        )
        .def_rw("mouse_pos_value", &ImGuiTestInputs::MousePosValue, "Own non-rounded copy of MousePos in order facilitate simulating mouse movement very slow speed and high-framerate")
        .def_rw("mouse_wheel", &ImGuiTestInputs::MouseWheel, "")
        .def_rw("mouse_hovered_viewport", &ImGuiTestInputs::MouseHoveredViewport, "")
        .def_rw("mouse_buttons_value", &ImGuiTestInputs::MouseButtonsValue, "FIXME-TESTS: Use simulated_io.MouseDown[] ?")
        .def_rw("host_esc_down", &ImGuiTestInputs::HostEscDown, "")
        .def_rw("host_esc_down_duration", &ImGuiTestInputs::HostEscDownDuration, "Maintain our own DownDuration for host/backend ESC key so we can abort.")
        ;


    auto pyClassImGuiTestEngine =
        nb::class_<ImGuiTestEngine>
            (m, "TestEngine", "[Internal] Test Engine Context")
        .def_rw("io", &ImGuiTestEngine::IO, "")
        .def_rw("ui_context_target", &ImGuiTestEngine::UiContextTarget, "imgui context for testing")
        .def_rw("ui_context_active", &ImGuiTestEngine::UiContextActive, "imgui context for testing == UiContextTarget or None")
        .def_rw("started", &ImGuiTestEngine::Started, "")
        .def_rw("ui_context_has_hooks", &ImGuiTestEngine::UiContextHasHooks, "")
        .def_rw("batch_start_time", &ImGuiTestEngine::BatchStartTime, "")
        .def_rw("batch_end_time", &ImGuiTestEngine::BatchEndTime, "")
        .def_rw("frame_count", &ImGuiTestEngine::FrameCount, "")
        .def_rw("override_delta_time", &ImGuiTestEngine::OverrideDeltaTime, "Inject custom delta time into imgui context to simulate clock passing faster than wall clock time.")
        .def_rw("test_context", &ImGuiTestEngine::TestContext, "Running test context")
        .def_rw("tests_source_lines_dirty", &ImGuiTestEngine::TestsSourceLinesDirty, "")
        .def_rw("gather_task", &ImGuiTestEngine::GatherTask, "")
        .def_rw("find_by_label_task", &ImGuiTestEngine::FindByLabelTask, "")
        .def_rw("inputs", &ImGuiTestEngine::Inputs, "Inputs")
        .def_rw("abort", &ImGuiTestEngine::Abort, "")
        .def_rw("ui_select_and_scroll_to_test", &ImGuiTestEngine::UiSelectAndScrollToTest, "")
        .def_rw("ui_selected_test", &ImGuiTestEngine::UiSelectedTest, "")
        .def_rw("ui_metrics_open", &ImGuiTestEngine::UiMetricsOpen, "")
        .def_rw("ui_debug_log_open", &ImGuiTestEngine::UiDebugLogOpen, "")
        .def_rw("ui_capture_tool_open", &ImGuiTestEngine::UiCaptureToolOpen, "")
        .def_rw("ui_stack_tool_open", &ImGuiTestEngine::UiStackToolOpen, "")
        .def_rw("ui_perf_tool_open", &ImGuiTestEngine::UiPerfToolOpen, "")
        .def_rw("ui_log_height", &ImGuiTestEngine::UiLogHeight, "")
        .def_rw("perf_ref_delta_time", &ImGuiTestEngine::PerfRefDeltaTime, "")
        .def_rw("post_swap_called", &ImGuiTestEngine::PostSwapCalled, "")
        .def_rw("tool_debug_reboot_ui_context", &ImGuiTestEngine::ToolDebugRebootUiContext, "Completely shutdown and recreate the dear imgui context in place")
        .def_rw("tool_slow_down", &ImGuiTestEngine::ToolSlowDown, "")
        .def_rw("tool_slow_down_ms", &ImGuiTestEngine::ToolSlowDownMs, "")
        .def_rw("backup_config_run_speed", &ImGuiTestEngine::BackupConfigRunSpeed, "")
        .def_rw("backup_config_no_throttle", &ImGuiTestEngine::BackupConfigNoThrottle, "")
        .def(nb::init<>(),
            "Functions")
        ;


    m.def("find_item_info",
        ImGuiTestEngine_FindItemInfo,
        nb::arg("engine"), nb::arg("id_"), nb::arg("debug_id"),
        "(private API)",
        nb::rv_policy::reference);

    m.def("yield_",
        ImGuiTestEngine_Yield,
        nb::arg("engine"),
        "(private API)");

    m.def("set_delta_time",
        ImGuiTestEngine_SetDeltaTime,
        nb::arg("engine"), nb::arg("delta_time"),
        "(private API)");

    m.def("get_frame_count",
        ImGuiTestEngine_GetFrameCount,
        nb::arg("engine"),
        "(private API)");

    m.def("pass_filter",
        ImGuiTestEngine_PassFilter,
        nb::arg("test"), nb::arg("filter"),
        "(private API)");

    m.def("run_test",
        ImGuiTestEngine_RunTest,
        nb::arg("engine"), nb::arg("ctx"), nb::arg("test"), nb::arg("run_flags"),
        "(private API)");

    m.def("bind_im_gui_context",
        ImGuiTestEngine_BindImGuiContext,
        nb::arg("engine"), nb::arg("ui_ctx"),
        "(private API)");

    m.def("unbind_im_gui_context",
        ImGuiTestEngine_UnbindImGuiContext,
        nb::arg("engine"), nb::arg("ui_ctx"),
        "(private API)");

    m.def("reboot_ui_context",
        ImGuiTestEngine_RebootUiContext,
        nb::arg("engine"),
        "(private API)");

    m.def("update_tests_source_lines",
        ImGuiTestEngine_UpdateTestsSourceLines,
        nb::arg("engine"),
        "(private API)");

    m.def("get_status_name",
        ImGuiTestEngine_GetStatusName,
        nb::arg("v"),
        "(private API)",
        nb::rv_policy::reference);

    m.def("get_run_speed_name",
        ImGuiTestEngine_GetRunSpeedName,
        nb::arg("v"),
        "(private API)",
        nb::rv_policy::reference);

    m.def("get_verbose_level_name",
        ImGuiTestEngine_GetVerboseLevelName,
        nb::arg("v"),
        "(private API)",
        nb::rv_policy::reference);
    ////////////////////    </generated_from:imgui_te_internal.h>    ////////////////////


    ////////////////////    <generated_from:imgui_te_ui.h>    ////////////////////
    m.def("show_test_engine_windows",
        [](ImGuiTestEngine * engine, bool p_open) -> bool
        {
            auto ImGuiTestEngine_ShowTestEngineWindows_adapt_modifiable_immutable_to_return = [](ImGuiTestEngine * engine, bool p_open) -> bool
            {
                bool * p_open_adapt_modifiable = & p_open;

                ImGuiTestEngine_ShowTestEngineWindows(engine, p_open_adapt_modifiable);
                return p_open;
            };

            return ImGuiTestEngine_ShowTestEngineWindows_adapt_modifiable_immutable_to_return(engine, p_open);
        },     nb::arg("engine"), nb::arg("p_open"));

    m.def("open_source_file",
        ImGuiTestEngine_OpenSourceFile, nb::arg("engine"), nb::arg("source_filename"), nb::arg("source_line_no"));
    ////////////////////    </generated_from:imgui_te_ui.h>    ////////////////////

    // </litgen_pydef> // Autogenerated code end
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


    nb::implicitly_convertible<ImGuiID, ImGuiTestRef>();
    nb::implicitly_convertible<const char*, ImGuiTestRef>();
}
#endif // #ifdef HELLOIMGUI_WITH_TEST_ENGINE
