// Part of ImGui Bundle - MIT License - Copyright (c) 2022-2024 Pascal Thomet - https://github.com/pthom/imgui_bundle
#ifdef IMGUI_BUNDLE_WITH_IMPLOT
#include <nanobind/nanobind.h>
#include <nanobind/stl/string.h>
#include <nanobind/stl/array.h>
#include <nanobind/stl/optional.h>
#include <nanobind/stl/vector.h>
#include <nanobind/stl/function.h>
#include <nanobind/stl/tuple.h>
#include <nanobind/stl/optional.h>
#include <nanobind/make_iterator.h>
#include <nanobind/ndarray.h>

#include "imgui.h"
#include "implot/implot.h"
#include "implot/implot_internal.h"


namespace nb = nanobind;


nb::class_<ImPlotPoint>* pyClassImPlotPointPtr  = nullptr;


void implot_binding_manual(nb::module_& m);

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// <litgen_glue_code>  // Autogenerated code below! Do not edit!

// </litgen_glue_code> // Autogenerated code end
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

void py_init_module_implot(nb::module_& m)
{
    m.attr("IMPLOT_AUTO") = -1.f;
    m.attr("AUTO") = -1.f;
    m.attr("IMPLOT_AUTO_COL") = ImVec4(0.f, 0.f, 0.f, -1.f);
    m.attr("AUTO_COL") = ImVec4(0.f, 0.f, 0.f, -1.f);
    m.attr("version") = IMPLOT_VERSION;

    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // <litgen_pydef> // Autogenerated code below! Do not edit!
    ////////////////////    <generated_from:implot.h>    ////////////////////
    // #ifndef IMGUI_DISABLE
    //
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //


    auto pyClassBoxedValue =
        nb::class_<BoxedValue>
            (m, "BoxedValue", "")
        .def("__init__", [](BoxedValue * self, double value = double())
        {
            new (self) BoxedValue();  // placement new
            auto r_ctor_ = self;
            r_ctor_->value = value;
        },
        nb::arg("value") = double()
        )
        .def_rw("value", &BoxedValue::value, "")
        ;
    // #endif
    //


    auto pyEnumImAxis_ =
        nb::enum_<ImAxis_>(m, "ImAxis_", nb::is_arithmetic(), nb::is_flag(), "Axis indices. The values assigned may change; NEVER hardcode these.")
            .value("x1", ImAxis_X1, "enabled by default")
            .value("x2", ImAxis_X2, "disabled by default")
            .value("x3", ImAxis_X3, "disabled by default")
            .value("y1", ImAxis_Y1, "enabled by default")
            .value("y2", ImAxis_Y2, "disabled by default")
            .value("y3", ImAxis_Y3, "disabled by default")
            .value("count", ImAxis_COUNT, "bookeeping");


    auto pyEnumFlags_ =
        nb::enum_<ImPlotFlags_>(m, "Flags_", nb::is_arithmetic(), nb::is_flag(), "Options for plots (see BeginPlot).")
            .value("none", ImPlotFlags_None, "default")
            .value("no_title", ImPlotFlags_NoTitle, "the plot title will not be displayed (titles are also hidden if preceeded by double hashes, e.g. \"##MyPlot\")")
            .value("no_legend", ImPlotFlags_NoLegend, "the legend will not be displayed")
            .value("no_mouse_text", ImPlotFlags_NoMouseText, "the mouse position, in plot coordinates, will not be displayed inside of the plot")
            .value("no_inputs", ImPlotFlags_NoInputs, "the user will not be able to interact with the plot")
            .value("no_menus", ImPlotFlags_NoMenus, "the user will not be able to open context menus")
            .value("no_box_select", ImPlotFlags_NoBoxSelect, "the user will not be able to box-select")
            .value("no_frame", ImPlotFlags_NoFrame, "the ImGui frame will not be rendered")
            .value("equal", ImPlotFlags_Equal, "x and y axes pairs will be constrained to have the same units/pixel")
            .value("crosshairs", ImPlotFlags_Crosshairs, "the default mouse cursor will be replaced with a crosshair when hovered")
            .value("canvas_only", ImPlotFlags_CanvasOnly, "");


    auto pyEnumAxisFlags_ =
        nb::enum_<ImPlotAxisFlags_>(m, "AxisFlags_", nb::is_arithmetic(), nb::is_flag(), "Options for plot axes (see SetupAxis).")
            .value("none", ImPlotAxisFlags_None, "default")
            .value("no_label", ImPlotAxisFlags_NoLabel, "the axis label will not be displayed (axis labels are also hidden if the supplied string name is None)")
            .value("no_grid_lines", ImPlotAxisFlags_NoGridLines, "no grid lines will be displayed")
            .value("no_tick_marks", ImPlotAxisFlags_NoTickMarks, "no tick marks will be displayed")
            .value("no_tick_labels", ImPlotAxisFlags_NoTickLabels, "no text labels will be displayed")
            .value("no_initial_fit", ImPlotAxisFlags_NoInitialFit, "axis will not be initially fit to data extents on the first rendered frame")
            .value("no_menus", ImPlotAxisFlags_NoMenus, "the user will not be able to open context menus with right-click")
            .value("no_side_switch", ImPlotAxisFlags_NoSideSwitch, "the user will not be able to switch the axis side by dragging it")
            .value("no_highlight", ImPlotAxisFlags_NoHighlight, "the axis will not have its background highlighted when hovered or held")
            .value("opposite", ImPlotAxisFlags_Opposite, "axis ticks and labels will be rendered on the conventionally opposite side (i.e, right or top)")
            .value("foreground", ImPlotAxisFlags_Foreground, "grid lines will be displayed in the foreground (i.e. on top of data) instead of the background")
            .value("invert", ImPlotAxisFlags_Invert, "the axis will be inverted")
            .value("auto_fit", ImPlotAxisFlags_AutoFit, "axis will be auto-fitting to data extents")
            .value("range_fit", ImPlotAxisFlags_RangeFit, "axis will only fit points if the point is in the visible range of the **orthogonal** axis")
            .value("pan_stretch", ImPlotAxisFlags_PanStretch, "panning in a locked or constrained state will cause the axis to stretch if possible")
            .value("lock_min", ImPlotAxisFlags_LockMin, "the axis minimum value will be locked when panning/zooming")
            .value("lock_max", ImPlotAxisFlags_LockMax, "the axis maximum value will be locked when panning/zooming")
            .value("lock", ImPlotAxisFlags_Lock, "")
            .value("no_decorations", ImPlotAxisFlags_NoDecorations, "")
            .value("aux_default", ImPlotAxisFlags_AuxDefault, "");


    auto pyEnumSubplotFlags_ =
        nb::enum_<ImPlotSubplotFlags_>(m, "SubplotFlags_", nb::is_arithmetic(), nb::is_flag(), "Options for subplots (see BeginSubplot)")
            .value("none", ImPlotSubplotFlags_None, "default")
            .value("no_title", ImPlotSubplotFlags_NoTitle, "the subplot title will not be displayed (titles are also hidden if preceeded by double hashes, e.g. \"##MySubplot\")")
            .value("no_legend", ImPlotSubplotFlags_NoLegend, "the legend will not be displayed (only applicable if ImPlotSubplotFlags_ShareItems is enabled)")
            .value("no_menus", ImPlotSubplotFlags_NoMenus, "the user will not be able to open context menus with right-click")
            .value("no_resize", ImPlotSubplotFlags_NoResize, "resize splitters between subplot cells will be not be provided")
            .value("no_align", ImPlotSubplotFlags_NoAlign, "subplot edges will not be aligned vertically or horizontally")
            .value("share_items", ImPlotSubplotFlags_ShareItems, "items across all subplots will be shared and rendered into a single legend entry")
            .value("link_rows", ImPlotSubplotFlags_LinkRows, "link the y-axis limits of all plots in each row (does not apply to auxiliary axes)")
            .value("link_cols", ImPlotSubplotFlags_LinkCols, "link the x-axis limits of all plots in each column (does not apply to auxiliary axes)")
            .value("link_all_x", ImPlotSubplotFlags_LinkAllX, "link the x-axis limits in every plot in the subplot (does not apply to auxiliary axes)")
            .value("link_all_y", ImPlotSubplotFlags_LinkAllY, "link the y-axis limits in every plot in the subplot (does not apply to auxiliary axes)")
            .value("col_major", ImPlotSubplotFlags_ColMajor, "subplots are added in column major order instead of the default row major order");


    auto pyEnumLegendFlags_ =
        nb::enum_<ImPlotLegendFlags_>(m, "LegendFlags_", nb::is_arithmetic(), nb::is_flag(), "Options for legends (see SetupLegend)")
            .value("none", ImPlotLegendFlags_None, "default")
            .value("no_buttons", ImPlotLegendFlags_NoButtons, "legend icons will not function as hide/show buttons")
            .value("no_highlight_item", ImPlotLegendFlags_NoHighlightItem, "plot items will not be highlighted when their legend entry is hovered")
            .value("no_highlight_axis", ImPlotLegendFlags_NoHighlightAxis, "axes will not be highlighted when legend entries are hovered (only relevant if x/y-axis count > 1)")
            .value("no_menus", ImPlotLegendFlags_NoMenus, "the user will not be able to open context menus with right-click")
            .value("outside", ImPlotLegendFlags_Outside, "legend will be rendered outside of the plot area")
            .value("horizontal", ImPlotLegendFlags_Horizontal, "legend entries will be displayed horizontally")
            .value("sort", ImPlotLegendFlags_Sort, "legend entries will be displayed in alphabetical order");


    auto pyEnumMouseTextFlags_ =
        nb::enum_<ImPlotMouseTextFlags_>(m, "MouseTextFlags_", nb::is_arithmetic(), nb::is_flag(), "Options for mouse hover text (see SetupMouseText)")
            .value("none", ImPlotMouseTextFlags_None, "default")
            .value("no_aux_axes", ImPlotMouseTextFlags_NoAuxAxes, "only show the mouse position for primary axes")
            .value("no_format", ImPlotMouseTextFlags_NoFormat, "axes label formatters won't be used to render text")
            .value("show_always", ImPlotMouseTextFlags_ShowAlways, "always display mouse position even if plot not hovered");


    auto pyEnumDragToolFlags_ =
        nb::enum_<ImPlotDragToolFlags_>(m, "DragToolFlags_", nb::is_arithmetic(), nb::is_flag(), "Options for DragPoint, DragLine, DragRect")
            .value("none", ImPlotDragToolFlags_None, "default")
            .value("no_cursors", ImPlotDragToolFlags_NoCursors, "drag tools won't change cursor icons when hovered or held")
            .value("no_fit", ImPlotDragToolFlags_NoFit, "the drag tool won't be considered for plot fits")
            .value("no_inputs", ImPlotDragToolFlags_NoInputs, "lock the tool from user inputs")
            .value("delayed", ImPlotDragToolFlags_Delayed, "tool rendering will be delayed one frame; useful when applying position-constraints");


    auto pyEnumColormapScaleFlags_ =
        nb::enum_<ImPlotColormapScaleFlags_>(m, "ColormapScaleFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ColormapScale")
            .value("none", ImPlotColormapScaleFlags_None, "default")
            .value("no_label", ImPlotColormapScaleFlags_NoLabel, "the colormap axis label will not be displayed")
            .value("opposite", ImPlotColormapScaleFlags_Opposite, "render the colormap label and tick labels on the opposite side")
            .value("invert", ImPlotColormapScaleFlags_Invert, "invert the colormap bar and axis scale (this only affects rendering; if you only want to reverse the scale mapping, make scale_min > scale_max)");


    auto pyEnumItemFlags_ =
        nb::enum_<ImPlotItemFlags_>(m, "ItemFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for ANY PlotX function")
            .value("none", ImPlotItemFlags_None, "")
            .value("no_legend", ImPlotItemFlags_NoLegend, "the item won't have a legend entry displayed")
            .value("no_fit", ImPlotItemFlags_NoFit, "the item won't be considered for plot fits");


    auto pyEnumLineFlags_ =
        nb::enum_<ImPlotLineFlags_>(m, "LineFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotLine")
            .value("none", ImPlotLineFlags_None, "default")
            .value("segments", ImPlotLineFlags_Segments, "a line segment will be rendered from every two consecutive points")
            .value("loop", ImPlotLineFlags_Loop, "the last and first point will be connected to form a closed loop")
            .value("skip_nan", ImPlotLineFlags_SkipNaN, "NaNs values will be skipped instead of rendered as missing data")
            .value("no_clip", ImPlotLineFlags_NoClip, "markers (if displayed) on the edge of a plot will not be clipped")
            .value("shaded", ImPlotLineFlags_Shaded, "a filled region between the line and horizontal origin will be rendered; use PlotShaded for more advanced cases");


    auto pyEnumScatterFlags_ =
        nb::enum_<ImPlotScatterFlags_>(m, "ScatterFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotScatter")
            .value("none", ImPlotScatterFlags_None, "default")
            .value("no_clip", ImPlotScatterFlags_NoClip, "markers on the edge of a plot will not be clipped");


    auto pyEnumStairsFlags_ =
        nb::enum_<ImPlotStairsFlags_>(m, "StairsFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotStairs")
            .value("none", ImPlotStairsFlags_None, "default")
            .value("pre_step", ImPlotStairsFlags_PreStep, "the y value is continued constantly to the left from every x position, i.e. the interval (x[i-1], x[i]] has the value y[i]")
            .value("shaded", ImPlotStairsFlags_Shaded, "a filled region between the stairs and horizontal origin will be rendered; use PlotShaded for more advanced cases");


    auto pyEnumShadedFlags_ =
        nb::enum_<ImPlotShadedFlags_>(m, "ShadedFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotShaded (placeholder)")
            .value("none", ImPlotShadedFlags_None, "default");


    auto pyEnumBarsFlags_ =
        nb::enum_<ImPlotBarsFlags_>(m, "BarsFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotBars")
            .value("none", ImPlotBarsFlags_None, "default")
            .value("horizontal", ImPlotBarsFlags_Horizontal, "bars will be rendered horizontally on the current y-axis");


    auto pyEnumBarGroupsFlags_ =
        nb::enum_<ImPlotBarGroupsFlags_>(m, "BarGroupsFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotBarGroups")
            .value("none", ImPlotBarGroupsFlags_None, "default")
            .value("horizontal", ImPlotBarGroupsFlags_Horizontal, "bar groups will be rendered horizontally on the current y-axis")
            .value("stacked", ImPlotBarGroupsFlags_Stacked, "items in a group will be stacked on top of each other");


    auto pyEnumErrorBarsFlags_ =
        nb::enum_<ImPlotErrorBarsFlags_>(m, "ErrorBarsFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotErrorBars")
            .value("none", ImPlotErrorBarsFlags_None, "default")
            .value("horizontal", ImPlotErrorBarsFlags_Horizontal, "error bars will be rendered horizontally on the current y-axis");


    auto pyEnumStemsFlags_ =
        nb::enum_<ImPlotStemsFlags_>(m, "StemsFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotStems")
            .value("none", ImPlotStemsFlags_None, "default")
            .value("horizontal", ImPlotStemsFlags_Horizontal, "stems will be rendered horizontally on the current y-axis");


    auto pyEnumInfLinesFlags_ =
        nb::enum_<ImPlotInfLinesFlags_>(m, "InfLinesFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotInfLines")
            .value("none", ImPlotInfLinesFlags_None, "default")
            .value("horizontal", ImPlotInfLinesFlags_Horizontal, "lines will be rendered horizontally on the current y-axis");


    auto pyEnumPieChartFlags_ =
        nb::enum_<ImPlotPieChartFlags_>(m, "PieChartFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotPieChart")
            .value("none", ImPlotPieChartFlags_None, "default")
            .value("normalize", ImPlotPieChartFlags_Normalize, "force normalization of pie chart values (i.e. always make a full circle if sum < 0)")
            .value("ignore_hidden", ImPlotPieChartFlags_IgnoreHidden, "ignore hidden slices when drawing the pie chart (as if they were not there)")
            .value("exploding", ImPlotPieChartFlags_Exploding, "Explode legend-hovered slice");


    auto pyEnumHeatmapFlags_ =
        nb::enum_<ImPlotHeatmapFlags_>(m, "HeatmapFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotHeatmap")
            .value("none", ImPlotHeatmapFlags_None, "default")
            .value("col_major", ImPlotHeatmapFlags_ColMajor, "data will be read in column major order");


    auto pyEnumHistogramFlags_ =
        nb::enum_<ImPlotHistogramFlags_>(m, "HistogramFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotHistogram and PlotHistogram2D")
            .value("none", ImPlotHistogramFlags_None, "default")
            .value("horizontal", ImPlotHistogramFlags_Horizontal, "histogram bars will be rendered horizontally (not supported by PlotHistogram2D)")
            .value("cumulative", ImPlotHistogramFlags_Cumulative, "each bin will contain its count plus the counts of all previous bins (not supported by PlotHistogram2D)")
            .value("density", ImPlotHistogramFlags_Density, "counts will be normalized, i.e. the PDF will be visualized, or the CDF will be visualized if Cumulative is also set")
            .value("no_outliers", ImPlotHistogramFlags_NoOutliers, "exclude values outside the specifed histogram range from the count toward normalizing and cumulative counts")
            .value("col_major", ImPlotHistogramFlags_ColMajor, "data will be read in column major order (not supported by PlotHistogram)");


    auto pyEnumDigitalFlags_ =
        nb::enum_<ImPlotDigitalFlags_>(m, "DigitalFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotDigital (placeholder)")
            .value("none", ImPlotDigitalFlags_None, "default");


    auto pyEnumImageFlags_ =
        nb::enum_<ImPlotImageFlags_>(m, "ImageFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotImage (placeholder)")
            .value("none", ImPlotImageFlags_None, "default");


    auto pyEnumTextFlags_ =
        nb::enum_<ImPlotTextFlags_>(m, "TextFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotText")
            .value("none", ImPlotTextFlags_None, "default")
            .value("vertical", ImPlotTextFlags_Vertical, "text will be rendered vertically");


    auto pyEnumDummyFlags_ =
        nb::enum_<ImPlotDummyFlags_>(m, "DummyFlags_", nb::is_arithmetic(), nb::is_flag(), "Flags for PlotDummy (placeholder)")
            .value("none", ImPlotDummyFlags_None, "default");


    auto pyEnumCond_ =
        nb::enum_<ImPlotCond_>(m, "Cond_", nb::is_arithmetic(), nb::is_flag(), "Represents a condition for SetupAxisLimits etc. (same as ImGuiCond, but we only support a subset of those enums)")
            .value("none", ImPlotCond_None, "No condition (always set the variable), same as _Always")
            .value("always", ImPlotCond_Always, "No condition (always set the variable)")
            .value("once", ImPlotCond_Once, "Set the variable once per runtime session (only the first call will succeed)");


    auto pyEnumCol_ =
        nb::enum_<ImPlotCol_>(m, "Col_", nb::is_arithmetic(), nb::is_flag(), "Plot styling colors.")
            .value("line", ImPlotCol_Line, "plot line/outline color (defaults to next unused color in current colormap)")
            .value("fill", ImPlotCol_Fill, "plot fill color for bars (defaults to the current line color)")
            .value("marker_outline", ImPlotCol_MarkerOutline, "marker outline color (defaults to the current line color)")
            .value("marker_fill", ImPlotCol_MarkerFill, "marker fill color (defaults to the current line color)")
            .value("error_bar", ImPlotCol_ErrorBar, "error bar color (defaults to ImGuiCol_Text)")
            .value("frame_bg", ImPlotCol_FrameBg, "plot frame background color (defaults to ImGuiCol_FrameBg)")
            .value("plot_bg", ImPlotCol_PlotBg, "plot area background color (defaults to ImGuiCol_WindowBg)")
            .value("plot_border", ImPlotCol_PlotBorder, "plot area border color (defaults to ImGuiCol_Border)")
            .value("legend_bg", ImPlotCol_LegendBg, "legend background color (defaults to ImGuiCol_PopupBg)")
            .value("legend_border", ImPlotCol_LegendBorder, "legend border color (defaults to ImPlotCol_PlotBorder)")
            .value("legend_text", ImPlotCol_LegendText, "legend text color (defaults to ImPlotCol_InlayText)")
            .value("title_text", ImPlotCol_TitleText, "plot title text color (defaults to ImGuiCol_Text)")
            .value("inlay_text", ImPlotCol_InlayText, "color of text appearing inside of plots (defaults to ImGuiCol_Text)")
            .value("axis_text", ImPlotCol_AxisText, "axis label and tick lables color (defaults to ImGuiCol_Text)")
            .value("axis_grid", ImPlotCol_AxisGrid, "axis grid color (defaults to 25% ImPlotCol_AxisText)")
            .value("axis_tick", ImPlotCol_AxisTick, "axis tick color (defaults to AxisGrid)")
            .value("axis_bg", ImPlotCol_AxisBg, "background color of axis hover region (defaults to transparent)")
            .value("axis_bg_hovered", ImPlotCol_AxisBgHovered, "axis hover color (defaults to ImGuiCol_ButtonHovered)")
            .value("axis_bg_active", ImPlotCol_AxisBgActive, "axis active color (defaults to ImGuiCol_ButtonActive)")
            .value("selection", ImPlotCol_Selection, "box-selection color (defaults to yellow)")
            .value("crosshairs", ImPlotCol_Crosshairs, "crosshairs color (defaults to ImPlotCol_PlotBorder)")
            .value("count", ImPlotCol_COUNT, "");


    auto pyEnumStyleVar_ =
        nb::enum_<ImPlotStyleVar_>(m, "StyleVar_", nb::is_arithmetic(), nb::is_flag(), "Plot styling variables.")
            .value("line_weight", ImPlotStyleVar_LineWeight, "float,  plot item line weight in pixels")
            .value("marker", ImPlotStyleVar_Marker, "int,    marker specification")
            .value("marker_size", ImPlotStyleVar_MarkerSize, "float,  marker size in pixels (roughly the marker's \"radius\")")
            .value("marker_weight", ImPlotStyleVar_MarkerWeight, "float,  plot outline weight of markers in pixels")
            .value("fill_alpha", ImPlotStyleVar_FillAlpha, "float,  alpha modifier applied to all plot item fills")
            .value("error_bar_size", ImPlotStyleVar_ErrorBarSize, "float,  error bar whisker width in pixels")
            .value("error_bar_weight", ImPlotStyleVar_ErrorBarWeight, "float,  error bar whisker weight in pixels")
            .value("digital_bit_height", ImPlotStyleVar_DigitalBitHeight, "float,  digital channels bit height (at 1) in pixels")
            .value("digital_bit_gap", ImPlotStyleVar_DigitalBitGap, "float,  digital channels bit padding gap in pixels")
            .value("plot_border_size", ImPlotStyleVar_PlotBorderSize, "float,  thickness of border around plot area")
            .value("minor_alpha", ImPlotStyleVar_MinorAlpha, "float,  alpha multiplier applied to minor axis grid lines")
            .value("major_tick_len", ImPlotStyleVar_MajorTickLen, "ImVec2, major tick lengths for X and Y axes")
            .value("minor_tick_len", ImPlotStyleVar_MinorTickLen, "ImVec2, minor tick lengths for X and Y axes")
            .value("major_tick_size", ImPlotStyleVar_MajorTickSize, "ImVec2, line thickness of major ticks")
            .value("minor_tick_size", ImPlotStyleVar_MinorTickSize, "ImVec2, line thickness of minor ticks")
            .value("major_grid_size", ImPlotStyleVar_MajorGridSize, "ImVec2, line thickness of major grid lines")
            .value("minor_grid_size", ImPlotStyleVar_MinorGridSize, "ImVec2, line thickness of minor grid lines")
            .value("plot_padding", ImPlotStyleVar_PlotPadding, "ImVec2, padding between widget frame and plot area, labels, or outside legends (i.e. main padding)")
            .value("label_padding", ImPlotStyleVar_LabelPadding, "ImVec2, padding between axes labels, tick labels, and plot edge")
            .value("legend_padding", ImPlotStyleVar_LegendPadding, "ImVec2, legend padding from plot edges")
            .value("legend_inner_padding", ImPlotStyleVar_LegendInnerPadding, "ImVec2, legend inner padding from legend edges")
            .value("legend_spacing", ImPlotStyleVar_LegendSpacing, "ImVec2, spacing between legend entries")
            .value("mouse_pos_padding", ImPlotStyleVar_MousePosPadding, "ImVec2, padding between plot edge and interior info text")
            .value("annotation_padding", ImPlotStyleVar_AnnotationPadding, "ImVec2, text padding around annotation labels")
            .value("fit_padding", ImPlotStyleVar_FitPadding, "ImVec2, additional fit padding as a percentage of the fit extents (e.g. ImVec2(0.1,0.1) adds 10% to the fit extents of X and Y)")
            .value("plot_default_size", ImPlotStyleVar_PlotDefaultSize, "ImVec2, default size used when ImVec2(0,0) is passed to BeginPlot")
            .value("plot_min_size", ImPlotStyleVar_PlotMinSize, "ImVec2, minimum size plot frame can be when shrunk")
            .value("count", ImPlotStyleVar_COUNT, "");


    auto pyEnumScale_ =
        nb::enum_<ImPlotScale_>(m, "Scale_", nb::is_arithmetic(), nb::is_flag(), "Axis scale")
            .value("linear", ImPlotScale_Linear, "default linear scale")
            .value("time", ImPlotScale_Time, "date/time scale")
            .value("log10", ImPlotScale_Log10, "base 10 logartithmic scale")
            .value("sym_log", ImPlotScale_SymLog, "symmetric log scale");


    auto pyEnumMarker_ =
        nb::enum_<ImPlotMarker_>(m, "Marker_", nb::is_arithmetic(), nb::is_flag(), "Marker specifications.")
            .value("none", ImPlotMarker_None, "no marker")
            .value("circle", ImPlotMarker_Circle, "a circle marker (default)")
            .value("square", ImPlotMarker_Square, "a square maker")
            .value("diamond", ImPlotMarker_Diamond, "a diamond marker")
            .value("up", ImPlotMarker_Up, "an upward-pointing triangle marker")
            .value("down", ImPlotMarker_Down, "an downward-pointing triangle marker")
            .value("left", ImPlotMarker_Left, "an leftward-pointing triangle marker")
            .value("right", ImPlotMarker_Right, "an rightward-pointing triangle marker")
            .value("cross", ImPlotMarker_Cross, "a cross marker (not fillable)")
            .value("plus", ImPlotMarker_Plus, "a plus marker (not fillable)")
            .value("asterisk", ImPlotMarker_Asterisk, "a asterisk marker (not fillable)")
            .value("count", ImPlotMarker_COUNT, "");


    auto pyEnumColormap_ =
        nb::enum_<ImPlotColormap_>(m, "Colormap_", nb::is_arithmetic(), nb::is_flag(), "Built-in colormaps")
            .value("deep", ImPlotColormap_Deep, "a.k.a. seaborn deep             (qual=True,  n=10) (default)")
            .value("dark", ImPlotColormap_Dark, "a.k.a. matplotlib \"Set1\"        (qual=True,  n=9 )")
            .value("pastel", ImPlotColormap_Pastel, "a.k.a. matplotlib \"Pastel1\"     (qual=True,  n=9 )")
            .value("paired", ImPlotColormap_Paired, "a.k.a. matplotlib \"Paired\"      (qual=True,  n=12)")
            .value("viridis", ImPlotColormap_Viridis, "a.k.a. matplotlib \"viridis\"     (qual=False, n=11)")
            .value("plasma", ImPlotColormap_Plasma, "a.k.a. matplotlib \"plasma\"      (qual=False, n=11)")
            .value("hot", ImPlotColormap_Hot, "a.k.a. matplotlib/MATLAB \"hot\"  (qual=False, n=11)")
            .value("cool", ImPlotColormap_Cool, "a.k.a. matplotlib/MATLAB \"cool\" (qual=False, n=11)")
            .value("pink", ImPlotColormap_Pink, "a.k.a. matplotlib/MATLAB \"pink\" (qual=False, n=11)")
            .value("jet", ImPlotColormap_Jet, "a.k.a. MATLAB \"jet\"             (qual=False, n=11)")
            .value("twilight", ImPlotColormap_Twilight, "a.k.a. matplotlib \"twilight\"    (qual=False, n=11)")
            .value("rd_bu", ImPlotColormap_RdBu, "red/blue, Color Brewer          (qual=False, n=11)")
            .value("br_bg", ImPlotColormap_BrBG, "brown/blue-green, Color Brewer  (qual=False, n=11)")
            .value("pi_yg", ImPlotColormap_PiYG, "pink/yellow-green, Color Brewer (qual=False, n=11)")
            .value("spectral", ImPlotColormap_Spectral, "color spectrum, Color Brewer    (qual=False, n=11)")
            .value("greys", ImPlotColormap_Greys, "white/black                     (qual=False, n=2 )");


    auto pyEnumLocation_ =
        nb::enum_<ImPlotLocation_>(m, "Location_", nb::is_arithmetic(), nb::is_flag(), "Used to position items on a plot (e.g. legends, labels, etc.)")
            .value("center", ImPlotLocation_Center, "center-center")
            .value("north", ImPlotLocation_North, "top-center")
            .value("south", ImPlotLocation_South, "bottom-center")
            .value("west", ImPlotLocation_West, "center-left")
            .value("east", ImPlotLocation_East, "center-right")
            .value("north_west", ImPlotLocation_NorthWest, "top-left")
            .value("north_east", ImPlotLocation_NorthEast, "top-right")
            .value("south_west", ImPlotLocation_SouthWest, "bottom-left")
            .value("south_east", ImPlotLocation_SouthEast, "bottom-right");


    auto pyEnumBin_ =
        nb::enum_<ImPlotBin_>(m, "Bin_", nb::is_arithmetic(), nb::is_flag(), "Enums for different automatic histogram binning methods (k = bin count or w = bin width)")
            .value("sqrt", ImPlotBin_Sqrt, "k = sqrt(n)")
            .value("sturges", ImPlotBin_Sturges, "k = 1 + log2(n)")
            .value("rice", ImPlotBin_Rice, "k = 2 * cbrt(n)")
            .value("scott", ImPlotBin_Scott, "w = 3.49 * sigma / cbrt(n)");


    auto pyClassImPlotPoint =
        nb::class_<ImPlotPoint>
            (m, "Point", "")
        .def_rw("x", &ImPlotPoint::x, "")
        .def_rw("y", &ImPlotPoint::y, "")
        .def(nb::init<>())
        .def(nb::init<double, double>(),
            nb::arg("_x"), nb::arg("_y"))
        .def(nb::init<const ImVec2 &>(),
            nb::arg("p"))
        .def("__getitem__",
            nb::overload_cast<size_t>(&ImPlotPoint::operator[]),
            nb::arg("idx"),
            "(private API)",
            nb::rv_policy::reference)
        .def("__getitem__",
            nb::overload_cast<size_t>(&ImPlotPoint::operator[], nb::const_),
            nb::arg("idx"),
            "(private API)")
        ;


    auto pyClassImPlotRange =
        nb::class_<ImPlotRange>
            (m, "Range", "Range defined by a min/max value.")
        .def_rw("min", &ImPlotRange::Min, "")
        .def_rw("max", &ImPlotRange::Max, "")
        .def(nb::init<>())
        .def(nb::init<double, double>(),
            nb::arg("_min"), nb::arg("_max"))
        .def("contains",
            [](const ImPlotRange & self, double value) -> bool
            {
                auto Contains_adapt_force_lambda = [&self](double value) -> bool
                {
                    auto lambda_result = self.Contains(value);
                    return lambda_result;
                };

                return Contains_adapt_force_lambda(value);
            },
            nb::arg("value"),
            "(private API)")
        .def("size",
            &ImPlotRange::Size, "(private API)")
        .def("clamp",
            &ImPlotRange::Clamp,
            nb::arg("value"),
            "(private API)")
        ;


    auto pyClassImPlotRect =
        nb::class_<ImPlotRect>
            (m, "Rect", "Combination of two range limits for X and Y axes. Also an AABB defined by Min()/Max().")
        .def_rw("x", &ImPlotRect::X, "")
        .def_rw("y", &ImPlotRect::Y, "")
        .def(nb::init<>())
        .def(nb::init<double, double, double, double>(),
            nb::arg("x_min"), nb::arg("x_max"), nb::arg("y_min"), nb::arg("y_max"))
        .def("contains",
            [](const ImPlotRect & self, const ImPlotPoint & p) -> bool
            {
                auto Contains_adapt_force_lambda = [&self](const ImPlotPoint & p) -> bool
                {
                    auto lambda_result = self.Contains(p);
                    return lambda_result;
                };

                return Contains_adapt_force_lambda(p);
            },
            nb::arg("p"),
            "(private API)")
        .def("contains",
            [](const ImPlotRect & self, double x, double y) -> bool
            {
                auto Contains_adapt_force_lambda = [&self](double x, double y) -> bool
                {
                    auto lambda_result = self.Contains(x, y);
                    return lambda_result;
                };

                return Contains_adapt_force_lambda(x, y);
            },
            nb::arg("x"), nb::arg("y"),
            "(private API)")
        .def("size",
            &ImPlotRect::Size, "(private API)")
        .def("clamp",
            nb::overload_cast<const ImPlotPoint &>(&ImPlotRect::Clamp),
            nb::arg("p"),
            "(private API)")
        .def("clamp",
            nb::overload_cast<double, double>(&ImPlotRect::Clamp),
            nb::arg("x"), nb::arg("y"),
            "(private API)")
        .def("min",
            &ImPlotRect::Min, "(private API)")
        .def("max",
            &ImPlotRect::Max, "(private API)")
        ;


    auto pyClassImPlotStyle =
        nb::class_<ImPlotStyle>
            (m, "Style", " Plot style structure\n(has support for copy.copy)")
        .def_rw("line_weight", &ImPlotStyle::LineWeight, "= 1,      item line weight in pixels")
        .def_rw("marker", &ImPlotStyle::Marker, "= ImPlotMarker_None, marker specification")
        .def_rw("marker_size", &ImPlotStyle::MarkerSize, "= 4,      marker size in pixels (roughly the marker's \"radius\")")
        .def_rw("marker_weight", &ImPlotStyle::MarkerWeight, "= 1,      outline weight of markers in pixels")
        .def_rw("fill_alpha", &ImPlotStyle::FillAlpha, "= 1,      alpha modifier applied to plot fills")
        .def_rw("error_bar_size", &ImPlotStyle::ErrorBarSize, "= 5,      error bar whisker width in pixels")
        .def_rw("error_bar_weight", &ImPlotStyle::ErrorBarWeight, "= 1.5,    error bar whisker weight in pixels")
        .def_rw("digital_bit_height", &ImPlotStyle::DigitalBitHeight, "= 8,      digital channels bit height (at y = 1.0) in pixels")
        .def_rw("digital_bit_gap", &ImPlotStyle::DigitalBitGap, "= 4,      digital channels bit padding gap in pixels")
        .def_rw("plot_border_size", &ImPlotStyle::PlotBorderSize, "= 1,      line thickness of border around plot area")
        .def_rw("minor_alpha", &ImPlotStyle::MinorAlpha, "= 0.25    alpha multiplier applied to minor axis grid lines")
        .def_rw("major_tick_len", &ImPlotStyle::MajorTickLen, "= 10,10   major tick lengths for X and Y axes")
        .def_rw("minor_tick_len", &ImPlotStyle::MinorTickLen, "= 5,5     minor tick lengths for X and Y axes")
        .def_rw("major_tick_size", &ImPlotStyle::MajorTickSize, "= 1,1     line thickness of major ticks")
        .def_rw("minor_tick_size", &ImPlotStyle::MinorTickSize, "= 1,1     line thickness of minor ticks")
        .def_rw("major_grid_size", &ImPlotStyle::MajorGridSize, "= 1,1     line thickness of major grid lines")
        .def_rw("minor_grid_size", &ImPlotStyle::MinorGridSize, "= 1,1     line thickness of minor grid lines")
        .def_rw("plot_padding", &ImPlotStyle::PlotPadding, "= 10,10   padding between widget frame and plot area, labels, or outside legends (i.e. main padding)")
        .def_rw("label_padding", &ImPlotStyle::LabelPadding, "= 5,5     padding between axes labels, tick labels, and plot edge")
        .def_rw("legend_padding", &ImPlotStyle::LegendPadding, "= 10,10   legend padding from plot edges")
        .def_rw("legend_inner_padding", &ImPlotStyle::LegendInnerPadding, "= 5,5     legend inner padding from legend edges")
        .def_rw("legend_spacing", &ImPlotStyle::LegendSpacing, "= 5,0     spacing between legend entries")
        .def_rw("mouse_pos_padding", &ImPlotStyle::MousePosPadding, "= 10,10   padding between plot edge and interior mouse location text")
        .def_rw("annotation_padding", &ImPlotStyle::AnnotationPadding, "= 2,2     text padding around annotation labels")
        .def_rw("fit_padding", &ImPlotStyle::FitPadding, "= 0,0     additional fit padding as a percentage of the fit extents (e.g. ImVec2(0.1,0.1) adds 10% to the fit extents of X and Y)")
        .def_rw("plot_default_size", &ImPlotStyle::PlotDefaultSize, "= 400,300 default size used when ImVec2(0,0) is passed to BeginPlot")
        .def_rw("plot_min_size", &ImPlotStyle::PlotMinSize, "= 200,150 minimum size plot frame can be when shrunk")
        // #ifdef IMGUI_BUNDLE_PYTHON_API
        //
        .def("color_",
            &ImPlotStyle::Color_,
            nb::arg("idx_color"),
            "Array of styling colors (index from implot.Col_.xxx)",
            nb::rv_policy::reference)
        .def("set_color_",
            &ImPlotStyle::SetColor_,
            nb::arg("idx_color"), nb::arg("color"),
            "Array of styling colors (index from implot.Col_.xxx)")
        // #endif
        //
        .def_rw("colormap", &ImPlotStyle::Colormap, "The current colormap. Set this to either an ImPlotColormap_ enum or an index returned by AddColormap.")
        .def_rw("use_local_time", &ImPlotStyle::UseLocalTime, "= False,  axis labels will be formatted for your timezone when ImPlotAxisFlag_Time is enabled")
        .def_rw("use_iso8601", &ImPlotStyle::UseISO8601, "= False,  dates will be formatted according to ISO 8601 where applicable (e.g. YYYY-MM-DD, YYYY-MM, --MM-DD, etc.)")
        .def_rw("use24_hour_clock", &ImPlotStyle::Use24HourClock, "= False,  times will be formatted using a 24 hour clock")
        .def(nb::init<>())
        .def("__copy__",  [](const ImPlotStyle &self) {
            return ImPlotStyle(self);
        })    ;


    auto pyClassImPlotInputMap =
        nb::class_<ImPlotInputMap>
            (m, "InputMap", "Input mapping structure. Default values listed. See also MapInputDefault, MapInputReverse.")
        .def_rw("pan", &ImPlotInputMap::Pan, "LMB    enables panning when held,")
        .def_rw("pan_mod", &ImPlotInputMap::PanMod, "none   optional modifier that must be held for panning/fitting")
        .def_rw("fit", &ImPlotInputMap::Fit, "LMB    initiates fit when double clicked")
        .def_rw("select", &ImPlotInputMap::Select, "RMB    begins box selection when pressed and confirms selection when released")
        .def_rw("select_cancel", &ImPlotInputMap::SelectCancel, "LMB    cancels active box selection when pressed; cannot be same as Select")
        .def_rw("select_mod", &ImPlotInputMap::SelectMod, "none   optional modifier that must be held for box selection")
        .def_rw("select_horz_mod", &ImPlotInputMap::SelectHorzMod, "Alt    expands active box selection horizontally to plot edge when held")
        .def_rw("select_vert_mod", &ImPlotInputMap::SelectVertMod, "Shift  expands active box selection vertically to plot edge when held")
        .def_rw("menu", &ImPlotInputMap::Menu, "RMB    opens context menus (if enabled) when clicked")
        .def_rw("override_mod", &ImPlotInputMap::OverrideMod, "Ctrl   when held, all input is ignored; used to enable axis/plots as DND sources")
        .def_rw("zoom_mod", &ImPlotInputMap::ZoomMod, "none   optional modifier that must be held for scroll wheel zooming")
        .def_rw("zoom_rate", &ImPlotInputMap::ZoomRate, "0.1   zoom rate for scroll (e.g. 0.1 = 10% plot range every scroll click); make negative to invert")
        .def(nb::init<>())
        ;


    m.def("create_context",
        ImPlot::CreateContext,
        "Creates a new ImPlot context. Call this after ImGui::CreateContext.",
        nb::rv_policy::reference);

    m.def("destroy_context",
        ImPlot::DestroyContext,
        nb::arg("ctx") = nb::none(),
        "Destroys an ImPlot context. Call this before ImGui::DestroyContext. None = destroy current context.");

    m.def("get_current_context",
        ImPlot::GetCurrentContext,
        "Returns the current ImPlot context. None if no context has ben set.",
        nb::rv_policy::reference);

    m.def("set_current_context",
        ImPlot::SetCurrentContext,
        nb::arg("ctx"),
        "Sets the current ImPlot context.");

    m.def("set_imgui_context",
        ImPlot::SetImGuiContext,
        nb::arg("ctx"),
        " Sets the current **ImGui** context. This is ONLY necessary if you are compiling\n ImPlot as a DLL (not recommended) separate from your ImGui compilation. It\n sets the global variable GImGui, which is not shared across DLL boundaries.\n See GImGui documentation in imgui.cpp for more details.");

    m.def("begin_plot",
        [](const char * title_id, const std::optional<const ImVec2> & size = std::nullopt, ImPlotFlags flags = 0) -> bool
        {
            auto BeginPlot_adapt_mutable_param_with_default_value = [](const char * title_id, const std::optional<const ImVec2> & size = std::nullopt, ImPlotFlags flags = 0) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(-1,0);
                }();

                auto lambda_result = ImPlot::BeginPlot(title_id, size_or_default, flags);
                return lambda_result;
            };

            return BeginPlot_adapt_mutable_param_with_default_value(title_id, size, flags);
        },
        nb::arg("title_id"), nb::arg("size").none() = nb::none(), nb::arg("flags") = 0,
        " Starts a 2D plotting context. If this function returns True, EndPlot() MUST\n be called! You are encouraged to use the following convention:\n\n if (BeginPlot(...)) {\n     PlotLine(...);\n     ...\n     EndPlot();\n }\n\n Important notes:\n\n - #title_id must be unique to the current ImGui ID scope. If you need to avoid ID\n   collisions or don't want to display a title in the plot, use double hashes\n   (e.g. \"MyPlot##HiddenIdText\" or \"##NoTitle\").\n - #size is the **frame** size of the plot widget, not the plot area. The default\n   size of plots (i.e. when ImVec2(0,0)) can be modified in your ImPlotStyle.\n\n\nPython bindings defaults:\n    If size is None, then its default value will be: ImVec2(-1,0)");

    m.def("end_plot",
        ImPlot::EndPlot, " Only call EndPlot() if BeginPlot() returns True! Typically called at the end\n of an if statement conditioned on BeginPlot(). See example above.");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //


    auto pyClassSubplotsRowColRatios =
        nb::class_<ImPlot::SubplotsRowColRatios>
            (m, "SubplotsRowColRatios", "")
        .def("__init__", [](ImPlot::SubplotsRowColRatios * self, const std::optional<const std::vector<float>> & row_ratios = std::nullopt, const std::optional<const std::vector<float>> & col_ratios = std::nullopt)
        {
            new (self) ImPlot::SubplotsRowColRatios();  // placement new
            auto r_ctor_ = self;
            if (row_ratios.has_value())
                r_ctor_->row_ratios = row_ratios.value();
            else
                r_ctor_->row_ratios = std::vector<float>();
            if (col_ratios.has_value())
                r_ctor_->col_ratios = col_ratios.value();
            else
                r_ctor_->col_ratios = std::vector<float>();
        },
        nb::arg("row_ratios").none() = nb::none(), nb::arg("col_ratios").none() = nb::none()
        )
        .def_rw("row_ratios", &ImPlot::SubplotsRowColRatios::row_ratios, "")
        .def_rw("col_ratios", &ImPlot::SubplotsRowColRatios::col_ratios, "")
        ;


    m.def("begin_subplots",
        ImPlot::BeginSubplotsWithRatios, nb::arg("title_id"), nb::arg("rows"), nb::arg("cols"), nb::arg("size"), nb::arg("flags") = 0, nb::arg("row_col_ratios") = nb::none());
    // #endif
    //

    m.def("end_subplots",
        ImPlot::EndSubplots, " Only call EndSubplots() if BeginSubplots() returns True! Typically called at the end\n of an if statement conditioned on BeginSublots(). See example above.");

    m.def("setup_axis",
        [](ImAxis axis, std::optional<std::string> label = std::nullopt, ImPlotAxisFlags flags = 0)
        {
            auto SetupAxis_adapt_const_char_pointer_with_default_null = [](ImAxis axis, std::optional<std::string> label = std::nullopt, ImPlotAxisFlags flags = 0)
            {
                const char * label_adapt_default_null = nullptr;
                if (label.has_value())
                    label_adapt_default_null = label.value().c_str();

                ImPlot::SetupAxis(axis, label_adapt_default_null, flags);
            };

            SetupAxis_adapt_const_char_pointer_with_default_null(axis, label, flags);
        },
        nb::arg("axis"), nb::arg("label").none() = nb::none(), nb::arg("flags") = 0,
        "Enables an axis or sets the label and/or flags for an existing axis. Leave #label = None for no label.");

    m.def("setup_axis_limits",
        [](ImAxis axis, double v_min, double v_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
        {
            auto SetupAxisLimits_adapt_mutable_param_with_default_value = [](ImAxis axis, double v_min, double v_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
            {

                const ImPlotCond& cond_or_default = [&]() -> const ImPlotCond {
                    if (cond.has_value())
                        return cond.value();
                    else
                        return ImPlotCond_Once;
                }();

                ImPlot::SetupAxisLimits(axis, v_min, v_max, cond_or_default);
            };

            SetupAxisLimits_adapt_mutable_param_with_default_value(axis, v_min, v_max, cond);
        },
        nb::arg("axis"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("cond").none() = nb::none(),
        " Sets an axis range limits. If ImPlotCond_Always is used, the axes limits will be locked. Inversion with v_min > v_max is not supported; use SetupAxisLimits instead.\n\n\nPython bindings defaults:\n    If cond is None, then its default value will be: Cond_Once");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("setup_axis_links",
        [](ImAxis axis, BoxedValue * link_min, BoxedValue * link_max)
        {
            auto SetupAxisLinks_adapt_force_lambda = [](ImAxis axis, BoxedValue * link_min, BoxedValue * link_max)
            {
                ImPlot::SetupAxisLinks(axis, link_min, link_max);
            };

            SetupAxisLinks_adapt_force_lambda(axis, link_min, link_max);
        },
        nb::arg("axis"), nb::arg("link_min"), nb::arg("link_max"),
        "Links an axis range limits to external values. Use BoxedValue to transmit values (which will be updated after EndPlot).");
    // #endif
    //

    m.def("setup_axis_format",
        nb::overload_cast<ImAxis, const char *>(ImPlot::SetupAxisFormat),
        nb::arg("axis"), nb::arg("fmt"),
        "Sets the format of numeric axis labels via formater specifier (default=\"%g\"). Formated values will be double (i.e. use %f).");

    m.def("setup_axis_scale",
        nb::overload_cast<ImAxis, ImPlotScale>(ImPlot::SetupAxisScale),
        nb::arg("axis"), nb::arg("scale"),
        "Sets an axis' scale using built-in options.");

    m.def("setup_axis_limits_constraints",
        ImPlot::SetupAxisLimitsConstraints,
        nb::arg("axis"), nb::arg("v_min"), nb::arg("v_max"),
        "Sets an axis' limits constraints.");

    m.def("setup_axis_zoom_constraints",
        ImPlot::SetupAxisZoomConstraints,
        nb::arg("axis"), nb::arg("z_min"), nb::arg("z_max"),
        "Sets an axis' zoom constraints.");

    m.def("setup_axes",
        ImPlot::SetupAxes,
        nb::arg("x_label"), nb::arg("y_label"), nb::arg("x_flags") = 0, nb::arg("y_flags") = 0,
        "Sets the label and/or flags for primary X and Y axes (shorthand for two calls to SetupAxis).");

    m.def("setup_axes_limits",
        [](double x_min, double x_max, double y_min, double y_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
        {
            auto SetupAxesLimits_adapt_mutable_param_with_default_value = [](double x_min, double x_max, double y_min, double y_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
            {

                const ImPlotCond& cond_or_default = [&]() -> const ImPlotCond {
                    if (cond.has_value())
                        return cond.value();
                    else
                        return ImPlotCond_Once;
                }();

                ImPlot::SetupAxesLimits(x_min, x_max, y_min, y_max, cond_or_default);
            };

            SetupAxesLimits_adapt_mutable_param_with_default_value(x_min, x_max, y_min, y_max, cond);
        },
        nb::arg("x_min"), nb::arg("x_max"), nb::arg("y_min"), nb::arg("y_max"), nb::arg("cond").none() = nb::none(),
        " Sets the primary X and Y axes range limits. If ImPlotCond_Always is used, the axes limits will be locked (shorthand for two calls to SetupAxisLimits).\n\n\nPython bindings defaults:\n    If cond is None, then its default value will be: Cond_Once");

    m.def("setup_legend",
        ImPlot::SetupLegend,
        nb::arg("location"), nb::arg("flags") = 0,
        "Sets up the plot legend. This can also be called immediately after BeginSubplots when using ImPlotSubplotFlags_ShareItems.");

    m.def("setup_mouse_text",
        ImPlot::SetupMouseText,
        nb::arg("location"), nb::arg("flags") = 0,
        "Set the location of the current plot's mouse position text (default = South|East).");

    m.def("setup_finish",
        ImPlot::SetupFinish, " Explicitly finalize plot setup. Once you call this, you cannot make anymore Setup calls for the current plot!\n Note that calling this function is OPTIONAL; it will be called by the first subsequent setup-locking API call.");

    m.def("set_next_axis_limits",
        [](ImAxis axis, double v_min, double v_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
        {
            auto SetNextAxisLimits_adapt_mutable_param_with_default_value = [](ImAxis axis, double v_min, double v_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
            {

                const ImPlotCond& cond_or_default = [&]() -> const ImPlotCond {
                    if (cond.has_value())
                        return cond.value();
                    else
                        return ImPlotCond_Once;
                }();

                ImPlot::SetNextAxisLimits(axis, v_min, v_max, cond_or_default);
            };

            SetNextAxisLimits_adapt_mutable_param_with_default_value(axis, v_min, v_max, cond);
        },
        nb::arg("axis"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("cond").none() = nb::none(),
        " Sets an upcoming axis range limits. If ImPlotCond_Always is used, the axes limits will be locked.\n\n\nPython bindings defaults:\n    If cond is None, then its default value will be: Cond_Once");
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("set_next_axis_links",
        [](ImAxis axis, BoxedValue * link_min, BoxedValue * link_max)
        {
            auto SetNextAxisLinks_adapt_force_lambda = [](ImAxis axis, BoxedValue * link_min, BoxedValue * link_max)
            {
                ImPlot::SetNextAxisLinks(axis, link_min, link_max);
            };

            SetNextAxisLinks_adapt_force_lambda(axis, link_min, link_max);
        },
        nb::arg("axis"), nb::arg("link_min"), nb::arg("link_max"),
        "Links an upcoming axis range limits to external values. Use BoxedValue to transmit values (which will be updated after EndPlot).");
    // #endif
    //

    m.def("set_next_axis_to_fit",
        ImPlot::SetNextAxisToFit,
        nb::arg("axis"),
        "Set an upcoming axis to auto fit to its data.");

    m.def("set_next_axes_limits",
        [](double x_min, double x_max, double y_min, double y_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
        {
            auto SetNextAxesLimits_adapt_mutable_param_with_default_value = [](double x_min, double x_max, double y_min, double y_max, const std::optional<const ImPlotCond> & cond = std::nullopt)
            {

                const ImPlotCond& cond_or_default = [&]() -> const ImPlotCond {
                    if (cond.has_value())
                        return cond.value();
                    else
                        return ImPlotCond_Once;
                }();

                ImPlot::SetNextAxesLimits(x_min, x_max, y_min, y_max, cond_or_default);
            };

            SetNextAxesLimits_adapt_mutable_param_with_default_value(x_min, x_max, y_min, y_max, cond);
        },
        nb::arg("x_min"), nb::arg("x_max"), nb::arg("y_min"), nb::arg("y_max"), nb::arg("cond").none() = nb::none(),
        " Sets the upcoming primary X and Y axes range limits. If ImPlotCond_Always is used, the axes limits will be locked (shorthand for two calls to SetupAxisLimits).\n\n\nPython bindings defaults:\n    If cond is None, then its default value will be: Cond_Once");

    m.def("set_next_axes_to_fit",
        ImPlot::SetNextAxesToFit, "Sets all upcoming axes to auto fit to their data.");

    m.def("plot_line",
        [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotLineFlags flags = 0, int offset = 0)
        {
            auto PlotLine_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotLineFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotLine(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotLine(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotLine(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotLine(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotLine(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotLine(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotLine(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotLine(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotLine(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotLine(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotLine(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotLine(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotLine_adapt_exclude_params = [&PlotLine_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotLineFlags flags = 0, int offset = 0)
            {
                PlotLine_adapt_c_buffers(label_id, values, xscale, xstart, flags, offset, -1);
            };

            PlotLine_adapt_exclude_params(label_id, values, xscale, xstart, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("xscale") = 1, nb::arg("xstart") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_line",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotLineFlags flags = 0, int offset = 0)
        {
            auto PlotLine_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotLineFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotLine(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotLine(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotLine(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotLine(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotLine(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotLine(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotLine(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotLine(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotLine(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotLine(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotLine(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotLine(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotLine_adapt_exclude_params = [&PlotLine_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotLineFlags flags = 0, int offset = 0)
            {
                PlotLine_adapt_c_buffers(label_id, xs, ys, flags, offset, -1);
            };

            PlotLine_adapt_exclude_params(label_id, xs, ys, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_scatter",
        [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotScatterFlags flags = 0, int offset = 0)
        {
            auto PlotScatter_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotScatterFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotScatter(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotScatter(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotScatter(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotScatter(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotScatter(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotScatter(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotScatter(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotScatter(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotScatter(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotScatter(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotScatter(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotScatter(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotScatter_adapt_exclude_params = [&PlotScatter_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotScatterFlags flags = 0, int offset = 0)
            {
                PlotScatter_adapt_c_buffers(label_id, values, xscale, xstart, flags, offset, -1);
            };

            PlotScatter_adapt_exclude_params(label_id, values, xscale, xstart, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("xscale") = 1, nb::arg("xstart") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_scatter",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotScatterFlags flags = 0, int offset = 0)
        {
            auto PlotScatter_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotScatterFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotScatter(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotScatter(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotScatter(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotScatter(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotScatter(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotScatter(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotScatter(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotScatter(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotScatter(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotScatter(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotScatter(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotScatter(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotScatter_adapt_exclude_params = [&PlotScatter_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotScatterFlags flags = 0, int offset = 0)
            {
                PlotScatter_adapt_c_buffers(label_id, xs, ys, flags, offset, -1);
            };

            PlotScatter_adapt_exclude_params(label_id, xs, ys, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_stairs",
        [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotStairsFlags flags = 0, int offset = 0)
        {
            auto PlotStairs_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotStairsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotStairs(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotStairs(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotStairs(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotStairs(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotStairs(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotStairs(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotStairs(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotStairs(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotStairs(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotStairs(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotStairs(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotStairs(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), xscale, xstart, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotStairs_adapt_exclude_params = [&PlotStairs_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double xscale = 1, double xstart = 0, ImPlotStairsFlags flags = 0, int offset = 0)
            {
                PlotStairs_adapt_c_buffers(label_id, values, xscale, xstart, flags, offset, -1);
            };

            PlotStairs_adapt_exclude_params(label_id, values, xscale, xstart, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("xscale") = 1, nb::arg("xstart") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_stairs",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotStairsFlags flags = 0, int offset = 0)
        {
            auto PlotStairs_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotStairsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotStairs(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotStairs(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotStairs(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotStairs(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotStairs(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotStairs(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotStairs(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotStairs(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotStairs(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotStairs(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotStairs(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotStairs(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotStairs_adapt_exclude_params = [&PlotStairs_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotStairsFlags flags = 0, int offset = 0)
            {
                PlotStairs_adapt_c_buffers(label_id, xs, ys, flags, offset, -1);
            };

            PlotStairs_adapt_exclude_params(label_id, xs, ys, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_shaded",
        [](const char * label_id, const nb::ndarray<> & values, double yref = 0, double xscale = 1, double xstart = 0, ImPlotShadedFlags flags = 0, int offset = 0)
        {
            auto PlotShaded_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double yref = 0, double xscale = 1, double xstart = 0, ImPlotShadedFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotShaded(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotShaded(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotShaded(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotShaded(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotShaded(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotShaded(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotShaded(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotShaded(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotShaded(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotShaded(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotShaded(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotShaded(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), yref, xscale, xstart, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotShaded_adapt_exclude_params = [&PlotShaded_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double yref = 0, double xscale = 1, double xstart = 0, ImPlotShadedFlags flags = 0, int offset = 0)
            {
                PlotShaded_adapt_c_buffers(label_id, values, yref, xscale, xstart, flags, offset, -1);
            };

            PlotShaded_adapt_exclude_params(label_id, values, yref, xscale, xstart, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("yref") = 0, nb::arg("xscale") = 1, nb::arg("xstart") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_shaded",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double yref = 0, ImPlotShadedFlags flags = 0, int offset = 0)
        {
            auto PlotShaded_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double yref = 0, ImPlotShadedFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotShaded(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotShaded(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotShaded(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotShaded(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotShaded(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotShaded(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotShaded(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotShaded(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotShaded(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotShaded(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotShaded(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotShaded(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), yref, flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotShaded_adapt_exclude_params = [&PlotShaded_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double yref = 0, ImPlotShadedFlags flags = 0, int offset = 0)
            {
                PlotShaded_adapt_c_buffers(label_id, xs, ys, yref, flags, offset, -1);
            };

            PlotShaded_adapt_exclude_params(label_id, xs, ys, yref, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("yref") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_shaded",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys1, const nb::ndarray<> & ys2, ImPlotShadedFlags flags = 0, int offset = 0)
        {
            auto PlotShaded_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys1, const nb::ndarray<> & ys2, ImPlotShadedFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys1.ndim() == 1 && ys1.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys1_from_pyarray = ys1.data();
                size_t ys1_count = ys1.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys2.ndim() == 1 && ys2.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys2_from_pyarray = ys2.data();
                size_t ys2_count = ys2.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys2_stride = stride;
                if (ys2_stride == -1)
                    ys2_stride = (int)ys2.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys2 = ys2.dtype().code;
                size_t sizeof_item_ys2 = ys2.dtype().bits / 8;
                char ys2_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys2, sizeof_item_ys2);

                // call the correct template version by casting
                if (ys2_type == 'B')
                    ImPlot::PlotShaded(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys1_from_pyarray), static_cast<const uint8_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'b')
                    ImPlot::PlotShaded(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys1_from_pyarray), static_cast<const int8_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'H')
                    ImPlot::PlotShaded(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys1_from_pyarray), static_cast<const uint16_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'h')
                    ImPlot::PlotShaded(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys1_from_pyarray), static_cast<const int16_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'I')
                    ImPlot::PlotShaded(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys1_from_pyarray), static_cast<const uint32_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'i')
                    ImPlot::PlotShaded(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys1_from_pyarray), static_cast<const int32_t *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'L')
                    ImPlot::PlotShaded(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys1_from_pyarray), static_cast<const np_uint_l *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'l')
                    ImPlot::PlotShaded(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys1_from_pyarray), static_cast<const np_int_l *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'f')
                    ImPlot::PlotShaded(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys1_from_pyarray), static_cast<const float *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'd')
                    ImPlot::PlotShaded(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys1_from_pyarray), static_cast<const double *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'g')
                    ImPlot::PlotShaded(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys1_from_pyarray), static_cast<const long double *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                else if (ys2_type == 'q')
                    ImPlot::PlotShaded(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys1_from_pyarray), static_cast<const long long *>(ys2_from_pyarray), static_cast<int>(ys2_count), flags, offset, ys2_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys2_type + "') for param ys2");
            };
            auto PlotShaded_adapt_exclude_params = [&PlotShaded_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys1, const nb::ndarray<> & ys2, ImPlotShadedFlags flags = 0, int offset = 0)
            {
                PlotShaded_adapt_c_buffers(label_id, xs, ys1, ys2, flags, offset, -1);
            };

            PlotShaded_adapt_exclude_params(label_id, xs, ys1, ys2, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys1"), nb::arg("ys2"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_bars",
        [](const char * label_id, const nb::ndarray<> & values, double bar_size = 0.67, double shift = 0, ImPlotBarsFlags flags = 0, int offset = 0)
        {
            auto PlotBars_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double bar_size = 0.67, double shift = 0, ImPlotBarsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotBars(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotBars(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotBars(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotBars(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotBars(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotBars(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotBars(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotBars(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotBars(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotBars(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotBars(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotBars(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), bar_size, shift, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotBars_adapt_exclude_params = [&PlotBars_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double bar_size = 0.67, double shift = 0, ImPlotBarsFlags flags = 0, int offset = 0)
            {
                PlotBars_adapt_c_buffers(label_id, values, bar_size, shift, flags, offset, -1);
            };

            PlotBars_adapt_exclude_params(label_id, values, bar_size, shift, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("bar_size") = 0.67, nb::arg("shift") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_bars",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double bar_size, ImPlotBarsFlags flags = 0, int offset = 0)
        {
            auto PlotBars_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double bar_size, ImPlotBarsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotBars(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotBars(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotBars(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotBars(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotBars(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotBars(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotBars(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotBars(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotBars(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotBars(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotBars(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotBars(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), bar_size, flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotBars_adapt_exclude_params = [&PlotBars_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double bar_size, ImPlotBarsFlags flags = 0, int offset = 0)
            {
                PlotBars_adapt_c_buffers(label_id, xs, ys, bar_size, flags, offset, -1);
            };

            PlotBars_adapt_exclude_params(label_id, xs, ys, bar_size, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("bar_size"), nb::arg("flags") = 0, nb::arg("offset") = 0);
    // #ifdef IMGUI_BUNDLE_PYTHON_API
    //

    m.def("plot_bar_groups",
        [](const std::vector<std::string> & label_ids, const nb::ndarray<> & values, double group_size = 0.67, double shift = 0, ImPlotBarGroupsFlags flags = 0)
        {
            auto PlotBarGroups_adapt_c_buffers = [](const std::vector<std::string> & label_ids, const nb::ndarray<> & values, double group_size = 0.67, double shift = 0, ImPlotBarGroupsFlags flags = 0)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'b')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'H')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'h')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'I')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'i')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'L')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'l')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'f')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'd')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'g')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                else if (values_type == 'q')
                    ImPlot::PlotBarGroups(label_ids, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), group_size, shift, flags);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };

            PlotBarGroups_adapt_c_buffers(label_ids, values, group_size, shift, flags);
        },
        nb::arg("label_ids"), nb::arg("values"), nb::arg("group_size") = 0.67, nb::arg("shift") = 0, nb::arg("flags") = 0,
        " Plots a group of bars.\n - values should be a **1 dimension** numpy array of values.\n - label_ids should be a list of strings corresponding to bars labels");
    // #endif
    //

    m.def("plot_error_bars",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & err, ImPlotErrorBarsFlags flags = 0, int offset = 0)
        {
            auto PlotErrorBars_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & err, ImPlotErrorBarsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (err.ndim() == 1 && err.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * err_from_pyarray = err.data();
                size_t err_count = err.shape(0);

                // process stride default value (which was a sizeof in C++)
                int err_stride = stride;
                if (err_stride == -1)
                    err_stride = (int)err.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_err = err.dtype().code;
                size_t sizeof_item_err = err.dtype().bits / 8;
                char err_type = _nanobind_buffer_type_to_letter_code(dtype_code_err, sizeof_item_err);

                // call the correct template version by casting
                if (err_type == 'B')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<const uint8_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'b')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<const int8_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'H')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<const uint16_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'h')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<const int16_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'I')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<const uint32_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'i')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<const int32_t *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'L')
                    ImPlot::PlotErrorBars(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<const np_uint_l *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'l')
                    ImPlot::PlotErrorBars(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<const np_int_l *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'f')
                    ImPlot::PlotErrorBars(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<const float *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'd')
                    ImPlot::PlotErrorBars(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<const double *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'g')
                    ImPlot::PlotErrorBars(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<const long double *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                else if (err_type == 'q')
                    ImPlot::PlotErrorBars(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<const long long *>(err_from_pyarray), static_cast<int>(err_count), flags, offset, err_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + err_type + "') for param err");
            };
            auto PlotErrorBars_adapt_exclude_params = [&PlotErrorBars_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & err, ImPlotErrorBarsFlags flags = 0, int offset = 0)
            {
                PlotErrorBars_adapt_c_buffers(label_id, xs, ys, err, flags, offset, -1);
            };

            PlotErrorBars_adapt_exclude_params(label_id, xs, ys, err, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("err"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_error_bars",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & neg, const nb::ndarray<> & pos, ImPlotErrorBarsFlags flags = 0, int offset = 0)
        {
            auto PlotErrorBars_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & neg, const nb::ndarray<> & pos, ImPlotErrorBarsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (neg.ndim() == 1 && neg.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * neg_from_pyarray = neg.data();
                size_t neg_count = neg.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (pos.ndim() == 1 && pos.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * pos_from_pyarray = pos.data();
                size_t pos_count = pos.shape(0);

                // process stride default value (which was a sizeof in C++)
                int pos_stride = stride;
                if (pos_stride == -1)
                    pos_stride = (int)pos.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_pos = pos.dtype().code;
                size_t sizeof_item_pos = pos.dtype().bits / 8;
                char pos_type = _nanobind_buffer_type_to_letter_code(dtype_code_pos, sizeof_item_pos);

                // call the correct template version by casting
                if (pos_type == 'B')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<const uint8_t *>(neg_from_pyarray), static_cast<const uint8_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'b')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<const int8_t *>(neg_from_pyarray), static_cast<const int8_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'H')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<const uint16_t *>(neg_from_pyarray), static_cast<const uint16_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'h')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<const int16_t *>(neg_from_pyarray), static_cast<const int16_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'I')
                    ImPlot::PlotErrorBars(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<const uint32_t *>(neg_from_pyarray), static_cast<const uint32_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'i')
                    ImPlot::PlotErrorBars(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<const int32_t *>(neg_from_pyarray), static_cast<const int32_t *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'L')
                    ImPlot::PlotErrorBars(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<const np_uint_l *>(neg_from_pyarray), static_cast<const np_uint_l *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'l')
                    ImPlot::PlotErrorBars(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<const np_int_l *>(neg_from_pyarray), static_cast<const np_int_l *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'f')
                    ImPlot::PlotErrorBars(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<const float *>(neg_from_pyarray), static_cast<const float *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'd')
                    ImPlot::PlotErrorBars(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<const double *>(neg_from_pyarray), static_cast<const double *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'g')
                    ImPlot::PlotErrorBars(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<const long double *>(neg_from_pyarray), static_cast<const long double *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                else if (pos_type == 'q')
                    ImPlot::PlotErrorBars(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<const long long *>(neg_from_pyarray), static_cast<const long long *>(pos_from_pyarray), static_cast<int>(pos_count), flags, offset, pos_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + pos_type + "') for param pos");
            };
            auto PlotErrorBars_adapt_exclude_params = [&PlotErrorBars_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, const nb::ndarray<> & neg, const nb::ndarray<> & pos, ImPlotErrorBarsFlags flags = 0, int offset = 0)
            {
                PlotErrorBars_adapt_c_buffers(label_id, xs, ys, neg, pos, flags, offset, -1);
            };

            PlotErrorBars_adapt_exclude_params(label_id, xs, ys, neg, pos, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("neg"), nb::arg("pos"), nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_stems",
        [](const char * label_id, const nb::ndarray<> & values, double ref = 0, double scale = 1, double start = 0, ImPlotStemsFlags flags = 0, int offset = 0)
        {
            auto PlotStems_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double ref = 0, double scale = 1, double start = 0, ImPlotStemsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotStems(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotStems(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotStems(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotStems(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotStems(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotStems(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotStems(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotStems(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotStems(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotStems(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotStems(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotStems(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), ref, scale, start, flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotStems_adapt_exclude_params = [&PlotStems_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, double ref = 0, double scale = 1, double start = 0, ImPlotStemsFlags flags = 0, int offset = 0)
            {
                PlotStems_adapt_c_buffers(label_id, values, ref, scale, start, flags, offset, -1);
            };

            PlotStems_adapt_exclude_params(label_id, values, ref, scale, start, flags, offset);
        },     nb::arg("label_id"), nb::arg("values"), nb::arg("ref") = 0, nb::arg("scale") = 1, nb::arg("start") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_stems",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double ref = 0, ImPlotStemsFlags flags = 0, int offset = 0)
        {
            auto PlotStems_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double ref = 0, ImPlotStemsFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotStems(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotStems(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotStems(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotStems(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotStems(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotStems(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotStems(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotStems(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotStems(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotStems(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotStems(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotStems(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), ref, flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotStems_adapt_exclude_params = [&PlotStems_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, double ref = 0, ImPlotStemsFlags flags = 0, int offset = 0)
            {
                PlotStems_adapt_c_buffers(label_id, xs, ys, ref, flags, offset, -1);
            };

            PlotStems_adapt_exclude_params(label_id, xs, ys, ref, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("ref") = 0, nb::arg("flags") = 0, nb::arg("offset") = 0);

    m.def("plot_inf_lines",
        [](const char * label_id, const nb::ndarray<> & values, ImPlotInfLinesFlags flags = 0, int offset = 0)
        {
            auto PlotInfLines_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, ImPlotInfLinesFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                // process stride default value (which was a sizeof in C++)
                int values_stride = stride;
                if (values_stride == -1)
                    values_stride = (int)values.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotInfLines(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'b')
                    ImPlot::PlotInfLines(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'H')
                    ImPlot::PlotInfLines(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'h')
                    ImPlot::PlotInfLines(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'I')
                    ImPlot::PlotInfLines(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'i')
                    ImPlot::PlotInfLines(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'L')
                    ImPlot::PlotInfLines(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'l')
                    ImPlot::PlotInfLines(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'f')
                    ImPlot::PlotInfLines(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'd')
                    ImPlot::PlotInfLines(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'g')
                    ImPlot::PlotInfLines(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                else if (values_type == 'q')
                    ImPlot::PlotInfLines(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), flags, offset, values_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotInfLines_adapt_exclude_params = [&PlotInfLines_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, ImPlotInfLinesFlags flags = 0, int offset = 0)
            {
                PlotInfLines_adapt_c_buffers(label_id, values, flags, offset, -1);
            };

            PlotInfLines_adapt_exclude_params(label_id, values, flags, offset);
        },
        nb::arg("label_id"), nb::arg("values"), nb::arg("flags") = 0, nb::arg("offset") = 0,
        "Plots infinite vertical or horizontal lines (e.g. for references or asymptotes).");

    m.def("plot_pie_chart",
        [](const std::vector<std::string> & label_ids, const nb::ndarray<> & values, double x, double y, double radius, const char * label_fmt = "%.1f", double angle0 = 90, ImPlotPieChartFlags flags = 0)
        {
            auto PlotPieChart_adapt_c_buffers = [](const char * const label_ids[], const nb::ndarray<> & values, double x, double y, double radius, const char * label_fmt = "%.1f", double angle0 = 90, ImPlotPieChartFlags flags = 0)
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    ImPlot::PlotPieChart(label_ids, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'b')
                    ImPlot::PlotPieChart(label_ids, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'H')
                    ImPlot::PlotPieChart(label_ids, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'h')
                    ImPlot::PlotPieChart(label_ids, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'I')
                    ImPlot::PlotPieChart(label_ids, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'i')
                    ImPlot::PlotPieChart(label_ids, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'L')
                    ImPlot::PlotPieChart(label_ids, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'l')
                    ImPlot::PlotPieChart(label_ids, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'f')
                    ImPlot::PlotPieChart(label_ids, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'd')
                    ImPlot::PlotPieChart(label_ids, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'g')
                    ImPlot::PlotPieChart(label_ids, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                else if (values_type == 'q')
                    ImPlot::PlotPieChart(label_ids, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), x, y, radius, label_fmt, angle0, flags);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotPieChart_adapt_c_string_list_no_count = [&PlotPieChart_adapt_c_buffers](const std::vector<std::string> & label_ids, const nb::ndarray<> & values, double x, double y, double radius, const char * label_fmt = "%.1f", double angle0 = 90, ImPlotPieChartFlags flags = 0)
            {
                std::vector<const char *> label_ids_ptrs;
                label_ids_ptrs.reserve(label_ids.size());
                for (const auto& v: label_ids)
                    label_ids_ptrs.push_back(v.c_str());

                PlotPieChart_adapt_c_buffers(label_ids_ptrs.data(), values, x, y, radius, label_fmt, angle0, flags);
            };

            PlotPieChart_adapt_c_string_list_no_count(label_ids, values, x, y, radius, label_fmt, angle0, flags);
        },     nb::arg("label_ids"), nb::arg("values"), nb::arg("x"), nb::arg("y"), nb::arg("radius"), nb::arg("label_fmt") = "%.1f", nb::arg("angle0") = 90, nb::arg("flags") = 0);

    m.def("plot_histogram",
        [](const char * label_id, const nb::ndarray<> & values, int bins = ImPlotBin_Sturges, double bar_scale = 1.0, const std::optional<const ImPlotRange> & range = std::nullopt, ImPlotHistogramFlags flags = 0) -> double
        {
            auto PlotHistogram_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, int bins = ImPlotBin_Sturges, double bar_scale = 1.0, ImPlotRange range = ImPlotRange(), ImPlotHistogramFlags flags = 0) -> double
            {
                // Check if the array is 1D and C-contiguous
                if (! (values.ndim() == 1 && values.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * values_from_pyarray = values.data();
                size_t values_count = values.shape(0);

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_values = values.dtype().code;
                size_t sizeof_item_values = values.dtype().bits / 8;
                char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_values, sizeof_item_values);

                // call the correct template version by casting
                if (values_type == 'B')
                    return ImPlot::PlotHistogram(label_id, static_cast<const uint8_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'b')
                    return ImPlot::PlotHistogram(label_id, static_cast<const int8_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'H')
                    return ImPlot::PlotHistogram(label_id, static_cast<const uint16_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'h')
                    return ImPlot::PlotHistogram(label_id, static_cast<const int16_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'I')
                    return ImPlot::PlotHistogram(label_id, static_cast<const uint32_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'i')
                    return ImPlot::PlotHistogram(label_id, static_cast<const int32_t *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'L')
                    return ImPlot::PlotHistogram(label_id, static_cast<const np_uint_l *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'l')
                    return ImPlot::PlotHistogram(label_id, static_cast<const np_int_l *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'f')
                    return ImPlot::PlotHistogram(label_id, static_cast<const float *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'd')
                    return ImPlot::PlotHistogram(label_id, static_cast<const double *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'g')
                    return ImPlot::PlotHistogram(label_id, static_cast<const long double *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                else if (values_type == 'q')
                    return ImPlot::PlotHistogram(label_id, static_cast<const long long *>(values_from_pyarray), static_cast<int>(values_count), bins, bar_scale, range, flags);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
            };
            auto PlotHistogram_adapt_mutable_param_with_default_value = [&PlotHistogram_adapt_c_buffers](const char * label_id, const nb::ndarray<> & values, int bins = ImPlotBin_Sturges, double bar_scale = 1.0, const std::optional<const ImPlotRange> & range = std::nullopt, ImPlotHistogramFlags flags = 0) -> double
            {

                const ImPlotRange& range_or_default = [&]() -> const ImPlotRange {
                    if (range.has_value())
                        return range.value();
                    else
                        return ImPlotRange();
                }();

                auto lambda_result = PlotHistogram_adapt_c_buffers(label_id, values, bins, bar_scale, range_or_default, flags);
                return lambda_result;
            };

            return PlotHistogram_adapt_mutable_param_with_default_value(label_id, values, bins, bar_scale, range, flags);
        },
        nb::arg("label_id"), nb::arg("values"), nb::arg("bins") = ImPlotBin_Sturges, nb::arg("bar_scale") = 1.0, nb::arg("range").none() = nb::none(), nb::arg("flags") = 0,
        " Plots a horizontal histogram. #bins can be a positive integer or an ImPlotBin_ method. If #range is left unspecified, the min/max of #values will be used as the range.\n Otherwise, outlier values outside of the range are not binned. The largest bin count or density is returned.\n\n\nPython bindings defaults:\n    If range is None, then its default value will be: Range()");

    m.def("plot_histogram_2d",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, int x_bins = ImPlotBin_Sturges, int y_bins = ImPlotBin_Sturges, const std::optional<const ImPlotRect> & range = std::nullopt, ImPlotHistogramFlags flags = 0) -> double
        {
            auto PlotHistogram2D_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, int x_bins = ImPlotBin_Sturges, int y_bins = ImPlotBin_Sturges, ImPlotRect range = ImPlotRect(), ImPlotHistogramFlags flags = 0) -> double
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'b')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'H')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'h')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'I')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'i')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'L')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'l')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'f')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'd')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'g')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                else if (ys_type == 'q')
                    return ImPlot::PlotHistogram2D(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), x_bins, y_bins, range, flags);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotHistogram2D_adapt_mutable_param_with_default_value = [&PlotHistogram2D_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, int x_bins = ImPlotBin_Sturges, int y_bins = ImPlotBin_Sturges, const std::optional<const ImPlotRect> & range = std::nullopt, ImPlotHistogramFlags flags = 0) -> double
            {

                const ImPlotRect& range_or_default = [&]() -> const ImPlotRect {
                    if (range.has_value())
                        return range.value();
                    else
                        return ImPlotRect();
                }();

                auto lambda_result = PlotHistogram2D_adapt_c_buffers(label_id, xs, ys, x_bins, y_bins, range_or_default, flags);
                return lambda_result;
            };

            return PlotHistogram2D_adapt_mutable_param_with_default_value(label_id, xs, ys, x_bins, y_bins, range, flags);
        },
        nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("x_bins") = ImPlotBin_Sturges, nb::arg("y_bins") = ImPlotBin_Sturges, nb::arg("range").none() = nb::none(), nb::arg("flags") = 0,
        " Plots two dimensional, bivariate histogram as a heatmap. #x_bins and #y_bins can be a positive integer or an ImPlotBin. If #range is left unspecified, the min/max of\n #xs an #ys will be used as the ranges. Otherwise, outlier values outside of range are not binned. The largest bin count or density is returned.\n\n\nPython bindings defaults:\n    If range is None, then its default value will be: Rect()");

    m.def("plot_digital",
        [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotDigitalFlags flags = 0, int offset = 0)
        {
            auto PlotDigital_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotDigitalFlags flags = 0, int offset = 0, int stride = -1)
            {
                // Check if the array is 1D and C-contiguous
                if (! (xs.ndim() == 1 && xs.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * xs_from_pyarray = xs.data();
                size_t xs_count = xs.shape(0);

                // Check if the array is 1D and C-contiguous
                if (! (ys.ndim() == 1 && ys.stride(0) == 1))
                    throw std::runtime_error("The array must be 1D and contiguous");

                // convert nb::ndarray to C standard buffer (const)
                const void * ys_from_pyarray = ys.data();
                size_t ys_count = ys.shape(0);

                // process stride default value (which was a sizeof in C++)
                int ys_stride = stride;
                if (ys_stride == -1)
                    ys_stride = (int)ys.itemsize();

                using np_uint_l = uint64_t;
                using np_int_l = int64_t;

                // Define a lambda to compute the letter code for the buffer type
                auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                {
                    #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                        const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                            {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                            {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                            {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                        }};
                    #undef DCODE
                    for (const auto& [code_val, size, letter] : mappings)
                        if (code_val == dtype_code && size == sizeof_item)
                            return letter;
                    throw std::runtime_error("Unsupported dtype");
                };

                // Compute the letter code for the buffer type
                uint8_t dtype_code_ys = ys.dtype().code;
                size_t sizeof_item_ys = ys.dtype().bits / 8;
                char ys_type = _nanobind_buffer_type_to_letter_code(dtype_code_ys, sizeof_item_ys);

                // call the correct template version by casting
                if (ys_type == 'B')
                    ImPlot::PlotDigital(label_id, static_cast<const uint8_t *>(xs_from_pyarray), static_cast<const uint8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'b')
                    ImPlot::PlotDigital(label_id, static_cast<const int8_t *>(xs_from_pyarray), static_cast<const int8_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'H')
                    ImPlot::PlotDigital(label_id, static_cast<const uint16_t *>(xs_from_pyarray), static_cast<const uint16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'h')
                    ImPlot::PlotDigital(label_id, static_cast<const int16_t *>(xs_from_pyarray), static_cast<const int16_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'I')
                    ImPlot::PlotDigital(label_id, static_cast<const uint32_t *>(xs_from_pyarray), static_cast<const uint32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'i')
                    ImPlot::PlotDigital(label_id, static_cast<const int32_t *>(xs_from_pyarray), static_cast<const int32_t *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'L')
                    ImPlot::PlotDigital(label_id, static_cast<const np_uint_l *>(xs_from_pyarray), static_cast<const np_uint_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'l')
                    ImPlot::PlotDigital(label_id, static_cast<const np_int_l *>(xs_from_pyarray), static_cast<const np_int_l *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'f')
                    ImPlot::PlotDigital(label_id, static_cast<const float *>(xs_from_pyarray), static_cast<const float *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'd')
                    ImPlot::PlotDigital(label_id, static_cast<const double *>(xs_from_pyarray), static_cast<const double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'g')
                    ImPlot::PlotDigital(label_id, static_cast<const long double *>(xs_from_pyarray), static_cast<const long double *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                else if (ys_type == 'q')
                    ImPlot::PlotDigital(label_id, static_cast<const long long *>(xs_from_pyarray), static_cast<const long long *>(ys_from_pyarray), static_cast<int>(ys_count), flags, offset, ys_stride);
                // If we reach this point, the array type is not supported!
                else
                    throw std::runtime_error(std::string("Bad array type ('") + ys_type + "') for param ys");
            };
            auto PlotDigital_adapt_exclude_params = [&PlotDigital_adapt_c_buffers](const char * label_id, const nb::ndarray<> & xs, const nb::ndarray<> & ys, ImPlotDigitalFlags flags = 0, int offset = 0)
            {
                PlotDigital_adapt_c_buffers(label_id, xs, ys, flags, offset, -1);
            };

            PlotDigital_adapt_exclude_params(label_id, xs, ys, flags, offset);
        },     nb::arg("label_id"), nb::arg("xs"), nb::arg("ys"), nb::arg("flags") = 0, nb::arg("offset") = 0);
    // #ifdef IMGUI_HAS_TEXTURES
    //

    m.def("plot_image",
        [](const char * label_id, ImTextureRef tex_ref, const ImPlotPoint & bounds_min, const ImPlotPoint & bounds_max, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt, ImPlotImageFlags flags = 0)
        {
            auto PlotImage_adapt_mutable_param_with_default_value = [](const char * label_id, ImTextureRef tex_ref, const ImPlotPoint & bounds_min, const ImPlotPoint & bounds_max, const std::optional<const ImVec2> & uv0 = std::nullopt, const std::optional<const ImVec2> & uv1 = std::nullopt, const std::optional<const ImVec4> & tint_col = std::nullopt, ImPlotImageFlags flags = 0)
            {

                const ImVec2& uv0_or_default = [&]() -> const ImVec2 {
                    if (uv0.has_value())
                        return uv0.value();
                    else
                        return ImVec2(0, 0);
                }();

                const ImVec2& uv1_or_default = [&]() -> const ImVec2 {
                    if (uv1.has_value())
                        return uv1.value();
                    else
                        return ImVec2(1, 1);
                }();

                const ImVec4& tint_col_or_default = [&]() -> const ImVec4 {
                    if (tint_col.has_value())
                        return tint_col.value();
                    else
                        return ImVec4(1, 1, 1, 1);
                }();

                ImPlot::PlotImage(label_id, tex_ref, bounds_min, bounds_max, uv0_or_default, uv1_or_default, tint_col_or_default, flags);
            };

            PlotImage_adapt_mutable_param_with_default_value(label_id, tex_ref, bounds_min, bounds_max, uv0, uv1, tint_col, flags);
        },
        nb::arg("label_id"), nb::arg("tex_ref"), nb::arg("bounds_min"), nb::arg("bounds_max"), nb::arg("uv0").none() = nb::none(), nb::arg("uv1").none() = nb::none(), nb::arg("tint_col").none() = nb::none(), nb::arg("flags") = 0,
        " Plots an axis-aligned image. #bounds_min/bounds_max are in plot coordinates (y-up) and #uv0/uv1 are in texture coordinates (y-down).\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * uv0: ImVec2(0, 0)\n        * uv1: ImVec2(1, 1)\n        * tint_col: ImVec4(1, 1, 1, 1)");
    // #endif
    //

    m.def("plot_text",
        [](const char * text, double x, double y, const std::optional<const ImVec2> & pix_offset = std::nullopt, ImPlotTextFlags flags = 0)
        {
            auto PlotText_adapt_mutable_param_with_default_value = [](const char * text, double x, double y, const std::optional<const ImVec2> & pix_offset = std::nullopt, ImPlotTextFlags flags = 0)
            {

                const ImVec2& pix_offset_or_default = [&]() -> const ImVec2 {
                    if (pix_offset.has_value())
                        return pix_offset.value();
                    else
                        return ImVec2(0,0);
                }();

                ImPlot::PlotText(text, x, y, pix_offset_or_default, flags);
            };

            PlotText_adapt_mutable_param_with_default_value(text, x, y, pix_offset, flags);
        },
        nb::arg("text"), nb::arg("x"), nb::arg("y"), nb::arg("pix_offset").none() = nb::none(), nb::arg("flags") = 0,
        " Plots a centered text label at point x,y with an optional pixel offset. Text color can be changed with ImPlot::PushStyleColor(ImPlotCol_InlayText, ...).\n\n\nPython bindings defaults:\n    If pix_offset is None, then its default value will be: ImVec2(0,0)");

    m.def("plot_dummy",
        ImPlot::PlotDummy,
        nb::arg("label_id"), nb::arg("flags") = 0,
        "Plots a dummy item (i.e. adds a legend entry colored by ImPlotCol_Line)");

    m.def("drag_point",
        [](int id, double x, double y, const ImVec4 & col, float size = 4, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
        {
            auto DragPoint_adapt_modifiable_immutable_to_return = [](int id, double x, double y, const ImVec4 & col, float size = 4, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
            {
                double * x_adapt_modifiable = & x;
                double * y_adapt_modifiable = & y;
                bool * out_clicked_adapt_modifiable = nullptr;
                if (out_clicked.has_value())
                    out_clicked_adapt_modifiable = & (*out_clicked);
                bool * out_hovered_adapt_modifiable = nullptr;
                if (out_hovered.has_value())
                    out_hovered_adapt_modifiable = & (*out_hovered);
                bool * held_adapt_modifiable = nullptr;
                if (held.has_value())
                    held_adapt_modifiable = & (*held);

                bool r = ImPlot::DragPoint(id, x_adapt_modifiable, y_adapt_modifiable, col, size, flags, out_clicked_adapt_modifiable, out_hovered_adapt_modifiable, held_adapt_modifiable);
                return std::make_tuple(r, x, y, out_clicked, out_hovered, held);
            };

            return DragPoint_adapt_modifiable_immutable_to_return(id, x, y, col, size, flags, out_clicked, out_hovered, held);
        },
        nb::arg("id_"), nb::arg("x"), nb::arg("y"), nb::arg("col"), nb::arg("size") = 4, nb::arg("flags") = 0, nb::arg("out_clicked").none() = nb::none(), nb::arg("out_hovered").none() = nb::none(), nb::arg("held").none() = nb::none(),
        "Shows a draggable point at x,y. #col defaults to ImGuiCol_Text.");

    m.def("drag_line_x",
        [](int id, double x, const ImVec4 & col, float thickness = 1, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
        {
            auto DragLineX_adapt_modifiable_immutable_to_return = [](int id, double x, const ImVec4 & col, float thickness = 1, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
            {
                double * x_adapt_modifiable = & x;
                bool * out_clicked_adapt_modifiable = nullptr;
                if (out_clicked.has_value())
                    out_clicked_adapt_modifiable = & (*out_clicked);
                bool * out_hovered_adapt_modifiable = nullptr;
                if (out_hovered.has_value())
                    out_hovered_adapt_modifiable = & (*out_hovered);
                bool * held_adapt_modifiable = nullptr;
                if (held.has_value())
                    held_adapt_modifiable = & (*held);

                bool r = ImPlot::DragLineX(id, x_adapt_modifiable, col, thickness, flags, out_clicked_adapt_modifiable, out_hovered_adapt_modifiable, held_adapt_modifiable);
                return std::make_tuple(r, x, out_clicked, out_hovered, held);
            };

            return DragLineX_adapt_modifiable_immutable_to_return(id, x, col, thickness, flags, out_clicked, out_hovered, held);
        },
        nb::arg("id_"), nb::arg("x"), nb::arg("col"), nb::arg("thickness") = 1, nb::arg("flags") = 0, nb::arg("out_clicked").none() = nb::none(), nb::arg("out_hovered").none() = nb::none(), nb::arg("held").none() = nb::none(),
        "Shows a draggable vertical guide line at an x-value. #col defaults to ImGuiCol_Text.");

    m.def("drag_line_y",
        [](int id, double y, const ImVec4 & col, float thickness = 1, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
        {
            auto DragLineY_adapt_modifiable_immutable_to_return = [](int id, double y, const ImVec4 & col, float thickness = 1, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
            {
                double * y_adapt_modifiable = & y;
                bool * out_clicked_adapt_modifiable = nullptr;
                if (out_clicked.has_value())
                    out_clicked_adapt_modifiable = & (*out_clicked);
                bool * out_hovered_adapt_modifiable = nullptr;
                if (out_hovered.has_value())
                    out_hovered_adapt_modifiable = & (*out_hovered);
                bool * held_adapt_modifiable = nullptr;
                if (held.has_value())
                    held_adapt_modifiable = & (*held);

                bool r = ImPlot::DragLineY(id, y_adapt_modifiable, col, thickness, flags, out_clicked_adapt_modifiable, out_hovered_adapt_modifiable, held_adapt_modifiable);
                return std::make_tuple(r, y, out_clicked, out_hovered, held);
            };

            return DragLineY_adapt_modifiable_immutable_to_return(id, y, col, thickness, flags, out_clicked, out_hovered, held);
        },
        nb::arg("id_"), nb::arg("y"), nb::arg("col"), nb::arg("thickness") = 1, nb::arg("flags") = 0, nb::arg("out_clicked").none() = nb::none(), nb::arg("out_hovered").none() = nb::none(), nb::arg("held").none() = nb::none(),
        "Shows a draggable horizontal guide line at a y-value. #col defaults to ImGuiCol_Text.");

    m.def("drag_rect",
        [](int id, double x1, double y1, double x2, double y2, const ImVec4 & col, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, double, double, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
        {
            auto DragRect_adapt_modifiable_immutable_to_return = [](int id, double x1, double y1, double x2, double y2, const ImVec4 & col, ImPlotDragToolFlags flags = 0, std::optional<bool> out_clicked = std::nullopt, std::optional<bool> out_hovered = std::nullopt, std::optional<bool> held = std::nullopt) -> std::tuple<bool, double, double, double, double, std::optional<bool>, std::optional<bool>, std::optional<bool>>
            {
                double * x1_adapt_modifiable = & x1;
                double * y1_adapt_modifiable = & y1;
                double * x2_adapt_modifiable = & x2;
                double * y2_adapt_modifiable = & y2;
                bool * out_clicked_adapt_modifiable = nullptr;
                if (out_clicked.has_value())
                    out_clicked_adapt_modifiable = & (*out_clicked);
                bool * out_hovered_adapt_modifiable = nullptr;
                if (out_hovered.has_value())
                    out_hovered_adapt_modifiable = & (*out_hovered);
                bool * held_adapt_modifiable = nullptr;
                if (held.has_value())
                    held_adapt_modifiable = & (*held);

                bool r = ImPlot::DragRect(id, x1_adapt_modifiable, y1_adapt_modifiable, x2_adapt_modifiable, y2_adapt_modifiable, col, flags, out_clicked_adapt_modifiable, out_hovered_adapt_modifiable, held_adapt_modifiable);
                return std::make_tuple(r, x1, y1, x2, y2, out_clicked, out_hovered, held);
            };

            return DragRect_adapt_modifiable_immutable_to_return(id, x1, y1, x2, y2, col, flags, out_clicked, out_hovered, held);
        },
        nb::arg("id_"), nb::arg("x1"), nb::arg("y1"), nb::arg("x2"), nb::arg("y2"), nb::arg("col"), nb::arg("flags") = 0, nb::arg("out_clicked").none() = nb::none(), nb::arg("out_hovered").none() = nb::none(), nb::arg("held").none() = nb::none(),
        "Shows a draggable and resizeable rectangle.");

    m.def("annotation",
        nb::overload_cast<double, double, const ImVec4 &, const ImVec2 &, bool, bool>(ImPlot::Annotation), nb::arg("x"), nb::arg("y"), nb::arg("col"), nb::arg("pix_offset"), nb::arg("clamp"), nb::arg("round") = false);

    m.def("annotation",
        [](double x, double y, const ImVec4 & col, const ImVec2 & pix_offset, bool clamp, const char * fmt)
        {
            auto Annotation_adapt_variadic_format = [](double x, double y, const ImVec4 & col, const ImVec2 & pix_offset, bool clamp, const char * fmt)
            {
                ImPlot::Annotation(x, y, col, pix_offset, clamp, "%s", fmt);
            };

            Annotation_adapt_variadic_format(x, y, col, pix_offset, clamp, fmt);
        },     nb::arg("x"), nb::arg("y"), nb::arg("col"), nb::arg("pix_offset"), nb::arg("clamp"), nb::arg("fmt"));

    m.def("tag_x",
        nb::overload_cast<double, const ImVec4 &, bool>(ImPlot::TagX), nb::arg("x"), nb::arg("col"), nb::arg("round") = false);

    m.def("tag_x",
        [](double x, const ImVec4 & col, const char * fmt)
        {
            auto TagX_adapt_variadic_format = [](double x, const ImVec4 & col, const char * fmt)
            {
                ImPlot::TagX(x, col, "%s", fmt);
            };

            TagX_adapt_variadic_format(x, col, fmt);
        },     nb::arg("x"), nb::arg("col"), nb::arg("fmt"));

    m.def("tag_y",
        nb::overload_cast<double, const ImVec4 &, bool>(ImPlot::TagY), nb::arg("y"), nb::arg("col"), nb::arg("round") = false);

    m.def("tag_y",
        [](double y, const ImVec4 & col, const char * fmt)
        {
            auto TagY_adapt_variadic_format = [](double y, const ImVec4 & col, const char * fmt)
            {
                ImPlot::TagY(y, col, "%s", fmt);
            };

            TagY_adapt_variadic_format(y, col, fmt);
        },     nb::arg("y"), nb::arg("col"), nb::arg("fmt"));

    m.def("set_axis",
        ImPlot::SetAxis, nb::arg("axis"));

    m.def("set_axes",
        ImPlot::SetAxes, nb::arg("x_axis"), nb::arg("y_axis"));

    m.def("pixels_to_plot",
        [](const ImVec2 & pix, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
        {
            auto PixelsToPlot_adapt_mutable_param_with_default_value = [](const ImVec2 & pix, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::PixelsToPlot(pix, x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return PixelsToPlot_adapt_mutable_param_with_default_value(pix, x_axis, y_axis);
        },
        nb::arg("pix"), nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("pixels_to_plot",
        [](float x, float y, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
        {
            auto PixelsToPlot_adapt_mutable_param_with_default_value = [](float x, float y, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::PixelsToPlot(x, y, x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return PixelsToPlot_adapt_mutable_param_with_default_value(x, y, x_axis, y_axis);
        },
        nb::arg("x"), nb::arg("y"), nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("plot_to_pixels",
        [](const ImPlotPoint & plt, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImVec2
        {
            auto PlotToPixels_adapt_mutable_param_with_default_value = [](const ImPlotPoint & plt, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImVec2
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::PlotToPixels(plt, x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return PlotToPixels_adapt_mutable_param_with_default_value(plt, x_axis, y_axis);
        },
        nb::arg("plt"), nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("plot_to_pixels",
        [](double x, double y, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImVec2
        {
            auto PlotToPixels_adapt_mutable_param_with_default_value = [](double x, double y, const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImVec2
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::PlotToPixels(x, y, x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return PlotToPixels_adapt_mutable_param_with_default_value(x, y, x_axis, y_axis);
        },
        nb::arg("x"), nb::arg("y"), nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        "Python bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("get_plot_pos",
        ImPlot::GetPlotPos, "Get the current Plot position (top-left) in pixels.");

    m.def("get_plot_size",
        ImPlot::GetPlotSize, "Get the curent Plot size in pixels.");

    m.def("get_plot_mouse_pos",
        [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
        {
            auto GetPlotMousePos_adapt_mutable_param_with_default_value = [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotPoint
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::GetPlotMousePos(x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return GetPlotMousePos_adapt_mutable_param_with_default_value(x_axis, y_axis);
        },
        nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        " Returns the mouse position in x,y coordinates of the current plot. Passing IMPLOT_AUTO uses the current axes.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("get_plot_limits",
        [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotRect
        {
            auto GetPlotLimits_adapt_mutable_param_with_default_value = [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotRect
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::GetPlotLimits(x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return GetPlotLimits_adapt_mutable_param_with_default_value(x_axis, y_axis);
        },
        nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        " Returns the current plot axis range.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("is_plot_hovered",
        ImPlot::IsPlotHovered, "Returns True if the plot area in the current plot is hovered.");

    m.def("is_axis_hovered",
        ImPlot::IsAxisHovered,
        nb::arg("axis"),
        "Returns True if the axis label area in the current plot is hovered.");

    m.def("is_subplots_hovered",
        ImPlot::IsSubplotsHovered, "Returns True if the bounding frame of a subplot is hovered.");

    m.def("is_plot_selected",
        ImPlot::IsPlotSelected, "Returns True if the current plot is being box selected.");

    m.def("get_plot_selection",
        [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotRect
        {
            auto GetPlotSelection_adapt_mutable_param_with_default_value = [](const std::optional<const ImAxis> & x_axis = std::nullopt, const std::optional<const ImAxis> & y_axis = std::nullopt) -> ImPlotRect
            {

                const ImAxis& x_axis_or_default = [&]() -> const ImAxis {
                    if (x_axis.has_value())
                        return x_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImAxis& y_axis_or_default = [&]() -> const ImAxis {
                    if (y_axis.has_value())
                        return y_axis.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::GetPlotSelection(x_axis_or_default, y_axis_or_default);
                return lambda_result;
            };

            return GetPlotSelection_adapt_mutable_param_with_default_value(x_axis, y_axis);
        },
        nb::arg("x_axis").none() = nb::none(), nb::arg("y_axis").none() = nb::none(),
        " Returns the current plot box selection bounds. Passing IMPLOT_AUTO uses the current axes.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * x_axis: IMPLOT_AUTO\n        * y_axis: IMPLOT_AUTO");

    m.def("cancel_plot_selection",
        ImPlot::CancelPlotSelection, "Cancels a the current plot box selection.");

    m.def("hide_next_item",
        [](bool hidden = true, const std::optional<const ImPlotCond> & cond = std::nullopt)
        {
            auto HideNextItem_adapt_mutable_param_with_default_value = [](bool hidden = true, const std::optional<const ImPlotCond> & cond = std::nullopt)
            {

                const ImPlotCond& cond_or_default = [&]() -> const ImPlotCond {
                    if (cond.has_value())
                        return cond.value();
                    else
                        return ImPlotCond_Once;
                }();

                ImPlot::HideNextItem(hidden, cond_or_default);
            };

            HideNextItem_adapt_mutable_param_with_default_value(hidden, cond);
        },
        nb::arg("hidden") = true, nb::arg("cond").none() = nb::none(),
        " Hides or shows the next plot item (i.e. as if it were toggled from the legend).\n Use ImPlotCond_Always if you need to forcefully set this every frame.\n\n\nPython bindings defaults:\n    If cond is None, then its default value will be: Cond_Once");

    m.def("begin_aligned_plots",
        ImPlot::BeginAlignedPlots,
        nb::arg("group_id"), nb::arg("vertical") = true,
        " Align axis padding over multiple plots in a single row or column. #group_id must\n be unique. If this function returns True, EndAlignedPlots() must be called.");

    m.def("end_aligned_plots",
        ImPlot::EndAlignedPlots, "Only call EndAlignedPlots() if BeginAlignedPlots() returns True!");

    m.def("begin_legend_popup",
        ImPlot::BeginLegendPopup,
        nb::arg("label_id"), nb::arg("mouse_button") = 1,
        "Begin a popup for a legend entry.");

    m.def("end_legend_popup",
        ImPlot::EndLegendPopup, "End a popup for a legend entry.");

    m.def("is_legend_entry_hovered",
        ImPlot::IsLegendEntryHovered,
        nb::arg("label_id"),
        "Returns True if a plot item legend entry is hovered.");

    m.def("begin_drag_drop_target_plot",
        ImPlot::BeginDragDropTargetPlot, "Turns the current plot's plotting area into a drag and drop target. Don't forget to call EndDragDropTarget!");

    m.def("begin_drag_drop_target_axis",
        ImPlot::BeginDragDropTargetAxis,
        nb::arg("axis"),
        "Turns the current plot's X-axis into a drag and drop target. Don't forget to call EndDragDropTarget!");

    m.def("begin_drag_drop_target_legend",
        ImPlot::BeginDragDropTargetLegend, "Turns the current plot's legend into a drag and drop target. Don't forget to call EndDragDropTarget!");

    m.def("end_drag_drop_target",
        ImPlot::EndDragDropTarget, "Ends a drag and drop target (currently just an alias for ImGui::EndDragDropTarget).");

    m.def("begin_drag_drop_source_plot",
        ImPlot::BeginDragDropSourcePlot,
        nb::arg("flags") = 0,
        "Turns the current plot's plotting area into a drag and drop source. You must hold Ctrl. Don't forget to call EndDragDropSource!");

    m.def("begin_drag_drop_source_axis",
        ImPlot::BeginDragDropSourceAxis,
        nb::arg("axis"), nb::arg("flags") = 0,
        "Turns the current plot's X-axis into a drag and drop source. You must hold Ctrl. Don't forget to call EndDragDropSource!");

    m.def("begin_drag_drop_source_item",
        ImPlot::BeginDragDropSourceItem,
        nb::arg("label_id"), nb::arg("flags") = 0,
        "Turns an item in the current plot's legend into drag and drop source. Don't forget to call EndDragDropSource!");

    m.def("end_drag_drop_source",
        ImPlot::EndDragDropSource, "Ends a drag and drop source (currently just an alias for ImGui::EndDragDropSource).");

    m.def("get_style",
        ImPlot::GetStyle, nb::rv_policy::reference);

    m.def("set_style",
        ImPlot::SetStyle, nb::arg("style"));

    m.def("style_colors_auto",
        ImPlot::StyleColorsAuto,
        nb::arg("dst") = nb::none(),
        "Style plot colors for current ImGui style (default).");

    m.def("style_colors_classic",
        ImPlot::StyleColorsClassic,
        nb::arg("dst") = nb::none(),
        "Style plot colors for ImGui \"Classic\".");

    m.def("style_colors_dark",
        ImPlot::StyleColorsDark,
        nb::arg("dst") = nb::none(),
        "Style plot colors for ImGui \"Dark\".");

    m.def("style_colors_light",
        ImPlot::StyleColorsLight,
        nb::arg("dst") = nb::none(),
        "Style plot colors for ImGui \"Light\".");

    m.def("push_style_color",
        nb::overload_cast<ImPlotCol, ImU32>(ImPlot::PushStyleColor), nb::arg("idx"), nb::arg("col"));

    m.def("push_style_color",
        nb::overload_cast<ImPlotCol, const ImVec4 &>(ImPlot::PushStyleColor), nb::arg("idx"), nb::arg("col"));

    m.def("pop_style_color",
        ImPlot::PopStyleColor,
        nb::arg("count") = 1,
        "Undo temporary style color modification(s). Undo multiple pushes at once by increasing count.");

    m.def("push_style_var",
        nb::overload_cast<ImPlotStyleVar, float>(ImPlot::PushStyleVar),
        nb::arg("idx"), nb::arg("val"),
        "Temporarily modify a style variable of float type. Don't forget to call PopStyleVar!");

    m.def("push_style_var",
        nb::overload_cast<ImPlotStyleVar, int>(ImPlot::PushStyleVar),
        nb::arg("idx"), nb::arg("val"),
        "Temporarily modify a style variable of int type. Don't forget to call PopStyleVar!");

    m.def("push_style_var",
        nb::overload_cast<ImPlotStyleVar, const ImVec2 &>(ImPlot::PushStyleVar),
        nb::arg("idx"), nb::arg("val"),
        "Temporarily modify a style variable of ImVec2 type. Don't forget to call PopStyleVar!");

    m.def("pop_style_var",
        ImPlot::PopStyleVar,
        nb::arg("count") = 1,
        "Undo temporary style variable modification(s). Undo multiple pushes at once by increasing count.");

    m.def("set_next_line_style",
        [](const std::optional<const ImVec4> & col = std::nullopt, float weight = IMPLOT_AUTO)
        {
            auto SetNextLineStyle_adapt_mutable_param_with_default_value = [](const std::optional<const ImVec4> & col = std::nullopt, float weight = IMPLOT_AUTO)
            {

                const ImVec4& col_or_default = [&]() -> const ImVec4 {
                    if (col.has_value())
                        return col.value();
                    else
                        return IMPLOT_AUTO_COL;
                }();

                ImPlot::SetNextLineStyle(col_or_default, weight);
            };

            SetNextLineStyle_adapt_mutable_param_with_default_value(col, weight);
        },
        nb::arg("col").none() = nb::none(), nb::arg("weight") = IMPLOT_AUTO,
        " Set the line color and weight for the next item only.\n\n\nPython bindings defaults:\n    If col is None, then its default value will be: IMPLOT_AUTO_COL");

    m.def("set_next_fill_style",
        [](const std::optional<const ImVec4> & col = std::nullopt, float alpha_mod = IMPLOT_AUTO)
        {
            auto SetNextFillStyle_adapt_mutable_param_with_default_value = [](const std::optional<const ImVec4> & col = std::nullopt, float alpha_mod = IMPLOT_AUTO)
            {

                const ImVec4& col_or_default = [&]() -> const ImVec4 {
                    if (col.has_value())
                        return col.value();
                    else
                        return IMPLOT_AUTO_COL;
                }();

                ImPlot::SetNextFillStyle(col_or_default, alpha_mod);
            };

            SetNextFillStyle_adapt_mutable_param_with_default_value(col, alpha_mod);
        },
        nb::arg("col").none() = nb::none(), nb::arg("alpha_mod") = IMPLOT_AUTO,
        " Set the fill color for the next item only.\n\n\nPython bindings defaults:\n    If col is None, then its default value will be: IMPLOT_AUTO_COL");

    m.def("set_next_marker_style",
        [](const std::optional<const ImPlotMarker> & marker = std::nullopt, float size = IMPLOT_AUTO, const std::optional<const ImVec4> & fill = std::nullopt, float weight = IMPLOT_AUTO, const std::optional<const ImVec4> & outline = std::nullopt)
        {
            auto SetNextMarkerStyle_adapt_mutable_param_with_default_value = [](const std::optional<const ImPlotMarker> & marker = std::nullopt, float size = IMPLOT_AUTO, const std::optional<const ImVec4> & fill = std::nullopt, float weight = IMPLOT_AUTO, const std::optional<const ImVec4> & outline = std::nullopt)
            {

                const ImPlotMarker& marker_or_default = [&]() -> const ImPlotMarker {
                    if (marker.has_value())
                        return marker.value();
                    else
                        return IMPLOT_AUTO;
                }();

                const ImVec4& fill_or_default = [&]() -> const ImVec4 {
                    if (fill.has_value())
                        return fill.value();
                    else
                        return IMPLOT_AUTO_COL;
                }();

                const ImVec4& outline_or_default = [&]() -> const ImVec4 {
                    if (outline.has_value())
                        return outline.value();
                    else
                        return IMPLOT_AUTO_COL;
                }();

                ImPlot::SetNextMarkerStyle(marker_or_default, size, fill_or_default, weight, outline_or_default);
            };

            SetNextMarkerStyle_adapt_mutable_param_with_default_value(marker, size, fill, weight, outline);
        },
        nb::arg("marker").none() = nb::none(), nb::arg("size") = IMPLOT_AUTO, nb::arg("fill").none() = nb::none(), nb::arg("weight") = IMPLOT_AUTO, nb::arg("outline").none() = nb::none(),
        " Set the marker style for the next item only.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * marker: IMPLOT_AUTO\n        * fill: IMPLOT_AUTO_COL\n        * outline: IMPLOT_AUTO_COL");

    m.def("set_next_error_bar_style",
        [](const std::optional<const ImVec4> & col = std::nullopt, float size = IMPLOT_AUTO, float weight = IMPLOT_AUTO)
        {
            auto SetNextErrorBarStyle_adapt_mutable_param_with_default_value = [](const std::optional<const ImVec4> & col = std::nullopt, float size = IMPLOT_AUTO, float weight = IMPLOT_AUTO)
            {

                const ImVec4& col_or_default = [&]() -> const ImVec4 {
                    if (col.has_value())
                        return col.value();
                    else
                        return IMPLOT_AUTO_COL;
                }();

                ImPlot::SetNextErrorBarStyle(col_or_default, size, weight);
            };

            SetNextErrorBarStyle_adapt_mutable_param_with_default_value(col, size, weight);
        },
        nb::arg("col").none() = nb::none(), nb::arg("size") = IMPLOT_AUTO, nb::arg("weight") = IMPLOT_AUTO,
        " Set the error bar style for the next item only.\n\n\nPython bindings defaults:\n    If col is None, then its default value will be: IMPLOT_AUTO_COL");

    m.def("get_last_item_color",
        ImPlot::GetLastItemColor, "Gets the last item primary color (i.e. its legend icon color)");

    m.def("get_style_color_name",
        ImPlot::GetStyleColorName,
        nb::arg("idx"),
        "Returns the null terminated string name for an ImPlotCol.",
        nb::rv_policy::reference);

    m.def("get_marker_name",
        ImPlot::GetMarkerName,
        nb::arg("idx"),
        "Returns the null terminated string name for an ImPlotMarker.",
        nb::rv_policy::reference);

    m.def("get_colormap_count",
        ImPlot::GetColormapCount, "Returns the number of available colormaps (i.e. the built-in + user-added count).");

    m.def("get_colormap_name",
        ImPlot::GetColormapName,
        nb::arg("cmap"),
        "Returns a null terminated string name for a colormap given an index. Returns None if index is invalid.",
        nb::rv_policy::reference);

    m.def("get_colormap_index",
        ImPlot::GetColormapIndex,
        nb::arg("name"),
        "Returns an index number for a colormap given a valid string name. Returns -1 if name is invalid.");

    m.def("push_colormap",
        nb::overload_cast<ImPlotColormap>(ImPlot::PushColormap),
        nb::arg("cmap"),
        "Temporarily switch to one of the built-in (i.e. ImPlotColormap_XXX) or user-added colormaps (i.e. a return value of AddColormap). Don't forget to call PopColormap!");

    m.def("push_colormap",
        nb::overload_cast<const char *>(ImPlot::PushColormap),
        nb::arg("name"),
        "Push a colormap by string name. Use built-in names such as \"Default\", \"Deep\", \"Jet\", etc. or a string you provided to AddColormap. Don't forget to call PopColormap!");

    m.def("pop_colormap",
        ImPlot::PopColormap,
        nb::arg("count") = 1,
        "Undo temporary colormap modification(s). Undo multiple pushes at once by increasing count.");

    m.def("next_colormap_color",
        ImPlot::NextColormapColor, " Returns the next color from the current colormap and advances the colormap for the current plot.\n Can also be used with no return value to skip colors if desired. You need to call this between Begin/EndPlot!");

    m.def("get_colormap_size",
        [](const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> int
        {
            auto GetColormapSize_adapt_mutable_param_with_default_value = [](const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> int
            {

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::GetColormapSize(cmap_or_default);
                return lambda_result;
            };

            return GetColormapSize_adapt_mutable_param_with_default_value(cmap);
        },
        nb::arg("cmap").none() = nb::none(),
        " Returns the size of a colormap.\n\n\nPython bindings defaults:\n    If cmap is None, then its default value will be: IMPLOT_AUTO");

    m.def("get_colormap_color",
        [](int idx, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> ImVec4
        {
            auto GetColormapColor_adapt_mutable_param_with_default_value = [](int idx, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> ImVec4
            {

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::GetColormapColor(idx, cmap_or_default);
                return lambda_result;
            };

            return GetColormapColor_adapt_mutable_param_with_default_value(idx, cmap);
        },
        nb::arg("idx"), nb::arg("cmap").none() = nb::none(),
        " Returns a color from a colormap given an index >= 0 (modulo will be performed).\n\n\nPython bindings defaults:\n    If cmap is None, then its default value will be: IMPLOT_AUTO");

    m.def("sample_colormap",
        [](float t, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> ImVec4
        {
            auto SampleColormap_adapt_mutable_param_with_default_value = [](float t, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> ImVec4
            {

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::SampleColormap(t, cmap_or_default);
                return lambda_result;
            };

            return SampleColormap_adapt_mutable_param_with_default_value(t, cmap);
        },
        nb::arg("t"), nb::arg("cmap").none() = nb::none(),
        " Sample a color from the current colormap given t between 0 and 1.\n\n\nPython bindings defaults:\n    If cmap is None, then its default value will be: IMPLOT_AUTO");

    m.def("colormap_scale",
        [](const char * label, double scale_min, double scale_max, const std::optional<const ImVec2> & size = std::nullopt, const char * format = "%g", ImPlotColormapScaleFlags flags = 0, const std::optional<const ImPlotColormap> & cmap = std::nullopt)
        {
            auto ColormapScale_adapt_mutable_param_with_default_value = [](const char * label, double scale_min, double scale_max, const std::optional<const ImVec2> & size = std::nullopt, const char * format = "%g", ImPlotColormapScaleFlags flags = 0, const std::optional<const ImPlotColormap> & cmap = std::nullopt)
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0,0);
                }();

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                ImPlot::ColormapScale(label, scale_min, scale_max, size_or_default, format, flags, cmap_or_default);
            };

            ColormapScale_adapt_mutable_param_with_default_value(label, scale_min, scale_max, size, format, flags, cmap);
        },
        nb::arg("label"), nb::arg("scale_min"), nb::arg("scale_max"), nb::arg("size").none() = nb::none(), nb::arg("format") = "%g", nb::arg("flags") = 0, nb::arg("cmap").none() = nb::none(),
        " Shows a vertical color scale with linear spaced ticks using the specified color map. Use double hashes to hide label (e.g. \"##NoLabel\"). If scale_min > scale_max, the scale to color mapping will be reversed.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * size: ImVec2(0,0)\n        * cmap: IMPLOT_AUTO");

    m.def("colormap_slider",
        [](const char * label, float t, ImVec4 * out = nullptr, const char * format = "", const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> std::tuple<bool, float>
        {
            auto ColormapSlider_adapt_mutable_param_with_default_value = [](const char * label, float * t, ImVec4 * out = nullptr, const char * format = "", const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> bool
            {

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::ColormapSlider(label, t, out, format, cmap_or_default);
                return lambda_result;
            };
            auto ColormapSlider_adapt_modifiable_immutable_to_return = [&ColormapSlider_adapt_mutable_param_with_default_value](const char * label, float t, ImVec4 * out = nullptr, const char * format = "", const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> std::tuple<bool, float>
            {
                float * t_adapt_modifiable = & t;

                bool r = ColormapSlider_adapt_mutable_param_with_default_value(label, t_adapt_modifiable, out, format, cmap);
                return std::make_tuple(r, t);
            };

            return ColormapSlider_adapt_modifiable_immutable_to_return(label, t, out, format, cmap);
        },
        nb::arg("label"), nb::arg("t"), nb::arg("out") = nb::none(), nb::arg("format") = "", nb::arg("cmap").none() = nb::none(),
        " Shows a horizontal slider with a colormap gradient background. Optionally returns the color sampled at t in [0 1].\n\n\nPython bindings defaults:\n    If cmap is None, then its default value will be: IMPLOT_AUTO");

    m.def("colormap_button",
        [](const char * label, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> bool
        {
            auto ColormapButton_adapt_mutable_param_with_default_value = [](const char * label, const std::optional<const ImVec2> & size = std::nullopt, const std::optional<const ImPlotColormap> & cmap = std::nullopt) -> bool
            {

                const ImVec2& size_or_default = [&]() -> const ImVec2 {
                    if (size.has_value())
                        return size.value();
                    else
                        return ImVec2(0,0);
                }();

                const ImPlotColormap& cmap_or_default = [&]() -> const ImPlotColormap {
                    if (cmap.has_value())
                        return cmap.value();
                    else
                        return IMPLOT_AUTO;
                }();

                auto lambda_result = ImPlot::ColormapButton(label, size_or_default, cmap_or_default);
                return lambda_result;
            };

            return ColormapButton_adapt_mutable_param_with_default_value(label, size, cmap);
        },
        nb::arg("label"), nb::arg("size").none() = nb::none(), nb::arg("cmap").none() = nb::none(),
        " Shows a button with a colormap gradient brackground.\n\n\nPython bindings defaults:\n    If any of the params below is None, then its default value below will be used:\n        * size: ImVec2(0,0)\n        * cmap: IMPLOT_AUTO");

    m.def("bust_color_cache",
        [](std::optional<std::string> plot_title_id = std::nullopt)
        {
            auto BustColorCache_adapt_const_char_pointer_with_default_null = [](std::optional<std::string> plot_title_id = std::nullopt)
            {
                const char * plot_title_id_adapt_default_null = nullptr;
                if (plot_title_id.has_value())
                    plot_title_id_adapt_default_null = plot_title_id.value().c_str();

                ImPlot::BustColorCache(plot_title_id_adapt_default_null);
            };

            BustColorCache_adapt_const_char_pointer_with_default_null(plot_title_id);
        },
        nb::arg("plot_title_id").none() = nb::none(),
        " When items in a plot sample their color from a colormap, the color is cached and does not change\n unless explicitly overriden. Therefore, if you change the colormap after the item has already been plotted,\n item colors will NOT update. If you need item colors to resample the new colormap, then use this\n function to bust the cached colors. If #plot_title_id is None, then every item in EVERY existing plot\n will be cache busted. Otherwise only the plot specified by #plot_title_id will be busted. For the\n latter, this function must be called in the same ImGui ID scope that the plot is in. You should rarely if ever\n need this function, but it is available for applications that require runtime colormap swaps (e.g. Heatmaps demo).");

    m.def("get_input_map",
        ImPlot::GetInputMap,
        "Provides access to input mapping structure for permanant modifications to controls for pan, select, etc.",
        nb::rv_policy::reference);

    m.def("map_input_default",
        ImPlot::MapInputDefault,
        nb::arg("dst") = nb::none(),
        "Default input mapping: pan = LMB drag, box select = RMB drag, fit = LMB double click, context menu = RMB click, zoom = scroll.");

    m.def("map_input_reverse",
        ImPlot::MapInputReverse,
        nb::arg("dst") = nb::none(),
        "Reverse input mapping: pan = RMB drag, box select = LMB drag, fit = LMB double click, context menu = RMB click, zoom = scroll.");

    m.def("item_icon",
        nb::overload_cast<const ImVec4 &>(ImPlot::ItemIcon), nb::arg("col"));

    m.def("item_icon",
        nb::overload_cast<ImU32>(ImPlot::ItemIcon), nb::arg("col"));

    m.def("colormap_icon",
        ImPlot::ColormapIcon, nb::arg("cmap"));

    m.def("get_plot_draw_list",
        ImPlot::GetPlotDrawList,
        "Get the plot draw list for custom rendering to the current plot area. Call between Begin/EndPlot.",
        nb::rv_policy::reference);

    m.def("push_plot_clip_rect",
        ImPlot::PushPlotClipRect,
        nb::arg("expand") = 0,
        "Push clip rect for rendering to current plot area. The rect can be expanded or contracted by #expand pixels. Call between Begin/EndPlot.");

    m.def("pop_plot_clip_rect",
        ImPlot::PopPlotClipRect, "Pop plot clip rect. Call between Begin/EndPlot.");

    m.def("show_style_selector",
        ImPlot::ShowStyleSelector,
        nb::arg("label"),
        "Shows ImPlot style selector dropdown menu.");

    m.def("show_colormap_selector",
        ImPlot::ShowColormapSelector,
        nb::arg("label"),
        "Shows ImPlot colormap selector dropdown menu.");

    m.def("show_input_map_selector",
        ImPlot::ShowInputMapSelector,
        nb::arg("label"),
        "Shows ImPlot input map selector dropdown menu.");

    m.def("show_style_editor",
        ImPlot::ShowStyleEditor,
        nb::arg("ref") = nb::none(),
        "Shows ImPlot style editor block (not a window).");

    m.def("show_user_guide",
        ImPlot::ShowUserGuide, "Add basic help/info block for end users (not a window).");

    m.def("show_metrics_window",
        [](std::optional<bool> p_popen = std::nullopt) -> std::optional<bool>
        {
            auto ShowMetricsWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_popen = std::nullopt) -> std::optional<bool>
            {
                bool * p_popen_adapt_modifiable = nullptr;
                if (p_popen.has_value())
                    p_popen_adapt_modifiable = & (*p_popen);

                ImPlot::ShowMetricsWindow(p_popen_adapt_modifiable);
                return p_popen;
            };

            return ShowMetricsWindow_adapt_modifiable_immutable_to_return(p_popen);
        },
        nb::arg("p_popen").none() = nb::none(),
        "Shows ImPlot metrics/debug information window.");

    m.def("show_demo_window",
        [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
        {
            auto ShowDemoWindow_adapt_modifiable_immutable_to_return = [](std::optional<bool> p_open = std::nullopt) -> std::optional<bool>
            {
                bool * p_open_adapt_modifiable = nullptr;
                if (p_open.has_value())
                    p_open_adapt_modifiable = & (*p_open);

                ImPlot::ShowDemoWindow(p_open_adapt_modifiable);
                return p_open;
            };

            return ShowDemoWindow_adapt_modifiable_immutable_to_return(p_open);
        },
        nb::arg("p_open").none() = nb::none(),
        "Shows the ImPlot demo window (add implot_demo.cpp to your sources!)");

    m.def("show_all_demos",
        ImPlot::ShowAllDemos, "Bundle: ShowAllDemos is extracted from ShowDemoWindow, so that it can be used without creating an ImGui window.");
    // #endif
    ////////////////////    </generated_from:implot.h>    ////////////////////

    // </litgen_pydef> // Autogenerated code end
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  AUTOGENERATED CODE END !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


    ///////////////////////////////////////////////////////////////////////////
    // MANUAL BINDINGS BELOW
    ///////////////////////////////////////////////////////////////////////////
    pyClassImPlotPointPtr = & pyClassImPlotPoint;
    implot_binding_manual(m);
}


void implot_binding_manual(nb::module_& m)
{
    IM_ASSERT(pyClassImPlotPointPtr != nullptr);
    nb::class_<ImPlotPoint> pyClassImPlotPoint = *pyClassImPlotPointPtr;


    // this is an internal function that should only be called with a tuple, list or array argument
    auto cast_to_point = [](nb::handle obj) -> ImPlotPoint
    {
        if (len(obj) != 2)
            throw std::invalid_argument("Python tuple/list/array to implot.Point: size should be 2!");
        auto floats = nb::cast<std::vector<float>>(obj);
        return ImPlotPoint(floats[0], floats[1]);
    };

    // Add a new colormap, inputs are np.array[uint32]
    m.def("add_colormap",
        [](const char * name, const nb::ndarray<> & cols, bool qual=true)
        {
            const void * values_from_pyarray = cols.data();

            int ndim = (int)cols.ndim();
            int rows = static_cast<int>(cols.shape(0));
            int columns = ndim >= 2 ? static_cast<int>(cols.shape(1)) : -1;

            bool is_uint32 = [&cols]()
            {
                auto uint_type = static_cast<uint8_t>(nb::dlpack::dtype_code::UInt);
                uint8_t buffer_type = cols.dtype().code;
                bool is_uint = buffer_type == uint_type;
                bool is32bits = cols.dtype().bits == 32;
                return is_uint && is32bits;
            }();

            bool is_float = [&cols]()
            {
                auto float_type = static_cast<uint8_t>(nb::dlpack::dtype_code::Float);
                uint8_t buffer_type = cols.dtype().code;
                bool is_float = buffer_type == float_type;
                bool is_float_size = cols.dtype().bits / 8 == sizeof(float);
                return is_float && is_float_size;
            }();

            if (is_uint32 && (ndim == 1))
                return ImPlot::AddColormap(name, static_cast<const uint32_t *>(values_from_pyarray), rows, qual);
            else if (is_float && (ndim == 2) && (columns == 4))
                return ImPlot::AddColormap(name, static_cast<const ImVec4 *>(values_from_pyarray), rows, qual);
            else
                throw std::runtime_error(std::string("Bad array type, expected either 1D array of type uint32, or 2D (Nx4 ->RGBA) array of type float32. Got matrix of ndim =") + std::to_string(ndim) + " for cols.");

        }, nb::arg("name"), nb::arg("cols"), nb::arg("qual") = true,
        "Add a new colormap."
        );

    // SetupAxisTicks, cf https://github.com/pthom/imgui_bundle/issues/81
    m.def("setup_axis_ticks",
          [](ImAxis axis, double v_min, double v_max, int n_ticks,
              const std::optional<std::vector<std::string>>& labels = std::nullopt, bool keep_default=false)
          {
              if (!labels.has_value() || labels.value().empty())
                  ImPlot::SetupAxisTicks(axis, v_min, v_max, n_ticks, nullptr, keep_default);
              else
              {
                  IM_ASSERT(labels.value().size() == n_ticks && "The number of labels should match the number of ticks");
                  std::vector<const char*> label_char;
                  for (std::string const& str : labels.value()){
                      label_char.push_back(str.c_str());
                  }
                  ImPlot::SetupAxisTicks(axis, v_min, v_max, n_ticks, label_char.data(), keep_default);
              }
          }, nb::arg("axis"), nb::arg("v_min"), nb::arg("v_max"), nb::arg("n_ticks"), nb::arg("labels"), nb::arg("keep_default"),
          "Sets an axis' ticks and optionally the labels for the next plot. To keep the default ticks, set #keep_default=true."
    );
    m.def("setup_axis_ticks",
          [](ImAxis axis, std::vector<double> values,
              const std::optional<std::vector<std::string>>& labels = std::nullopt, bool keep_default=false)
          {
              int n_ticks = static_cast<int>(values.size());

              if (!labels.has_value() || labels.value().empty())
                  ImPlot::SetupAxisTicks(axis, values.data(), n_ticks, nullptr, keep_default);
              else
              {
                  std::vector<const char*> label_char;
                  for (std::string const& str : labels.value()){
                      label_char.push_back(str.c_str());
                  }
                  ImPlot::SetupAxisTicks(axis, values.data(), n_ticks, label_char.data(), keep_default);
              }
          }, nb::arg("axis"), nb::arg("values"), nb::arg("labels"), nb::arg("keep_default"),
          "Sets an axis' ticks and optionally the labels for the next plot. To keep the default ticks, set #keep_default=true."
    );

    // PlotHeatmap, cf https://github.com/pthom/imgui_bundle/issues/81
    m.def("plot_heatmap",
          [](const char * label_id, const nb::ndarray<> & values, double scale_min = 0, double scale_max=0, const char * label_fmt="%.1f", const ImPlotPoint& bounds_min=ImPlotPoint(0,0), const ImPlotPoint& bounds_max=ImPlotPoint(1,1), ImPlotHeatmapFlags flags = 0)
          {
              auto PlotHeatmap_adapt_c_buffers = [](const char * label_id, const nb::ndarray<> & values, double scale_min = 0, double scale_max=0, const char * label_fmt="%.1f",  const ImPlotPoint& bounds_min=ImPlotPoint(0,0), const ImPlotPoint& bounds_max=ImPlotPoint(1,1),  ImPlotHeatmapFlags flags = 0)
              {
                  // convert nb::array to C standard buffer (const)
                  const void * values_from_pyarray = values.data();

                  using np_uint_l = uint64_t;
                  using np_int_l = int64_t;

                  if (values.ndim() != 2)
                      throw std::runtime_error("plot_heatmap expects a numpy bidimensional array ");
                  int rows = static_cast<int>(values.shape(0));
                  int cols = static_cast<int>(values.shape(1));

                  // Define a lambda to compute the letter code for the buffer type
                  auto _nanobind_buffer_type_to_letter_code = [](uint8_t dtype_code, size_t sizeof_item)  -> char
                  {
                      #define DCODE(T) static_cast<uint8_t>(nb::dlpack::dtype_code::T)
                      const std::array<std::tuple<uint8_t, size_t, char>, 11> mappings = {{
                           {DCODE(UInt), 1, 'B'}, {DCODE(UInt), 2, 'H'}, {DCODE(UInt), 4, 'I'}, {DCODE(UInt), 8, 'L'},
                           {DCODE(Int), 1, 'b'}, {DCODE(Int), 2, 'h'}, {DCODE(Int), 4, 'i'}, {DCODE(Int), 8, 'l'},
                           {DCODE(Float), 4, 'f'}, {DCODE(Float), 8, 'd'}, {DCODE(Float), 16, 'g'}
                      }};
                      #undef DCODE
                      for (const auto& [code_val, size, letter] : mappings)
                          if (code_val == dtype_code && size == sizeof_item)
                              return letter;
                      throw std::runtime_error("Unsupported dtype");
                  };

                  // Compute the letter code for the buffer type
                  uint8_t dtype_code_buffer = values.dtype().code;
                  size_t sizeof_item_buffer = values.dtype().bits / 8;
                  char values_type = _nanobind_buffer_type_to_letter_code(dtype_code_buffer, sizeof_item_buffer);

                  // call the correct template version by casting
                  if (values_type == 'B')
                      ImPlot::PlotHeatmap(label_id, static_cast<const uint8_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'b')
                      ImPlot::PlotHeatmap(label_id, static_cast<const int8_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'H')
                      ImPlot::PlotHeatmap(label_id, static_cast<const uint16_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'h')
                      ImPlot::PlotHeatmap(label_id, static_cast<const int16_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'I')
                      ImPlot::PlotHeatmap(label_id, static_cast<const uint32_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'i')
                      ImPlot::PlotHeatmap(label_id, static_cast<const int32_t *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'L')
                      ImPlot::PlotHeatmap(label_id, static_cast<const np_uint_l *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'l')
                      ImPlot::PlotHeatmap(label_id, static_cast<const np_int_l *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'f')
                      ImPlot::PlotHeatmap(label_id, static_cast<const float *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'd')
                      ImPlot::PlotHeatmap(label_id, static_cast<const double *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'g')
                      ImPlot::PlotHeatmap(label_id, static_cast<const long double *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                  else if (values_type == 'q')
                      ImPlot::PlotHeatmap(label_id, static_cast<const long long *>(values_from_pyarray), rows, cols, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
                      // If we reach this point, the array type is not supported!
                  else
                      throw std::runtime_error(std::string("Bad array type ('") + values_type + "') for param values");
              };

              PlotHeatmap_adapt_c_buffers(label_id, values, scale_min, scale_max, label_fmt, bounds_min, bounds_max, flags);
          },
          nb::arg("label_id"), nb::arg("values"), nb::arg("scale_min")= 0, nb::arg("scale_max") = 0, nb::arg("label_fmt")="%.1f", nb::arg("bounds_min")=ImPlotPoint(0,0), nb::arg("bounds_max")=ImPlotPoint(1,1), nb::arg("flags")=0
    );


    pyClassImPlotPoint.def("__str__", [](const ImPlotPoint& self) -> std::string {
       char r[100];
       snprintf(r, 100, "ImPlotPoint(%f, %f)", self.x, self.y);
       return r;
    });
    pyClassImPlotPoint.def("__repr__", [](const ImPlotPoint& self) -> std::string {
        char r[100];
        snprintf(r, 100, "ImPlotPoint(%f, %f)", self.x, self.y);
        return r;
    });
    pyClassImPlotPoint.def("__len__", [](const ImPlotPoint& self) -> size_t {
        return 2;
    });
    pyClassImPlotPoint.def("__iter__", [](const ImPlotPoint& self) {
            return nb::make_iterator(nb::type<ImPlotPoint>(), "iterator", &self.x, &self.x+2);
        },
        nb::keep_alive<0, 1>() /* Essential: keep object alive while iterator exists */
    );
    pyClassImPlotPoint.def("__init__", [&cast_to_point](ImPlotPoint *self, nb::tuple t) {
        new (self) ImPlotPoint();
        *self = cast_to_point(std::move(t));
    });
    pyClassImPlotPoint.def("__init__", [&cast_to_point](ImPlotPoint *self, nb::list t) {
        new (self) ImPlotPoint();
        *self = cast_to_point(std::move(t));
    });
//    pyClassImPlotPoint.def("__init__", [](ImPlotPoint *self, nb::ndarray<> t) {
//        new (self) ImPlotPoint();
//        *self = cast_to_point(t);
//    });

    nb::implicitly_convertible<nb::tuple, ImPlotPoint>();
    nb::implicitly_convertible<nb::list, ImPlotPoint>();
    //nb::implicitly_convertible<nb::array, ImPlotPoint>();
    nb::implicitly_convertible<ImVec2, ImPlotPoint>();

    pyClassImPlotPoint.def("__init__", [](ImPlotPoint *self, ImPlotPoint ipp) {
        new (self) ImPlotPoint();
        *self = ImPlotPoint(ipp.x, ipp.y);
    }, nb::arg("implotpoint"));

}
#endif // IMGUI_BUNDLE_WITH_IMPLOT
