# CMake functions to simplify the creation of demo apps using imgui_bundle
# Those are convenience functions, mostly used in the C++ and in the tutorials.

function(ibd_add_demo_cpp)
    # Creates an app using imgui_bundle_add_app, and link it with demo_utils
    # Usage:
    #     ibd_add_demo_cpp(demo_name demo_file_1.cpp demo_file_2.cpp)
    set(args ${ARGN})
    list(GET args 0 demo_name)
    set(demos_asset_folder ${IMGUI_BUNDLE_PATH}/bindings/imgui_bundle/demos_assets)
    imgui_bundle_add_app(${args} ASSETS_LOCATION ${demos_asset_folder})
    if(MSVC)
        hello_imgui_msvc_target_set_folder(${demo_name} demos_cpp)
    endif()
    target_link_libraries(${demo_name} PUBLIC demo_utils)
    if (EMSCRIPTEN)
        # Bundle demos_assets
        hello_imgui_bundle_assets_from_folder(${demo_name} ${demos_cpp_folder}/../demos_assets)
    endif()
endfunction()

function(ibd_add_single_file_demo_cpp demo_cpp_file optional_prefix)
    # Creates a demo app
    # Usage:
    #     ibd_add_single_file_demo_cpp(demo_file.cpp)
    # or
    #    ibd_add_single_file_demo_cpp(demo_file.cpp prefix_)
    get_filename_component(demo_name "${demo_cpp_file}" NAME_WE)
    ibd_add_demo_cpp(${optional_prefix}${demo_name} ${demo_cpp_file})
endfunction()

function(ibd_add_auto_demo demo_cpp_file optional_prefix)
    # Creates a demo app, and provides main() automatically
    # Usage:
    #     ibd_add_auto_demo(demo_file.cpp)
    # (demo_file.cpp should provide a function named according to the containing file, e.g. `void demo_file()`)
    get_filename_component(demo_cpp_file "${demo_cpp_file}" NAME)
    get_filename_component(demo_name "${demo_cpp_file}" NAME_WE)
    ibd_add_demo_cpp(${optional_prefix}${demo_name} ${demo_cpp_file})

    # Read the contents of the cpp file
    file(READ "${demo_cpp_file}" demo_cpp_file_content)

    # If the cpp file does not contain a main function, provide it
    if (NOT "${demo_cpp_file_content}" MATCHES "int main")
        message(STATUS "${demo_cpp_file} does not contain a main function, adding it!")
        set(main_file ${CMAKE_CURRENT_BINARY_DIR}/${demo_cpp_file}.autogenerated.cpp)
        configure_file(${demos_cpp_folder}/_auto_main/_auto_main.cpp.in ${main_file})
        target_sources(${demo_name} PRIVATE ${main_file})
    endif()

endfunction()

function(ibd_add_this_folder_demos optional_prefix)
    # Creates demo apps for all cpp files in the current folder
    FILE(GLOB demo_cpp_files *.cpp)
    FOREACH(demo_cpp_file ${demo_cpp_files})
        ibd_add_single_file_demo_cpp(${demo_cpp_file} "${optional_prefix}")
    ENDFOREACH()
endfunction()

function(ibd_add_this_folder_auto_demos optional_prefix)
    # Creates demo apps for all cpp files in the current folder, and provide a main for them
    FILE(GLOB demo_cpp_files *.cpp)
    FOREACH(demo_cpp_file ${demo_cpp_files})
        ibd_add_auto_demo(${demo_cpp_file} "${optional_prefix}")
    ENDFOREACH()
endfunction()


function(ibd_add_this_folder_as_demos_library)
    file(GLOB sources *.cpp)
    get_filename_component(folder_name "${CMAKE_CURRENT_LIST_DIR}" NAME_WE)
    set(demos_library_name ${folder_name})
    add_library(${demos_library_name} ${sources})
    target_link_libraries(${demos_library_name} PRIVATE imgui_bundle)
    target_compile_definitions(${demos_library_name} PRIVATE IMGUI_BUNDLE_BUILD_DEMO_AS_LIBRARY)
endfunction()
