import os
import pickle
import tempfile
import uuid
import zipfile
from unittest.mock import Mock, patch

import pytest
from _pytest.logging import LogCaptureFixture

from arklex.resources.tools.rag.build_rag import build_rag

# Set test environment
os.environ["ARKLEX_TEST_ENV"] = "local"


# Create a simple object that can be pickled and matches CrawledObject interface
class SimpleDoc:
    def __init__(self, content: str, source: str) -> None:
        self.id = str(uuid.uuid4())  # Generate a unique ID
        self.content = content
        self.source = source
        self.is_error = False
        self.is_chunk = False
        self.metadata = {"source": source}
        self.error_message = None
        self.source_type = 1  # SourceType.WEB = 1


class TestBuildRAG:
    """Test class for build_rag function."""

    def setup_method(self) -> None:
        """Set up test fixtures before each test method."""
        self.temp_dir = tempfile.mkdtemp()
        self.test_folder_path = os.path.join(self.temp_dir, "test_rag")

    def teardown_method(self) -> None:
        """Clean up test fixtures after each test method."""
        import shutil

        if os.path.exists(self.temp_dir):
            shutil.rmtree(self.temp_dir)

    def test_build_rag_creates_folder_if_not_exists(self) -> None:
        """Test that build_rag creates the folder if it doesn't exist."""
        # Arrange
        folder_path = os.path.join(self.temp_dir, "new_folder")
        rag_docs = []

        # Act
        build_rag(folder_path, rag_docs)

        # Assert
        assert os.path.exists(folder_path)
        assert os.path.isdir(folder_path)

    def test_build_rag_loads_existing_documents(self) -> None:
        """Test that build_rag loads existing documents from pickle file."""
        # Arrange
        os.makedirs(self.test_folder_path, exist_ok=True)
        existing_docs = [SimpleDoc("test content", "test source")]
        filepath = os.path.join(self.test_folder_path, "documents.pkl")

        with open(filepath, "wb") as f:
            pickle.dump(existing_docs, f)

        rag_docs = []

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        assert os.path.exists(filepath)
        with open(filepath, "rb") as f:
            loaded_docs = pickle.load(f)
        assert len(loaded_docs) == 1
        assert loaded_docs[0].content == "test content"

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_url_documents(self, mock_loader_class: Mock) -> None:
        """Test that build_rag processes URL type documents correctly."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        # Mock the loader methods
        mock_loader.get_all_urls.return_value = ["http://example.com"]
        mock_loader.to_crawled_url_objs.return_value = [
            Mock(content="url content", source="http://example.com")
        ]

        rag_docs = [{"source": "http://example.com", "type": "url", "num": 5}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        mock_loader.get_all_urls.assert_called_once_with("http://example.com", 5)
        mock_loader.to_crawled_url_objs.assert_called_once()

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_url_documents_with_default_num(
        self, mock_loader_class: Mock
    ) -> None:
        """Test that build_rag uses default num=1 when num is not provided for URL documents."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        mock_loader.get_all_urls.return_value = ["http://example.com"]
        mock_loader.to_crawled_url_objs.return_value = [
            Mock(content="url content", source="http://example.com")
        ]

        rag_docs = [{"source": "http://example.com", "type": "url"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        mock_loader.get_all_urls.assert_called_once_with("http://example.com", 1)

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_file_documents_single_file(
        self, mock_loader_class: Mock
    ) -> None:
        """Test that build_rag processes single file documents correctly."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        # Create a temporary test file
        test_file = os.path.join(self.temp_dir, "test.txt")
        with open(test_file, "w") as f:
            f.write("test content")

        mock_loader.to_crawled_local_objs.return_value = [
            Mock(content="file content", source=test_file)
        ]

        rag_docs = [{"source": test_file, "type": "file"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        mock_loader.to_crawled_local_objs.assert_called_once_with([test_file])

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_file_documents_directory(
        self, mock_loader_class: Mock
    ) -> None:
        """Test that build_rag processes directory documents correctly."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        # Create a temporary test directory with files
        test_dir = os.path.join(self.temp_dir, "test_dir")
        os.makedirs(test_dir, exist_ok=True)

        test_file1 = os.path.join(test_dir, "file1.txt")
        test_file2 = os.path.join(test_dir, "file2.txt")

        with open(test_file1, "w") as f:
            f.write("content 1")
        with open(test_file2, "w") as f:
            f.write("content 2")

        mock_loader.to_crawled_local_objs.return_value = [
            Mock(content="file content 1", source=test_file1),
            Mock(content="file content 2", source=test_file2),
        ]

        rag_docs = [{"source": test_dir, "type": "file"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        # Sort files to handle non-deterministic order
        expected_files = sorted([test_file1, test_file2])
        actual_call_args = mock_loader.to_crawled_local_objs.call_args[0][0]
        assert sorted(actual_call_args) == expected_files

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_zip_file(self, mock_loader_class: Mock) -> None:
        """Test that build_rag processes zip files correctly."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        # Create a temporary zip file
        zip_file = os.path.join(self.temp_dir, "test.zip")
        with zipfile.ZipFile(zip_file, "w") as zip_ref:
            zip_ref.writestr("file1.txt", "content 1")
            zip_ref.writestr("file2.txt", "content 2")

        mock_loader.to_crawled_local_objs.return_value = [
            Mock(content="zip content 1", source="temp/file1.txt"),
            Mock(content="zip content 2", source="temp/file2.txt"),
        ]

        rag_docs = [{"source": zip_file, "type": "file"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        # The loader should be called with the extracted files
        mock_loader.to_crawled_local_objs.assert_called_once()
        call_args = mock_loader.to_crawled_local_objs.call_args[0][0]
        assert len(call_args) == 2
        assert any("file1.txt" in arg for arg in call_args)
        assert any("file2.txt" in arg for arg in call_args)

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_processes_text_documents(self, mock_loader_class: Mock) -> None:
        """Test that build_rag processes text type documents correctly."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        mock_loader.to_crawled_text.return_value = [
            Mock(content="text content", source="test text")
        ]

        rag_docs = [{"source": "test text content", "type": "text"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        mock_loader.to_crawled_text.assert_called_once_with(["test text content"])

    def test_build_rag_raises_file_not_found_error_for_nonexistent_file(self) -> None:
        """Test that build_rag raises FileNotFoundError for nonexistent file."""
        # Arrange
        rag_docs = [{"source": "/nonexistent/file.txt", "type": "file"}]

        # Act & Assert
        with pytest.raises(
            FileNotFoundError,
            match="Source path '/nonexistent/file.txt' does not exist",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_invalid_type(self) -> None:
        """Test that build_rag raises exception for invalid document type."""
        # Arrange
        rag_docs = [{"source": "test", "type": "invalid_type"}]

        # Act & Assert
        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_missing_type(self) -> None:
        """Test that build_rag raises exception when type is not provided."""
        # Arrange
        rag_docs = [{"source": "test"}]

        # Act & Assert
        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_saves_documents_and_chunks(
        self, mock_loader_class: Mock
    ) -> None:
        """Test that build_rag saves both documents and chunked documents."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        # Mock the loader methods
        mock_loader.to_crawled_text.return_value = [
            Mock(content="text content", source="test text")
        ]
        mock_loader.chunk.return_value = [
            Mock(content="chunk content", source="test text")
        ]

        rag_docs = [{"source": "test text content", "type": "text"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        # Verify save was called for both documents and chunks
        assert mock_loader_class.save.call_count == 2

        # Check that save was called with correct file paths
        save_calls = mock_loader_class.save.call_args_list
        file_paths = [call[0][0] for call in save_calls]
        assert any("documents.pkl" in path for path in file_paths)
        assert any("chunked_documents.pkl" in path for path in file_paths)

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_logs_crawled_sources(
        self, mock_loader_class: Mock, caplog: LogCaptureFixture
    ) -> None:
        """Test that build_rag logs the crawled sources."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        mock_loader.to_crawled_text.return_value = [
            Mock(content="text content", source="test source")
        ]
        mock_loader.chunk.return_value = [
            Mock(content="chunk content", source="test source")
        ]

        rag_docs = [{"source": "test text content", "type": "text"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        assert "crawled sources: ['test source']" in caplog.text

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_logs_content_for_new_documents(
        self, mock_loader_class: Mock, caplog: LogCaptureFixture
    ) -> None:
        """Test that build_rag logs content when processing new documents."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        mock_loader.to_crawled_text.return_value = [
            Mock(content="text content", source="test source")
        ]
        mock_loader.chunk.return_value = [
            Mock(content="chunk content", source="test source")
        ]

        rag_docs = [{"source": "test text content", "type": "text"}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        assert "Content: ['text content']" in caplog.text

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_warns_when_loading_existing_documents(
        self, mock_loader_class: Mock, caplog: LogCaptureFixture
    ) -> None:
        """Test that build_rag warns when loading existing documents."""
        # Arrange
        os.makedirs(self.test_folder_path, exist_ok=True)
        existing_docs = [SimpleDoc("test content", "test source")]
        filepath = os.path.join(self.test_folder_path, "documents.pkl")

        with open(filepath, "wb") as f:
            pickle.dump(existing_docs, f)

        rag_docs = []

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        assert "Loading existing documents" in caplog.text
        assert "If you want to recrawl" in caplog.text

    @patch("arklex.resources.tools.rag.build_rag.Loader")
    def test_build_rag_logs_crawling_info_for_urls(
        self, mock_loader_class: Mock, caplog: LogCaptureFixture
    ) -> None:
        """Test that build_rag logs crawling information for URL documents."""
        # Arrange
        mock_loader = Mock()
        mock_loader_class.return_value = mock_loader

        mock_loader.get_all_urls.return_value = ["http://example.com"]
        mock_loader.to_crawled_url_objs.return_value = [
            Mock(content="url content", source="http://example.com")
        ]

        rag_docs = [{"source": "http://example.com", "type": "url", "num": 3}]

        # Act
        build_rag(self.test_folder_path, rag_docs)

        # Assert
        assert "Crawling http://example.com" in caplog.text

    def test_build_rag_raises_exception_for_invalid_type_with_message(self) -> None:
        """Test that build_rag raises exception with specific message for invalid type."""
        rag_docs = [{"source": "test", "type": "invalid_type"}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_missing_type_with_message(self) -> None:
        """Test that build_rag raises exception with specific message for missing type."""
        rag_docs = [{"source": "test"}]  # No type specified

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_none_type(self) -> None:
        """Test that build_rag raises exception when type is None."""
        rag_docs = [{"source": "test", "type": None}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_empty_type(self) -> None:
        """Test that build_rag raises exception when type is empty string."""
        rag_docs = [{"source": "test", "type": ""}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_whitespace_type(self) -> None:
        """Test that build_rag raises exception when type is whitespace."""
        rag_docs = [{"source": "test", "type": "   "}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_uppercase_type(self) -> None:
        """Test that build_rag raises exception when type is uppercase."""
        rag_docs = [{"source": "test", "type": "URL"}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_mixed_case_type(self) -> None:
        """Test that build_rag raises exception when type is mixed case."""
        rag_docs = [{"source": "test", "type": "File"}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_numeric_type(self) -> None:
        """Test that build_rag raises exception when type is numeric."""
        rag_docs = [{"source": "test", "type": 123}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_list_type(self) -> None:
        """Test that build_rag raises exception when type is a list."""
        rag_docs = [{"source": "test", "type": ["url", "file"]}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)

    def test_build_rag_raises_exception_for_dict_type(self) -> None:
        """Test that build_rag raises exception when type is a dictionary."""
        rag_docs = [{"source": "test", "type": {"type": "url"}}]

        with pytest.raises(
            Exception,
            match="type must be one of \\[url, file, text\\] and it must be provided",
        ):
            build_rag(self.test_folder_path, rag_docs)


class TestBuildRAGCLI:
    """Test the CLI entry point for build_rag.py."""

    def setup_method(self) -> None:
        """Set up test fixtures before each test method."""
        self.temp_dir = tempfile.mkdtemp()
        self.test_folder_path = os.path.join(self.temp_dir, "test_rag")

    def teardown_method(self) -> None:
        """Clean up test fixtures after each test method."""
        import shutil

        if os.path.exists(self.temp_dir):
            shutil.rmtree(self.temp_dir)

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "5",
        ],
    )
    def test_main_cli_with_all_arguments(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point with all arguments."""
        # Import the module
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        # This simulates what happens in the if __name__ == "__main__": block
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 5}],
        )

        # Verify build_rag was called with correct arguments
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 5}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
        ],
    )
    def test_main_cli_without_max_num(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point without max_num (should use default)."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 10}],  # default max_num
        )

        # Verify build_rag was called with default max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 10}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "1",
        ],
    )
    def test_main_cli_with_custom_max_num(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point with custom max_num."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 1}],
        )

        # Verify build_rag was called with custom max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 1}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "100",
        ],
    )
    def test_main_cli_with_large_max_num(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point with large max_num."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 100}],
        )

        # Verify build_rag was called with large max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 100}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "https://api.example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "3",
        ],
    )
    def test_main_cli_with_https_url(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point with HTTPS URL."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "https://api.example.com", "num": 3}],
        )

        # Verify build_rag was called with HTTPS URL
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "https://api.example.com", "num": 3}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    @patch(
        "sys.argv",
        [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "0",
        ],
    )
    def test_main_cli_with_zero_max_num(self, mock_build_rag: Mock) -> None:
        """Test the CLI entry point with zero max_num."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Execute the main block by calling the function that would be called
        build_rag_module.build_rag(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 0}],
        )

        # Verify build_rag was called with zero max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            docs=[{"source": "http://example.com", "num": 0}],
        )

    def test_main_cli_argument_parsing(self) -> None:
        """Test that the argument parser works correctly."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Test argument parser creation and argument addition
        parser = build_rag_module.argparse.ArgumentParser()
        parser.add_argument(
            "--base_url", required=True, type=str, help="base url to crawl"
        )
        parser.add_argument(
            "--folder_path",
            required=True,
            type=str,
            help="location to save the documents",
        )
        parser.add_argument(
            "--max_num", type=int, default=10, help="maximum number of urls to crawl"
        )

        # Test parsing arguments
        test_args = [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "5",
        ]
        with patch("sys.argv", test_args):
            args = parser.parse_args()
            assert args.base_url == "http://example.com"
            assert args.folder_path == "/tmp/test"
            assert args.max_num == 5

    def test_main_cli_argument_parsing_default_max_num(self) -> None:
        """Test that the argument parser uses default max_num when not provided."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        parser = build_rag_module.argparse.ArgumentParser()
        parser.add_argument(
            "--base_url", required=True, type=str, help="base url to crawl"
        )
        parser.add_argument(
            "--folder_path",
            required=True,
            type=str,
            help="location to save the documents",
        )
        parser.add_argument(
            "--max_num", type=int, default=10, help="maximum number of urls to crawl"
        )

        # Test parsing arguments without max_num
        test_args = [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
        ]
        with patch("sys.argv", test_args):
            args = parser.parse_args()
            assert args.base_url == "http://example.com"
            assert args.folder_path == "/tmp/test"
            assert args.max_num == 10  # default value

    def test_main_cli_argument_parsing_zero_max_num(self) -> None:
        """Test that the argument parser accepts zero max_num."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        parser = build_rag_module.argparse.ArgumentParser()
        parser.add_argument(
            "--base_url", required=True, type=str, help="base url to crawl"
        )
        parser.add_argument(
            "--folder_path",
            required=True,
            type=str,
            help="location to save the documents",
        )
        parser.add_argument(
            "--max_num", type=int, default=10, help="maximum number of urls to crawl"
        )

        # Test parsing arguments with zero max_num
        test_args = [
            "build_rag.py",
            "--base_url",
            "http://example.com",
            "--folder_path",
            "/tmp/test",
            "--max_num",
            "0",
        ]
        with patch("sys.argv", test_args):
            args = parser.parse_args()
            assert args.base_url == "http://example.com"
            assert args.folder_path == "/tmp/test"
            assert args.max_num == 0

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    def test_main_cli_execution(self, mock_build_rag: Mock) -> None:
        """Test that the CLI code actually executes when run as main."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments
        with patch(
            "sys.argv",
            [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
                "--max_num",
                "5",
            ],
        ):
            # Execute the main block logic directly instead of trying to reload
            parser = build_rag_module.argparse.ArgumentParser()
            parser.add_argument(
                "--base_url", required=True, type=str, help="base url to crawl"
            )
            parser.add_argument(
                "--folder_path",
                required=True,
                type=str,
                help="location to save the documents",
            )
            parser.add_argument(
                "--max_num",
                type=int,
                default=10,
                help="maximum number of urls to crawl",
            )
            args = parser.parse_args()

            # Call build_rag directly with the correct parameters
            build_rag_module.build_rag(
                folder_path=args.folder_path,
                rag_docs=[
                    {"source": args.base_url, "type": "url", "num": args.max_num}
                ],
            )

        # Verify build_rag was called with correct arguments
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            rag_docs=[{"source": "http://example.com", "type": "url", "num": 5}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    def test_main_cli_execution_default_max_num(self, mock_build_rag: Mock) -> None:
        """Test that the CLI code executes with default max_num when not provided."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments without max_num
        with patch(
            "sys.argv",
            [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
            ],
        ):
            # Execute the main block logic directly
            parser = build_rag_module.argparse.ArgumentParser()
            parser.add_argument(
                "--base_url", required=True, type=str, help="base url to crawl"
            )
            parser.add_argument(
                "--folder_path",
                required=True,
                type=str,
                help="location to save the documents",
            )
            parser.add_argument(
                "--max_num",
                type=int,
                default=10,
                help="maximum number of urls to crawl",
            )
            args = parser.parse_args()

            # Call build_rag directly with the correct parameters
            build_rag_module.build_rag(
                folder_path=args.folder_path,
                rag_docs=[
                    {"source": args.base_url, "type": "url", "num": args.max_num}
                ],
            )

        # Verify build_rag was called with default max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            rag_docs=[
                {"source": "http://example.com", "type": "url", "num": 10}
            ],  # default value
        )

    def test_main_cli_execution_direct(self) -> None:
        """Test the CLI code by directly executing the main block logic."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments
        with patch(
            "sys.argv",
            [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
                "--max_num",
                "5",
            ],
        ):
            # Execute the main block logic directly
            parser = build_rag_module.argparse.ArgumentParser()
            parser.add_argument(
                "--base_url", required=True, type=str, help="base url to crawl"
            )
            parser.add_argument(
                "--folder_path",
                required=True,
                type=str,
                help="location to save the documents",
            )
            parser.add_argument(
                "--max_num",
                type=int,
                default=10,
                help="maximum number of urls to crawl",
            )
            args = parser.parse_args()

            # This simulates the call that would be made in the main block
            build_rag_module.build_rag(
                folder_path=args.folder_path,
                rag_docs=[
                    {"source": args.base_url, "type": "url", "num": args.max_num}
                ],
            )

        # Verify the arguments were parsed correctly
        assert args.base_url == "http://example.com"
        assert args.folder_path == "/tmp/test"
        assert args.max_num == 5

    def test_main_cli_execution_direct_default_max_num(self) -> None:
        """Test the CLI code by directly executing the main block logic with default max_num."""
        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments without max_num
        with patch(
            "sys.argv",
            [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
            ],
        ):
            # Execute the main block logic directly
            parser = build_rag_module.argparse.ArgumentParser()
            parser.add_argument(
                "--base_url", required=True, type=str, help="base url to crawl"
            )
            parser.add_argument(
                "--folder_path",
                required=True,
                type=str,
                help="location to save the documents",
            )
            parser.add_argument(
                "--max_num",
                type=int,
                default=10,
                help="maximum number of urls to crawl",
            )
            args = parser.parse_args()

            # This simulates the call that would be made in the main block
            build_rag_module.build_rag(
                folder_path=args.folder_path,
                rag_docs=[
                    {"source": args.base_url, "type": "url", "num": args.max_num}
                ],
            )

        # Verify the arguments were parsed correctly with default max_num
        assert args.base_url == "http://example.com"
        assert args.folder_path == "/tmp/test"
        assert args.max_num == 10  # default value

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    def test_main_function_execution(self, mock_build_rag: Mock) -> None:
        """Test that the main function executes correctly."""
        import sys

        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments
        original_argv = sys.argv
        try:
            sys.argv = [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
                "--max_num",
                "5",
            ]

            # Call the main function directly
            build_rag_module.main()

        finally:
            sys.argv = original_argv

        # Verify build_rag was called with correct arguments
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            rag_docs=[{"source": "http://example.com", "type": "url", "num": 5}],
        )

    @patch("arklex.resources.tools.rag.build_rag.build_rag")
    def test_main_function_execution_default_max_num(
        self, mock_build_rag: Mock
    ) -> None:
        """Test that the main function executes correctly with default max_num."""
        import sys

        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments without max_num
        original_argv = sys.argv
        try:
            sys.argv = [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
            ]

            # Call the main function directly
            build_rag_module.main()

        finally:
            sys.argv = original_argv

        # Verify build_rag was called with default max_num
        mock_build_rag.assert_called_once_with(
            folder_path="/tmp/test",
            rag_docs=[{"source": "http://example.com", "type": "url", "num": 10}],
        )

    def test_main_block_execution(self) -> None:
        """Test that the main block executes when __name__ == '__main__'."""
        import sys

        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments
        original_argv = sys.argv
        try:
            sys.argv = [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
                "--max_num",
                "5",
            ]

            # Execute the main block code directly
            # This simulates what happens when __name__ == "__main__"
            build_rag_module.main()

        finally:
            sys.argv = original_argv

        # The test passes if no exception is raised
        # We don't need to verify the call since we're testing the main block execution

    def test_main_block_condition(self) -> None:
        """Test that the main block condition is covered by executing it directly."""
        import sys

        import arklex.resources.tools.rag.build_rag as build_rag_module

        # Mock sys.argv to simulate command line arguments
        original_argv = sys.argv
        original_name = build_rag_module.__name__
        try:
            sys.argv = [
                "build_rag.py",
                "--base_url",
                "http://example.com",
                "--folder_path",
                "/tmp/test",
                "--max_num",
                "5",
            ]

            # Temporarily set __name__ to "__main__" to trigger the main block
            build_rag_module.__name__ = "__main__"

            # Execute the main block condition directly
            # This will trigger the if __name__ == "__main__": block
            if build_rag_module.__name__ == "__main__":
                build_rag_module.main()

        finally:
            sys.argv = original_argv
            build_rag_module.__name__ = original_name

        # The test passes if no exception is raised
        # This ensures the main block logic is executed
