# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_utils.ipynb.

# %% auto 0
__all__ = ['modjoin', 'is_mod', 'is_mod_imp', 'is_var_imp', 'is_mod_or_var', 'is_mod_avail', 'loader_from_spec',
           'module_from_str', 'getmodule']

# %% ../nbs/01_utils.ipynb 3
import sys
from importlib.util import (LazyLoader, find_spec, module_from_spec)
from typing import Optional, Any

# %% ../nbs/01_utils.ipynb 5
from ipos.types import (
    Module, Loader, ModuleSpec, VariableDict, 
    FallbackType, ImpItemsType, ImpSubSpecType
)

# %% ../nbs/01_utils.ipynb 7
def modjoin(*parts: str) -> str:
    """
    Join module name parts into a valid module path.

    Parameters
    ----------
    *parts : str
        Name parts of the module to be joined.

    Returns
    -------
    str
        Valid module path.
    """
    return ('.'.join(parts)).rstrip('.')

def is_mod(module: Any) -> bool:
    """
    Check if a given object is a module.

    Parameters
    ----------
    module : Any
        Object to be checked.

    Returns
    -------
    bool
        True if the object is a module, False otherwise.
    """
    return isinstance(module, Module)

def is_mod_imp(name: str) -> bool:
    """
    Check if a given module name is in the system module list.

    Parameters
    ----------
    name : str
        Name of the module to be checked.

    Returns
    -------
    bool
        True if the module is in the system module list, False otherwise.
    """
    return name in sys.modules

def is_var_imp(name: str) -> bool:
    """
    Check if a given variable name exists in the global namespace.

    Parameters
    ----------
    name : str
        Name of the variable to be checked.

    Returns
    -------
    bool
        True if the variable exists in the global namespace, False otherwise.
    """
    return name in globals()


def is_mod_or_var(name: str):
    """
    Check if a given name exists either as a module or a variable.

    Parameters
    ----------
    name : str
        Name to be checked.

    Returns
    -------
    bool
        True if the name exists either as a module or a variable, False otherwise.
    """
    return is_var_imp(name) or is_mod_imp(name)

def is_mod_avail(name: str) -> bool:
    """
    Check if a given module name is available for import.

    Parameters
    ----------
    name : str
        Name of the module to be checked.

    Returns
    -------
    bool
        True if the module is available for import, False otherwise.
    """
    if is_mod_imp(name): return True
    elif find_spec(name) is not None: return True
    return False

def loader_from_spec(spec: ModuleSpec, lazy: bool = False) -> Loader:
    """
    Get a loader from a given module specification.

    Parameters
    ----------
    spec : ModuleSpec
        Module specification from which to get the loader.
    lazy : bool, optional
        If True, return a LazyLoader; if False, return the original loader.

    Returns
    -------
    Loader
        Loader for the module.
    """
    loader = spec.loader
    if lazy: loader = LazyLoader(loader)
    return loader

def module_from_str(
    name: str, 
    lazy: bool = False, 
    alias: Optional[str] = None,
    inject: Optional[bool] = True,
    inject_both: Optional[bool] = False,
) -> Module:
    """
    Import a module from a given string name.

    Parameters
    ----------
    name : str
        Name of the module to import.
    lazy : bool, optional
        If True, use lazy import; if False, use regular import.
    alias : str, optional
        Alias to use for the module.
    inject : bool, optional
        If True, inject the module into sys.modules.
    inject_both : bool, optional
        If True, inject the module into sys.modules with both its original name and alias.

    Returns
    -------
    Module
        The imported module.
    """
    spec = find_spec(name)    
    loader = loader_from_spec(spec, lazy=lazy)
    
    module = module_from_spec(spec)    
    modname = name if alias is None else alias
    
    if inject_both: inject = True
    if inject:
        sys.modules[modname] = module
    if inject_both:
        sys.modules[name] = module

    loader.exec_module(module)
    return module

def getmodule(
    name: str, lazy: Optional[bool] = False, 
    alias: Optional[str] = None, 
    inject: Optional[bool] = True,
    inject_both: Optional[bool] = False,
) -> Module:
    """
    Get a module by name, importing it if necessary.

    Parameters
    ----------
    name : str
        Name of the module to get.
    lazy : bool, optional
        If True, use lazy import; if False, use regular import.
    alias : str, optional
        Alias to use for the module.
    inject : bool, optional
        If True, inject the module into sys.modules.
    inject_both : bool, optional
        If True, inject the module into sys.modules with both its original name and alias.

    Returns
    -------
    Module
        The requested module.

    Raises
    ------
    ModuleNotFoundError
        If the module cannot be found.
    """
    if is_mod_avail(name):
        return module_from_str(name, lazy, alias, inject, inject_both)
    raise ModuleNotFoundError

