import argparse
import os
from .recognizer import SimpleFaceRecognizer
from .add_faces import add_faces_from_folder
from .remove_faces import remove_face_database


def main():
    parser = argparse.ArgumentParser(
        description="facial_recognition CLI Tool — Add, recognize, or remove faces"
    )
    subparsers = parser.add_subparsers(dest="command")

    # --- Subcommand: add_faces ---
    add_parser = subparsers.add_parser(
        "add_faces",
        help="Add all faces from the current folder to the database"
    )
    add_parser.add_argument(
        "--threshold",
        type=float,
        default=0.3,
        help="Cosine similarity threshold for adding faces (default: 0.3)"
    )

    # --- Subcommand: recognize ---
    rec_parser = subparsers.add_parser(
        "recognize",
        help="Recognize faces from an image"
    )
    rec_parser.add_argument(
        "image_path",
        help="Path to the image file for recognition"
    )
    rec_parser.add_argument(
        "--threshold",
        type=float,
        default=0.3,
        help="Cosine similarity threshold for recognition (default: 0.3)"
    )

    # --- Subcommand: remove_faces ---
    subparsers.add_parser(
        "remove_faces",
        help="Remove the local face database file"
    )

    # Parse arguments
    args = parser.parse_args()

    # --- No command provided ---
    if not args.command:
        parser.print_help()
        return

    # --- Handle commands ---
    if args.command == "add_faces":
        folder = os.getcwd()
        add_faces_from_folder(folder)

    elif args.command == "recognize":
        recognizer = SimpleFaceRecognizer()
        recognizer.recognize_file(args.image_path, threshold=args.threshold)

    elif args.command == "remove_faces":
        remove_face_database()


if __name__ == "__main__":
    main()
