import tkinter as tk
from tkinter import ttk
from typing import Callable, List, Dict, Any, Optional, Tuple

class FormField:
    """表单字段组件 - 标签 + 输入框"""
    
    def __init__(self, parent, label_text, widget_type="entry", **kwargs):
        self.frame = ttk.Frame(parent)
        self.label = ttk.Label(self.frame, text=label_text, width=15, anchor="e")
        self.label.pack(side="left", padx=(0, 10))
        
        if widget_type == "entry":
            self.widget = ttk.Entry(self.frame, **kwargs)
        elif widget_type == "combobox":
            self.widget = ttk.Combobox(self.frame, **kwargs)
        elif widget_type == "text":
            self.widget = tk.Text(self.frame, height=3, **kwargs)
        elif widget_type == "spinbox":
            self.widget = ttk.Spinbox(self.frame, from_=0, to=100, **kwargs)
            
        self.widget.pack(side="left", fill="x", expand=True)
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)
    
    def get(self):
        if hasattr(self.widget, 'get'):
            return self.widget.get()
        elif isinstance(self.widget, tk.Text):
            return self.widget.get("1.0", "end-1c")
        return None
    
    def set(self, value):
        if hasattr(self.widget, 'delete') and hasattr(self.widget, 'insert'):
            self.widget.delete(0, tk.END)
            self.widget.insert(0, str(value))
        elif isinstance(self.widget, tk.Text):
            self.widget.delete("1.0", tk.END)
            self.widget.insert("1.0", str(value))
        elif hasattr(self.widget, 'set'):
            self.widget.set(value)

class LoginForm:
    """登录表单组件"""
    
    def __init__(self, parent, on_login: Callable = None, title: str = "用户登录"):
        self.frame = ttk.Frame(parent)
        self.on_login = on_login
        
        # 标题
        self.title_label = ttk.Label(self.frame, text=title, font=("Arial", 14, "bold"))
        self.title_label.pack(pady=10)
        
        # 用户名字段
        self.username_field = FormField(self.frame, "用户名:", "entry")
        self.username_field.pack(fill="x", padx=20, pady=5)
        
        # 密码字段
        self.password_field = FormField(self.frame, "密码:", "entry", show="*")
        self.password_field.pack(fill="x", padx=20, pady=5)
        
        # 记住我复选框
        self.remember_var = tk.BooleanVar()
        self.remember_check = ttk.Checkbutton(
            self.frame, 
            text="记住密码", 
            variable=self.remember_var
        )
        self.remember_check.pack(pady=5)
        
        # 登录按钮
        self.login_button = ttk.Button(
            self.frame, 
            text="登录", 
            command=self._on_login_click
        )
        self.login_button.pack(pady=10)
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)
    
    def _on_login_click(self):
        if self.on_login:
            credentials = {
                'username': self.username_field.get(),
                'password': self.password_field.get(),
                'remember': self.remember_var.get()
            }
            self.on_login(credentials)
    
    def get_credentials(self):
        return {
            'username': self.username_field.get(),
            'password': self.password_field.get(),
            'remember': self.remember_var.get()
        }

class DataTable:
    """简单数据表格组件"""
    
    def __init__(self, parent, columns: List[str], data: List[List] = None, height=10):
        self.frame = ttk.Frame(parent)
        self.columns = columns
        self.data = data or []
        
        # 创建Treeview
        self.tree = ttk.Treeview(self.frame, columns=columns, show="headings", height=height)
        
        # 设置列标题
        for col in columns:
            self.tree.heading(col, text=col)
            self.tree.column(col, width=100)
        
        # 滚动条
        scrollbar = ttk.Scrollbar(self.frame, orient="vertical", command=self.tree.yview)
        self.tree.configure(yscrollcommand=scrollbar.set)
        
        # 布局
        self.tree.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="right", fill="y")
        
        # 添加初始数据
        self._refresh_data()
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)
    
    def add_row(self, row_data: List):
        """添加一行数据"""
        self.data.append(row_data)
        self._refresh_data()
    
    def set_data(self, data: List[List]):
        """设置所有数据"""
        self.data = data
        self._refresh_data()
    
    def clear(self):
        """清空数据"""
        self.data = []
        self._refresh_data()
    
    def get_selected_row(self):
        """获取选中的行数据"""
        selection = self.tree.selection()
        if selection:
            item = selection[0]
            return self.tree.item(item)['values']
        return None
    
    def _refresh_data(self):
        """刷新表格数据"""
        # 清空现有数据
        for item in self.tree.get_children():
            self.tree.delete(item)
        
        # 添加新数据
        for row in self.data:
            self.tree.insert("", "end", values=row)

class ProgressDialog:
    """进度对话框组件"""
    
    def __init__(self, parent, title="处理中", message="请稍候..."):
        self.top = tk.Toplevel(parent)
        self.top.title(title)
        self.top.geometry("300x100")
        self.top.transient(parent)
        self.top.grab_set()
        
        # 消息标签
        self.message_label = ttk.Label(self.top, text=message)
        self.message_label.pack(pady=10)
        
        # 进度条
        self.progress = ttk.Progressbar(self.top, mode='indeterminate')
        self.progress.pack(fill="x", padx=20, pady=5)
        
        # 开始动画
        self.progress.start(10)
    
    def close(self):
        """关闭对话框"""
        self.progress.stop()
        self.top.destroy()

class CardWidget:
    """卡片式组件"""
    
    def __init__(self, parent, title: str, content: str = ""):
        self.frame = ttk.Frame(parent, relief="solid", borderwidth=1)
        
        # 标题
        self.title_label = ttk.Label(
            self.frame, 
            text=title, 
            font=("Arial", 12, "bold"),
            background="#e8e8e8"
        )
        self.title_label.pack(fill="x", padx=5, pady=5)
        
        # 内容
        self.content_label = ttk.Label(
            self.frame, 
            text=content, 
            wraplength=200
        )
        self.content_label.pack(fill="x", padx=10, pady=10)
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)
    
    def set_content(self, content: str):
        """设置内容"""
        self.content_label.config(text=content)
    
    def set_title(self, title: str):
        """设置标题"""
        self.title_label.config(text=title)

class StatusBar:
    """状态栏组件"""
    
    def __init__(self, parent, initial_text="就绪"):
        self.frame = ttk.Frame(parent)
        self.status_var = tk.StringVar(value=initial_text)
        
        self.status_label = ttk.Label(
            self.frame, 
            textvariable=self.status_var,
            relief="sunken",
            anchor="w"
        )
        self.status_label.pack(fill="x", padx=5, pady=2)
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)
    
    def set_text(self, text: str):
        """设置状态文本"""
        self.status_var.set(text)
    
    def clear(self):
        """清空状态文本"""
        self.status_var.set("")

class Toolbar:
    """工具栏组件"""
    
    def __init__(self, parent, buttons: List[Tuple[str, Callable]]):
        self.frame = ttk.Frame(parent)
        
        for text, command in buttons:
            button = ttk.Button(self.frame, text=text, command=command)
            button.pack(side="left", padx=2)
    
    def pack(self, **kwargs):
        self.frame.pack(**kwargs)