import tkinter as tk
from typing import Dict, Any, Optional, Callable, List
from .widgets import WidgetFactory
from .layout import LayoutManager
from .styles import StyleManager
from .dialogs import DialogManager

class SimpleGUI:
    """
    SimpleGUI 主类
    提供创建和管理GUI的简单接口
    """
    
    def __init__(self, title="SimpleGUI", size="400x300"):
        self.root = tk.Tk()
        self.root.title(title)
        self.root.geometry(size)
        
        # 初始化各个管理器
        self.widgets = WidgetFactory(self.root)
        self.layout = LayoutManager(self.root)
        self.style = StyleManager()
        self.dialogs = DialogManager(self.root)
        
        # 存储组件的字典
        self._components = {}
        
        # 应用默认样式
        self.style.apply_default_style()
        
    def run(self):
        """运行GUI主循环"""
        self.root.mainloop()
        
    def set_title(self, title: str):
        """设置窗口标题"""
        self.root.title(title)
        
    def set_size(self, width: int, height: int):
        """设置窗口大小"""
        self.root.geometry(f"{width}x{height}")
        
    def set_icon(self, icon_path: str):
        """设置窗口图标"""
        try:
            self.root.iconbitmap(icon_path)
        except:
            pass  # 图标设置失败时静默处理
    
    # 快捷方法 - 委托给widgets
    def add_label(self, text: str, name: str = None, **kwargs):
        """添加标签"""
        widget = self.widgets.create_label(text, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_button(self, text: str, command: Callable = None, name: str = None, **kwargs):
        """添加按钮"""
        widget = self.widgets.create_button(text, command, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_entry(self, default_text: str = "", name: str = None, **kwargs):
        """添加输入框"""
        widget = self.widgets.create_entry(default_text, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_textbox(self, default_text: str = "", height: int = 5, name: str = None, **kwargs):
        """添加文本框"""
        widget = self.widgets.create_textbox(default_text, height, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_combobox(self, values: List, default: str = None, name: str = None, **kwargs):
        """添加下拉框"""
        widget = self.widgets.create_combobox(values, default, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_checkbox(self, text: str, default: bool = False, name: str = None, **kwargs):
        """添加复选框"""
        widget = self.widgets.create_checkbox(text, default, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_radiobuttons(self, options: List, default: str = None, name: str = None, **kwargs):
        """添加单选按钮"""
        widget = self.widgets.create_radiobuttons(options, default, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_frame(self, name: str = None, **kwargs):
        """添加框架"""
        widget = self.widgets.create_frame(name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_separator(self, **kwargs):
        """添加分隔线"""
        return self.widgets.create_separator(**kwargs)
    
    def add_spinbox(self, from_: int, to: int, default: int = None, name: str = None, **kwargs):
        """添加数字选择框"""
        widget = self.widgets.create_spinbox(from_, to, default, name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    def add_progressbar(self, name: str = None, **kwargs):
        """添加进度条"""
        widget = self.widgets.create_progressbar(name, **kwargs)
        if name:
            self._components[name] = widget
        return widget
    
    # 布局方法
    def set_grid_layout(self, rows: int, columns: int):
        """设置为网格布局"""
        self.layout.set_grid(rows, columns)
    
    def set_pack_layout(self):
        """设置为包布局"""
        self.layout.set_pack()
    
    # 对话框方法
    def show_info(self, title: str, message: str):
        """显示信息对话框"""
        return self.dialogs.show_info(title, message)
    
    def show_error(self, title: str, message: str):
        """显示错误对话框"""
        return self.dialogs.show_error(title, message)
    
    def show_warning(self, title: str, message: str):
        """显示警告对话框"""
        return self.dialogs.show_warning(title, message)
    
    def ask_question(self, title: str, question: str) -> bool:
        """显示问题对话框"""
        return self.dialogs.ask_question(title, question)
    
    def ask_ok_cancel(self, title: str, question: str) -> bool:
        """显示确定取消对话框"""
        return self.dialogs.ask_ok_cancel(title, question)
    
    def ask_yes_no(self, title: str, question: str) -> bool:
        """显示是否对话框"""
        return self.dialogs.ask_yes_no(title, question)
    
    def open_file(self, title: str = "打开文件", filetypes: List = None) -> str:
        """打开文件对话框"""
        return self.dialogs.open_file(title, filetypes)
    
    def save_file(self, title: str = "保存文件", filetypes: List = None) -> str:
        """保存文件对话框"""
        return self.dialogs.save_file(title, filetypes)
    
    def select_directory(self, title: str = "选择目录") -> str:
        """选择目录对话框"""
        return self.dialogs.select_directory(title)
    
    # 组件访问
    def get_component(self, name: str):
        """通过名称获取组件"""
        return self._components.get(name)
    
    def update_component(self, name: str, **kwargs):
        """更新组件属性"""
        component = self.get_component(name)
        if component:
            for key, value in kwargs.items():
                if hasattr(component, key):
                    getattr(component, key)(value)
                else:
                    component.config(**{key: value})
    
    def destroy_component(self, name: str):
        """销毁组件"""
        component = self.get_component(name)
        if component:
            component.destroy()
            del self._components[name]