import tkinter as tk
from tkinter import messagebox, filedialog, simpledialog
from typing import List, Optional, Tuple

class DialogManager:
    """
    对话框管理器
    负责管理各种系统对话框
    """
    
    def __init__(self, parent):
        self.parent = parent
    
    def show_info(self, title: str, message: str):
        """显示信息对话框"""
        return messagebox.showinfo(title, message)
    
    def show_error(self, title: str, message: str):
        """显示错误对话框"""
        return messagebox.showerror(title, message)
    
    def show_warning(self, title: str, message: str):
        """显示警告对话框"""
        return messagebox.showwarning(title, message)
    
    def ask_question(self, title: str, question: str) -> bool:
        """显示问题对话框"""
        return messagebox.askyesno(title, question)
    
    def ask_ok_cancel(self, title: str, question: str) -> bool:
        """显示确定取消对话框"""
        return messagebox.askokcancel(title, question)
    
    def ask_yes_no(self, title: str, question: str) -> bool:
        """显示是否对话框"""
        return messagebox.askyesno(title, question)
    
    def ask_retry_cancel(self, title: str, question: str) -> bool:
        """显示重试取消对话框"""
        return messagebox.askretrycancel(title, question)
    
    def open_file(self, title: str = "打开文件", 
                  filetypes: List[Tuple[str, str]] = None) -> Optional[str]:
        """打开文件对话框"""
        if filetypes is None:
            filetypes = [("所有文件", "*.*")]
            
        return filedialog.askopenfilename(
            title=title,
            filetypes=filetypes
        )
    
    def save_file(self, title: str = "保存文件",
                  filetypes: List[Tuple[str, str]] = None) -> Optional[str]:
        """保存文件对话框"""
        if filetypes is None:
            filetypes = [("所有文件", "*.*")]
            
        return filedialog.asksaveasfilename(
            title=title,
            filetypes=filetypes,
            defaultextension=filetypes[0][1] if filetypes else ""
        )
    
    def select_directory(self, title: str = "选择目录") -> Optional[str]:
        """选择目录对话框"""
        return filedialog.askdirectory(title=title)
    
    def ask_string(self, title: str, prompt: str, initial_value: str = "") -> Optional[str]:
        """询问字符串输入"""
        return simpledialog.askstring(title, prompt, initialvalue=initial_value)
    
    def ask_integer(self, title: str, prompt: str, initial_value: int = 0, 
                   min_value: int = None, max_value: int = None) -> Optional[int]:
        """询问整数输入"""
        return simpledialog.askinteger(title, prompt, initialvalue=initial_value,
                                     minvalue=min_value, maxvalue=max_value)
    
    def ask_float(self, title: str, prompt: str, initial_value: float = 0.0,
                 min_value: float = None, max_value: float = None) -> Optional[float]:
        """询问浮点数输入"""
        return simpledialog.askfloat(title, prompt, initialvalue=initial_value,
                                   minvalue=min_value, maxvalue=max_value)
    
    def show_custom_dialog(self, title: str, message: str, buttons: List[str]) -> str:
        """显示自定义按钮的对话框"""
        return messagebox.askquestion(title, message, icon='info', 
                                    type=messagebox.OKCANCEL, default=messagebox.OK)