import tkinter as tk
from tkinter import ttk
from typing import Dict, List, Tuple, Any

class LayoutManager:
    """
    布局管理器
    负责管理组件的布局和排列
    """
    
    def __init__(self, root):
        self.root = root
        self.current_layout = "pack"  # 当前布局方式
        self.grid_config = {"rows": 1, "columns": 1}
        self._frames = {}  # 存储框架
    
    def set_pack(self):
        """设置为包布局"""
        self.current_layout = "pack"
    
    def set_grid(self, rows: int = 1, columns: int = 1):
        """设置为网格布局"""
        self.current_layout = "grid"
        self.grid_config = {"rows": rows, "columns": columns}
    
    def create_grid_frame(self, name: str, rows: int, columns: int, **kwargs) -> ttk.Frame:
        """创建网格布局的框架"""
        frame = ttk.Frame(self.root, **kwargs)
        
        # 配置网格权重
        for i in range(rows):
            frame.grid_rowconfigure(i, weight=1)
        for j in range(columns):
            frame.grid_columnconfigure(j, weight=1)
        
        frame.pack(fill="both", expand=True, padx=10, pady=10)
        
        if name:
            self._frames[name] = frame
            
        return frame
    
    def create_tabbed_frame(self, names: List[str], name: str = None, **kwargs) -> ttk.Notebook:
        """创建标签页框架"""
        notebook = ttk.Notebook(self.root, **kwargs)
        
        tabs = {}
        for tab_name in names:
            tab_frame = ttk.Frame(notebook)
            notebook.add(tab_frame, text=tab_name)
            tabs[tab_name] = tab_frame
        
        notebook.pack(fill="both", expand=True, padx=10, pady=10)
        
        if name:
            self._frames[name] = {"notebook": notebook, "tabs": tabs}
            
        return notebook, tabs
    
    def create_scrollable_frame(self, name: str = None, **kwargs) -> Tuple[ttk.Frame, ttk.Scrollbar]:
        """创建可滚动的框架"""
        # 主框架
        main_frame = ttk.Frame(self.root, **kwargs)
        main_frame.pack(fill="both", expand=True)
        
        # 创建Canvas和滚动条
        canvas = tk.Canvas(main_frame)
        scrollbar = ttk.Scrollbar(main_frame, orient="vertical", command=canvas.yview)
        
        # 可滚动的内部框架
        scrollable_frame = ttk.Frame(canvas)
        
        # 配置Canvas
        scrollable_frame.bind(
            "<Configure>",
            lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
        )
        
        canvas.create_window((0, 0), window=scrollable_frame, anchor="nw")
        canvas.configure(yscrollcommand=scrollbar.set)
        
        # 布局
        canvas.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="right", fill="y")
        
        if name:
            self._frames[name] = {
                "main_frame": main_frame,
                "canvas": canvas,
                "scrollable_frame": scrollable_frame,
                "scrollbar": scrollbar
            }
            
        return scrollable_frame
    
    def create_paned_window(self, orientation: str = "horizontal", name: str = None, **kwargs) -> ttk.PanedWindow:
        """创建可调整的面板窗口"""
        paned = ttk.PanedWindow(self.root, orient=orientation, **kwargs)
        paned.pack(fill="both", expand=True, padx=10, pady=10)
        
        if name:
            self._frames[name] = paned
            
        return paned
    
    def get_frame(self, name: str):
        """通过名称获取框架"""
        return self._frames.get(name)
    
    def arrange_in_grid(self, widget, row: int, column: int, **kwargs):
        """在网格中排列组件"""
        if self.current_layout == "grid":
            widget.grid(row=row, column=column, **kwargs)
        else:
            widget.pack(**kwargs)