"""
SimpleGUI 主模块
提供库的入口点和示例代码
"""

import sys
import os

# 添加当前目录到Python路径，确保可以导入其他模块
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

from .core import SimpleGUI
from .widgets import WidgetFactory
from .layout import LayoutManager
from .styles import StyleManager
from .dialogs import DialogManager
from .components import (
    FormField, 
    LoginForm, 
    DataTable, 
    ProgressDialog, 
    CardWidget,
    StatusBar,
    Toolbar
)
from .utils import Validation, FileHandler

def demo_basic():
    """基础组件演示"""
    gui = SimpleGUI("SimpleGUI - 基础演示", "500x400")
    
    gui.add_label("SimpleGUI 基础组件演示", font=("Arial", 14, "bold"))
    gui.add_separator()
    
    # 输入组件
    gui.add_label("输入组件示例:")
    gui.add_entry("默认文本", name="entry_demo")
    gui.add_combobox(["选项1", "选项2", "选项3"], name="combobox_demo")
    gui.add_spinbox(0, 100, 50, name="spinbox_demo")
    
    # 选择组件
    gui.add_label("选择组件示例:")
    gui.add_checkbox("复选框示例", name="checkbox_demo")
    gui.add_radiobuttons([
        ("选项A", "A"),
        ("选项B", "B"),
        ("选项C", "C")
    ], name="radiobuttons_demo")
    
    # 按钮
    def show_inputs():
        entry_value = gui.get_component("entry_demo").get()
        combobox_value = gui.get_component("combobox_demo").get()
        spinbox_value = gui.get_component("spinbox_demo").get()
        checkbox_value = gui.get_component("checkbox_demo").get()
        radio_value = gui.get_component("radiobuttons_demo").get()
        
        message = f"""
        输入值:
        文本框: {entry_value}
        下拉框: {combobox_value}
        数字框: {spinbox_value}
        复选框: {checkbox_value}
        单选按钮: {radio_value}
        """
        gui.show_info("输入结果", message)
    
    gui.add_button("显示输入值", show_inputs)
    gui.add_button("退出", gui.root.quit)
    
    gui.run()

def demo_advanced():
    """高级组件演示"""
    gui = SimpleGUI("SimpleGUI - 高级组件演示", "600x500")
    
    # 使用预定义组件
    gui.add_label("高级组件演示", font=("Arial", 16, "bold"))
    gui.add_separator()
    
    # 登录表单
    def on_login(credentials):
        gui.show_info("登录信息", f"用户名: {credentials['username']}\n记住密码: {credentials['remember']}")
    
    login_form = LoginForm(gui.root, on_login, "用户登录")
    login_form.pack(fill="x", padx=20, pady=10)
    
    gui.add_separator()
    
    # 数据表格
    columns = ["ID", "姓名", "年龄", "城市"]
    data = [
        [1, "张三", 25, "北京"],
        [2, "李四", 30, "上海"],
        [3, "王五", 28, "广州"]
    ]
    table = DataTable(gui.root, columns, data, height=5)
    table.pack(fill="both", expand=True, padx=20, pady=10)
    
    # 卡片组件
    card_frame = gui.add_frame("card_frame")
    gui.widgets.set_parent(card_frame)
    
    card1 = CardWidget(card_frame, "卡片1", "这是第一个卡片的内容")
    card1.pack(side="left", fill="both", expand=True, padx=5, pady=5)
    
    card2 = CardWidget(card_frame, "卡片2", "这是第二个卡片的内容")
    card2.pack(side="left", fill="both", expand=True, padx=5, pady=5)
    
    gui.widgets.reset_parent()
    
    # 状态栏
    status_bar = StatusBar(gui.root, "就绪")
    status_bar.pack(side="bottom", fill="x")
    
    gui.run()

def demo_form():
    """表单演示"""
    gui = SimpleGUI("SimpleGUI - 表单演示", "500x450")
    
    gui.add_label("用户注册表单", font=("Arial", 16, "bold"))
    gui.add_separator()
    
    # 创建表单框架
    form_frame = gui.add_frame("form_frame")
    gui.widgets.set_parent(form_frame)
    
    # 表单字段
    username_field = FormField(form_frame, "用户名:", "entry")
    username_field.pack(fill="x", pady=5)
    
    email_field = FormField(form_frame, "邮箱:", "entry")
    email_field.pack(fill="x", pady=5)
    
    age_field = FormField(form_frame, "年龄:", "spinbox")
    age_field.pack(fill="x", pady=5)
    
    city_field = FormField(form_frame, "城市:", "combobox")
    city_field.widget['values'] = ["北京", "上海", "广州", "深圳", "杭州"]
    city_field.pack(fill="x", pady=5)
    
    bio_field = FormField(form_frame, "个人简介:", "text")
    bio_field.pack(fill="x", pady=5)
    
    gui.widgets.reset_parent()
    
    def submit_form():
        """提交表单"""
        username = username_field.get()
        email = email_field.get()
        age = age_field.get()
        city = city_field.get()
        bio = bio_field.get()
        
        # 验证
        errors = []
        if not Validation.is_required(username):
            errors.append("用户名不能为空")
        if not Validation.is_email(email):
            errors.append("邮箱格式不正确")
        if not Validation.is_integer(age) or not (1 <= int(age) <= 150):
            errors.append("年龄必须是1-150之间的整数")
        
        if errors:
            gui.show_error("表单错误", "\n".join(errors))
        else:
            message = f"""
            注册信息:
            用户名: {username}
            邮箱: {email}
            年龄: {age}
            城市: {city}
            个人简介: {bio}
            """
            gui.show_info("注册成功", message)
    
    gui.add_button("提交注册", submit_form)
    gui.run()

def demo_dialogs():
    """对话框演示"""
    gui = SimpleGUI("SimpleGUI - 对话框演示", "400x300")
    
    gui.add_label("对话框功能演示", font=("Arial", 14, "bold"))
    gui.add_separator()
    
    def show_info_dialog():
        gui.show_info("信息", "这是一个信息对话框")
    
    def show_error_dialog():
        gui.show_error("错误", "这是一个错误对话框")
    
    def show_warning_dialog():
        gui.show_warning("警告", "这是一个警告对话框")
    
    def show_question_dialog():
        result = gui.ask_question("问题", "您确定要继续吗？")
        gui.show_info("结果", f"用户选择: {'是' if result else '否'}")
    
    def show_file_dialog():
        file_path = gui.open_file("选择文件", [("文本文件", "*.txt"), ("所有文件", "*.*")])
        if file_path:
            gui.show_info("文件选择", f"选择的文件: {file_path}")
    
    def show_directory_dialog():
        dir_path = gui.select_directory("选择目录")
        if dir_path:
            gui.show_info("目录选择", f"选择的目录: {dir_path}")
    
    # 对话框按钮
    gui.add_button("信息对话框", show_info_dialog)
    gui.add_button("错误对话框", show_error_dialog)
    gui.add_button("警告对话框", show_warning_dialog)
    gui.add_button("问题对话框", show_question_dialog)
    gui.add_button("文件选择", show_file_dialog)
    gui.add_button("目录选择", show_directory_dialog)
    
    gui.run()

def demo_layout():
    """布局演示"""
    gui = SimpleGUI("SimpleGUI - 布局演示", "700x500")
    
    gui.add_label("布局管理器演示", font=("Arial", 16, "bold"))
    gui.add_separator()
    
    # 创建标签页
    notebook, tabs = gui.layout.create_tabbed_frame(["网格布局", "滚动布局", "面板布局"], "notebook_demo")
    
    # 网格布局标签页
    grid_frame = tabs["网格布局"]
    gui.widgets.set_parent(grid_frame)
    
    for i in range(3):
        for j in range(3):
            label = gui.add_label(f"单元格 ({i},{j})")
            label.grid(row=i, column=j, padx=5, pady=5, sticky="nsew")
    
    gui.widgets.reset_parent()
    
    # 滚动布局标签页
    scroll_frame = gui.layout.create_scrollable_frame("scroll_frame")
    gui.widgets.set_parent(scroll_frame)
    
    for i in range(20):
        gui.add_label(f"滚动项目 {i+1}")
    
    gui.widgets.reset_parent()
    
    # 面板布局标签页
    paned = gui.layout.create_paned_window("horizontal", "paned_demo")
    
    left_frame = ttk.Frame(paned)
    right_frame = ttk.Frame(paned)
    
    paned.add(left_frame, weight=1)
    paned.add(right_frame, weight=1)
    
    # 左侧面板
    gui.widgets.set_parent(left_frame)
    gui.add_label("左侧面板", font=("Arial", 12, "bold"))
    for i in range(5):
        gui.add_button(f"左侧按钮 {i+1}", lambda i=i: print(f"左侧 {i+1}"))
    
    # 右侧面板
    gui.widgets.set_parent(right_frame)
    gui.add_label("右侧面板", font=("Arial", 12, "bold"))
    for i in range(5):
        gui.add_button(f"右侧按钮 {i+1}", lambda i=i: print(f"右侧 {i+1}"))
    
    gui.widgets.reset_parent()
    
    gui.run()

def main():
    """主函数 - 显示演示选择界面"""
    choice_gui = SimpleGUI("SimpleGUI 演示选择", "300x400")
    
    choice_gui.add_label("SimpleGUI 库演示", font=("Arial", 16, "bold"))
    choice_gui.add_separator()
    choice_gui.add_label("请选择要运行的演示:")
    
    # 演示选择按钮
    choice_gui.add_button("1. 基础组件演示", demo_basic)
    choice_gui.add_button("2. 高级组件演示", demo_advanced)
    choice_gui.add_button("3. 表单演示", demo_form)
    choice_gui.add_button("4. 对话框演示", demo_dialogs)
    choice_gui.add_button("5. 布局演示", demo_layout)
    choice_gui.add_separator()
    choice_gui.add_button("退出", choice_gui.root.quit)
    
    choice_gui.run()

# 导出主要函数
__all__ = [
    'main',
    'demo_basic',
    'demo_advanced',
    'demo_form',
    'demo_dialogs',
    'demo_layout'
]

if __name__ == "__main__":
    main()