import tkinter as tk
from tkinter import ttk
from typing import Dict, Any, List

class StyleManager:
    """
    样式管理器
    负责管理GUI的外观和主题
    """
    
    def __init__(self):
        self.style = ttk.Style()
        self.themes = {}
        self.current_theme = "default"
        
    def apply_default_style(self):
        """应用默认样式"""
        # 配置基础样式
        self.style.configure("TFrame", background="#f0f0f0")
        self.style.configure("TLabel", background="#f0f0f0", font=("Arial", 10))
        self.style.configure("TButton", font=("Arial", 10))
        self.style.configure("TEntry", font=("Arial", 10))
        self.style.configure("TCombobox", font=("Arial", 10))
        
        # 存储默认主题
        self.themes["default"] = {
            "TFrame": {"background": "#f0f0f0"},
            "TLabel": {"background": "#f0f0f0", "font": ("Arial", 10)},
            "TButton": {"font": ("Arial", 10)},
            "TEntry": {"font": ("Arial", 10)},
            "TCombobox": {"font": ("Arial", 10)}
        }
    
    def create_theme(self, name: str, styles: Dict[str, Dict]):
        """创建新主题"""
        self.themes[name] = styles
    
    def apply_theme(self, name: str):
        """应用主题"""
        if name in self.themes:
            theme = self.themes[name]
            for widget_type, styles in theme.items():
                self.style.configure(widget_type, **styles)
            self.current_theme = name
    
    def configure_widget(self, widget_type: str, **kwargs):
        """配置特定组件类型的样式"""
        self.style.configure(widget_type, **kwargs)
    
    def map_widget(self, widget_type: str, states: Dict[str, Dict]):
        """配置组件的状态样式"""
        self.style.map(widget_type, **states)
    
    def get_available_themes(self) -> List[str]:
        """获取可用的主题列表"""
        return list(self.themes.keys())
    
    def set_font(self, font_family: str = "Arial", size: int = 10, bold: bool = False):
        """设置全局字体"""
        font_config = f"{font_family} {size}"
        if bold:
            font_config += " bold"
            
        # 更新所有组件的字体
        self.style.configure("TLabel", font=font_config)
        self.style.configure("TButton", font=font_config)
        self.style.configure("TEntry", font=font_config)
        self.style.configure("TCombobox", font=font_config)
    
    def set_color_scheme(self, background: str = "#f0f0f0", foreground: str = "#000000", 
                        accent: str = "#0078d4"):
        """设置颜色方案"""
        self.style.configure(".", background=background, foreground=foreground)
        self.style.configure("TButton", background=accent, foreground="white")
        
        # 配置按钮悬停效果
        self.style.map("TButton",
                      background=[('active', self._lighten_color(accent, 20)),
                                 ('pressed', self._darken_color(accent, 20))])
    
    def _lighten_color(self, color: str, percent: int) -> str:
        """调亮颜色"""
        # 简化的颜色处理，实际使用时可以完善
        return color
    
    def _darken_color(self, color: str, percent: int) -> str:
        """调暗颜色"""
        # 简化的颜色处理，实际使用时可以完善
        return color