import os
import re
from typing import Any, Callable, List, Tuple, Optional
from tkinter import messagebox

class Validation:
    """验证工具类"""
    
    @staticmethod
    def is_email(email: str) -> bool:
        """验证邮箱格式"""
        pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        return re.match(pattern, email) is not None
    
    @staticmethod
    def is_phone(phone: str) -> bool:
        """验证手机号格式"""
        pattern = r'^1[3-9]\d{9}$'
        return re.match(pattern, phone) is not None
    
    @staticmethod
    def is_number(text: str) -> bool:
        """验证是否为数字"""
        try:
            float(text)
            return True
        except ValueError:
            return False
    
    @staticmethod
    def is_integer(text: str) -> bool:
        """验证是否为整数"""
        try:
            int(text)
            return True
        except ValueError:
            return False
    
    @staticmethod
    def is_required(text: str) -> bool:
        """验证是否非空"""
        return bool(text and text.strip())
    
    @staticmethod
    def is_length_between(text: str, min_len: int, max_len: int) -> bool:
        """验证长度范围"""
        return min_len <= len(text) <= max_len
    
    @staticmethod
    def create_validator(validation_func: Callable, error_message: str) -> Callable:
        """创建验证器"""
        def validator(text: str) -> Tuple[bool, str]:
            if validation_func(text):
                return True, ""
            else:
                return False, error_message
        return validator

class FileHandler:
    """文件处理工具类"""
    
    @staticmethod
    def read_text_file(file_path: str) -> Optional[str]:
        """读取文本文件"""
        try:
            with open(file_path, 'r', encoding='utf-8') as file:
                return file.read()
        except Exception as e:
            messagebox.showerror("错误", f"读取文件失败: {str(e)}")
            return None
    
    @staticmethod
    def write_text_file(file_path: str, content: str) -> bool:
        """写入文本文件"""
        try:
            with open(file_path, 'w', encoding='utf-8') as file:
                file.write(content)
            return True
        except Exception as e:
            messagebox.showerror("错误", f"写入文件失败: {str(e)}")
            return False
    
    @staticmethod
    def get_file_extension(file_path: str) -> str:
        """获取文件扩展名"""
        return os.path.splitext(file_path)[1].lower()
    
    @staticmethod
    def get_file_size(file_path: str) -> Optional[int]:
        """获取文件大小（字节）"""
        try:
            return os.path.getsize(file_path)
        except:
            return None
    
    @staticmethod
    def is_image_file(file_path: str) -> bool:
        """判断是否为图片文件"""
        image_extensions = {'.jpg', '.jpeg', '.png', '.gif', '.bmp', '.tiff', '.webp'}
        return FileHandler.get_file_extension(file_path) in image_extensions

class StringUtils:
    """字符串工具类"""
    
    @staticmethod
    def truncate(text: str, max_length: int, ellipsis: str = "...") -> str:
        """截断字符串"""
        if len(text) <= max_length:
            return text
        return text[:max_length - len(ellipsis)] + ellipsis
    
    @staticmethod
    def to_camel_case(text: str) -> str:
        """转换为驼峰命名"""
        words = re.split(r'[_\s-]+', text)
        return words[0].lower() + ''.join(word.capitalize() for word in words[1:])
    
    @staticmethod
    def to_snake_case(text: str) -> str:
        """转换为蛇形命名"""
        text = re.sub(r'([a-z])([A-Z])', r'\1_\2', text)
        return text.lower()

class NumberUtils:
    """数字工具类"""
    
    @staticmethod
    def format_number(number: float, decimals: int = 2) -> str:
        """格式化数字"""
        return f"{number:.{decimals}f}"
    
    @staticmethod
    def parse_number(text: str, default: float = 0.0) -> float:
        """解析数字"""
        try:
            return float(text)
        except ValueError:
            return default

class DateUtils:
    """日期工具类"""
    
    @staticmethod
    def validate_date(date_str: str, format: str = "%Y-%m-%d") -> bool:
        """验证日期格式"""
        try:
            from datetime import datetime
            datetime.strptime(date_str, format)
            return True
        except ValueError:
            return False