import tkinter as tk
from tkinter import ttk
from typing import Dict, List, Callable, Any, Optional, Union

class WidgetFactory:
    """
    组件工厂类
    负责创建和管理所有基础GUI组件
    """
    
    def __init__(self, root):
        self.root = root
        self._widgets = {}  # 存储所有创建的组件
        self._current_parent = root  # 当前父组件
    
    def set_parent(self, parent):
        """设置当前父组件"""
        self._current_parent = parent
    
    def reset_parent(self):
        """重置父组件为根窗口"""
        self._current_parent = self.root
    
    def create_label(self, text: str, name: str = None, **kwargs) -> ttk.Label:
        """创建标签"""
        label = ttk.Label(self._current_parent, text=text, **kwargs)
        label.pack(pady=2, padx=10, anchor="w")
        
        if name:
            self._widgets[name] = label
        return label
    
    def create_button(self, text: str, command: Callable = None, name: str = None, **kwargs) -> ttk.Button:
        """创建按钮"""
        button = ttk.Button(self._current_parent, text=text, command=command, **kwargs)
        button.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = button
        return button
    
    def create_entry(self, default_text: str = "", name: str = None, **kwargs) -> ttk.Entry:
        """创建输入框"""
        entry = ttk.Entry(self._current_parent, **kwargs)
        entry.insert(0, default_text)
        entry.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = entry
        return entry
    
    def create_textbox(self, default_text: str = "", height: int = 5, name: str = None, **kwargs) -> tk.Text:
        """创建多行文本框"""
        frame = ttk.Frame(self._current_parent)
        frame.pack(pady=5, padx=10, fill="both", expand=True)
        
        # 添加滚动条
        scrollbar = ttk.Scrollbar(frame)
        scrollbar.pack(side="right", fill="y")
        
        textbox = tk.Text(frame, height=height, yscrollcommand=scrollbar.set, **kwargs)
        textbox.insert("1.0", default_text)
        textbox.pack(side="left", fill="both", expand=True)
        
        scrollbar.config(command=textbox.yview)
        
        if name:
            self._widgets[name] = textbox
            
        return textbox
    
    def create_combobox(self, values: List, default: str = None, name: str = None, **kwargs) -> ttk.Combobox:
        """创建下拉框"""
        combobox = ttk.Combobox(self._current_parent, values=values, **kwargs)
        
        if default and default in values:
            combobox.set(default)
        elif values:
            combobox.set(values[0])
            
        combobox.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = combobox
            
        return combobox
    
    def create_checkbox(self, text: str, default: bool = False, name: str = None, **kwargs) -> tk.BooleanVar:
        """创建复选框"""
        var = tk.BooleanVar(value=default)
        checkbox = ttk.Checkbutton(self._current_parent, text=text, variable=var, **kwargs)
        checkbox.pack(pady=2, padx=10, anchor="w")
        
        if name:
            self._widgets[name] = var
            
        return var
    
    def create_radiobuttons(self, options: List, default: str = None, name: str = None, **kwargs) -> tk.StringVar:
        """创建单选按钮组"""
        var = tk.StringVar()
        
        if default:
            var.set(default)
        elif options:
            var.set(options[0][1])
            
        frame = ttk.Frame(self._current_parent)
        frame.pack(pady=5, padx=10, fill="x")
        
        for text, value in options:
            radio = ttk.Radiobutton(
                frame, 
                text=text, 
                variable=var, 
                value=value,
                **kwargs
            )
            radio.pack(side="left", padx=(0, 10))
            
        if name:
            self._widgets[name] = var
            
        return var
    
    def create_frame(self, name: str = None, **kwargs) -> ttk.Frame:
        """创建框架"""
        frame = ttk.Frame(self._current_parent, **kwargs)
        frame.pack(pady=5, padx=10, fill="both", expand=True)
        
        if name:
            self._widgets[name] = frame
            
        return frame
    
    def create_separator(self, **kwargs) -> ttk.Separator:
        """创建分隔线"""
        separator = ttk.Separator(self._current_parent, orient="horizontal", **kwargs)
        separator.pack(fill="x", pady=10, padx=10)
        return separator
    
    def create_spinbox(self, from_: int, to: int, default: int = None, name: str = None, **kwargs) -> ttk.Spinbox:
        """创建数字选择框"""
        spinbox = ttk.Spinbox(self._current_parent, from_=from_, to=to, **kwargs)
        
        if default is not None and from_ <= default <= to:
            spinbox.set(default)
        else:
            spinbox.set(from_)
            
        spinbox.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = spinbox
            
        return spinbox
    
    def create_progressbar(self, name: str = None, **kwargs) -> ttk.Progressbar:
        """创建进度条"""
        progressbar = ttk.Progressbar(self._current_parent, **kwargs)
        progressbar.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = progressbar
            
        return progressbar
    
    def create_scale(self, from_: int, to: int, default: int = None, name: str = None, **kwargs) -> ttk.Scale:
        """创建滑动条"""
        scale = ttk.Scale(self._current_parent, from_=from_, to=to, **kwargs)
        
        if default is not None and from_ <= default <= to:
            scale.set(default)
            
        scale.pack(pady=5, padx=10, fill="x")
        
        if name:
            self._widgets[name] = scale
            
        return scale
    
    def create_listbox(self, items: List = None, name: str = None, **kwargs) -> tk.Listbox:
        """创建列表框"""
        frame = ttk.Frame(self._current_parent)
        frame.pack(pady=5, padx=10, fill="both", expand=True)
        
        # 添加滚动条
        scrollbar = ttk.Scrollbar(frame)
        scrollbar.pack(side="right", fill="y")
        
        listbox = tk.Listbox(frame, yscrollcommand=scrollbar.set, **kwargs)
        
        if items:
            for item in items:
                listbox.insert(tk.END, item)
                
        listbox.pack(side="left", fill="both", expand=True)
        scrollbar.config(command=listbox.yview)
        
        if name:
            self._widgets[name] = listbox
            
        return listbox
    
    def get_widget(self, name: str):
        """通过名称获取组件"""
        return self._widgets.get(name)
    
    def destroy_widget(self, name: str):
        """销毁组件"""
        widget = self.get_widget(name)
        if widget:
            widget.destroy()
            del self._widgets[name]