"""Create aggregation metadata using the aggregation service in fmu-dataio."""

from __future__ import annotations

from pathlib import Path

import numpy as np
import xtgeo

import fmu.dataio
import fmu.dataio._utils


def create_aggregation_metadata():
    """Create aggregation metadata by aggregating one surface across
    3 emulated realizations from the example case and store the metadata.

    The aggregation service can export both file and metadata to disk. For this case we
    only want to create the metadata.
    """

    # First we get the input data (the individual surfaces from each realization).
    # We use the surfaces and metadata generated by the example exports
    casepath = Path("../").resolve()
    relative_path = (
        "share/results/maps/topvolantis--ds_extract_geogrid.gri"  # exists in all reals
    )
    surface_path = casepath / "example_exports" / relative_path

    # We will emulate that we have 3 realizations we want to aggregate the surface for
    realization1_surface = xtgeo.surface_from_file(surface_path)
    realization1_metadata = fmu.dataio.read_metadata(surface_path)

    realization2_surface = xtgeo.surface_from_file(surface_path)
    realization2_metadata = fmu.dataio.read_metadata(surface_path)

    realization3_surface = xtgeo.surface_from_file(surface_path)
    realization3_metadata = fmu.dataio.read_metadata(surface_path)

    # Prepare the dataset for the surfaces we want to aggregate
    surfaces = [realization1_surface, realization2_surface, realization3_surface]
    source_surfaces = xtgeo.Surfaces(surfaces)
    source_metadata = [
        realization1_metadata,
        realization2_metadata,
        realization3_metadata,
    ]

    # We want to aggregate the mean of the surfaces
    operation = "mean"

    # This is the ID we assign to this set of aggregations
    aggregation_id = str(fmu.dataio._utils.uuid_from_string("something_very_unique"))

    # Initialize an AggregatedData object for this set of aggregation
    exp = fmu.dataio.AggregatedData(
        source_metadata=source_metadata,
        aggregation_id=aggregation_id,
        casepath=casepath,
        name="aggregated_surfaces",
    )

    # Aggregate the mean of the surfaces
    print("Aggregating the mean of the surfaces...")
    aggregated_surface = source_surfaces.apply(np.nanmean, axis=0)

    # Generate metadata for the aggregated surface and export to file
    print("Generating metadata for the aggregated surface and exporting to file...")
    exp.export(aggregated_surface, operation=operation)


def main():
    print("\nCreating aggregation metadata...")
    create_aggregation_metadata()
    print("Done creating aggregation metadata.\n")


if __name__ == "__main__":
    main()
