import abc
import logging
from . import entity_registry as entity_registry
from .entity import Entity as Entity
from .entity_component import EntityComponent as EntityComponent
from .json import json_bytes as json_bytes
from .storage import Store as Store
from .typing import ConfigType as ConfigType, VolDictType as VolDictType
from _typeshed import Incomplete
from abc import abstractmethod
from collections.abc import Awaitable, Callable, Iterable
from dataclasses import dataclass
from homeassistant.components import websocket_api as websocket_api
from homeassistant.const import CONF_ID as CONF_ID
from homeassistant.core import CALLBACK_TYPE as CALLBACK_TYPE, HomeAssistant as HomeAssistant, callback as callback
from homeassistant.exceptions import HomeAssistantError as HomeAssistantError
from homeassistant.util import slugify as slugify
from typing import Any, TypedDict

STORAGE_VERSION: int
SAVE_DELAY: int
CHANGE_ADDED: str
CHANGE_UPDATED: str
CHANGE_REMOVED: str

@dataclass(slots=True)
class CollectionChange:
    change_type: str
    item_id: str
    item: Any
    item_hash: str | None = ...
type ChangeListener = Callable[[str, str, dict], Awaitable[None]]
type ChangeSetListener = Callable[[Iterable[CollectionChange]], Awaitable[None]]

class CollectionError(HomeAssistantError): ...

class ItemNotFound(CollectionError):
    item_id: Incomplete
    def __init__(self, item_id: str) -> None: ...

class IDManager:
    collections: list[dict[str, Any]]
    def __init__(self) -> None: ...
    def add_collection(self, collection: dict[str, Any]) -> None: ...
    def has_id(self, item_id: str) -> bool: ...
    def generate_id(self, suggestion: str) -> str: ...

class CollectionEntity(Entity, metaclass=abc.ABCMeta):
    @classmethod
    @abstractmethod
    def from_storage(cls, config: ConfigType) -> CollectionEntity: ...
    @classmethod
    @abstractmethod
    def from_yaml(cls, config: ConfigType) -> CollectionEntity: ...
    @abstractmethod
    async def async_update_config(self, config: ConfigType) -> None: ...

class ObservableCollection[_ItemT]:
    id_manager: Incomplete
    data: dict[str, _ItemT]
    listeners: list[ChangeListener]
    change_set_listeners: list[ChangeSetListener]
    def __init__(self, id_manager: IDManager | None) -> None: ...
    @callback
    def async_items(self) -> list[_ItemT]: ...
    @callback
    def async_add_listener(self, listener: ChangeListener) -> Callable[[], None]: ...
    @callback
    def async_add_change_set_listener(self, listener: ChangeSetListener) -> Callable[[], None]: ...
    async def notify_changes(self, change_set: Iterable[CollectionChange]) -> None: ...

class YamlCollection(ObservableCollection[dict]):
    logger: Incomplete
    def __init__(self, logger: logging.Logger, id_manager: IDManager | None = None) -> None: ...
    @staticmethod
    def create_entity(entity_class: type[CollectionEntity], config: ConfigType) -> CollectionEntity: ...
    async def async_load(self, data: list[dict]) -> None: ...

class SerializedStorageCollection(TypedDict):
    items: list[dict[str, Any]]

class StorageCollection[_ItemT, _StoreT: SerializedStorageCollection](ObservableCollection[_ItemT], metaclass=abc.ABCMeta):
    store: Incomplete
    def __init__(self, store: Store[_StoreT], id_manager: IDManager | None = None) -> None: ...
    @staticmethod
    def create_entity(entity_class: type[CollectionEntity], config: ConfigType) -> CollectionEntity: ...
    @property
    def hass(self) -> HomeAssistant: ...
    async def _async_load_data(self) -> _StoreT | None: ...
    async def async_load(self) -> None: ...
    @abstractmethod
    async def _process_create_data(self, data: dict) -> dict: ...
    @callback
    @abstractmethod
    def _get_suggested_id(self, info: dict) -> str: ...
    @abstractmethod
    async def _update_data(self, item: _ItemT, update_data: dict) -> _ItemT: ...
    @abstractmethod
    def _create_item(self, item_id: str, data: dict) -> _ItemT: ...
    @abstractmethod
    def _deserialize_item(self, data: dict) -> _ItemT: ...
    @abstractmethod
    def _serialize_item(self, item_id: str, item: _ItemT) -> dict: ...
    async def async_create_item(self, data: dict) -> _ItemT: ...
    async def async_update_item(self, item_id: str, updates: dict) -> _ItemT: ...
    async def async_delete_item(self, item_id: str) -> None: ...
    @callback
    def _async_schedule_save(self) -> None: ...
    @callback
    def _base_data_to_save(self) -> SerializedStorageCollection: ...
    @abstractmethod
    @callback
    def _data_to_save(self) -> _StoreT: ...
    def _hash_item(self, item: dict) -> str: ...

class DictStorageCollection(StorageCollection[dict, SerializedStorageCollection], metaclass=abc.ABCMeta):
    def _create_item(self, item_id: str, data: dict) -> dict: ...
    def _deserialize_item(self, data: dict) -> dict: ...
    def _serialize_item(self, item_id: str, item: dict) -> dict: ...
    @callback
    def _data_to_save(self) -> SerializedStorageCollection: ...

class IDLessCollection(YamlCollection):
    counter: int
    async def async_load(self, data: list[dict]) -> None: ...

_GROUP_BY_KEY: Incomplete

@dataclass(slots=True, frozen=True)
class _CollectionLifeCycle[_EntityT: Entity = Entity]:
    domain: str
    platform: str
    entity_component: EntityComponent[_EntityT]
    collection: StorageCollection | YamlCollection
    entity_class: type[CollectionEntity]
    ent_reg: entity_registry.EntityRegistry
    entities: dict[str, CollectionEntity]
    @callback
    def async_setup(self) -> None: ...
    def _entity_removed(self, item_id: str) -> None: ...
    @callback
    def _add_entity(self, change_set: CollectionChange) -> CollectionEntity: ...
    async def _remove_entity(self, change_set: CollectionChange) -> None: ...
    async def _update_entity(self, change_set: CollectionChange) -> None: ...
    async def _collection_changed(self, change_set: Iterable[CollectionChange]) -> None: ...

@callback
def sync_entity_lifecycle[_EntityT: Entity = Entity](hass: HomeAssistant, domain: str, platform: str, entity_component: EntityComponent[_EntityT], collection: StorageCollection | YamlCollection, entity_class: type[CollectionEntity]) -> None: ...

class StorageCollectionWebsocket[_StorageCollectionT: StorageCollection]:
    storage_collection: Incomplete
    api_prefix: Incomplete
    model_name: Incomplete
    create_schema: Incomplete
    update_schema: Incomplete
    _remove_subscription: CALLBACK_TYPE | None
    _subscribers: set[tuple[websocket_api.ActiveConnection, int]]
    def __init__(self, storage_collection: _StorageCollectionT, api_prefix: str, model_name: str, create_schema: VolDictType, update_schema: VolDictType) -> None: ...
    @property
    def item_id_key(self) -> str: ...
    @callback
    def async_setup(self, hass: HomeAssistant) -> None: ...
    @callback
    def ws_list_item(self, hass: HomeAssistant, connection: websocket_api.ActiveConnection, msg: dict) -> None: ...
    async def ws_create_item(self, hass: HomeAssistant, connection: websocket_api.ActiveConnection, msg: dict) -> None: ...
    @callback
    def _ws_subscribe(self, hass: HomeAssistant, connection: websocket_api.ActiveConnection, msg: dict) -> None: ...
    async def ws_update_item(self, hass: HomeAssistant, connection: websocket_api.ActiveConnection, msg: dict) -> None: ...
    async def ws_delete_item(self, hass: HomeAssistant, connection: websocket_api.ActiveConnection, msg: dict) -> None: ...

class DictStorageCollectionWebsocket(StorageCollectionWebsocket[DictStorageCollection]): ...
