from .tool import *

from sympy import *
import matplotlib.pyplot as plt
from time import time
from csv import writer


class BasicEconomy:
    def __init__(self, k0: float, n: float, alpha: float, life: int):
        """
        This basic macroeconomic model is adapted from a dynamically multiple Ramsey Model, which could be extended into many occasions.
        It follows the philosophy and methods of the New Classical Macro Theory where macroeconomic model must have micro-foundation,
        and thus basic assumptions in the theory, for instance, Say's Law, Inda Conditions and CRS, are accepted here.
        For the perspective of production, the model adopts the traditional choice: Cobb Douglas Production Function,
        while Logarithmic form is chosen as the utility function.
        In order to simplify calculation, labor is normalized into 1, and TFP(A) is assumed as a const 1.
        All variables below are counted by consumer goods, reflecting the real economy.
        Therefore, monetary factors are neglected in this very brief model.
        As a result of the New Classical Macro Assumptions, this model has it structural limitation,
        which means it is absolutely a terrible idea to apply this model to analyze an economy with insufficient aggregate demand, such as China.

        :param k0: initial capital stock
        :param n: population growth rate, the world average level is about 0.0085
        :param alpha: capital intensity, or the capital share in Solow Growth Model, describing the endowment structure of this economy
        :param life: the life of this economy, which can be assigned casually
        :return A graphic demonstrating the development footprints of the simulated economy in the form of png
        :return A table recording the detailed data, with a sequence of real GDP, rent, wage, capital, consumption, consumer utility and expected rent of next period
        :raise All parameter must be POSITIVE, if not, raise ValueError.
        """
        self.version = int(time())
        self.k0 = k0 # initial capital stock
        self.n = n # population growth rate
        self.alpha = alpha # capital intensity
        self.beta = 0.98 # stochastic discount factor
        self.y = [] # Real GDP
        self.gamma = [] # Define Gamma0 = k0 + AF(k0, l) = c1 + k1
        self.rent = []
        self.wage = []
        self.capital = []
        self.consumption = []
        self.utility = []
        self.expected_rent = []
        self.initial_period()
        self.expect_rent()
        for p in range(0, life):
            print(f'----------{p + 1}----------')
            self.period(p)
            self.expect_rent()
        self.record()
        self.demonstrate()

    def check(self):
        for i in [self.k0, self.n, self.alpha]:
            if i < 0:
                raise ValueError

    def initial_period(self):
        generation1 = Cobb_Douglas_Production_Function(self.k0, 1, self.alpha) # Assume l = 1
        y1 = generation1.y()
        self.y.append(y1)
        gamma1 = self.k0 + y1
        self.gamma.append(gamma1)
        r1 = generation1.rent()
        self.rent.append(r1)
        print(f'Current intrest rate: {r1}')
        self.expected_rent.append(r1)
        w1 = generation1.wage()
        self.wage.append(w1)

        x = symbols('x')
        eq = (1+self.beta)*x-(self.alpha/(1+r1)*(x**self.alpha))-self.beta*gamma1
        k1 = solve(eq, x, positive=True)
        k1 = k1[0]
        self.capital.append(k1)

        c1 = gamma1 - k1
        self.consumption.append(c1)
        u1 = Utility(c1).logarithmic()
        self.utility.append(u1)

    def expect_rent(self):
        r1 = self.rent[-1]
        re1 = self.expected_rent[-1]
        re2 = 2 * r1 - re1
        self.expected_rent.append(re2)
        print(f'Expected intrest rate of next period: {re2}')

    def period(self, period):
        k1 = self.capital[-1]
        generation2 = Cobb_Douglas_Production_Function(k1, 1, self.alpha)
        y2 = generation2.y()
        self.y.append(y2)
        print(f'Real GDP: {y2}')
        gamma2 = k1 + y2
        self.gamma.append(gamma2)
        r2 = generation2.rent()
        self.rent.append(r2)
        print(f'Current intrest rate: {r2}')
        w2 = generation2.wage()
        self.wage.append(w2)

        re2 = self.expected_rent[-1]
        x = symbols('x')
        eq = (1+self.beta)*x-(self.alpha*((1+self.n)**period)/(1+re2)*(x**self.alpha))-self.beta*gamma2
        k2 = solve(eq, x, positive=True)
        k2 = k2[0]
        self.capital.append(k2)

        c2 = gamma2 - k2
        self.consumption.append(c2)
        u2 = Utility(c2).logarithmic()
        self.utility.append(u2)
        print(f'Consumer utility: {u2}')

    def record(self):
        file_name = f'BasicEconomy_{self.version}.csv'
        with open(file=file_name, mode='a', encoding='utf-8', newline='') as f:
            csv_writer = writer(f)
            csv_writer.writerows(
                [self.y, self.rent, self.wage, self.capital, self.consumption, self.utility, self.expected_rent]
            )
        print(f'***Record as {file_name} successfully.')

    def demonstrate(self):
        file_name = f'BasicEconomy_{self.version}.png'
        plt.plot(range(0, len(self.y)), self.y, label='Real GDP')
        plt.plot(range(0, len(self.consumption)), self.consumption, 'b--', label='Consumption')
        plt.suptitle(f'version :{self.version}')
        plt.xlabel('Period')
        plt.ylabel('Value per Population')
        plt.title(f'Basic Economy (α={self.alpha})')
        plt.legend()
        plt.savefig(fname=file_name, dpi=1980)
        plt.show()


if __name__ == '__main__':
    state = BasicEconomy(1.0, 0.0085, 0.4, 20)