# Component Relationships Documentation

## Overview

This document describes how components in the refactored TunaCode agent system interact and depend on each other. The architecture follows a modular design with clear separation of concerns and well-defined interfaces.

## Architecture Overview

```
┌─────────────────────────────────────────────────────────────┐
│                    Main Agent (main.py)                     │
│                    (Orchestration Layer)                    │
└─────────────────────┬───────────────────────────────────────┘
                      │
                      ▼
┌─────────────────────────────────────────────────────────────┐
│                Agent Components Package                     │
│              (Modular Component Layer)                     │
└─────────────────────┬───────────────────────────────────────┘
                      │
    ┌─────────────────┼─────────────────┐
    │                 │                 │
    ▼                 ▼                 ▼
┌─────────┐    ┌─────────────┐    ┌─────────────┐
│ State   │    │ Processing  │    │ Tool        │
│ Mgmt    │    │ Components  │    │ Components  │
└─────────┘    └─────────────┘    └─────────────┘
    │                 │                 │
    │                 │                 │
    ▼                 ▼                 ▼
┌─────────┐    ┌─────────────┐    ┌─────────────┐
│ Response│    │ Message     │    │ Execution   │
│ State   │    │ Handlers    │    │ Components  │
└─────────┘    └─────────────┘    └─────────────┘
```

## Component Interaction Flows

### 1. Request Processing Pipeline

```
User Request → Main Agent → Response State → Message Handler → Tool Execution → Response Generation
```

**Step-by-step Flow**:

1. **Main Agent** receives user request
2. **Response State** initializes state machine to `USER_INPUT`
3. **Message Handler** constructs model request
4. **State Transition** to `ASSISTANT` state
5. **Tool Execution** processes tool calls (if any)
6. **State Transition** to `TOOL_EXECUTION` state
7. **Response Generation** handles results
8. **State Transition** to `RESPONSE` state
9. **Task Completion** checks for completion markers
10. **Response State** updates completion status

### 2. State Management Flow

```
Response State ← → State Transition Rules ← → Agent State Enum
        ↓
    Legacy Boolean Flags (backward compatibility)
```

**Key Interactions**:

- **Response State** uses **State Transition Rules** for validation
- **Agent State Enum** provides type-safe state values
- **Legacy Flags** are maintained for backward compatibility
- **State Machine** enforces transition logic

### 3. Tool Execution Flow

```
JSON Tool Parser → Tool Buffer → Tool Executor → Parallel Execution
        ↓                ↓                ↓
   Tool Messages → Tool Validation → Result Aggregation
```

**Parallel Processing**:

- **Tool Buffer** batches tool calls for efficiency
- **Tool Executor** runs tools in parallel
- **JSON Tool Parser** handles JSON-formatted tool calls
- **Result Aggregation** combines tool results

#### Rejection Guidance Loop

- When the user rejects a tool via option 3, **ToolUI** now prompts for corrective guidance and stores it on `ToolConfirmationResponse.instructions`.
- **ToolHandler.process_confirmation** detects guided aborts and calls `agent_helpers.create_user_message` so the agent receives the user's corrective instructions on the next turn.
- The REPL suppresses the legacy "Operation aborted" banner once guidance is captured, keeping the transcript focused on the new instructions instead of noise.

## Dependency Graph

### Core Dependencies

```
Main Agent (main.py)
├── Response State (response_state.py)
│   ├── State Transition (state_transition.py)
│   └── Agent State Enum (types.py)
├── Message Handler (message_handler.py)
├── Tool Components
│   ├── Tool Executor (tool_executor.py)
│   ├── Tool Buffer (tool_buffer.py)
│   └── JSON Tool Parser (json_tool_parser.py)
├── Result Wrapper (result_wrapper.py)
├── Agent Helpers (agent_helpers.py)
└── Streaming (streaming.py)
```

### State Management Dependencies

```
Response State (response_state.py)
├── State Transition Rules (state_transition.py)
│   ├── Agent State Enum (types.py)
│   └── InvalidStateTransitionError (exceptions.py)
├── Legacy Boolean Flags (internal)
└── State Machine Logic (internal)
```

### Tool Execution Dependencies

```
Tool Execution Pipeline
├── JSON Tool Parser (json_tool_parser.py)
│   └── Tool Call Extraction Logic
├── Tool Buffer (tool_buffer.py)
│   └── Tool Call Management
├── Tool Executor (tool_executor.py)
│   ├── Parallel Execution
│   └── Error Handling
└── Result Aggregation (via message_handler.py)
```

## Component Contracts

### 1. State Management Contract

**Interface**: `ResponseState`
- **Provides**: State machine with enum-based states
- **Requires**: State transition rules from `state_transition.py`
- **Guarantees**: Thread-safe state transitions with validation
- **Backward Compatibility**: Legacy boolean flag accessors

### 2. Tool Execution Contract

**Interface**: `ToolExecutor`
- **Provides**: Parallel tool execution with error handling
- **Requires**: Tool call specifications and callbacks
- **Guarantees**: Atomic execution or graceful failure
- **Fallback**: JSON parsing for malformed tool calls

### 3. Message Processing Contract

**Interface**: `MessageHandler`
- **Provides**: Message construction and tool result handling
- **Requires**: Tool results and response content
- **Guarantees**: Proper message formatting and context
- **Error Handling**: Tool message patching for recovery

### 4. Result Wrapper Contract

**Interface**: `AgentRunWithState`
- **Provides**: Enhanced response with state tracking
- **Requires**: Agent response and state information
- **Guarantees**: State consistency and response integrity
- **Fallback**: `AgentRunWrapper` for degraded operation

## Data Flow Diagrams

### 1. State Transition Flow

```
USER_INPUT ──→ ASSISTANT ──→ TOOL_EXECUTION ──→ RESPONSE
    │              │               │               │
    │              │               │               │
    └──────┐       │               │               │
           ▼       ▼               ▼               ▼
    (Completion Detection ← State Machine Validation)
```

**State Transition Rules**:
- USER_INPUT → ASSISTANT: Always allowed
- ASSISTANT → TOOL_EXECUTION: When tools needed
- ASSISTANT → RESPONSE: Direct response without tools
- TOOL_EXECUTION → RESPONSE: After tool execution
- RESPONSE → USER_INPUT: For follow-up interactions
- RESPONSE → COMPLETED: When task completion detected

### 2. Tool Call Processing Flow

```
Agent Response → JSON Tool Parser → Tool Buffer → Tool Executor → Tool Results
       ↓                ↓               ↓               ↓               ↓
Tool Call Detection → Tool Validation → Tool Batching → Parallel Execution → Result Aggregation
```

**Error Handling Paths**:
- Malformed JSON → JSON parsing fallback
- Tool Execution Failure → Error result with context
- Tool Timeout → Timeout handling and retry logic
- Invalid Arguments → Argument validation errors

### 3. Response Generation Flow

```
Tool Results → Message Handler → Response Generation → Response State → Final Response
      ↓              ↓               ↓                ↓              ↓
Result Format → Message Construction → Content Assembly → State Update → User Output
```

**Completion Detection**:
- Content scanned for completion markers
- State machine updated accordingly
- Legacy flags synchronized
- Final response formatted

## Error Handling and Recovery

### 1. State Machine Errors

```
Invalid State Transition → InvalidStateTransitionError → Error Handler → Recovery
```

**Recovery Strategies**:
- Log error with context
- Attempt state reset to safe state
- Notify user of transition failure
- Continue with degraded functionality

### 2. Tool Execution Errors

```
Tool Failure → Error Detection → Fallback Logic → Error Recovery
```

**Recovery Strategies**:
- JSON parsing fallback for malformed tool calls
- Parallel execution with individual tool isolation
- Tool timeout handling
- Graceful degradation with user notification

### 3. Message Processing Errors

```
Message Error → Detection → Patching → Recovery
```

**Recovery Strategies**:
- Tool message patching
- Context preservation
- Fallback message construction
- Error context inclusion

## Performance Considerations

### 1. Parallel Processing

```
Tool Buffer ──→ Parallel Tool Executor ──→ Result Aggregation
       ↓                  ↓                    ↓
Tool Batching    Concurrent Execution    Result Collection
```

**Optimizations**:
- Tools run in parallel when possible
- Batching reduces overhead
- Result aggregation is efficient
- Minimal locking for thread safety

### 2. State Management

```
State Machine ──→ Transition Validation ──→ State Update
       ↓                 ↓                    ↓
Enum-based        Efficient Rules      Thread-safe Updates
```

**Optimizations**:
- Enum-based states are lightweight
- Transition rules are pre-computed
- State updates are atomic
- Minimal synchronization overhead

## Integration Points

### 1. External System Integration

```
Main Agent ──→ MCP Servers ──→ External Tools
       ↓           ↓              ↓
State Mgmt   Tool Discovery   Tool Execution
```

### 2. UI Integration

```
Response Generation ──→ UI Updates ──→ User Feedback
         ↓                ↓              ↓
State Tracking    Real-time Display   Input Handling
```

### 3. Testing Integration

```
Component Testing ──→ Mock Components ──→ Test Verification
        ↓                  ↓                ↓
Isolated Units    Controlled Behavior    Expected Results
```

## Future Extension Points

### 1. New States

The state machine can be extended with new states:
- Add new enum values to `AgentState`
- Define transition rules for new states
- Update state machine logic
- Maintain backward compatibility

### 2. New Tool Types

The tool execution system can be extended:
- Add new tool parsers for different formats
- Extend tool executor for new execution modes
- Add tool result processors
- Maintain existing tool compatibility

### 3. New Response Types

The response system can be extended:
- Add new result wrapper types
- Extend response state tracking
- Add new completion detection rules
- Maintain existing response formats

This modular architecture ensures that components can be extended independently while maintaining system integrity and backward compatibility.
