"""
Module: tunacode.cli.main

Enhanced CLI entry point with better styling while staying CLI-based.
"""

import asyncio
import logging

import typer

from tunacode.cli.repl import repl
from tunacode.configuration.settings import ApplicationSettings
from tunacode.core.state import StateManager
from tunacode.core.tool_handler import ToolHandler
from tunacode.exceptions import UserAbortError
from tunacode.setup import setup
from tunacode.ui import console as ui
from tunacode.ui.config_dashboard import show_config_dashboard
from tunacode.utils.system import check_for_updates

app_settings = ApplicationSettings()
app = typer.Typer(help="TunaCode - OS AI-powered development assistant")
state_manager = StateManager()

logger = logging.getLogger(__name__)


def _handle_background_task_error(task: asyncio.Task) -> None:
    """Error callback for background tasks to prevent unhandled exceptions.

    This callback ensures that background task failures are logged but don't
    crash the CLI. Tasks are marked as 'done' after this callback executes,
    preventing 'Task was destroyed but pending' warnings.
    """
    try:
        # Retrieve exception if task failed
        exception = task.exception()
        if exception is not None:
            task_name = task.get_name()
            logger.warning(
                "Background task '%s' failed: %s",
                task_name,
                exception,
                exc_info=exception,
            )
    except asyncio.CancelledError:
        # Task was cancelled, which is expected behavior
        pass
    except Exception as e:
        # Failsafe: log any error in the callback itself
        logger.error("Error in background task error callback: %s", e, exc_info=True)


@app.command()
def main(
    version: bool = typer.Option(False, "--version", "-v", help="Show version and exit."),
    run_setup: bool = typer.Option(False, "--setup", help="Run setup process."),
    wizard: bool = typer.Option(
        False, "--wizard", help="Run interactive setup wizard for guided configuration."
    ),
    show_config: bool = typer.Option(
        False, "--show-config", help="Show configuration dashboard and exit."
    ),
    baseurl: str = typer.Option(
        None, "--baseurl", help="API base URL (e.g., https://openrouter.ai/api/v1)"
    ),
    model: str = typer.Option(None, "--model", help="Default model to use (e.g., openai/gpt-4)"),
    key: str = typer.Option(None, "--key", help="API key for the provider"),
    context: int = typer.Option(
        None, "--context", help="Maximum context window size for custom models"
    ),
):
    """Start TunaCode - Your AI-powered development assistant"""

    # Configure logging to suppress INFO messages by default
    logging.basicConfig(level=logging.WARNING, force=True)

    async def async_main():
        if version:
            await ui.version()
            return

        if show_config:
            await ui.banner()
            show_config_dashboard()
            return

        await ui.banner()

        # Start update check in background
        update_task = asyncio.create_task(asyncio.to_thread(check_for_updates), name="update_check")
        update_task.add_done_callback(_handle_background_task_error)

        cli_config = {
            "baseurl": baseurl,
            "model": model,
            "key": key,
            "custom_context_window": context,
        }
        cli_config = {k: v for k, v in cli_config.items() if v is not None}

        try:
            await setup(run_setup or wizard, state_manager, cli_config, wizard_mode=wizard)

            # Initialize ToolHandler after setup
            tool_handler = ToolHandler(state_manager)
            state_manager.set_tool_handler(tool_handler)

            await repl(state_manager)
        except (KeyboardInterrupt, UserAbortError):
            update_task.cancel()
            return
        except Exception as e:
            from tunacode.exceptions import ConfigurationError

            if isinstance(e, ConfigurationError):
                # Display the configuration error message
                await ui.error(str(e))
                update_task.cancel()  # Cancel the update check
                return
            import traceback

            await ui.error(f"{str(e)}\n\nTraceback:\n{traceback.format_exc()}")

        # Gracefully handle update check result (may have failed in background)
        try:
            has_update, latest_version = await update_task
            if has_update:
                await ui.update_available(latest_version)
        except Exception:
            # Update check failed; error already logged by callback
            pass

        # Normal exit - cleanup MCP servers
        try:
            from tunacode.core.agents import cleanup_mcp_servers

            await cleanup_mcp_servers()
        except Exception:
            pass  # Best effort cleanup

    asyncio.run(async_main())


if __name__ == "__main__":
    app()
