# Sarvam Python Library

[![fern shield](https://img.shields.io/badge/%F0%9F%8C%BF-Built%20with%20Fern-brightgreen)](https://buildwithfern.com?utm_source=github&utm_medium=github&utm_campaign=readme&utm_source=https%3A%2F%2Fgithub.com%2Fsarvamai%2Fsarvam-python-sdk)
[![pypi](https://img.shields.io/pypi/v/sarvamai)](https://pypi.python.org/pypi/sarvamai)

The Sarvam Python library provides convenient access to the Sarvam API from Python.

## Documentation

API reference documentation is available [here](https://docs.sarvam.ai/api-reference-docs/speech-to-text/).

## Installation

```sh
pip install sarvamai
```

## Reference

A full reference for this library is available [here](#References).

## Usage

Instantiate and use the client with the following:

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.text.translate(
    input="input",
    source_language_code="auto",
    target_language_code="bn-IN",
)
```

## Async Client

The SDK also exports an `async` client so that you can make non-blocking calls to our API.

```python
import asyncio

from sarvamai import AsyncSarvamAI

client = AsyncSarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)


async def main() -> None:
    await client.text.translate(
        input="input",
        source_language_code="auto",
        target_language_code="bn-IN",
    )


asyncio.run(main())
```

## Exception Handling

When the API returns a non-success status code (4xx or 5xx response), a subclass of the following error
will be thrown.

```python
from sarvamai.core.api_error import ApiError

try:
    client.text.translate(...)
except ApiError as e:
    print(e.status_code)
    print(e.body)
```

## Advanced

### Retries

The SDK is instrumented with automatic retries with exponential backoff. A request will be retried as long
as the request is deemed retryable and the number of retry attempts has not grown larger than the configured
retry limit (default: 2).

A request is deemed retryable when any of the following HTTP status codes is returned:

- [408](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/408) (Timeout)
- [429](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/429) (Too Many Requests)
- [5XX](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/500) (Internal Server Errors)

Use the `max_retries` request option to configure this behavior.

```python
client.text.translate(..., request_options={
    "max_retries": 1
})
```

### Timeouts

The SDK defaults to a 60 second timeout. You can configure this with a timeout option at the client or request level.

```python

from sarvamai import SarvamAI

client = SarvamAI(
    ...,
    timeout=20.0,
)


# Override timeout for a specific method
client.text.translate(..., request_options={
    "timeout_in_seconds": 1
})
```

### Custom Client

You can override the `httpx` client to customize it for your use-case. Some common use-cases include support for proxies
and transports.

```python
import httpx
from sarvamai import SarvamAI

client = SarvamAI(
    ...,
    httpx_client=httpx.Client(
        proxies="http://my.test.proxy.example.com",
        transport=httpx.HTTPTransport(local_address="0.0.0.0"),
    ),
)
```

## Contributing

While we value open-source contributions to this SDK, this library is generated programmatically.
Additions made directly to this library would have to be moved over to our generation code,
otherwise they would be overwritten upon the next generated release. Feel free to open a PR as
a proof of concept, but know that we will not be able to merge it as-is. We suggest opening
an issue first to discuss with us!

On the other hand, contributions to the README are always very welcome!

---

## References

### Text
<details><summary><code>client.text.<a href="src/sarvamai/text/client.py">translate</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

**Translation** converts text from one language to another while preserving its meaning.
For Example: **'मैं ऑफिस जा रहा हूँ'** translates to **'I am going to the office'** in English, where the script and language change, but the original meaning remains the same.

Available languages:
- **`bn-IN`**: Bengali
- **`en-IN`**: English
- **`gu-IN`**: Gujarati
- **`hi-IN`**: Hindi
- **`kn-IN`**: Kannada
- **`ml-IN`**: Malayalam
- **`mr-IN`**: Marathi
- **`od-IN`**: Odia
- **`pa-IN`**: Punjabi
- **`ta-IN`**: Tamil
- **`te-IN`**: Telugu

#### Newly added languages:
- **`as-IN`**: Assamese
- **`brx-IN`**: Bodo
- **`doi-IN`**: Dogri
- **`kok-IN`**: Konkani
- **`ks-IN`**: Kashmiri
- **`mai-IN`**: Maithili
- **`mni-IN`**: Manipuri (Meiteilon)
- **`ne-IN`**: Nepali
- **`sa-IN`**: Sanskrit
- **`sat-IN`**: Santali
- **`sd-IN`**: Sindhi
- **`ur-IN`**: Urdu

For hands-on practice, you can explore the notebook tutorial on [Translate API Tutorial](https://github.com/sarvamai/sarvam-ai-cookbook/blob/main/notebooks/translate/Translate_API_Tutorial.ipynb).
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.text.translate(
    input="input",
    source_language_code="auto",
    target_language_code="bn-IN",
)

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**input:** `str` — The text you want to translate is the input text that will be processed by the translation model. The maximum is 1000 characters for Mayura:v1 and 2000 characters for Sarvam-Translate:v1.
    
</dd>
</dl>

<dl>
<dd>

**source_language_code:** `TranslateSourceLanguage` 

Source language code for translation input.

**mayura:v1 Languages:** Bengali, English, Gujarati, Hindi, Kannada, Malayalam, Marathi, Odia, Punjabi, Tamil, Telugu

**sarvam-translate:v1 Languages:** All mayura:v1 languages and Assamese, Bodo, Dogri, Konkani, Kashmiri, Maithili, Manipuri, Nepali, Sanskrit, Santali, Sindhi, Urdu

**Note:** mayura:v1 supports automatic language detection using 'auto' as the source language code.

    
</dd>
</dl>

<dl>
<dd>

**target_language_code:** `TranslateTargetLanguage` 

The language code of the translated text. This specifies the target language for translation.

**mayura:v1 Languages:** Bengali, English, Gujarati, Hindi, Kannada, Malayalam, Marathi, Odia, Punjabi, Tamil, Telugu

**sarvam-translate:v1 Languages:** All mayura:v1 and Assamese, Bodo, Dogri, Konkani, Kashmiri, Maithili, Manipuri, Nepali, Sanskrit, Santali, Sindhi, Urdu

    
</dd>
</dl>

<dl>
<dd>

**speaker_gender:** `typing.Optional[TranslateSpeakerGender]` — Please specify the gender of the speaker for better translations.
    
</dd>
</dl>

<dl>
<dd>

**mode:** `typing.Optional[TranslateMode]` 

Specifies the tone or style of the translation.

**Model Support:**
- **mayura:v1**: Supports formal, classic-colloquial, and modern-colloquial modes
- **sarvam-translate:v1**: Only formal mode is supported

**Default:** formal
    
</dd>
</dl>

<dl>
<dd>

**model:** `typing.Optional[TranslateModel]` 

Specifies the translation model to use.
- mayura:v1: Supports 12 languages with all modes, output scripts, and automatic language detection.
- sarvam-translate:v1: Supports all 22 scheduled languages of India, formal mode only.
    
</dd>
</dl>

<dl>
<dd>

**enable_preprocessing:** `typing.Optional[bool]` 

This will enable custom preprocessing of the input text which can result in better translations.
 Recommendation- You can switch on whenever there is some complex text with difficult vocabulary and sentences, for which you want simple translations that people can understand.
    
</dd>
</dl>

<dl>
<dd>

**output_script:** `typing.Optional[TransliterateMode]` 

**output_script**: This is an optional parameter which controls the transliteration style applied to the output text.

**Transliteration**: Converting text from one script to another while preserving pronunciation.

For mayura:v1 - We support transliteration with four options:
- **`null`**(default): No transliteration applied.
- **`roman`**: Transliteration in Romanized script.
- **`fully-native`**: Transliteration in the native script with formal style.
- **`spoken-form-in-native`**: Transliteration in the native script with spoken style.

For sarvam-translate:v1 - Transliteration is not supported.
#### Example:
English: Your EMI of Rs. 3000 is pending.
Default modern translation: आपका Rs. 3000 का EMI pending है (when `null` is passed).

With postprocessing enabled:
- **roman output**: aapka Rs. 3000 ka EMI pending hai.
    
</dd>
</dl>

<dl>
<dd>

**numerals_format:** `typing.Optional[NumeralsFormat]` 

`numerals_format` is an optional parameter with two options (supported for both mayura:v1 and sarvam-translate:v1):

- **`international`** (default): Uses regular numerals (0-9).
- **`native`**: Uses language-specific native numerals.

#### Example:
- If `international` format is selected, we use regular numerals (0-9). For example: `मेरा phone number है: 9840950950`.
- If `native` format is selected, we use language-specific native numerals, like: `मेरा phone number है: ९८४०९५०९५०`.
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

<details><summary><code>client.text.<a href="src/sarvamai/text/client.py">identify_language</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

Identifies the language (e.g., en-IN, hi-IN) and script (e.g., Latin, Devanagari) of the input text, supporting multiple languages.
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.text.identify_language(
    input="input",
)

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**input:** `str` — The text input for language and script identification.
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

<details><summary><code>client.text.<a href="src/sarvamai/text/client.py">transliterate</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

**Transliteration** converts text from one script to another while preserving the original pronunciation. For example, **'नमस्ते'** becomes **'namaste'** in English, and **'how are you'** can be written as **'हाउ आर यू'** in Devanagari. This process ensures that the sound of the original text remains intact, even when written in a different script.

Transliteration is useful when you want to represent words phonetically across different writing systems, such as converting **'मैं ऑफिस जा रहा हूँ'** to **'main office ja raha hun'** in English letters.

**Translation**, on the other hand, converts text from one language to another while preserving the meaning rather than pronunciation. For example, **'मैं ऑफिस जा रहा हूँ'** translates to **'I am going to the office'** in English, changing both the script and the language while conveying the intended message.
#### Examples of **Transliteration**:
- **'Good morning'** becomes **'गुड मॉर्निंग'** in Hindi, where the pronunciation is preserved but the meaning is not translated.
- **'सुप्रभात'** becomes **'suprabhat'** in English.

Available languages:
- **`en-IN`**: English
- **`hi-IN`**: Hindi
- **`bn-IN`**: Bengali
- **`gu-IN`**: Gujarati
- **`kn-IN`**: Kannada
- **`ml-IN`**: Malayalam
- **`mr-IN`**: Marathi
- **`od-IN`**: Odia
- **`pa-IN`**: Punjabi
- **`ta-IN`**: Tamil
- **`te-IN`**: Telugu

For hands-on practice, you can explore the notebook tutorial on [Transliterate API Tutorial](https://github.com/sarvamai/sarvam-ai-cookbook/blob/main/notebooks/transliterate/Transliterate_API_Tutorial.ipynb).
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.text.transliterate(
    input="input",
    source_language_code="auto",
    target_language_code="bn-IN",
)

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**input:** `str` — The text you want to transliterate.
    
</dd>
</dl>

<dl>
<dd>

**source_language_code:** `TransliterateSourceLanguage` 

The language code of the input text. This specifies the source language for transliteration.



 Note:  The source language should either be an Indic language or English. As we supports both Indic-to-English and English-to-Indic transliteration.

    
</dd>
</dl>

<dl>
<dd>

**target_language_code:** `TranslatiterateTargetLanguage` 

The language code of the transliteration text. This specifies the target language for transliteration.



 Note:The target language should either be an Indic language or English. As we supports both Indic-to-English and English-to-Indic transliteration.

    
</dd>
</dl>

<dl>
<dd>

**numerals_format:** `typing.Optional[NumeralsFormat]` 

`numerals_format` is an optional parameter with two options:

- **`international`** (default): Uses regular numerals (0-9).
- **`native`**: Uses language-specific native numerals.

#### Example:
- If `international` format is selected, we use regular numerals (0-9). For example: `मेरा phone number है: 9840950950`.
- If `native` format is selected, we use language-specific native numerals, like: `मेरा phone number है: ९८४०९५०९५०`.
    
</dd>
</dl>

<dl>
<dd>

**spoken_form_numerals_language:** `typing.Optional[SpokenFormNumeralsFormat]` 

`spoken_form_numerals_language` is an optional parameter with two options and only works when spoken_form is true:

- **`english`** : Numbers in the text will be spoken in English.
- **`native(default)`**: Numbers in the text will be spoken in the native language.

#### Examples:
- **Input:** "मेरे पास ₹200 है"
  - If `english` format is selected: "मेरे पास टू हन्डर्ड रूपीस है"
  - If `native` format is selected: "मेरे पास दो सौ रुपये है"

    
</dd>
</dl>

<dl>
<dd>

**spoken_form:** `typing.Optional[bool]` 

  - Default: `False`
  - Converts text into a natural spoken form when `True`.
  - **Note:** No effect if output language is `en-IN`.

#### Example:
- **Input:** `मुझे कल 9:30am को appointment है`
  - **Output:** `मुझे कल सुबह साढ़े नौ बजे को अपॉइंटमेंट है`
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

### SpeechToText
<details><summary><code>client.speech_to_text.<a href="src/sarvamai/speech_to_text/client.py">transcribe</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

### Real-Time Speech to Text API

This API transcribes speech to text in multiple Indian languages and English. Supports real-time transcription for interactive applications.

#### Available Options:
- **Real-Time API** (Current Endpoint): For quick responses under 30 seconds with immediate results
- **Batch API**: For longer audio files, requires following a notebook script - [View Notebook](https://github.com/sarvamai/sarvam-ai-cookbook/tree/main/notebooks/stt/stt-batch-api)
  - Supports diarization (speaker identification)

#### Note:
- Pricing differs for Real-Time and Batch APIs
- Diarization is only available in Batch API with separate pricing
- Please refer to [dashboard.sarvam.ai](https://dashboard.sarvam.ai) for detailed pricing information
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.speech_to_text.transcribe()

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**file:** `from __future__ import annotations

core.File` — See core.File for more documentation
    
</dd>
</dl>

<dl>
<dd>

**model:** `typing.Optional[SpeechToTextModel]` 

Specifies the model to use for speech-to-text conversion.
Note:- Default model is `saarika:v2`
    
</dd>
</dl>

<dl>
<dd>

**language_code:** `typing.Optional[SpeechToTextLanguage]` 

Specifies the language of the input audio. This parameter is required to ensure accurate transcription.
 For the `saarika:v1` model, this parameter is mandatory.
 For the `saarika:v2` model, it is optional.
`unknown`: Use this when the language is not known; the API will detect it automatically. 
Note:- that the `saarika:v1` model does not support `unknown` language code.
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

<details><summary><code>client.speech_to_text.<a href="src/sarvamai/speech_to_text/client.py">translate</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

### Real-Time Speech to Text Translation API

This API automatically detects the input language, transcribes the speech, and translates the text to English.

#### Available Options:
- **Real-Time API** (Current Endpoint): For quick responses under 30 seconds with immediate results
- **Batch API**: For longer audio files, requires following a notebook script - [View Notebook](https://github.com/sarvamai/sarvam-ai-cookbook/tree/main/notebooks/stt-translate/stt-translate-batch-api)
  - Supports diarization (speaker identification)

#### Note:
- Pricing differs for Real-Time and Batch APIs
- Diarization is only available in Batch API with separate pricing
- Please refer to [dashboard.sarvam.ai](https://dashboard.sarvam.ai) for detailed pricing information
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.speech_to_text.translate()

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**file:** `from __future__ import annotations

core.File` — See core.File for more documentation
    
</dd>
</dl>

<dl>
<dd>

**prompt:** `typing.Optional[str]` — Conversation context can be passed as a prompt to boost model accuracy. However, the current system is at an experimentation stage and doesn’t match the prompt performance of large language models.
    
</dd>
</dl>

<dl>
<dd>

**model:** `typing.Optional[SpeechToTextTranslateModel]` — Model to be used for converting speech to text in target language
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

### TextToSpeech
<details><summary><code>client.text_to_speech.<a href="src/sarvamai/text_to_speech/client.py">convert</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

This is the model to convert text into spoken audio. 
The output is a wave file encoded as a base64 string.
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.text_to_speech.convert(
    text="text",
    target_language_code="bn-IN",
)

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**text:** `str` 
    
</dd>
</dl>

<dl>
<dd>

**target_language_code:** `TextToSpeechLanguage` — The language of the text is BCP-47 format
    
</dd>
</dl>

<dl>
<dd>

**speaker:** `typing.Optional[TextToSpeechSpeaker]` 

The speaker voice to be used for the output audio.

**Default:** Meera

**Model Compatibility (Speakers compatible with respective models):**
- **bulbul:v1:**
  - Female: Diya, Maya, Meera, Pavithra, Maitreyi, Misha
  - Male: Amol, Arjun, Amartya, Arvind, Neel, Vian

- **bulbul:v2:**
  - Female: Anushka, Manisha, Vidya, Arya
  - Male: Abhilash, Karun, Hitesh

**Note:** Speaker selection must match the chosen model version.
    
</dd>
</dl>

<dl>
<dd>

**pitch:** `typing.Optional[float]` — Controls the pitch of the audio. Lower values result in a deeper voice, while higher values make it sharper. The suitable range is between -0.75 and 0.75. Default is 0.0.
    
</dd>
</dl>

<dl>
<dd>

**pace:** `typing.Optional[float]` — Controls the speed of the audio. Lower values result in slower speech, while higher values make it faster. The suitable range is between 0.5 and 2.0. Default is 1.0.
    
</dd>
</dl>

<dl>
<dd>

**loudness:** `typing.Optional[float]` — Controls the loudness of the audio. Lower values result in quieter audio, while higher values make it louder. The suitable range is between 0.3 and 3.0. Default is 1.0.
    
</dd>
</dl>

<dl>
<dd>

**speech_sample_rate:** `typing.Optional[SpeechSampleRate]` — Specifies the sample rate of the output audio. Supported values are 8000, 16000, 22050, 24000 Hz. If not provided, the default is 22050 Hz.
    
</dd>
</dl>

<dl>
<dd>

**enable_preprocessing:** `typing.Optional[bool]` —  Controls whether normalization of English words and numeric entities (e.g., numbers, dates) is performed. Set to true for better handling of mixed-language text. Default is false.
    
</dd>
</dl>

<dl>
<dd>

**model:** `typing.Optional[TextToSpeechModel]` — Specifies the model to use for text-to-speech conversion. Default is bulbul:v1.
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>

### Chat
<details><summary><code>client.chat.<a href="src/sarvamai/chat/client.py">completions</a>(...)</code></summary>
<dl>
<dd>

#### 📝 Description

<dl>
<dd>

<dl>
<dd>

Calls Sarvam LLM API to get the chat completion. Supported model(s): `sarvam-m`.
</dd>
</dl>
</dd>
</dl>

#### 🔌 Usage

<dl>
<dd>

<dl>
<dd>

```python
from sarvamai import SarvamAI

client = SarvamAI(
    api_subscription_key="YOUR_API_SUBSCRIPTION_KEY",
)
client.chat.completions(
    messages=[{"content": "content", "role": "assistant"}],
)

```
</dd>
</dl>
</dd>
</dl>

#### ⚙️ Parameters

<dl>
<dd>

<dl>
<dd>

**messages:** `typing.Sequence[ChatCompletionRequestMessageParams]` — A list of messages comprising the conversation so far.
    
</dd>
</dl>

<dl>
<dd>

**temperature:** `typing.Optional[float]` 

What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic.
We generally recommend altering this or `top_p` but not both.
    
</dd>
</dl>

<dl>
<dd>

**top_p:** `typing.Optional[float]` 

An alternative to sampling with temperature, called nucleus sampling,
where the model considers the results of the tokens with top_p probability
mass. So 0.1 means only the tokens comprising the top 10% probability mass
are considered.

We generally recommend altering this or `temperature` but not both.
    
</dd>
</dl>

<dl>
<dd>

**reasoning_effort:** `typing.Optional[ReasoningEffort]` 
    
</dd>
</dl>

<dl>
<dd>

**max_tokens:** `typing.Optional[int]` — The maximum number of tokens that can be generated in the chat completion.
    
</dd>
</dl>

<dl>
<dd>

**stream:** `typing.Optional[bool]` 

If set to true, the model response data will be streamed to the client
as it is generated using [server-sent events](https://developer.mozilla.org/en-US/docs/Web/API/Server-sent_events/Using_server-sent_events#Event_stream_format).
    
</dd>
</dl>

<dl>
<dd>

**stop:** `typing.Optional[StopConfigurationParams]` 
    
</dd>
</dl>

<dl>
<dd>

**n:** `typing.Optional[int]` — How many chat completion choices to generate for each input message. Note that you will be charged based on the number of generated tokens across all of the choices. Keep `n` as `1` to minimize costs.
    
</dd>
</dl>

<dl>
<dd>

**seed:** `typing.Optional[int]` 

This feature is in Beta.
If specified, our system will make a best effort to sample deterministically, such that repeated requests with the same `seed` and parameters should return the same result.
Determinism is not guaranteed, and you should refer to the `system_fingerprint` response parameter to monitor changes in the backend.
    
</dd>
</dl>

<dl>
<dd>

**frequency_penalty:** `typing.Optional[float]` 

Number between -2.0 and 2.0. Positive values penalize new tokens based on
their existing frequency in the text so far, decreasing the model's
likelihood to repeat the same line verbatim.
    
</dd>
</dl>

<dl>
<dd>

**presence_penalty:** `typing.Optional[float]` 

Number between -2.0 and 2.0. Positive values penalize new tokens based on
whether they appear in the text so far, increasing the model's likelihood
to talk about new topics.
    
</dd>
</dl>

<dl>
<dd>

**wiki_grounding:** `typing.Optional[bool]` — If this parameter is enabled, then the model uses a RAG based approach to retrieve relevant chunks from Wikipedia and uses them to answer the question. This is particularly useful for queries seeking factual information.
    
</dd>
</dl>

<dl>
<dd>

**request_options:** `typing.Optional[RequestOptions]` — Request-specific configuration.
    
</dd>
</dl>
</dd>
</dl>


</dd>
</dl>
</details>
