"""
ML Context Detection Utilities

This module provides utilities to detect when binary data appears to be ML model weights
or other legitimate ML content, helping to reduce false positives in security scanning.
"""

import struct
from typing import Any


def analyze_binary_for_ml_context(data: bytes, file_size: int = 0) -> dict[str, Any]:
    """
    Analyze binary data to determine if it appears to be ML model weights.

    Args:
        data: Binary data chunk to analyze
        file_size: Total file size for statistical analysis

    Returns:
        Dict containing context analysis with confidence scores
    """
    context = {
        "appears_to_be_weights": False,
        "weight_confidence": 0.0,
        "pattern_density": 0.0,
        "float_ratio": 0.0,
        "statistical_expectation": 0.0,
        "file_size_factor": 0.0,
    }

    if len(data) < 16:
        return context

    # Analyze floating-point data patterns
    float_analysis = _analyze_float_patterns(data)
    context["float_ratio"] = float_analysis["float_ratio"]

    # Calculate statistical expectation for random patterns
    if file_size > 0:
        context["statistical_expectation"] = _calculate_pattern_expectation(len(data), file_size)
        context["file_size_factor"] = min(file_size / (100 * 1024 * 1024), 1.0)  # Cap at 100MB

    # Detect weight-like characteristics
    weight_indicators = _detect_weight_characteristics(data)
    context.update(weight_indicators)

    # Calculate overall confidence
    confidence_factors = [
        context["float_ratio"] * 0.4,  # 40% weight on float patterns
        weight_indicators.get("distribution_score", 0) * 0.3,  # 30% on distribution
        context["file_size_factor"] * 0.2,  # 20% on file size
        weight_indicators.get("range_score", 0) * 0.1,  # 10% on value ranges
    ]

    context["weight_confidence"] = sum(confidence_factors)
    context["appears_to_be_weights"] = context["weight_confidence"] > 0.6

    return context


def should_ignore_executable_signature(
    signature: bytes,
    offset: int,
    ml_context: dict[str, Any],
    pattern_density: float = 0.0,  # Changed from int to float
    total_patterns: int = 0,
) -> bool:
    """
    Determine if an executable signature should be ignored based on ML context.

    Args:
        signature: The detected executable signature
        offset: Position in file where signature was found
        ml_context: ML context analysis from analyze_binary_for_ml_context
        pattern_density: Number of patterns per MB
        total_patterns: Total number of patterns found

    Returns:
        True if the signature should be ignored as a likely false positive
    """
    # Handle specific signatures with custom logic first
    if signature == b"MZ":
        # MZ signatures in the middle of files are often false positives in weight data
        # Be more permissive for MZ patterns not at file start
        if offset > 100:  # Not at very beginning (allow for small headers)
            return (
                ml_context.get("weight_confidence", 0) > 0.3  # Lower threshold for MZ
                or pattern_density < 5  # Very sparse patterns likely coincidental
            )
        return False  # Always flag MZ at file start

    # Shell script shebangs are the most common false positive in weight data
    if signature == b"#!/":
        return _should_ignore_shebang_pattern(ml_context, pattern_density, total_patterns)

    # Always flag actual executables at file start (for other signatures)
    if offset < 1024:  # First 1KB should not contain weight data
        return False

    if signature == b"\x7fELF":
        # ELF signatures are longer and less likely to be coincidental
        return (
            ml_context.get("appears_to_be_weights", False)
            and ml_context.get("weight_confidence", 0) > 0.8
            and pattern_density > 50  # Very high density suggests coincidental
        )

    # Default: don't ignore other signatures
    return False


def _should_ignore_shebang_pattern(
    ml_context: dict[str, Any],
    pattern_density: float,  # Changed from int to float to handle 0.0 values
    total_patterns: int,
) -> bool:
    """Specific logic for shell script shebang patterns in ML context."""
    # Strong indicators this is weight data
    weight_confidence = ml_context.get("weight_confidence", 0)

    # Lower threshold for shebang patterns since they're most common false positive
    # BERT has 0.69 confidence which should be ignored as false positive
    if weight_confidence > 0.6:  # Lowered from 0.7 to 0.6
        # For large ML model files, be more permissive with pattern counts
        expected_patterns = ml_context.get("statistical_expectation", 0)
        file_size_factor = ml_context.get("file_size_factor", 0)

        # Adjust multiplier based on file size and confidence
        # Large ML models (>100MB) can have more coincidental patterns
        if file_size_factor > 0.5:  # File > 50MB
            multiplier = 10 + (weight_confidence - 0.6) * 25  # 10x to 20x for very confident
        else:
            multiplier = 5 + (weight_confidence - 0.6) * 12.5  # 5x to 10x for smaller files

        if total_patterns <= expected_patterns * multiplier:
            return True

        # Also consider very low pattern density as indicator of coincidental patterns
        # BERT shows 0.0 density with 8 patterns - this should be ignored
        if pattern_density < 2.0:  # Less than 2 patterns per MB is very sparse
            return True

    # Even with moderate confidence (0.5-0.6), ignore if pattern density is extremely low
    # This handles cases like BERT with many patterns but spread across very large files
    if weight_confidence > 0.5 and pattern_density < 1.0:  # Less than 1 pattern per MB
        return True

    # AGGRESSIVE FIX: For very large ML models with low density, always ignore shebang patterns
    # BERT case: 0.697 confidence, 0.0 density, 8 patterns in 440MB file
    if weight_confidence > 0.5 and pattern_density <= 0.1:  # Nearly 0 density patterns
        return True

    # Very high pattern density might indicate actual embedded content
    if pattern_density > 100:  # More than 100 patterns per MB
        return weight_confidence > 0.5

    return False


def _analyze_float_patterns(data: bytes) -> dict[str, float]:
    """Analyze data for floating-point patterns typical of ML weights."""
    if len(data) < 8:
        return {"float_ratio": 0.0}

    valid_floats = 0
    total_attempts = 0

    # Sample every 4 bytes as potential float32
    for i in range(0, len(data) - 4, 4):
        try:
            value = struct.unpack("f", data[i : i + 4])[0]
            total_attempts += 1

            # Check if it's a reasonable float value
            if (
                value == value  # Not NaN
                and abs(value) < 1e10  # Not extremely large
                and (value == 0.0 or abs(value) > 1e-10)  # Not extremely small (unless zero)
            ):
                valid_floats += 1

        except (struct.error, OverflowError):
            total_attempts += 1

    float_ratio = valid_floats / total_attempts if total_attempts > 0 else 0.0
    return {"float_ratio": float_ratio}


def _detect_weight_characteristics(data: bytes) -> dict[str, float]:
    """Detect characteristics typical of neural network weights."""
    if len(data) < 32:
        return {"distribution_score": 0.0, "range_score": 0.0}

    # Sample some float values for statistical analysis
    values = []
    for i in range(0, min(len(data) - 4, 1000), 8):  # Sample every 8 bytes, max 125 samples
        try:
            value = struct.unpack("f", data[i : i + 4])[0]
            if value == value and abs(value) < 1e6:  # Filter out NaN and extreme values
                values.append(value)
        except (struct.error, OverflowError):
            continue

    if len(values) < 10:
        return {"distribution_score": 0.0, "range_score": 0.0}

    # Analyze distribution characteristics
    mean_val = sum(values) / len(values)
    variance = sum((x - mean_val) ** 2 for x in values) / len(values)
    std_dev = variance**0.5

    # Weight-like characteristics:
    # 1. Values typically in range [-10, 10]
    range_score = sum(1 for v in values if -10 <= v <= 10) / len(values)

    # 2. Distribution is often roughly normal-ish (not too skewed)
    # Simple check: most values within 3 standard deviations
    within_3_sigma = sum(1 for v in values if abs(v - mean_val) <= 3 * std_dev) / len(values)
    distribution_score = min(within_3_sigma, 1.0)

    return {
        "distribution_score": distribution_score,
        "range_score": range_score,
        "sample_size": len(values),
        "mean": mean_val,
        "std_dev": std_dev,
    }


def _calculate_pattern_expectation(chunk_size: int, file_size: int) -> float:
    """
    Calculate expected number of 3-byte patterns in random data.

    For a 3-byte pattern like '#!/' (0x23 0x21 0x2f), the probability of
    any 3-byte sequence matching is 1/(256^3) = 1/16,777,216.
    """
    if chunk_size < 3:
        return 0.0

    possible_positions = max(chunk_size - 2, 1)  # Number of 3-byte positions in chunk
    probability_per_position = 1.0 / (256.0**3)  # Probability of specific 3-byte pattern

    return possible_positions * probability_per_position


def get_ml_context_explanation(ml_context: dict[str, Any], pattern_count: int) -> str:
    """Generate human-readable explanation of why a pattern was ignored."""
    if not ml_context.get("appears_to_be_weights", False):
        return ""

    explanations = []

    if ml_context.get("weight_confidence", 0) > 0.7:
        explanations.append(f"High confidence ML weights (score: {ml_context['weight_confidence']:.2f})")

    if ml_context.get("float_ratio", 0) > 0.6:
        explanations.append(f"High floating-point data ratio ({ml_context['float_ratio']:.1%})")

    expected = ml_context.get("statistical_expectation", 0)
    if expected > 0 and pattern_count > 2 and pattern_count <= expected * 3:
        explanations.append(f"Pattern count ({pattern_count}) within statistical expectation ({expected:.1f})")

    return "; ".join(explanations) if explanations else "Appears to be legitimate ML weight data"
