"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const index_1 = require("./index");
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const mockApp = new core_1.App();
const stack = new core_1.Stack(mockApp, 'testing-stack', { env });
/**
 * Example create an Secret for testing.
 */
const secret = new aws_secretsmanager_1.CfnSecret(stack, 'Mysecret', {
    secretString: JSON.stringify({
        KEY1: 'VALUE1',
        KEY2: 'VALUE2',
        KEY3: 'VALUE3',
    }),
});
const layer = new index_1.SecretManagerWrapperLayer(stack, 'SecretManagerWrapperLayer');
const lambda = new aws_lambda_1.Function(stack, 'fn', {
    runtime: aws_lambda_1.Runtime.PYTHON_3_9,
    code: aws_lambda_1.Code.fromInline(`
import os
def hander(events, contexts):
    env = {}
    env['KEY1'] = os.environ.get('KEY1', 'Not Found')
    env['KEY2'] = os.environ.get('KEY2', 'Not Found')
    env['KEY3'] = os.environ.get('KEY3', 'Not Found')
    return env
    `),
    handler: 'index.hander',
    layers: [layer],
    timeout: core_1.Duration.minutes(1),
    /**
     * you need to define this 4 environment various.
     */
    environment: {
        AWS_LAMBDA_EXEC_WRAPPER: '/opt/get-secrets-layer',
        SECRET_REGION: stack.region,
        SECRET_ARN: secret.ref,
        API_TIMEOUT: '5000',
    },
});
/**
 * Add Permission for lambda get secret value from secret manager.
 */
lambda.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
    effect: aws_iam_1.Effect.ALLOW,
    actions: ['secretsmanager:GetSecretValue'],
    // Also you can use find from context.
    resources: [secret.ref],
}));
/**
 * For Testing.
 */
const FnUrl = lambda.addFunctionUrl({
    authType: aws_lambda_1.FunctionUrlAuthType.NONE,
});
new core_1.CfnOutput(stack, 'FnUrl', {
    value: FnUrl.url,
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW50ZWcuaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw4Q0FBMkQ7QUFDM0Qsb0RBQW1GO0FBQ25GLG9FQUF3RDtBQUN4RCx3Q0FBZ0U7QUFDaEUsbUNBQW9EO0FBQ3BELE1BQU0sR0FBRyxHQUFHO0lBQ1YsTUFBTSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCO0lBQ3RDLE9BQU8sRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQjtDQUN6QyxDQUFDO0FBQ0YsTUFBTSxPQUFPLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztBQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztBQUUzRDs7R0FFRztBQUNILE1BQU0sTUFBTSxHQUFHLElBQUksOEJBQVMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO0lBQzlDLFlBQVksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDO1FBQzNCLElBQUksRUFBRSxRQUFRO1FBQ2QsSUFBSSxFQUFFLFFBQVE7UUFDZCxJQUFJLEVBQUUsUUFBUTtLQUNmLENBQUM7Q0FDSCxDQUFDLENBQUM7QUFFSCxNQUFNLEtBQUssR0FBRyxJQUFJLGlDQUF5QixDQUFDLEtBQUssRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO0FBRWhGLE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO0lBQ3ZDLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7SUFDM0IsSUFBSSxFQUFFLGlCQUFJLENBQUMsVUFBVSxDQUFDOzs7Ozs7OztLQVFuQixDQUFDO0lBQ0osT0FBTyxFQUFFLGNBQWM7SUFDdkIsTUFBTSxFQUFFLENBQUMsS0FBSyxDQUFDO0lBQ2YsT0FBTyxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBQzVCOztPQUVHO0lBQ0gsV0FBVyxFQUFFO1FBQ1gsdUJBQXVCLEVBQUUsd0JBQXdCO1FBQ2pELGFBQWEsRUFBRSxLQUFLLENBQUMsTUFBTTtRQUMzQixVQUFVLEVBQUUsTUFBTSxDQUFDLEdBQUc7UUFDdEIsV0FBVyxFQUFFLE1BQU07S0FDcEI7Q0FDRixDQUFDLENBQUM7QUFFSDs7R0FFRztBQUNILE1BQU0sQ0FBQyxJQUFLLENBQUMsb0JBQW9CLENBQy9CLElBQUkseUJBQWUsQ0FBQztJQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO0lBQ3BCLE9BQU8sRUFBRSxDQUFDLCtCQUErQixDQUFDO0lBQzFDLHNDQUFzQztJQUN0QyxTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDO0NBQ3hCLENBQUMsQ0FDSCxDQUFDO0FBRUY7O0dBRUc7QUFDSCxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsY0FBYyxDQUFDO0lBQ2xDLFFBQVEsRUFBRSxnQ0FBbUIsQ0FBQyxJQUFJO0NBQ25DLENBQUMsQ0FBQztBQUVILElBQUksZ0JBQVMsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO0lBQzVCLEtBQUssRUFBRSxLQUFLLENBQUMsR0FBRztDQUNqQixDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBFZmZlY3QsIFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgRnVuY3Rpb24sIFJ1bnRpbWUsIENvZGUsIEZ1bmN0aW9uVXJsQXV0aFR5cGUgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IENmblNlY3JldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQgeyBBcHAsIFN0YWNrLCBDZm5PdXRwdXQsIER1cmF0aW9uIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBTZWNyZXRNYW5hZ2VyV3JhcHBlckxheWVyIH0gZnJvbSAnLi9pbmRleCc7XG5jb25zdCBlbnYgPSB7XG4gIHJlZ2lvbjogcHJvY2Vzcy5lbnYuQ0RLX0RFRkFVTFRfUkVHSU9OLFxuICBhY2NvdW50OiBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5ULFxufTtcbmNvbnN0IG1vY2tBcHAgPSBuZXcgQXBwKCk7XG5jb25zdCBzdGFjayA9IG5ldyBTdGFjayhtb2NrQXBwLCAndGVzdGluZy1zdGFjaycsIHsgZW52IH0pO1xuXG4vKipcbiAqIEV4YW1wbGUgY3JlYXRlIGFuIFNlY3JldCBmb3IgdGVzdGluZy5cbiAqL1xuY29uc3Qgc2VjcmV0ID0gbmV3IENmblNlY3JldChzdGFjaywgJ015c2VjcmV0Jywge1xuICBzZWNyZXRTdHJpbmc6IEpTT04uc3RyaW5naWZ5KHtcbiAgICBLRVkxOiAnVkFMVUUxJyxcbiAgICBLRVkyOiAnVkFMVUUyJyxcbiAgICBLRVkzOiAnVkFMVUUzJyxcbiAgfSksXG59KTtcblxuY29uc3QgbGF5ZXIgPSBuZXcgU2VjcmV0TWFuYWdlcldyYXBwZXJMYXllcihzdGFjaywgJ1NlY3JldE1hbmFnZXJXcmFwcGVyTGF5ZXInKTtcblxuY29uc3QgbGFtYmRhID0gbmV3IEZ1bmN0aW9uKHN0YWNrLCAnZm4nLCB7XG4gIHJ1bnRpbWU6IFJ1bnRpbWUuUFlUSE9OXzNfOSxcbiAgY29kZTogQ29kZS5mcm9tSW5saW5lKGBcbmltcG9ydCBvc1xuZGVmIGhhbmRlcihldmVudHMsIGNvbnRleHRzKTpcbiAgICBlbnYgPSB7fVxuICAgIGVudlsnS0VZMSddID0gb3MuZW52aXJvbi5nZXQoJ0tFWTEnLCAnTm90IEZvdW5kJylcbiAgICBlbnZbJ0tFWTInXSA9IG9zLmVudmlyb24uZ2V0KCdLRVkyJywgJ05vdCBGb3VuZCcpXG4gICAgZW52WydLRVkzJ10gPSBvcy5lbnZpcm9uLmdldCgnS0VZMycsICdOb3QgRm91bmQnKVxuICAgIHJldHVybiBlbnZcbiAgICBgKSxcbiAgaGFuZGxlcjogJ2luZGV4LmhhbmRlcicsXG4gIGxheWVyczogW2xheWVyXSxcbiAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxKSxcbiAgLyoqXG4gICAqIHlvdSBuZWVkIHRvIGRlZmluZSB0aGlzIDQgZW52aXJvbm1lbnQgdmFyaW91cy5cbiAgICovXG4gIGVudmlyb25tZW50OiB7XG4gICAgQVdTX0xBTUJEQV9FWEVDX1dSQVBQRVI6ICcvb3B0L2dldC1zZWNyZXRzLWxheWVyJyxcbiAgICBTRUNSRVRfUkVHSU9OOiBzdGFjay5yZWdpb24sXG4gICAgU0VDUkVUX0FSTjogc2VjcmV0LnJlZixcbiAgICBBUElfVElNRU9VVDogJzUwMDAnLFxuICB9LFxufSk7XG5cbi8qKlxuICogQWRkIFBlcm1pc3Npb24gZm9yIGxhbWJkYSBnZXQgc2VjcmV0IHZhbHVlIGZyb20gc2VjcmV0IG1hbmFnZXIuXG4gKi9cbmxhbWJkYS5yb2xlIS5hZGRUb1ByaW5jaXBhbFBvbGljeShcbiAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgYWN0aW9uczogWydzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZSddLFxuICAgIC8vIEFsc28geW91IGNhbiB1c2UgZmluZCBmcm9tIGNvbnRleHQuXG4gICAgcmVzb3VyY2VzOiBbc2VjcmV0LnJlZl0sXG4gIH0pLFxuKTtcblxuLyoqXG4gKiBGb3IgVGVzdGluZy5cbiAqL1xuY29uc3QgRm5VcmwgPSBsYW1iZGEuYWRkRnVuY3Rpb25Vcmwoe1xuICBhdXRoVHlwZTogRnVuY3Rpb25VcmxBdXRoVHlwZS5OT05FLFxufSk7XG5cbm5ldyBDZm5PdXRwdXQoc3RhY2ssICdGblVybCcsIHtcbiAgdmFsdWU6IEZuVXJsLnVybCxcbn0pOyJdfQ==