# STIX mapping

## Standard Objects

### TLPs SMOs

At script run time user can set `tlp_level`; either white, green, amber or red.

Each mode maps to a STIX marking definition object;

* Clear: `marking-definition--94868c89-83c2-464b-929b-a1a8aa3c8487`
* Green: `marking-definition--bab4a63c-aed9-4cf5-a766-dfca5abac2bb`
* Amber: `marking-definition--55d920b0-5e8b-4f79-9ee9-91f868d9b421`
* Amber+Strict: `marking-definition--939a9414-2ddd-4d32-a0cd-375ea402b003`
* Red: `marking-definition--e828b379-4e03-4974-9ac4-e53a884c97c1`

Depending on the value set by user, the generated STIX objects will contain a `marking-definition--` reference in the `object_marking_refs` field to the corresponding TLP level.

### Marking definitions

All objects created by txt2stix also have a standard marking definition in the `object_marking_refs` property.

The default STIX 2.1 marking definition for txt2stix is always used and imported from https://raw.githubusercontent.com/muchdogesec/stix4doge/main/objects/marking-definition/txt2stix.json

This object will be printed in the final bundle, and in all objects generated by txt2stix in the `object_marking_refs` property.

### Identities (`identity`) SDOs

txt2stix assigns a `created_by_ref` property to all SDOs and SROs it creates to their `created_by_ref` field.

The default STIX 2.1 identity for txt2stix is imported from https://raw.githubusercontent.com/muchdogesec/stix4doge/main/objects/identity/txt2stix.json

This object will be printed in the final bundle, and the ID will be used for all `created_by_ref` generated by txt2stix.

Often a user will want to use a custom ID of their own to generate objects. They can do this using the `use_identity` flag in the CLI.

Two things will happen if a custom identity is used in this way. The identity JSON object will be printed in the final bundle, and the ID will be used for all `created_by_ref` generated by txt2stix.

### Report SDO (`report`)

All files uploaded are represented as a unique [STIX Report SDO](https://docs.oasis-open.org/cti/stix/v2.1/os/stix-v2.1-os.html#_n8bjzg1ysgdq) that take the following structure;

```json
{
    "type": "report",
    "spec_version": "2.1",
    "id": "report--<REPORT ID PASSED OR UUID V4 GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<ITEM INGEST DATE>",
    "modified": "<ITEM INGEST DATE>",
    "name": "<NAME ENTERED ON UPLOAD>",
    "description": "<FULL BODY OF TEXT FROM FILE>",
    "confidence": "<CONFIDENCE VALUE PASSED AT CLI, IF EXISTS, ELSE NOT PRINTED>",
    "published": "<ITEM INGEST DATE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "labels": [
        "<LABELS ADDED BY USER>", "<CLASSIFICATION TAGS IF AI CONTENT CHECK PROVIDER SET>"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix Report MD5",
            "external_id": "<MD5 HASH OF DESCRIPTION FIELD>"
        }   
    ],
    "object_refs": [
        "<LIST OF ALL VISIBLE EXTRACTED SDO OBJECTS EXCEPT NOTE>"
    ]
}
```

Note, the `object_refs` contains all references that are referenced by objects in the report (SDOs, SROs, SCOs, marking definitions, etc.). This includes extracted objects (i.e. Indicator SDOs, Vulnerability SDOs, Software SCOs, Relationship SROs etc.). The only object it does not include is the Note object created automatically.

`<REPORT OBJECT ID>` should match the report object UUID, e.g. if this was 

## STIX Mapping (locally created objects)

All extractions detected by txt2stix are converted to STIX.

In the extractions config (`includes/extractions/ai|lookup|pattern/config.yaml)`, a `stix-mapping` property exists for all extractions. 

Below shows how the data extracted is mapped to a set of STIX objects for each `stix-mapping` type.

### stix-mapping: `ipv4-addr`

Objects always created:

* `indicator`
* `ipv4-addr`
* `relationship` (`ipv4-addr` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: ipv4-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "ipv4: <EXTRACTED IPV4 OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ ipv4-addr:value = '<EXTRACTED IPV4 OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "ipv4-addr",
    "spec_version": "2.1",
    "id": "ipv4-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED IPV4 OBSERVABLE VALUE>"
}
```

### stix-mapping: `ipv4-addr-port`

Objects always created:

* `indicator`
* `ipv4-addr` 
* `relationship` (`ipv4-addr` -> `indicator`)
* `network-traffic`

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: ipv4-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "ipv4: <EXTRACTED IPV4 OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ ipv4-addr:value = '<EXTRACTED IPV4 OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "ipv4-addr",
    "spec_version": "2.1",
    "id": "ipv4-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED IPV4 OBSERVABLE VALUE>"
}
```

When a port is reported, it could be either a source or destination port.

Generally, threat intel research reports cover destination ports when reported with an IP (they report on what was seen). Therefore, a conscious decision was made that txt2stix always classifies IPs with port numbers as showing destination ports.

```json
{
    "type": "network-traffic",
    "spec_version": "2.1",
    "id": "network-traffic--<GENERATED BY STIX2 LIBRARY>",
    "dst_ref": "ipv4-addr--<IPV4 OBJECT ID>",
    "dst_port": "<EXTRACTED IPV4 PORT VALUE>",
    "protocols": [
        "ipv4"
    ]
}
```

### stix-mapping: `ipv6-addr`

Objects always created:

* `indicator`
* `ipv6-addr`
* `relationship` (`ipv6-addr` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: ipv6-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "ipv6: <EXTRACTED IPV6 OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ ipv6-addr:value = '<EXTRACTED IPV6 OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "ipv6-addr",
    "spec_version": "2.1",
    "id": "ipv4-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED IPV6 OBSERVABLE VALUE>"
}
```

### stix-mapping: `ipv6-addr-port`

Objects always created:

* `indicator`
* `ipv6-addr`
* `relationship` (`ipv6-addr` -> `indicator`)
* `network-traffic`

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: ipv6-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "ipv6: <EXTRACTED IPV6 OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ ipv6-addr:value = '<EXTRACTED IPV6 OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "ipv6-addr",
    "spec_version": "2.1",
    "id": "ipv4-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED IPV6 OBSERVABLE VALUE>"
}
```

```json
{
    "type": "network-traffic",
    "spec_version": "2.1",
    "id": "network-traffic--<GENERATED BY STIX2 LIBRARY>",
    "dst_ref": "ipv6-addr--<IPV6 OBJECT ID>",
    "dst_port": "<EXTRACTED IPV6 PORT VALUE>",
    "protocols": [
        "ipv4"
    ]
}
```

### stix-mapping: `domain-name`

Objects always created:

* `indicator`
* `domain-name`
* `relationship` (`domain-name` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: domain-name is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Domain: <EXTRACTED DOMAIN NAME OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ domain-name:value = '<EXTRACTED DOMAIN NAME OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "domain-name",
    "spec_version": "2.1",
    "id": "domain-name--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED DOMAIN NAME VALUE>"
}
```

### stix-mapping: `url`

Objects always created:

* `indicator`
* `url`
* `relationship` (`url` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: url is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "URL: <EXTRACTED URL OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ url:value = '<EXTRACTED URL OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{  
    "type": "url",
    "spec_version": "2.1",
    "id": "url--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED URL VALUE>"
}
```

### stix-mapping: `file`

Objects always created:

* `indicator`
* `file`
* `relationship` (`file` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: file is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "File name: <EXTRACTED FILE NAME OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ file:name = '<EXTRACTED FILE NAME OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "file",
    "spec_version": "2.1",
    "id": "file--<GENERATED BY STIX2 LIBRARY>",
    "name": "<EXTRACTED FILE NAME VALUE>"
}
```

### stix-mapping: `directory`

Objects always created:

* `indicator`
* `directory`
* `relationship` (`directory` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO:`indicator` `extracted-from` `report`
* AI mode relationship SROs: directory is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Directory: <EXTRACTED DIRECTORY OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ directory:path = '<EXTRACTED DIRECTORY OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "directory",
    "spec_version": "2.1",
    "id": "directory--<GENERATED BY STIX2 LIBRARY>",
    "path": "<EXTRACTED DIRECTORY OBSERVABLE VALUE>"
}
```

### stix-mapping: `directory-file`

Objects always created:

* `indicator`
* `directory`
* `relationship` (`directory` -> `indicator`)
* `file`

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: directory is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Directory: <EXTRACTED DIRECTORY OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ directory:path = '<EXTRACTED DIRECTORY OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "directory",
    "spec_version": "2.1",
    "id": "directory--<GENERATED BY STIX2 LIBRARY>",
    "path": "<EXTRACTED DIRECTORY OBSERVABLE VALUE>",

}
```

```json
{
    "type": "file",
    "spec_version": "2.1",
    "id": "file--<GENERATED BY STIX2 LIBRARY>",
    "name": "<EXTRACTED FILE NAME VALUE>"
}
```

Note, we decided not to add the `parent_directory_ref` to the file object, as we may see the same file in different directories on different txt2stix runs. As the ID contributing properties of the file object do not consider `parent_directory_ref` it means all filenames with different paths will always have the same ID, which is not what we wanted. Thus, we standardise the filenames to always include the same properties, and use an SRO to join it to the directory.

```json
{
    "type": "relationship",
    "spec_version": "2.1",
    "id": "relationship--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<SIGNALS CORPS IDENTITY ID>",
    "created": "<REPORT CREATED DATE>",
    "modified": "<REPORT CREATED DATE>",
    "relationship_type": "directory",
    "source_ref": "file--<FILE OBJECT>",
    "target_ref": "directory--<DIRECTORY OBJECT>",
    "description": "<FILE NAME> is in the directory <DIRECTORY PATH>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `file-hash`

Objects always created:

* `indicator`
* `file`
* `relationship` (`file` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: file is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "<FILE HASH TYPE>: <EXTRACTED FILE HASH OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ file:hashes.<FILE HASH TYPE> = '<EXTRACTED FILE HASH OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        },
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
    ]
}
```

```json
{
    "type": "file",
    "spec_version": "2.1",
    "id": "file--<GENERATED BY STIX2 LIBRARY>",
    "hashes": {
        "<FILE HASH TYPE>": "<EXTRACTED FILE HASH OBSERVABLE VALUE>"
    }
}
```

### stix-mapping: `email-addr`

Objects always created:

* `indicator`
* `email-addr`
* `relationship` (`email-addr` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: email-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Email Address: <EXTRACTED EMAIL ADDRESS OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ email-addr:value = '<EXTRACTED EMAIL ADDRESS OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "email-addr",
    "spec_version": "2.1",
    "id": "email-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED EMAIL ADDRESS OBSERVABLE VALUE>"
}
```

### stix-mapping: `mac-addr`

Objects always created:

* `indicator`
* `mac-addr`
* `relationship` (`mac-addr` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: mac-addr is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "MAC Address: <EXTRACTED MAC ADDRESS OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ mac-addr:value = '<EXTRACTED MAC ADDRESS OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "mac-addr",
    "spec_version": "2.1",
    "id": "mac-addr--<GENERATED BY STIX2 LIBRARY>",
    "value": "<EXTRACTED MAC ADDRESS OBSERVABLE VALUE>"
}
```

### stix-mapping: `windows-registry-key`

Objects always created:

* `indicator`
* `windows-registry-key`
* `relationship` (`windows-registry-key` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: windows-registry-key is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Windows Registry Key: <EXTRACTED WINDOWS REGISTRY KEY OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ windows-registry-key:key = '<EXTRACTED WINDOWS REGISTRY KEY OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "windows-registry-key",
    "spec_version": "2.1",
    "id": "windows-registry-key--<GENERATED BY STIX2 LIBRARY>",
    "key": "<EXTRACTED WINDOWS REGISTRY KEY OBSERVABLE VALUE>"
}
```

### stix-mapping: `autonomous-system`

Objects always created:

* `indicator`
* `autonomous-system`
* `relationship` (`autonomous-system` -> `indicator`)

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: autonomous-system is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "AS<EXTRACTED NUMERICAL AS OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ autonomous-system:number = '<EXTRACTED NUMERICAL AS OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "autonomous-system",
    "spec_version": "2.1",
    "id": "autonomous-system--<GENERATED BY STIX2 LIBRARY>",
    "number": "<EXTRACTED NUMERICAL AS OBSERVABLE VALUE>"
}
```

### stix-mapping: `user-agent`

Objects always created:

* `indicator`
* `user-agent`
* `relationship` (`user-agent` -> `indicator`)
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/user-agent.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: user-agent is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "User Agent: <EXTRACTED FULL USER AGENT STRING>",
    "pattern_type": "stix",
    "pattern": "[ user-agent:string = '<EXTRACTED FULL USER AGENT STRING>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "user-agent",
    "spec_version": "2.1",
    "id": "user-agent--<GENERATED BY STIX2 LIBRARY>",
    "string": "<EXTRACTED FULL USER AGENT STRING>",
    "extensions": {
        "extension-definition--7ca5afee-0e4e-5813-b643-de51538658cc" : {
            "extension_type" : "new-sco"
        }
    }
}
```

### stix-mapping: `cryptocurrency-wallet`

Objects always created:

* `indicator`
* `cryptocurrency-wallet`
* `relationship` (`cryptocurrency-wallet` -> `indicator`)
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/cryptocurrency-wallet.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: cryptocurrency-wallet is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
  {
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "<CRYPTO TYPE> Wallet: <EXTRACTED CRYPTOCURRENCY OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ cryptocurrency-wallet:address = '<EXTRACTED CRYPTOCURRENCY OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

The `cryptocurrency-wallet` object is generated by [crypto2stix](https://github.com/muchdogesec/crypto2stix).

The crypto2stix equivilant command is;

```shell
python3 crypto2stix.py --wallet HASH --wallet_only
```

Which only generates the wallet object.

### stix-mapping: `cryptocurrency-wallet-with-transaction`

Objects always created:

* `indicator`
* `cryptocurrency-wallet`
* `relationship` (`cryptocurrency-wallet` -> `indicator`)
* `cryptocurrency-transaction` for all crypto transactions that exist related to the wallet (is not always generated if lookup unsuccessful)
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/cryptocurrency-wallet.json
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/cryptocurrency-transaction.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: cryptocurrency-wallet is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
  {
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "<CRYPTO TYPE> Wallet: <EXTRACTED CRYPTOCURRENCY OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ cryptocurrency-wallet:address = '<EXTRACTED CRYPTOCURRENCY OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

The `cryptocurrency-transaction` object is generated by [crypto2stix](https://github.com/muchdogesec/crypto2stix).

The crypto2stix equivilant command is;

```shell
python3 crypto2stix.py --wallet HASH --transactions_only
```

Which generates the wallet object and any transactions associated with it.

### stix-mapping: `cryptocurrency-transaction`

Objects always created:

* `indicator`
* `cryptocurrency-transaction`
* `relationship` (`cryptocurrency-transaction` -> `indicator`)
* `cryptocurrency-wallet` for wallets seen in transaction identified by crypto2stix (is not always generated if lookup unsuccessful)
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/cryptocurrency-transaction.json
* `extensions-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/cryptocurrency-wallet.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: cryptocurrency-transaction is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
  {
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "<CRYPTO TYPE> Transaction: <EXTRACTED TRANSACTION HASH>",
    "pattern_type": "stix",
    "pattern": "[ cryptocurrency-transaction:hash = '<EXTRACTED CRYPTOCURRENCY HASH VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "cryptocurrency-transaction",
    "spec_version": "2.1",
    "id": "cryptocurrency-transaction--<UUIDV5>",
    "currency_symbol": "<EXTRACTED CRYPTOCURRENCY OBSERVABLE VALUE>",
    "hash": "<EXTRACTED TRANSACTION HASH>",
    "timestamp": "2022-10-02T15:22:21Z",
    "extensions": {
        "extension-definition--151d042d-4dcf-5e44-843f-1024440318e5" : {
            "extension_type" : "new-sco"
        }
    }
}
```

The `cryptocurrency-transaction` object is generated by [crypto2stix](https://github.com/muchdogesec/crypto2stix).

The crypto2stix equivilant command is;

```shell
python3 crypto2stix.py --transaction HASH
```

This will also generate all `cryptocurrency-wallets` seen in the transaction.

### stix-mapping: `bank-card`

Objects always created:

* `indicator`
* `bank-card` (with `relationship` to `indicator`)
* `relationship` (`bank-card` -> `indicator`)
* `identity` (with `relationship` to `bank-card`) generated by creditcard2stix (is not always generated if lookup unsuccessful)
* `extension-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/bank-card.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: bank-card is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "<CARD TYPE>: <EXTRACTED CREDIT CARD OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ bank-card:number = '<EXTRACTED CREDIT CARD OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

The `bank-card` object is generated by [creditcard2stix](https://github.com/muchdogesec/creditcard2stix). This will require users to enter an `BIN_LIST_API_KEY` in the `.env` file.

### stix-mapping: `bank-account`

Objects always created:

* `indicator`
* `bank-account`
* `relationship` (`bank-card` -> `indicator`)
* `extension-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/bank-account.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: bank-card is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Bank account: <EXTRACTED IBAN OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ bank-account:iban_number = '<EXTRACTED IBAN OBSERVABLE VALUE>' ]",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "bank-account",
    "spec_version": "2.1",
    "id": "bank-account--<UUIDV5>",
    "iban_number": "<FULL IBAN NUMBER INCLUDING COUNTRY CODE>",
    "extensions": {
        "extension-definition--f19f3291-6a84-5674-b311-d75a925d5bd9": {
            "extension_type" : "new-sco"
        }
    }
}
```

To ensure duplicate `bank-account` objects are not created for the same values, a UUIDv5 address is generated for the ID as follows;

* Namespace = `00abedb4-aa42-466c-9c01-fed23315a9b7` (this is the default MITRE namespace used in the stix2 python lib https://github.com/oasis-open/cti-python-stix2/blob/50fd81fd6ba4f26824a864319305bc298e89bb45/stix2/base.py#L29)
* Value = `<iban_number>`

### stix-mapping: `phone-number`

Objects always created:

* `indicator`
* `phone-number`
* `relationship` (`phone-number` -> `indicator`)
* `extension-definition`: https://raw.githubusercontent.com/muchdogesec/stix2extensions/main/extension-definitions/scos/phone-number.json

Relationship mode object generation behaviour:

* Standard relationship SRO: `indicator` `extracted-from` `report`
* AI mode relationship SROs: phone-number is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "indicator",
    "spec_version": "2.1",
    "id": "indicator--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "indicator_types": [
        "unknown"
    ],
    "name": "Phone Number: <EXTRACTED PHONE OBSERVABLE VALUE>",
    "pattern_type": "stix",
    "pattern": "[ phone-number:number = '<EXTRACTED PHONE OBSERVABLE VALUE>'",
    "valid_from": "<REPORT CREATED PROPERTY VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

```json
{
    "type": "phone-number",
    "spec_version": "2.1",
    "id": "phone-number--<UUIDV5>",
    "number": "<EXTRACTED PHONE OBSERVABLE VALUE>",
    "extensions": {
        "extension-definition--14a97ee2-e666-5ada-a6bd-b7177f79e211" : {
            "extension_type" : "new-sco"
        }
    }
}
```

To ensure duplicate `phone-number` objects are not created for the same values, a UUIDv5 address is generated for the ID as follows;

* Namespace = `00abedb4-aa42-466c-9c01-fed23315a9b7` (this is the default MITRE namespace used in the stix2 python lib https://github.com/oasis-open/cti-python-stix2/blob/50fd81fd6ba4f26824a864319305bc298e89bb45/stix2/base.py#L29)
* Value = `<number>`

### stix-mapping: `attack-pattern`

Objects created:

* `attack-pattern`

Relationship mode object generation behaviour:

* Standard relationship SRO: `attack-pattern` `extracted-from` `report`
* AI mode relationship SROs: attack-pattern is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "attack-pattern",
    "spec_version": "2.1",
    "id": "campaign--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `campaign`

Objects created:

* `campaign`

Relationship mode object generation behaviour:

* Standard relationship SRO: `campaign` `extracted-from` `report`
* AI mode relationship SROs: campaign is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "campaign",
    "spec_version": "2.1",
    "id": "campaign--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `course-of-action`

Objects created:

* `course-of-action`

Relationship mode object generation behaviour:

* Standard relationship SRO: `course-of-action` `extracted-from` `report`
* AI mode relationship SROs: course-of-action is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "course-of-action",
    "spec_version": "2.1",
    "id": "course-of-action--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `infrastructure`

Objects created:

* `infrastructure`

Relationship mode object generation behaviour:

* Standard relationship SRO: `infrastructure` `extracted-from` `report`
* AI mode relationship SROs: infrastructure is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type":"infrastructure",
    "spec_version": "2.1",
    "id":"infrastructure--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "infrastructure_types": ["unknown"],
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `intrusion-set`

Objects created:

* `intrusion-set`

Relationship mode object generation behaviour:

* Standard relationship SRO: `intrusion-set` `extracted-from` `report`
* AI mode relationship SROs: intrusion-set is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "intrusion-set",
    "spec_version": "2.1",
    "id": "intrusion-set--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `malware`

Objects created:

* `malware`

Relationship mode object generation behaviour:

* Standard relationship SRO: `malware` `extracted-from` `report`
* AI mode relationship SROs: malware is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "malware",
    "spec_version": "2.1",
    "id": "malware--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "malware_types": ["unknown"],
    "is_family": true,
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `threat-actor`

Objects created:

* `threat-actor`

Relationship mode object generation behaviour:

* Standard relationship SRO: `threat-actor` `extracted-from` `report`
* AI mode relationship SROs: threat-actor is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "threat-actor",
    "spec_version": "2.1",
    "id": "threat-actor--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "threat_actor_types": "unknown",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `tool`

Objects created:

* `tool`

Relationship mode object generation behaviour:

* Standard relationship SRO: `tool` `extracted-from` `report`
* AI mode relationship SROs: tool is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "tool",
    "spec_version": "2.1",
    "id": "tool--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "tool_types": "unknown",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### stix-mapping: `identity`

Objects created:

* `identity`

Relationship mode object generation behaviour:

* Standard relationship SRO: `identity` `extracted-from` `report`
* AI mode relationship SROs: identity is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

```json
{
    "type": "identity",
    "spec_version": "2.1",
    "id": "identity--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT OR CUSTOM IDENTITY OBJECT ID>",
    "created": "<REPORT CREATED PROPERTY VALUE>",
    "modified": "<REPORT MODIFIED PROPERTY VALUE>",
    "name": "<EXTRACTED VALUE>",
    "identity_class": "unspecified",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>",
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

## STIX Mapping (remote created objects)

Some objects created for extractions do not need to be generated by txt2stix, they can be looked up from an external databases.

### stix-mapping: `ctibutler-mitre-attack-enterprise-id`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/attack-enterprise/objects/:attack_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-attack-enterprise-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-enterprise/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-mitre-attack-enterprise-alias`

Takes the extracted alias and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-enterprise/objects/?alias=<VALUE>
```

### stix-mapping: `ctibutler-mitre-attack-mobile-id`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/attack-mobile/objects/:attack_id/
```

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-attack-mobile-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-mobile/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-mitre-attack-mobile-alias`

Takes the extracted alias and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-mobile/objects/?alias=<VALUE>
```

### stix-mapping: `ctibutler-mitre-attack-ics-id`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/attack-ics/objects/:attack_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-attack-ics-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-ics/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-mitre-attack-ics-alias`

Takes the extracted alias and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/attack-ics/objects/?alias=<VALUE>
```

### stix-mapping: `ctibutler-mitre-capec-id`

Takes the extracted ID and passes it to 

```shell
GET CTIBUTLER_BASE_URL/v1/capec/objects/:capec_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported `course-of-action` / `attack-pattern` objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-capec-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/capec/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-mitre-cwe-id`

Takes the extracted ID and passes it to 

```shell
GET CTIBUTLER_BASE_URL/v1/cwe/objects/:cwe_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported `weakness` object is (`source_ref`) object connected to Report
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-cwe-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/cwe/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-mitre-atlas-id`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/atlas/objects/:atlas_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-atlas-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/atlas/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-disarm-id`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/disarm/objects/:disarm_id/
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported object(s) is (`source_ref`) object connected to Report with type `extracted-from`
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `ctibutler-mitre-atlas-name`

Takes the extracted name and passes it to;

```shell
GET CTIBUTLER_BASE_URL/v1/atlas/objects/?name=<VALUE>
```

### stix-mapping: `ctibutler-location`

Takes the extracted ID and passes it to

```shell
GET CTIBUTLER_BASE_URL/v1/location/objects/?alpha2_code=ID
```

`CTIBUTLER_APIKEY` in request passed if set.

All the objects returned are imported.

Relationship mode object generation behaviour:

* Standard relationship SRO: `location` `extracted-from` `report`
* AI mode relationship SROs: location is connected as `source_ref` or `target_ref` to 0 or more objects based on AI analysis

### stix-mapping: `vulmatch-cve-id`

Takes the extracted ID and passes it to 

```shell
GET VULMATCH_BASE_URL/v1/cve/objects/:cve_id/
```

`VULMATCH_APIKEY` in request passed if set.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported `vulnerability` object is (`source_ref`) object connected to Report
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

### stix-mapping: `vulmatch-cpe-id`

Takes the extracted ID and passes it to 

```shell
GET VULMATCH_BASE_URL/v1/cpe/objects/:cpe_id/
```

`VULMATCH_APIKEY` in request passed if set.

Relationship mode object generation behaviour:

* Standard relationship SRO: Imported `software` object is (`source_ref`) object connected to Report
* AI mode relationship SROs: all imported objects are connected as source or target object (depending on if extraction is source or target)

## Relationship objects

### Fixed SCO to Indicator relationships for extractions

In some extractions, SROs are created to link extractions (SCO linked to Indicator)

These relationships are modelled as follows;

```json
{
    "type": "relationship",
    "spec_version": "2.1",
    "id": "relationship--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT/CUSTOM IDENTITY ID>",
    "created": "<REPORT CREATED DATE>",
    "modified": "<REPORT CREATED DATE>",
    "relationship_type": "<DEFINED BY EXTRACTION>",
    "source_ref": "<SCO ID>",
    "target_ref": "indicator--<ID>",
    "description": "<SOURCE OBJECT NAME> is found in <REPORT NAME>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>"
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### Relationships generated in `standard` relationship mode

In standard mode, one SRO is created for each extraction back to the source report object created for the job. 

```json
{
    "type": "relationship",
    "spec_version": "2.1",
    "id": "relationship--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT/CUSTOM IDENTITY ID>",
    "created": "<REPORT CREATED DATE>",
    "modified": "<REPORT CREATED DATE>",
    "relationship_type": "extracted-from",
    "source_ref": "<SOURCE OBJECT ID AS DEFINED BY STIX EXTRACTION>",
    "target_ref": "report--<REPORT OBJECT ID CREATED FOR JOB>",
    "description": "<SOURCE OBJECT NAME> is found in <REPORT NAME>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>"
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

### Relationships generated in `ai` relationship mode

The LLM generates a JSON file describing how objects created from extractions are connected.

The Relationship SRO created for each relationship identified will be generated as follows;

```json
{
    "type": "relationship",
    "spec_version": "2.1",
    "id": "relationship--<GENERATED BY STIX2 LIBRARY>",
    "created_by_ref": "identity--<DEFAULT/CUSTOM IDENTITY ID>",
    "created": "<REPORT CREATED DATE>",
    "modified": "<REPORT CREATED DATE>",
    "relationship_type": "<OPENAI REPORTED RELATIONSHIP_TYPE>",
    "source_ref": "<SOURCE OBJECT ID AS DEFINED BY STIX CONVERSION TYPE>",
    "target_ref": "<TARGET OBJECT ID AS DEFINED BY STIX CONVERSION TYPE>",
    "description": "<SOURCE OBJECT NAME FOR SDO OR FOR SDO VALUE/HASHES/PATH/KEY/NUMBER/STRING/ADDRESS/HASH/SYMBOL/IBAN_NUMBER/DEFAULT TO ID IF NO VALUE> <relationship_type> <TARGET OBJECT SDO/SCO NAME/VALUE/ETC.>",
    "object_marking_refs": [
        "marking-definition--<TLP LEVEL SET>"
        "marking-definition--f92e15d9-6afc-5ae2-bb3e-85a1fd83a3b5"
    ],
    "external_references": [
        {
            "source_name": "txt2stix_report_id",
            "external_id": "<UUID OF REPORT OBJECT>"
        },
        {
            "source_name": "txt2stix_extraction_type",
            "external_id": "<EXTRACTION SLUG>_<EXTRACTION_VERSION>"
        }
    ]
}
```

Note, a source object can be linked to many target objects in the same report. Thus, a source object might create many SROs, representing links to target objects.

Similarly, in the case of MITRE ATT&CK an extraction might import one or more STIX objects. In this case, an SRO should be created for each imported object to the source/target object identified.

## Bundle (output)

The output of txt2stix is a STIX bundle file.

This bundle takes the format;

```json
{
    "type": "bundle",
    "id": "bundle--<SAME AS REPORT UUID PART>",
    "objects": [
        "<ALL STIX JSON DEFAULT AND EXTRACTED OBJECTS>"
    ]
}
```

The objects include all SROs generated for the input.

The filename of the bundle takes the format: `bundle--<ID>.json`