#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
# Copyright 2016-2025 European Commission (JRC);
# Licensed under the EUPL (the 'Licence');
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl
import time
import unittest
import ddt
import schedula as sh
from formulas.cell import Cell
from formulas.functions import Error
from formulas.functions.date import DEFAULT_DATE

DEFAULT_DATE[0] = 2019


def inp_ranges(*rng):
    return dict.fromkeys(rng, sh.EMPTY)


@ddt.ddt
class TestCell(unittest.TestCase):
    @ddt.idata([
        ('A1', '=MULTINOMIAL(0,0)', {}, "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=MULTINOMIAL(2,3,4)', {}, "<Ranges>(A1)=[[1260.0]]"),
        ('A1', '=SUMX2MY2({1,2},{3,4})', {}, "<Ranges>(A1)=[[-20.0]]"),
        ('A1', '=HYPGEOMDIST(0,,,)', {}, "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=VDB(2400,300,10*12,0,3.2)', {},
         "<Ranges>(A1)=[[125.61774074074076]]"),
        ('A1', '=VDB(2400,300,10*12,0,18)', {},
         "<Ranges>(A1)=[[626.5255161762734]]"),
        ('A1', '=VDB(2400,300,10*12,2.2,12.2)', {},
         "<Ranges>(A1)=[[357.82510255658264]]"),
        ('A1', '=VDB(2400,300,10*12,0,12.2)', {},
         "<Ranges>(A1)=[[444.89399144547156]]"),
        ('A1', '=VDB(2400,300,10*12,0,2.2)', {},
         "<Ranges>(A1)=[[87.0688888888889]]"),
        ('A1', '=VDB(2400,300,10*12,0,1.2)', {},
         "<Ranges>(A1)=[[47.86666666666667]]"),
        ('A1', '=VDB(2,0,10*12,1.2,18,1)', {}, "<Ranges>(A1)=[[0.28]]"),
        ('A1', '=VDB(2400,300,10*365,0,1)', {},
         "<Ranges>(A1)=[[1.3150684931506849]]"),
        ('A1', '=VDB(2400,300,10*12,0,1)', {}, "<Ranges>(A1)=[[40.0]]"),
        ('A1', '=VDB(2400,300,10,0,1)', {}, "<Ranges>(A1)=[[480.0]]"),
        ('A1', '=VDB(2400,300,10*12,6,18)', {},
         "<Ranges>(A1)=[[396.30605326475074]]"),
        ('A1', '=ODDLPRICE("2005-4-15","2009-7-1","2000-6-30",0.1,6.5%,1,4)',
         {}, "<Ranges>(A1)=[[23.549290501737403]]"),
        ('A1', '=ODDLPRICE("2005-4-15","2009-7-1","2000-6-13",0.1,6.5%,1,4)',
         {}, "<Ranges>(A1)=[[23.447810199522166]]"),
        ('A1', '=ODDLPRICE("15/4/2005","15/4/2009","15/4/2000",0.1,6.5%,1,4)',
         {}, "<Ranges>(A1)=[[22.22222222222223]]"),
        ('A1', '=ODDLPRICE("2005-4-15","2009-7-30","2000-6-13",0.1,6.5%,1,4)',
         {}, "<Ranges>(A1)=[[23.78356047854338]]"),
        ('A1', '=WORKDAY.INTL("2020-1-1","2024-1-1",#VALUE!)', {},
         "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=WORKDAY.INTL("2020-1-1","2024-1-1",22)', {},
         "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=WORKDAY.INTL("2020-1-1","2024-1-1","00")', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=WORKDAY("2020-1-1",22)', {}, "<Ranges>(A1)=[[43861]]"),
        ('A1', '=NETWORKDAYS("2020-1-1","2024-1-1",0.7)', {},
         "<Ranges>(A1)=[[1044]]"),
        ('A1', '=NETWORKDAYS.INTL(DATE(2006,1,1),DATE(2006,2,1),"0010001",'
               '{"2006/1/2","2006/1/16"})', {}, "<Ranges>(A1)=[[20]]"),
        ('A1', '=NETWORKDAYS.INTL(DATE(2006,1,1),DATE(2006,2,1),7,'
               '{"2006/1/2","2006/1/16"})', {}, "<Ranges>(A1)=[[22]]"),
        ('A1', '=NETWORKDAYS.INTL(DATE(2006,2,28),DATE(2006,1,31))',
         {}, "<Ranges>(A1)=[[-21]]"),
        ('A1', '=NETWORKDAYS.INTL(DATE(2006,1,1),DATE(2006,1,31))',
         {}, "<Ranges>(A1)=[[22]]"),
        ('A1', '=NETWORKDAYS.INTL(DATE(2006,1,1),DATE(2006,1,31),22)',
         {}, "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=DAYS360("2000-2-29","2002-2-28",1)', {},
         "<Ranges>(A1)=[[719]]"),
        ('A1', '=DAYS360("2000-2-29","2002-2-28")', {}, "<Ranges>(A1)=[[718]]"),
        ('A1', '=DAYS360("2000-2-29","2002-2-27")', {}, "<Ranges>(A1)=[[717]]"),
        ('A1', '=DAYS360("2000-2-28","2002-2-28")', {}, "<Ranges>(A1)=[[720]]"),
        ('A1', '=DAYS(-1,"2022-10-13")', {}, "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=DAYS("1900-1-1","2022-10-13")', {}, "<Ranges>(A1)=[[-44846]]"),
        ('A1', '=TBILLYIELD(DATE(2024,4,1),DATE(2025,3,4),0.3)', {},
         "<Ranges>(A1)=[[355.0148367952523]]"),
        ('A1', '=TBILLPRICE(DATE(2024,4,1),DATE(2025,3,4),0.3)', {},
         "<Ranges>(A1)=[[71.91666666666667]]"),
        ('A1', '=TBILLEQ(DATE(2024,4,1),DATE(2025,3,4),0.3)', {},
         "<Ranges>(A1)=[[0.3883686423017574]]"),
        ('A1', '=TBILLEQ(DATE(2022, 11, 15), DATE(2023, 3, 31), 5%)', {},
         "<Ranges>(A1)=[[0.05167044167610419]]"),
        ('A1:C1',
         '=AMORDEGRC(2200, "2012-05-11","2022-12-31",96,{0,1,2},0.19,0)', {},
         "<Ranges>(A1:C1)=[[2104 36 0]]"),
        ('A1', '=AMORDEGRC(2200, "2012-05-11","2022-12-31",96,1,0.5,0)', {},
         "<Ranges>(A1)=[[#NUM!]]"),
        ('A1:G1',
         '=AMORDEGRC(2200, "2022-05-11","2022-12-31",0,{0,1,2,3,4,5,6},0.19,0)',
         {}, "<Ranges>(A1:G1)=[[534 633 393 243 199 199 0]]"),
        ('A1:E1',
         '=AMORDEGRC(2200, "2022-05-11","2022-12-31",0,{0,1,2,3,4},0.29,0)', {},
         "<Ranges>(A1:E1)=[[611 691 449 449 0]]"),
        ('A1:C1',
         '=AMORDEGRC(2200, "2022-05-11","2022-12-31",2000,{0,1,2},0.09,0)', {},
         "<Ranges>(A1:C1)=[[200 450 0]]"),
        ('A1:M1', '=AMORDEGRC(2200, "2022-05-11","2022-12-31",96,{0,1,2,3,4,5,'
                  '6,7,8,9,10,11,12},0.09,0)', {},
         "<Ranges>(A1:M1)=[[316 424 329 255 197 153 119 92 71 55 95 0 0]]"),
        ('A1:M1', '=AMORDEGRC(2200, "2022-05-11","2022-12-31",0,{0,1,2,3,4,'
                  '5,6,7,8,9,10,11,12},0.09,0)', {},
         "<Ranges>(A1:M1)=[[316 424 329 255 197 153 119 92 71 55 95 95 0]]"),
        ('A1:F1', '=AMORDEGRC(100, "15/1/2005","15/4/2005",6,{0,1,2,3,4,5},'
                  '0.2,0)', {}, "<Ranges>(A1:F1)=[[10 36 22 16 16 0]]"),
        ('A1', '=YIELDMAT("1999-02-15","1999-04-13","1998-11-11",0.061,98.5,3)',
         {}, "<Ranges>(A1)=[[0.15688886539342684]]"),
        ('A1', '=YIELD("15/5/2011","31/7/2011",5.75%,1200,100,4,0)', {},
         "<Ranges>(A1)=[[-4.336510848300115]]"),
        ('A1', '=YIELD("15/4/2011","31/7/2011",5.75%,1200,100,4,0)', {},
         "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=YIELD("15/4/2005","31/7/2009",3,6.5%,100,4)', {},
         "<Ranges>(A1)=[[24.958782326908345]]"),
        ('A1:C1', '=AMORLINC(100, "2005-1-11","2005-4-15",7,{0,1,2},0.2,0)', {},
         "<Ranges>(A1:C1)=[[5.222222222222222 20.0 20.0]]"),
        ('A1:C1', '=AMORLINC(100, "2005-1-11","2009-4-15",7,{0,1,2},0.2,0)', {},
         "<Ranges>(A1:C1)=[[85.22222222222223 7.7777777777777715 0]]"),
        ('A1:D1', '=AMORLINC(100, "15/1/2005","15/4/2009",12,{0,1,2.1,3.3},'
                  '0.2,0)', {}, "<Ranges>(A1:D1)=[[85.0 3.0 0 0]]"),
        ('A1:D1',
         '=PRICEMAT("15/4/2005","31/7/2009","15/4/2000",5.75%,100,{0,2,3,4})',
         {}, "<Ranges>(A1:D1)=[[-28.393524135260716 -28.8120358587144 "
             "-28.40934137362836\n  -28.393331073227436]]"),
        ('A1:E1', '=PRICEDISC("15/4/2005","31/7/2009",5.75%,100,{0,1,2,3,4})',
         {}, "<Ranges>(A1:E1)=[[75.30694444444444 75.31215772179627 "
             "74.95555555555555 75.2986301369863\n  75.32291666666666]]"),
        ('A1:C1', '=PRICE("15/4/2005","31/7/2009",5.75%,6.50%,100,4,{0,1,4})',
         {}, "<Ranges>(A1:C1)=[[97.20919305618973 97.20891452090883 "
             "97.20919305618973]]"),
        ('A1', '=INDEX(B1:D2 (B1:C1,B2:C2), 1, 1, 2)',
         {'B1:C1': [[2, 3]], 'B2:C2': [[4, 5]]}, '<Ranges>(A1)=[[4]]'),
        # ('A1', '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",4,1,4,1,0)', {},
        # "<Ranges>(A1)=[[19.05909646]]"),
        # ('A1', '=ACCRINT("13/10/1999","15/4/2005","27/7/2009",4,1,4,1,0)', {},
        # "<Ranges>(A1)=[[18.39951691]]"),
        # ('A1:B1', '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",4,1,1,{2,3},1)',
        # {}, "<Ranges>(A1:B1)=[[17.05555556 16.98630137]]"),
        # ('A1:B1', '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",1,1,2,{2,3},1)',
        # {}, "<Ranges>(A1:B1)=[[4.2555555556 4.2452054795]]"),
        # ('A1:C1',
        # '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{1,2,3},1)', {},
        # "<Ranges>(A1:C1)=[[9.453663004 9.455555556 9.452739726]]"),
        # ('A1:B1', '=ACCRINT("15/4/2000","15/4/2005","15/4/2009",4,1,4,{2,3},1)',
        # {}, "<Ranges>(A1:B1)=[[36.0 35.986]]"),
        # ('A1', '=ACCRINT("15/4/2000","15/4/2005","15/4/2009",4,1,4,{1},0)', {},
        # "<Ranges>(A1)=[[17.23333333]]"),
        # ('A1', '=ACCRINT("13/10/1999","15/4/2005","27/7/2009",1,1,4,{1},0)', {},
        # "<Ranges>(A1)=[[4.599879227]]"),
        # ('A1', '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{1},0)', {},
        # "<Ranges>(A1)=[[4.764774115]]"),
        ('A1', '=ACCRINT("13/10/1999","15/4/2005","27/7/2009",1,-1,4,{2},0)',
         {}, "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=ACCRINT("13/10/1999","15/4/2005","27/7/2009",1,1,4,{2},0)', {},
         "<Ranges>(A1)=[[4.6]]"),
        ('A1', '=ACCRINT("13/10/1999","30/4/2005","27/7/2009",1,1,4,{2},0)', {},
         "<Ranges>(A1)=[[4.599999999999999]]"),
        ('A1', '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{2},0)', {},
         "<Ranges>(A1)=[[4.7666666666666675]]"),
        ('A1:B1', '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{0,4},0)',
         {}, "<Ranges>(A1:B1)=[[4.705555555555556 4.705555555555556]]"),
        ('A1:B1', '=ACCRINT("14/2/2000","15/4/2005","27/7/2009",1,1,4,{0,4},0)',
         {}, "<Ranges>(A1:B1)=[[4.7027777777777775 4.7027777777777775]]"),
        ('A1', '=ACCRINT("2024-01-15","2024-07-22","2024-05-31",1,1,4,1,1)', {},
         "<Ranges>(A1)=[[0.37616459627329196]]"),
        ('A1', '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",1,1,1,1,1)', {},
         "<Ranges>(A1)=[[4.2465753424657535]]"),
        ('A1', '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",1,1,2,1,1)', {},
         "<Ranges>(A1)=[[4.247252747252747]]"),
        ('A1:C1',
         '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",4,1,1,{0,1,4},1)', {},
         "<Ranges>(A1:C1)=[[17.0 16.986301369863014 17.0]]"),
        ('A1:C1',
         '=ACCRINT("15/1/2005","15/4/2005","15/4/2009",1,1,2,{0,1,4},1)', {},
         "<Ranges>(A1:C1)=[[4.25 4.247252747252747 4.25]]"),
        ('A1:B1', '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{0,4},1)',
         {}, "<Ranges>(A1:B1)=[[9.455555555555556 9.455555555555556]]"),
        ('A1:D1',
         '=ACCRINT("13/2/2000","15/4/2005","27/7/2009",1,1,4,{0,2,3,4},0)', {},
         "<Ranges>(A1:D1)=[[4.705555555555556 4.7666666666666675 "
         "4.701369863013699\n  4.705555555555556]]"),
        ('A1', '=FVSCHEDULE(TRUE,{#N/A,2})', {}, "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=FVSCHEDULE("1/1/1987",{1,2})', {},
         "<Ranges>(A1)=[[190668.0]]"),
        ('A1', '=FVSCHEDULE(1,{7,6})', {}, "<Ranges>(A1)=[[56.0]]"),
        ('A1', '=DOLLARFR(1.7816666666666663,3.2)', {},
         "<Ranges>(A1)=[[1.2345]]"),
        ('A1', '=DOLLARDE(1.2345,3.2)', {},
         "<Ranges>(A1)=[[1.7816666666666663]]"),
        ('A1', '=DISC(43107,54058,97.975,100,1)', {},
         "<Ranges>(A1)=[[0.0006754155608119472]]"),
        ('A1', '=DB(10000,1000,2,1,4)', {}, "<Ranges>(A1)=[[2280.0]]"),
        ('A1', '=DB(10000,1000,2,2,4)', {},
         "<Ranges>(A1)=[[5280.4800000000005]]"),
        ('A1', '=DB(10000,1000,2,3,4)', {},
         "<Ranges>(A1)=[[1112.4211199999997]]"),
        ('A1', '=RECEIVED(32000,43232,"26/8/1989",0.01,1)', {},
         "<Ranges>(A1)=[[47287.694223698265]]"),
        ('A1', '=INTRATE(32000,43232,"26/8/1989",1400,1)', {},
         "<Ranges>(A1)=[[-0.031128414555493433]]"),
        ('A1', '=EFFECT(3.2,6)', {}, "<Ranges>(A1)=[[11.996292038408773]]"),
        ('A1', '=MIRR({-7,2,-1,4,-3;7,2,-1,4,4},4,3)', {},
         "<Ranges>(A1)=[[2.0249544173762937]]"),
        ('A1', '=DURATION("31/08/2030","31/12/2050",TRUE,4,0,1)', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=PDURATION(12,2,1)', {},
         "<Ranges>(A1)=[[-0.27023815442731974]]"),
        ('A1', '=MDURATION(32000,43232,9%,8%,1)', {},
         "<Ranges>(A1)=[[10.94163076865601]]"),
        ('A1', '=DURATION(32000,43232,9%,8%,1)', {},
         "<Ranges>(A1)=[[11.816961230148491]]"),
        ('A1', '=ACCRINTM("13/2/2000","27/7/2009",TRUE,1000,1)', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1:E1', '=ACCRINTM("13/2/2000","27/7/2009",0.3,1000,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[2836.666666666667 2834.930194360799 "
         "2876.6666666666665\n  2837.2602739726026 2836.666666666667]]"),
        ('A1:E1', '=COUPDAYBS("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[75 74 74 74 75]]"),
        ('A1:E1', '=COUPDAYSNC("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[15 15 15 15 15]]"),
        ('A1:E1', '=COUPDAYS("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[90.0 89 90.0 91.25 90.0]]"),
        ('A1:E1', '=COUPPCD("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[38383 38383 38383 38383 38383]]"),
        ('A1:E1', '=COUPNUM("15/4/2000","15/4/2009",2,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[18 18 18 18 18]]"),
        ('A1:E1', '=COUPNUM("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[18 18 18 18 18]]"),
        ('A1:E1', '=COUPNCD("15/4/2005","31/7/2009",4,{0,1,2,3,4})', {},
         "<Ranges>(A1:E1)=[[38472 38472 38472 38472 38472]]"),
        ('A1',
         '=FORECAST.ETS.STAT({3,3,1,2,3,2,1,3,2,1},{3,7,5,6,7,8,1,3,4,9},4,4,1)',
         {}, "<Ranges>(A1)=[[0]]"),
        ('A1:B2', '=MAP({1,2;3,3},{3,4;4,5},LAMBDA(a,b,a+b+3))', {},
         "<Ranges>(A1:B2)=[[ 7.  9.]\n [10. 11.]]"),
        ('A1:C1', '=SCAN({1,2}, {1,2,3}, LAMBDA(a,b,SUM(a)+b^2))', {},
         "<Ranges>(A1:C1)=[[ 4.  8. 17.]]"),
        ('A1', '=REDUCE(1, {1,2,3}, LAMBDA(a,b,a+b^2))', {},
         "<Ranges>(A1)=[[15.0]]"),
        ('A1', '=MAKEARRAY(TRUE,TRUE,LAMBDA(x,y,x+1+y))', {},
         "<Ranges>(A1)=[[3.0]]"),
        ('A1:B1', '=MAKEARRAY(2,{2,1},LAMBDA(x,y,x+1+y))', {},
         "<Ranges>(A1:B1)=[[3.0 3.0]]"),
        ('A1:B2', '=MAKEARRAY(2,2,LAMBDA(x,y,x+1+y))', {},
         "<Ranges>(A1:B2)=[[3.0 4.0]\n [4.0 5.0]]"),
        ('D1:E1', '=BYCOL(A1:B2,_XLETA.SUM)',
         {'A1:B2': [[1, 3], [Error.errors['#VALUE!'], 3]]},
         "<Ranges>(D1:E1)=[[#VALUE! 6.0]]"),
        ('A1:B1', '=BYCOL({1,2;3,4},_XLETA.SUM)', {},
         "<Ranges>(A1:B1)=[[4.0 6.0]]"),
        ('D1', '=ISFORMULA(A1:A2)', {'A1:A2': [[3, 1]]},
         "<Ranges>(D1)=[[False]]"),
        ('D1', '=N(A1:A2)', {'A1:A2': [[3, 1]]}, "<Ranges>(D1)=[[3]]"),
        ('D1', '=TYPE("1/1/1900")', {}, "<Ranges>(D1)=[[2]]"),
        ('D1', '=N("1/1/1900")', {}, "<Ranges>(D1)=[[0]]"),
        ('D1', '=ERROR.TYPE(#VALUE!)', {}, "<Ranges>(D1)=[[3]]"),
        ('D1', '=ISREF(1)', {}, "<Ranges>(D1)=[[False]]"),
        ('D1', '=ISREF(A1)', {'A1': [[sh.EMPTY]]}, "<Ranges>(D1)=[[True]]"),
        ('D1', '=TRIMRANGE(A1:C4,3,3)', {'A1:C4': [
            [sh.EMPTY, sh.EMPTY, sh.EMPTY],
            [sh.EMPTY, sh.EMPTY, sh.EMPTY],
            [sh.EMPTY, sh.EMPTY, sh.EMPTY],
            [sh.EMPTY, sh.EMPTY, sh.EMPTY]
        ]},
         "<Ranges>(D1)=[[#REF!]]"),
        ('D1', '=TRIMRANGE(A1:C4,3,3)', {'A1:C4': [
            [sh.EMPTY, sh.EMPTY, sh.EMPTY],
            [sh.EMPTY, 0, sh.EMPTY],
            [sh.EMPTY, sh.EMPTY, sh.EMPTY],
            [sh.EMPTY, sh.EMPTY, sh.EMPTY]
        ]},
         "<Ranges>(D1)=[[0]]"),
        ('A1:C1', '=TAKE({1,2,3;4,5,6;7,8,9},1)', {},
         "<Ranges>(A1:C1)=[[1 2 3]]"),
        ('A1:E2', '=WRAPCOLS({1,2,3,4,5,6,7,8,9},2,6)', {},
         "<Ranges>(A1:E2)=[[1 3 5 7 9]\n "
         "[2 4 6 8 6]]"),
        ('A1:F2', '=HSTACK({1},{1;2},{1,2},{1,2;3,4})', {},
         "<Ranges>(A1:F2)=[[1 1 1 2 1 2]\n "
         "[#N/A 2 #N/A #N/A 3 4]]"),
        ('A1:B3', '=GROUPBY({"r1";2;2;3},{"v1";-1;5;3},_XLETA.AVERAGE)', {},
         "<Ranges>(A1:B3)=[[2 2.0]\n "
         "[3 3.0]\n "
         "['Total' 2.3333333333333335]]"),
        ('A1:B4', '=GROUPBY({"r1";2;2;3},{"v1";1;5;3},_XLETA.SUM,3)', {},
         "<Ranges>(A1:B4)=[['r1' 'v1']\n "
         "[2 6.0]\n "
         "[3 3.0]\n "
         "['Total' 9.0]]"),
        ('A1:D4',
         '=PIVOTBY(C69:C87,F69:F87,I69:I87,_XLETA.SUM,1,1,2,1,2,K69:K87)',
         {'C69:C87': [["row-2"],
                      ["r2"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r3"],
                      ["r3"]],
          'F69:F87': [["col-1"],
                      ["b"],
                      ["b"],
                      ["b"],
                      ["b"],
                      ["b"],
                      ["b"],
                      ["a"],
                      ["a"],
                      ["a"],
                      ["a"],
                      ["b"],
                      ["a"],
                      ["b"],
                      ["a"],
                      ["a"],
                      ["a"],
                      ["b"],
                      ["b"]],
          'I69:I87': [["val1"],
                      [1],
                      [6],
                      [-9],
                      [-1],
                      [Error.errors["#VALUE!"]],
                      [3],
                      [7],
                      [-4],
                      [-7],
                      [9],
                      [7],
                      [5],
                      [9],
                      [8],
                      [8],
                      [-6],
                      [1],
                      [-2]],
          'K69:K87': [["filter"],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [0],
                      [0],
                      [0], ]},
         "<Ranges>(A1:D4)=[['' 'b' 'a' 'Total']\n "
         "['r3' 9.0 15.0 24.0]\n "
         "['r2' 7.0 22.0 29.0]\n "
         "['Total' 16.0 37.0 53.0]]"),
        ('A1:F5',
         '=PIVOTBY(C69:C87,F69:G87,I69:I87,_XLETA.SUM,1,1,2,1,3,K69:K87)',
         {'C69:C87': [["row-2"],
                      ["r2"],
                      ["r2"],
                      ["r3"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r1"]],
          'F69:G87': [["col-1", "col-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-1"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-2"],
                      ["c1", "c1-2"],
                      ["c2", "c2-2"],
                      ["c1", "c1-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"]],
          'I69:I87': [["val1"],
                      [1],
                      [6],
                      [-9],
                      [-1],
                      [Error.errors["#VALUE!"]],
                      [3],
                      [-7],
                      [-4],
                      [-7],
                      [9],
                      [7],
                      [5],
                      [9],
                      [8],
                      [-8],
                      [-6],
                      [1],
                      [-2]],
          'K69:K87': [["filter"],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [0],
                      [0],
                      [0], ]},
         "<Ranges>(A1:F5)=[['' 'c1' 'c1' 'c2' 'c2' 'Total']\n "
         "['' 'c1-2' 'c1-1' 'c2-1' 'c2-2' '']\n "
         "['r1' '' 9.0 -15.0 '' -6.0]\n "
         "['r2' 7.0 '' 8.0 14.0 29.0]\n "
         "['Total' 7.0 9.0 -7.0 14.0 23.0]]"),
        ('A1:F5',
         '=PIVOTBY(C69:C87,F69:G87,I69:I87,_XLETA.SUM,1,1,1,1,{1,-2},K69:K87)',
         {'C69:C87': [["row-2"],
                      ["r2"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r2"],
                      ["r1"],
                      ["r1"]],
          'F69:G87': [["col-1", "col-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-1"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-2"],
                      ["c1", "c1-2"],
                      ["c2", "c2-2"],
                      ["c1", "c1-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-1"],
                      ["c2", "c2-2"],
                      ["c1", "c1-2"],
                      ["c1", "c1-2"]],
          'I69:I87': [["val1"],
                      [1],
                      [6],
                      [-9],
                      [-1],
                      [Error.errors["#VALUE!"]],
                      [3],
                      [-7],
                      [-4],
                      [-7],
                      [9],
                      [7],
                      [5],
                      [9],
                      [8],
                      [-8],
                      [-6],
                      [1],
                      [-2]],
          'K69:K87': [["filter"],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [0],
                      [0],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [1],
                      [0],
                      [0],
                      [0], ]},
         "<Ranges>(A1:F5)=[['' 'c1' 'c1' 'c2' 'c2' 'Total']\n "
         "['' 'c1-2' 'c1-1' 'c2-2' 'c2-1' '']\n "
         "['r1' '' 9.0 '' -15.0 -6.0]\n "
         "['r2' 7.0 '' 14.0 8.0 29.0]\n "
         "['Total' 7.0 9.0 14.0 -7.0 23.0]]"),
        ('A1:H10',
         '=PIVOTBY({"r1","r2";2,1;2,2;3,1},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.PERCENTOF,3,2,1,0,1,{"f";1;1;1},3)',
         {},
         "<Ranges>(A1:H10)=[['' '' 'c1, c2' '' '' '' '' '']\n "
         "['' '' 1 1 2 2 2 2]\n "
         "['' '' 1 1 1 1 2 2]\n "
         "['r1' 'r2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2']\n "
         "[2 1 1.0 1.0 '' '' '' '']\n "
         "[2 2 '' '' 1.0 1.0 '' '']\n "
         "[2 '' 1.0 1.0 1.0 1.0 '' '']\n "
         "[3 1 '' '' '' '' 1.0 1.0]\n "
         "[3 '' '' '' '' '' 1.0 1.0]\n "
         "['Grand Total' '' 1.0 1.0 0.4 0.7142857142857143 0.6 0.2857142857142857]]"),
        ('A1:D4', '=PIVOTBY({2;2;3},{1;2;2},{1;2;3},_XLETA.IF)', {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 #VALUE! #VALUE! #VALUE!]\n "
         "[3 '' #VALUE! #VALUE!]\n ['Total' #VALUE! #VALUE! #VALUE!]]"),
        ('A1:D4',
         '=PIVOTBY({"r2";"r2";"r3";"r2";"r2";"r2";"r3";"r2";"r3"},{"c1";"c1";"c2";"c2";"c1";"c2";"c1";"c2";"c2"},{1;-1;-7;-1;7;5;9;8;-8},_XLETA.PERCENTOF,0,1,1,1,1,{1;1;1;1;1;1;1;1;1},3)',
         {},
         "<Ranges>(A1:D4)=[['' 'c1' 'c2' 'Total']\n "
         "['r2' 0.3684210526315789 0.631578947368421 1.0]\n "
         "['r3' -1.5 2.5 1.0]\n "
         "['Total' 1.2307692307692308 -0.23076923076923078 1.0]]"),
        ('A1:H10',
         '=PIVOTBY({"r1","r2";2,1;2,2;3,1},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.PERCENTOF,3,2,1,0,1,{"f";1;1;1},1)',
         {},
         "<Ranges>(A1:H10)=[['' '' 'c1, c2' '' '' '' '' '']\n "
         "['' '' 1 1 2 2 2 2]\n "
         "['' '' 1 1 1 1 2 2]\n "
         "['r1' 'r2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2']\n "
         "[2 1 1.0 1.0 '' '' '' '']\n "
         "[2 2 '' '' 1.0 1.0 '' '']\n "
         "[2 '' 0.3333333333333333 0.375 0.6666666666666666 0.625 '' '']\n "
         "[3 1 '' '' '' '' 1.0 1.0]\n "
         "[3 '' '' '' '' '' 1.0 1.0]\n "
         "['Grand Total' '' 0.16666666666666666 0.3 0.3333333333333333 0.5 0.5 0.2]]"),
        ('A1:H10',
         '=PIVOTBY({"r1","r2";2,1;2,2;3,1},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.PERCENTOF,3,2,1,0,1,{"f";1;1;1},0)',
         {},
         "<Ranges>(A1:H10)=[['' '' 'c1, c2' '' '' '' '' '']\n "
         "['' '' 1 1 2 2 2 2]\n "
         "['' '' 1 1 1 1 2 2]\n "
         "['r1' 'r2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2']\n "
         "[2 1 1.0 1.0 '' '' '' '']\n "
         "[2 2 '' '' 1.0 1.0 '' '']\n "
         "[2 '' 1.0 1.0 1.0 1.0 '' '']\n "
         "[3 1 '' '' '' '' 1.0 1.0]\n "
         "[3 '' '' '' '' '' 1.0 1.0]\n "
         "['Grand Total' '' 1.0 1.0 1.0 1.0 1.0 1.0]]"),
        ('A1:H10',
         '=PIVOTBY({"r1","r2";2,1;2,2;3,1},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.PERCENTOF,3,2,1,0,1,{"f";1;1;1},2)',
         {},
         "<Ranges>(A1:H10)=[['' '' 'c1, c2' '' '' '' '' '']\n "
         "['' '' 1 1 2 2 2 2]\n "
         "['' '' 1 1 1 1 2 2]\n "
         "['r1' 'r2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2']\n "
         "[2 1 0.16666666666666666 0.3 '' '' '' '']\n "
         "[2 2 '' '' 0.3333333333333333 0.5 '' '']\n "
         "[2 '' 0.16666666666666666 0.3 0.3333333333333333 0.5 '' '']\n "
         "[3 1 '' '' '' '' 0.5 0.2]\n "
         "[3 '' '' '' '' '' 0.5 0.2]\n "
         "['Grand Total' '' 0.16666666666666666 0.3 0.3333333333333333 0.5 0.5 0.2]]"),
        ('A1:O10',
         '=PIVOTBY(C69:E87,F69:H87,I69:J87,_XLETA.SUM,1,1,1,0,1,K69:L87)',
         {'C69:E87': [["row-2", "row-1", "row-3"],
                      ["r2", "r2-1", "r2-1-1"],
                      ["r2", "r2-2", "r2-2-2"],
                      ["r1", "r1-1", "r1-1-2"],
                      ["r2", "r2-2", "r2-2-1"],
                      ["r1", "r1-2", "r1-2-2"],
                      ["r2", "r2-1", "r2-1-2"],
                      ["r1", "r1-2", "r1-2-1"],
                      ["r2", "r2-2", "r2-2-2"],
                      ["r2", "r2-1", "r2-1-2"],
                      ["r2", "r2-2", "r2-2-1"],
                      ["r2", "r2-2", "r2-2-2"],
                      ["r2", "r2-2", "r2-2-1"],
                      ["r1", "r1-1", "r1-1-2"],
                      ["r2", "r2-1", "r2-1-2"],
                      ["r1", "r1-2", "r1-2-2"],
                      ["r2", "r2-2", "r2-2-2"],
                      ["r1", "r1-2", "r1-2-2"],
                      ["r1", "r1-1", "r1-1-2"]],
          'F69:H87': [["col-1", "col-2", "col-3"],
                      ["c1", "c1-2", "c1-2-1"],
                      ["c1", "c1-1", "c1-1-2"],
                      ["c1", "c1-2", "c1-2-2"],
                      ["c1", "c1-2", "c1-2-2"],
                      ["c1", "c1-2", "c1-2-1"],
                      ["c1", "c1-2", "c1-2-1"],
                      ["c2", "c2-1", "c2-1-2"],
                      ["c2", "c2-1", "c2-1-2"],
                      ["c2", "c2-1", "c2-1-2"],
                      ["c2", "c2-2", "c2-2-2"],
                      ["c1", "c1-2", "c1-2-1"],
                      ["c2", "c2-2", "c2-2-2"],
                      ["c1", "c1-1", "c1-1-1"],
                      ["c2", "c2-1", "c2-1-1"],
                      ["c2", "c2-1", "c2-1-2"],
                      ["c2", "c2-2", "c2-2-1"],
                      ["c1", "c1-2", "c1-2-1"],
                      ["c1", "c1-2", "c1-2-2"]],
          'I69:J87': [["val1", "val2"],
                      [1, 5],
                      [6, 9],
                      [-9, 8],
                      [-1, 7],
                      [Error.errors["#VALUE!"], 3],
                      [3, -7],
                      [-7, -4],
                      [-4, 1],
                      [-7, 6],
                      [-1, -9],
                      [7, -1],
                      [5, 7],
                      [9, 3],
                      [8, -7],
                      [-8, -4],
                      [-6, -7],
                      [1, -1],
                      [-2, -2], ],
          'K69:L87': [["filter", ""],
                      [1, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY],
                      [1, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY],
                      [1, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY],
                      [1, sh.EMPTY],
                      [1, sh.EMPTY],
                      [1, sh.EMPTY],
                      [1, sh.EMPTY],
                      [1, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY],
                      [0, sh.EMPTY], ]},
         "<Ranges>(A1:O10)=[['' '' '' 'c1' 'c1' 'c1' 'c1' 'c1' 'c1' 'c2' 'c2' 'c2' 'c2' 'c2' 'c2']\n "
         "['' '' '' 'c1-1' 'c1-1' 'c1-2' 'c1-2' 'c1-2' 'c1-2' 'c2-1' 'c2-1' 'c2-1'\n  'c2-1' 'c2-2' 'c2-2']\n "
         "['' '' '' 'c1-1-1' 'c1-1-1' 'c1-2-1' 'c1-2-1' 'c1-2-2' 'c1-2-2' 'c2-1-1'\n  'c2-1-1' 'c2-1-2' 'c2-1-2' 'c2-2-2' 'c2-2-2']\n "
         "['r1' 'r1-1' 'r1-1-2' 9.0 3.0 '' '' '' '' '' '' '' '' '' '']\n "
         "['r1' 'r1-2' 'r1-2-1' '' '' '' '' '' '' '' '' -7.0 -4.0 '' '']\n "
         "['r2' 'r2-1' 'r2-1-1' '' '' 1.0 5.0 '' '' '' '' '' '' '' '']\n "
         "['r2' 'r2-1' 'r2-1-2' '' '' '' '' '' '' 8.0 -7.0 '' '' '' '']\n "
         "['r2' 'r2-2' 'r2-2-1' '' '' '' '' -1.0 7.0 '' '' '' '' 4.0 -2.0]\n "
         "['r2' 'r2-2' 'r2-2-2' '' '' 7.0 -1.0 '' '' '' '' '' '' '' '']\n "
         "['Total' '' '' 9.0 3.0 8.0 4.0 -1.0 7.0 8.0 -7.0 -7.0 -4.0 4.0 -2.0]]"),
        ('A1:N10',
         '=PIVOTBY({"r1","r2";2,1;2,2;3,1},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.SUM)',
         {},
         "<Ranges>(A1:N10)=[['' '' 'c1, c2' '' '' '' '' '' '' '' '' '' '' '']\n "
         "['' '' 1 1 1 1 2 2 2 2 2 2 'Grand Total' 'Grand Total']\n "
         "['' '' 1 1 '' '' 1 1 2 2 '' '' '' '']\n "
         "['r1' 'r2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2' 'v1' 'v2']\n "
         "[2 1 1.0 3.0 1.0 3.0 '' '' '' '' '' '' 1.0 3.0]\n "
         "[2 2 '' '' '' '' 2.0 5.0 '' '' 2.0 5.0 2.0 5.0]\n "
         "[2 '' 1.0 3.0 1.0 3.0 2.0 5.0 '' '' 2.0 5.0 3.0 8.0]\n "
         "[3 1 '' '' '' '' '' '' 3.0 2.0 3.0 2.0 3.0 2.0]\n "
         "[3 '' '' '' '' '' '' '' 3.0 2.0 3.0 2.0 3.0 2.0]\n "
         "['Grand Total' '' 1.0 3.0 1.0 3.0 2.0 5.0 3.0 2.0 5.0 7.0 6.0 10.0]]"),
        ('A1:I5',
         '=PIVOTBY({"r1";2;2;3},{"c1","c2";1,1;2,1;2,2},{"v1","v2";1,3;2,5;3,2},_XLETA.SUM,1,1,1,1,1)',
         {},
         "<Ranges>(A1:I5)=[['' 1 1 2 2 2 2 'Total' 'Total']\n "
         "['' 1 1 1 1 2 2 '' '']\n "
         "[2 1.0 3.0 2.0 5.0 '' '' 3.0 8.0]\n "
         "[3 '' '' '' '' 3.0 2.0 3.0 2.0]\n "
         "['Total' 1.0 3.0 2.0 5.0 3.0 2.0 6.0 10.0]]"),
        ('A1:D4', '=PIVOTBY({2;2;3},{1;2;2},{1;2;3},LAMBDA(x,SUM(x,1)))', {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 2.0 3.0 4.0]\n "
         "[3 '' 4.0 4.0]\n ['Total' 2.0 6.0 7.0]]"),
        ('A1:D4',
         '=PIVOTBY({"r1";2;2;3},{"c1";1;1;2},{"v1";1;5;3},_XLETA.SUM,1,1,1,1,1,{"a";1;0;2})',
         {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 1.0 '' 1.0]\n "
         "[3 '' 3.0 3.0]\n ['Total' 1.0 3.0 4.0]]"),
        ('A1:D4',
         '=PIVOTBY({"r1";2;2;3},{"c1";1;1;2},{"v1";1;5;3},_XLETA.SUM)',
         {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 6.0 '' 6.0]\n "
         "[3 '' 3.0 3.0]\n ['Total' 6.0 3.0 9.0]]"),
        ('A1:D4',
         '=PIVOTBY({"r1";2;2;3},{"c1";1;1;2},{"v1";1;5;3},_XLETA.SUM)',
         {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 6.0 '' 6.0]\n "
         "[3 '' 3.0 3.0]\n ['Total' 6.0 3.0 9.0]]"),
        ('A1:D4', '=PIVOTBY({2;2;3},{1;2;2},{1;2;3},_XLETA.SUM)', {},
         "<Ranges>(A1:D4)=[['' 1 2 'Total']\n [2 1.0 2.0 3.0]\n "
         "[3 '' 3.0 3.0]\n ['Total' 1.0 5.0 6.0]]"),
        ('A1', '=PERCENTOF({1,2,3},{1,TRUE,"2","3.0"})', {},
         "<Ranges>(A1)=[[6.0]]"),
        ('A1', '=LET(z,3,y,A4,x,A2/2,SUM(y)+SUM((x+1)*x,(y+x+A2+A3)))',
         {'A2': [[2]], 'A3': [[3]], 'A4': [[2]]},
         "<Ranges>(A1)=[[12.0]]"),
        ('A1', '=LET(X,1,Y,E9,Z,L10,232+X+Y)',
         {'L10': [[2]], 'E9': [[2.9]]},
         "<Ranges>(A1)=[[235.9]]"),
        ('A1', '=LAMBDA(z,y,x,SUM(y)+SUM((x+1)*x,(y+x+A2+A3)))(3,2,1)',
         {'A2': [[2]], 'A3': [[3]]},
         "<Ranges>(A1)=[[12.0]]"),
        ('A1', '=LAMBDA(x,y,z,SUM(y)+SUM(x+1,(y+x+A2+A3)))(1,2)',
         {'A2': [[2]], 'A3': [[3]]},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=LAMBDA(x,y,z,SUM(y)+SUM(x+1,(y+x+A2+A3)))',
         {'A2': [[2]], 'A3': [[3]]},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1:C2', '=SORTBY({1,2,3;3,2,1},{-2,0,0},1)', {},
         "<Ranges>(A1:C2)=[[1 2 3]\n [3 2 1]]"),
        ('A1:C2', '=SORTBY({1,2,3;4,5,6},{3,2,2},1,{5,1,4},-1)', {},
         "<Ranges>(A1:C2)=[[3 2 1]\n [6 5 4]]"),
        ('A1:B2', '=SORT({2,1;1,2})', {}, "<Ranges>(A1:B2)=[[1 2]\n [2 1]]"),
        ('A1:O1', '=SORT(A2:O2,1,1,1)', {'A2:O2': [[
            Error.errors['#VALUE!'], 0, 3, 2, True, False, 2, "b",
            Error.errors['#DIV/0!'], Error.errors['#N/A'],
            Error.errors['#NAME?'], Error.errors['#REF!'],
            Error.errors['#NULL!'], Error.errors['#NUM!'], sh.EMPTY
        ]]},
         "<Ranges>(A1:O1)=[[0 2 2 3 'b' False True #NULL! #DIV/0! #VALUE! #REF! #NAME? #NUM! #N/A 0]]"),
        ('A1:B2', '=SORT({2,1;3,4},1,-1,1)', {},
         "<Ranges>(A1:B2)=[[2 1]\n [3 4]]"),
        ('A1:D1', '=UNIQUE({1,2;1,2},B2:E2)', {
            'B2:E2': [[1, Error.errors['#VALUE!'], sh.EMPTY, "b"]]
        }, "<Ranges>(A1:D1)=[[1 #VALUE! 0 #VALUE!]]"),
        ('A1:B2', '=UNIQUE({2,1,2,1},{1,1;2,2})', {},
         "<Ranges>(A1:B2)=[[2 0]\n [0 0]]"),
        ('A1', '=UNIQUE({2,1,2,1},1,1)', {}, "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=UNIQUE({2,1,2},1,1)', {}, "<Ranges>(A1)=[[1]]"),
        ('A1:A3', '=TOCOL(B1:C2,2.9)', {
            'B1:C2': [[1, 2], [Error.errors['#VALUE!'], sh.EMPTY]]
        }, "<Ranges>(A1:A3)=[[1]\n [2]\n [0]]"),
        ('A1:A3', '=TOCOL({1,2,3},{1,2,3})', {},
         "<Ranges>(A1:A3)=[[1]\n [1]\n [1]]"),
        ('A1:A3', '=TOCOL({1,2,3})', {}, "<Ranges>(A1:A3)=[[1]\n [2]\n [3]]"),
        ('A1', '=ROWS((D73:G73,I72:J73,K73:N73))', {},
         "<Ranges>(A1)=[[#REF!]]"),
        ('A1:B1', '=CHOOSE({1,2},{1,2},{3,4})', {}, "<Ranges>(A1:B1)=[[1 4]]"),
        ('A1', '=CHOOSE(-1,"Wide",115,"world",8)', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=CHOOSE(3,"Wide",115,"world",8)', {},
         "<Ranges>(A1)=[['world']]"),
        ('A1', '=AREAS(B2:D4)', {'B2:D4': None}, "<Ranges>(A1)=[[1]]"),
        ('A1', '=AREAS(((F4:G4,G4:I4,F6:G6,G6:I6) G3:G7))', {},
         "<Ranges>(A1)=[[4]]"),
        ('A1', '=AREAS((B2:D4,E5,F6:I9))', {}, "<Ranges>(A1)=[[3]]"),
        ('A1', '=AREAS(B2:D4 B2)', {}, "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH(1,{4.1,3,2.1,2,1},1,2)', {}, "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH(-2,{"",4.1,"a",1.1},1,2)', {}, "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH(4.1,{"","a",1.1,4.1},0,2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH(4.1,{"",4.1,"a",1.1},1,2)', {}, "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(1.1,{-1.1,2.1,3.1,4.1},1)', {}, "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(4.1,{1.1,4.1,"","a"},0,2)', {}, "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(4.1,{1.1,4.1,"","a"},0,-2)', {}, "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(4.1,{"","a",1.1,4.1},0,-2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH(-2,{FALSE,2.1,TRUE,4.1},-1)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH("ß?",{1,"ssa",2.1,"c bau c",4.1},2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH("ß",{1,"Ss",2.1,"c bau c",4.1},1,2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH("ß",{1,"Ss",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(A2,{-1.1,2.1,3.1,4.1},1)',
         {'A2': [[sh.EMPTY]]},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=MATCH(1.1,{-1.1,2.1,3.1,4.1},1)', {}, "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH("ss",{1,"SS","c bau c",4.1},0,2)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=XLOOKUP(-2,{-1.1,2.1,3.1,4.1},{1,2,3,4}, "pippo")', {},
         "<Ranges>(A1)=[['pippo']]"),
        ('A1', '=XMATCH(0,{1,2,3,4,5},-1,2)', {}, "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH(3.5,{9,4.1,3,2.1,2},-1,-2)', {}, "<Ranges>(A1)=[[3]]"),
        ('A1', '=XMATCH(5,{9,4.1,3,2.1,2},-1,-2)', {}, "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(FALSE,{"a",3,FALSE,4.1,3,2.1,2,FALSE,"b"},-1,-2)', {},
         "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH(FALSE,{"a",4.1,3,2.1,2,TRUE,"b"},-1,2)', {},
         "<Ranges>(A1)=[[5]]"),
        ('A1', '=XMATCH(FALSE,{"a",4.1,3,2.1,2,TRUE,"b"},-1,-2)', {},
         "<Ranges>(A1)=[[1]]"),
        ('A1', '=XMATCH(FALSE,{"a",FALSE,4.1,3,2.1,2,FALSE,"b"},1,2)', {},
         "<Ranges>(A1)=[[7]]"),
        ('A1', '=XMATCH(FALSE,{"a",FALSE,4.1,3,2.1,2,FALSE,"b"},1,-2)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH(FALSE,{"a",4.1,3,2.1,2,TRUE,"b"},1,2)', {},
         "<Ranges>(A1)=[[6]]"),
        ('A1', '=XMATCH(FALSE,{"a",4.1,3,2.1,2,TRUE,"b"},1,-2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH(TRUE,{"a",4.1,3,2.1,2,TRUE,"b"},1,2)', {},
         "<Ranges>(A1)=[[6]]"),
        ('A1', '=XMATCH(TRUE,{"a",4.1,3,2.1,2,TRUE,"b"},1,-2)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH(3,{"a",4.1,3,2.1,2,1,"b"},1,-2)', {},
         "<Ranges>(A1)=[[3]]"),
        ('A1', '=XMATCH(1,{1,7,2.1,3,4.1},1,-2)', {}, "<Ranges>(A1)=[[5]]"),
        ('A1', '=XMATCH("A?",{1,"aa",2.1,"c bau c",4.1},2)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH("A?",{1,"aa",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=XMATCH("A?",{1,"a?",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=XMATCH("ß?",{1,"ssa",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=MATCH("A?",{1,"aa",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1', '=MATCH("ß?",{1,"ssa",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=MATCH("ß",{1,"ss",2.1,"c bau c",4.1},0)', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=TREND({2;3;4;21;2},{2,1;2,2;5,3;2,4;4,1},{1,2,3})', {},
         "<Ranges>(A1)=[[#REF!]]"),

        ('A1:A5', '=GROWTH({2;3;4;21;2},{1,2,1;2,2,2;1,5,3;6,2,4;2,4,1})', {},
         "<Ranges>(A1:A5)=[[ 1.77710473]\n"
         " [ 3.46042359]\n"
         " [ 3.92199614]\n"
         " [20.28858899]\n"
         " [ 2.05996214]]"),
        ('A1', '=PROB({1,2,3,4,5},{0.1,0.2,0.3,0.3,0.1},2,4)', {},
         "<Ranges>(A1)=[[0.8]]"),
        ('A1:C2', '=FREQUENCY(B1:B5,{1,3;5,3})',
         {'B1:B5': [2, 3, 0, -1, sh.EMPTY]},
         "<Ranges>(A1:C2)=[[2 2 2]\n [2 2 2]]"),
        ('A1:A7', '=FREQUENCY({1,2,3,2,1,3,2,1,6,5,4.9,0,-1,-2},{1,3;5,3;3,3})',
         {}, "<Ranges>(A1:A7)=[[6]\n [5]\n [2]\n [0]\n [0]\n [0]\n [1]]"),
        ('A1', '=Z.TEST({4,"","13/09/1878"},0,3)', {},
         "<Ranges>(A1)=[[#DIV/0!]]"),
        ('A1', '=Z.TEST(#REF!,#NULL!, #N/A)', {},
         "<Ranges>(A1)=[[#NULL!]]"),
        ('A1', '=RANK.AVG(0,{0,-10,0,2.2,-0.1},1)', {}, "<Ranges>(A1)=[[3.5]]"),
        ('A1', '=RANK.AVG(20,{50,40,20,20,10},1)', {}, "<Ranges>(A1)=[[2.5]]"),
        ('A1', '=RANK.EQ(20,{50,40,20,20,10},1)', {}, "<Ranges>(A1)=[[2.0]]"),
        ('A1', '=RANK.AVG(20,{50,40,20,20,10},0)', {}, "<Ranges>(A1)=[[3.5]]"),
        ('A1', '=RANK.EQ(20,{50,40,20,20,10},0)', {}, "<Ranges>(A1)=[[3.0]]"),
        ('A1', '=POISSON.DIST(0,,)', {}, "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=PERMUT(100,3)', {}, "<Ranges>(A1)=[[970200.0]]"),
        ('A1',
         '=PERCENTRANK.INC({0,0.1,0.7,0.002,4,"","13/01/1900",TRUE},TRUE,4)',
         {}, "<Ranges>(A1)=[[0.7727]]"),
        ('A1',
         '=PERCENTRANK.INC({0,0.1,0.7,0.002,4,"","13/01/1900",TRUE},0.7,4)',
         {}, "<Ranges>(A1)=[[0.75]]"),
        ('A1', '=PERCENTRANK.EXC({0,0.1,3,0.002,4,"","13/01/1900",TRUE},0,4)',
         {}, "<Ranges>(A1)=[[0.1666]]"),
        ('A1', '=PERCENTRANK.EXC({0,0.1,3,0.002,4,"","13/01/1900",TRUE},4,4)',
         {}, "<Ranges>(A1)=[[0.8333]]"),
        ('A1', '=PERCENTRANK.EXC({0,0.1,3,0.002,4,"","13/01/1900",TRUE},3,4)',
         {}, "<Ranges>(A1)=[[0.6666]]"),
        ('A1', '=PERCENTRANK.EXC({0,0.1,0.7,0.002,4,"","13/01/1900",TRUE},3,4)',
         {}, "<Ranges>(A1)=[[0.7828]]"),
        ('A1', '=MODE.SNGL({"",TRUE,"29/02/1900"})', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=MODE.SNGL({"",TRUE,"29/02/1900"})', {},
         "<Ranges>(A1)=[[#N/A]]"),
        ('A1', '=MODE.SNGL({0.2,0.3},{0.3,-0.1},{-0.1,0},"29/02/1900")', {},
         "<Ranges>(A1)=[[0.3]]"),
        ('A1', '=MODE.SNGL({-0.1,0},{0.2,0.3},{0.3,-0.1},"29/02/1900")', {},
         "<Ranges>(A1)=[[-0.1]]"),
        ('A1', '=MODE.SNGL({0.1,0.2},{0.2,0.3},{0.3,-0.1},"")', {},
         "<Ranges>(A1)=[[0.2]]"),
        ('A1:A3', '=MODE.MULT(0,1,2,3,2,1,0,1,2,3,6,7,0)', {},
         "<Ranges>(A1:A3)=[[0]\n [1]\n [2]]"),
        ('A1', '=HYPGEOM.DIST(0,2,3,3,1)', {}, "<Ranges>(A1)=[[0.0]]"),
        ('A1', '=HYPGEOM.DIST(0,0,0,0,0)', {}, "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=F.TEST({0.1,0.9,3.2,0.7},{0.1,0.2,0.3})', {},
         "<Ranges>(A1)=[[0.010767145675124415]]"),
        ('A1', '=COVARIANCE.P({0.2,0.3,-0.1},{-0.1,"","29/02/1900"})', {},
         "<Ranges>(A1)=[[0.0]]"),
        ('A1', '=COVARIANCE.P({2,4,8},{5,11,12})', {},
         "<Ranges>(A1)=[[6.4444444444444455]]"),
        ('A1', '=COVARIANCE.S({2,4,8},{5,11,12})', {},
         "<Ranges>(A1)=[[9.666666666666668]]"),
        ('A1', '=CONFIDENCE.NORM(0.05,2.5,50)', {},
         "<Ranges>(A1)=[[0.6929519121748386]]"),
        ('A1', '=CHISQ.TEST({#NULL!,3,#REF!},{5,#VALUE!,5})', {},
         "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=CHISQ.TEST(A2:B4,A6:B8)', {
            'A2:B4': [[58, 35], [11, 25], [10, 23]],
            'A6:B8': [[45.35, 47.65], [17.56, 18.44], [16.09, 16.91]]
        }, "<Ranges>(A1)=[[0.0003081920170083095]]"),
        ('A1', '=BINOM.DIST.RANGE(0,0,4/7,0.1)', {},
         "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=BINOM.DIST(0.7,".1",".3",FALSE)', {},
         "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=BINOM.DIST.RANGE(0.7,".1",".3",0.5)', {},
         "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=BINOM.DIST.RANGE(60,0.75,48)', {},
         "<Ranges>(A1)=[[0.08397496742904753]]"),
        ('A1', '=BINOM.INV(0.7,0.9,0.9)', {},
         "<Ranges>(A1)=[[0.0]]"),
        ('A1', '=BINOM.INV(40,0.2,0.7)', {},
         "<Ranges>(A1)=[[9.0]]"),
        ('A1', '=BETA.DIST(1,"29/02/1900",0.7,1)', {},
         "<Ranges>(A1)=[[1.0]]"),
        ('A1', '=LOGNORM.INV(0.1, 2, 2)', {},
         "<Ranges>(A1)=[[0.5694392746085293]]"),
        ('A1', '=LOGNORM.INV(0.1, 2, 2)', {},
         "<Ranges>(A1)=[[0.5694392746085293]]"),
        ('A1', '=LOGNORM.DIST(4,1,2,FALSE)', {},
         "<Ranges>(A1)=[[0.048946227003151065]]"),
        ('A1', '=LOGNORM.DIST(4,1,2,TRUE)', {},
         "<Ranges>(A1)=[[0.5765781482392448]]"),
        ('A1', '=T(A2)', {"A2": (sh.EMPTY,)}, "<Ranges>(A1)=[['']]"),
        ('A1', '=@TEXTAFTER("1,2,3,4,5,6,7,8",{"9"},1,1,1)', {},
         "<Ranges>(A1)=[['']]"),
        ('A1', '=TEXTAFTER("1,2,3,4,5,6,7,8",{"3","4","5"},2)', {},
         "<Ranges>(A1)=[[',5,6,7,8']]"),
        ('A1:C3',
         '=TEXTSPLIT("1,2.3;4,,5AA6.7",{",","."},{";","a"},TRUE,1,"pi")', {},
         "<Ranges>(A1:C3)=[['1' '2' '3']\n ['4' '5' 'pi']\n ['6' '7' 'pi']]"),
        ('A1', '=FIXED("123456789076754345682",-12)', {},
         "<Ranges>(A1)=[['123,456,789,000,000,000,000']]"),
        ('A1', '=FIXED("123456789076754345682.2121234567832131341","3", 0)', {},
         "<Ranges>(A1)=[['123,456,789,076,754,000,000.000']]"),
        ('A1', '=FIXED(10001,"36", "FALSe")', {},
         "<Ranges>(A1)=[['10,001.000000000000000000000000000000000000']]"),
        ('A1', '=FIXED(6,5)', {}, "<Ranges>(A1)=[['6.00000']]"),
        ('A1', '=VALUETOTEXT(A2,1)', {
            "A2": (sh.EMPTY,)
        }, "<Ranges>(A1)=[['']]"),
        ('A1', '=ARRAYTOTEXT(A2:B3,1)', {
            "A2:B3": [[1, sh.EMPTY], [3, 4]]
        }, "<Ranges>(A1)=[['{1,;3,4}']]"),
        ('A1', "=ARRAYTOTEXT({1,2;3,4},0)", {},
         "<Ranges>(A1)=[['1, 2, 3, 4']]"),
        ('A1', '=REGEXEXTRACT(A2,"\d",1)', {
            "A2": (sh.EMPTY,)
        }, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=REGEXEXTRACT("abc","[ac]+", 2)', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1:C1', '=REGEXEXTRACT("1a2b3c","((\d)[bc]+|(3c))", 2)', {},
         "<Ranges>(A1:C1)=[['2b' '2' #N/A]]"),
        ('A1:B1', '=REGEXEXTRACT("abc","([ac]+)", 1)', {},
         "<Ranges>(A1:B1)=[['a' 'c']]"),
        ('A1', '=REGEXTEST(0, "^\d+$")', {}, '<Ranges>(A1)=[[True]]'),
        ('A1', '=NUMBERVALUE("(20(027)", "(")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=NUMBERVALUE("250(027", "(")', {}, '<Ranges>(A1)=[[250.027]]'),
        ('A1', '=NUMBERVALUE(A2,".",",.,")', {
            "A2": (sh.EMPTY,)
        }, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=NUMBERVALUE("(2500.27%)%",".",",.,")', {},
         '<Ranges>(A1)=[[-0.250027]]'),
        ('A1', '=NUMBERVALUE("(2.500,27)", ",", ".")', {},
         '<Ranges>(A1)=[[-2500.27]]'),
        ('A1', '=VALUETOTEXT("1987-08-26")', {},
         '<Ranges>(A1)=[[\'1987-08-26\']]'),
        ('A1', '="Aß"="ass"', {}, '<Ranges>(A1)=[[True]]'),
        ('A1', '=CODE(CHAR(1))', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=BAHTTEXT(1234)', {},
         '<Ranges>(A1)=[[\'หนึ่งพันสองร้อยสามสิบสี่บาทถ้วน\']]'),
        ('A1', '=ASC("カタカナガッツ・ポーズヴァイオリン「全角」かな、カナ。")', {},
         '<Ranges>(A1)=[[\'ｶﾀｶﾅｶﾞｯﾂ･ﾎﾟｰｽﾞｳﾞｧｲｵﾘﾝ｢全角｣かな､ｶﾅ｡\']]'),
        ('A1', '=IMARGUMENT("0")', {}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=IMCONJUGATE("0")', {}, '<Ranges>(A1)=[[\'0\']]'),
        ('A1', '=IMSUM(A2:B2,"7+2ids")', {
            "A2:B2": ("7+2i", sh.EMPTY,)
        }, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=IMCOT("0")', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=IMPOWER("7+25i",2)', {}, '<Ranges>(A1)=[[\'-576+350i\']]'),
        ('A1', '=IMTAN(TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=IMDIV(A2,"7+25i")', {"A2": (sh.EMPTY,)},
         '<Ranges>(A1)=[[\'0\']]'),
        ('A1', '=IMDIV("7+2i",A2)', {
            "A2": (sh.EMPTY,)
        }, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=IMDIV("5+25i","1+1i")', {},
         '<Ranges>(A1)=[[\'15+10i\']]'),
        ('A1', '=IMPRODUCT(A2,B2)', {
            "A2": Error.errors["#N/A"], "B2": Error.errors["#DIV/0!"]
        }, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=IMPRODUCT("7+25i","7+25i")', {},
         '<Ranges>(A1)=[[\'-576+350i\']]'),
        ('A1', '=IMSUM(2,2)', {}, '<Ranges>(A1)=[[\'4\']]'),
        ('A1', '=IMLOG2("7+25i")', {},
         '<Ranges>(A1)=[[\'4.698302390590929+1.8723117688508721i\']]'),
        ('A1', '=COMPLEX(0,1)', {}, '<Ranges>(A1)=[[\'i\']]'),
        ('A1', '=COMPLEX(3,4)', {}, '<Ranges>(A1)=[[\'3+4i\']]'),
        ('A1', '=COMPLEX(3,4,"j")', {}, '<Ranges>(A1)=[[\'3+4j\']]'),
        ('A1', '=COMPLEX(1,0)', {}, '<Ranges>(A1)=[[\'1\']]'),
        ('A1', '=COMPLEX(1E-22, 2)', {}, '<Ranges>(A1)=[[\'1E-22+2i\']]'),
        ('A1', '=IMABS("5+12i")', {}, '<Ranges>(A1)=[[13.0]]'),
        ('A1', '=TEXT(130.33332, "a/p aM/pm a/P")', {},
         '<Ranges>(A1)=[[\'a AM a\']]'),
        ('A1', '=TEXT(130.63332, "a/p aM/pm a/P am/pm")', {},
         '<Ranges>(A1)=[[\'p PM P PM\']]'),
        ('A1', '=TEXT(45141.2, "A/P a/P A/p a/p am/pm")', {},
         '<Ranges>(A1)=[[\'A a A a AM\']]'),
        ('A1', '=TEXT(1.23, "#mm")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=TEXT("s", "A/P a/P A/p a/p am/pm")', {},
         '<Ranges>(A1)=[[\'s\']]'),
        ('A1', '=CONVERT(1,"Pang3","L")', {}, '<Ranges>(A1)=[[1e+18]]'),
        ('A1', '=CONVERT(1,"pc","m")', {},
         '<Ranges>(A1)=[[3.0856775812815532e+16]]'),
        ('A1', '=CONVERT(1,"kgal","L")', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=CONVERT(3,"mi","km")', {}, '<Ranges>(A1)=[[4.828032]]'),
        ('A1', '=BITLSHIFT(4,-1)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=BITAND(13,25.3)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=BITAND(13,25)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=BITAND(1,5)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=BESSELI(B1,"3.2")', {'B1': (sh.EMPTY,)},
         '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=BESSELI(2,-1)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=BESSELI(2.3,2)', {}, '<Ranges>(A1)=[[1.0054316636559144]]'),
        ('A1', '=MATCH("ß",{"ß","ss"})', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH("1",{1,"1"})', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH(1,{1,"1"})', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=SEARCH("ß","prtessr")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SUMIFS({1, 2}, {"ß","ss"}, "ß")', {}, '<Ranges>(A1)=[[3.0]]'),
        ('A1', '=EOMONTH(1,-3)', {}, "<Ranges>(A1)=[[#NUM!]]"),
        ('A1', '=EOMONTH(444,3)', {}, "<Ranges>(A1)=[[547]]"),
        ('A1', '=EOMONTH(444,-3)', {}, "<Ranges>(A1)=[[366]]"),
        ('A1', '=TEXT(36852.98609953704, "yyyy-mm-dd-HH:MM:SS")', {},
         "<Ranges>(A1)=[['2000-11-22-23:39:59']]"),
        ('A1',
         '=TEXT("4/30/1900  10:39:48 AM", "pp  yy MM \sal\e\s mm yy ss  trtrt mm mm dd mm ss mm hh mm ss ss mm mm")',
         {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1',
         '=TEXT("4/30/1900  10:39:48 AM", "pp  yy MM \sal\e\s mm yy ss  trtrt mm mm dd mm ss mm hh mm ss ss mm")',
         {},
         "<Ranges>(A1)=[['pp  00 04 sales 04 00 48  trtrt 39 04 30 39 48 04 10 39 48 48 04']]"),
        ('A1', '=SUBSTITUTE("0, 1.9,2.3,10,-1,ciao, TRUE",B1,"")', {
            'B1': (sh.EMPTY,)
        }, "<Ranges>(A1)=[['0, 1.9,2.3,10,-1,ciao, TRUE']]"),
        ('A1', '=T("A")', {}, "<Ranges>(A1)=[['A']]"),
        ('A1', '=T(TRUE)', {}, "<Ranges>(A1)=[['']]"),
        ('A1:A2', '=FILTER({"a";2.1},TRUE)', {},
         "<Ranges>(A1:A2)=[['a']\n [2.1]]"),
        ('A1:B1', '=FILTER({"a",2.1},TRUE)', {}, "<Ranges>(A1:B1)=[['a' 2.1]]"),
        ('A1', '=FILTER({"a",2.1,"b",4.1},{FALSE,FALSE,FALSE,FALSE})', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1:C1', '=FILTER({1,2,3;4,5,6},A2:C3)', {
            'A2:C3': [[1, 1, 1], [1, 1, 1]]
        }, "<Ranges>(A1:C1)=[[#VALUE! #VALUE! #VALUE!]]"),
        ('A1:C1', '=FILTER({1,2,3},A2:C2)', {
            'A2:C2': (1, 1, sh.EMPTY)
        }, "<Ranges>(A1:C1)=[[1 2 #N/A]]"),
        ('A1:C1', '=FILTER({1,2,3},A2:C2)', {
            'A2:C2': (1, sh.EMPTY, 1)
        }, "<Ranges>(A1:C1)=[[1 3 #N/A]]"),
        ('A1', '=TEXTJOIN(" ",FALSE, B2:B5)', {
            'B2:B5': (sh.EMPTY, "sun", sh.EMPTY, "come")
        }, "<Ranges>(A1)=[[' sun  come']]"),
        ('A1', '=TEXTJOIN(" ",TRUE, "The", "sun", "will", "come.")', {},
         "<Ranges>(A1)=[['The sun will come.']]"),
        ('A1:B2', '=FILTER({#VALUE!,2,"b",0;"a",3,"b",4.3},{FALSE,-2,0,4.1})',
         {}, "<Ranges>(A1:B2)=[[2 0]\n [3 4.3]]"),
        ('A1', '=FILTER({"a",2.1,"b",4.1},{FALSE,FALSE,FALSE,FALSE})', {},
         "<Ranges>(A1)=[[#VALUE!]]"),
        ('A1', '=FILTER({"a",2.1,"b",4.1},{FALSE,FALSE,FALSE,FALSE},2)', {},
         "<Ranges>(A1)=[[2]]"),
        ('A1:D1', '=FILTER({"a",2.1,"b",4.1},{TRUE,-2.1,7,4.1})', {},
         "<Ranges>(A1:D1)=[['a' 2.1 'b' 4.1]]"),
        ('A1', '=LOOKUP("b",{"a",2.1,"b",4.1})', {}, "<Ranges>(A1)=[['b']]"),
        ('A1', '=SUM(B1:D1  (  B1:B2  ,  D1:D2  ))',
         {'B1': 1, 'D1': 1}, '<Ranges>(A1)=[[2.0]]'),
        ('A1', '=SUM(MUNIT({1,#NUM!;3,4;7,8}))', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=SUM(MUNIT({1,2;3,4;7,8}))', {}, '<Ranges>(A1)=[[6.0]]'),
        ('A1', '=MUNIT(0)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=MUNIT(1)', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=MINVERSE({1,3,8,5;1,3,6,1})', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=MDETERM({1,3,8,5;1,3,6,1;1,1,1,0;7,3,10,2})', {},
         '<Ranges>(A1)=[[87.99999999999996]]'),
        ('A1', '=MDETERM({1,3,8,5;1,3,6,1})', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=MDETERM({3,6;1,1})', {}, '<Ranges>(A1)=[[-3.0]]'),
        ('A1', '=MDETERM({3,6,1;1,1,0;3,10,2})', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=SUM(MINVERSE({1,1;1,1}))', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=SUM(MINVERSE({1,"2",-1;3,4,-1;0,2,0}))', {},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SUM(MINVERSE({1,2,-1;3,4,-1;0,2,0}))', {},
         '<Ranges>(A1)=[[-0.5]]'),
        ('A1', '=MMULT({0,-10},{0.9;9})', {}, '<Ranges>(A1)=[[-90.]]'),
        ('A1', '=MMULT({"0",-10},{0.9;9})', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=MMULT({0,-10, 1},{0.9;9})', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=PERCENTILE.INC({0,-10,0.9,0.1,-0.1},0.9)', {},
         '<Ranges>(A1)=[[0.5800000000000001]]'),
        ('A1', '=NORM.S.INV(0.7)', {}, '<Ranges>(A1)=[[0.5244005127080407]]'),
        ('A1', '=QUARTILE.EXC({#REF!,5},5)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=QUARTILE({#REF!,5},5)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=QUARTILE(B2:B5,0)',
         {'B2:B5': (sh.EMPTY, sh.EMPTY, sh.EMPTY, sh.EMPTY)},
         '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=QUARTILE(B2:B5,-1)',
         {'B2:B5': (1, 1, 1, 1)}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=QUARTILE(B2:B5,5)',
         {'B2:B5': (1, 1, 1, 1)}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=QUARTILE(B2:B5,1)',
         {'B2:B5': (1, 2, 3, sh.EMPTY)}, '<Ranges>(A1)=[[1.5]]'),
        ('A1', '=QUARTILE(B2:B5,2)',
         {'B2:B5': (1, 2, 3, 4)}, '<Ranges>(A1)=[[2.5]]'),
        ('A1', "=IPMT((3 / 12), 1, (3 * 12), 8000)", {},
         '<Ranges>(A1)=[[-2000.0]]'),
        ('A1', '=IF(,4,)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=IF(,,)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=IF(,,A2)', {'A2': 5}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=CODE(A2)', {'A2': [[sh.EMPTY]]}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=CHAR(1)', {}, '<Ranges>(A1)=[[\'\\x01\']]'),
        ('A1', '=CHAR(157)', {}, '<Ranges>(A1)=[[\'ù\']]'),
        ('A1', '=TEXT("a", "mm")', {}, '<Ranges>(A1)=[[\'a\']]'),
        ('A1', '=VALUE("26/08/1987 12:00")', {},
         '<Ranges>(A1)=[[32015.5]]'),
        ('A1', '=TEXT("26/08/87", "#.0")', {}, '<Ranges>(A1)=[[\'32015.0\']]'),
        ('A1', '=TEXT("a", "#mm")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=TEXT(1.23, "#mm")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=TEXT(130.33332, "hm hmm hmmm hmmmm")', {},
         '<Ranges>(A1)=[[\'759 759 7May 7May\']]'),
        ('A1', '=TEXT(1.23, "#""°F""""")', {},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=TEXT(45141, "m mm mMm mmmm mmmmm")', {},
         '<Ranges>(A1)=[[\'8 08 Aug August A\']]'),
        ('A1', '=TEXT(45141.2, "A/P a/P A/p a/p am/pm")', {},
         '<Ranges>(A1)=[[\'A a A a AM\']]'),
        ('A1', '=TEXT(45141.6, "A/P a/P A/p a/p am/pm")', {},
         '<Ranges>(A1)=[[\'P P p p PM\']]'),
        ('A1', '=TEXT(1234567898, "[<=9999999]###-####;(###) ###-####")', {},
         '<Ranges>(A1)=[[\'(123) 456-7898\']]'),
        ('A1', '=TEXT(2,"xxx")', {}, '<Ranges>(A1)=[[\'xxx\']]'),
        ('A1', '=TEXT(2,"[<=1]x;[=2]xx;xxx")', {}, '<Ranges>(A1)=[[\'xx\']]'),
        ('A1', '=TEXT(0.000122, "0.00E-##")', {},
         '<Ranges>(A1)=[[\'1.22E-4\']]'),
        ('A1', '=TEXT(0.000122, "0.00E+00")', {},
         '<Ranges>(A1)=[[\'1.22E-04\']]'),
        ('A1', '=TEXT(12200000, "0.00E-00")', {},
         '<Ranges>(A1)=[[\'1.22E07\']]'),
        ('A1', '=TEXT(12200000, "0.00E+00")', {},
         '<Ranges>(A1)=[[\'1.22E+07\']]'),
        ('A1', '=TEXT(-1234.567, "$#,##0.00")', {},
         '<Ranges>(A1)=[[\'-$1,234.57\']]'),
        ('A1', '=TEXT(123456, "##0° 00\' 00\'\'")', {},
         '<Ranges>(A1)=[["12° 34\' 56\'\'"]]'),
        ('A1', '=TEXT(1234.567, "$#,##0.00")', {},
         '<Ranges>(A1)=[[\'$1,234.57\']]'),
        ('A1', '=TEXT(0.285,"0.0%")', {}, '<Ranges>(A1)=[[\'28.5%\']]'),
        ('A1', '=TEXT(1234,"5000700")', {}, '<Ranges>(A1)=[[\'5012734\']]'),
        ('A1', '=TEXT(1234578.555,"#,##00000000####0### .  # ,0.00.0,00")', {},
         '<Ranges>(A1)=[[\'0,000,000,01,234,578 .  5 ,5.50.000\']]'),
        ('A1', '=TEXT(1234567890,"0,## ##, ##")', {},
         '<Ranges>(A1)=[[\'1,234,56 7,8 90\']]'),
        ('A1', '=TEXT(1000000, "##,00.00")', {},
         '<Ranges>(A1)=[[\'1,000,000.00\']]'),
        ('A1', '=TEXT(1.23, "#%%%")', {}, '<Ranges>(A1)=[[\'1230000%%%\']]'),
        ('A1', '=TEXT(10.93,"#"".505#""°F""#""")', {},
         '<Ranges>(A1)=[[\'11.505#°F#\']]'),
        ('A1', '=TEXT(1,"00###.##.0000.00")', {},
         '<Ranges>(A1)=[[\'001..0000.00\']]'),
        ('A1',
         '=TEXT(1234578.555, "###0$0000,  ,00 , 0########    .  # ,0.00.0,00")',
         {}, '<Ranges>(A1)=[[\'0$0000  ,00 , 01234578    .  5 ,5.50.000\']]'),
        ('A1', '=TEXT(10.93, "#"" ""°F"" """)', {},
         '<Ranges>(A1)=[[\'11 °F \']]'),

        ('A1', '=AND(TRUE,"0",#REF!)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=VALUE(#NAME?)', {}, '<Ranges>(A1)=[[#NAME?]]'),
        ('A1', '=VALUE(FALSE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=VALUE(0)', {}, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=VALUE("26/08/1987")', {}, '<Ranges>(A1)=[[32015.0]]'),
        ('A1', '=ROMAN(0)', {}, '<Ranges>(A1)=[[\'\']]'),
        ('A1', '=ROMAN(1)', {}, '<Ranges>(A1)=[[\'I\']]'),
        ('A1', '=ROMAN(4000)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=ROMAN(1, 5)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=ADDRESS(2,3,2,FALSE)', {}, '<Ranges>(A1)=[[\'R2C[3]\']]'),
        ('A1', '=ADDRESS(2,3,2)', {}, '<Ranges>(A1)=[[\'C$2\']]'),
        ('A1', '=ADDRESS(2,3)', {}, '<Ranges>(A1)=[[\'$C$2\']]'),
        ('A1', '=ADDRESS(1,1,1, FALSE,"sss\'")', {},
         '<Ranges>(A1)=[["\'sss\'\'\'!R1C1"]]'),
        ('A1', '=LARGE({-1.1,4.1,#REF!},1)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=LARGE({-1.1,4.1,#REF!},"c")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=LARGE({-1.1,4.1,#REF!},#N/A)', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=LARGE(#N/A,#VALUE!)', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=SLOPE(A2,{1})', {'A2': [[sh.EMPTY]]},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=FORECAST(32,{2,3,4},{1,1,1})', {}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=NPER(0,4,5)', {}, '<Ranges>(A1)=[[-1.25]]'),
        ('A1', '=CUMIPMT(0.09/12,30*12,125000,13,24,4)', {},
         '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=CUMIPMT(3,5.5,125000,1,6,1)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATEDIF("01/01/01","06/02/08","yd")', {},
         '<Ranges>(A1)=[[152]]'),
        ('A1', '=DATEDIF("01/09/01","06/02/08","YD")', {},
         '<Ranges>(A1)=[[144]]'),
        ('A1', '=DATEDIF("12/31/2001","12/29/2008","MD")', {},
         '<Ranges>(A1)=[[28]]'),
        ('A1', '=DATEDIF("12/20/2001","12/19/2008","MD")', {},
         '<Ranges>(A1)=[[29]]'),
        ('A1', '=DATEDIF("12/29/2001","12/29/2008","MD")', {},
         '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATEDIF("01/31/2001","01/29/2008","YD")', {},
         '<Ranges>(A1)=[[363]]'),
        ('A1', '=DATEDIF("01/01/2001","02/29/2008","YD")', {},
         '<Ranges>(A1)=[[59]]'),
        ('A1', '=DATEDIF("01/01/2001","03/01/2008","YD")', {},
         '<Ranges>(A1)=[[59]]'),
        ('A1', '=DATEDIF("01/09/00","06/02/08","YD")', {},
         '<Ranges>(A1)=[[145]]'),
        ('A1', '=DATEDIF("01/31/2001","03/29/2007","YD")', {},
         '<Ranges>(A1)=[[57]]'),
        ('A1', '=DATEDIF("01/31/01","02/29/08","YM")', {},
         '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATEDIF(1, 0, "YD")', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATEDIF(-1, 222222, "YD")', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATEDIF(-1, 222222, "D")', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATEDIF(0, 222222, "YD")', {}, '<Ranges>(A1)=[[154]]'),
        ('A1', '=DATEDIF(1, 222222, "YD")', {}, '<Ranges>(A1)=[[153]]'),
        ('A1', '=DATEDIF("01/09/01","02/29/08","YD")', {},
         '<Ranges>(A1)=[[51]]'),
        ('A1', '=DATEDIF(60,222222,"YD")', {}, '<Ranges>(A1)=[[94]]'),
        ('A1', '=DATEDIF(222,222290,"YD")', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATEDIF(222,222289,"YD")', {}, '<Ranges>(A1)=[[364]]'),
        ('A1', '=DATEDIF(222,222222,"YD")', {}, '<Ranges>(A1)=[[297]]'),
        ('A1', '=DATEDIF(222,222290,"YM")', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATEDIF(222,222289,"YM")', {}, '<Ranges>(A1)=[[11]]'),
        ('A1', '=DATEDIF(222,222222,"YM")', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=DATEDIF(222,222290,"MD")', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATEDIF(222,222289,"MD")', {}, '<Ranges>(A1)=[[30]]'),
        ('A1', '=DATEDIF(222,222222,"MD")', {}, '<Ranges>(A1)=[[24]]'),
        ('A1', '=DATEDIF(222,222290,"Y")', {}, '<Ranges>(A1)=[[608]]'),
        ('A1', '=DATEDIF(222,222289,"Y")', {}, '<Ranges>(A1)=[[607]]'),
        ('A1', '=DATEDIF(222,222222,"Y")', {}, '<Ranges>(A1)=[[607]]'),
        ('A1', '=DATEDIF(222,222290,"M")', {}, '<Ranges>(A1)=[[7296]]'),
        ('A1', '=DATEDIF(222,222289,"M")', {}, '<Ranges>(A1)=[[7295]]'),
        ('A1', '=DATEDIF(222,222222,"M")', {}, '<Ranges>(A1)=[[7293]]'),
        ('A1', '=DATEDIF(222,222290,"D")', {}, '<Ranges>(A1)=[[222068]]'),
        ('A1', '=DATEDIF(222,222289,"D")', {}, '<Ranges>(A1)=[[222067]]'),
        ('A1', '=DATEDIF(222,222222,"D")', {}, '<Ranges>(A1)=[[222000]]'),
        ('A1', '=EDATE(1, TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=EDATE(1, "DSD")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=WEEKDAY(-1, 21)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DAY(1)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(-1, 21)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=WEEKNUM(1, 22)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=WEEKNUM(0, 21)', {}, '<Ranges>(A1)=[[52]]'),
        ('A1', '=WEEKNUM(1, 21)', {}, '<Ranges>(A1)=[[52]]'),
        ('A1', '=WEEKNUM(2, 21)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(8, 21)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(15, 21)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(22, 21)', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=WEEKNUM(29, 21)', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=WEEKNUM(9, 21)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(16, 21)', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=WEEKNUM(23, 21)', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=WEEKNUM(30, 21)', {}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=WEEKNUM(9, 21)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(64, 21)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(65, 21)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(0, 11)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 12)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 13)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 14)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 15)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 16)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(0, 17)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=WEEKNUM(0, 1)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=WEEKNUM(0, 2)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(60, 21)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 11)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 12)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 13)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 14)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 15)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 16)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 17)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 1)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 2)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(11690, 21)', {}, '<Ranges>(A1)=[[53]]'),
        ('A1', '=WEEKNUM(12055, 21)', {}, '<Ranges>(A1)=[[52]]'),
        ('A1', '=WEEKNUM(12056, 21)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12057, 21)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12425, 11)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12425, 12)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12425, 13)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12425, 14)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12425, 15)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12425, 16)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12425, 17)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12425, 1)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12425, 2)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(60, 11)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 12)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 13)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(60, 14)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 15)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 16)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 17)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 1)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=WEEKNUM(60, 2)', {}, '<Ranges>(A1)=[[10]]'),
        ('A1', '=WEEKNUM(11326)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(36534)', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=WEEKNUM(36527)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(12420)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(12421)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(11323)', {}, '<Ranges>(A1)=[[53]]'),
        ('A1', '=WEEKNUM(11325)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(11327)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(36533)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=WEEKNUM(36526)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=WEEKNUM(36525)', {}, '<Ranges>(A1)=[[53]]'),
        ('A1', '=WEEKDAY(36525)', {}, '<Ranges>(A1)=[[6]]'),
        ('A1', '=WEEKDAY(0,17)', {}, '<Ranges>(A1)=[[7]]'),
        ('A1', '=WEEKDAY("a",FALSE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=WEEKDAY(0,FALSE)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=WEEKDAY(0,11)', {}, '<Ranges>(A1)=[[6]]'),
        ('A1', '=WEEKDAY("2.9", "3.9")', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=WEEKDAY(TRUE, 2)', {}, '<Ranges>(A1)=[[7]]'),
        ('A1', '=WEEKDAY(DATE(1008,5,23))', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=CUMIPMT(0.09/12,30*12,125000,13,24,TRUE)', {},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=PRODUCT({3,4},{8,9})', {}, '<Ranges>(A1)=[[864.0]]'),
        ('A1', '=SLOPE({3,4},{8,9})', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=CORREL(A2:A4,{8,2})', {'A2:A4': [[sh.EMPTY] * 3]},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=CORREL({"2",4,1},{7,"",TRUE})', {},
         '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=CORREL(A2:A3,{8,2})', {'A2:A3': [[sh.EMPTY] * 2]},
         '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=CORREL(A2,{8,2})', {'A2': [[sh.EMPTY]]},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=CORREL({"2",4,1},{7,3,6})', {}, '<Ranges>(A1)=[[-1.0]]'),
        ('A1', '=IPMT(0.1/12, 60, 3*12, 8000)', {},
         '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=PMT(0.08/12,0,10000)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=FORECAST(1,A2,{8,2})', {'A2': [[sh.EMPTY]]},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=FORECAST(1,A2:A3,A2:A3)', {'A2:A3': [[sh.EMPTY, sh.EMPTY]]},
         '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=FORECAST(A2,{3,4},{8,9})', {'A2': [[sh.EMPTY]]},
         '<Ranges>(A1)=[[-5.0]]'),
        ('A1', '=FORECAST(#N/A,{"3"},{9})', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=FORECAST(5,{3},{9})', {}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=FORECAST(5,{3},{8,9})', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=FORECAST(2,A2:A5,{7,TRUE,8,9})',
         {'A2:A5': [[sh.EMPTY, 4, 3, 4]]}, '<Ranges>(A1)=[[-3.0]]'),
        ('A1', '=FORECAST(2,{"1",4,3,4},{7,TRUE,8,9})', {},
         '<Ranges>(A1)=[[-3.0]]'),
        ('A1', '=FORECAST(2,{"1","4",3,4},{7,4,8,9})', {},
         '<Ranges>(A1)=[[-3.0]]'),
        ('A1', '=FORECAST(5,{"a",2,3}, {7,8,9})', {}, '<Ranges>(A1)=[[-1.0]]'),
        ('A1', '=FORECAST(5,{1,2,3}, {7,8,9})', {}, '<Ranges>(A1)=[[-1.0]]'),
        ('A1', '=SEARCH(5,45)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=SEARCH("n", "printer")', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=SEARCH("BASE","database",5.8)', {}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=SEARCH("BASE","database",6.1)', {},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SEARCH("BASE","database",FALSE)', {},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SEARCH("BASE","database",TRUE)', {}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=SEARCH("x","database")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=EDATE(0, 0)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=EDATE(0.7, "15-Jan-11")', {}, '<Ranges>(A1)=[[1234460]]'),
        ('A1', '=EDATE(2.8, ".7")', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=EDATE(2.4, TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=EDATE("15-Jan-11", 1)', {}, '<Ranges>(A1)=[[40589]]'),
        ('A1', '=EDATE(1, "2")', {}, '<Ranges>(A1)=[[61]]'),
        ('A1', '=EDATE(2.4, 1)', {}, '<Ranges>(A1)=[[33]]'),
        ('A1', '=A2 %', {'A2': [[20]]}, '<Ranges>(A1)=[[0.2]]'),
        ('A3', '=@F1:F3', {'F1:F3': [[1], [2], [3]]},
         '<Ranges>(A3)=[[3]]'),
        ('A3', '=SINGLE(F1:F3)', {'F1:F3': [[1], [2], [3]]},
         '<Ranges>(A3)=[[3]]'),
        ('A3', '=COLUMN(SINGLE(F1:F3))', {'F1:F3': [[1], [2], [3]]},
         '<Ranges>(A3)=[[6]]'),
        ('A4', '=COLUMN(SINGLE(F1:F3))', {'F1:F3': [[1], [2], [3]]},
         '<Ranges>(A4)=[[#VALUE!]]'),
        ('A4', '=SINGLE(F1:F3)', {'F1:F3': [[1], [2], [3]]},
         '<Ranges>(A4)=[[#VALUE!]]'),
        ('A1', '=__xludf.DUMMYFUNCTION("(GOOGLEFINANCE($A1,""Name""))")', {},
         "<Ranges>(A1)=[[#NAME?]]"),
        ('A1', '=IFS(-1,A2,1, TRUE)', {'A2': [[sh.EMPTY]]},
         "<Ranges>(A1)=[[0]]"),
        ('A1', '=A2 =A3', {'A2': [[1]], 'A3': [[1]]}, "<Ranges>(A1)=[[True]]"),
        ('A1', '=A2 =-A3', {'A2': [[1]], 'A3': [[-1]]},
         "<Ranges>(A1)=[[True]]"),
        ('A1', '=A2 A3', {'A2': [[1]], 'A3': [[-1]]},
         "<Ranges>(A1)=[[#NULL!]]"),
        ('A1', '="\\n"&SUM(1,2)', {}, "<Ranges>(A1)=[['\\\\n3']]"),
        ('A1', '=SUM("2","4", A2:B2)',
         {'A2:B2': [[True, False]]}, '<Ranges>(A1)=[[6.0]]'),
        ('A1', '=SUM("2","4","ciao")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SUM("2","4")', {}, '<Ranges>(A1)=[[6.0]]'),
        ('A1', '=STDEVPA(B1)',
         {'B1': [[sh.EMPTY]]}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=STDEV.P(B1)',
         {'B1': [[sh.EMPTY]]}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=STDEV.P(1)', {}, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=STDEV.S(1)', {}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=STDEV.S(1, "a")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=STDEV.S(1, "3", 2)', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=STDEV.P(2, 3)', {}, '<Ranges>(A1)=[[0.5]]'),
        ('A1', '=STDEV.S(1, 2, 3)', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=CONCAT(1, TRUE, 3)', {}, '<Ranges>(A1)=[[\'1TRUE3\']]'),

        ('A1', '=LARGE({-1.1,10.1;"40",-2},1.1)', {}, '<Ranges>(A1)=[[-1.1]]'),
        ('A1', '=LARGE(A2:H2,"01/01/1900")', {
            'A2:H2': [[0.1, -10, 0.9, 2.2, -0.1, sh.EMPTY, "02/01/1900", True]]
        }, '<Ranges>(A1)=[[2.2]]'),

        ('A1:B1', '=SMALL(A2:B2,A3:B3)', {
            'A2:B2': [[4, sh.EMPTY]], 'A3:B3': [[1, Error.errors['#N/A']]]
        }, '<Ranges>(A1:B1)=[[4.0 #N/A]]'),
        ('A1', '=SMALL({-1.1,10.1;4.1,"40"},4)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=SMALL({-1.1,10.1;"40",TRUE},2)', {}, '<Ranges>(A1)=[[10.1]]'),
        ('A1', '=LARGE({-1.1,10.1;4.1,"40"},4)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=LARGE({-1.1,10.1;"40",-2},2)', {}, '<Ranges>(A1)=[[-1.1]]'),
        ('A1', '=LOOKUP(2,{-1.1,2.1,3.1,4.1},{#REF!,2.1,3.1,4.1})', {},
         '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=XIRR({-10000,2750,3250,4250,2},'
               '      {"39448",39508,39859,"30/10/2008", TRUE}, 0)',
         {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=XIRR({-10,1,1,2,3},{3,6,7,4,9})',
         {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=IRR({2,0,4},A2)',
         {'A2': [[sh.EMPTY]]}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=IRR({7,2,-1,4,-3;7,2,-1,4,4},A2)',
         {'A2': [[3]]}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=XNPV(0.02,{-10000,2750,3250,4250},{0,39508,39859,39751})',
         {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=NPV(1, A3:C3)', {'A3:C3': [[5, 4, 3]]},
         '<Ranges>(A1)=[[3.875]]'),
        ('A1', '=NPV(D2, A2:C2)',
         {'D2': [["ciao"]], 'A2:C2': [[5, sh.EMPTY, Error.errors['#N/A']]]},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=NPV(5, A2:D2)',
         {'A2:D2': [[5, sh.EMPTY, 'ciao', Error.errors['#N/A']]]},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=COUNT(0,345,TRUE,#VALUE!,"26/08")', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=MAX("")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=MAXA("")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=COUNTIF(A2:B2, "<>-1")', {'A2:B2': [[-1, 2]]},
         '<Ranges>(A1)=[[1]]'),
        ('A1', '=COUNTA(A2)', {'A2': [[sh.EMPTY]]}, '<Ranges>(A1)=[[0]]'),
        ('A1:C1', '=COUNTIF(A2:F2,{"60","29/02/1900","*0"})',
         {'A2:F2': [[60, "29/02/1900", sh.EMPTY, 0, "*", "AUG-98"]]},
         '<Ranges>(A1:C1)=[[2 2 1]]'),
        ('A1:G1', '=COUNTIF(A2:E2,{"<=FALSE",0,"",#VALUE!,"~*",FALSE})',
         {'A2:E2': [[sh.EMPTY, 0, Error.errors['#VALUE!'], "*", False]]},
         '<Ranges>(A1:G1)=[[1 1 1 1 1 1 #N/A]]'),
        ('A1', '=MAX("29/02/1900")', {}, '<Ranges>(A1)=[[60.0]]'),
        ('A1', '=COUNT(A2:E2,"26/08")',
         {'A2:E2': [[0, 345, True, Error.errors['#VALUE!'], "26/08"]]},
         '<Ranges>(A1)=[[3]]'),
        ('A1:G1', '=SUMIF(A2:E2,{"<=FALSE",0,"",#VALUE!,"~*",FALSE},A3:E3)',
         {'A2:E2': [[sh.EMPTY, 0, Error.errors['#VALUE!'], "*", False]],
          'A3:E3': [[11, 7, 5, 9, 2]]},
         '<Ranges>(A1:G1)=[[2.0 7.0 11.0 5.0 9.0 2.0 #N/A]]'),
        ('A1:C1', '=SUMIF(A2:F2,{"60","29/02/1900","*0"},A3:F3)',
         {'A2:F2': [[60, "29/02/1900", sh.EMPTY, 0, "*", "AUG-98"]],
          'A3:F3': [[1, 3, 11, 7, 9, 13]]},
         '<Ranges>(A1:C1)=[[4.0 4.0 3.0]]'),
        ('A1', '=YEARFRAC("31/8/1987","29/8/1986",0)', {},
         '<Ranges>(A1)=[[1.0055555555555555]]'),
        ('A1', '=YEARFRAC("31/8/1987","29/8/1986",4)', {},
         '<Ranges>(A1)=[[1.0027777777777778]]'),
        ('A1', '=YEARFRAC("30/8/1987","29/8/1986",0)', {},
         '<Ranges>(A1)=[[1.0027777777777778]]'),
        ('A1', '=YEARFRAC("31/8/1987","30/8/1986",0)', {},
         '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=YEARFRAC("30/8/1987","31/8/1986",0)', {},
         '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=YEARFRAC(0,345,TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=YEARFRAC("26/8/1987 05:00 AM",345,4)', {},
         '<Ranges>(A1)=[[86.71111111111111]]'),
        ('A1', '=YEARFRAC("26/8/1987 05:00 AM",345,1)', {},
         '<Ranges>(A1)=[[86.71043215830248]]'),
        ('A1', '=YEARFRAC(2,1462,1)', {},
         '<Ranges>(A1)=[[3.9978094194961664]]'),
        ('A1', '=YEARFRAC(0,4382,1)', {}, '<Ranges>(A1)=[[12.0]]'),
        ('A1', '=YEARFRAC(1462,4382,1)', {},
         '<Ranges>(A1)=[[7.994524298425736]]'),
        ('A1', '=YEARFRAC(1462,4383,1)', {},
         '<Ranges>(A1)=[[7.997262149212868]]'),
        ('A1', '=YEARFRAC(3,368,1)', {}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=YEARFRAC(366,2000,1)', {},
         '<Ranges>(A1)=[[4.4746691008671835]]'),
        ('A1', '=YEARFRAC(1,6000,1)', {}, '<Ranges>(A1)=[[16.4250281848929]]'),
        ('A1', '=YEARFRAC(1,2000,1)', {}, '<Ranges>(A1)=[[5.474212688270196]]'),
        ('A1', '=YEARFRAC(1,2000,0)', {}, '<Ranges>(A1)=[[5.475]]'),
        ('A1', '=YEARFRAC(1,2000,2)', {}, '<Ranges>(A1)=[[5.552777777777778]]'),
        ('A1', '=YEARFRAC(1,2000,3)', {}, '<Ranges>(A1)=[[5.476712328767123]]'),
        ('A1', '=YEARFRAC(1,2000,4)', {}, '<Ranges>(A1)=[[5.475]]'),
        ('A1', '=HOUR(-1)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=HOUR(0.4006770833)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=MINUTE(2.4)', {}, '<Ranges>(A1)=[[36]]'),
        ('A1', '=MINUTE(0.4006770833)', {}, '<Ranges>(A1)=[[36]]'),
        ('A1', '=SECOND(0.4006770833)', {}, '<Ranges>(A1)=[[58]]'),
        ('A1', '=SECOND(0.4006770834)', {}, '<Ranges>(A1)=[[59]]'),
        ('A1', '=SECOND(0.4)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=SECOND("22-Aug-2011 9:36 AM")', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=TIMEVALUE("22-Aug-2011 9:36 AM")', {}, '<Ranges>(A1)=[[0.4]]'),
        ('A1', '=TIMEVALUE("9:36 AM")', {}, '<Ranges>(A1)=[[0.4]]'),
        ('A1', '=DAY(0)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=MONTH(0.7)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=TIME(24,0,0)', {}, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=TIME(36,12*60,6*60*60)', {}, '<Ranges>(A1)=[[0.25]]'),
        ('A1', '=TIME(36,0,6*60*60)', {}, '<Ranges>(A1)=[[0.75]]'),
        ('A1', '=TIME(12,0,6*60*60)', {}, '<Ranges>(A1)=[[0.75]]'),
        ('A1', '=TIME(0,0,6*60*60)', {}, '<Ranges>(A1)=[[0.25]]'),
        ('A1', '=DAY(2958466)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DAY(60)', {}, '<Ranges>(A1)=[[29]]'),
        ('A1', '=DAY(50)', {}, '<Ranges>(A1)=[[19]]'),
        ('A1', '=DAY(100)', {}, '<Ranges>(A1)=[[9]]'),
        ('A1', '=DAY("29/2/1900")', {}, '<Ranges>(A1)=[[29]]'),
        ('A1', '=DAY("22 August 20")', {}, '<Ranges>(A1)=[[22]]'),
        ('A1', '=DATEVALUE("22 August 20")', {}, '<Ranges>(A1)=[[44065]]'),
        ('A1', '=DATEVALUE("01/01/00")', {}, '<Ranges>(A1)=[[36526]]'),
        ('A1', '=DATEVALUE("01/01/99")', {}, '<Ranges>(A1)=[[36161]]'),
        ('A1', '=DATEVALUE("01/01/29")', {}, '<Ranges>(A1)=[[47119]]'),
        ('A1', '=DATEVALUE("01/01/30")', {}, '<Ranges>(A1)=[[10959]]'),
        ('A1', '=DATEVALUE("8/22/2011")', {}, '<Ranges>(A1)=[[40777]]'),
        ('A1', '=DATEVALUE("22-MAY-2011")', {}, '<Ranges>(A1)=[[40685]]'),
        ('A1', '=DATEVALUE("2011/02/23")', {}, '<Ranges>(A1)=[[40597]]'),
        ('A1', '=DATEVALUE("5-JUL")', {}, '<Ranges>(A1)=[[43651]]'),
        ('A1', '=DATEVALUE("8/1987")', {}, '<Ranges>(A1)=[[31990]]'),
        ('A1', '=DATEVALUE("26/8/1987")', {}, '<Ranges>(A1)=[[32015]]'),
        ('A1', '=DATEVALUE("29/2/1900")', {}, '<Ranges>(A1)=[[60]]'),
        ('A1', '=DATE(1900,3,"")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=DATE(1900,3,"-1")', {}, '<Ranges>(A1)=[[59]]'),
        ('A1', '=DATE(1900,3,-1)', {}, '<Ranges>(A1)=[[59]]'),
        ('A1', '=DATE(1900,3,0)', {}, '<Ranges>(A1)=[[60]]'),
        ('A1', '=DATE(9999,12,32)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATE(9999,12,31)', {}, '<Ranges>(A1)=[[2958465]]'),
        ('A1', '=DATE(-1,2,29)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATE(1899,2,29)', {}, '<Ranges>(A1)=[[693657]]'),
        ('A1', '=DATE(1900,1,0)', {}, '<Ranges>(A1)=[[0]]'),
        ('A1', '=DATE(1900,0,0)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATE(1900,2,28)', {}, '<Ranges>(A1)=[[59]]'),
        ('A1', '=DATE(1900,2,29)', {}, '<Ranges>(A1)=[[60]]'),
        ('A1', '=DATE(1904,2,29)', {}, '<Ranges>(A1)=[[1521]]'),
        ('A1', '=DATE(0,11.9,-40.1)', {}, '<Ranges>(A1)=[[264]]'),
        ('A1', '=DATE(0,11.4,1.9)', {}, '<Ranges>(A1)=[[306]]'),
        ('A1', '=DATE(0,11,0)', {}, '<Ranges>(A1)=[[305]]'),
        ('A1', '=DATE(1,-1.9,1.1)', {}, '<Ranges>(A1)=[[275]]'),
        ('A1', '=DATE(1,-1,0)', {}, '<Ranges>(A1)=[[305]]'),
        ('A1', '=DATE(0,0,0)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DATE(2020,2,29)', {}, '<Ranges>(A1)=[[43890]]'),
        ('A1', '=OR({0,0,0},FALSE,"0")', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=OR(B1,FALSE)', {'B1': [['0']]}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=OR("0",FALSE)', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=XOR({0,0,0},FALSE,FALSE)', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=XOR({0,0},FALSE,FALSE)', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=XOR(TRUE,TRUE)', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=XOR(TRUE(),TRUE())', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=OR(TRUE,#REF!,"0")', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=OR(FALSE,"0",#REF!)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=INDEX({2,3;4,5},FALSE,"0")', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=INDEX({2,3;4,5},0,0)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=INDEX({2,3;4,5}, -1)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=INDEX({2,3;4,5}, 0, 0, -1)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=INDEX({2,3;4,5}, 0, -1)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=INDEX(B1:C1, 1, 1)', {'B1:C1': [[sh.EMPTY, 2]]},
         '<Ranges>(A1)=[[0]]'),
        ('A1', '=INDEX(B1:C2, -1)', {'B1:C2': [[1, 2], [3, 4]]},
         '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=SUM(B1:D1 (B1:D1,B1:C1))',
         {'B1:D1': [[2, 3, 4]], 'B1:C1': [[2, 3]]}, '<Ranges>(A1)=[[14.0]]'),
        ('A1', '=INDEX(B1:D2 (B1:C1,B2:C2), 1, 1, 2)',
         {'B1:C1': [[2, 3]], 'B2:C2': [[4, 5]]}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=INDEX((D1:D2:B1:C1, B2:C2), 1, 1, 2)',
         {'B1:D2': [[2, 3, 6], [4, 5, 7]], 'B2:C2': [[4, 5]]},
         '<Ranges>(A1)=[[4]]'),
        ('A1', '=INDEX({2,3;4,5},#NAME?)', {}, '<Ranges>(A1)=[[#NAME?]]'),
        ('A1:B2', '=INDEX({2,3;4,5},{1,2})', {},
         '<Ranges>(A1:B2)=[[2 4]\n [2 4]]'),
        ('A1', '=INDEX(C1:D2,1)', {'C1:D2': [[2, 3], [4, 5]]},
         '<Ranges>(A1)=[[#REF!]]'),
        ('A1:B1', '=INDEX(C1:D1,1)', {'C1:D1': [2, 3]},
         '<Ranges>(A1:B1)=[[2 2]]'),
        ('A1:B1', '=INDEX({2,3;4,5},1)', {}, '<Ranges>(A1:B1)=[[2 3]]'),
        ('A1', '=INDEX({2,3;4,5},1)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=INDEX({2,3,4},1,1)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=INDEX({2,3,4},2,1)', {}, '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=INDEX({2,3,4},2)', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=LOOKUP(2,{-1.1,2.1,3.1,4.1})', {}, '<Ranges>(A1)=[[-1.1]]'),
        ('A1', '=LOOKUP(3,{-1.1,2.1,3.1,4.1})', {}, '<Ranges>(A1)=[[2.1]]'),
        ('A1', '=SWITCH(TRUE,1,0,,,TRUE,1,7)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=SWITCH(TRUE,1,0,#REF!,3,TRUE,1,7)', {},
         '<Ranges>(A1)=[[#REF!]]'),
        ('A1:D1', '=SWITCH({0,1,TRUE},1,0,,,TRUE,1,7)', {},
         '<Ranges>(A1:D1)=[[0 0 1 #N/A]]'),
        ('A1', '=SWITCH(1,2,0,1,4,,4,5)', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=GCD(5.2, -1, TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=GCD(5.2, -1)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=GCD(5.2, 10)', {}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=GCD(#NAME?, #VALUE!, #N/A)', {}, '<Ranges>(A1)=[[#NAME?]]'),
        ('A1', '=GCD(55, 15, 5)', {}, '<Ranges>(A1)=[[5]]'),
        ('A1', '=5%', {}, '<Ranges>(A1)=[[0.05]]'),
        ('A1', '=IF(#NAME?, #VALUE!, #N/A)', {}, '<Ranges>(A1)=[[#NAME?]]'),
        ('A1', '=IF(TRUE, #VALUE!, #N/A)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=IF(FALSE, #VALUE!, #N/A)', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=IF(TRUE, "1a", "2b")', {}, '<Ranges>(A1)=[[\'1a\']]'),
        ('A1', '=IFS(TRUE, "FIRST")', {}, '<Ranges>(A1)=[[\'FIRST\']]'),
        ('A1', '=IFS(FALSE, "FIRST", TRUE, "SECOND")', {},
         '<Ranges>(A1)=[[\'SECOND\']]'),
        ('A1', '=IFS(FALSE, "FIRST", FALSE, "SECOND", TRUE, "THIRD")', {},
         '<Ranges>(A1)=[[\'THIRD\']]'),
        ('A1', '=IFS(FALSE, "FIRST", FALSE, "SECOND", TRUE,)', {},
         '<Ranges>(A1)=[[0]]'),
        ('A1', '=IFS(FALSE, "FIRST", FALSE, "SECOND")', {},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=IF(,FALSE,"PASS")', {}, '<Ranges>(A1)=[[\'PASS\']]'),
        ('A1', '=ROW(4:7)', inp_ranges('4:7'), '<Ranges>(A1)=[[4]]'),
        ('A1', '=ROW(B8:D8:F7:H8 D7:E8)',
         inp_ranges('B8:D8', 'F7:H8', 'D7:E8'), '<Ranges>(A1)=[[7]]'),
        ('A1', '=COLUMN(B8:D8:F7:H8 D7:E7)',
         inp_ranges('B8:D8', 'F7:H8', 'D7:E7'), '<Ranges>(A1)=[[4]]'),
        ('A1', '=COLUMN((B8:D8:F7:H8) D7:E7)',
         inp_ranges('B8:D8', 'F7:H8', 'D7:E7'), '<Ranges>(A1)=[[4]]'),
        ('A1:C3', '=ROW(D1:E1)', inp_ranges('D1:E1'),
         '<Ranges>(A1:C3)=[[1 1 1]\n [1 1 1]\n [1 1 1]]'),
        ('A1:C3', '=ROW(D1:D2)', inp_ranges('D1:D2'),
         '<Ranges>(A1:C3)=[[1 1 1]\n [2 2 2]\n [#N/A #N/A #N/A]]'),
        ('A1:C3', '=ROW(D1:E2)', inp_ranges('D1:E2'),
         '<Ranges>(A1:C3)=[[1 1 1]\n [2 2 2]\n [#N/A #N/A #N/A]]'),
        ('A11', '=ROW(B55:D55:F54:H55 D54:E54)',
         inp_ranges('B55:D55', 'F54:H55', 'D54:E54'), '<Ranges>(A11)=[[54]]'),
        ('A11', '=ROW(B53:D54 C54:E54)', inp_ranges('B53:D54', 'C54:E54'),
         '<Ranges>(A11)=[[54]]'),
        ('A11', '=ROW(L45)', inp_ranges('L45'), '<Ranges>(A11)=[[45]]'),
        ('A11', '=ROW()', {}, '<Ranges>(A11)=[[11]]'),
        ('A1', '=REF()', {}, "<Ranges>(A1)=[[#NAME?]]"),
        ('A1', '=REF', {}, "<Ranges>(A1)=[[#REF!]]"),
        ('A1', '=(-INT(2))', {}, '<Ranges>(A1)=[[-2.0]]'),
        ('A1', '=(1+1)+(1+1)', {}, '<Ranges>(A1)=[[4.0]]'),
        ('A1', '=IFERROR(INDIRECT("aa") * 100,"")', {}, "<Ranges>(A1)=[['']]"),
        ('A1', '=( 1 + 2 + 3)*(4 + 5)^(1/5)', {},
         '<Ranges>(A1)=[[9.311073443492159]]'),
        ('A1', '={1,2;1,2}', {}, '<Ranges>(A1)=[[1]]'),
        ('A1:B2', '={1,2;1,2}', {}, '<Ranges>(A1:B2)=[[1 2]\n [1 2]]'),
        ('A1', '=PI()', {}, '<Ranges>(A1)=[[3.141592653589793]]'),
        ('A1', '=INT(1)%+3', {}, '<Ranges>(A1)=[[3.01]]'),
        ('A1', '=SUM({1, 3; 4, 2})', {}, '<Ranges>(A1)=[[10.0]]'),
        ('A1', '=" "" a"', {}, '<Ranges>(A1)=[[\' " a\']]'),
        ('A1', '=#NULL!', {}, "<Ranges>(A1)=[[#NULL!]]"),
        ('A1', '=1 + 2', {}, '<Ranges>(A1)=[[3.0]]'),
        ('A1', '=AVERAGE(((123 + 4 + AVERAGE({1,2}))))', {},
         '<Ranges>(A1)=[[128.5]]'),
        ('A1', '="a" & "b"""', {}, '<Ranges>(A1)=[[\'ab"\']]'),
        ('A1', '=SUM(B2:B4)',
         {'B2:B4': (sh.EMPTY, sh.EMPTY, sh.EMPTY)}, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=SUM(B2:B4)',
         {'B2:B4': (sh.EMPTY, 1, sh.EMPTY)}, '<Ranges>(A1)=[[1.0]]'),
        ('A1', '=MATCH("*b?u*",{"a",2.1,"ds  bau  dsd",4.1},0)', {},
         '<Ranges>(A1)=[[3]]'),
        ('A1', '=MATCH(4.1,{FALSE,2.1,TRUE,4.1},-1)', {},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=HLOOKUP(-1.1,{-1.1,2.1,3.1,4.1;5,6,7,8},"2.9",0)', {},
         '<Ranges>(A1)=[[5]]'),
        ('A1', '=HLOOKUP(-1.1,{-1.1,2.1,3.1,4.1;5,6,7,8},3,0)', {},
         '<Ranges>(A1)=[[#REF!]]'),
        ('A1', '=MATCH(1.1,{"b",4.1,"a",1.1})', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=MATCH(1.1,{4.1,2.1,3.1,1.1})', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=MATCH(4.1,{4.1,"b","a",1.1})', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=MATCH(4.1,{"b",4.1,"a",1.1})', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH(4.1,{4.1,"b","a",5.1},-1)', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=MATCH(4.1,{"b",4.1,"a",5.1},-1)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH("b",{"b",4.1,"a",1.1})', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=MATCH(3,{-1.1,2.1,3.1,4.1})', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH(-1.1,{"b",4.1,"a",1.1})', {}, '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=MATCH(-1.1,{4.1,2.1,3.1,1.1},-1)', {}, '<Ranges>(A1)=[[4]]'),
        ('A1', '=MATCH(-1.1,{-1.1,2.1,3.1,4.1})', {}, '<Ranges>(A1)=[[1]]'),
        ('A1', '=MATCH(2.1,{4.1,2.1,3.1,1.1})', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH(2.1,{4.1,2.1,3.1,1.1},-1)', {}, '<Ranges>(A1)=[[2]]'),
        ('A1', '=MATCH(2,{4.1,2.1,3.1,1.1},-1)', {}, '<Ranges>(A1)=[[3]]'),
        ('A1', '=LOOKUP(2.1,{4.1,2.1,3.1,1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'ML\']]'),
        ('A1', '=LOOKUP(A2,B1:B4,{"L","ML","MR","R"})',
         {'A2': [[sh.EMPTY]], 'B1:B4': (0, sh.EMPTY, 2, 3)},
         '<Ranges>(A1)=[[\'L\']]'),
        ('A1', '=LOOKUP(A2,B1:B4,{"L","ML","MR","R"})',
         {'A2': [[sh.EMPTY]], 'B1:B4': (1, sh.EMPTY, 2, 3)},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=LOOKUP(1,B1:B4,{"L","ML","MR","R"})',
         {'B1:B4': (1, sh.EMPTY, 2, 3)}, '<Ranges>(A1)=[[\'L\']]'),
        ('A1', '=LOOKUP(1,{1,2,3,4},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'L\']]'),
        ('A1', '=LOOKUP(1,{1,"",3,4},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'L\']]'),
        ('A1', '=LOOKUP("b",{"b",4.1,"a",1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'MR\']]'),
        ('A1', '=LOOKUP(TRUE,{TRUE,4.1,FALSE,1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'MR\']]'),
        ('A1', '=LOOKUP(4.1,{"b",4.1,"a",1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'ML\']]'),
        ('A1', '=LOOKUP(2,{"b",4.1,"a",1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[#N/A]]'),
        ('A1', '=LOOKUP(4.1,{4.1,2.1,3.1,1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'R\']]'),
        ('A1', '=LOOKUP(4,{4.1,2.1,3.1,1.1},{"L","ML","MR","R"})', {},
         '<Ranges>(A1)=[[\'R\']]'),
        ('A1:D1', '=IF({0,-0.2,0},2,{1})', {},
         '<Ranges>(A1:D1)=[[1 2 1 #N/A]]'),
        ('A1', '=HEX2DEC(9999999999)', {}, '<Ranges>(A1)=[[-439804651111]]'),
        ('A1', '=HEX2BIN(9999999999)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=HEX2BIN("FFFFFFFE00")', {}, '<Ranges>(A1)=[[\'1000000000\']]'),
        ('A1', '=HEX2BIN("1ff")', {}, '<Ranges>(A1)=[[\'111111111\']]'),
        ('A1', '=HEX2OCT("FF0000000")', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=HEX2OCT("FFE0000000")', {}, '<Ranges>(A1)=[[\'4000000000\']]'),
        ('A1', '=HEX2OCT("1FFFFFFF")', {}, '<Ranges>(A1)=[[\'3777777777\']]'),
        ('A1', '=DEC2HEX(-439804651111)', {},
         '<Ranges>(A1)=[[\'9999999999\']]'),
        ('A1', '=DEC2BIN(TRUE)', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=DEC2BIN(#DIV/0!)', {}, '<Ranges>(A1)=[[#DIV/0!]]'),
        ('A1', '=DEC2BIN("a")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=DEC2BIN(4,6)', {}, '<Ranges>(A1)=[[\'000100\']]'),
        ('A1', '=DEC2BIN(4,-2)', {}, '<Ranges>(A1)=[[#NUM!]]'),
        ('A1', '=DEC2BIN(4,"a")', {}, '<Ranges>(A1)=[[#VALUE!]]'),
        ('A1', '=ROUND(2.5,0)', {}, '<Ranges>(A1)=[[3.0]]'),
        ('A1', '=ROUND(2.35,1)', {}, '<Ranges>(A1)=[[2.4]]'),
        ('A1', '=TRUE()', {}, '<Ranges>(A1)=[[True]]'),
        ('A1', '=FALSE()', {}, '<Ranges>(A1)=[[False]]'),
        ('A1', '=CONCAT("con", "cat", "enate")', {},
         '<Ranges>(A1)=[[\'concatenate\']]'),
        ('A1', '=CONCAT(A2:E2)', {
            'A2:E2': [["h", "e", "l", "l", 0.15]]
        }, '<Ranges>(A1)=[[\'hell0.15\']]'),
        ('A1', '=CONCAT(A2:E2, A3:E3, "curl")', {
            'A2:E2': [["h", "e", "l", "l", "o"]],
            'A3:E3': [["h", "e", "l", "l", "o"]]
        }, '<Ranges>(A1)=[[\'hellohellocurl\']]'),
        ('A1', '=SUMSQ(B2:B4)',
         {'B2:B4': (sh.EMPTY, sh.EMPTY, sh.EMPTY)}, '<Ranges>(A1)=[[0.0]]'),
        ('A1', '=SUMSQ(B2:B4)',
         {'B2:B4': (sh.EMPTY, 2, 3)}, '<Ranges>(A1)=[[13.0]]'),
        # ('A1:D1', '=IF({0,-0.2,0},{2,3},{1})', {},
        #  '<Ranges>(A1:D1)=[[1 2 1 #N/A]]'),
        # ('A1:D1', '=IF({0,-2,0},{2,3},{1,4})', {},
        #  '<Ranges>(A1:D1)=[[1 2 #N/A #N/A]]')
    ])
    def test_output(self, case):
        self.maxDiff = None
        reference, formula, inputs, result = case
        dsp = sh.Dispatcher()
        cell = Cell(reference, formula).compile()
        assert cell.add(dsp)
        output = str(dsp(inputs)[cell.output])
        self.assertEqual(
            result, output,
            'Formula({}): {} != {}'.format(formula, result, output)
        )

    @ddt.idata([
        ('A1', '=ODDFYIELD("2008-11-11","2021-3-1","2008-10-15","2009-3-1",0.07'
               '85,0.0625,100,2,1)', {}, 26.628136487894576),
        ('A1', '=ODDFPRICE("2008-11-11","2021-3-1","2007-10-15","2014-3-1",0.07'
               '85,0.0625,100,4,0)', {}, 106.06073620223454),
        ('A1', '=ODDFPRICE("2019-2-1","2022-2-15","2018-12-1","2019-2-15",0.05,'
               '0.06,100,2,0)', {}, 97.26007079112537),
        ('A1', '=ODDFPRICE("2008-11-11","2021-3-1","2008-10-15","2009-3-1",0.07'
               '85,0.0625,100,2,1)', {}, 113.59771747407883),
        ('A1', '=FORECAST.ETS(18,{1,2,3,2,1,3,2,1},{1,3,5,7,9,11,13,14},1,0)',
         {}, 1.839980682),
        ('A1', '=FORECAST.ETS(216,{1,2,3,2,1,2,3,2,1,1},{1,6,10,14,18,22,26,'
               '30,31,38},1,0)', {}, -3.764168605),
        ('A1', '=FORECAST.ETS(216,{1,2,3,2,1,2,3,2,1,1},{1,6,10,14,18,22,26,'
               '30,31,38},4,0)', {}, -2.95439433),
        ('A1', '=FORECAST.ETS(16,{1,2,3,1,2,3},{-0.1,2,4,6,8,10})',
         {}, 2.784629391),
        ('A1', '=FORECAST.ETS(18,{1,2,3,2,1,3,2,1},{1,3,5,7,9,11,13,15},8,0)',
         {}, 1.67465843),
        ('A1', '=FORECAST.ETS(18,{3,1,2,3,2,1,3,2,1},{7,5,6,7,8,1,3,4,9},2)',
         {}, 1.989369378),
        ('A1', '=FORECAST.ETS(18,{3,1,2,3,2,1,3,2,1},{7,5,6,7,8,1,3,4,9},3)',
         {}, 2.178149198),
        ('A1', '=FORECAST.ETS(18,{3,1,2,3,2,1,3,2,1},{7,5,6,7,8,1,3,4,9},4)',
         {}, 2.0),
        ('A1', '=FORECAST.ETS(18,{3,3,1,2,3,2,1,3,2,1},{3,7,5,6,7,8,1,3,4,9})',
         {}, 1.791180882),
        ('A1', '=FORECAST.ETS(18,{1,2,3,2,1,3,2,1},{1,3,5,7,9,11,13,15},8,0)',
         {}, 1.67465843),
        ('A1', '=FORECAST.ETS(18.5,{3,3,1,2,3,2,1,3,2,1},{3,7,5,6,7,8,1,'
               '3,4,9},4)', {}, 2.5),
        ('A1', '=FORECAST.ETS(18,{1,2,3,2,1,3,2,1},{1,3,5,7,9,11,13,15},4)',
         {}, 1.500116781),
        ('A1', '=FORECAST.ETS(18,{1,2,3,2,1,3,2,1},{1,3,5,7,9,11,13,14},4,0)',
         {}, 1.839980682),
    ])
    def test_result(self, case):
        reference, formula, inputs, result = case
        dsp = sh.Dispatcher()
        cell = Cell(reference, formula).compile()
        assert cell.add(dsp)
        output = dsp(inputs)[cell.output].value.item()
        self.assertAlmostEqual(
            result, output, 0,
            'Formula({}): {} != {}'.format(formula, result, output)
        )

    @ddt.idata([
        ('A1', '=F.TEST({0.1,0.9;3.2,0.7},{0.1,0.2;0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[0\.0028587453134692\d*\]\]"),
        ('A1', '=F.TEST({0.1,0.9;3.2,0.7},{0.1,0.2,0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[0\.0028587453134692\d*\]\]"),
        ('A1', '=F.TEST({0.1,0.9,3.2,0.7},{0.1,0.2;0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[0\.0028587453134692\d*\]\]"),
        ('A1', '=F.TEST({0.1,0.9,3.2,0.7},{0.1,0.2,0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[0\.0028587453134692\d*\]\]"),
        ('A1', '=NEGBINOM.DIST(5,3,0.4,1)', {},
         "<Ranges>\(A1\)=\[\[0\.68460544\d*\]\]"),
        ('A1', '=Z.TEST({0,-0.1,10,3.2},-0.9)', {},
         "<Ranges>\(A1\)=\[\[0\.039007077373921\d*\]\]"),
        ('A1', '=BINOM.DIST(5,10,0.2,FALSE)', {},
         "<Ranges>\(A1\)=\[\[0\.02642411519999\d*\]\]"),
        ('A1', '=CHISQ.TEST({0.1,0.9,3.2,0.7},{0.1,0.2;0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[9\.791450352809\d*e-07\]\]"),
        ('A1', '=CHISQ.TEST({0.1,0.9;3.2,0.7},{0.1,0.2,0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[2\.9987440798444\d*e-08\]\]"),
        ('A1', '=CHISQ.TEST({0.1,0.9;3.2,0.7},{0.1,0.2;0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[2\.9987440798444\d*e-08\]\]"),
        ('A1', '=CHISQ.TEST({0.1,0.9,3.2,0.7},{0.1,0.2,0.3,0.4})', {},
         "<Ranges>\(A1\)=\[\[9\.791450352809\d*e-07\]\]"),
        ('A1', '=CHISQ.TEST({0.7,""},{4,""})', {},
         "<Ranges>\(A1\)=\[\[0\.09894293606729\d*\]\]"),
        ('A1', '=CHISQ.TEST({0.7,1},{4,""})', {},
         "<Ranges>\(A1\)=\[\[0\.09894293606729\d*\]\]"),
        ('A1', '=CONFIDENCE.T(0.05,1,50)', {},
         "<Ranges>\(A1\)=\[\[0\.28419685549572\d*\]\]"),
        ('A1:D5', '=LOGEST({2;3;4;21;2},{1,2,1;2,2,2;1,5,3;6,2,4;2,4,1},0,1)',
         {},
         "<Ranges>\(A1:D5\)=\[\[1\.5664334466775\d* 0\.9610027927469\d* "
         "1\.2402383938034\d* 1\.0\]\n "
         "\[0\.05066588583718\d* 0\.03537061734839\d* 0\.08272771053197\d* "
         "#N/A\]\n \[0\.9972327485569\d* 0\.13595424611813\d* #N/A #N/A\]\n "
         "\[240\.24626820088\d* 2 #N/A #N/A\]\n "
         "\[13\.32181680404\d* 0\.03696711407509\d* #N/A #N/A\]\]"),
        ('A1:C5', '=LINEST({2;3;4;21;2},{2,1;2,2;5,3;2,4;4,1},0,1)', {},
         "<Ranges>\(A1:C5\)=\[\[5\.375451263537\d* -1\.8375451263537\d* "
         "0\.0\]\n \[0\.9772930582193\d* 1\.2778559569377\d* #N/A\]\n "
         "\[0\.8919709363432\d* 4\.131415260871\d* #N/A\]\n "
         "\[12\.38515228426\d* 3 #N/A\]\n "
         "\[422\.79422382671\d* 51\.20577617328\d* #N/A\]\]"),
        ('A1:D5', '=LOGEST({2;3;4;21;2},{1,2,1;2,2,2;1,5,3;6,2,4;2,4,1},1,1)',
         {},
         "<Ranges>\(A1:D5\)=\[\[1\.5659258509260\d* 0\.9654962343041\d* "
         "1\.243497877058\d*\n  0\.9790291254962\d*\]\n "
         "\[0\.31095496747040\d* 0\.08119886017046\d* 0\.08470544656430\d*\n  "
         "0\.11682066611763\d*\]\n "
         "\[0\.9902997881386\d* 0\.1918233037998\d* #N/A #N/A\]\n "
         "\[25\.522632966471\d* 1 #N/A #N/A\]\n "
         "\[3\.75654157465\d* 0\.0367961798806\d* #N/A #N/A\]\]"),
        ('A1:C5', '=LINEST({2;3;4;21;2},{2,1;2,2;5,3;2,4;4,1},1,1)', {},
         "<Ranges>\(A1:C5\)=\[\[5\.235294117647\d* -2\.0000000000000\d* "
         "0\.882352941176\d*\]\n "
         "\[7\.194288391990\d* 1\.7822655773580\d* 1\.9331384382535\d*\]\n "
         "\[0\.81120531422078\d* 5\.041008302500\d* #N/A\]\n "
         "\[2\.86450617283\d* 2 #N/A\]\n "
         "\[218\.37647058823\d* 50\.823529411764\d* #N/A\]\]"),
        ('A1:B1', '=LINEST(A2:C2,A3:C3,0,0)',
         {'A2:C2': [[1, 2, 3]], 'A3:C3': [[1, 2, 3]]},
         "<Ranges>\(A1:B1\)=\[\[1\.0+\d?\d? 0\.0+\d?\d?\]\]"),
        ('A1:A5', '=TREND({2;3;4;21;2},{2,1;2,2;5,3;2,4;4,1})', {},
         "<Ranges>\(A1:A5\)=\[\[2\.11764705882355\d*\]\n"
         " \[7\.352941176470\d*\]\n"
         " \[6\.588235294117\d*\]\n"
         " \[17\.8235294117\d*\]\n"
         " \[-1\.882352941176\d*\]\]"),
        ('A1', '=NORMSDIST(2,TRUE)', {},
         '<Ranges>\(A1\)=\[\[0\.977249868051820\d*\]\]'),
        ('A1', '=XIRR({-10,1,5,0.0001,4.5},{1,20,4,4,5},"26/08/1987")',
         {}, '<Ranges>\(A1\)=\[\[38\.321500577844\d*\]\]'),
        ('A1', '=XIRR({-10000,2750,3250,4250},'
               '      {"39448",39508,39859,"30/10/2008"}, 0)',
         {}, '<Ranges>\(A1\)=\[\[0\.03379137764398\d*\]\]'),
        ('A1', '=XIRR({-10,1,1,2,7},{3,6,7,4,9})', {},
         '<Ranges>\(A1\)=\[\[1937\.5566679300\d*\]\]'),
        ('A1', '=XIRR({-10000,2750,3250,4250},{39448,39508,39859,39751}, 1)',
         {}, '<Ranges>\(A1\)=\[\[0\.03379137764432\d*\]\]'),
        ('A1', '=IRR({-7,2,-1,4,-3;7,2,-1,4,4},1)', {},
         '<Ranges>\(A1\)=\[\[0\.19086464188385\d*\]\]'),
        ('A1', '=FV(0.06,-10,-200,-500,1)', {},
         '<Ranges>\(A1\)=\[\[-1281\.1410664423\d*\]\]'),
        ('A1', '=FV(0.06,-10,200,-500,1)', {},
         '<Ranges>\(A1\)=\[\[1839\.5358433574\d*\]\]'),
        ('A1', '=FV(0.06/12, 10, -200, -500, 1)', {},
         '<Ranges>\(A1\)=\[\[2581\.4033740601\d*\]\]'),
        ('A1', '=RATE(1500,-200,8000)', {},
         '<Ranges>\(A1\)=\[\[0\.025000000047328\d*\]\]'),
        ('A1', '=RATE(43,-200,8000)', {},
         '<Ranges>\(A1\)=\[\[0\.003331551825239\d*\]\]'),
        ('A1', '=PV(100,3,22.4)', {},
         '<Ranges>\(A1\)=\[\[-0\.22399978258780\d*\]\]'),
        ('A1', '=PV(100,3.1,22)', {},
         '<Ranges>\(A1\)=\[\[-0\.21999986540577\d*\]\]'),
        ('A1', '=PV(100,3,22)', {},
         '<Ranges>\(A1\)=\[\[-0\.21999978647016\d*\]\]'),
        ('A1', '=XNPV(0.02,{-10000,2750,3250,4250},{39448,39508,39859,39751})',
         {}, '<Ranges>\(A1\)=\[\[100\.10102845727\d*\]\]'),
        ('A1', '=NPV(-0.1, {-0.1,2,0,4,5})', {},
         '<Ranges>\(A1\)=\[\[16\.922200206608\d*\]\]'),
        ('A1', '=CUMIPMT(0.09/12,30*12,125000,13,24,0)', {},
         '<Ranges>\(A1\)=\[\[-11135\.232130750\d*\]\]'),
        ('A1', '=CUMIPMT(3,5.5,125000,3,5,1)', {},
         '<Ranges>\(A1\)=\[\[-265999\.0229604\d*\]\]'),
        ('A1', '=CUMIPMT(3,5.5,125000,1,5,1)', {},
         '<Ranges>\(A1\)=\[\[-359611\.6267708\d*\]\]'),
        ('A1', '=RATE(4*12, -200, 8000)', {},
         '<Ranges>\(A1\)=\[\[0\.00770147248823\d*\]\]'),
        ('A1', '=PV(0.08/12, 12*20, 500000,,0)', {},
         '<Ranges>\(A1\)=\[\[-59777145\.85118\d*\]\]'),
        ('A1', '=NPER(0.12/12, -100, -1000, 10000, 1)', {},
         '<Ranges>\(A1\)=\[\[59\.67386567429\d*\]\]'),
        ('A1', '=NPER(0.12/12, -100, -1000)', {},
         '<Ranges>\(A1\)=\[\[-9\.578594039813\d*\]\]'),
        ('A1', '=IPMT(0.1/12, "01/01/1900", 3*12, 8000)', {},
         '<Ranges>\(A1\)=\[\[-66\.66666666666\d*\]\]'),
        ('A1', '=PPMT(0.1/12, 1, 2*12, 2000)', {},
         '<Ranges>\(A1\)=\[\[-75\.62318600836\d*\]\]'),
        ('A1', '=PMT("33","3","33","10000","1")', {},
         '<Ranges>\(A1\)=\[\[-32\.2771768657\d*\]\]'),
        ('A1', '=PMT(0.08/12,10,10000,10000,1)', {},
         '<Ranges>\(A1\)=\[\[-1994\.1034887930\d*\]\]'),
        ('A1', '=PMT(0.08/12,10,10000,,1)', {},
         '<Ranges>\(A1\)=\[\[-1030\.1643271779\d*\]\]'),
        ('A1', '=PMT(0.08/12,10,10000)', {},
         '<Ranges>\(A1\)=\[\[-1037\.0320893591\d*\]\]'),
    ])
    def test_output_regex(self, case):
        reference, formula, inputs, regex = case
        dsp = sh.Dispatcher()
        cell = Cell(reference, formula).compile()
        assert cell.add(dsp)
        output = str(dsp(inputs)[cell.output])
        self.assertRegex(
            output, regex,
            'Formula({}): {} != {}'.format(formula, regex, output)
        )

    @ddt.idata([
        ('A1:D1', '=IF({0,-0.2,0},{2,3},{1})', {}),  # BroadcastError
        ('A1:D1', '=IF({0,-2,0},{2,3},{1,4})', {}),  # BroadcastError
    ])
    def test_invalid(self, case):
        reference, formula, inputs = case
        with self.assertRaises(sh.DispatcherError):
            dsp = sh.Dispatcher(raises=True)
            cell = Cell(reference, formula).compile()
            assert cell.add(dsp)
            dsp(inputs)

    @ddt.idata([
        ('A1', '=NOW()', 1),
        ('A1', '=RAND()', 0),
        # ('A1', '=TODAY()'),
    ])
    def test_impure(self, case):
        reference, formula, dt = case
        dsp = sh.Dispatcher()
        cell = Cell(reference, formula).compile()
        assert cell.add(dsp)
        out = str(dsp()[cell.output])
        time.sleep(dt)
        self.assertNotEqual(out, str(dsp()[cell.output]))
