import os
from typing import Type, TypeVar
from newsflash.widgets.base import Widget, widget_registry
from newsflash.widgets import Notifications

from django.http import HttpRequest
from django.core.management import execute_from_command_line
from django.core.wsgi import get_wsgi_application
from django.core.handlers.wsgi import WSGIHandler
from django.template.loader import render_to_string


T = TypeVar("T")
W = TypeVar("W", bound=Widget)


class App:
    navbar: bool = False

    routes: dict[str, str] = {}

    def __init__(self, routes: dict[str, str] | None = None) -> None:
        self.widget_registry = {}
        if routes is not None:
            self.routes = routes

    def widget(self, cls: Type[W]) -> Type[W]:
        _cls = cls()
        self.register_widget(_cls)
        return cls

    def register_widget(self, widget: Widget) -> None:
        assert widget.id is not None
        widget_registry[widget.id] = widget

    def query_one(self, id: str, type: Type[T]) -> T | None:
        notifications = Notifications()
        if id == "notifications" and isinstance(notifications, type):
            return notifications

        if id in widget_registry:
            widget = widget_registry[id]
            if isinstance(widget, type):
                return widget

        return None

    def render(self, request: HttpRequest) -> str:
        rendered_widgets = {
            widget_id: widget.render(request)
            for widget_id, widget in widget_registry.items()
        }

        return render_to_string(
            template_name=self.routes["/"],
            context=rendered_widgets,
            request=request,
        )

    def run(self):
        from newsflash.web.app.urls import set_urlpatterns

        os.environ.setdefault("DJANGO_SETTINGS_MODULE", "newsflash.web.settings")
        set_urlpatterns(self)
        execute_from_command_line(["manage.py", "runserver"])

    def get_wsgi_application(self) -> WSGIHandler:
        from newsflash.web.app.urls import set_urlpatterns

        os.environ.setdefault("DJANGO_SETTINGS_MODULE", "newsflash.web.settings")
        set_urlpatterns(self)
        application = get_wsgi_application()
        return application
