import sys
import os
from pathlib import Path

sys.path.append(str(Path(__file__).resolve().parents[1]))

import pytest

from telegram_markdown import render_markdown_to_telegram

os.environ.setdefault("BOT_TOKEN", "dummy")

from bot import _prepare_model_payload, _prepare_model_payload_variants  # noqa: E402


@pytest.mark.parametrize(
    "source, expected",
    [
        ("**bold** text", "*bold* text"),
        ("__italic__", "*italic*"),
        ("`code_snippet`", "`code_snippet`"),
        ("普通文本 (test)", "普通文本 \\(test\\)"),
        ("raw *literal* star", "raw _literal_ star"),
        ("带感叹号! 以及加号+", "带感叹号\\! 以及加号\\+"),
        ("multiline\nline2", "multiline\nline2"),
    ],
)
def test_render_inline_markdown(source: str, expected: str) -> None:
    assert render_markdown_to_telegram(source) == expected


def test_render_code_block_with_language() -> None:
    markdown = "```python\nprint('hello')\n```"
    rendered = render_markdown_to_telegram(markdown)
    assert rendered == "```python\nprint('hello')\n```"


def test_render_bullet_list() -> None:
    markdown = "- item 1\n- item 2"
    rendered = render_markdown_to_telegram(markdown)
    assert rendered.splitlines() == ["- item 1", "- item 2"]


def test_render_ordered_list() -> None:
    markdown = "1. first\n2. second"
    rendered = render_markdown_to_telegram(markdown)
    assert rendered.splitlines() == ["1. first", "2. second"]


def test_render_heading_and_paragraph_spacing() -> None:
    markdown = "# Title\n\nParagraph line."
    rendered = render_markdown_to_telegram(markdown)
    assert rendered == "*Title*\n\nParagraph line\\."


def test_render_blockquote() -> None:
    markdown = "> quote"
    assert render_markdown_to_telegram(markdown) == "> quote"


def test_render_link() -> None:
    markdown = "[link](https://example.com/path?a=1&b=2)"
    assert render_markdown_to_telegram(markdown) == "[link](https://example.com/path?a=1&b=2)"


@pytest.mark.parametrize(
    "text, expect_primary, expect_change",
    [
        ("A. 采用方案", "A\\. 采用方案", False),
        ("示例.\n第二行", "示例\\.\n第二行", False),
        ("- [ ] 待办事项", "- \\[ \\] 待办事项", False),
        ("- [ ] 待办事项\n- [ ] 第二条", "- \\[ \\] 待办事项\n- \\[ \\] 第二条", False),
        ("任务编码：/TASK_0023", None, False),
        ("纯文本，无需转义", None, False),
        ("带内联代码 `A.` 示例", None, False),
        ("```\nA. print\n```", None, False),
        ("段落结束. 继续下一句", "段落结束\\. 继续下一句", False),
        ("- [ ] 包含句点. 并继续", "- \\[ \\] 包含句点\\. 并继续", False),
    ],
)
def test_prepare_model_payload_variants(text: str, expect_primary: str | None, expect_change: bool) -> None:
    primary, fallback = _prepare_model_payload_variants(text)
    strict = _prepare_model_payload(text)
    if expect_primary is not None:
        assert primary == expect_primary
    else:
        assert primary == strict
    if expect_change:
        assert fallback is not None
        assert fallback != primary
        assert fallback == strict
    else:
        if fallback is None:
            assert primary == strict
        else:
            assert fallback == primary == strict
