import { Component } from "./component";
import { FileBase, IResolver } from "./file";
import { Project } from "./project";
/**
 * Supported AI coding assistants and their instruction file locations.
 */
export declare enum AiAgent {
    /**
     * GitHub Copilot - .github/copilot-instructions.md
     */
    GITHUB_COPILOT = "GitHub Copilot",
    /**
     * Cursor IDE - .cursor/rules/project.md
     */
    CURSOR = "Cursor",
    /**
     * Claude Code - CLAUDE.md
     */
    CLAUDE = "Claude",
    /**
     * Amazon Q - .amazonq/rules/project.md
     */
    AMAZON_Q = "Amazon Q",
    /**
     * Kiro - .kiro/steering/project.md
     */
    KIRO = "Kiro"
}
/**
 * Options for configuring AI tool instruction files.
 */
export interface AiInstructionsOptions {
    /**
     * Which AI agents to generate instruction files for.
     *
     * @default - All agents: [AiAgent.GITHUB_COPILOT, AiAgent.CURSOR, AiAgent.CLAUDE, AiAgent.AMAZON_Q]
     */
    readonly agents?: AiAgent[];
    /**
     * General instructions applicable to all agents.
     *
     * @default - no agent specific instructions
     */
    readonly instructions?: string[];
    /**
     * Per-agent custom instructions. Allows different instructions for different AI tools.
     *
     * @default - no agent specific instructions
     * @example
     * {
     *   [AiAgent.GITHUB_COPILOT]: {
     *     instructions: ["Use descriptive commit messages."]
     *   },
     *   [AiAgent.CURSOR]: {
     *     instructions: ["Prefer functional patterns.", "Always add tests."]
     *   }
     * }
     */
    readonly agentSpecificInstructions?: Record<string, string[]>;
    /**
     * Include default instructions for projen and general best practices.
     *
     * Default instructions will only be included for agents provided in the `agents` option.
     * If `agents` is not provided, default instructions will be included for all agents.
     *
     * @default true
     */
    readonly includeDefaultInstructions?: boolean;
}
/**
 * Generates instruction files for AI coding assistants with projen-specific guidance.
 *
 * This component creates configuration files that help AI tools like GitHub Copilot,
 * Cursor IDE, Claude Code, and Amazon Q understand that the project is managed by projen
 * and should follow projen conventions.
 *
 * @example
 * const project = new TypeScriptProject({
 *   name: "my-project",
 *   defaultReleaseBranch: "main",
 * });
 *
 * // Basic usage - generates files for all supported AI agents
 * new AiInstructions(project);
 *
 * // Custom usage - specify which agents and add custom instructions
 * new AiInstructions(project, {
 *   agents: [AiAgent.GITHUB_COPILOT, AiAgent.CURSOR],
 *   agentSpecificInstructions: {
 *     [AiAgent.GITHUB_COPILOT]: ["Always use descriptive commit messages."],
 *   },
 * });
 *
 * // Add more instructions after instantiation
 * const ai = new AiInstructions(project);
 * ai.addInstructions("Use functional programming patterns.");
 * ai.addInstructions("Always write comprehensive tests.");
 */
export declare class AiInstructions extends Component {
    /**
     * Returns projen-specific instructions for AI agents.
     */
    static projen(project: Project): string;
    /**
     * Returns development best practices instructions for AI agents.
     */
    static bestPractices(project: Project): string;
    private readonly agents;
    private readonly files;
    constructor(project: Project, options?: AiInstructionsOptions);
    /**
     * Create or return the instructions file.
     */
    private ensureInstructionsFile;
    /**
     * Adds instructions that will be included for all selected AI agents.
     *
     * @param instructions The instructions to add.
     * @example
     * aiInstructions.addInstructions("Always use TypeScript strict mode.");
     * aiInstructions.addInstructions("Prefer functional programming.", "Avoid mutations.");
     */
    addInstructions(...instructions: string[]): void;
    /**
     * Add instructions for a specific AI agent.
     *
     * This can also be used to add instructions for an AI agent that was previously not enabled.
     *
     * @param agent The AI agent to add instructions for
     * @param instructions The instruction(s) to add
     * @example
     * aiInstructions.addAgentSpecificInstructions(AiAgent.GITHUB_COPILOT, "Use descriptive commit messages.");
     */
    addAgentSpecificInstructions(agent: AiAgent, ...instructions: string[]): void;
    /**
     * Get the file path for a given AI agent.
     */
    private getAgentFilePath;
}
export declare class AiInstructionsFile extends FileBase {
    private readonly instructions;
    /**
     * Adds instructions to the instruction file.
     */
    addInstructions(...instructions: string[]): void;
    protected synthesizeContent(resolver: IResolver): string | undefined;
}
