"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const util_1 = require("../util");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_22_X;
        const entrypoint = (0, util_1.normalizePersistedPath)(options.entrypoint);
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(entrypoint);
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path.posix.join(path.dirname(entrypoint), path.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? (0, case_1.pascal)(path.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: runtime.defaultExternals,
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path.join(project.outdir, bundle.outfile);
        const constructAbs = path.join(project.outdir, constructFile);
        const relativeOutfile = path.relative(path.dirname(constructAbs), path.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${(0, internal_1.convertToPosixPath)(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${(0, internal_1.convertToPosixPath)(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${(0, internal_1.convertToPosixPath)(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.98.16" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget, 
    /**
     * Options for this runtime.
     */
    options) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
        this.defaultExternals = options?.defaultExternals ?? ["@aws-sdk/*"];
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.98.16" };
/**
 * Node.js 10.x
 * @deprecated Node.js 10 runtime has been deprecated on Jul 30, 2021
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 12.x
 * @deprecated Node.js 12 runtime has been deprecated on Mar 31, 2023
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 14.x
 * @deprecated Node.js 14 runtime has been deprecated on Dec 4, 2023
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 16.x
 * @deprecated Node.js 16 runtime has been deprecated on Jun 12, 2024
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 18.x
 *
 * @deprecated: Node.js 18 runtime has been deprecated on Sep 1, 2025
 */
LambdaRuntime.NODEJS_18_X = new LambdaRuntime("nodejs18.x", "node18");
/**
 * Node.js 20.x
 */
LambdaRuntime.NODEJS_20_X = new LambdaRuntime("nodejs20.x", "node20");
/**
 * Node.js 22.x
 */
LambdaRuntime.NODEJS_22_X = new LambdaRuntime("nodejs22.x", "node22");
//# sourceMappingURL=data:application/json;base64,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