"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cdk8sDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Manages dependencies on the CDK8s.
 */
class Cdk8sDeps extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.dependencyType = options.dependencyType;
        this._packageNames = this.packageNames();
        const framework = determineFrameworkVersion(options);
        this.cdk8sVersion = framework.range;
        this.cdk8sMajorVersion = framework.major;
        this.cdk8sMinimumVersion = framework.minimum;
        // constructs library
        this.addConstructsDependency(options.constructsVersion, options.constructsVersionPinning);
        this.addFrameworkDependency(options.cdk8sVersion, options.cdk8sVersionPinning);
        if (options.cdk8sCliDependency) {
            this.addClientDependency(options.cdk8sCliVersion, options.cdk8sCliVersionPinning);
        }
        if (options.cdk8sPlus ?? true) {
            const k8MinorVersion = options.k8sMinorVersion ?? 22;
            this.addCdk8sPlusDependency(k8MinorVersion, options.cdk8sPlusVersion, 
            // (project instanceof Cdk8sPythonApp) ? true : false,
            options.cdk8sPlusVersionPinning);
        }
    }
    getVersionRequirement(requestedVersion, v1Version, v2Version, pinning) {
        const defaultVersion = this.cdk8sMajorVersion === 1 ? v1Version : v2Version;
        const version = `${requestedVersion ?? defaultVersion}`;
        const versionRequirement = pinning ? `${version}` : `^${version}`;
        return versionRequirement;
    }
    addCdk8sPlusDependency(k8MinorVersion, requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.0.0-beta.222", "2.0.0-rc.26", pinning);
        const cdk8sPlusMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sPlusMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s-plus version '${versionRequirement}'`);
        }
        const cdk8sPlusLib = this._packageNames.cdk8sPlus.concat("-", k8MinorVersion.toString());
        this.project.deps.addDependency(`${cdk8sPlusLib}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    addClientDependency(requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.1.34", "2.0.28", pinning);
        const cdk8sCliMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sCliMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s version '${versionRequirement}'`);
        }
        this.project.deps.addDependency(`${this._packageNames.cdk8sClient}@${versionRequirement}`, dependencies_1.DependencyType.BUILD);
        return versionRequirement;
    }
    addFrameworkDependency(requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.6.33", "2.3.33", pinning);
        const cdk8sMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s version '${versionRequirement}'`);
        }
        this.project.deps.addDependency(`${this._packageNames.cdk8s}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    addConstructsDependency(requestedVersion, pinning) {
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const versionRequirement = this.getVersionRequirement(requestedVersion, "3.4.39", "10.1.42", pinning);
        const constructsMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdk8sMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error("AWS CDK 1.x requires constructs 3.x");
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error("AWS CDK 2.x requires constructs 10.x");
                }
                break;
        }
        this.project.deps.addDependency(`${this._packageNames.constructs}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
}
exports.Cdk8sDeps = Cdk8sDeps;
_a = JSII_RTTI_SYMBOL_1;
Cdk8sDeps[_a] = { fqn: "projen.cdk8s.Cdk8sDeps", version: "0.98.16" };
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdk8sVersion);
    if (!ver) {
        throw new Error(`"cdk8sVersion" cannot be parsed as a semver version: ${options.cdk8sVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdk8sVersionPinning
            ? options.cdk8sVersion
            : `^${options.cdk8sVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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