"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoQueue = exports.MergeMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const gh = require("../github");
/**
 * The merge method used to add the PR to the merge queue
 *
 * Behavior can be further configured in repository settings.
 */
var MergeMethod;
(function (MergeMethod) {
    MergeMethod["SQUASH"] = "squash";
    MergeMethod["MERGE"] = "merge";
    MergeMethod["REBASE"] = "rebase";
})(MergeMethod || (exports.MergeMethod = MergeMethod = {}));
/**
 * Automatically add pull requests to the merge queue
 * PRs will be merged once they pass required checks.
 */
class AutoQueue extends component_1.Component {
    constructor(scope, options = {}) {
        super(scope);
        const workflowEngine = gh.GitHub.of(this.project);
        if (!workflowEngine) {
            throw new Error(`Cannot add ${new.target.name} to project without GitHub enabled. Please enable GitHub for this project.`);
        }
        const labels = options.labels ?? [];
        const usernames = options.allowedUsernames ?? [];
        const conditions = [];
        if (labels.length > 0) {
            conditions.push("(" +
                labels
                    .map((l) => `contains(github.event.pull_request.labels.*.name, '${l}')`)
                    .join(" || ") +
                ")");
        }
        if (usernames.length > 0) {
            conditions.push("(" +
                usernames
                    .map((u) => `github.event.pull_request.user.login == '${u}'`)
                    .join(" || ") +
                ")");
        }
        let needsEditedEvent = false;
        if (options.targetBranches) {
            // Branch conditions, based off the 'opened' or 'edited' events.
            //
            // The current workflow will only run if the target branch is one of the intended
            // ones, so we only need to check if the event type is correct.
            needsEditedEvent = true;
            const isOpened = `github.event.action == 'opened'`;
            const isBranchChanged = `(github.event.action == 'edited' && github.event.changes.base)`;
            conditions.push(`(${isOpened} || ${isBranchChanged})`);
        }
        const credentials = options.projenCredentials ?? workflowEngine.projenCredentials;
        const mergeMethod = options.mergeMethod ?? MergeMethod.SQUASH;
        const autoQueueJob = {
            name: "Set AutoQueue on PR #${{ github.event.number }}",
            runsOn: options.runsOn ?? ["ubuntu-latest"],
            permissions: {
                pullRequests: gh.workflows.JobPermission.WRITE,
                contents: gh.workflows.JobPermission.WRITE,
            },
            if: conditions.length ? conditions.join(" && ") : undefined,
            steps: [
                ...credentials.setupSteps,
                {
                    uses: "peter-evans/enable-pull-request-automerge@v3",
                    with: {
                        token: credentials.tokenRef,
                        "pull-request-number": "${{ github.event.number }}",
                        "merge-method": mergeMethod,
                    },
                },
            ],
        };
        const workflow = workflowEngine.addWorkflow("auto-queue");
        workflow.on({
            // The 'pull request' event gives the workflow 'read-only' permissions on some
            // pull requests (such as the ones from dependabot) when using the `GITHUB_TOKEN`
            // security token. This prevents the workflow from approving these pull requests.
            // Github has placed this guard so as to prevent security attacks by simply opening
            // a pull request and triggering a workflow on a commit that was not vetted to make
            // unintended changes to the repository.
            //
            // Instead use the 'pull request target' event here that gives the Github workflow
            // 'read-write' permissions. This is safe because, this event, unlike the 'pull request'
            // event references the BASE commit of the pull request and not the HEAD commit.
            //
            // We only enable auto-queue when a PR is opened, reopened or moving from Draft to Ready,
            // or retargeted to a different branch. Specifically, if a user disables auto-queue we try very hard to avoid
            // accidentally re-enabling it.
            //
            // The 'edited' trigger is only used to detect base branch changes.
            pullRequestTarget: {
                types: [
                    "opened",
                    "reopened",
                    "ready_for_review",
                    ...(needsEditedEvent ? ["edited"] : []),
                ],
                branches: options.targetBranches,
            },
        });
        workflow.addJobs({ enableAutoQueue: autoQueueJob });
    }
}
exports.AutoQueue = AutoQueue;
_a = JSII_RTTI_SYMBOL_1;
AutoQueue[_a] = { fqn: "projen.github.AutoQueue", version: "0.98.16" };
//# sourceMappingURL=data:application/json;base64,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