"use strict";
// @see https://docs.gitlab.com/ee/ci/yaml/
// Schema from: https://gitlab.com/gitlab-org/gitlab/-/raw/master/app/assets/javascripts/editor/schema/ci.json
// Initial TypeScript Interfaces from: https://app.quicktype.io/#
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowWhen = exports.Strategy = exports.LinkType = exports.DefaultElement = exports.KubernetesEnum = exports.DeploymentTier = exports.Action = exports.JobWhen = exports.PullPolicy = exports.CacheWhen = exports.CachePolicy = void 0;
/**
 * Configure the upload and download behaviour of a cache.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachepolicy
 */
var CachePolicy;
(function (CachePolicy) {
    /** Only download the cache when the job starts, but never upload changes when the job finishes. */
    CachePolicy["PULL"] = "pull";
    /** Only upload a cache when the job finishes, but never download the cache when the job starts. */
    CachePolicy["PUSH"] = "push";
    /** The job downloads the cache when the job starts, and uploads changes to the cache when the job ends. */
    CachePolicy["PULL_PUSH"] = "pull-push";
})(CachePolicy || (exports.CachePolicy = CachePolicy = {}));
/**
 * Configure when artifacts are uploaded depended on job status.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachewhen
 */
var CacheWhen;
(function (CacheWhen) {
    /** Upload artifacts regardless of job status. */
    CacheWhen["ALWAYS"] = "always";
    /** Upload artifacts only when the job fails. */
    CacheWhen["ON_FAILURE"] = "on_failure";
    /** Upload artifacts only when the job succeeds (this is the default). */
    CacheWhen["ON_SUCCESS"] = "on_success";
})(CacheWhen || (exports.CacheWhen = CacheWhen = {}));
/**
 * Describes the conditions for when to pull an image.
 * @see https://docs.gitlab.com/ee/ci/yaml/#servicepull_policy
 */
var PullPolicy;
(function (PullPolicy) {
    PullPolicy["ALWAYS"] = "always";
    PullPolicy["NEVER"] = "never";
    PullPolicy["IF_NOT_PRESENT"] = "if-not-present";
})(PullPolicy || (exports.PullPolicy = PullPolicy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * @see https://docs.gitlab.com/ee/ci/yaml/#when
 */
var JobWhen;
(function (JobWhen) {
    JobWhen["ALWAYS"] = "always";
    JobWhen["DELAYED"] = "delayed";
    JobWhen["MANUAL"] = "manual";
    JobWhen["NEVER"] = "never";
    JobWhen["ON_FAILURE"] = "on_failure";
    JobWhen["ON_SUCCESS"] = "on_success";
})(JobWhen || (exports.JobWhen = JobWhen = {}));
/**
 * Specifies what this job will do. 'start' (default) indicates the job will start the
 * deployment. 'prepare' indicates this will not affect the deployment. 'stop' indicates
 * this will stop the deployment.
 */
var Action;
(function (Action) {
    Action["PREPARE"] = "prepare";
    Action["START"] = "start";
    Action["STOP"] = "stop";
})(Action || (exports.Action = Action = {}));
/**
 * Explicitly specifies the tier of the deployment environment if non-standard environment
 * name is used.
 */
var DeploymentTier;
(function (DeploymentTier) {
    DeploymentTier["DEVELOPMENT"] = "development";
    DeploymentTier["OTHER"] = "other";
    DeploymentTier["PRODUCTION"] = "production";
    DeploymentTier["STAGING"] = "staging";
    DeploymentTier["TESTING"] = "testing";
})(DeploymentTier || (exports.DeploymentTier = DeploymentTier = {}));
/**
 * Filter job based on if Kubernetes integration is active.
 */
var KubernetesEnum;
(function (KubernetesEnum) {
    KubernetesEnum["ACTIVE"] = "active";
})(KubernetesEnum || (exports.KubernetesEnum = KubernetesEnum = {}));
var DefaultElement;
(function (DefaultElement) {
    DefaultElement["AFTER_SCRIPT"] = "after_script";
    DefaultElement["ARTIFACTS"] = "artifacts";
    DefaultElement["BEFORE_SCRIPT"] = "before_script";
    DefaultElement["CACHE"] = "cache";
    DefaultElement["IMAGE"] = "image";
    DefaultElement["INTERRUPTIBLE"] = "interruptible";
    DefaultElement["RETRY"] = "retry";
    DefaultElement["SERVICES"] = "services";
    DefaultElement["TAGS"] = "tags";
    DefaultElement["TIMEOUT"] = "timeout";
})(DefaultElement || (exports.DefaultElement = DefaultElement = {}));
/**
 * The content kind of what users can download via url.
 */
var LinkType;
(function (LinkType) {
    LinkType["IMAGE"] = "image";
    LinkType["OTHER"] = "other";
    LinkType["PACKAGE"] = "package";
    LinkType["RUNBOOK"] = "runbook";
})(LinkType || (exports.LinkType = LinkType = {}));
/**
 * You can mirror the pipeline status from the triggered pipeline to the source bridge job
 * by using strategy: depend or mirror
 * @see https://docs.gitlab.com/ee/ci/yaml/#triggerstrategy
 */
var Strategy;
(function (Strategy) {
    /** Not recommended, use mirror instead. The trigger job status shows failed, success, or running, depending on the downstream pipeline status. */
    Strategy["DEPEND"] = "depend";
    /** Mirrors the status of the downstream pipeline exactly. */
    Strategy["MIRROR"] = "mirror";
})(Strategy || (exports.Strategy = Strategy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * The value can only be 'always' or 'never' when used with workflow.
 * @see https://docs.gitlab.com/ee/ci/yaml/#workflowrules
 */
var WorkflowWhen;
(function (WorkflowWhen) {
    WorkflowWhen["ALWAYS"] = "always";
    WorkflowWhen["NEVER"] = "never";
})(WorkflowWhen || (exports.WorkflowWhen = WorkflowWhen = {}));
//# sourceMappingURL=data:application/json;base64,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