"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = exports.ChecksumPolicy = exports.UpdatePolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    "@xsi:schemaLocation": "http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd",
    "@xmlns": "http://maven.apache.org/POM/4.0.0",
    "@xmlns:xsi": "http://www.w3.org/2001/XMLSchema-instance",
};
class UpdatePolicy {
    /**
     * Updates at an interval of X minutes.
     */
    static interval(minutes) {
        return `interval:${minutes}`;
    }
}
exports.UpdatePolicy = UpdatePolicy;
_a = JSII_RTTI_SYMBOL_1;
UpdatePolicy[_a] = { fqn: "projen.java.UpdatePolicy", version: "0.98.16" };
UpdatePolicy.ALWAYS = "always";
UpdatePolicy.DAILY = "daily";
UpdatePolicy.NEVER = "never";
var ChecksumPolicy;
(function (ChecksumPolicy) {
    ChecksumPolicy["IGNORE"] = "ignore";
    ChecksumPolicy["FAIL"] = "fail";
    ChecksumPolicy["WARN"] = "warn";
})(ChecksumPolicy || (exports.ChecksumPolicy = ChecksumPolicy = {}));
/**
 * A Project Object Model or POM is the fundamental unit of work in Maven. It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 */
class Pom extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.properties = {};
        this.repositories = [];
        this.pluginRepositories = [];
        this.fileName = "pom.xml";
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = options.packaging ?? "jar";
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        this.parentPom = options.parentPom;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * Adds a key/value property to the pom.
     * @param key the key
     * @param value the value
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`)
     * @param options plugin options
     */
    addPlugin(spec, options = {}) {
        for (const dep of options.dependencies ?? []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    /**
     * Adds a repository to the pom
     * @param repository the repository to add
     */
    addRepository(repository) {
        this.repositories.push(repository);
    }
    /*
     * Adds a repository for plugins to the pom
     * @param repository the repository to add
     */
    addPluginRepository(repository) {
        this.pluginRepositories.push(repository);
    }
    synthPom() {
        return (0, _resolve_1.resolve)({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: "4.0.0",
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                parent: this.parentPom,
                ...this.synthRepositories(),
                ...this.synthDependencies(),
                ...this.synthPluginRepositories(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: "test",
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
    synthRepositories() {
        if (this.repositories.length === 0) {
            return;
        }
        return {
            repositories: { repository: this.repositories },
        };
    }
    synthPluginRepositories() {
        if (this.pluginRepositories.length === 0) {
            return;
        }
        return {
            pluginRepositories: { pluginRepository: this.pluginRepositories },
        };
    }
}
exports.Pom = Pom;
_b = JSII_RTTI_SYMBOL_1;
Pom[_b] = { fqn: "projen.java.Pom", version: "0.98.16" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split("/");
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? (0, semver_1.toMavenVersionRange)(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    return {
        configuration: options.configuration,
        dependencies: options.dependencies && options.dependencies.length > 0
            ? {
                dependency: options.dependencies?.map((d) => mavenCoords(dependencies_1.Dependencies.parseDependency(d))),
            }
            : undefined,
        executions: {
            execution: options.executions?.map((e) => ({
                id: e.id,
                goals: e.goals.map((goal) => ({ goal })),
                phase: e.phase,
                configuration: e.configuration,
            })),
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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