"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const license_checker_1 = require("./license-checker");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const consts_1 = require("../build/private/consts");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const biome_1 = require("./biome/biome");
const util_1 = require("./util");
const constants_1 = require("../github/constants");
const util_2 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const projenrc_json_1 = require("../projenrc-json");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const util_3 = require("../util");
const path_2 = require("../util/path");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease || (exports.AutoRelease = AutoRelease = {}));
/**
 * Node.js project.
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    /**
     * The .npmrc file
     */
    get npmrc() {
        if (!this._npmrc) {
            this._npmrc = new javascript_1.NpmConfig(this, { omitEmpty: true });
        }
        return this._npmrc;
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * The minimum node version required by this package to function.
     *
     * This value indicates the package is incompatible with older versions.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version supported by this package.
     *
     * The value indicates the package is incompatible with newer versions.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    constructor(options) {
        super({
            ...options,
            // Node projects have the specific projen version locked via lockfile, so we can skip the @<VERSION> part of the top-level project
            projenCommand: options.projenCommand ?? "npx projen",
        });
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.workflowPackageCache = options.workflowPackageCache ?? false;
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_2.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        const normalizedArtifactsDirectory = (0, util_3.normalizePersistedPath)(this.artifactsDirectory);
        this.artifactsJavascriptDirectory = path_1.posix.join(normalizedArtifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                case node_package_1.NodePackageManager.YARN_CLASSIC:
                case node_package_1.NodePackageManager.YARN_BERRY:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                case node_package_1.NodePackageManager.BUN:
                    return "bun run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        const envCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.PNPM:
                    return '$(pnpm -c exec "node --print process.env.PATH")';
                case node_package_1.NodePackageManager.BUN:
                    return '$(bun --eval "console.log(process.env.PATH)")';
                default:
                    return '$(npx -c "node --print process.env.PATH")';
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", envCommand);
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        const projen = options.projenDevDependency ?? (this.parent ? false : true);
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
            if (!this.deps.isDependencySatisfied("constructs", dependencies_1.DependencyType.BUILD, "^10.0.0")) {
                this.addDevDeps(`constructs@^10.0.0`);
            }
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const workflowPermissions = {
            idToken: this.determineIdTokenPermissions(options),
        };
        const buildWorkflowOptions = options.buildWorkflowOptions ?? {};
        if (buildEnabled && (this.github || github_1.GitHub.of(this.root))) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
                ...buildWorkflowOptions,
                preBuildSteps: this.renderWorkflowSetup({
                    installStepConfiguration: {
                        workingDirectory: this.determineInstallWorkingDirectory(),
                    },
                    mutable: buildWorkflowOptions.mutableBuild ?? options.mutableBuild ?? true,
                }).concat(buildWorkflowOptions.preBuildSteps ?? []),
                postBuildSteps: [...(options.postBuildSteps ?? [])],
                ...(0, runner_options_1.filteredRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
            this.maybeAddCodecovIgnores(options);
        }
        // Build release tasks array
        const releaseTasks = [];
        // Add security audit before release if enabled
        if (options.auditDeps) {
            const auditTask = this.addAuditTask(options);
            const runOn = options.auditDepsOptions?.runOn ?? "build";
            if (runOn === "release") {
                releaseTasks.push(auditTask);
            }
            else if (runOn === "build") {
                this.preCompileTask.spawn(auditTask);
            }
            // "manual" mode doesn't add to any task
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            // Add build task
            releaseTasks.push(this.buildTask);
            this.release = new release_1.Release(this, {
                versionFile: "package.json", // this is where "version" is set after bump
                tasks: releaseTasks,
                branch: options.defaultReleaseBranch ?? "main",
                ...options,
                artifactsDirectory: this.artifactsDirectory,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({
                        installStepConfiguration: {
                            workingDirectory: this.determineInstallWorkingDirectory(),
                        },
                        mutable: false,
                    }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.maybeAddCodecovIgnores(options);
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = (0, release_1.isAwsCodeArtifactRegistry)(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    npmProvenance: this.package.npmProvenance,
                    trustedPublishing: options.npmTrustedPublishing ?? false,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot((0, util_3.deepMerge)([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, (0, util_3.deepMerge)([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            const projenrcJsFile = new projenrc_1.Projenrc(this, options.projenrcJsOptions);
            this.npmignore?.exclude(`/${projenrcJsFile.filePath}`);
        }
        else if (options.projenrcJson) {
            const projenrcJsonFile = projenrc_json_1.ProjenrcJson.of(this);
            if (projenrcJsonFile) {
                this.npmignore?.exclude(`/${projenrcJsonFile.filePath}`);
            }
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        const shouldPackage = options.package ?? true;
        if (release && !shouldPackage) {
            this.logger.warn("When `release` is enabled, `package` must also be enabled as it is required by release. Force enabling `package`.");
        }
        if (release || shouldPackage) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            const pkgMgr = this.package.packageManager === node_package_1.NodePackageManager.PNPM
                ? "pnpm"
                : "npm";
            this.packageTask.exec(`${pkgMgr} pack --pack-destination ${this.artifactsJavascriptDirectory}`);
        }
        if ((0, util_3.multipleSelected)([options.biome, options.prettier])) {
            throw new Error("Only one of biome and prettier can be enabled.");
        }
        if (options.biome ?? false) {
            this.biome = new biome_1.Biome(this, { ...options.biomeOptions });
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
        // For PNPM, the default resolution mode is "lowest", which leads to any non-versioned (ie '*') dependencies being
        // resolved to the lowest available version, which is unlikely to be expected behaviour for users. We set resolution
        // mode to "highest" to match the behaviour of yarn and npm.
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("resolution-mode", "highest");
        }
        if (options.checkLicenses) {
            new license_checker_1.LicenseChecker(this, options.checkLicenses);
        }
    }
    determineInstallWorkingDirectory() {
        if (this.parent) {
            return (0, path_2.ensureRelativePathStartsWithDot)((0, path_1.relative)(".", this.root.outdir));
        }
        return;
    }
    determineIdTokenPermissions(options) {
        const { codeCovTokenSecret, scopedPackagesOptions, codeArtifactOptions } = options;
        const useCodeCoveOidc = this.useCodecov(options) && !codeCovTokenSecret;
        const hasScopedPackages = (scopedPackagesOptions ?? []).length > 0;
        const useCodeArtifactOidc = codeArtifactOptions?.authProvider ===
            release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        if (useCodeCoveOidc || (useCodeArtifactOidc && hasScopedPackages)) {
            return workflows_model_1.JobPermission.WRITE;
        }
        return undefined;
    }
    useCodecov(options) {
        // Use Codecov when it is enabled or if or a secret token name is passed in
        // AND jest must be configured
        return (options.codeCov || options.codeCovTokenSecret) && this.jest?.config;
    }
    maybeAddCodecovIgnores(options) {
        if (this.useCodecov(options)) {
            this.addGitIgnore("codecov");
            this.addGitIgnore("codecov.*");
        }
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if (this.useCodecov(options)) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v5",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest?.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest?.config.coverageDirectory,
                            use_oidc: true,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts) {
        for (const [name, command] of Object.entries(scripts)) {
            this.package.setScript(name, command);
        }
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-access-key-id": (0, util_2.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": (0, util_2.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: (0, util_2.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: (0, util_2.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v4",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === node_package_1.NodePackageManager.BUN) {
            install.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        if (this.package.packageManager !== node_package_1.NodePackageManager.BUN) {
            if (this.nodeVersion || this.workflowPackageCache) {
                const pm = this.package.packageManager;
                const cache = (0, util_1.isYarnClassic)(pm) || (0, util_1.isYarnBerry)(pm)
                    ? "yarn"
                    : pm === node_package_1.NodePackageManager.PNPM
                        ? "pnpm"
                        : "npm";
                install.push({
                    name: "Setup Node.js",
                    uses: "actions/setup-node@v5",
                    with: {
                        ...(this.nodeVersion && {
                            "node-version": this.nodeVersion,
                        }),
                        ...(this.workflowPackageCache && {
                            cache,
                        }),
                    },
                });
            }
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
            ...(options.installStepConfiguration ?? {}),
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
    /**
     * Adds a security audit task.
     */
    addAuditTask(options) {
        const auditLevel = options.auditDepsOptions?.level ?? "high";
        const auditProdOnly = options.auditDepsOptions?.prodOnly ?? false;
        // Create audit task
        const auditTask = this.tasks.addTask("audit", {
            description: "Run security audit",
        });
        // Add package manager specific audit command
        const auditCommand = this.getAuditCommand(auditLevel, auditProdOnly);
        auditTask.exec(auditCommand);
        return auditTask;
    }
    /**
     * Gets the appropriate audit command for the package manager.
     */
    getAuditCommand(level, prodOnly) {
        const levelFlag = this.getAuditLevelFlag(level);
        const prodFlag = prodOnly ? this.getAuditProdFlag() : "";
        switch (this.packageManager) {
            case node_package_1.NodePackageManager.NPM:
                return `npm audit${levelFlag}${prodFlag}`;
            case node_package_1.NodePackageManager.YARN:
            case node_package_1.NodePackageManager.YARN_CLASSIC:
                // Use Node.js to call yarn and handle exit code cross-platform
                const threshold = this.getYarnClassicThreshold(level);
                return `node -e "const { execSync } = require('child_process'); try { execSync('yarn audit${levelFlag}${prodFlag}', {stdio: 'inherit'}); } catch(e) { process.exit(e.status < ${threshold} ? 0 : 1); }"`;
            case node_package_1.NodePackageManager.YARN2:
            case node_package_1.NodePackageManager.YARN_BERRY:
                return `yarn npm audit --recursive${levelFlag}${prodFlag}`;
            case node_package_1.NodePackageManager.PNPM:
                return `pnpm audit${levelFlag}${prodFlag}`;
            case node_package_1.NodePackageManager.BUN:
                return `bun audit${levelFlag}${prodFlag}`;
            default:
                throw new Error(`Unsupported package manager: ${this.packageManager}`);
        }
    }
    /**
     * Gets the threshold value for yarn classic based on vulnerability level.
     */
    getYarnClassicThreshold(level) {
        switch (level) {
            case "low":
                return 2; // Pass for exit code < 2, fail for >= 2 (LOW and above)
            case "moderate":
                return 4; // Pass for exit code < 4, fail for >= 4 (MODERATE and above)
            case "high":
                return 8; // Pass for exit code < 8, fail for >= 8 (HIGH and above)
            case "critical":
                return 16; // Pass for exit code < 16, fail for >= 16 (CRITICAL)
            default:
                return 2;
        }
    }
    /**
     * Gets the audit level flag for the package manager.
     */
    getAuditLevelFlag(level) {
        switch (this.packageManager) {
            case node_package_1.NodePackageManager.NPM:
                return ` --audit-level=${level}`;
            case node_package_1.NodePackageManager.YARN:
            case node_package_1.NodePackageManager.YARN_CLASSIC:
                return ` --level ${level}`;
            case node_package_1.NodePackageManager.YARN2:
            case node_package_1.NodePackageManager.YARN_BERRY:
                return ` --severity ${level}`;
            case node_package_1.NodePackageManager.PNPM:
                return ` --audit-level ${level}`;
            case node_package_1.NodePackageManager.BUN:
                return ` --audit-level ${level}`;
            default:
                return "";
        }
    }
    /**
     * Gets the production-only flag for the package manager.
     */
    getAuditProdFlag() {
        switch (this.packageManager) {
            case node_package_1.NodePackageManager.NPM:
                return " --omit=dev";
            case node_package_1.NodePackageManager.YARN:
            case node_package_1.NodePackageManager.YARN_CLASSIC:
                return " --groups dependencies";
            case node_package_1.NodePackageManager.YARN2:
            case node_package_1.NodePackageManager.YARN_BERRY:
                return " --environment production";
            case node_package_1.NodePackageManager.PNPM:
                return " --prod";
            case node_package_1.NodePackageManager.BUN:
                return " --production";
            default:
                return "";
        }
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.98.16" };
//# sourceMappingURL=data:application/json;base64,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