"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = exports.TestFailureBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// inspired by https://github.com/cdk8s-team/cdk8s-core/blob/2.x/src/json-patch.ts
// under Apache 2.0 license
const fast_json_patch_1 = require("fast-json-patch");
var TestFailureBehavior;
(function (TestFailureBehavior) {
    /**
     * Skip the current patch operation and continue with the next operation.
     */
    TestFailureBehavior["SKIP"] = "skip";
    /**
     * Fail the whole file synthesis.
     */
    TestFailureBehavior["FAIL_SYNTHESIS"] = "fail";
})(TestFailureBehavior || (exports.TestFailureBehavior = TestFailureBehavior = {}));
const TEST_FAILURE_BEHAVIOR_SYMBOL = Symbol.for("testFailureBehavior");
/**
 * Utility for applying RFC-6902 JSON-Patch to a document.
 *
 * Use the the `JsonPatch.apply(doc, ...ops)` function to apply a set of
 * operations to a JSON document and return the result.
 *
 * Operations can be created using the factory methods `JsonPatch.add()`,
 * `JsonPatch.remove()`, etc.
 *
 * @example
 *
 *const output = JsonPatch.apply(input,
 *  JsonPatch.replace('/world/hi/there', 'goodbye'),
 *  JsonPatch.add('/world/foo/', 'boom'),
 *  JsonPatch.remove('/hello'));
 *
 */
class JsonPatch {
    /**
     * Applies a set of JSON-Patch (RFC-6902) operations to `document` and returns the result.
     * @param document The document to patch
     * @param ops The operations to apply
     * @returns The result document
     */
    static apply(document, ...ops) {
        try {
            const result = (0, fast_json_patch_1.applyPatch)(document, (0, fast_json_patch_1.deepClone)(ops.map((o) => o._toJson())));
            return result.newDocument;
        }
        catch (e) {
            if (e instanceof fast_json_patch_1.JsonPatchError && e.name === "TEST_OPERATION_FAILED") {
                const op = ops[e.index];
                if (TEST_FAILURE_BEHAVIOR_SYMBOL in op) {
                    const failureBehavior = op[TEST_FAILURE_BEHAVIOR_SYMBOL];
                    if (failureBehavior === TestFailureBehavior.SKIP) {
                        return document;
                    }
                }
            }
            throw e;
        }
    }
    /**
     * Adds a value to an object or inserts it into an array. In the case of an
     * array, the value is inserted before the given index. The - character can be
     * used instead of an index to insert at the end of an array.
     *
     * @example JsonPatch.add('/biscuits/1', { "name": "Ginger Nut" })
     */
    static add(path, value) {
        return new JsonPatch({
            op: "add",
            path,
            value,
        });
    }
    /**
     * Removes a value from an object or array.
     *
     * @example JsonPatch.remove('/biscuits')
     * @example JsonPatch.remove('/biscuits/0')
     */
    static remove(path) {
        return new JsonPatch({ op: "remove", path });
    }
    /**
     * Replaces a value. Equivalent to a “remove” followed by an “add”.
     *
     * @example JsonPatch.replace('/biscuits/0/name', 'Chocolate Digestive')
     */
    static replace(path, value) {
        return new JsonPatch({
            op: "replace",
            path,
            value,
        });
    }
    /**
     * Copies a value from one location to another within the JSON document. Both
     * from and path are JSON Pointers.
     *
     * @example JsonPatch.copy('/biscuits/0', '/best_biscuit')
     */
    static copy(from, path) {
        return new JsonPatch({ op: "copy", from, path });
    }
    /**
     * Moves a value from one location to the other. Both from and path are JSON Pointers.
     *
     * @example JsonPatch.move('/biscuits', '/cookies')
     */
    static move(from, path) {
        return new JsonPatch({ op: "move", from, path });
    }
    /**
     * Tests that the specified value is set in the document. If the test fails,
     * then the patch as a whole should not apply.
     *
     * @example JsonPatch.test('/best_biscuit/name', 'Choco Leibniz')
     */
    static test(path, value, failureBehavior = TestFailureBehavior.SKIP) {
        const patch = new JsonPatch({
            op: "test",
            path,
            value,
        });
        Object.defineProperty(patch, TEST_FAILURE_BEHAVIOR_SYMBOL, {
            writable: false,
            value: failureBehavior,
        });
        return patch;
    }
    /**
     * Escapes a json pointer path
     * @param path The raw pointer
     * @return the Escaped path
     */
    static escapePath(path) {
        return (0, fast_json_patch_1.escapePathComponent)(path);
    }
    constructor(operation) {
        this.operation = operation;
    }
    /**
     * Returns the JSON representation of this JSON patch operation.
     *
     * @internal
     */
    _toJson() {
        return this.operation;
    }
}
exports.JsonPatch = JsonPatch;
_a = JSII_RTTI_SYMBOL_1;
JsonPatch[_a] = { fqn: "projen.JsonPatch", version: "0.98.16" };
//# sourceMappingURL=data:application/json;base64,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