"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PoetryPyproject = exports.Poetry = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const TOML = require("@iarna/toml");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const task_runtime_1 = require("../task-runtime");
const toml_1 = require("../toml");
const util_1 = require("../util");
/**
 * Manage project dependencies, virtual environments, and packaging through the
 * poetry CLI tool.
 */
class Poetry extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.pythonExec = options.pythonExec ?? "python";
        this.installTask = project.addTask("install", {
            description: "Install dependencies and update lockfile",
            exec: "poetry update",
        });
        this.installCiTask = project.addTask("install:ci", {
            description: "Install dependencies with frozen lockfile",
            exec: "poetry check --lock && poetry install",
        });
        this.project.tasks.addEnvironment("VIRTUAL_ENV", 
        // Create .venv on the first run if it doesn't already exist
        "$(poetry env info -p || poetry run poetry env info -p)");
        this.project.tasks.addEnvironment("PATH", "$(echo $(poetry env info -p)/bin:$PATH)");
        project.packageTask.exec("poetry build");
        this.publishTestTask = project.addTask("publish:test", {
            description: "Uploads the package against a test PyPI endpoint.",
            exec: "poetry publish -r testpypi",
        });
        this.publishTask = project.addTask("publish", {
            description: "Uploads the package to PyPI.",
            exec: "poetry publish",
        });
        this.pyProject = new PoetryPyproject(project, {
            name: project.name,
            version: options.version,
            description: options.description ?? "",
            license: options.license,
            authors: [`${options.authorName} <${options.authorEmail}>`],
            homepage: options.homepage,
            classifiers: options.classifiers,
            ...options.poetryOptions,
            dependencies: () => this.synthDependencies(),
            devDependencies: () => this.synthDevDependencies(),
        });
        new toml_1.TomlFile(project, "poetry.toml", {
            committed: false,
            obj: {
                repositories: {
                    testpypi: {
                        url: "https://test.pypi.org/legacy/",
                    },
                },
            },
        });
    }
    synthDependencies() {
        const dependencies = {};
        let pythonDefined = false;
        for (const pkg of this.project.deps.all) {
            if (pkg.name === "python") {
                pythonDefined = true;
            }
            if (pkg.type === dependencies_1.DependencyType.RUNTIME) {
                dependencies[pkg.name] = pkg.version ?? "*";
            }
        }
        if (!pythonDefined) {
            // Python version must be defined for poetry projects. Default to ^3.8.
            dependencies.python = "^3.8";
        }
        return this.permitDepsWithTomlInlineTables(dependencies);
    }
    synthDevDependencies() {
        const dependencies = {};
        for (const pkg of this.project.deps.all) {
            if ([dependencies_1.DependencyType.DEVENV, dependencies_1.DependencyType.TEST].includes(pkg.type)) {
                dependencies[pkg.name] = pkg.version ?? "*";
            }
        }
        return this.permitDepsWithTomlInlineTables(dependencies);
    }
    /**
     * Parses dependency values that may include TOML inline tables, converting them into JavaScript objects.
     * If a dependency value cannot be parsed as a TOML inline table (indicating it is a plain SemVer string),
     * it is left unchanged. This allows to support the full range of Poetry's dependency specification.
     * @see https://python-poetry.org/docs/dependency-specification/
     * @see https://toml.io/en/v1.0.0#inline-table
     *
     * @example
     * // Given a `dependencies` object like this:
     * const dependencies = {
     *   "mypackage": "{ version = '1.2.3', extras = ['extra1', 'extra2'] }",
     *   "anotherpackage": "^2.3.4"
     * };
     * // After parsing, the resulting object would be:
     * {
     *   "mypackage": {
     *     version: "1.2.3",
     *     extras: ["extra1", "extra2"]
     *   },
     *   "anotherpackage": "^2.3.4"
     * }
     * // Note: The value of `anotherpackage` remains unchanged as it is a plain SemVer string.
     *
     * @param dependencies An object where each key is a dependency name and each value is a string that might be
     * either a SemVer string or a TOML inline table string.
     * @returns A new object where each key is a dependency name and each value is either the original SemVer string
     * or the parsed JavaScript object representation of the TOML inline table.
     */
    permitDepsWithTomlInlineTables(dependencies) {
        const parseTomlInlineTable = (dependencyValue) => {
            try {
                // Attempt parsing the `dependencyValue` as a TOML inline table
                return TOML.parse(`dependencyKey = ${dependencyValue}`).dependencyKey;
            }
            catch {
                // If parsing fails, treat the `dependencyValue` as a plain SemVer string
                return dependencyValue;
            }
        };
        return Object.fromEntries(Object.entries(dependencies).map(([dependencyKey, dependencyValue]) => {
            return [dependencyKey, parseTomlInlineTable(dependencyValue)];
        }));
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.DEVENV);
    }
    /**
     * Initializes the virtual environment if it doesn't exist (called during post-synthesis).
     */
    setupEnvironment() {
        const result = (0, util_1.execOrUndefined)("which poetry", {
            cwd: this.project.outdir,
        });
        if (!result) {
            this.project.logger.info("Unable to setup an environment since poetry is not installed. Please install poetry (https://python-poetry.org/docs/) or use a different component for managing environments such as 'venv'.");
        }
        let envPath = (0, util_1.execOrUndefined)("poetry env info -p", {
            cwd: this.project.outdir,
        });
        if (!envPath) {
            this.project.logger.info("Setting up a virtual environment...");
            (0, util_1.exec)(`poetry env use ${this.pythonExec}`, { cwd: this.project.outdir });
            envPath = (0, util_1.execOrUndefined)("poetry env info -p", {
                cwd: this.project.outdir,
            });
            this.project.logger.info(`Environment successfully created (located in ${envPath}}).`);
        }
    }
    /**
     * Installs dependencies (called during post-synthesis).
     */
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        // If the pyproject.toml file has changed, update the lockfile prior to installation
        if (this.pyProject.file.changed) {
            runtime.runTask(this.installTask.name);
        }
        else {
            runtime.runTask(this.installCiTask.name);
        }
    }
}
exports.Poetry = Poetry;
_a = JSII_RTTI_SYMBOL_1;
Poetry[_a] = { fqn: "projen.python.Poetry", version: "0.98.16" };
/**
 * Represents configuration of a pyproject.toml file for a Poetry project.
 *
 * @see https://python-poetry.org/docs/pyproject/
 */
class PoetryPyproject extends component_1.Component {
    constructor(project, options) {
        super(project);
        const { dependencies, devDependencies, ...otherOptions } = options;
        const decamelizedOptions = (0, util_1.decamelizeKeysRecursively)(otherOptions, {
            separator: "-",
        });
        const tomlStructure = {
            tool: {
                poetry: {
                    ...decamelizedOptions,
                    dependencies: dependencies,
                    group: {
                        dev: {
                            dependencies: devDependencies,
                        },
                    },
                },
            },
            "build-system": {
                requires: ["poetry-core"],
                "build-backend": "poetry.core.masonry.api",
            },
        };
        this.file = new toml_1.TomlFile(project, "pyproject.toml", {
            omitEmpty: false,
            obj: tomlStructure,
        });
    }
}
exports.PoetryPyproject = PoetryPyproject;
_b = JSII_RTTI_SYMBOL_1;
PoetryPyproject[_b] = { fqn: "projen.python.PoetryPyproject", version: "0.98.16" };
//# sourceMappingURL=data:application/json;base64,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