"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const runner_options_1 = require("../runner-options");
const version_1 = require("../version");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.18.0" } },
    dotnet: { dotnet: { version: "6.x" } },
};
const PUBLISH_JOB_PREFIX = "release_";
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this._gitHubPostPublishing = [];
        // List of publish jobs added to the publisher
        // Maps between the basename and the jobname
        this.publishJobs = {};
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "lts/*";
        this.workflowContainerImage = options.workflowContainerImage;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.publishTasks = options.publishTasks ?? false;
        this.runsOn = options.workflowRunsOn;
        this.runsOnGroup = options.workflowRunsOnGroup;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Adds post publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPostPublishingSteps(...steps) {
        this._gitHubPostPublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: version_1.CHANGES_SINCE_LAST_RELEASE,
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const jobName = "github";
        this.addPublishJob(jobName, (_branch, branchOptions) => {
            return {
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                postPublishSteps: options.postPublishSteps ?? this._gitHubPostPublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                needs: Object.entries(this.publishJobs)
                    .filter(([name, _]) => name != jobName)
                    .map(([_, job]) => job),
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.githubReleaseCommand(options, branchOptions),
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                },
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        if (options.trustedPublishing && options.npmTokenSecret) {
            throw new Error("Cannot use npmTokenSecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of NPM tokens. " +
                "Remove the npmTokenSecret option to use trusted publishing.");
        }
        const trustedPublisher = options.trustedPublishing ? "true" : undefined;
        const npmProvenance = options.npmProvenance ? "true" : undefined;
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const needsIdTokenWrite = isAwsCodeArtifactWithOidc || trustedPublisher || npmProvenance;
        const npmToken = trustedPublisher
            ? undefined
            : (0, node_package_1.defaultNpmToken)(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v5",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob("npm", (_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            return {
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps,
                postPublishSteps: options.postPublishSteps ?? [],
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                    NPM_CONFIG_PROVENANCE: npmProvenance,
                    NPM_TRUSTED_PUBLISHER: trustedPublisher,
                },
                permissions: {
                    idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        if (options.trustedPublishing && options.nugetApiKeySecret) {
            throw new Error("Cannot use nugetApiKeySecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of API keys. " +
                "Remove the nugetApiKeySecret option to use trusted publishing.");
        }
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        const needsIdTokenWrite = options.trustedPublishing;
        this.addPublishJob("nuget", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            env: {
                NUGET_TRUSTED_PUBLISHER: options.trustedPublishing
                    ? "true"
                    : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: options.trustedPublishing
                    ? undefined
                    : secret(isGitHubPackages
                        ? "GITHUB_TOKEN"
                        : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
                NUGET_USERNAME: options.trustedPublishing
                    ? secret(options.nugetUsernameSecret ?? "NUGET_USERNAME")
                    : undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : "central-ossrh");
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        if (mavenServerId === "central-ossrh" && options.mavenEndpoint != null) {
            throw new Error('Custom endpoints are not supported when publishing to Maven Central (mavenServerId: "central-ossrh"). Please remove "mavenEndpoint" from the options.');
        }
        this.addPublishJob("maven", (_branch, branchOptions) => ({
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        if (options.trustedPublishing &&
            (options.twineUsernameSecret || options.twinePasswordSecret)) {
            throw new Error("Cannot use twineUsernameSecret and twinePasswordSecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of username/password credentials. " +
                "Remove the twineUsernameSecret and twinePasswordSecret options to use trusted publishing.");
        }
        let permissions = { contents: workflows_model_1.JobPermission.READ };
        const prePublishSteps = options.prePublishSteps ?? [];
        let workflowEnv = {};
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.twineRegistryUrl);
        if (isAwsCodeArtifact) {
            const { domain, account, region } = awsCodeArtifactInfoFromUrl(options.twineRegistryUrl);
            const { authProvider, roleToAssume, accessKeyIdSecret, secretAccessKeySecret, } = options.codeArtifactOptions ?? {};
            const useOidcAuth = authProvider === CodeArtifactAuthProvider.GITHUB_OIDC;
            if (useOidcAuth) {
                if (!roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
                permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
                prePublishSteps.push({
                    name: "Configure AWS Credentials via GitHub OIDC Provider",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "role-to-assume": roleToAssume,
                        "aws-region": region,
                    },
                });
            }
            prePublishSteps.push({
                name: "Generate CodeArtifact Token",
                run: `echo "TWINE_PASSWORD=$(aws codeartifact get-authorization-token --domain ${domain} --domain-owner ${account} --region ${region} --query authorizationToken --output text)" >> $GITHUB_ENV`,
                env: useOidcAuth
                    ? undefined
                    : {
                        AWS_ACCESS_KEY_ID: secret(accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID"),
                        AWS_SECRET_ACCESS_KEY: secret(secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY"),
                    },
            });
            workflowEnv = { TWINE_USERNAME: "aws" };
        }
        else if (options.trustedPublishing) {
            permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
            workflowEnv = {
                PYPI_TRUSTED_PUBLISHER: "true",
            };
            // attestations default to true, only disable when explicitly requested
            if (options.attestations === false) {
                workflowEnv.PYPI_DISABLE_ATTESTATIONS = "true";
            }
        }
        else {
            workflowEnv = {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            };
        }
        this.addPublishJob("pypi", (_branch, branchOptions) => ({
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            permissions,
            prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv,
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob("golang", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(
    /**
     * The basename of the publish job (should be lowercase).
     * Will be extended with a prefix.
     */
    basename, factory) {
        const jobname = `${PUBLISH_JOB_PREFIX}${basename}`;
        this.publishJobs[basename] = jobname;
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [env, expression] of workflowEnvEntries) {
                requiredEnv.push(env);
                jobEnv[env] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${basename.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v5",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR, // this must be "dist" for publib
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
                ...opts.postPublishSteps,
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            const container = this.workflowContainerImage
                ? {
                    image: this.workflowContainerImage,
                }
                : undefined;
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        shell: "bash",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    ...(opts.environment ? { environment: opts.environment } : {}),
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId, ...(opts.needs ?? [])],
                    ...(0, runner_options_1.filteredRunsOnOptions)(this.runsOn, this.runsOnGroup),
                    container,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_SHA",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.98.16" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
/**
 * Parses info about code artifact domain from given AWS code artifact url
 * @param url Of code artifact domain
 * @returns domain, account, and region of code artifact domain
 */
function awsCodeArtifactInfoFromUrl(url) {
    const captureRegex = /([a-z0-9-]+)-(.+)\.d\.codeartifact\.(.+)\.amazonaws\.com/;
    const matches = url?.match(captureRegex) ?? [];
    const [_, domain, account, region] = matches;
    return { domain, account, region };
}
//# sourceMappingURL=data:application/json;base64,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