"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tasks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const component_1 = require("./component");
const json_1 = require("./json");
const logging_1 = require("./logging");
const task_1 = require("./task");
const task_runtime_1 = require("./task-runtime");
/**
 * Defines project tasks.
 *
 * Tasks extend the projen CLI by adding subcommands to it. Task definitions are
 * synthesized into `.projen/tasks.json`.
 */
class Tasks extends component_1.Component {
    constructor(project) {
        super(project);
        this._tasks = {};
        this._env = {};
        new json_1.JsonFile(project, task_runtime_1.TaskRuntime.MANIFEST_FILE, {
            omitEmpty: true,
            obj: {
                tasks: (() => this.renderTasks()),
                env: (() => this.renderEnv()),
            },
        });
    }
    /**
     * All tasks.
     */
    get all() {
        return Object.values(this._tasks);
    }
    /**
     * Adds a task to a project.
     * @param name The name of the task
     * @param options Task options.
     */
    addTask(name, options = {}) {
        const task = new task_1.Task(name, options);
        if (this._tasks[name]) {
            throw new Error(`A task with the name ${name} already exists. To override it, call removeTask first and then create the new task.`);
        }
        this._tasks[name] = task;
        return task;
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        const dependentTasks = this.all.filter((task) => task.steps.find((step) => step.spawn == name));
        if (dependentTasks.length > 0) {
            const errList = dependentTasks.map((depTask) => depTask.name).join(", ");
            throw new Error(`Unable to remove task "${name}" because the following tasks depend on it: ${errList}`);
        }
        const task = this._tasks[name];
        if (task) {
            delete this._tasks[name];
            return task;
        }
        else {
            return undefined;
        }
    }
    /**
     * Adds global environment.
     * @param name Environment variable name
     * @param value Value
     */
    addEnvironment(name, value) {
        this._env[name] = value;
    }
    /**
     * Returns a copy of the currently global environment for this project.
     */
    get env() {
        return {
            ...this._env,
        };
    }
    /**
     * Finds a task by name. Returns `undefined` if the task cannot be found.
     * @param name The name of the task
     */
    tryFind(name) {
        return this._tasks[name];
    }
    synthesize() {
        if (this.project.ejected) {
            // Insert a task-runner script so that tasks can be run after ejecting
            fs.mkdirSync(path.join(this.project.outdir, "scripts"), {
                recursive: true,
            });
            fs.copyFileSync(path.join(__dirname, "..", "lib", "run-task.cjs"), path.join(this.project.outdir, "scripts", "run-task.cjs"));
            fs.chmodSync(path.join(this.project.outdir, "scripts", "run-task.cjs"), "755");
        }
    }
    renderTasks() {
        const tasks = {};
        for (const task of Object.values(this._tasks).sort((x, y) => x.name.localeCompare(y.name))) {
            tasks[task.name] = task._renderSpec();
        }
        return tasks;
    }
    renderEnv() {
        return Object.keys(this._env).reduce((prev, curr) => ({
            ...prev,
            [curr]: this.getEnvString(curr, this._env[curr]),
        }), {});
    }
    /**
     * Ensure that environment variables are persisted as strings
     * to prevent type errors when parsing from tasks.json in future
     */
    getEnvString(name, value) {
        if (typeof value !== "string" && value !== undefined) {
            (0, logging_1.warn)(`Received non-string value for environment variable ${name}. Value will be stringified.`);
            return String(value);
        }
        else {
            return value;
        }
    }
}
exports.Tasks = Tasks;
_a = JSII_RTTI_SYMBOL_1;
Tasks[_a] = { fqn: "projen.Tasks", version: "0.98.16" };
//# sourceMappingURL=data:application/json;base64,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