# Migration Guide: v0.1.0 to v0.2.0

## Overview

This migration guide covers the upgrade from aiogram-sentinel v0.1.0 to v0.2.0. The main changes include expanded hooks system, Redis prefix support, and improved error handling.

## Breaking Changes

| Old API | New API | Impact | Migration Required |
|---------|---------|--------|-------------------|
| `Sentinel.setup(dp, config)` | `Sentinel(config=config)` | High | Yes |
| `@rate_limit(limit, window)` | `@sentinel_rate_limit(limit, window)` | Medium | Optional |
| `@debounce(delay)` | `@sentinel_debounce(delay)` | Medium | Optional |
| `@require_registered()` | `@sentinel_require_registered()` | Medium | Optional |

## Code Modifications

### Before (v0.1.0)

```python
from aiogram import Bot, Dispatcher
from aiogram_sentinel import Sentinel, SentinelConfig
from aiogram_sentinel import rate_limit, debounce, require_registered

# Old setup method
config = SentinelConfig(
    backend="redis",
    redis_url="redis://localhost:6379",
    throttling_default_max=10,
)

# Old setup call
router, backends = await Sentinel.setup(dp, config)

# Old decorators
@rate_limit(5, 60)
@debounce(2)
@require_registered()
async def protected_handler(message: Message):
    await message.answer("Protected command!")
```

### After (v0.2.0)

```python
from aiogram import Bot, Dispatcher
from aiogram_sentinel import Sentinel, SentinelConfig
from aiogram_sentinel.decorators import sentinel_rate_limit, sentinel_debounce, sentinel_require_registered

# New setup method
config = SentinelConfig(
    backend="redis",
    redis_url="redis://localhost:6379",
    redis_prefix="my_bot:",  # New: Redis prefix support
    throttling_default_max=10,
)

# New setup call
sentinel = Sentinel(config=config)
dp.message.middleware(sentinel.middleware)

# New decorators (old ones still work but deprecated)
@sentinel_rate_limit(limit=5, window=60)
@sentinel_debounce(delay=2)
@sentinel_require_registered()
async def protected_handler(message: Message):
    await message.answer("Protected command!")
```

## New Features

### Redis Prefix Support

```python
# v0.2.0: Namespace your Redis keys
config = SentinelConfig(
    backend="redis",
    redis_url="redis://localhost:6379",
    redis_prefix="my_bot:",  # Keys will be prefixed with "my_bot:"
)
```

### Expanded Hooks System

```python
# v0.2.0: More hook types available
async def on_rate_limited_hook(user_id: int, limit: int, window: int):
    """Called when user hits rate limit."""
    logger.warning(f"User {user_id} rate limited: {limit}/{window}")

async def on_debounced_hook(user_id: int, delay: int):
    """Called when message is debounced."""
    logger.info(f"Message from {user_id} debounced for {delay}s")

# Register hooks
sentinel = Sentinel(
    config=config,
    on_rate_limited=on_rate_limited_hook,
    on_debounced=on_debounced_hook,
)
```

## Deprecations & Removal Timeline

- **v0.2.0**: `Sentinel.setup()` method deprecated
- **v0.2.0**: Old decorator names deprecated (still functional)
- **v0.3.0**: `Sentinel.setup()` method removed
- **v1.0.0**: Old decorator names removed

## Automatic Migration Tools

### Migration Script

```bash
# Run the migration script to update your code
python -m aiogram_sentinel.migrate v0.1.0 v0.2.0 --path /path/to/your/bot
```

### Manual Migration Checklist

- [ ] Update `Sentinel.setup()` calls to new `Sentinel()` constructor
- [ ] Add `dp.message.middleware(sentinel.middleware)` registration
- [ ] Update decorator imports to use `sentinel_` prefix
- [ ] Add Redis prefix configuration if using Redis backend
- [ ] Test all middleware functionality
- [ ] Update any custom hooks to use new hook system

## Common Issues

### Issue: "Sentinel.setup() not found"

**Solution**: Replace with new constructor pattern:

```python
# Old
router, backends = await Sentinel.setup(dp, config)

# New
sentinel = Sentinel(config=config)
dp.message.middleware(sentinel.middleware)
```

### Issue: "Decorator not found"

**Solution**: Update imports:

```python
# Old
from aiogram_sentinel import rate_limit, debounce, require_registered

# New
from aiogram_sentinel.decorators import sentinel_rate_limit, sentinel_debounce, sentinel_require_registered
```

## Need Help?

If you encounter issues during migration:
- Check the [Troubleshooting Guide](../troubleshooting.md)
- Open an issue on [GitHub](https://github.com/ArmanAvanesyan/aiogram-sentinel/issues)
- Join our [Discussions](https://github.com/ArmanAvanesyan/aiogram-sentinel/discussions)
