# Migration Guide: v0.2.0 to v1.0.0

## Overview

This migration guide covers the upgrade from aiogram-sentinel v0.2.0 to v1.0.0. The main changes include token bucket rate limiting, stable API commitment, and enhanced performance optimizations.

## Breaking Changes

| Old API | New API | Impact | Migration Required |
|---------|---------|--------|-------------------|
| `throttling_default_max` | `rate_limit_default_tokens` | High | Yes |
| `throttling_default_per_seconds` | `rate_limit_default_refill_rate` | High | Yes |
| `@sentinel_rate_limit(limit, window)` | `@sentinel_rate_limit(tokens, refill_rate)` | Medium | Optional |
| `MemoryStorage()` | `MemoryStorage(config)` | Low | Optional |

## Code Modifications

### Before (v0.2.0)

```python
from aiogram_sentinel import Sentinel, SentinelConfig
from aiogram_sentinel.storage import MemoryStorage

# Old configuration
config = SentinelConfig(
    throttling_default_max=10,           # 10 messages
    throttling_default_per_seconds=60,   # per 60 seconds
    backend="memory",
)

# Old storage initialization
storage = MemoryStorage()

# Old rate limit decorator
@sentinel_rate_limit(limit=5, window=30)
async def handler(message: Message):
    await message.answer("Rate limited!")
```

### After (v1.0.0)

```python
from aiogram_sentinel import Sentinel, SentinelConfig
from aiogram_sentinel.storage import MemoryStorage

# New configuration with token bucket
config = SentinelConfig(
    rate_limit_default_tokens=10,        # 10 tokens in bucket
    rate_limit_default_refill_rate=0.17, # ~10 tokens per 60 seconds
    backend="memory",
)

# New storage initialization (config required)
storage = MemoryStorage(config)

# New rate limit decorator
@sentinel_rate_limit(tokens=5, refill_rate=0.33)  # 5 tokens, ~10 per 30 seconds
async def handler(message: Message):
    await message.answer("Rate limited!")
```

## New Features

### Token Bucket Rate Limiting

v1.0.0 introduces token bucket algorithm for more accurate rate limiting:

```python
# Token bucket configuration
config = SentinelConfig(
    rate_limit_default_tokens=20,        # Initial bucket size
    rate_limit_default_refill_rate=0.5,  # 0.5 tokens per second
    rate_limit_burst_multiplier=2.0,     # Allow 2x burst
)

# Per-handler token bucket
@sentinel_rate_limit(
    tokens=5,           # 5 tokens in bucket
    refill_rate=0.17,   # ~1 token per 6 seconds
    burst_multiplier=1.5 # Allow 1.5x burst
)
async def slow_handler(message: Message):
    await message.answer("Slow command!")
```

### Enhanced Performance

```python
# v1.0.0: Optimized memory usage
config = SentinelConfig(
    memory_cleanup_interval=300,  # Cleanup every 5 minutes
    memory_max_entries=10000,     # Limit memory usage
    redis_connection_pool_size=10, # Redis connection pooling
)
```

### Stable API Commitment

v1.0.0 marks the stable API. No breaking changes will be introduced in v1.x.x releases.

## Deprecations & Removal Timeline

- **v1.0.0**: Old rate limiting parameters deprecated
- **v1.1.0**: Old parameters removed
- **v1.0.0**: `MemoryStorage()` without config deprecated
- **v1.1.0**: Config required for `MemoryStorage()`

## Migration Steps

### 1. Update Configuration

```python
# Calculate refill rate from old parameters
# old_max = 10, old_window = 60
# new_tokens = 10, new_refill_rate = 10/60 = 0.167

config = SentinelConfig(
    rate_limit_default_tokens=10,        # Same as old_max
    rate_limit_default_refill_rate=0.167, # old_max / old_window
)
```

### 2. Update Decorators

```python
# Old
@sentinel_rate_limit(limit=5, window=30)

# New
@sentinel_rate_limit(tokens=5, refill_rate=0.167)  # 5/30 = 0.167
```

### 3. Update Storage Initialization

```python
# Old
storage = MemoryStorage()

# New
storage = MemoryStorage(config)
```

## Performance Improvements

### Memory Usage

- **v0.2.0**: ~1KB per active user
- **v1.0.0**: ~0.5KB per active user (50% reduction)

### Rate Limiting Accuracy

- **v0.2.0**: Sliding window (approximate)
- **v1.0.0**: Token bucket (exact)

### Redis Operations

- **v0.2.0**: 2-3 Redis calls per rate limit check
- **v1.0.0**: 1 Redis call per rate limit check

## Common Issues

### Issue: "Invalid refill_rate value"

**Solution**: Ensure refill_rate is between 0 and 1:

```python
# Wrong
refill_rate=60  # Too high

# Correct
refill_rate=0.167  # 10 tokens per 60 seconds
```

### Issue: "MemoryStorage requires config"

**Solution**: Pass config to MemoryStorage:

```python
# Old
storage = MemoryStorage()

# New
storage = MemoryStorage(config)
```

## Testing Your Migration

```python
# Test rate limiting behavior
import asyncio
from aiogram_sentinel import Sentinel, SentinelConfig

async def test_rate_limiting():
    config = SentinelConfig(
        rate_limit_default_tokens=5,
        rate_limit_default_refill_rate=0.5,  # 0.5 tokens per second
    )
    
    sentinel = Sentinel(config=config)
    
    # Test rapid requests
    for i in range(10):
        allowed, remaining, reset_after = await sentinel.rate_limiter_backend.allow(
            "test_user", 5, 0.5
        )
        print(f"Request {i}: allowed={allowed}, remaining={remaining}")
        await asyncio.sleep(0.1)

asyncio.run(test_rate_limiting())
```

## Need Help?

If you encounter issues during migration:
- Check the [Troubleshooting Guide](../troubleshooting.md)
- Open an issue on [GitHub](https://github.com/ArmanAvanesyan/aiogram-sentinel/issues)
- Join our [Discussions](https://github.com/ArmanAvanesyan/aiogram-sentinel/discussions)
